/*
 *  Copyright (C) 2001 Philip Langdale, Matthew Aubury
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"

#include "ProgressListener2.h"

#include "mozilla.h"
#include "mime.h"
#include "misc_general.h"
#include "misc_gui.h"
#include "glade.h"
#include "dialog.h"
#include "favicon.h"
#include "eel-gconf-extensions.h"
#include "embed.h"

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <gtk/gtkclist.h>
#include <gtk/gtkentry.h>
#include <gtk/gtkprogress.h>
#include <gtk/gtkoptionmenu.h>
#include <gtk/gtkmain.h>
#include <libgnome/gnome-exec.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnomeui/gnome-stock.h>
#include <libgnomeui/gnome-popup-menu.h>
#include <libgnomeui/gnome-app.h>
#include <libgnomeui/gnome-app-helper.h>
#include <libgnomeui/gnome-winhints.h>
#include <libgnomevfs/gnome-vfs-mime.h>

#include "nsXPIDLString.h"
#include "nsIChannel.h"
#include "nsIMIMEService.h"
#include "nsIFTPChannel.h"
#include "nsCRT.h"
#include "nsCOMPtr.h"

extern "C" {
/*
 * Columns of the download dialog
 */
enum
{
	COL_PERCENT = 0,
	COL_STATUS,
	COL_FILENAME,
	COL_SPEED,
	COL_RECEIVED,
	COL_SIZE,
	COL_ELAPSED,
	COL_REMAINING,
	COL_SOURCE,
	COL_DESTINATION,

	/* must be last! */
	COL_MAX
};

/*
 * Widgets needed by C code
 */
typedef struct
{
	GtkWidget *window;
	GtkWidget *clist;
	GtkWidget *details_frame;
	GtkWidget *details_location;
	GtkWidget *details_file;
	GtkWidget *details_status;
	GtkWidget *details_elapsed;
	GtkWidget *details_remaining;
	GtkWidget *details_progress;
	GtkWidget *details_button;
	GtkWidget *keep_open_check;
	GtkWidget *pause_button;
	GtkWidget *resume_button;
	GtkWidget *abort_button;
} DownloadDialogWidgets;

/**
 * Details information
 */
typedef struct
{
	gchar *location;
	gchar *file;;
	gchar *status;
	gchar *elapsed;
	gchar *remaining;
	gfloat progress;
} DetailsInfo;

/*
 * Global widgets declaration
 */
static DownloadDialogWidgets *dialog = NULL;
static GHashTable *details_hash;

/*
 * Local C function prototypes
 */
void download_dialog_pause_cb (GtkButton *button, 
			       DownloadDialogWidgets *widgets);
void download_dialog_resume_cb (GtkButton *button, 
				DownloadDialogWidgets *widgets);
void download_dialog_abort_cb (GtkButton *button, 
			       DownloadDialogWidgets *widgets);
void download_clist_select_row_cb (GtkCList *clist, gint row, 
				   gint column, GdkEvent *ev, 
				   DownloadDialogWidgets *widgets);
void download_clist_unselect_row_cb (GtkCList *clist, gint row, 
				     gint column, GdkEvent *ev, 
				     DownloadDialogWidgets *widgets);
gboolean download_clist_button_press_event_cb (GtkCList *clist,
					       GdkEventButton *event,
				               DownloadDialogWidgets *widgets);
gboolean download_dialog_delete_cb (GtkWidget *window, GdkEventAny *event,
				    DownloadDialogWidgets *widgets);
void download_dialog_details_cb (GtkToggleButton *button, 
				 DownloadDialogWidgets *widgets);
void download_dialog_keep_open_cb (GtkToggleButton *button, 
			           DownloadDialogWidgets *widgets);

static void selection_changed (DownloadDialogWidgets *widgets);
}

static gboolean timeout_listener (GProgressListener2 *Progress);

NS_IMPL_ISUPPORTS4 (GProgressListener2, nsIDownload, nsIWebProgressListener,
		    nsIProgressDialog, nsISupportsWeakReference)

//---------------------------------------------------------------------------

//---------------------------------------------------------------------------

GProgressListener2::GProgressListener2 (void) : mLauncher(nsnull),
						mPersist(nsnull),
						mHandler(nsnull),
						mObserver(nsnull),
#if MOZILLA_SNAPSHOT > 1
						mMIMEInfo(nsnull),
#endif
						mPercentComplete(0)
{
	NS_INIT_ISUPPORTS ();
}

GProgressListener2::~GProgressListener2 ()
{
	/* destructor code */
}

NS_METHOD GProgressListener2::InitForPersist (nsIWebBrowserPersist *aPersist,
					      nsIDOMWindow *aParent, 
					      nsIURI *aURI,
					      nsIFile *aFile, 
					      DownloadAction aAction,
					      gpointer info,
					      PRInt64 aTimeDownloadStarted)
{
	nsresult rv;

	/* fill in download details */
	mParent = aParent;
	mAction = aAction;
	mNoDialog = (mAction == ACTION_FAVICON || mAction == ACTION_SET_PIXMAP ||
		     mAction == ACTION_ADDBOOKMARK || mAction == ACTION_SILENT);
	mUri = aURI;
	mFile = aFile;
	mPersist = aPersist;
	mTimeDownloadStarted = aTimeDownloadStarted;
	mCallbackInfo = info;

	/* do remaining init */
	rv = PrivateInit ();

	/* pick up progress messages */
	mPersist->SetProgressListener (this);

	/* done */
	return rv;
}

NS_METHOD GProgressListener2::InitForDownload (nsIHelperAppLauncher *aLauncher,
					       nsISupports *aContext,
					       GContentHandler *aHandler,
					       DownloadAction aAction)
{
	nsresult rv;

	mNoDialog = 0;

	/* fill in download details */
	mParent = do_QueryInterface (aContext);
	mAction = aAction;
	mNoDialog = (mAction == ACTION_FAVICON || mAction == ACTION_SET_PIXMAP ||
		     mAction == ACTION_ADDBOOKMARK || mAction == ACTION_SILENT);
	mHandler = aHandler;
	mLauncher = aLauncher;

#if MOZILLA_SNAPSHOT > 11
	rv = mLauncher->GetSource(getter_AddRefs(mUri));
	rv = mLauncher->GetTimeDownloadStarted(&mTimeDownloadStarted);
	rv = mLauncher->GetTargetFile(getter_AddRefs(mFile));
#else
	rv = mLauncher->GetDownloadInfo (getter_AddRefs (mUri),
					 &mTimeDownloadStarted,
					 getter_AddRefs (mFile));
#endif

	/* do remaining init */
	rv = PrivateInit ();

	/* pick up progress messages */
	mLauncher->SetWebProgressListener (this);

	/* done */
	return rv;
}

NS_METHOD GProgressListener2::PrivateInit (void)
{
	GtkWidget *aParentWidget;
	GtkCList *clist;
	nsresult rv;

	if (mAction == ACTION_FAVICON || mAction == ACTION_FAVICON_EDITOR ||
	    mAction == ACTION_SET_PIXMAP)
	{
		mTimer = g_timer_new ();
		g_timer_start (mTimer);
		mTimeoutFunc = gtk_timeout_add
			(10000, (GtkFunction) timeout_listener, this);
	}

	/* setup this download */
	mInterval            = 500000;     /* in microsecs == 500ms == 0.5s */
	mPriorKRate          = 0;
	mRateChanges         = 0;
	mRateChangeLimit     = 2;          /* only update rate every second */
	mCheckedCanPause     = PR_FALSE;
	mCanPause            = PR_FALSE;
	mIsPaused            = PR_FALSE;
	mAbort               = PR_FALSE;
	PRInt64 now          = PR_Now ();
	mLastUpdate          = now;

//	mStartTime           = (mTimeDownloadStarted != 0 ? 
//				mTimeDownloadStarted : now);
	mStartTime           = now; //Stupid mozilla race condition

	mElapsed             = now - mStartTime;

	if (!mNoDialog)
	{
		if(!GTK_IS_CLIST(dialog->clist))
		{
			downloader_dialog_init();
		}

		/* increment window count while progress is under way */
		window_count++;

		/* init details structure */
		DetailsInfo *dinfo = g_new0 (DetailsInfo, 1);
		g_hash_table_insert (details_hash, this, dinfo);
		
		/* get parent mozilla widget */
		aParentWidget = mozilla_find_gtk_parent (mParent);

		/* initialise all columns to empty */
		gchar *text[COL_MAX];
		for (int i = 0; i < COL_MAX; i++) 
		{ 
			text[i] = NULL; 
		}

		/* setup filename */
		nsCAutoString leafName;
		rv = mFile->GetNativeLeafName (leafName);
		text[COL_FILENAME] = nsCRT::strdup (leafName.get());

		nsCAutoString path;
		rv = mFile->GetNativePath (path);
		text[COL_DESTINATION] = nsCRT::strdup (path.get());

		nsCAutoString spec;
		rv = mUri ->GetSpec (spec);
		text[COL_SOURCE] = nsCRT::strdup (spec.get());

		dinfo->location = g_strdup (text[COL_SOURCE]);
		dinfo->file = g_strdup (text[COL_DESTINATION]);

		/* add row and setup data */
		clist = GTK_CLIST (dialog->clist);
		int row = gtk_clist_append (clist, text);
		gtk_clist_set_row_data (clist, row, this);

		/* insert an empty progress bar */
		gtk_clist_set_pixmap (clist, row, COL_STATUS, 
				      misc_gui_render_bar (BAR_ACTIVE, 0),
				      NULL);
		dinfo->status = g_strdup (_("Starting download..."));

		/* free allocated strings */
		nsMemory::Free (text[COL_FILENAME]);
		nsMemory::Free (text[COL_SOURCE]);
		nsMemory::Free (text[COL_DESTINATION]);
	
		/* make sure the dialog is shown, moving it to the current
		 * workspace if it's mapped and it's not there already */
		if (GTK_WIDGET_MAPPED (dialog->window) &&
		    gnome_win_hints_get_current_workspace () !=
		    gnome_win_hints_get_workspace (dialog->window))
		{
			gnome_win_hints_set_workspace (dialog->window,
				gnome_win_hints_get_current_workspace ());
		}
		else gtk_widget_show_now (dialog->window);

		g_assert (clist->rows > 0);
	}

	/* done */
	return NS_OK;
}

/* nsIProgressDialog impl */
NS_IMETHODIMP GProgressListener2::Init(nsIURI *aSource,
				       nsILocalFile *aTarget,
				       const PRUnichar *aDisplayName,
#if MOZILLA_SNAPSHOT > 1
				       nsIMIMEInfo *aMIMEInfo,
#else
				       const PRUnichar *aOpeningWith,
#endif
				       PRInt64 aStartTime,
				       nsIWebBrowserPersist *aPersist)
{
	mUri = aSource;
	mFile = aTarget;
	mTimeDownloadStarted = aStartTime;
	mStartTime = aStartTime;
	mPersist = aPersist;
#if MOZILLA_SNAPSHOT > 1
	mMIMEInfo = aMIMEInfo;
#endif

	mAction = ACTION_NONE;
#if MOZILLA_SNAPSHOT > 1
	if(mMIMEInfo)
	{
		nsMIMEInfoHandleAction mimeAction;
		if(NS_SUCCEEDED(mMIMEInfo->GetPreferredAction(&mimeAction)))
		{
			mAction = (mimeAction == nsIMIMEInfo::useHelperApp) ?
				  ACTION_SAVEFORHELPER : ACTION_NONE;
		}
	}
#endif
	mNoDialog = 0;

	return PrivateInit();
}

NS_IMETHODIMP GProgressListener2::Open(nsIDOMWindow *aParent)
{
	mParent = aParent;
	mNoDialog = 0;

	return NS_OK;
}

/* attribute long long startTime; */
NS_IMETHODIMP GProgressListener2::GetStartTime(PRInt64 *aStartTime)
{
	*aStartTime = mStartTime;
	return NS_OK;
}

/* attribute nsIURI source; */
NS_IMETHODIMP GProgressListener2::GetSource(nsIURI * *aSource)
{
	NS_IF_ADDREF(*aSource = mUri);
	return NS_OK;
}

/* attribute nsILocalFile target; */
NS_IMETHODIMP GProgressListener2::GetTarget(nsILocalFile * *aTarget)
{
	nsCOMPtr<nsILocalFile> aLocalFile = do_QueryInterface(mFile);
	NS_IF_ADDREF(*aTarget = aLocalFile);
	return NS_OK;
}

#if MOZILLA_SNAPSHOT > 1
NS_IMETHODIMP GProgressListener2::GetMIMEInfo(nsIMIMEInfo * *aMIMEInfo)
{
	NS_IF_ADDREF(*aMIMEInfo = mMIMEInfo);
	return NS_OK;
}
#else
NS_IMETHODIMP GProgressListener2::GetOpeningWith(PRUnichar * *aOpeningWith)
{
	*aOpeningWith = NULL;
	return NS_OK;
}
#endif

/* attribute nsIObserver observer; */
NS_IMETHODIMP GProgressListener2::GetObserver(nsIObserver * *aObserver)
{
	NS_IF_ADDREF(*aObserver = mObserver);
	return NS_OK;
}
NS_IMETHODIMP GProgressListener2::SetObserver(nsIObserver * aObserver)
{
	mObserver = aObserver;
	return NS_OK;
}

/* readonly attribute nsIWebBrowserPersist persist; */
NS_IMETHODIMP GProgressListener2::GetPersist(nsIWebBrowserPersist * *aPersist)
{
	NS_IF_ADDREF(*aPersist = mPersist);
	return NS_OK;
}

/* readonly attribute nsIDOMWindow dialog; */
NS_IMETHODIMP GProgressListener2::GetDialog(nsIDOMWindow * *aDialog)
{
	*aDialog = nsnull;
	return NS_OK;
}

#if MOZILLA_SNAPSHOT > 1
/* attribute PRBool cancelDownloadOnClose; */
NS_IMETHODIMP  GProgressListener2::GetCancelDownloadOnClose(PRBool *aCancelDownloadOnClose)
{
    return NS_ERROR_NOT_IMPLEMENTED;
}
NS_IMETHODIMP  GProgressListener2::SetCancelDownloadOnClose(PRBool aCancelDownloadOnClose)
{
    return NS_ERROR_NOT_IMPLEMENTED;
}
#endif

/* nsIDownload Impl */

NS_IMETHODIMP GProgressListener2::SetDialog(nsIDOMWindow *aDialog)
{
	return NS_OK;
}

/* readonly attribute PRInt32 percentComplete; */
NS_IMETHODIMP GProgressListener2::GetPercentComplete(PRInt32 *aPercentComplete)
{
	return *aPercentComplete = mPercentComplete;
}

/* attribute wstring displayName; */
NS_IMETHODIMP GProgressListener2::GetDisplayName(PRUnichar * *aDisplayName)
{
	*aDisplayName = nsnull;
	return NS_OK;
}
NS_IMETHODIMP GProgressListener2::SetDisplayName(const PRUnichar * aDisplayName)
{
	return NS_OK;
}

/* attribute nsIWebProgressListener listener; */
NS_IMETHODIMP GProgressListener2::GetListener(nsIWebProgressListener * *aListener)
{
	*aListener = nsnull;
	return NS_OK;
}
NS_IMETHODIMP GProgressListener2::SetListener(nsIWebProgressListener * aListener)
{
	return NS_OK;
}

/* nsIWebProgressListener impl */

/*
 * void onStateChange (in nsIWebProgress aWebProgress, 
 *                     in nsIRequest aRequest, 
 *		       in long aStateFlags, 
 *		       in unsigned long aStatus);
 */
NS_IMETHODIMP GProgressListener2::OnStateChange (nsIWebProgress *aWebProgress,
						 nsIRequest *aRequest,
						 PRUint32 aStateFlags,
						 PRUint32 aStatus)
{
	if (mAbort) return NS_ERROR_FAILURE;

	if (aStateFlags & nsIWebProgressListener::STATE_STOP)
	{
		nsCAutoString filename;
		mFile->GetNativePath (filename);

		switch (mAction)
		{
		case ACTION_VIEWSOURCE:
			misc_general_launch_external_viewer (filename.get ());
			break;

		case ACTION_SETBACKGROUND:
			gchar *command;

			/* build command */
			command = g_strconcat ("background-properties-capplet "
					       "--init-session-settings "
					       "--background-image=",
					       filename.get(), NULL);

			/* execute it synchronously */
			gnome_execute_shell (NULL, command);

			/* free */
			g_free (command);
			break;

		case ACTION_SAVEFORHELPER:
#if MOZILLA_SNAPSHOT > 1
			LaunchHelperApp();
#else
			if (mLauncher)
			{
				mLauncher->CloseProgressWindow ();
			}
			if (mHandler)
			{
				mHandler->LaunchHelperApp ();
			}
#endif
			break;

		case ACTION_ADDBOOKMARK:
			SaveImageInfo *info; 
			info = (SaveImageInfo *) mCallbackInfo;
			bookmarks_update_images (filename.get());
			g_free (info->text);
			g_free (info->url);
			g_free (info->smarturl);
			g_free (info);
			break;
			
		case ACTION_FAVICON_EDITOR:
		case ACTION_FAVICON:
			if (mTimer) g_timer_destroy (mTimer);
			gtk_timeout_remove (mTimeoutFunc);
			favicon_download_completed
				((FaviconInfo *) mCallbackInfo);
			break;
			
		case ACTION_SET_PIXMAP:
			{
				if (mTimer) g_timer_destroy (mTimer);
				gtk_timeout_remove (mTimeoutFunc);

				SetPixmapInfo *i =
					(SetPixmapInfo *) mCallbackInfo;

				gchar *fn;
				nsCAutoString cfn;
				mFile->GetNativePath(cfn);
				fn = nsCRT::strdup(cfn.get());

				if (i->pixmap == NULL ||
				    !GTK_IS_PIXMAP (i->pixmap) ||
				    GTK_OBJECT (i->pixmap)->ref_count <= 0 ||
				    !i->url || !i->selected_url ||
				    strcmp (i->url, i->selected_url))
				{
					unlink (fn);
					nsMemory::Free (fn);
					g_free (i);
					break;
				}

				/* build a pixmap and update the given
				 * widget */
				PixmapData *pix =
					misc_gui_pixmap_data_new_from_file (
								fn, FALSE);

				if (pix)
				{
					gtk_pixmap_set
						(GTK_PIXMAP (i->pixmap),
						 pix->pixmap, pix->mask);
					gtk_widget_show
						(GTK_WIDGET (i->pixmap));
					if (pix->pixmap)
						gdk_pixmap_unref (pix->pixmap);
					if (pix->mask)
						gdk_bitmap_unref (pix->mask);
					g_free (pix);
				}

				/* remove file since we dont need it
				 * anymore */
				unlink (fn);
				g_free (fn);
				g_free (i);
			}
			break;
			
		case ACTION_NONE:
		case ACTION_SILENT:
			if (mLauncher)
			{
				mLauncher->CloseProgressWindow ();
			}
		}

		Cleanup();
	}

	/* done */
	return NS_OK;
}

/*
 * void onProgressChange (in nsIWebProgress aWebProgress, 
 *                        in nsIRequest aRequest, 
 *                        in long aCurSelfProgress, 
 *                        in long aMaxSelfProgress, 
 *                        in long aCurTotalProgress, 
 *                        in long aMaxTotalProgress); 
 */
NS_IMETHODIMP GProgressListener2::
			OnProgressChange (nsIWebProgress *aWebProgress,
					  nsIRequest *aRequest,
					  PRInt32 aCurSelfProgress,
					  PRInt32 aMaxSelfProgress,
					  PRInt32 aCurTotalProgress,
					  PRInt32 aMaxTotalProgress)
{
	gchar *statusstr, *tmp;

	if (mAbort) return NS_ERROR_FAILURE;

	if (mAction == ACTION_FAVICON || mAction == ACTION_FAVICON_EDITOR)
	{
		PRInt32 currentKBytes =
			(PRInt32)(aCurTotalProgress / 1024.0 + 0.5);
		PRInt32 totalKBytes =
			(PRInt32)(aMaxTotalProgress / 1024.0 + 0.5);

		if (currentKBytes >= 40 || totalKBytes >= 40)
		{
			Abort ();
		}

		return NS_OK;
	}

	if (mNoDialog) return NS_OK;

	GtkCList *clist = GTK_CLIST (dialog->clist);
	int row = gtk_clist_find_row_from_data (clist, this);
	DetailsInfo *dinfo = (DetailsInfo *)
		g_hash_table_lookup (details_hash, this);
	if (!dinfo) return NS_OK;
	char buffer[256];

	if (!mCheckedCanPause)
	{
		mCheckedCanPause = PR_TRUE;

		nsresult rv;
		nsCOMPtr<nsIFTPChannel> aChannel = 
			do_QueryInterface (aRequest, &rv);

		mCanPause = (NS_SUCCEEDED (rv) ? PR_TRUE : PR_FALSE);
	}
	mRequest = aRequest;

	PRInt64 now = PR_Now ();

	/* get out if we're updating too quickly */
	if ((now - mLastUpdate < mInterval) && 
	    (aMaxTotalProgress != -1) &&  
	    (aCurTotalProgress < aMaxTotalProgress))
	{
		return NS_OK;
	}

	/* freeze the list to stop multiple updates */
	gtk_clist_freeze (clist);

	/* compute elapsed time */
	mLastUpdate = now;
	mElapsed = now - mStartTime;
	FormatTime (mElapsed / 1000000, buffer);
	gtk_clist_set_text (clist, row, COL_ELAPSED, buffer);
	g_free (dinfo->elapsed);
	dinfo->elapsed = g_strdup (buffer);

	/* compute size done */
	PRInt32 currentKBytes = (PRInt32)(aCurTotalProgress / 1024.0 + 0.5);
	if (currentKBytes >= 10000)
	{
		sprintf (buffer, "%.2f MB", currentKBytes / 1024.0);
	}
	else
	{
		sprintf (buffer, "%d KB", currentKBytes);
	}
	gtk_clist_set_text (clist, row, COL_RECEIVED, buffer);
	statusstr = g_strdup_printf ("%s", buffer);

	/* compute total size */
	PRInt32 totalKBytes = (PRInt32)(aMaxTotalProgress / 1024.0 + 0.5);
	if (totalKBytes >= 10000)
	{
		sprintf (buffer, "%.2f MB", totalKBytes / 1024.0);
	}
	else
	{
		sprintf (buffer, "%d KB", totalKBytes);
	}
	gtk_clist_set_text (clist, row, COL_SIZE, buffer);
	tmp = g_strdup_printf (_("%s of %s"), statusstr, buffer);
	g_free (statusstr);
	statusstr = tmp;

	/* compute progress value */
	if (aMaxTotalProgress > 0)
	{
		gfloat progress = (gfloat)aCurTotalProgress /
				  (gfloat)aMaxTotalProgress;
		/* handle the case where mozilla gunzips on the fly and
		 * aCurTotalProgress becomes bigger than aMaxTotalProgress */
		if (progress > 1)
		{
			progress = 1;
		}

		mPercentComplete = NS_STATIC_CAST(PRInt32, progress);

		sprintf (buffer, "%.1f%%", progress * 100.0);
		gtk_clist_set_text (clist, row, COL_PERCENT, buffer);

		dinfo->progress = progress;
		int width = (int)((progress * (BAR_WIDTH - 2)) + 0.5);
		gtk_clist_set_pixmap (clist, row, COL_STATUS,
				      misc_gui_render_bar (BAR_ACTIVE, width),
				      NULL);
	}
	else
	{
		dinfo->progress = -1;
	}

	/* compute download rate */
	PRInt64 currentRate;
	if (mElapsed)
	{
		currentRate = ((PRInt64)(aCurTotalProgress)) * 1000000 / 
			mElapsed;
	}
	else
	{
		currentRate = 0;
	}		
	if (!mIsPaused)
	{
		if (currentRate)
		{
			PRFloat64 currentKRate = ((PRFloat64)currentRate)/1024;
			if (currentKRate != mPriorKRate)
			{
				if (mRateChanges++ == mRateChangeLimit)
				{
					mPriorKRate = currentKRate;
					mRateChanges = 0;
				}
				else
				{
					currentKRate = mPriorKRate;
				}
			}
			else
			{
			mRateChanges = 0;
			}
			sprintf (buffer, "%.1f KB/s", currentKRate);
			gtk_clist_set_text (clist, row, COL_SPEED, buffer);
			tmp = g_strdup_printf (_("%s at %s"),
					       statusstr, buffer);
			g_free (statusstr);
			statusstr = tmp;
		}
		else
		{
			gtk_clist_set_text (clist, row, COL_SPEED, "?? KB/s");
			tmp = g_strdup_printf (_("%s at KB/s"), statusstr);
			g_free (statusstr);
			statusstr = tmp;
		}
		
		g_free (dinfo->status);
		dinfo->status = statusstr;
	}
	
	/* compute time remaining */
	if (currentRate && (aMaxTotalProgress > 0))
	 {
		PRInt32 remaining = 
			(PRInt32)((aMaxTotalProgress - aCurTotalProgress)
				   /currentRate +.5);

		FormatTime (remaining, buffer);
 		gtk_clist_set_text (clist, row, COL_REMAINING, buffer);
		g_free (dinfo->remaining);
		dinfo->remaining = g_strdup (buffer);
	}
	else
	{
 		gtk_clist_set_text (clist, row, COL_REMAINING, "??");
		g_free (dinfo->remaining);
		dinfo->remaining = g_strdup ("??");
	}

	/* thaw the list to show updates */
	gtk_clist_thaw (clist);

	/* make sure buttons are up to date */
	selection_changed (dialog);

	/* done */
	return NS_OK;
}

/* void onLocationChange (in nsIWebProgress aWebProgress, in nsIRequest aRequest, in nsIURI location); */
NS_IMETHODIMP GProgressListener2::
			OnLocationChange(nsIWebProgress *aWebProgress,
					 nsIRequest *aRequest, nsIURI *location)
{
    return NS_OK;
}

/* void onStatusChange (in nsIWebProgress aWebProgress, in nsIRequest aRequest, in nsresult aStatus, in wstring aMessage); */
NS_IMETHODIMP GProgressListener2::
			OnStatusChange(nsIWebProgress *aWebProgress,
				       nsIRequest *aRequest, nsresult aStatus,
				       const PRUnichar *aMessage)
{
    return NS_OK;
}

/* void onSecurityChange (in nsIWebProgress aWebProgress, in nsIRequest aRequest, in long state); */
NS_IMETHODIMP GProgressListener2::
			OnSecurityChange(nsIWebProgress *aWebProgress,
					 nsIRequest *aRequest,
					 PRUint32 state)
{
    return NS_OK;
}

//---------------------------------------------------------------------------

void GProgressListener2::FormatTime (PRUint32 aTime, char *buffer)
{
	PRUint32 secs = (PRUint32)(aTime + .5);
	PRUint32 hours = secs / 3600;
	secs -= hours * 3600;
	PRUint32 mins = secs / 60;
	secs -= mins * 60;

	if (hours)
	{
		sprintf (buffer, "%u:%02u.%02u", hours, mins, secs);
	}
	else
	{
		sprintf (buffer, "%02u.%02u", mins, secs);
	}
}

//---------------------------------------------------------------------------

NS_METHOD GProgressListener2::LaunchHelperApp (void)
{
	if (!mMIMEInfo)
		return NS_ERROR_FAILURE;

	nsresult rv;

	nsCOMPtr<nsIFile> helperFile;
	rv = mMIMEInfo->GetPreferredApplicationHandler(getter_AddRefs(helperFile));
	if(NS_FAILED(rv)) return NS_ERROR_FAILURE;

	nsCAutoString helperFileName;
	rv = helperFile->GetNativePath(helperFileName);
	if(NS_FAILED(rv)) return NS_ERROR_FAILURE;

	nsMIMEInfoHandleAction mimeAction;
	rv = mMIMEInfo->GetPreferredAction(&mimeAction);
	if(NS_FAILED(rv)) return NS_ERROR_FAILURE;

	nsCOMPtr<nsIExternalHelperAppService> helperService =
		do_GetService (NS_EXTERNALHELPERAPPSERVICE_CONTRACTID, &rv);
	if (NS_SUCCEEDED(rv))
	{
		nsCOMPtr<nsPIExternalAppLauncher> appLauncher =
			do_QueryInterface (helperService, &rv);
		if (NS_SUCCEEDED(rv))
		{
			appLauncher->DeleteTemporaryFileOnExit(mFile);
		}
	}

	nsCAutoString aFileName;
	mFile->GetNativePath(aFileName);
	if(NS_FAILED(rv)) return NS_ERROR_FAILURE;

	gchar *tmp_dir = g_get_tmp_dir ();

	nsXPIDLString helperDesc;
	rv = mMIMEInfo->GetApplicationDescription(getter_Copies(helperDesc));
	if(NS_FAILED(rv)) return NS_ERROR_FAILURE;

	if(helperDesc.Equals(NS_LITERAL_STRING("runInTerminal")))
	{
		const char *argv[4] = { "gnome-terminal", "-e",
					helperFileName.get(),
					aFileName.get() };
		gnome_execute_async (tmp_dir, 4, const_cast<char **>(argv));
	}
	else
	{
		const char *argv[2] = { helperFileName.get(),
					aFileName.get() };
		gnome_execute_async (tmp_dir, 2, const_cast<char **>(argv));
	}

	return NS_OK;
}

//---------------------------------------------------------------------------

nsresult GProgressListener2::Cleanup (void)
{
	gint row;

	mAbort = PR_TRUE;

	if (mNoDialog) return NS_OK;

	/* decrement window count as this progess is finished */
	window_count--;

	row = gtk_clist_find_row_from_data
		(GTK_CLIST (dialog->clist), this);

	g_assert (row >= 0);

        /* remove the appropriate row */
	gtk_clist_remove (GTK_CLIST (dialog->clist), row);

	/* if no more rows, hide the dialog */
	if (GTK_CLIST (dialog->clist)->rows == 0 &&
	    !GTK_TOGGLE_BUTTON (dialog->keep_open_check)->active)
	{
		gtk_widget_hide (dialog->window);
	}

	/* hide details frame if there's no selection left */
	if (!GTK_CLIST (dialog->clist)->selection)
	{
		gtk_toggle_button_set_active
			(GTK_TOGGLE_BUTTON
			 (dialog->details_button), FALSE);
	}

	/* cleanup direct stuff */
	DetailsInfo *dinfo = (DetailsInfo *)
		g_hash_table_lookup (details_hash, this);
	g_free (dinfo->location);
	g_free (dinfo->file);
	g_free (dinfo->status);
	g_free (dinfo->elapsed);
	g_free (dinfo->remaining);
	g_free (dinfo);
	g_hash_table_remove (details_hash, this);

	/* if there are no more windows left, exit galeon */
	if (g_list_length (all_windows) == 0 && 
	    !galeon_server_mode && window_count == 0)
	{
		galeon_exit (TRUE, TRUE);
	}

	return NS_OK;
}

nsresult GProgressListener2::Pause (void)
{
	nsresult rv;

	if (mCanPause && !mIsPaused)
	{
		rv = mRequest->Suspend ();
		if (NS_SUCCEEDED (rv))
		{
			mIsPaused = PR_TRUE;
		}
	}
	else
	{
		rv = NS_ERROR_FAILURE;
	}

	return rv;
}

nsresult GProgressListener2::Resume (void)
{
	nsresult rv;

	if (mCanPause && mIsPaused)
	{
		rv = mRequest->Resume ();
		if (NS_SUCCEEDED (rv))
		{
			mIsPaused = PR_FALSE;
		}
	}
	else
	{
		rv = NS_ERROR_FAILURE;
	}

	return rv;
}

nsresult GProgressListener2::Abort (void)
{
	mAction = ACTION_NONE;

	if (mIsPaused)
	{
		Resume ();
	}

	if (mPersist || mLauncher)
	{
		Cleanup();
	}

	if (mObserver)
	{
		mObserver->Observe(NS_ISUPPORTS_CAST(nsIProgressDialog*, this),
					     "oncancel", nsnull);
		OnStateChange(nsnull, nsnull,
			      nsIWebProgressListener::STATE_STOP, 0);
	}

	if (mPersist)
	{
		return mPersist->CancelSave ();
	}

	if (mLauncher)
	{
		return mLauncher->Cancel ();
	}
	else
	{
		return NS_ERROR_FAILURE;
	}
}

PRBool GProgressListener2::CanPause (void)
{
	return (mCanPause && !mIsPaused);
}

PRBool GProgressListener2::CanResume (void)
{
	return (mCanPause && mIsPaused);
}

/*
 * Global dialog initialisation
 */
void
downloader_dialog_init (void)
{
	GladeXML *gxml;
	GtkCList *clist;

	/* allocate */
	dialog = g_new0 (DownloadDialogWidgets, 1);
	g_assert (dialog != NULL);

	/* build the interface */
	gxml = glade_widget_new ("galeon.glade", "download_manager_dialog",
				 &(dialog->window), dialog);
	
	/* lookup needed widgets */
	dialog->clist = glade_xml_get_widget (gxml, "clist");
	dialog->details_frame = glade_xml_get_widget (gxml, "details_frame");
	dialog->details_location =
		glade_xml_get_widget (gxml, "details_location");
	dialog->details_file =
		glade_xml_get_widget (gxml, "details_file");
	dialog->details_status =
		glade_xml_get_widget (gxml, "details_status");
	dialog->details_elapsed =
		glade_xml_get_widget (gxml, "details_elapsed");
	dialog->details_remaining =
		glade_xml_get_widget (gxml, "details_remaining");
	dialog->details_progress =
		glade_xml_get_widget (gxml, "details_progress");
	dialog->keep_open_check = glade_xml_get_widget (gxml, "keep_open_check");
	dialog->details_button = glade_xml_get_widget (gxml, "details_button");
	dialog->pause_button = glade_xml_get_widget (gxml, "pause_button");
	dialog->resume_button = glade_xml_get_widget (gxml, "resume_button");
	dialog->abort_button = glade_xml_get_widget (gxml, "abort_button");

	/* dispense with XML interface */
	gtk_object_unref (GTK_OBJECT (gxml));
	
	/* fixup limitations of glade */
	clist = GTK_CLIST (dialog->clist);
	gtk_clist_set_column_justification (clist, COL_PERCENT, 
					    GTK_JUSTIFY_RIGHT);
	gtk_clist_set_column_justification (clist, COL_SPEED, 
					    GTK_JUSTIFY_RIGHT);
	gtk_clist_set_column_justification (clist, COL_RECEIVED, 
					    GTK_JUSTIFY_RIGHT);
	gtk_clist_set_column_justification (clist, COL_SIZE, 
					    GTK_JUSTIFY_RIGHT);
	gtk_clist_set_column_justification (clist, COL_ELAPSED, 
					    GTK_JUSTIFY_RIGHT);
	gtk_clist_set_column_justification (clist, COL_REMAINING, 
					    GTK_JUSTIFY_RIGHT);

	/* init details hash */
	details_hash = g_hash_table_new (g_direct_hash, g_direct_equal);

	/* setup keep open check */
	gtk_toggle_button_set_active
		(GTK_TOGGLE_BUTTON (dialog->keep_open_check),
		 eel_gconf_get_boolean (CONF_DOWNLOADING_KEEP_OPEN));
}

//////////////////////////////////////////////////////////////////////////////
// begin ProgressDialog callbacks.
//////////////////////////////////////////////////////////////////////////////

void 
download_dialog_pause_cb (GtkButton *button, DownloadDialogWidgets *widgets)
{
	GtkCList *clist = GTK_CLIST (widgets->clist);
	GProgressListener2 *Progress;
	GList *l;
	int row;

	/* iterate over selection */
	for (l = clist->selection; l != NULL; l = g_list_next (l))
	{
		DetailsInfo *dinfo;
			
		/* get this row */
		row = GPOINTER_TO_INT (l->data);
		Progress = (GProgressListener2 *)
			gtk_clist_get_row_data (clist, row);
		dinfo = (DetailsInfo *)
			g_hash_table_lookup (details_hash, Progress);

		/* pause it */
		if (NS_SUCCEEDED (Progress->Pause ()))
		{
			gtk_clist_set_text (clist, row, COL_SPEED, 
 					    _("Paused"));
			dinfo->status = g_strdup (_("Paused"));
		}
	}

	/* update buttons */
	selection_changed (widgets);
}

void 
download_dialog_resume_cb (GtkButton *button, DownloadDialogWidgets *widgets)
{
	GtkCList *clist = GTK_CLIST (widgets->clist);
	GProgressListener2 *Progress;
	GList *l;
	int row;

	/* iterate over selection */
	for (l = clist->selection; l != NULL; l = g_list_next (l))
	{
		DetailsInfo *dinfo;

		/* get this row */
		row = GPOINTER_TO_INT (l->data);
		Progress = (GProgressListener2 *)
			gtk_clist_get_row_data (clist, row);
		dinfo = (DetailsInfo *)
			g_hash_table_lookup (details_hash, Progress);

		/* resume it */
		if (NS_SUCCEEDED (Progress->Resume ()))
		{
			gtk_clist_set_text (clist, row, COL_SPEED, 
					    _("Resuming..."));
			dinfo->status = g_strdup (_("Resuming..."));
		}
	}

	/* update buttons */
	selection_changed (widgets);
}

void
download_dialog_abort_cb (GtkButton *button, DownloadDialogWidgets *widgets)
{
	GtkCList *clist = GTK_CLIST (widgets->clist);
	GProgressListener2 *Progress;
	GList *l;
	int row;

	l = clist->selection;

	/* iterate over selection */
	while (l != NULL)
	{
		/* get this row */
		row = GPOINTER_TO_INT (l->data);
		Progress = (GProgressListener2 *)
			gtk_clist_get_row_data (clist, row);

		l = l->next;

		/* abort it */
		Progress->Abort ();
	}

	/* update buttons */
	selection_changed (widgets);
}

void
download_clist_select_row_cb (GtkCList *clist, gint row, gint column,
			      GdkEvent *ev, DownloadDialogWidgets *widgets)
{
	selection_changed (widgets);
}

void
download_clist_unselect_row_cb (GtkCList *clist, gint row, gint column,
				GdkEvent *ev, DownloadDialogWidgets *widgets)
{
	selection_changed (widgets);
}

gboolean
download_clist_button_press_event_cb (GtkCList *clist,
			              GdkEventButton *event,
				      DownloadDialogWidgets *widgets)
{
	GProgressListener2 *Progress;
	DetailsInfo *dinfo;
	gint r, c;

	/* get selection */
	gtk_clist_get_selection_info (clist, (gint) event->x, (gint) event->y,
				      &r, &c);
	Progress = (GProgressListener2 *)
		gtk_clist_get_row_data (clist, r);
	dinfo = (DetailsInfo *)
		g_hash_table_lookup (details_hash, Progress);

	/* contextmenu */
	if (event->button == 3)
	{
		GtkWidget *menu;
		gint action;

		/* context menu */
		static GnomeUIInfo menu_uiinfo[] =
		{
			GNOMEUIINFO_ITEM_STOCK (N_("Copy source location"),
						NULL, NULL, GNOME_STOCK_MENU_COPY),
			GNOMEUIINFO_ITEM_STOCK (N_("Copy destination location"),
						NULL, NULL, GNOME_STOCK_MENU_COPY),
			GNOMEUIINFO_END
		};
	
		/* show context menu */
		menu = gnome_popup_menu_new (menu_uiinfo);
		
		/*accout for the case where nothing is selected */
		if(!dinfo)
		{
			gtk_widget_set_sensitive (menu_uiinfo[0].widget, FALSE);
			gtk_widget_set_sensitive (menu_uiinfo[1].widget, FALSE);
		}
		
		action = gnome_popup_menu_do_popup_modal 
			(GTK_WIDGET (menu), NULL, NULL, event, NULL);

		/* do appropiate action */
		switch (action)
		{
		case 0:
			embed_copy_text_to_clipboard (dinfo->location,
					   	      (GaleonEmbed *)
						      (all_embeds->data));
			break;
		case 1:
			embed_copy_text_to_clipboard (dinfo->file,
						      (GaleonEmbed *)
						      (all_embeds->data));
			break;
		default:
			break;
		}
		/* destroy the popup menu */
		gtk_widget_unref (menu);
		
		return TRUE;
	}
	return FALSE;
}

gboolean
download_dialog_delete_cb (GtkWidget *window, GdkEventAny *event,
			   DownloadDialogWidgets *widgets)
{
	GtkCList *clist = GTK_CLIST (widgets->clist);
	GProgressListener2 *Progress;
	GtkWidget *dialog;
	gboolean choice;

	/* if no downloads left, close window */
	if (clist->rows == 0)
	{
		gtk_widget_hide (widgets->window);
		return TRUE;
	}

	/* build question dialog */
	dialog = gnome_question_dialog_modal_parented
		(_("Cancel all pending downloads?"), 
		 NULL, NULL, GTK_WINDOW (window));

	/* run it */
	choice = gnome_dialog_run_and_close (GNOME_DIALOG (dialog));

	/* do the appropriate thing */
	if (!choice)
	{
		/* iterate over all rows */
		while (clist->rows != 0)
		{
			/* get this row */
			Progress = (GProgressListener2 *)
				gtk_clist_get_row_data (clist, 0);

			/* abort it */
			Progress->Abort ();
		}
	}

	/* done, but never actually delete the dialog! */
	return TRUE;
}

void
download_dialog_details_cb (GtkToggleButton *button, 
			    DownloadDialogWidgets *widgets)
{
	if (gtk_toggle_button_get_active (button))
	{
		if (!GTK_CLIST (widgets->clist)->selection)
		{
			gtk_toggle_button_set_active (button, FALSE);
			return;
		}
		gtk_widget_show (GTK_WIDGET (widgets->details_frame));
		gtk_object_set_data (GTK_OBJECT (widgets->details_frame),
				     "row",
				     GTK_CLIST (widgets->clist)->selection->data);
		selection_changed (widgets);
	}
	else
	{
		gtk_widget_hide (GTK_WIDGET (widgets->details_frame));
		gtk_object_set_data (GTK_OBJECT (widgets->details_frame),
				     "row", GINT_TO_POINTER (-1));
	}
}

void
download_dialog_keep_open_cb (GtkToggleButton *button, 
			      DownloadDialogWidgets *widgets)
{
	eel_gconf_set_boolean (CONF_DOWNLOADING_KEEP_OPEN, button->active);
}

static void
selection_changed (DownloadDialogWidgets *widgets)
{
	GtkCList *clist = GTK_CLIST (widgets->clist);
	GProgressListener2 *Progress;
	gboolean can_pause, can_resume;
	DetailsInfo *dinfo;
	GList *l;
	int row;

	/* initial conditions */
	can_pause = can_resume = FALSE;

	/* iterate over selection */
	for (l = clist->selection; l != NULL; l = g_list_next (l))
	{
                /* get this row */
		row = GPOINTER_TO_INT (l->data);
		Progress = (GProgressListener2 *)
			gtk_clist_get_row_data (clist, row);

		/* test it */
		can_pause |= (Progress->CanPause ());
		can_resume |= (Progress->CanResume ());
	}

	/* setup buttons */
	gtk_widget_set_sensitive (widgets->pause_button, can_pause);
	gtk_widget_set_sensitive (widgets->resume_button, can_resume);
	gtk_widget_set_sensitive (widgets->abort_button, 
				  clist->selection != NULL);

	/* update details frame */
	gtk_widget_set_sensitive (widgets->details_button,
				  (clist->selection != NULL));
	if (!clist->selection) return;

	gtk_object_set_data (GTK_OBJECT (widgets->details_frame),
			     "row", clist->selection->data);
	
	Progress = (GProgressListener2 *)
		gtk_clist_get_row_data (clist,
				GPOINTER_TO_INT (clist->selection->data));
	dinfo = (DetailsInfo *)
		g_hash_table_lookup (details_hash, Progress);
	gtk_entry_set_text (GTK_ENTRY (widgets->details_location),
			    dinfo->location);
	gtk_entry_set_text (GTK_ENTRY (widgets->details_file),
			    dinfo->file);
	gtk_label_set_text (GTK_LABEL (widgets->details_status),
			    dinfo->status);
	gtk_label_set_text (GTK_LABEL (widgets->details_elapsed),
			    dinfo->elapsed);
	gtk_label_set_text (GTK_LABEL (widgets->details_remaining),
			    dinfo->remaining);
	if (dinfo->progress >= 0 && dinfo->progress <= 1.0)
	{
		gtk_progress_set_percentage
			(GTK_PROGRESS (widgets->details_progress),
			 dinfo->progress);
	}
	else
	{
		gtk_progress_set_format_string
			(GTK_PROGRESS (widgets->details_progress),
			 "?? %%");
	}
}

static gboolean
timeout_listener (GProgressListener2 *Progress)
{
	/* a timeout of 2 minutes */
	if (g_timer_elapsed (Progress->mTimer, NULL) >= 120)
	{
		g_timer_destroy (Progress->mTimer);
		Progress->Abort ();
		return FALSE;
	}

	return TRUE;
}
