/*
 *  Copyright (C) 2001 Philip Langdale
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* Things to be aware of:
 *
 * This filepicker, like the mozilla one, does not make an attempt
 * to verify the validity of the initial directory you pass it.
 * It does check that the user doesn't give it a garbage path
 * during use, but it is the caller's responsibility to give a
 * sensible initial path.
 *
 * At the current moment, we instantiate the filepicker directly
 * in our contenthandler where there is path verification code
 * and else where through our C wrapper, which also does verification.
 * If, at a future date, you need to instantiate filepicker without
 * using the C wrapper, please verify the initial path. See
 * ContentHandler for a way to do this.
 */

#include "galeon.h"
#include "mozilla.h"
#include "dialog.h"
#include "misc_string.h"
#include "eel-gconf-extensions.h"

#include <gtk/gtkmain.h>
#include <gtk/gtkcheckbutton.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkfilesel.h>
#include <gtk/gtkhbbox.h>
#include <gtk/gtkoptionmenu.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-messagebox.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnomeui/gnome-stock.h>

#include "nsCRT.h"
#include "nsCOMPtr.h"
#include "nsISupportsArray.h"
#include "nsIServiceManager.h"

#include "nsString.h"
#include "nsXPIDLString.h"
#include "nsIPrefService.h"
#include "nsIURI.h"
#include "nsIFileURL.h"
#include "nsIChannel.h"
#include "nsIFileChannel.h"
#include "nsNetCID.h"
#include "nsILocalFile.h"
#include "nsIDOMWindow.h"
#include "nsIDOMWindowInternal.h"
#include "nsIPromptService.h"
#include "nsReadableUtils.h"

#include <libgnome/gnome-util.h>

#include "FilePicker.h"


/* Implementation file */
NS_IMPL_ISUPPORTS1(GFilePicker, nsIFilePicker)

GFilePicker::GFilePicker (PRBool showContentCheck, FileFormat *fileFormats) : mSaveContent(PR_FALSE)
{
	NS_INIT_ISUPPORTS();
	/* member initializers and constructor code */

	mShowContentCheck = showContentCheck;
	mFileFormats = fileFormats;
	mFile = do_CreateInstance (NS_LOCAL_FILE_CONTRACTID);
	mDisplayDirectory = do_CreateInstance (NS_LOCAL_FILE_CONTRACTID);
	mDisplayDirectory->InitWithNativePath(nsDependentCString(g_get_home_dir()));
}

GFilePicker::~GFilePicker()
{
  /* destructor code */
}

////////////////////////////////////////////////////////////////////////////////
// begin nsIFilePicker impl
////////////////////////////////////////////////////////////////////////////////

/* void init (in nsIDOMWindowInternal parent, in wstring title, in short mode); */
NS_IMETHODIMP GFilePicker::Init(nsIDOMWindowInternal *parent, 
				 const PRUnichar *title, PRInt16 mode)
{
	mParentInternal = parent;
	mParent = do_QueryInterface (mParentInternal);
	mParentWidget = mozilla_find_gtk_parent (mParent);
	
	mTitle = title;
	mMode = mode;
	
	return NS_OK;
}

/* void appendFilters (in long filterMask); */
NS_IMETHODIMP GFilePicker::AppendFilters(PRInt32 filterMask)
{
	//This function cannot be implemented due to the crippled
	//nature of GtkFileSelection, but NS_ERROR_NOT_IMPLEMENTED
	//is interpreted as a terminal error by some callers.
	return NS_OK;
}

/* void appendFilter (in wstring title, in wstring filter); */
NS_IMETHODIMP GFilePicker::AppendFilter(const PRUnichar *title,
					const PRUnichar *filter)
{
	//GtkFileSelection is crippled, so we can't provide a short-list
	//of filters to choose from. We provide minimal functionality
	//by using the most recent AppendFilter call as the active filter.
	mFilter = filter;
	return NS_OK;
}

/* attribute long filterIndex; */
NS_IMETHODIMP GFilePicker::GetFilterIndex(PRInt32 *aFilterIndex)
{
	return NS_ERROR_NOT_IMPLEMENTED;
}
NS_IMETHODIMP GFilePicker::SetFilterIndex(PRInt32 aFilterIndex)
{
	return NS_OK;
}

/* attribute wstring defaultString; */
NS_IMETHODIMP GFilePicker::GetDefaultString(PRUnichar * *aDefaultString)
{
	*aDefaultString = ToNewUnicode (mDefaultString);
	return NS_OK;
}
NS_IMETHODIMP GFilePicker::SetDefaultString(const PRUnichar * aDefaultString)
{
	if (aDefaultString)
		mDefaultString = aDefaultString;
	else
		mDefaultString = NS_LITERAL_STRING ("");
	return NS_OK;
}

/* attribute wstring defaultExtension; */
// Again, due to the crippled file selector, we can't really
// do anything here.
NS_IMETHODIMP GFilePicker::GetDefaultExtension(PRUnichar * *aDefaultExtension)
{
    return NS_ERROR_NOT_IMPLEMENTED;
}

NS_IMETHODIMP GFilePicker::SetDefaultExtension(const PRUnichar * aDefaultExtension)
{
    return NS_OK;
}

/* attribute nsILocalFile displayDirectory; */
NS_IMETHODIMP GFilePicker::GetDisplayDirectory(nsILocalFile * *aDisplayDirectory)
{
	NS_IF_ADDREF (*aDisplayDirectory = mDisplayDirectory);
	return NS_OK;
}
NS_IMETHODIMP GFilePicker::SetDisplayDirectory(nsILocalFile * aDisplayDirectory)
{
	mDisplayDirectory = aDisplayDirectory;
	return NS_OK;
}

/* readonly attribute nsILocalFile file; */
NS_IMETHODIMP GFilePicker::GetFile(nsILocalFile * *aFile)
{
	NS_IF_ADDREF (*aFile = mFile);
	return NS_OK;
}

/* readonly attribute nsIFileURL fileURL; */
NS_IMETHODIMP GFilePicker::GetFileURL(nsIFileURL * *aFileURL)
{
	nsCOMPtr<nsIFileURL> fileURL = 
		do_CreateInstance (NS_STANDARDURL_CONTRACTID);
	fileURL->SetFile (mFile);
	NS_IF_ADDREF (*aFileURL = fileURL);
	return NS_OK;
}

#if MOZILLA_SNAPSHOT > 2
/* readonly attribute nsISimpleEnumerator files; */
NS_IMETHODIMP GFilePicker::GetFiles(nsISimpleEnumerator * *aFiles)
{
    return NS_ERROR_NOT_IMPLEMENTED;
}
#endif

/* short show (); */
NS_IMETHODIMP GFilePicker::Show(PRInt16 *_retval)
{
	char *cTitle = mozilla_unicode_to_locale (mTitle.get());

	mFileSelector = gtk_file_selection_new (cTitle);
	g_free (cTitle);

	char *cFileName;
	if (mMode == nsIFilePicker::modeGetFolder)
	{
		cFileName = NULL;
	}
	else
	{
		cFileName = mozilla_unicode_to_locale (mDefaultString.get());
	}

	nsCAutoString cDirName;
	mDisplayDirectory->GetNativePath (cDirName);

	const nsCAutoString cFullPath(cDirName + NS_LITERAL_CSTRING("/") +
				      nsDependentCString(cFileName ? cFileName : ""));
	g_free (cFileName);

	gtk_file_selection_set_filename (GTK_FILE_SELECTION(mFileSelector),
				 	 cFullPath.get());

	if (!mFilter.IsEmpty())
	{
		char *cFilter = mozilla_unicode_to_locale (mFilter.get());
		gtk_file_selection_complete (GTK_FILE_SELECTION(mFileSelector),
					     cFilter);
		g_free (cFilter);
	}

	if (mParentWidget)
		gtk_window_set_transient_for (GTK_WINDOW(mFileSelector),
					      GTK_WINDOW(mParentWidget));

	gtk_signal_connect (GTK_OBJECT(GTK_FILE_SELECTION(mFileSelector)
			    ->ok_button),
			    "clicked",
			    GTK_SIGNAL_FUNC(filePicker_save_as_ok_cb),
			    (gpointer)this);
	gtk_signal_connect (GTK_OBJECT(GTK_FILE_SELECTION(mFileSelector)
			    ->cancel_button),
			    "clicked",
			    GTK_SIGNAL_FUNC(filePicker_save_as_cancel_cb),
			    (gpointer)this);
	gtk_signal_connect (GTK_OBJECT(GTK_FILE_SELECTION(mFileSelector)),
			    "delete-event",
			    GTK_SIGNAL_FUNC(filePicker_save_as_close_cb),
			    (gpointer)this);
	
	gtk_window_set_wmclass (GTK_WINDOW (mFileSelector), "dialog_filesel",
				"galeon_browser");

	if (mShowContentCheck)
	{
		GtkWidget *bbox = gtk_hbutton_box_new ();
		gtk_button_box_set_layout (GTK_BUTTON_BOX(bbox),
					   GTK_BUTTONBOX_END);
		gtk_button_box_set_spacing (GTK_BUTTON_BOX(bbox), 0);
        	gtk_box_pack_end (GTK_BOX(
        		GTK_FILE_SELECTION(mFileSelector)->action_area),
				  bbox, TRUE, TRUE, 0);

		GtkWidget *saveContent = 
			gtk_check_button_new_with_label(_("Save with content"));
		gtk_signal_connect(GTK_OBJECT(saveContent),
				   "clicked",
				   GTK_SIGNAL_FUNC(filePicker_save_content_cb),
				   (gpointer)this);

		gtk_box_pack_start(GTK_BOX(bbox), saveContent,
				   FALSE, FALSE, 0);

		gtk_widget_show_all(bbox);
	}

	if (mFileFormats)
	{
		mFormatChooser = gtk_option_menu_new ();
		GtkMenu *options = GTK_MENU (gtk_menu_new ());

		FileFormat *current = mFileFormats;
		while (current->description != NULL)
		{
			/* FIXME: the label should include the extensions too */
			gchar *label = current->description;
			GtkWidget *item = gtk_menu_item_new_with_label (label);
			gtk_widget_show (item);
			gtk_menu_append (options, item);
			current++;
		}
		gtk_option_menu_set_menu (GTK_OPTION_MENU (mFormatChooser), GTK_WIDGET (options));
		gtk_widget_show (mFormatChooser);
		gtk_box_pack_start (GTK_BOX (GTK_FILE_SELECTION (mFileSelector)->action_area), mFormatChooser,
				    FALSE, TRUE, 0);
	}
	else
	{
		mFormatChooser = NULL;
	}

	if (mMode == nsIFilePicker::modeGetFolder)
	{
		gtk_widget_set_sensitive (GTK_FILE_SELECTION(mFileSelector)
					  ->file_list, FALSE);
	}

	gtk_window_set_modal (GTK_WINDOW(mFileSelector), TRUE);

	mFileSelectorClosed = PR_FALSE;

	dialog_set_parent (mFileSelector, NULL);

	while (mFileSelectorClosed == PR_FALSE)
	{
		gtk_main_iteration();
	}

	if (mFileSelectorOk == PR_TRUE)
	{
		*_retval = nsIFilePicker::returnOK;
		if (mSaveContent == PR_TRUE)
			*_retval = returnOKSaveContent;
	}
	else
	{
		*_retval = nsIFilePicker::returnCancel;
	}

	return (mFileSelectorOk == PR_TRUE) ? NS_OK : NS_ERROR_FAILURE;
}

////////////////////////////////////////////////////////////////////////////////
// begin local public methods impl
////////////////////////////////////////////////////////////////////////////////

NS_IMETHODIMP GFilePicker::InitWithGtkWidget (GtkWidget *parentWidget, 
					      const char *title, PRInt16 mode)
{
	mParentWidget = parentWidget;
	
	PRUnichar *uTitle = mozilla_locale_to_unicode (title);
	mTitle = uTitle;
	g_free (uTitle);

	mMode = mode;

	mFile = do_CreateInstance (NS_LOCAL_FILE_CONTRACTID);

	return NS_OK;
}

NS_IMETHODIMP GFilePicker::SanityCheck (PRBool *retval)
{
	*retval = PR_TRUE;

	nsresult rv;
	PRBool dirExists, fileExists = PR_TRUE;

	if (mDisplayDirectory)
	{
		rv = mDisplayDirectory->Exists (&dirExists);
		g_return_val_if_fail (NS_SUCCEEDED(rv), rv);
	}
	else
	{
		dirExists = PR_FALSE;
	}

	if (mMode != nsIFilePicker::modeGetFolder)
	{
		rv = mFile->Exists (&fileExists);
		g_return_val_if_fail (NS_SUCCEEDED(rv), rv);
	}
	
	if (mMode == nsIFilePicker::modeSave && !fileExists)
	{
		return NS_OK;
	}
	
	if (!dirExists || !fileExists)
	{
		GtkWidget *errorDialog = gnome_message_box_new (
				_("The specified path does not exist."),
				GNOME_MESSAGE_BOX_ERROR,
				GNOME_STOCK_BUTTON_OK,
				NULL);
		gtk_window_set_wmclass (GTK_WINDOW (errorDialog),
					"dialog_alert",
					"galeon_browser");
		dialog_set_parent (errorDialog, mFileSelector);
		gtk_window_set_modal (GTK_WINDOW(errorDialog), TRUE);
		gnome_dialog_run_and_close (GNOME_DIALOG(errorDialog));
		*retval = PR_FALSE;
		return NS_OK;
	}

	PRBool correctType;
	char *errorText;
	if (mMode == nsIFilePicker::modeGetFolder)
	{
		rv = mDisplayDirectory->IsDirectory (&correctType);
		g_return_val_if_fail (NS_SUCCEEDED(rv), rv);
		errorText = g_strdup (_("A file was selected when a "
					"folder was expected."));
	}
	else
	{
		rv = mFile->IsFile (&correctType);
		g_return_val_if_fail (NS_SUCCEEDED(rv), rv);
		errorText = g_strdup (_("A folder was selected when a "
				        "file was expected."));
	}
	
	if(!correctType)
	{
		GtkWidget *errorDialog = gnome_message_box_new (
				errorText,
				GNOME_MESSAGE_BOX_ERROR,
				GNOME_STOCK_BUTTON_OK,
				NULL);
		gtk_window_set_wmclass (GTK_WINDOW (errorDialog),
					"dialog_alert",
					"galeon_browser");
		dialog_set_parent (errorDialog, mFileSelector);
		gtk_window_set_modal (GTK_WINDOW(errorDialog), TRUE);
		gnome_dialog_run_and_close (GNOME_DIALOG(errorDialog));
		*retval = PR_FALSE;
	}
	g_free (errorText);

	return NS_OK;
}

////////////////////////////////////////////////////////////////////////////////
// begin local private methods impl
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// begin FileSelector callbacks.
////////////////////////////////////////////////////////////////////////////////

void 
filePicker_save_as_ok_cb (GtkButton *button, GFilePicker *aFilePicker)
{
	nsresult rv;

	gchar *aFileName = 
		misc_string_simplify_uri (
			gtk_file_selection_get_filename (
				GTK_FILE_SELECTION (
						aFilePicker->mFileSelector)));
	if (!aFileName || strlen(aFileName) == 0) return;

	if (aFilePicker->mMode == nsIFilePicker::modeSave)
	{
		if (!dialog_check_overwrite_file (aFileName, 
						aFilePicker->mFileSelector))
		{
			g_free (aFileName);
			return;
		}
	}

	aFilePicker->mFile->InitWithNativePath (nsDependentCString(aFileName));

	if (aFilePicker->mMode == nsIFilePicker::modeGetFolder)
	{
		aFilePicker->mDisplayDirectory->
			InitWithNativePath (nsDependentCString(aFileName));
		aFilePicker->mDefaultString = NS_LITERAL_STRING("");
	}
	else
	{
		nsCOMPtr<nsIFile> aDirectory;
		aFilePicker->mFile->GetParent (getter_AddRefs(aDirectory));
		aFilePicker->mDisplayDirectory = do_QueryInterface (aDirectory);
		aFilePicker->mFile->GetLeafName (aFilePicker->mDefaultString);
	}

	g_free (aFileName);

	PRBool passesSanityCheck;
	rv = aFilePicker->SanityCheck (&passesSanityCheck);
	if (NS_SUCCEEDED(rv) && !passesSanityCheck) return;

	aFilePicker->mFileSelectorClosed = PR_TRUE;
	aFilePicker->mFileSelectorOk = PR_TRUE;

	if (aFilePicker->mFormatChooser)
	{
		gint i = 0;		
		GtkWidget *menu = gtk_option_menu_get_menu 
			(GTK_OPTION_MENU (aFilePicker->mFormatChooser));
		GList *iterator = GTK_MENU_SHELL (menu)->children;
		GtkWidget *selected = gtk_menu_get_active (GTK_MENU(menu));

		while (iterator)
		{
			if (iterator->data == selected) 
			{
				aFilePicker->mSelectedFileFormat = i;
				break;
			}
			iterator = iterator->next;
			i++;
		}
	}

	gtk_widget_destroy (GTK_WIDGET(aFilePicker->mFileSelector));
}

void
filePicker_save_as_cancel_cb (GtkButton *button, GFilePicker *aFilePicker)
{
	aFilePicker->mFileSelectorClosed = PR_TRUE;
	aFilePicker->mFileSelectorOk = PR_FALSE;

	gtk_widget_destroy (GTK_WIDGET(aFilePicker->mFileSelector));
}

gboolean
filePicker_save_as_close_cb (GtkWidget *dialog, GdkEvent *event,
			     GFilePicker *aFilePicker)
{
	aFilePicker->mFileSelectorClosed = PR_TRUE;
	aFilePicker->mFileSelectorOk = PR_FALSE;

	return FALSE;
}

void filePicker_save_content_cb (GtkToggleButton *button,
				 GFilePicker *aFilePicker)
{
	aFilePicker->mSaveContent = gtk_toggle_button_get_active (button) ?
				    PR_TRUE : PR_FALSE;
}	

////////////////////////////////////////////////////////////////////////////////
/**
 * show_file_picker: Shows a file picker. Can be configured to select a
 * file or a directory.
 * @parentWidget: Parent Widget for file picker.
 * @title: Title for file picker.
 * @directory: Initial directory to start in. 
 * @file: Initial filename to show in filepicker.
 * @mode: Mode to run filepicker in (modeOpen, modeSave, modeGetFolder)
 * @ret_fullpath: On a successful return, will hold the full path to selected
 *		file or directory.
 * @file_formats: an array of FileFormat structures to fill the format chooser
 *              optionmenu. NULL if not needed. The last item must have 
 *              description == NULL.
 * @ret_file_format: where to store the index of the format selected (can be
 *              NULL)
 * returns: TRUE for success, FALSE for failure.
 */
////////////////////////////////////////////////////////////////////////////////

extern "C" gboolean
show_file_picker (GtkWidget *parentWidget, const char* title,
		  const char* directory, const char* file, FilePickerMode mode,
		  char * *ret_fullpath, gboolean* ret_save_content, 
		  FileFormat *file_formats, gint *ret_file_format)
{
	PRBool showContentCheck;
	gchar *expanded_directory;

	if (ret_save_content == NULL)
		showContentCheck = PR_FALSE;
	else
		showContentCheck = PR_TRUE;

	GFilePicker *aFilePicker = new GFilePicker (showContentCheck, 
						    file_formats);

	PRUnichar *uDefaultString = mozilla_locale_to_unicode (file);

	/* expand the tilde, if there is one */
	expanded_directory = misc_string_expand_home_dir (directory);

	/* make sure the directory exists, and use the home directory
	 * otherwise */
	if (!expanded_directory ||
	    !g_file_test (expanded_directory, G_FILE_TEST_ISDIR))
	{
		if (expanded_directory) g_free (expanded_directory);
		expanded_directory = g_strdup (g_get_home_dir());
	}

	nsCOMPtr<nsILocalFile> aDir = 
				do_CreateInstance (NS_LOCAL_FILE_CONTRACTID);
	aDir->InitWithNativePath (nsDependentCString(expanded_directory));
	g_free (expanded_directory);

	aFilePicker->InitWithGtkWidget (parentWidget, title, mode);
	aFilePicker->SetDefaultString (uDefaultString);
	aFilePicker->SetDisplayDirectory (aDir);
	
	PRInt16 retval;
	aFilePicker->Show (&retval);

	if (ret_save_content != NULL)
	{
		if (retval == GFilePicker::returnOKSaveContent)
			*ret_save_content = TRUE;
		else
			*ret_save_content = FALSE;
	}
	if (ret_file_format != NULL)
	{
		*ret_file_format = aFilePicker->mSelectedFileFormat;
	}

	if (retval == nsIFilePicker::returnCancel)
	{
		delete aFilePicker;
		return FALSE;
	}
	else
	{
		if (*ret_fullpath)
			g_free (*ret_fullpath);
		nsCOMPtr<nsILocalFile> aFile;
		aFilePicker->GetFile (getter_AddRefs(aFile));
		nsCAutoString tempFullPathStr;
		aFile->GetNativePath (tempFullPathStr);
		*ret_fullpath = g_strdup (tempFullPathStr.get());
		delete aFilePicker;
		return TRUE;
	}
}
