/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 */

#include "galeon.h"
#include "window.h"
#include "dialog.h"
#include "misc_string.h"
#include "prefs.h"
#include "glade.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <gtk/gtkpaned.h>
#include <gtk/gtkwindow.h>
#include <gtk/gtkclist.h>
#include <gtk/gtkentry.h>
#include <gtk/gtktogglebutton.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-util.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-triggers.h>
#include <libgnomeui/gnome-dialog.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnomeui/gnome-winhints.h>
#include <libgnomeui/gnome-app.h>

static void dialog_ask_question_cb (gint reply, gpointer data);

/** Shared var for the next two functions */
static gboolean misc_ask_question_ret = FALSE;

/**
 * Asks a yes/no question and returns a boolean
 */
gboolean 
dialog_ask_question (const gchar *question, GtkWidget *parent)
{
	GtkWidget *dialog;
	if (parent)
	{
		dialog = gnome_question_dialog_parented
			(question, dialog_ask_question_cb, NULL,
			 GTK_WINDOW (parent));
		dialog_set_parent (dialog, parent);
	}
	else
	{
		dialog = gnome_question_dialog (question,
						dialog_ask_question_cb, NULL);
	}

	gnome_dialog_run (GNOME_DIALOG (dialog));
	return misc_ask_question_ret;
}

static void
dialog_ask_question_cb (gint reply, gpointer data) {
	misc_ask_question_ret = !reply;
}

/**
 * dialog_check_overwrite_file: If file exists, ask whether to overwrite
 */
gboolean
dialog_check_overwrite_file (const gchar *fname, GtkWidget *parent)
{
	gchar *question, modified[256];
	int ret = TRUE;
	struct stat st;
	GtkWidget *dialog;

	if (g_file_test (fname, G_FILE_TEST_ISFILE | G_FILE_TEST_ISLINK))
	{
		stat (fname, &st);
		misc_string_store_full_time_in_string (st.st_mtime, modified);
		question = g_strdup_printf 
			(_("File %s will be overwritten.\n"
			   "(%li bytes, modified %s)\n"
			   "If you choose yes, the contents will be lost.\n\n"
			   "Do you want to continue?"), fname, 
			 (unsigned long) st.st_size, 
			 modified);
		ret = dialog_ask_question (question, parent);
		g_free (question);
	}
	else if (g_file_test (fname, G_FILE_TEST_ISDIR))
	{
		question = g_strdup_printf ("%s is a directory.\n" 
					    "Please enter a filename.", fname);
		dialog = gnome_error_dialog (question);
		gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
		ret = FALSE;
		g_free (question);
	}

	return ret;
}

/**
 * dialog_confirm_check: for simple yes/no dialogs, with an additional
 * check button
 */
gboolean
dialog_confirm_check (GtkWidget *parent, const gchar *dialogTitle,
		      const gchar *text, const gchar *checkMsg, 
		      gboolean *checkValue)
{
	GtkWidget *dialog;
	GtkWidget *label;
	GtkWidget *check_button;
	GladeXML *gxml;
	gint ret;

	/* build and show the dialog */
	gxml = glade_widget_new ("prompts.glade", "confirm_check_dialog", 
				 &dialog, NULL);
	label = glade_xml_get_widget (gxml, "label");
	check_button = glade_xml_get_widget (gxml, "check_button");
	gtk_object_unref (GTK_OBJECT (gxml));
	
	
	/* set dynamic attributes */
	gtk_window_set_title (GTK_WINDOW (dialog), 
			      (dialogTitle == NULL ? N_("Galeon") : 
			       dialogTitle));
	gtk_label_set_text (GTK_LABEL (label), text);
	gtk_label_set_text (GTK_LABEL (GTK_BIN (check_button)->child), 
			    checkMsg);

	if (checkValue == NULL)
	{
		gtk_widget_hide (GTK_WIDGET (check_button));
	}
	else
	{
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check_button),
					      *checkValue);
	}

	/* make a suitable sound */
	gnome_triggers_vdo ("", "question", NULL);

	/* run dialog and capture return values */
	dialog_set_parent (dialog, parent);
	ret = gnome_dialog_run_and_close (GNOME_DIALOG (dialog));

	/* handle return values */
	if (checkValue != NULL)
		*checkValue = gtk_toggle_button_get_active 
					(GTK_TOGGLE_BUTTON (check_button));

	/* done */
	gtk_widget_destroy (dialog);
	
	return (ret == 0);
}

/**
 * dialog_set_parent: do the best job possible of parenting a dialog to a 
 * window. Generally try to make sure it's on the right layer and for
 * true dialogs make sure we stay on top of the parent.
 */ 
void
dialog_set_parent (GtkWidget *dialog, GtkWidget *parent_window)
{
	GaleonWindow *window = NULL;

	/* won't get far if this is true... */
	if (dialog == NULL)
	{
		return;
	}

	/* find GaleonWindow */
	if (parent_window != NULL)
	{
		window = gtk_object_get_data
			(GTK_OBJECT (parent_window), "GaleonWindow");
	}

	/* check if we're in fullscreen mode and stacked above dock */
	if (window != NULL && window->fullscreen_active && 
	    eel_gconf_get_boolean (CONF_WINDOWS_FULLSCREEN_STACKING))
	{
		/* get to the top (must be shown first) */
		gtk_widget_show (dialog);
		gnome_win_hints_set_layer (dialog, WIN_LAYER_ABOVE_DOCK);
	}
	
	/* if it's not a toplevel app (like e.g. the bm editor) */
	if (!(GNOME_IS_APP (dialog)) && parent_window != NULL)
	{
		/* then make sure it's got transient parenting */
		gtk_window_set_transient_for (GTK_WINDOW (dialog),
					      GTK_WINDOW (parent_window));
	}

	/* finally show it and raise it to the top */
	gtk_widget_show (dialog);
	gdk_window_raise (dialog->window);
}
