/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef __bookmarks_h
#define __bookmarks_h

#include "galeon_types.h"
#include <gtk/gtkwindow.h>
#include <gtk/gtkmenuitem.h>
#include <gtk/gtkmenu.h>
#include <gtk/gtktooltips.h>
#include <gtk/gtkentry.h>

#define BOOKMARKS_MENU_MAX_LENGTH 40 /** Max length of a bookmark menu item */
#define BOOKMARKS_TOOLTIP_MAX_LENGTH 30 /** Max length of a bookmark url tooltip */

typedef struct _BookmarkItem             BookmarkItem;

/**
 * BookmarkType: various types of bookmarks
 */ 
typedef enum
{
	BM_FOLDER,
	BM_SITE,
	BM_SEPARATOR,
	BM_AUTOBOOKMARKS
} BookmarkType;

/**
 * BookmarkAddFlags: flags to specify special instructions to use when
 * adding bookmarks or folders
 */
typedef enum
{
	BOOKMARK_ADD_ALLOW_EXPAND_TITLE	= 1 << 0, /* if the title is determined
						     to be "short", append the
						     url */
	BOOKMARK_ADD_NEVER_PROMPT_TITLE	= 1 << 1, /* regardless of the user's
						     prefs, don't prompt for
						     the title */
	BOOKMARK_ADD_FORCE_PROMPT_TITLE	= 1 << 2  /* prompt for the title */
} BookmarkAddFlags;

/**
 * Forward declaration
 */
typedef struct _BookmarkTreeItem BookmarkTreeItem;

/**
 * BookmarkItem: a single entry in bookmarks
 * All the fields are read-only, you should modify them using the provided 
 * funtions
 */
struct _BookmarkItem
{
	/* the type of this bookmark */
	BookmarkType type;

	/* the user provided name of the bookmark */
	gchar *name;

	/* the site it points at */
	gchar *url;

	/* nickname used as a shorthand for this bookmark */
	gchar *nick;
	
	/* any user provided comments about the bookmark */
	gchar *notes; 

	/* smart url for smart bookmark */
	gchar *smarturl;
	
	/* the pixmap file used to represent it */
	gchar *pixmap_file;

	/* the in-memory binary data of the pixmap */
	PixmapData *pixmap_data;

	/* the list of children if this is a folder */
	GList *list;

	/* a handles to find the bookmark in the editor dialogs */
	/* this is a list of BookmarkTreeItem */
	GList *tree_items;

	/* parent folder (NULL for root bookmark) */
	BookmarkItem *parent; 

	/* if TRUE and this is a folder, create a toolbar from it */
	gboolean create_toolbar : 1; 

	/* if TRUE, we add this bookmark to the document context menu */
	gboolean create_context_menu : 1;

	/* if TRUE and this is a folder, then it is shown open in the editor */
	gboolean expanded : 1; 

	/* if not NULL, then this is an alias. The pointed bookmark can be an
	 * alias too */
	BookmarkItem *alias_of;

	/* Points to the first alias of this bookmark, if any. If the 
	 * bookmark is an alias, then it points to the next alias */
	BookmarkItem *alias;

	/* time info (if they are 0, then they should be ignored) */
	GTime time_added;
	GTime time_modified;
	GTime time_visited;

	/* toolbar style */
	ToolbarStyle toolbar_style;

	/* accel data */
	guint accel_key;
	GdkModifierType accel_mods;
};

#define BOOKMARK_ITEM_IS_FOLDER(b) \
            ((((BookmarkItem *)(b))->type == BM_FOLDER) \
             || (((BookmarkItem *)(b))->type == BM_AUTOBOOKMARKS))

/**
 * Funtions to modify a bookmark
 */
void bookmark_set_name (BookmarkItem *b, const gchar *val);
void bookmark_set_url (BookmarkItem *b, const gchar *val);
void bookmark_set_smarturl (BookmarkItem *b, const gchar *val);
void bookmark_set_nick (BookmarkItem *b, const gchar *val);
void bookmark_set_notes (BookmarkItem *b, const gchar *val);
void bookmark_set_pixmap (BookmarkItem *b, const gchar *val);
void bookmark_set_create_toolbar (BookmarkItem *b, gboolean val);
void bookmark_set_create_context_menu (BookmarkItem *b, gboolean val);
void bookmark_set_expanded (BookmarkItem *b, gboolean val);
void bookmark_set_time_added (BookmarkItem *b, GTime val);
void bookmark_set_time_modified (BookmarkItem *b, GTime val);
void bookmark_set_time_visited (BookmarkItem *b, GTime val);
void bookmark_set_toolbar_style (BookmarkItem *b, ToolbarStyle val);
void bookmark_set_accel (BookmarkItem *b, guint accel_key, 
			 GdkModifierType accel_mods);
void bookmark_add_child (BookmarkItem *b, BookmarkItem *c, gint position);
void bookmark_unparent (BookmarkItem *c);

/**
 * Shared variables
 */

/** The root of the bookmarks tree */
extern BookmarkItem *bookmarks_root;

/** The root of the temporary bookmark tree */
extern BookmarkItem *default_bookmarks_root;

/** Do we need to save the bookmarks? */
extern gboolean bookmarks_dirty;

/** Do we need to recreate the bookmarks toolbars? */
extern gboolean bookmarks_tb_dirty;

/** The bookmarks editors list */
extern GList *bookmarks_editors;


/**
 * Functions
 */
void bookmarks_init (void);
void bookmarks_exit (void);
BookmarkItem *bookmarks_new_bookmark (BookmarkType type, gboolean escape_name,
				      const char *name, const char *url, 
				      const char *nick, const char *notes, 
				      const char *pixmap_file);
BookmarkItem *bookmarks_new_alias (BookmarkItem *bookmark);
void bookmarks_free_bookmark (BookmarkItem *b);
void bookmarks_update_alias (BookmarkItem *b);
void bookmarks_set_visited (BookmarkItem *b);
BookmarkItem *bookmarks_find_real_bookmark (BookmarkItem *b);
BookmarkItem *bookmarks_copy_bookmark (BookmarkItem *b);
void bookmarks_add_completion_recursively (BookmarkItem *b);
void bookmarks_load (void);
void bookmarks_updated (void);
void bookmarks_smart_bm_dialog (GaleonEmbed *embed, GaleonEmbed **newembed,
				BookmarkItem *b, LinkState state);
void bookmarks_insert_bookmark (BookmarkItem *b, BookmarkItem *near,
				GtkCListDragPos insert_pos);
void bookmarks_remove_recursively (BookmarkItem *b);
void bookmarks_remove_several (GList *l);

BookmarkItem *bookmarks_add_bookmark (const gchar *name, const gchar *url,
				      BookmarkItem *parent,
				      GtkWindow *parent_window,
				      BookmarkAddFlags flags);
BookmarkItem *bookmarks_add_smart_bookmark (const gchar *name,
					    const gchar *url,
			      		    const gchar *smarturl,
					    const gchar *pixmap_file,
					    BookmarkItem *parent,
					    GtkWindow *parent_window,
					    BookmarkAddFlags flags);
void bookmarks_update_images (const gchar *changed_image);
BookmarkItem *bookmarks_add_folder (const gchar *name, BookmarkItem *parent,
				    GtkWindow *parent_window,
				    BookmarkAddFlags flags);
BookmarkItem *bookmarks_add_item (BookmarkType type, const gchar *name,
				  const gchar *url, const gchar *smarturl,
				  const gchar *pixmap_file,
				  BookmarkItem *parent,
				  GtkWindow *parent_window,
				  BookmarkAddFlags flags);
gchar *bookmarks_parse_nick (const gchar *text, BookmarkItem *root);
gchar *bookmarks_substitute_argument (BookmarkItem *b, const gchar *arguments);
gchar *bookmarks_item_to_string (BookmarkItem *root);
BookmarkItem *bookmarks_item_from_string (char *string);
void bookmarks_import_netscape_activate_cb (GtkMenuItem *menuitem,
					    GaleonWindow *window);
void bookmarks_import_mozilla_activate_cb (GtkMenuItem *menuitem,
					   GaleonWindow *window);
GList *bookmarks_get_selections_sorted (GtkCList *clist);
gchar *bookmarks_encode_smb_string (gchar *str);
void bookmarks_folder_open_all_items (GaleonEmbed *embed, BookmarkItem *bi,
				      gboolean new_window, gboolean reuse);
gboolean bookmarks_is_ancestor (BookmarkItem *a, BookmarkItem *b);
BookmarkItem *bookmarks_find_by_nick (const gchar *nick, BookmarkItem *root,
				      gboolean wantsmart);
BookmarkItem *bookmarks_find_by_url (const gchar *url, BookmarkItem *root);
void bookmarks_merge_trees (BookmarkItem *root1, BookmarkItem *root2);
void bookmarks_set_root (BookmarkItem *new_root);
void bookmarks_sort_norec (BookmarkItem *b, gboolean folders_first, 
			   gboolean update_edit);
void bookmarks_sort_rec (BookmarkItem *b, gboolean folders_first, 
			 gboolean update_edit);
void bookmarks_folder_open_in_myportal (GaleonEmbed *embed, BookmarkItem *bi,
					gboolean new_window, gboolean reuse);
void bookmarks_load_icons (void);
gboolean bookmarks_bookmark_exists (BookmarkItem *start, BookmarkItem *b);
gint bookmarks_count_tree (BookmarkItem *root);

/* bookmarks_menus.c */
void bookmarks_menu_create (GaleonWindow *window);
void bookmarks_menu_recreate (GaleonWindow *window);
/* FIXME: dont export the next two? */
GtkMenu *bookmarks_create_copy_menu (BookmarkItem *source);
void bookmarks_menu_create_recursively (BookmarkItem *bookmark, GtkMenu *menu, 
					GtkMenuItem *file_menuitem,
					GtkTooltips *tooltips, 
					gboolean options, 
					gboolean disable_tearoffs,
					gboolean lock_accels);
void bookmarks_menu_create_extra_items (GtkMenu *menu,
					BookmarkItem *bookmark,
					BookmarksExtraItemsLocation opts_loc);
void bookmarks_menu_create_base_extra_items (GtkMenu *menu,
					     BookmarkItem *bookmark);
void bookmarks_item_create_extra_items (GtkMenu *extra_menu, 
					BookmarkItem *bookmark);
void bookmarks_item_create_base_extra_items (GtkMenu *menu,
					     BookmarkItem *bookmark);
void bookmarks_menu_sync_accel_labels (void);


gboolean bookmarks_button_press_event_cb (GtkWidget *widget,
					  GdkEventButton *event,
					  BookmarkItem *bi);
gboolean bookmarks_button_release_event_cb (GtkWidget *widget, 
					    GdkEventButton *event,
					    BookmarkItem *bi);
void bookmarks_activate_cb (GtkWidget *widget, BookmarkItem *bi);
void bookmarks_open_in_new_tab_activate_cb (GtkWidget *widget,
					    BookmarkItem *bi);
void bookmarks_open_in_new_window_activate_cb (GtkWidget *widget,
					       BookmarkItem *bi);
void bookmarks_copy_link_activate_cb (GtkWidget *widget, BookmarkItem *bi);
void bookmarks_add_activate_cb (GtkWidget *widget, BookmarkItem *bi);
void bookmarks_add_folder_activate_cb (GtkWidget *widget, BookmarkItem *bi);
void bookmarks_open_all_activate_cb (GtkWidget *widget, BookmarkItem *bi);
void bookmarks_open_all_release_cb (GtkWidget *widget, GdkEventButton *event,
				    BookmarkItem *bi);
void bookmarks_open_all_in_wins_activate_cb (GtkWidget *widget,
					     BookmarkItem *bi);
void bookmarks_open_all_in_wins_release_cb (GtkWidget *widget,
					    GdkEventButton *event,
					    BookmarkItem *bi);
void bookmarks_create_toolbar_activate_cb (GtkWidget *widget, BookmarkItem *bi);
void bookmarks_set_as_default_activate_cb (GtkWidget *widget, BookmarkItem *bi);
void bookmarks_regenerate_autobookmarks_cb (GtkWidget *widget,
					    BookmarkItem *bi);
void bookmarks_edit_activate_cb (GtkWidget *menuitem, BookmarkItem *bm);
void bookmarks_remove_activate_cb (GtkWidget *menuitem, BookmarkItem *bm);
void bookmarks_export_activate_cb (GtkWidget *menuitem, BookmarkItem *bm);
void bookmarks_import_activate_cb (GtkWidget *menuitem, BookmarkItem *bm);
void bookmarks_file_bookmark_cb (GtkWidget *widget, BookmarkItem *parent);
void bookmarks_editor_bookmark_copy_cb (GtkMenuItem *menuitem,
					BookmarkItem *parent);
void bookmarks_nick_entry_activate (GtkEntry *entry, BookmarkItem *b);
void bookmarks_open_in_myportal_cb (GtkWidget *widget, BookmarkItem *bi);
void bookmarks_add_accel_cb (GtkWidget *widget, guint accel_signal_id,
			     GtkAccelGroup *accel_group, guint accel_key,
			     GdkModifierType accel_mods, 
			     GtkAccelFlags accel_flags,
			     BookmarkItem *bi);
void bookmarks_remove_accel_cb (GtkWidget *widget, GtkAccelGroup *accel_group, 
			        guint accel_key, GdkModifierType accel_mods, 
			        BookmarkItem *bi);
void bookmarks_select_cb (GtkWidget *widget, BookmarkItem *bi);
void bookmarks_deselect_cb (GtkWidget *widget, BookmarkItem *bi);

/* bookmarks_toolbar.c */
void bookmarks_toolbars_create (GaleonWindow *window);
void bookmarks_toolbars_recreate (GaleonWindow *window);
void bookmarks_toolbars_set_visibility (GaleonWindow *window, 
					gboolean visible);
void bookmarks_toolbars_check_update (BookmarkItem *b);

#endif
