/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *  Copyright (C) 2001, 2002 Marco Pesenti Gritti, Ricardo Fernndez Pascual
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* Galeon includes */
#include "galeon.h"
#include "embed.h"
#include "autocompletion.h"
#include "bookmarks.h"
#include "bookmarks_editor.h"
#include "bookmarks_io.h"
#include "window.h"
#include "prefs.h"
#include "mozilla.h"
#include "themes.h"
#include "misc_string.h"
#include "misc_gui.h"
#include "glade.h"

/* GNOME includes */
#include <time.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <libgnomevfs/gnome-vfs.h>
#include <libgnome/gnome-util.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-config.h>
#include <libgnomevfs/gnome-vfs-mime.h>
#include <libgnomeui/gnome-dialog.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <gtk/gtkdnd.h>
#include <gtkmozembed.h>

#define NUM_BOOKMARK_BACKUPS 3

/** The pixmaps for the bookmarks */
PixmapData *folder_pixmap_data = NULL;
PixmapData *folder_open_pixmap_data = NULL;
PixmapData *default_folder_pixmap_data = NULL;
PixmapData *default_folder_open_pixmap_data = NULL;
PixmapData *site_pixmap_data = NULL;
PixmapData *smart_bm_fold_pixmap_data = NULL;
PixmapData *smart_bm_unfold_pixmap_data = NULL;

/** The root of the bookmark tree */
BookmarkItem *bookmarks_root = NULL;

/** The root of the temporary bookmark tree */
BookmarkItem *default_bookmarks_root = NULL;

/** The bookmarks need to be saved */
gboolean bookmarks_dirty = FALSE;

/** Do we need to recreate the bookmarks toolbars? */
gboolean bookmarks_tb_dirty = FALSE;

/** List of to-be freed bookmarks */
static GList *removed_bookmarks = NULL;

/** List of avaible bookmark translators */
GList *bookmarks_translators = NULL;

/** the default one */
/* FIXME: this should be static, but currently I need to exprt it to be able
   to import/export to own format. The UI needs to be changed */
BookmarkTranslator *bookmarks_default_translator = NULL;

/* the filename of the bookmarks */
gchar *bookmarks_filename = NULL;

/* DnD */
const GtkTargetEntry bookmarks_dnd_targets [] = {
	{ "GALEON_BOOKMARK", 0, DND_TARGET_GALEON_BOOKMARK },
	{ "GALEON_URL",      0, DND_TARGET_GALEON_URL      },
	{ "text/uri-list",   0, DND_TARGET_TEXT_URI_LIST   },
	{ "_NETSCAPE_URL",   0, DND_TARGET_NETSCAPE_URL    },
	{ "STRING",          0, DND_TARGET_STRING          }
};
const gint bookmarks_dnd_targets_num_items = (sizeof (bookmarks_dnd_targets) /
					      sizeof (GtkTargetEntry));

/* Private prototypes */
static void bookmarks_backup (void);
static void bookmarks_save (void);
static void bookmarks_remove_all_alias (BookmarkItem *b);
static void bookmarks_init_alias (BookmarkItem *alias, BookmarkItem *bookmark);
static BookmarkItem *bookmarks_merge_trees_look_for_equal 
        (BookmarkItem *folder, BookmarkItem *bookmark);
static gint bookmarks_compare_name_and_type (gconstpointer a, gconstpointer b);
static gint bookmarks_compare_name (gconstpointer a, gconstpointer b);
static gchar *bookmarks_get_bookmark_path (BookmarkItem *b);
static void bookmarks_free_enqueue (BookmarkItem *b);
static void bookmarks_free_removed_bookmarks (void);
static void bookmarks_update_images_recursively (BookmarkItem *b, 
						 const gchar *changed_image);

/**
 * These functions are not defined here, but are called only from here
 */
BookmarkTranslator *bookmarks_io_own_format_init (void);
BookmarkTranslator *bookmarks_io_xbel_init (void);
BookmarkTranslator *bookmarks_io_netscape_init (void);
BookmarkTranslator *bookmarks_io_mozilla_init (void);

/**
 * bookmarks_init: intialise bookmarks for the first time
 */
void
bookmarks_init (void)
{
	/* bookmarks loaders/savers */
	bookmarks_io_netscape_init ();
	bookmarks_io_mozilla_init ();
	bookmarks_io_own_format_init ();
	bookmarks_default_translator = bookmarks_io_xbel_init ();

	bookmarks_filename = g_strconcat (g_get_home_dir (),
			                  "/.galeon/bookmarks.xbel", NULL);
	bookmarks_load_icons ();
	bookmarks_load ();
}

/**
 * bookmarks_exit: shut down bookmarks
 */
void
bookmarks_exit (void)
{
	GList *l = g_list_copy (bookmarks_editors);
	GList *li;

	bookmarks_save ();

	/* close all editors */
	for (li = l; li != NULL; li = li->next) 
	{
		bookmarks_editor_hide_dialog (li->data);
	}
	g_list_free (l);

	g_free (folder_open_pixmap_data);
	g_free (folder_pixmap_data);
	g_free (default_folder_open_pixmap_data);
	g_free (default_folder_pixmap_data);
	g_free (site_pixmap_data);
	g_free (smart_bm_fold_pixmap_data);
	g_free (smart_bm_unfold_pixmap_data);

	folder_open_pixmap_data = NULL;
	folder_pixmap_data = NULL;
	default_folder_open_pixmap_data = NULL;
	default_folder_pixmap_data = NULL;
	site_pixmap_data = NULL;
	smart_bm_fold_pixmap_data = NULL;
	smart_bm_unfold_pixmap_data = NULL;

	bookmarks_remove_recursively (bookmarks_root);

	bookmarks_root = NULL;
}

/**
 * bookmarks_load: load bookmarks from xml file
 */
void
bookmarks_load (void)
{
	BookmarkItem *new_root = NULL;
	gchar *oldbookmarks_filename;

	if (bookmarks_default_translator == NULL)
	{
		g_error ("Don't know how to load bookmarks!");
	}

	default_bookmarks_root = NULL;

	/* Automatic migration to XBEL */
	oldbookmarks_filename = g_strconcat (g_get_home_dir (),
					     "/.galeon/bookmarks.xml", NULL);

	if (!g_file_exists (bookmarks_filename)
	    && g_file_exists (oldbookmarks_filename))
	{
		BookmarkTranslator *old = bookmarks_io_own_format_init ();
		GtkWidget *dialog = gnome_ok_dialog
			(_("There is a file bookmarks.xml but no "
			   "bookmarks.xbel in your ~/.galeon folder.\n"
			   "This probably means that this is the first "
			   "time that you run a post 1.0 version\n"
			   "of Galeon.\n\n"
			   "The bookmarks format of Galeon has changed to "
			   "XBEL for enhanced interoperatibility.\n"
			   "Your old bookmarks file will be loaded now and "
			   "it will be left untouched for backup.\n\n"
			   "Please, report any problems."));
		gnome_dialog_run (GNOME_DIALOG (dialog));
		
		new_root = old->load_from_file (oldbookmarks_filename, 
						&default_bookmarks_root);
		
		/* make sure bookmarks are saved */
		bookmarks_dirty = TRUE;
	}
	else
	{
		new_root = bookmarks_default_translator->load_from_file 
			(bookmarks_filename, &default_bookmarks_root);
	}

	if (new_root == NULL)
	{
		g_warning ("could not load bookmarks!");
	}

	g_free (oldbookmarks_filename);
	bookmarks_set_root (new_root);
}

/**
 * Sets the root of the bookmarks tree, freeing the previous tree and updating
 * everything
 */
void
bookmarks_set_root (BookmarkItem *new_root)
{
	GList *li;

	bookmarks_editor_freeze_all ();
	
	/* delete the old bookmarks */
	bookmarks_remove_recursively (bookmarks_root);
	
	bookmarks_root = new_root;
	if (!bookmarks_root) 
	{
		gchar* utfstr = mozilla_locale_to_utf8(_("Bookmarks"));
		bookmarks_root = bookmarks_new_bookmark 
			(BM_FOLDER, TRUE, utfstr, NULL, NULL,
			 NULL, NULL);
		g_free(utfstr);
	}
	
	if (default_bookmarks_root == NULL) {
		default_bookmarks_root = bookmarks_root;
	}

	/* generate the autobookmarks folder */
	autobookmarks_generate();
	
	bookmarks_add_completion_recursively (bookmarks_root);

	/* view new bookmarks */
	for (li = bookmarks_editors; li != NULL; li = li->next) 
	{
		/* FIXME: this sets the root bookmark of every editor
		   to the root bookmark. This is incorrect. 
		   (but not a big deal) */
		BookmarksEditorControls *current_controls = li->data;
		bookmarks_editor_set_root (current_controls, bookmarks_root);
	}
	bookmarks_editor_thaw_all ();
}

/**
 * bookmarks_load_icons: load the bookmarks icons
 */
void 
bookmarks_load_icons (void) 
{
	/* load pixmaps */
	folder_pixmap_data = themes_get_pixmap ("dir.png", FALSE);
	folder_open_pixmap_data = themes_get_pixmap ("dir_open.png", FALSE);
	default_folder_pixmap_data = themes_get_pixmap ("default.png", FALSE);
	default_folder_open_pixmap_data = themes_get_pixmap ("default_open.png",
							    FALSE);
	site_pixmap_data = themes_get_pixmap ("i-bookmark.png", FALSE);
	smart_bm_fold_pixmap_data = themes_get_pixmap ("smart-bm-fold.png",
						       FALSE);
	smart_bm_unfold_pixmap_data = themes_get_pixmap ("smart-bm-unfold.png",
						          FALSE);

	/* TODO: add icons for alias */
}

/**
 * bookmarks_updated: called when one or more change has taken place in 
 * the bookmarks which we wish to record, i.e. save to disk and display
 * in the menus and toolbars.
 */
void
bookmarks_updated (void)
{
	GList *l;

	/* force a save */
	bookmarks_dirty = TRUE;
	bookmarks_save ();

	/* update each window */
	for (l = all_windows; l != NULL; l = g_list_next (l))
	{
		GaleonWindow *window = l->data;

		if (!window || (window->magic != GALEON_WINDOW_MAGIC))
		{
			g_warning ("bookmaks_updated found unexpected things "
				   " in window list!");
			continue;
		}
		
		/* save window layout */
		window_save_layout (window);

		/* update bookmarks menubar */
		bookmarks_menu_recreate (window);
	
		/* update bookmarks toolbars */
		if (bookmarks_tb_dirty)
			bookmarks_toolbars_recreate (window);

		/* restore window layout */
		window_restore_layout (window);
	}

	bookmarks_tb_dirty = FALSE;

	/* reload any myportals */
	for (l = all_embeds; l; l = g_list_next (l))
	{
		GaleonEmbed *embed = l->data;

		if (embed->location && strncmp (embed->location,
						"myportal:", 9) == 0)
		{
			embed_reload (embed, GTK_MOZ_EMBED_FLAG_RELOADNORMAL);
		}
	}

	/* FIXME: update applet... */

	bookmarks_free_removed_bookmarks ();
}

/**
 * free removed bookmarks 
 */
static void
bookmarks_free_removed_bookmarks (void)
{
	GList *l;
	for (l = removed_bookmarks; l; l = g_list_next (l))
	{
		bookmarks_free_bookmark (l->data);
	}
	g_list_free (removed_bookmarks);
	removed_bookmarks = NULL;
}


/**
 * bookmarks_backup: rotate bookmark backups
 */
void
bookmarks_backup (void)
{
	gint count;

	/* backup/rotate the bookmarks */
	for (count = NUM_BOOKMARK_BACKUPS - 1; count >= 0; count--)
	{
		gchar *src_ext, *dest_ext;
		gchar *src_filename, *dest_filename;
		GnomeVFSURI *src_uri, *dest_uri;

		/* get the extensions */
		if (count > 0) src_ext = g_strdup_printf (".%d", count - 1);
		else src_ext = NULL;
		dest_ext = g_strdup_printf (".%d", count);

		/* create the filenames */
		src_filename = g_strconcat (bookmarks_filename,
					    src_ext, NULL);
		dest_filename = g_strconcat (bookmarks_filename,
					     dest_ext, NULL);

		/* create uris from the filenames */
		src_uri = gnome_vfs_uri_new (src_filename);
		dest_uri = gnome_vfs_uri_new (dest_filename);

		/* move the file */
		if (gnome_vfs_uri_exists (src_uri))
		{
			gnome_vfs_move_uri (src_uri,
					    dest_uri, TRUE);
		}

		/* free stuff */
		g_free (src_ext);
		g_free (dest_ext);
		g_free (src_filename);
		g_free (dest_filename);
		gnome_vfs_uri_unref (src_uri);
		gnome_vfs_uri_unref (dest_uri);
	}
}

/**
 * bookmarks_save: save all bookmarks to bookmarks_file, after backing up
 * the previous versions 
 */
void
bookmarks_save (void)
{
	gboolean ok;
	gchar *temp_filename;
	GnomeVFSURI *temp_uri;
	static gboolean backed_up_already = FALSE;

	/* make sure the bookmarks have been changed before we do anything */
	if (!bookmarks_dirty)
	{
		return;
	}

	if (bookmarks_default_translator == NULL)
	{
		g_warning ("Don't know how to save bookmarks!");
		return;
	}

	/* write the temp file */
	temp_filename = g_strconcat (bookmarks_filename, ".temp", NULL);
	ok = bookmarks_default_translator->save_to_file 
		(bookmarks_root, default_bookmarks_root, temp_filename);
	temp_uri = gnome_vfs_uri_new (temp_filename);

	/* if the file was written... */
	if (ok && gnome_vfs_uri_exists (temp_uri))
	{
		GnomeVFSResult result;

		/* back up the bookmarks, if we haven't already */
		if (!backed_up_already)
		{
			bookmarks_backup ();
			backed_up_already = TRUE;
		}

		/* move the temp file to the normal location */
		result = gnome_vfs_move (temp_filename,
					 bookmarks_filename, TRUE);

		/* make sure the move went okay */
		if (result == GNOME_VFS_OK)
		{
			/* bookmarks have been synched now */
			bookmarks_dirty = FALSE;
		}
		else
		{
			gnome_error_dialog (
				_("Unable to save bookmark file."));
		}
	}
	/* otherwise... */
	else
	{
		gnome_error_dialog (_("Unable to save bookmark file."));
	}

	/* free the temp file stuff */
	gnome_vfs_uri_unref (temp_uri);
	g_free (temp_filename);
}


/**
 * bookmarks_remove_recursively: recursively remove a tree of bookmarks items
 */
void 
bookmarks_remove_recursively (BookmarkItem *b) 
{
	
	if (b == NULL) return;

	bookmark_unparent (b);
	
	if (!b->alias_of) 
	{
		/* When removing a bookmark, we remove all its alias
		   Instead, we should check before removing if it has aliases
		   and warn the user */
		bookmarks_remove_all_alias (b);
	}
	else 
	{
		/* we are removing an alias, connect the aliased bookmark with 
		   it's next, if any */
		b->alias_of->alias = b->alias;
	}
	
	if (BOOKMARK_ITEM_IS_FOLDER (b) && !b->alias_of)
	{
		while (b->list)
		{
			bookmarks_remove_recursively (b->list->data);
		}
	}
	
	if (b == default_bookmarks_root) 
		default_bookmarks_root = bookmarks_root;

	bookmarks_editor_remove_tree_items (b);
	bookmarks_free_enqueue (b);
}

static void
bookmarks_free_enqueue (BookmarkItem *b)
{
	if (!g_list_find (removed_bookmarks, b))
	{
		removed_bookmarks = g_list_prepend (removed_bookmarks, b);
	}
}

/**
 * Remove a list of bookmarks.
 */
void
bookmarks_remove_several (GList *l)
{
	for (; l != NULL; l = g_list_next (l))
	{
		BookmarkItem *b = l->data;
		bookmarks_toolbars_check_update (b);
		bookmarks_remove_recursively (b);
		bookmarks_dirty = TRUE;
        }
}

/**
 * bookmark_exists: check whether a bookmark exists
 */
gboolean
bookmarks_bookmark_exists (BookmarkItem *start, BookmarkItem *b)
{
	GList *item;

	if (!b || !start)
		return FALSE;

	if (b == start)
		return TRUE;
	
	/* FIXME: I'm not sure if the following is really correct */
	if (start->alias_of)
		return FALSE;
	
	if (BOOKMARK_ITEM_IS_FOLDER (start))
	{
		for (item = start->list; item != NULL; item = item->next)
		{
			if (bookmarks_bookmark_exists (item->data, b))
				return TRUE;
		}
	}

	return FALSE;
}

gint 
bookmarks_count_tree (BookmarkItem *root)
{
	gint count;
	if (root == NULL) return 0;
	count = 1;
	if (BOOKMARK_ITEM_IS_FOLDER (root) && !root->alias_of)
	{
		GList *l;
		for (l = root->list; l != NULL; l = l->next)
		{
			count += bookmarks_count_tree (l->data);
		}
	}
	return count;
}

/**
 * bookmarks_new_bookmark: Creates a new allocated BookmarkItem
 * @type: the type of the bookmark
 * @escape_name: TRUE if the name needs to have _'s escaped, 
 *                       FALSE if they already are
 * @name: the name, if NULL the url is used
 * @url: the url (ignored if type != BM_SITE)
 * @nick: the nickname for the bookmark, if any
 * @notes: the comments aout the bookmark
 * 
 * This function allocates a BookmarkItem and initializes its fields to sane 
 * values.
 * Use it instead of g_new (BookmarkItem, 1) and manually initializing the 
 * fields. 
 * The returned BookmarkItem has it's parent set to NULL, so you have to add it
 * to the bookmarks list yourself (if you need it)
 * All the strings passed to this function are g_strduped
 * 
 * Return value: The new BookmarkItem
 **/
BookmarkItem *
bookmarks_new_bookmark (BookmarkType type, gboolean escape_name,
			const char *name, const char *url, 
			const char *nick, const char *notes, 
			const char *pixmap_file)
{
	BookmarkItem *bookmark;

	g_return_val_if_fail (!((type == BM_SITE) && (url == NULL)), NULL);
	g_return_val_if_fail (!((type == BM_FOLDER) && (name == NULL)), NULL);

	bookmark = g_new0 (BookmarkItem, 1);
	bookmark->type = type;

	switch (bookmark->type) 
	{
	case BM_AUTOBOOKMARKS:
	case BM_FOLDER:
		if (escape_name)
			bookmark->name = 
				misc_string_escape_uline_accel (name);
		else
			bookmark->name = g_strdup (name);
		bookmark->url = NULL;
		break;
	case BM_SITE:
		if (name)
		{
			if (escape_name)
				bookmark->name =
					misc_string_escape_uline_accel (name);
			else
				bookmark->name = g_strdup (name);
		}
		else
		{
			/* shorten our url title to 30 characters */
			gchar *temp_str = misc_string_shorten (url, 30);
			bookmark->name = misc_string_escape_uline_accel (
								temp_str);
			g_free (temp_str);
		}
		bookmark->url = g_strdup (url);
		/* we are adding the URL to the autocompletion list. That means
		 * that we can't free the string without removing it first from
		 * that list */
		auto_completion_add_url (bookmark->url);
		break;
	case BM_SEPARATOR:
		bookmark->name = bookmark->url = NULL;
		break;
	default:
		g_assert_not_reached ();
		break;
	}
	bookmark->list = NULL;
	bookmark->tree_items = NULL;
	bookmark->parent = NULL;
	bookmark->create_toolbar = FALSE;
	bookmark->toolbar_style = TOOLBAR_STYLE_HORIZONTAL;
	bookmark->create_context_menu = FALSE;
	if (nick) 
		bookmark->nick = g_strdup (nick);
	else
		bookmark->nick = g_strdup ("");
	if (notes)
		bookmark->notes = g_strdup (notes);
	else
		bookmark->notes = g_strdup ("");
	if (pixmap_file != NULL && strlen(pixmap_file) != 0)
	{
		bookmark->pixmap_file = g_strdup (pixmap_file);
		bookmark->pixmap_data = misc_gui_pixmap_data_new_from_file
			(pixmap_file, FALSE);
	}
	else
	{
		bookmark->pixmap_file = g_strdup ("");
		bookmark->pixmap_data = NULL;
	}
	bookmark->expanded = TRUE;
	bookmark->alias = NULL;
	bookmark->alias_of = NULL;
	bookmark->time_added = 0;
	bookmark->time_modified = 0;
	bookmark->time_visited = 0;
	bookmark->smarturl = NULL;

	return bookmark;
}

/**
 * bookmarks_new_alias: creates an alias of a bookmark. If the parameter is
 * NULL, returns the alias with all fields NULL, for creating an alias
 * later.
 */
BookmarkItem *
bookmarks_new_alias (BookmarkItem *bookmark)
{
	BookmarkItem *alias = g_new0 (BookmarkItem, 1);
	alias->tree_items = NULL;
	alias->parent = NULL;
	alias->create_toolbar = FALSE;
	alias->toolbar_style = TOOLBAR_STYLE_HORIZONTAL;
	alias->create_context_menu = FALSE;
	if (bookmark) bookmarks_init_alias (alias, bookmark);
	return alias;
}

/**
 * bookmarks_init_alias: initializes alias to point towards bookmark.
 */
static void
bookmarks_init_alias (BookmarkItem *alias, BookmarkItem *bookmark)
{
	if (!bookmark) return;

	/* Most are the same as the original bookmark */
	/* actually, they point to the same string, so be cautious */
	alias->type = bookmark->type;
	alias->name = bookmark->name;
	alias->url = bookmark->url;
	alias->smarturl = bookmark->smarturl;
	alias->list = bookmark->list;
	alias->nick = bookmark->nick;
	alias->notes = bookmark->notes;
	alias->pixmap_file = bookmark->pixmap_file;
	alias->pixmap_data = bookmark->pixmap_data;
	alias->expanded = TRUE; /* does not have sense here */
	/* insert in the list */
	alias->alias = bookmark->alias;
	alias->alias_of = bookmark;
	bookmark->alias = alias;
}

/**
 * Updates all aliases of a bookmark
 */
void
bookmarks_update_alias (BookmarkItem *b)
{
	BookmarkItem *alias;
	if (!b) return;
	while ((alias = b->alias) != NULL)
	{
		alias->name = b->name;
		alias->url = b->url;
		alias->list = b->list;
		alias->nick = b->nick;
		alias->notes = b->notes;
		alias->pixmap_file = b->pixmap_file;
		alias->pixmap_data = b->pixmap_data;
		/* note that time_added is not updated here */
		alias->time_visited = b->time_visited;
		alias->time_modified = b->time_modified;
		alias->smarturl = b->smarturl;
		alias->accel_key = b->accel_key;
		alias->accel_mods = b->accel_mods;
		bookmarks_editor_update_tree_items (alias);
		b = alias;
	}
}

/**
 * Finds the real bookmark of an alias
 */
BookmarkItem *
bookmarks_find_real_bookmark (BookmarkItem *b)
{
	if (!b->alias_of) 
		return b;
	else
		return bookmarks_find_real_bookmark (b->alias_of);
}

/**
 * Sets the real bookmark visited date and ensures it is saved.
 */
void bookmarks_set_visited (BookmarkItem *b)
{
	BookmarkItem *r;
	g_return_if_fail (b != NULL);
	r = bookmarks_find_real_bookmark (b);
	r->time_visited = time (NULL);
	bookmarks_update_alias (r);
	bookmarks_dirty = TRUE;
}

/**
 * bookmarks_copy_bookmark: copy a bookmark
 * @b: the bookmark
 * 
 * copy a bookmarks and its children
 * 
 * Return value: the new bookmark
 **/
/* FIXME: This fails miserably wrt aliases */
BookmarkItem *
bookmarks_copy_bookmark (BookmarkItem *b)
{
	GList *l;
	/* TODO: if it is an alias... */
	BookmarkItem *new = bookmarks_new_bookmark (b->type, FALSE, b->name,
						    b->url, b->nick, b->notes,
						    b->pixmap_file);

	if (b->smarturl)
	{
		g_free (new->smarturl);
		new->smarturl = g_strdup (b->smarturl);
	}

	new->create_toolbar = b->create_toolbar;
	new->toolbar_style = b->toolbar_style;
	new->create_context_menu = b->create_context_menu;
	new->expanded = b->expanded;
	/* should these times just copied here? 
	   should they be updated? */
	new->time_added = b->time_added;
	new->time_modified = b->time_modified;
	new->time_visited = b->time_visited;
	new->accel_key = b->accel_key;
	new->accel_mods = b->accel_mods;

	if (BOOKMARK_ITEM_IS_FOLDER (b)) 
	{
		for (l = b->list; l != NULL; l = g_list_next (l)) {
			BookmarkItem *child = l->data;
			BookmarkItem *new_child;
			if (BOOKMARK_ITEM_IS_FOLDER (b) && (b->alias_of))
				continue;
			new_child = bookmarks_copy_bookmark (child);
			new_child->parent = new;
			new->list = g_list_append (new->list, new_child);
		}
	}
	return new;
}

/**
 * bookmarks_free_bookmark: free a bookmark
 * @b: the bookmark to be freed, if NULL thenb do nothing
 * 
 * Frees the bookmarks and all the strings referenced by it and the list 
 * of children if it's a BM_FOLDER (but does not free the children). Use it 
 * instead of g_free. 
 **/
void
bookmarks_free_bookmark (BookmarkItem *b)
{
	if (! b) return;
	
	/* aliases fields point to the real bookmark fields */
	if (!b->alias_of) {
		g_free (b->name);
		/* remove the url from the completion list before freeing */
		auto_completion_remove_url (b->url);
		g_free (b->url);
		g_free (b->smarturl);
		g_free (b->nick);
		g_free (b->pixmap_file);
		if (b->pixmap_data)
			misc_gui_pixmap_data_free (b->pixmap_data);
		g_free (b->notes);
		g_list_free (b->list);
	}
	g_free (b);
	b = NULL;
}

/**
 * bookmarks_insert_bookmark: insert a bookmark
 * @b: the bookmark to insert
 * @near: a bookmark that should be near @b (a brother or its parent)
 **/
void
bookmarks_insert_bookmark (BookmarkItem *b, BookmarkItem *near,
			   GtkCListDragPos insert_pos)
{
	BookmarkItem *parent;
	GList *position_node;
	gint position;
	
	if (near == NULL)
	{
		parent = bookmarks_root;
	}
	else 
	{
		if (insert_pos == GTK_CLIST_DRAG_INTO 
		    && BOOKMARK_ITEM_IS_FOLDER (near) 
		    && !near->alias_of)
			parent = near;
		else
			parent = near->parent;
	}
	
	g_return_if_fail (parent != NULL);
	b->parent = parent;
	
	position_node = g_list_find (parent->list, near);
	if (!position_node)
	{
		position = 0;
	}
	else 
	{
		position = g_list_position (parent->list, position_node);
		switch (insert_pos) 
		{
		case GTK_CLIST_DRAG_NONE:
		case GTK_CLIST_DRAG_BEFORE:
			break;
		case GTK_CLIST_DRAG_INTO:
			if (near == NULL)
				break;
			if (BOOKMARK_ITEM_IS_FOLDER (near))
				position++;
			break;				
		case GTK_CLIST_DRAG_AFTER:
			position++;
			break;
		default:
			break;
		}
	}
	bookmark_add_child (parent, b, position);

	bookmarks_toolbars_check_update (b);
	bookmarks_editor_place_tree_items (b);
}

/**
 * bookmarks_move_bookmark: Moves a bookmark
 * @controls: the controls of the editor
 * @b: the bookmark to move
 * @where: if 0 then move up, else move down
 * Returns: 1 if bookmark is now first or last in the list, 0 otherwise
 *
 * Moves a bookmark up or down and updates both the edition ctree and the 
 * bookmarks structure 
 **/
gint
bookmarks_move_bookmark (BookmarksEditorControls *controls, 
			 BookmarkItem *b, int where)
{
	BookmarkItem *parent, *grandparent, *other;
	GList *pos, *other_pos;
	
	parent = b->parent;
	if (!parent)
		return 1;

	grandparent = parent->parent;
	pos = g_list_find (parent->list, b);

	if (!where) /* up */
		other_pos = g_list_previous(pos);
	else /* down */
		other_pos = g_list_next(pos);
	
	if (other_pos) {
		other = other_pos->data;
		if (other->type == BM_FOLDER) {
			/* expand the folder and move b into it */
			parent->list = g_list_remove_link (parent->list, pos);
			bookmarks_update_alias (parent);
			gtk_ctree_expand (GTK_CTREE (controls->ctree), 
					bookmarks_get_tree_item
					  (other, controls));
			
			if (!where) { 
				/* up */
				other->list = g_list_append (other->list, b);
				b->parent = other;
			} else {
				/* down */
				b->parent = other;
				other->list = g_list_prepend (other->list, b);
			}
		} else {
			gint newindex = g_list_position
				(parent->list, other_pos);
			b->parent->list = g_list_remove_link 
				(b->parent->list, pos);
			b->parent->list = g_list_insert 
				(parent->list, b, newindex);
		}
	} else {
		/* move b to its grandparent*/
		if  (!grandparent)
			return 1;
		
		parent->list = g_list_remove_link(parent->list, pos);
		bookmarks_update_alias (parent);
		
		if (!where) {
			/* up */
			grandparent->list = g_list_insert 
				(grandparent->list, b, g_list_position 
				 (grandparent->list, g_list_find(grandparent->list, 
								 parent)));
			b->parent = grandparent;
		} else {
			/* down */
			GList *next = g_list_find
				(grandparent->list, parent)->next;
			grandparent->list = g_list_insert 
				(grandparent->list, b, 
				 g_list_position 
				 (grandparent->list, next));
			b->parent = grandparent;
		}
	}
	bookmarks_editor_place_tree_items (b);
	bookmarks_update_alias (b->parent);
	gtk_ctree_select ( GTK_CTREE (controls->ctree),
			   bookmarks_get_tree_item (b, controls));

	return 0;
}

BookmarkItem *
bookmarks_add_bookmark (const gchar *name, const gchar *url,
			BookmarkItem *parent, GtkWindow *parent_window,
			BookmarkAddFlags flags)
{
	return bookmarks_add_item (BM_SITE, name, url, NULL,
				   NULL, parent, parent_window, flags);
}

BookmarkItem *
bookmarks_add_smart_bookmark (const gchar *name, const gchar *url,
			      const gchar *smarturl, const gchar *pixmap_file,
			      BookmarkItem *parent, GtkWindow *parent_window,
			      BookmarkAddFlags flags)
{
	return bookmarks_add_item (BM_SITE, name, url, smarturl,
				   pixmap_file, parent, parent_window, flags);
}

/**
 * Updates all the bookmarks that use an image. Called when the image
 * has finished downloading.
 */
void 
bookmarks_update_images (const gchar *changed_image)
{
	g_return_if_fail (changed_image != NULL);
	bookmarks_update_images_recursively (bookmarks_root, changed_image);
	/* update toolbars / menus */
	if (bookmarks_tb_dirty)
		bookmarks_updated ();

}

static void 
bookmarks_update_images_recursively (BookmarkItem *b, 
				     const gchar *changed_image)
{
	GList *l;
	g_return_if_fail (b != NULL);
	if (b->pixmap_file && !strcmp (changed_image, b->pixmap_file))
	{
		/* force to reread it */
		bookmark_set_pixmap (b, "");
		bookmark_set_pixmap (b, changed_image);
		/* update toolbars / menus */
		bookmarks_toolbars_check_update (b);
	}
	if (BOOKMARK_ITEM_IS_FOLDER (b) && !b->alias_of)
		for (l = b->list; l; l = l->next)
			bookmarks_update_images_recursively (l->data, 
							     changed_image);
}

BookmarkItem *
bookmarks_add_folder (const gchar *name, BookmarkItem *parent,
		      GtkWindow *parent_window, BookmarkAddFlags flags)
{
	return bookmarks_add_item (BM_FOLDER, name, NULL, NULL,
				   NULL, parent, parent_window, flags);
}


/**
 * bookmarks_add_item: adds a bookmark or folder.
 */
BookmarkItem *
bookmarks_add_item (BookmarkType type, const gchar *name,
		    const gchar *url, const gchar *smarturl,
		    const gchar *pixmap_file, BookmarkItem *parent,
		    GtkWindow *parent_window, BookmarkAddFlags flags)
{
	BookmarkItem *bookmark;
	gchar *real_name = NULL;
	gint short_length; /* the length that constitutes a "short" title */
	gint prompt; /* whether the user wants to be prompted for
			a bookmark title */

	/* return if we received a non-handled type */
	if (type != BM_FOLDER && type != BM_SITE) return NULL;

	/* get related prefs */
	short_length = eel_gconf_get_integer (CONF_BOOKMARKS_SHORTTEXT);
	prompt = eel_gconf_get_integer (CONF_BOOKMARKS_TITLE);

	switch (type)
	{
		case BM_FOLDER:
			if (name) real_name = misc_string_strip_newline (name);
			else real_name = g_strdup (_("Untitled folder"));
			break;
		case BM_SITE:
			/* use url if we didn't get a name */
			if (!name || name[0] == '\0')
				real_name = g_strdup (url);
			/* check if we're allowed to expand the title */
			else if (flags & BOOKMARK_ADD_ALLOW_EXPAND_TITLE)
			{
				/* use old url-only behavior
				 * if the short length is negative */
				if (short_length < 0)
					real_name = g_strdup (url);
				/* if necessary, append url to name */
				else if ((gint) strlen (name) <= short_length &&
					 strncmp (url, "javascript:", 11))
				{
					gchar *temp = g_strdup_printf (
							"%s - %s", name, url);
					real_name = misc_string_strip_newline (
									temp);
					g_free (temp);
				}
				/* otherwise, just use the name */
				else 
				{ 
					real_name = misc_string_strip_newline (
						name);
				}
			}
			/* otherwise, just use the name */
			else 
			{
				real_name = misc_string_strip_newline (name);
			}
			break;
		default:
			break;
	}

	/* prompt user for title if appropriate */
	if ((prompt && !(flags & BOOKMARK_ADD_NEVER_PROMPT_TITLE)) ||
	    (flags & BOOKMARK_ADD_FORCE_PROMPT_TITLE))
	{
		GladeXML *gxml;
		GtkWidget *dialog, *label, *entry;
		gint dialog_button;

		/* create dialog */
		gxml = glade_widget_new ("bookmarks.glade",
					 "bookmark_name_dialog",
					 NULL, NULL);
		dialog = glade_xml_get_widget (gxml, "bookmark_name_dialog");
		label = glade_xml_get_widget (gxml, "label");
		entry = glade_xml_get_widget (gxml, "entry");
		gtk_object_unref (GTK_OBJECT (gxml));

		/* set label and title */
		if (type == BM_SITE)
		{
			gtk_label_set_text (GTK_LABEL (label),
					(_("Please name the bookmark.")));
			gtk_window_set_title (GTK_WINDOW (dialog),
					(_("Bookmark Name")));
		}
		else if (type == BM_FOLDER)
		{
			gtk_label_set_text (GTK_LABEL (label),
					(_("Please name the folder.")));
			gtk_window_set_title (GTK_WINDOW (dialog),
					(_("Folder Name")));
		}

		/* set entry text and highlight it */
		gtk_entry_set_text (GTK_ENTRY (entry), real_name);
		gtk_editable_select_region (GTK_EDITABLE (entry), 0, -1);

		/* set parent window */
		if (parent_window)
			gnome_dialog_set_parent (GNOME_DIALOG (dialog),
						 parent_window);

		/* give entry the focus, and make its activate signal click
		 * the ok button */
		gtk_widget_grab_focus (entry);
		gnome_dialog_editable_enters (GNOME_DIALOG (dialog),
					      GTK_EDITABLE (entry));

		/* display the dialog */
		gtk_widget_show_all (GNOME_DIALOG (dialog)->vbox);
		dialog_button = gnome_dialog_run (GNOME_DIALOG (dialog));

		/* if the dialog is still open, get the text from the
		 * entry, convert to utf8, and then close the dialog */
		if (dialog_button > -1)
		{
			gchar *temp;

			temp = gtk_editable_get_chars (GTK_EDITABLE (entry),
						       0, -1);
			if (real_name) g_free (real_name);
			real_name = mozilla_locale_to_utf8 (temp);
			g_free (temp);

			gnome_dialog_close (GNOME_DIALOG (dialog));
		}

		/* if they didn't click ok, abort */
		if (dialog_button != 0)
		{
			if (real_name) g_free (real_name);
			return NULL;
		}

	}

	/* create new bookmark */
	bookmark = bookmarks_new_bookmark (type, TRUE, real_name,
					   url, NULL, NULL, pixmap_file);

	/* use the default folder if no parent was provided */
	if (!parent) parent = default_bookmarks_root;
	if (!parent) parent = bookmarks_root;
	
	/* if we received smart url, set it */
	if (type == BM_SITE && smarturl)
		bookmark_set_smarturl (bookmark, smarturl);

	bookmark->time_added = time (NULL);

	bookmark_add_child (parent, bookmark, -1);
	bookmarks_editor_place_tree_items (bookmark);
	bookmarks_toolbars_check_update (bookmark);
	bookmarks_updated ();

	/* free allocated string */
	if (real_name) g_free (real_name);

	return bookmark;
}

/**
 * bookmarks_parse_nick: returns the url of a given nick with one arg
 * @text: the text to try to parse
 * @root: the root bookmark for start the search, if NULL the usual tree will
 * be used
 * 
 * Tries to find a bookmarkitem wich has a nick like the first word of @text 
 * and if found return the URL result of substituting each %s in the 
 * BookmarkItem url by the 
 * text after the first " " in @text
 * 
 * Return value: the resulting URL or NULL if not matching nick found
 **/
gchar *
bookmarks_parse_nick (const gchar *text, BookmarkItem *root)
{
	gchar *nick;
	gchar *space;
	BookmarkItem *b;
	gchar *ntext = g_strdup (text);
	gchar *stext;
	gchar *ret;

	/* remove any extra spaces */
	stext = g_strstrip (ntext);

	/* find the nick */
	space = strstr (stext, " ");
	if (space)
	{
		gchar *translated_text;

		/* split out nickname and arguments */
		nick = g_strndup (stext, space - stext);

		/* find smart bookmark */
		b = bookmarks_find_by_nick (nick, root, TRUE);
		if (b == NULL) /* try for a normal if no smartbm found*/
			b = bookmarks_find_by_nick (nick, root, FALSE);

		g_free (nick);

		/* check we found it */
		if (b == NULL) 
		{
			ret = NULL;
		}
		else 
		{
			/* translate non-alphanumeric characters+
			   into %{} * format */
			translated_text = bookmarks_encode_smb_string (
								space + 1);

			/* do substitution */		
			ret = bookmarks_substitute_argument (b, 
							     translated_text);
			
			g_free (translated_text);
		}
	} 
	else 
	{
		/* the whole thing is the nick */
		nick = stext;
		
		/* find normal bookmark */
		b = bookmarks_find_by_nick (nick, root, FALSE);
		if (b == NULL) /* try for a smartbm if no normal found*/
			b = bookmarks_find_by_nick (nick, root, TRUE);
		
		/* check we found it */
		if (b == NULL)
		{
			ret = NULL;
		}
		else 
		{
			/* return copy of the url */
			ret = g_strdup(b->url);
		}
	}
	g_free (ntext);
	/* don't free stext */
	return ret;
}

/**
 * bookmarks_substitute_argument:
 **/
gchar *
bookmarks_substitute_argument (BookmarkItem *b, const gchar *arguments)
{
	gchar *ret = NULL;
	const gchar *s;
	int expected_args = 0;

	g_return_val_if_fail (b != NULL, NULL);
	g_return_val_if_fail (arguments != NULL, NULL);

	s = b->smarturl;
	if (!s || strlen (s) == 0)
		s = b->url;

	ret = g_strdup (s);

	/* count %s in the SmartURL */
	while ((s = strstr(s, "%s"))) 
	{
		++expected_args;
		++s;
	}
	
	if (expected_args <= 1) 
	{
		/*  do the substitution */
		while ( (s = strstr (ret, "%s")) ) 
		{
			gchar *new1, *new2;
			new1 = g_strndup (ret, s - ret);
			new2 = g_strconcat (new1, arguments, s + 2, NULL);
			g_free (new1);
			g_free (ret);
			ret = new2;
		}
	}
	else {
		/* try to substitute matching arguments */
		gchar **args;
		int arg = 0;
		args = g_strsplit(arguments, " ", -1);
		while ( (s = strstr (ret, "%s")) ) 
		{
			gchar *new1, *new2;
			if (!args[arg])
				break;
			new1 = g_strndup (ret, s - ret);
			new2 = g_strconcat (new1, args[arg++], s + 2, NULL);
			g_free (new1);
			g_free (ret);
			ret = new2;
		}
		g_strfreev (args);
	}

	return ret;
}

/**
 * bookmarks_find_by_nick: search for a BookmarkItem with a nick
 * @nick: the nick to search for.
 * @r: the root bookmark for start the search, if NULL the usual tree will be
 * used
 * @wantsmart: if we are trying to find a smart bookmark or not
 * 
 * Search a bookmarks folder recursivley for a bookmark with the given nick
 * 
 * Return value: the found BookmarkItem or NULL if nothing found
 **/
BookmarkItem *
bookmarks_find_by_nick (const gchar *nick, BookmarkItem *r, gboolean wantsmart)
{
	GList *l;
	BookmarkItem *b;
	if (!r) r = bookmarks_root;
	
	if (((r->nick) && r->nick[0])
	    && (!wantsmart 
		|| ((r->smarturl != NULL && r->smarturl[0] != 0) 
		    || (r->url && strstr (r->url, "%s") != NULL))))
	{
		/* maybe the bookmark defines several nicks, separated
		 * by commas or spaces */
		gchar **nicks = g_strsplit (r->nick, ", ", 999);
		gint i;
		gboolean found = FALSE;
		for (i = 0; nicks[i] != NULL; i++)
		{
			if (!strcmp (nicks[i], nick))
			{
				found = TRUE;
				break;
			}
		}
		g_strfreev (nicks);
		if (found) return r;
	}
	
	if ((r->type == BM_FOLDER) && !r->alias_of)
	{
		for (l = r->list; l != NULL; l = g_list_next (l))
		{
			b = bookmarks_find_by_nick (nick, l->data, wantsmart);
			if (b) return b;
		}
	}
	return NULL;
}

/**
 * bookmarks_find_by_url: search for a BookmarkItem with a url
 * @nick: the url to search for.
 * @r: the root bookmark for start the search, if NULL the usual tree will be
 * used
 * 
 * Search a bookmarks folder recursivley for a bookmark with the given url
 * 
 * Return value: the found BookmarkItem or NULL if nothing found
 **/
BookmarkItem *
bookmarks_find_by_url (const gchar *url, BookmarkItem *r)
{
	GList *l;
	BookmarkItem *b;
	if (!r) r = bookmarks_root;

	if (((r->url) && r->url[0]))
	{
		if (!strcmp (r->url, url)) return r;
	}

	if ((r->type == BM_FOLDER) && !r->alias_of)
	{
		for (l = r->list; l != NULL; l = g_list_next (l))
		{
			b = bookmarks_find_by_url (url, l->data);
			if (b) return b;
		}
	}
	return NULL;
}

/**
 * bookmarks_item_to_string: convert a bookmark to a string
 * @bookmark: the bookmark (may or may not be a folder).
 *
 * Return value: the bookmarks as a string.
 **/
gchar * 
bookmarks_item_to_string (BookmarkItem *bookmark)
{
	gchar *buf;
	bookmarks_default_translator->save_to_string (bookmark, 
						      default_bookmarks_root,
						      &buf);
	return buf;
}

/**
 * bookmarks_item_from_string: Read a bookmark from a string
 * @string: the string to parse.
 * Return value: The new allocated bookmark or NULL if error
 **/
BookmarkItem *
bookmarks_item_from_string (char *string)
{
	return bookmarks_default_translator->load_from_string (string, NULL);
}

/**
 * bookmarks_encode_smb_string: encodes the given string to be url-safe, by
 * replacing ' ' with '+' and replacing all unsafe characters with their
 * hexadecimal equivalents.  the returned string is newly-allocated, and
 * should be freed at some point.
 */
gchar *
bookmarks_encode_smb_string (gchar *str)
{
	gchar *enc_str, *enc_ptr;
	gint str_len, i;

	if (!str) return NULL;
	str_len = strlen (str);
	enc_str = enc_ptr = g_new (char, 3 * str_len + 1);

	for (i = 0; i < str_len; i++)
	{
		guchar ch = (unsigned char) str[i];

		/* if the character is alphanumeric, or one of the
		 * unreserved ascii characters, print it verbatim */
		if ((ch >= 'a' && ch <= 'z') ||
		    (ch >= 'A' && ch <= 'Z') ||
		    (ch >= '0' && ch <= '9') ||
		    ch == '-' || ch == '_' || ch == '.' || ch == '!' ||
		    ch == '~' || ch == '*' || ch == '(' || ch == ')' ||
		    ch == '\'')
		{
			*enc_ptr = ch;
			enc_ptr++;
		}
		/* replace spaces with pluses */
		else if (ch == ' ')
		{
			*enc_ptr = '+';
			enc_ptr++;
		}
		/* encode all other characters as %HH */
		else
		{
			gchar key_code[3];

			snprintf (key_code, 3, "%02X", ch);
			enc_ptr[0] = '%';
			enc_ptr[1] = key_code[0];
			enc_ptr[2] = key_code[1];
			enc_ptr += 3;
		}
	}

	/* terminate and return the string */
	*enc_ptr = '\0';
	return enc_str;
}

/**
 * bookmarks_folder_open_all_items: Loads all bookmarks in a folder 
 **/
void
bookmarks_folder_open_all_items (GaleonEmbed *embed, BookmarkItem *bi,
				 gboolean new_window, gboolean reuse)
{
	GList *l = bi->list;
	BookmarkItem *child = NULL;

	while (l) {
		child = l->data;
		if (child->url)
			break;
		l = l->next;
	}

	/* reuse the first tab/window if required */
	if (reuse && l && child->url)
	{
		embed_load_url (embed, child->url);
		l = l->next;
	}

	/* open all the rest of the items */
	while (l)
	{
		BookmarkItem *child = l->data;
		if (child->url)
		{
			embed_create_after_embed (embed, new_window,
						  child->url,
						  EMBED_CREATE_NEVER_JUMP);
		}
		l = l->next;
	}
}

/**
 * bookmarks_add_completion_recursively: recursively add bookmarks to the
 * auto completions table
 */
void 
bookmarks_add_completion_recursively (BookmarkItem *b)
{
	GList *item;

	if (!b) return;
	
	if (b->alias_of)
		return; /* the real one should have been added already */
	
	switch (b->type)
	{
	case BM_FOLDER:
	case BM_AUTOBOOKMARKS:
		for (item = b->list; item != NULL; item = item->next)
		{
			bookmarks_add_completion_recursively (item->data);
		}
		break;

	case BM_SITE:
		auto_completion_add_url (b->url);
		break;

	case BM_SEPARATOR:
		break;
	}

}

/**
 * Removes all alias of a bookmark 
 */
static void 
bookmarks_remove_all_alias (BookmarkItem *b)
{
	BookmarkItem *alias = b->alias;
	if (alias) {
		bookmarks_remove_all_alias (alias);
		bookmarks_remove_recursively (alias);
		b->alias = NULL;
	}
}

GtkCTreeNode *
bookmarks_get_tree_item (BookmarkItem *b,
			 BookmarksEditorControls *c)
{
	GList *l;
	for (l = b->tree_items; l; l = l->next) 
	{
		BookmarkTreeItem *ti = l->data;
		if (ti->controls == c)
			return ti->tree_item;
	}
	return NULL;
}

void
bookmarks_set_tree_item (BookmarkItem *b, BookmarksEditorControls *c,
			 GtkCTreeNode *n)
{
	GList *l;
	BookmarkTreeItem *ti;		
	for (l = b->tree_items; l; l = l->next) 
	{
		ti = l->data;
		if (ti->controls == c) 
		{
			if (n)
			{
				ti->tree_item = n;
			}
			else 
			{
				b->tree_items = g_list_remove_link 
					(b->tree_items, l);
				g_free (ti);
			}	
			return;
		}
	}
	/* not found, add if needed */
	if (n)
	{
		ti = g_new0 (BookmarkTreeItem, 1);
		ti->tree_item = n;
		ti->controls = c;
		b->tree_items = g_list_prepend (b->tree_items, ti);
	}
}

/**
 * Returns TRUE if "a" is an ancestor of "b"
 */
gboolean
bookmarks_is_ancestor (BookmarkItem *a, BookmarkItem *b)
{
	for (; b; b = b->parent)
		if (a == b) return TRUE;
	return FALSE;
}


/**
 * Merges two trees of bookmarks, adding bookmarks in root2 to root1
 * root1 is modified, and root2 is not.
 * This function is intended to be used when importing bookmarks
 * The policies are very simple, they are open to discussion but I wthink 
 * they cover most cases and are asy to implement. They are:
 *    - for categories:
 *            - two categories with the same name at the same level are merged
 *              recursively
 *            - categories in root2 not present in root1 are copied to root1
 *    - for items
 *            - two items are considered equal if they have the same url and
                smarturl
 *            - items not present in root1 are copied
 *    - for separators
 *            - they are added only if the previous item was added too, unless
 *              a whole folder is imported
 *    - for aliases
 *            - aliases are copied as if they were normal bookmarks 
 *              (for simplicity)
 */
void
bookmarks_merge_trees (BookmarkItem *root1, BookmarkItem *root2)
{
	GList *li;
	gboolean last_one_added = FALSE;

	g_return_if_fail (root1 != NULL);
	g_return_if_fail (root2 != NULL);
	
	if (!BOOKMARK_ITEM_IS_FOLDER (root2) && !(root2->alias_of))
	{
		BookmarkItem *item =
			bookmarks_merge_trees_look_for_equal (root1, root2);
		if (item == NULL)
		{
			if (root2->type != BM_SEPARATOR) 
			{
				BookmarkItem *b = bookmarks_copy_bookmark
					(root2);
				root1->list = g_list_append (root1->list, b);
				b->parent = root1;
			}
		}
	}
	
	for (li = root2->list; li != NULL; li = li->next) 
	{
		BookmarkItem *item = li->data;
		BookmarkItem *item2 = bookmarks_merge_trees_look_for_equal
			(root1, item);
		if (item2 == NULL) 
		{
			if ((item->type != BM_SEPARATOR) || last_one_added) 
			{
				BookmarkItem *b = bookmarks_copy_bookmark
					(item);
				root1->list = g_list_append (root1->list, b);
				b->parent = root1;
				last_one_added = TRUE;
			}
		} 
		else if (BOOKMARK_ITEM_IS_FOLDER (item2)) 
		{
			if (!(item2->alias_of || item->alias_of))
				bookmarks_merge_trees (item2, item);
			last_one_added = FALSE;
		}
		else 
		{
			last_one_added = FALSE;
		}
	}	
}

static BookmarkItem *
bookmarks_merge_trees_look_for_equal (BookmarkItem *folder, 
				      BookmarkItem *bookmark)
{
	GList *li;

	g_return_val_if_fail (folder != NULL, NULL);
	g_return_val_if_fail (bookmark != NULL, NULL);

	if (bookmark->type == BM_SEPARATOR) return NULL;
	for (li = folder->list; li != NULL; li = li->next) 
	{
		BookmarkItem *item = li->data;
		if ((item->type == BM_SITE) 
		    && (bookmark->type == BM_SITE)) {
			if (!strcmp (item->url, bookmark->url)) 
			{
				if (((item->smarturl != NULL) 
				     && (bookmark->smarturl != NULL)
				     && !strcmp (item->smarturl,
						 bookmark->smarturl))
				    || (item->smarturl == bookmark->smarturl))
				{
					return item;
				}
			}
		} 
		else if ((BOOKMARK_ITEM_IS_FOLDER (bookmark))
			 && (BOOKMARK_ITEM_IS_FOLDER (item))) 
		{
			if (!strcmp (item->name, bookmark->name)
			    && (!(item->alias_of || bookmark->alias_of)
				|| (item->alias_of && bookmark->alias_of))) 
				return item;
		}
	}
	return NULL;
}


void
bookmarks_sort_norec (BookmarkItem *b, gboolean folders_first, 
		      gboolean update_edit)
{
	if (BOOKMARK_ITEM_IS_FOLDER (b)) 
	{
		if (!b->alias_of) 
		{
			b->list = g_list_sort 
				(b->list, folders_first 
				 ? bookmarks_compare_name_and_type
				 : bookmarks_compare_name);
			bookmarks_update_alias (b);
		}
		if (update_edit)
			bookmarks_editor_place_tree_items (b);
	}
}

void 
bookmarks_sort_rec (BookmarkItem *b, gboolean folders_first,
		    gboolean update_edit)
{
	GList *l;
	bookmarks_sort_norec (b, folders_first, FALSE);
	if (BOOKMARK_ITEM_IS_FOLDER (b))
		for (l = b->list; l != NULL; l = l->next) 
		{
			BookmarkItem *b2 = l->data;
			if (!b2->alias_of) 
				bookmarks_sort_rec (b2, folders_first, FALSE);
		}

	if (update_edit)
		bookmarks_editor_place_tree_items (b);
}

static gint 
bookmarks_compare_name (gconstpointer a, gconstpointer b)
{
	BookmarkItem *b1 = (BookmarkItem *) a;
	BookmarkItem *b2 = (BookmarkItem *) b;

	return g_strcasecmp (b1->name, b2->name);
}

static gint 
bookmarks_compare_name_and_type (gconstpointer a, gconstpointer b)
{
	BookmarkItem *b1 = (BookmarkItem *) a;
	BookmarkItem *b2 = (BookmarkItem *) b;

	if (BOOKMARK_ITEM_IS_FOLDER (b1) && !BOOKMARK_ITEM_IS_FOLDER (b2))
		return -1;
	if (!BOOKMARK_ITEM_IS_FOLDER (b1) && BOOKMARK_ITEM_IS_FOLDER (b2))
		return 1;

	return g_strcasecmp (b1->name, b2->name);
}

/**
 * bookmarks_folder_open_in_myportal: Open the MyPortal page for a category
 **/
void
bookmarks_folder_open_in_myportal (GaleonEmbed *embed, BookmarkItem *bi,
				   gboolean new_window, gboolean reuse)
{
	gchar *url;
	gchar *path;

	g_return_if_fail (bi != NULL);

	if (!BOOKMARK_ITEM_IS_FOLDER (bi))
	{
		bi = bi->parent;
	}

	path = bookmarks_get_bookmark_path (bi);
	url = g_strconcat ("myportal:", path, NULL);
	g_free (path);

	if (reuse)
	{
		embed_load_url (embed, url);
	}
	else 
	{
		embed_create_after_embed (embed, new_window, url,
					  EMBED_CREATE_FORCE_JUMP);
	}
	g_free (url);
}

/**
 * Returns the path of the bookmark (for MyPortal)
 */
static gchar *
bookmarks_get_bookmark_path (BookmarkItem *b)
{
	gchar *stripped = misc_string_strip_uline_accel (b->name);
	gchar *escaped = misc_string_escape_path (stripped);
	gchar *path;
	if ((b->parent == NULL) || (b->parent == bookmarks_root)) 
	{
		path = g_strconcat ("/", escaped, NULL);
	}
	else
	{
		gchar *parent = bookmarks_get_bookmark_path (b->parent);
		path = g_strconcat (parent, "/", escaped, NULL);
		g_free (parent);
	}
	g_free (escaped);
	g_free (stripped);
	return path;
}

/*********************************************************************/

/**
 * Support functions for bookmarks loaders / savers
 */


BookmarkLoaderInfo *
bookmarks_io_loader_info_new (void)
{
	BookmarkLoaderInfo *li;
	li = g_new0 (BookmarkLoaderInfo, 1);
	li->id_bookmark = g_hash_table_new (NULL, NULL);
	return li;
}

void
bookmarks_io_loader_info_free (BookmarkLoaderInfo *li)
{
	if (li == NULL) return;
	g_list_free (li->unresolved_aliases);
	g_hash_table_destroy (li->id_bookmark);
	g_free (li);	
}

UnresolvedAliasInfo *
bookmarks_io_unresolved_alias_info_new (BookmarkItem *unresolved,
				     guint alias_of_id)
{
	UnresolvedAliasInfo *uai = g_new0 (UnresolvedAliasInfo, 1);
	uai->alias_of_id = alias_of_id;
	uai->alias = unresolved;
	return uai;
}

void
bookmarks_io_unresolved_alias_info_free (UnresolvedAliasInfo *uai)
{
	/* nothing interesting to do here */
	/* we might need to free the id if it were an string, this may happen 
	   in the future (I think XBEL allows the id to be an string) */
	g_free (uai);
}

void 
bookmarks_io_resolve_loaded_alias (BookmarkLoaderInfo *li)
{
	while (li->unresolved_aliases) 
	{
		UnresolvedAliasInfo *uai = li->unresolved_aliases->data;
		BookmarkItem *alias = uai->alias;
		BookmarkItem *real = g_hash_table_lookup (li->id_bookmark,
							  GINT_TO_POINTER 
							  (uai->alias_of_id));
		li->unresolved_aliases = g_list_remove 
			(li->unresolved_aliases, uai);
		bookmarks_io_unresolved_alias_info_free (uai);
		if (real)
		{
			bookmarks_init_alias (alias, real);
		}
		else
		{
			/* turn that bookmark into an error bookmark */
			GList *link = g_list_find (alias->parent->list, alias);
			BookmarkItem *eb = bookmarks_io_error_bookmarks_new
				(_("Alias Error"),
				 _("An error ocurred loading / pasting "
				   "bookmarks. There was an alias that "
				   "pointed to an invalid bookmark"));
			eb->parent = alias->parent;
			bookmarks_free_bookmark (alias);
			link->data = eb;
		}
	}
}

BookmarkItem *
bookmarks_io_error_bookmarks_new (const gchar *msg_title, 
				  const gchar *msg_notes)
{
	BookmarkItem *error_bookmark;
	gchar* eutfstr = mozilla_locale_to_utf8(msg_title);
	gchar* nutfstr = mozilla_locale_to_utf8(msg_notes);
	
	error_bookmark = bookmarks_new_bookmark 
		(BM_SITE, TRUE, eutfstr, "about:blank", NULL,
		 nutfstr, NULL);
	g_free(eutfstr);
	g_free(nutfstr);
	return error_bookmark;
}

FileFormat *
bookmarks_io_get_fileformats (void)
{
	FileFormat *ff = g_new0 (FileFormat, g_list_length 
				 (bookmarks_translators) + 1);
	GList *li;
	gint i;
	for (li = bookmarks_translators, i = 0; li != NULL; li = li->next, i++)
	{
		BookmarkTranslator *t = li->data;
		ff[i].description = t->format_name;
		ff[i].extensions = t->extensions;
		
	}
	return ff;
}

/*****************************************/

/**
 * Functions for modifying bookmarks 
 */
void
bookmark_set_name (BookmarkItem *b, const gchar *val)
{
	BookmarkItem *r = bookmarks_find_real_bookmark (b);
	r->name = g_strdup (val);
	bookmarks_editor_update_tree_items (r);
	bookmarks_update_alias (r);
}

void
bookmark_set_url (BookmarkItem *b, const gchar *val)
{
	BookmarkItem *r = bookmarks_find_real_bookmark (b);
	if (r->type != BM_SITE) return;
	auto_completion_remove_url (r->url);
	g_free (r->url);
	r->url = g_strdup (val);
	auto_completion_add_url (r->url);
	bookmarks_editor_update_tree_items (r);
	bookmarks_update_alias (r);
}

void
bookmark_set_smarturl (BookmarkItem *b, const gchar *val)
{
	BookmarkItem *r = bookmarks_find_real_bookmark (b);
	if (r->type != BM_SITE) return;
	g_free (r->smarturl);
	r->smarturl = g_strdup (val);
	bookmarks_update_alias (r);
}

void
bookmark_set_nick (BookmarkItem *b, const gchar *val)
{
	BookmarkItem *r = bookmarks_find_real_bookmark (b);
	if (r->type != BM_SITE) return;
	g_free (r->nick);
	r->nick = g_strdup (val);
	bookmarks_update_alias (r);
}

void
bookmark_set_notes (BookmarkItem *b, const gchar *val)
{
	BookmarkItem *r = bookmarks_find_real_bookmark (b);
	g_free (r->notes);
	r->notes = g_strdup (val);
	bookmarks_update_alias (r);
}

void
bookmark_set_pixmap (BookmarkItem *b, const gchar *val)
{
	BookmarkItem *r = bookmarks_find_real_bookmark (b);
	/* creating pixmaps is a bit expensive */
	if (r->pixmap_file && !strcmp (b->pixmap_file, val))
		return;
	g_free (r->pixmap_file);
	r->pixmap_file = g_strdup (val);
	
	if (r->pixmap_data)
		misc_gui_pixmap_data_free (r->pixmap_data);
	
	r->pixmap_data = misc_gui_pixmap_data_new_from_file (r->pixmap_file,
							     FALSE);
	
	bookmarks_update_alias (r);
}

void
bookmark_set_create_toolbar (BookmarkItem *b, gboolean val)
{
	b->create_toolbar = val;
}

void
bookmark_set_create_context_menu (BookmarkItem *b, gboolean val)
{
	b->create_context_menu = val;
}

void
bookmark_set_expanded (BookmarkItem *b, gboolean val)
{
	b->expanded = val;
}

void
bookmark_set_time_added (BookmarkItem *b, GTime val)
{
	b->time_added = val;
}

void
bookmark_set_time_modified (BookmarkItem *b, GTime val)
{
	BookmarkItem *r = bookmarks_find_real_bookmark (b);
	r->time_modified = val;
	bookmarks_update_alias (r);
}

void
bookmark_set_time_visited (BookmarkItem *b, GTime val)
{
	BookmarkItem *r = bookmarks_find_real_bookmark (b);
	r->time_visited = val;
	bookmarks_update_alias (r);
}

void
bookmark_set_toolbar_style (BookmarkItem *b, ToolbarStyle val)
{
	b->toolbar_style = val;
}

void
bookmark_set_accel (BookmarkItem *b, guint accel_key, 
		    GdkModifierType accel_mods)
{	
	BookmarkItem *r = bookmarks_find_real_bookmark (b);
	if (r->type != BM_SITE) return;
	r->accel_key = accel_key;
	r->accel_mods = accel_mods;
	bookmarks_update_alias (r);
}

void
bookmark_add_child (BookmarkItem *b, BookmarkItem *c, 
		    gint pos)
{
	BookmarkItem *r = bookmarks_find_real_bookmark (b);
	g_return_if_fail (BOOKMARK_ITEM_IS_FOLDER (r));
	bookmark_unparent (c);
	c->parent = b;
	b->list = g_list_insert (b->list, c, pos);	
	bookmarks_update_alias (r);
}

void 
bookmark_unparent (BookmarkItem *c)
{
	BookmarkItem *p = c->parent;
	BookmarkItem *rp;
	if (!p) return;
	rp = bookmarks_find_real_bookmark (p);
	rp->list = g_list_remove (rp->list, c);
	c->parent = NULL;
	bookmarks_update_alias (rp);
}


