/*
 * Copyright (c) 2005 Aleksander Piotrowski <aleksander.piotrowski@nic.com.pl>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#define GAIM_PLUGINS

#include "account.h"
#include "accountopt.h"
#include "debug.h"
#include "notify.h"
#include "request.h"
#include "server.h"
#include "util.h"
#include "version.h"
#include "xmlnode.h"
	
#include "tlen.h"
#include "chat.h"
#include "wb.h"

static GaimPlugin *my_protocol = NULL;

char *tlen_gender_list[] = {
	"Male or female",
	"Male",
	"Female",
};
const int tlen_gender_list_size = sizeof(tlen_gender_list) / sizeof(tlen_gender_list[0]);

TlenUserInfoElement tlen_user_template[] = {
	{"v",     "Show my details to others", TlenUIE_BOOL,   TlenUIE_RW, TlenUIE_DONTSHOW},
	{"first", "First name",                TlenUIE_STR,    TlenUIE_RW, TlenUIE_SHOW},
	{"last",  "Last name",                 TlenUIE_STR,    TlenUIE_RW, TlenUIE_SHOW},
	{"nick",  "Nickname",                  TlenUIE_STR,    TlenUIE_RW, TlenUIE_SHOW},
	{"email", "E-Mail address",            TlenUIE_STR,    TlenUIE_RW, TlenUIE_SHOW},
	{"c",     "City",                      TlenUIE_STR,    TlenUIE_RW, TlenUIE_SHOW},
	{"e",     "School",                    TlenUIE_STR,    TlenUIE_RW, TlenUIE_SHOW},
	{"s",     "Gender",                    TlenUIE_CHOICE, TlenUIE_RW, TlenUIE_SHOW},
	{"b",     "Birth year",                TlenUIE_STR,    TlenUIE_RW, TlenUIE_SHOW},
	{"j",     "Job",                       TlenUIE_INT,    TlenUIE_RO, TlenUIE_DONTSHOW},
	{"r",     "Looking for",               TlenUIE_INT,    TlenUIE_RO, TlenUIE_DONTSHOW},
	{"g",     "Voice",                     TlenUIE_INT,    TlenUIE_RO, TlenUIE_DONTSHOW},
	{"p",     "Plans",                     TlenUIE_INT,    TlenUIE_RO, TlenUIE_DONTSHOW}
};                                     

char *tlen_hash(const char *pass, const char *id);

/* Following two functions are from OpenBSD's ftp command.  Check:
 * http://www.openbsd.org/cgi-bin/cvsweb/src/usr.bin/ftp/fetch.c?rev=1.55&content-type=text/x-cvsweb-markup
 * for more information.
 *
 * Copyright (c) 1997 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Jason Thorpe and Luke Mewburn.
 * [...]
 */
char
hextochar(const char *str)
{
	char c, ret;

	c = str[0];
	ret = c;
	if (isalpha(c))
		ret -= isupper(c) ? 'A' - 10 : 'a' - 10;
	else
		ret -= '0';
	ret *= 16;

	c = str[1];
	ret += c;
	if (isalpha(c))
		ret -= isupper(c) ? 'A' - 10 : 'a' - 10;
	else
		ret -= '0';
	return ret;
}

static char *
urldecode(const char *str)
{
	char *ret, c;
	int i, reallen;

	if (str == NULL) {
		return NULL;
	}
	if ((ret = malloc(strlen(str)+1)) == NULL) {
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "urldecode: cannot malloc memory\n");
		return NULL;
	}
	for (i = 0, reallen = 0; str[i] != '\0'; i++, reallen++, ret++) {
		c = str[i];
		if (c == '+') {
			*ret = ' ';
			continue;
		}
		/* Can't use strtol here because next char after %xx may be
		 * a digit. */
		if (c == '%' && isxdigit(str[i+1]) && isxdigit(str[i+2])) {
			*ret = hextochar(&str[i+1]);
			i+=2;
			continue;
		}
		*ret = c;
	}
	*ret = '\0';

	return ret-reallen;
}

/* Stolen from libtlen sources */
static char *
urlencode(const char *msg)
{
	unsigned char *s;
	unsigned char *str;
	unsigned char *pos;

	str = calloc(1, 3 * strlen(msg) + 1);
	if (str == NULL ) {
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "cannot allocate memory for encoded string\n");
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- urlencode\n");

		return NULL;
	}

	s = (unsigned char *)msg;
	pos = str;

	while (*s != '\0') {
		if (*s == ' ') {
			*pos = '+';
			pos++;
		} else if ((*s < '0' && *s != '-' && *s != '.')
			|| (*s < 'A' && *s > '9')
			|| (*s > 'Z' && *s < 'a' && *s != '_')
			|| (*s > 'z')) {
			    sprintf((char *)pos, "%%%02X", *s);
			    pos += 3;
		} else {
			*pos = *s;
			pos++;
		}

		s++;
	}

	return (char *)str;
}

/* Converts msg from UTF to ISO */
static char *
fromutf(const char *msg)
{
	return g_convert(msg, strlen(msg), "ISO-8859-2", "UTF-8", NULL, NULL, NULL);
}

/* Converts msg from ISO to UTF */
static char *
toutf(const char *msg)
{
	return g_convert(msg, strlen(msg), "UTF-8", "ISO-8859-2", NULL, NULL, NULL);
}

char *
tlen_decode_and_convert(const char *str)
{
	char *decoded, *converted;

	if (str == NULL)
		return NULL;

	decoded = urldecode(str);
	if (decoded == NULL) {
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "tlen_decode_and_convert: unable to urldecode '%s'\n", str);
		return NULL;
	}

	converted = toutf(decoded);
	g_free(decoded);

	if (converted == NULL) {
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "tlen_decode_and_convert: unable to convert '%s'\n", decoded);
	}

	return converted;
}

char *
tlen_encode_and_convert(const char *str)
{
	char *encoded, *converted;

	if (str == NULL)
		return NULL;

	converted = fromutf(str);
	if (converted == NULL) {
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "tlen_encode_and_convert: unable to convert '%s'\n", str);
		return NULL;
	}

	encoded = urlencode(converted);
	g_free(converted);

	if (encoded == NULL) {
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "tlen_encode_and_convert: unable to urlencode '%s'\n", str);
	}

	return encoded;
}

static int
tlen_parse_subscription(const char *subscription)
{
	if (strcmp(subscription, "both") == 0) {
		return SUB_BOTH;
	} else if (strcmp(subscription, "none") == 0) {
		return SUB_NONE;
	} else if (strcmp(subscription, "to") == 0) {
		return SUB_TO;
	} else {
		return SUB_NONE;
	}
}

int
tlen_send(TlenSession *tlen, char *command)
{
	int ret;

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "-> tlen_send\n");

	if (tlen == NULL) {
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "tlen is NULL!\n");
	}

	if (tlen->fd < 0) {
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_send: tlen->fd < 0\n");
		return -1;
	}

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "query=%s\n", command);

	ret = write(tlen->fd, command, strlen(command));
	if (ret < 0) {
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "write(): %d, %s\n", errno, strerror(errno));
		gaim_connection_error(gaim_account_get_connection(tlen->account),
		                      _("Server has disconnected"));
	}

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_send: ret=%d\n", ret);

	return ret;
}

static void
tlen_parser_element_start(GMarkupParseContext *context,
	const char *element_name, const char **attrib_names,
	const char **attrib_values, gpointer user_data, GError **error)
{
	//xmlnode     *node;
	int          i, ret;
	//const char  *attribval;
	TlenSession *tlen = (TlenSession *) user_data;
	char         buf[TLEN_BUFSIZE];//, type[100], id[100];
	char        *hash;
	xmlnode     *xml;

	if (!element_name)
		return;

#if 0
	gaim_debug(GAIM_DEBUG_INFO, "tlen", "-> tlen_parser_element_start\n");

	/* Debugging */
	gaim_debug(GAIM_DEBUG_INFO, "tlen", "element_name=\"%s\"\n", element_name);
	for (i = 0; attrib_names[i]; i++) {
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "attrib_names[%d]=\"%s\", attrib_values[%d]=%s\n",
			i, attrib_names[i], i, attrib_values[i]);
	}
#endif

	/* Session start; tag <s> is closed at the end of session */
	if ((strcmp(element_name, "s") == 0) && !tlen->xml) {
		// XXX what if `i' is not here?
		for (i = 0; attrib_names[i]; i++) {
			if (strcmp(attrib_names[i], "i") == 0) {
				gaim_debug(GAIM_DEBUG_INFO, "tlen", "attrib_values[%d]=\"%s\"\n", i, attrib_values[i]);
				strncpy(tlen->session_id, attrib_values[i], sizeof(tlen->session_id) - 1);
				gaim_debug(GAIM_DEBUG_INFO, "tlen", "got session id=%s\n", tlen->session_id);

				/* Got session id?  Now it's time to authorize ourselves */
				gaim_connection_update_progress(tlen->gc, _("Authorizing"), 3, 4);

				hash = tlen_hash(tlen->password, tlen->session_id);
				gaim_debug(GAIM_DEBUG_INFO, "tlen", "hash=%s\n", hash);

				/* Free the password, zero it first */
				memset(tlen->password, 0, strlen(tlen->password));
				g_free(tlen->password);
				tlen->password = NULL;

				ret = snprintf(buf, sizeof(buf), TLEN_AUTH_QUERY,
					tlen->session_id, tlen->user, hash);
				free(hash);
				if (ret <= 0 || ret >= sizeof(buf)) {
					gaim_debug(GAIM_DEBUG_INFO, "tlen", "snprintf(): ret=%d\n", ret);
					return;
				}
				tlen_send(tlen, buf);
			}
		}
	} else {
		if (tlen->xml) {
			xml = xmlnode_new_child(tlen->xml, element_name);
		} else {
			xml = xmlnode_new(element_name);
		}

		for (i=0; attrib_names[i]; i++) {
			xmlnode_set_attrib(xml, attrib_names[i], attrib_values[i]);
		}

	       	tlen->xml = xml;
	}

	//gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_parser_element_start\n");
}

static char *
tlen_status2str(GaimPresence *presence) 
{
        if (gaim_presence_is_status_active(presence, "away")) {
                return UC_AWAY_DESCR;
        } else if (gaim_presence_is_status_active(presence, "available")) {
                return UC_AVAILABLE_DESCR;
        } else if (gaim_presence_is_status_active(presence, "chat")) {
                return UC_CHAT_DESCR;
        } else if (gaim_presence_is_status_active(presence, "dnd")) {
        	return UC_DND_DESCR;
        } else if (gaim_presence_is_status_active(presence, "xa")) {
        	return UC_XA_DESCR;
        } else {
        	return UC_UNAVAILABLE_DESCR;
        }
}


static void
tlen_list_emblems(GaimBuddy *b, const char **se, const char **sw, const char **nw, const char **ne)
{
	GaimPresence *presence = gaim_buddy_get_presence(b);
	TlenBuddy *tb = NULL;

	if (b->proto_data != NULL)
		tb = (TlenBuddy *) b->proto_data;
	
	gaim_debug(GAIM_DEBUG_INFO, "tlen", "-> tlen_list_emblems: %s, tb=%p\n", b->name, tb);
	
        if (!GAIM_BUDDY_IS_ONLINE(b)) {
		if (tb != NULL && tb->subscription != SUB_NONE)
                	*se = "offline";
		else
			*se = "notauthorized";
        } else if (gaim_presence_is_status_active(presence, "away")) {
                *se = "away";
        } else if (gaim_presence_is_status_active(presence, "available")) {
                *se = "online";
        } else if (gaim_presence_is_status_active(presence, "offline")) {
                *se = "offline";
        } else if (gaim_presence_is_status_active(presence, "chat")) {
                *se = "freeforchat";
        } else if (gaim_presence_is_status_active(presence, "dnd")) {
                *se = "dnd";
        } else if (gaim_presence_is_status_active(presence, "xa")) {
                *se = "extended_away";
        } else {
                *se = "offline";
        }

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_list_emblems: se=%s\n", *se);
}

static void
tlen_set_buddy_status(GaimAccount *account, GaimBuddy *buddy, xmlnode *presence)
{
	xmlnode *node;
	char *show, *desc = NULL, *st;

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "-> tlen_set_buddy_status: %s\n", buddy->name);

	show = (char *) xmlnode_get_attrib(presence, "type");
	if (!show) {
		node = xmlnode_get_child(presence, "show");
		if (!node) {
			gaim_debug(GAIM_DEBUG_INFO, "tlen", "presence change without show\n");
			gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_set_buddy_status\n");
			return;
		}
		show = xmlnode_get_data(node);
	}

	/* User has set status */
	node = xmlnode_get_child(presence, "status");
	if (node) {
		desc = xmlnode_get_data(node);
		if (desc) {
			desc = tlen_decode_and_convert(desc);
		}
	}

	if (strcmp(show, UC_AVAILABLE_TEXT) == 0) {
		st = "available";
	} else if (strcmp(show, UC_AWAY_TEXT) == 0) {
		st = "away";
	} else if (strcmp(show, UC_CHAT_TEXT) == 0) {
		st = "chat";
	} else if (strcmp(show, UC_XA_TEXT) == 0) {
		st = "xa";
	} else if (strcmp(show, UC_DND_TEXT) == 0) {
		st = "dnd";
	} else {
                st = "offline";
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "unknown status: %s\n", show);
        }

	gaim_debug_info("tlen", "st=%s\n", st);

	if (desc) {
		gaim_prpl_got_user_status(account, buddy->name, st, "message", desc, NULL);
		g_free(desc);
	} else
		gaim_prpl_got_user_status(account, buddy->name, st, NULL);

	gaim_debug_info("tlen", "<- tlen_set_buddy_status: desc=%s\n", desc ? desc : "NULL");
}

void
tlen_request_auth(GaimConnection *gc, char *name)
{
	char buf[256];
	TlenSession *tlen;

	tlen = gc->proto_data;

	snprintf(buf, sizeof(buf), TLEN_PRESENCE_SUBSCRIBE, name);
	tlen_send(tlen, buf);

}

static void
tlen_presence_authorize(TlenRequest *r)
{
	GaimBuddy *b;
	char buf[200];
	TlenSession *tlen;
	TlenBuddy *tb;

	tlen = r->gc->proto_data;

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "-> tlen_presence_authorize: r->from='%s'\n", r->from);

	snprintf(buf, sizeof(buf), TLEN_PRESENCE_ACCEPT, r->from);
	tlen_send(tlen, buf);

	b = gaim_find_buddy(r->gc->account, r->from);
	if (!b) {
		gaim_account_request_add(r->gc->account, r->from, NULL, NULL, NULL);
	} else {
		tb = b->proto_data;
		if (tb && tb->subscription == SUB_NONE) {
			tlen_request_auth(r->gc, r->from);
		}
	}

	g_free(r->from);
	g_free(r);

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_presence_authorize\n");
}

static void
tlen_presence_deny(TlenRequest *r)
{
	TlenSession *tlen;
	char buf[200];

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "-> tlen_presence_deny: r->from='%s'\n", r->from);

	tlen = r->gc->proto_data;

	/* First accept request and then ... */
	snprintf(buf, sizeof(buf), TLEN_PRESENCE_ACCEPT, r->from);
	tlen_send(tlen, buf);

	/* ... remove that buddy */
	snprintf(buf, sizeof(buf), TLEN_BUDDY_REMOVE, r->from);
	tlen_send(tlen, buf);

	g_free(r->from);
	g_free(r);

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_presence_deny\n");
}

static int
tlen_process_presence(TlenSession *tlen, xmlnode *xml)
{
	const char *from, *type;

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "-> tlen_process_presence\n");

	from = xmlnode_get_attrib(xml, "from");
	if (!from) {
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "<presence> without from\n");
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_process_presence\n");
		return 0;
	}

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "from=%s\n", from);

	type = xmlnode_get_attrib(xml, "type");
	gaim_debug(GAIM_DEBUG_INFO, "tlen", "type=%s\n", type ? type : "NONE");

	/* Subscribtion -- someone has agreed to add us to its buddy list */
	// <presence from='libtlendev@tlen.pl' type='subscribed'/>
	if (type && (strcmp(type, "subscribed") == 0)) {
		GaimBuddy *b;
		TlenBuddy *tb;

		b = gaim_find_buddy(tlen->gc->account, from);
		if (!b) {
			gaim_debug(GAIM_DEBUG_INFO, "tlen", "unknown buddy %s\n", from);
			gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_process_presence\n");
			return 0;
		}

		tb = b->proto_data;
		if (tb) {
			tb->subscription = SUB_TO;
		}
	/* Someone wants to add you to his buddy list */
	} else if (type && (strcmp(type, "subscribe") == 0)) {
		char *msg;
		TlenRequest *r;

		r = g_new0(TlenRequest, 1);
		r->gc = tlen->gc;
		r->from = g_strdup(from);

		msg = g_strdup_printf(_("The user %s wants to add you to their buddy list."), from);
		
                gaim_request_action(tlen->gc, NULL, msg, NULL, GAIM_DEFAULT_ACTION_NONE,
		    r, 2,
		    _("Authorize"), G_CALLBACK(tlen_presence_authorize),
		    _("Deny"), G_CALLBACK(tlen_presence_deny));

                g_free(msg);
	/* Status change */
	// <presence from='identyfikator@tlen.pl/t'><show>...</show><status>...</status></presence>
	} else {
		GaimBuddy *buddy;
		int size;

		buddy = gaim_find_buddy(tlen->gc->account, from);
		if (!buddy) {
			gaim_debug(GAIM_DEBUG_INFO, "tlen", "unknown buddy %s\n", from);
			gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_process_presence\n");
			return 0;
		}

		gaim_debug(GAIM_DEBUG_INFO, "tlen", "xml=%s\n", xmlnode_to_formatted_str(xml, &size));

		tlen_set_buddy_status(tlen->gc->account, buddy, xml);
	}

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_process_presence\n");

	return 0;
}

static int
tlen_process_message(TlenSession *tlen, xmlnode *xml)
{
	char *msg, *converted;
	const char *from, *stamp;
	xmlnode *body, *x, *wb;
	time_t sent = 0;

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "-> tlen_process_message\n");

	from = xmlnode_get_attrib(xml, "from");
	if (!from) {
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "message without 'from'\n");
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_process_message\n");
		return 0;
	}

	body = xmlnode_get_child(xml, "body");
	if (!body) {
		/* This could be a whiteboard message */
		wb = xmlnode_get_child(xml, "wb");
		if (wb == NULL) {
			gaim_debug(GAIM_DEBUG_INFO, "tlen", "message without a 'body'\n");
			gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_process_message\n");
		} else {
			tlen_wb_process(tlen, wb, from);
		}

		return 0;
	}
	
	msg = xmlnode_get_data(body);
	if (!msg) {
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "message with empty 'body'\n");
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_process_message\n");
		return 0;
	}

	converted = tlen_decode_and_convert(msg);
	gaim_debug(GAIM_DEBUG_INFO, "tlen", "msg=%s\n", converted);
	msg = g_markup_escape_text(converted, -1);
	g_free(converted);

	/* timestamp of an offline msg  */
	x = xmlnode_get_child(xml, "x");
	if (x) {
		stamp = xmlnode_get_attrib(x, "stamp");
		if (stamp) {
			sent = gaim_str_to_time(stamp, TRUE, NULL, NULL, NULL);
		}
	}

	serv_got_im(tlen->gc, from, msg, 0, sent == 0 ? time(NULL) : sent);


	gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_process_message\n");

	return 0;
}

/* This can either be a typing notification or a chat message */
static int
tlen_process_notification(TlenSession *tlen, xmlnode *xml)
{
	const char *from, *type;

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "-> tlen_process_notification\n");

	from = xmlnode_get_attrib(xml, "f");
	if (!from)
		return 0;

	/* Check if this is a chat message
       		<m f='261@c/~something' ...

	*/
	if (strstr(from, "@c") != NULL) {
		return tlen_chat_process_message(tlen, xml, from);
	}

	type = xmlnode_get_attrib(xml, "tp");
	if (!type)
		return 0;

	if (strcmp(type, "t") == 0) {
		 serv_got_typing(tlen->gc, from, 10, GAIM_TYPING);
	}
		
	gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_process_notification\n");

	return 0;
}

static void
tlen_pubdir_user_info(TlenSession *tlen, xmlnode *item)
{
	GaimNotifySearchResults *results;
	GaimNotifySearchColumn *column;
	int i;
	GList *row;
	xmlnode *node;
	char *decoded;

	gaim_debug_info("tlen", "-> tlen_pubdir_user_info\n");

	results = gaim_notify_searchresults_new();
	if (!results) {
		gaim_notify_error(tlen->gc, NULL,
			_("Unable to display public directory information."),
			NULL);
		return;
	}

        column = gaim_notify_searchresults_column_new(_("Tlen ID"));
        gaim_notify_searchresults_column_add(results, column);

	for (i = 0; i < sizeof(tlen_user_template)/sizeof(tlen_user_template[0]); i++) {
		if (tlen_user_template[i].display == TlenUIE_DONTSHOW)
			continue; 
		
		column = gaim_notify_searchresults_column_new(_(tlen_user_template[i].desc));
		gaim_notify_searchresults_column_add(results, column);
	}

	while (item) {
		row = NULL;

		row = g_list_append(row, (char *) xmlnode_get_attrib(item, "jid"));

		for (i = 0; i < sizeof(tlen_user_template)/sizeof(tlen_user_template[0]); i++) {
			if (tlen_user_template[i].display == TlenUIE_DONTSHOW)
				continue; 

			decoded = NULL;
			node = xmlnode_get_child(item, tlen_user_template[i].tag);

			if (!node) {
				gaim_debug_info("tlen", "%s -> %s (!node)\n", tlen_user_template[i].tag, "");
				row = g_list_append(row, "");
				continue;
			}
	
			if (tlen_user_template[i].format == TlenUIE_STR) {
				decoded = tlen_decode_and_convert(xmlnode_get_data(node));
			}

			gaim_debug_info("tlen", "%s -> %s\n", tlen_user_template[i].tag,
				decoded ? decoded : "NULL"); 

			if (strcmp(tlen_user_template[i].tag, "s") == 0) {
				int gender = atoi(xmlnode_get_data(node));
				if (gender < 0 || gender >= tlen_gender_list_size)
					gender = 0;

				row = g_list_append(row, _(tlen_gender_list[gender]));
			} else {
				row = g_list_append(row, decoded ? decoded : xmlnode_get_data(node));
			}
		}

		gaim_notify_searchresults_row_add(results, row);

		item = xmlnode_get_next_twin(item);
	}

	gaim_notify_searchresults(tlen->gc,
		_("Tlen.pl Public Directory"),
		_("Search results"), NULL, results,
		NULL, //(GaimNotifyCloseCallback)ggp_sr_close_cb,
		gaim_connection_get_account(tlen->gc));

#if 0
	if (item != NULL) {
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "item=%s\n", xmlnode_to_formatted_str(item, &size));
	}

	for (i = 0; i < sizeof(tlen_user_template)/sizeof(tlen_user_template[0]); i++) {
		if (!tlen_user_template[i].edit)
			continue; 

		intval = 0;
		nodeval = NULL;
		node = NULL;

		if (item != NULL) {
			node = xmlnode_get_child(item, tlen_user_template[i].tag);
			if (node) {
				nodeval = xmlnode_get_data(node);
				if (nodeval != NULL) {
					intval = atoi(nodeval);
					decoded = tlen_decode_and_convert(nodeval);
					gaim_debug_info("tlen", "%s == %s (%d)\n", tlen_user_template[i].desc, nodeval, intval);
				}
			}
		}

		if (strcmp(tlen_user_template[i].tag, "v") == 0) {
			field = gaim_request_field_bool_new(tlen_user_template[i].tag, 
					_(tlen_user_template[i].desc), nodeval ? intval : 1);
		} else if (strcmp(tlen_user_template[i].tag, "s") == 0) {
			field = gaim_request_field_choice_new(tlen_user_template[i].tag, 
					_(tlen_user_template[i].desc), intval);
			gaim_request_field_choice_add(field, _("Unknown"));
			gaim_request_field_choice_add(field, _("Male"));
			gaim_request_field_choice_add(field, _("Female"));
		} else if (strcmp(tlen_user_template[i].tag, "b") == 0) {
			field = gaim_request_field_int_new(tlen_user_template[i].tag, 
					_(tlen_user_template[i].desc), intval);
		} else {
			field = gaim_request_field_string_new(tlen_user_template[i].tag,
					_(tlen_user_template[i].desc), node ? decoded : "", FALSE);
		}

		nodeval = NULL;
		g_free(decoded);
		decoded = NULL;

		gaim_request_field_group_add_field(group, field);
	}

	gaim_request_fields(tlen->gc, _("Edit Tlen.pl public directory information"),
			_("Edit Tlen.pl public directory information"),
			_("All items below are optional."),
			fields,
			_("Save"), G_CALLBACK(tlen_pubdir_set_user_info),
			_("Cancel"), NULL,
			tlen->gc);
#endif
}

static GString *
tlen_pubdir_process_fields(GaimConnection *gc, GaimRequestFields *fields, int mode)
{
	GaimRequestField *field;
	const char *text = NULL;
	char buf[128], *encoded;
	int i, intval;
	GString *tubabuf;

	tubabuf = g_string_new("");

	for (i = 0; i < sizeof(tlen_user_template)/sizeof(tlen_user_template[0]); i++) {
		if (tlen_user_template[i].edit == TlenUIE_RO)
			continue;

                field = gaim_request_fields_get_field(fields, tlen_user_template[i].tag);

		if (mode == TlenUIE_MODE_SEARCH && strcmp(tlen_user_template[i].tag, "v") == 0)
			continue;

		if (mode == TlenUIE_MODE_SEARCH && strcmp(tlen_user_template[i].tag, "s") == 0
		    && gaim_request_field_choice_get_value(field) == 0)
			continue;

		switch (tlen_user_template[i].format) {
			case TlenUIE_STR:
				text  = gaim_request_field_string_get_value(field);
				break;
			case TlenUIE_CHOICE:
				intval = gaim_request_field_choice_get_value(field);
				goto handle_integer;
			case TlenUIE_INT:
				intval = gaim_request_field_int_get_value(field);
				goto handle_integer;
			case TlenUIE_BOOL:
				intval = gaim_request_field_bool_get_value(field);
handle_integer:
				snprintf(buf, sizeof(buf), "%d", intval);
				text = buf;
				break;
		}

		encoded = NULL;

		if (text) {
			encoded = tlen_encode_and_convert(text);
		}

		if (encoded && !(mode == TlenUIE_MODE_SEARCH && strlen(encoded) == 0)) {
			g_string_append_printf(tubabuf, "<%s>%s</%s>", tlen_user_template[i].tag,
					encoded, tlen_user_template[i].tag);
			g_free(encoded);
		}	

		gaim_debug_info("tlen", "%s -> %s\n", tlen_user_template[i].tag, text ? text : "NULL");
	}

	return tubabuf;
}

static void
tlen_pubdir_set_user_info(GaimConnection *gc, GaimRequestFields *fields)
{       
	TlenSession *tlen = gc->proto_data;
	GString *tubabuf;
	char *q, buf[512];

	gaim_debug_info("tlen", "-> tlen_pubdir_set_user_info\n");

	tubabuf = tlen_pubdir_process_fields(gc, fields, TlenUIE_MODE_EDIT);

	q = g_string_free(tubabuf, FALSE);
	snprintf(buf, sizeof(buf), "%s%s%s", TLEN_SET_PUBDIR_HEADER, q, TLEN_SET_PUBDIR_FOOTER);

	tlen_send(tlen, buf);

	g_free(q);
}

static void
tlen_pubdir_edit_user_info(TlenSession *tlen, xmlnode *item)
{
	xmlnode *node;
	char *nodeval, *decoded;
	int size, i, j, intval;
	GaimRequestFields *fields;
	GaimRequestFieldGroup *group;
	GaimRequestField *field;

	fields = gaim_request_fields_new();
	group = gaim_request_field_group_new(NULL);
	gaim_request_fields_add_group(fields, group);

	if (item != NULL) {
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "item=%s\n", xmlnode_to_formatted_str(item, &size));
	}

	for (i = 0; i < sizeof(tlen_user_template)/sizeof(tlen_user_template[0]); i++) {
		if (tlen_user_template[i].edit == TlenUIE_RO)
			continue; 

		intval = 0;
		nodeval = NULL;
		node = NULL;
		// XXX: does this fix the XXX below? decoded wasn't initialized!
		decoded = NULL;

		if (item != NULL) {
			node = xmlnode_get_child(item, tlen_user_template[i].tag);
			if (node) {
				nodeval = xmlnode_get_data(node);
				if (nodeval != NULL) {
					intval = atoi(nodeval);
					decoded = tlen_decode_and_convert(nodeval);
					gaim_debug_info("tlen", "%s == %s (%d)\n", tlen_user_template[i].desc, nodeval, intval);
				}
			}
		}

		if (strcmp(tlen_user_template[i].tag, "v") == 0) {
			field = gaim_request_field_bool_new(tlen_user_template[i].tag, 
					_(tlen_user_template[i].desc), nodeval ? intval : 1);
		} else if (strcmp(tlen_user_template[i].tag, "s") == 0) {
			field = gaim_request_field_choice_new(tlen_user_template[i].tag, 
					_(tlen_user_template[i].desc), intval);
			for (j = 0; j < tlen_gender_list_size; j++) {
				gaim_request_field_choice_add(field, _(tlen_gender_list[j]));
			}
#if 0
		} else if (strcmp(tlen_user_template[i].tag, "b") == 0) {
			field = gaim_request_field_int_new(tlen_user_template[i].tag, 
					_(tlen_user_template[i].desc), intval);
#endif
		} else {
			field = gaim_request_field_string_new(tlen_user_template[i].tag,
					_(tlen_user_template[i].desc), node ? decoded : "", FALSE);
		}

		nodeval = NULL;
		// XXX: On win32 freeing decoded leads to segfault
		g_free(decoded);

		gaim_request_field_group_add_field(group, field);
	}

	gaim_request_fields(tlen->gc, _("Edit Tlen.pl public directory information"),
			_("Edit Tlen.pl public directory information"),
			_("All items below are optional."),
			fields,
			_("Save"), G_CALLBACK(tlen_pubdir_set_user_info),
			_("Cancel"), NULL,
			tlen->gc);
}
		
static int
tlen_email_notification(TlenSession *tlen, xmlnode *xml)
{
	const char *from, *subject;
	char *from_decoded, *subject_decoded;
	GaimAccount *account = gaim_connection_get_account(tlen->gc);

	if (!gaim_account_get_check_mail(account))
		return 0;

	from = xmlnode_get_attrib(xml, "f");
	if (!from)
		return 0;
	from_decoded = tlen_decode_and_convert(from);

	subject = xmlnode_get_attrib(xml, "s");
	if (!subject) {
		g_free(from_decoded);
		return 0;
	}
	subject_decoded = tlen_decode_and_convert(subject);

	gaim_notify_email(tlen->gc, subject_decoded, from_decoded,
		gaim_account_get_username(tlen->account),
        	"http://poczta.o2.pl/", NULL, NULL);

	g_free(from_decoded);
	g_free(subject_decoded);

	return 0;
}

static void
tlen_set_away(GaimAccount *account, GaimStatus *status)
{
        GaimConnection *gc = gaim_account_get_connection(account);
        TlenSession *tlen = gc->proto_data;
        const char *status_id = gaim_status_get_id(status), *pattern;
	char *msg, *msg2 = NULL, buf[1024];

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "-> tlen_set_away\n");

        if (!gaim_status_is_active(status)) {
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_set_away\n");
                return;
	}

	/* Special "invisible" presence */
	if (strcmp(status_id, UC_INVISIBLE_TEXT) == 0) {
		tlen_send(tlen, TLEN_PRESENCE_INVISIBLE);
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_set_away\n");
		return;
	}
 
        msg = (char *) gaim_status_get_attr_string(status, "message");
	if (msg) {
		msg2 = fromutf(msg);
		if (msg2 == NULL) {
			gaim_debug(GAIM_DEBUG_INFO, "tlen", "tlen_set_away: can't convert msg\n");
			msg2 = g_strdup(msg);
		}

		msg = gaim_unescape_html(msg2);
		g_free(msg2);

		gaim_debug(GAIM_DEBUG_INFO, "tlen", "unescaped=%s\n", msg);

		msg2 = urlencode(msg);
		if (!msg2) {
			gaim_debug(GAIM_DEBUG_INFO, "tlen", "cannot urlencode away message\n");
			msg2 = g_strdup(msg);
		}
		g_free(msg);
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "encoded=%s\n", msg2);
	}

	if (msg2 != NULL)
		pattern = TLEN_PRESENCE_STATE;
	else
		pattern = TLEN_PRESENCE;

	snprintf(buf, sizeof(buf), pattern, status_id, msg2);
	g_free(msg2);

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "buf=%s\n", buf);

	tlen_send(tlen, buf);

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_set_away\n");
}

static int
tlen_process_iq(TlenSession *tlen, xmlnode *xml)
{
	const char *type, *id, *from;

	type = xmlnode_get_attrib(xml, "type");
       	id = xmlnode_get_attrib(xml, "id");
	from = xmlnode_get_attrib(xml, "from");

	if (!type) {
		return 0;
	}

	/* Process nodes from the chat server. We either have from='c' or
	 * from='123@c' */
	if (from != NULL && (strcmp(from, "c") == 0 || strstr(from, "@c") != NULL)) {
		return tlen_chat_process_iq(tlen, xml, type);
	}

	/* Yeah, sometimes id is not given ... */
	if (!id) {
		if (strcmp(type, "set") == 0) {
			xmlnode *query, *item;
			GaimBuddy *b;
			const char *subscription, *jid;
			TlenBuddy *tb;

			query = xmlnode_get_child(xml, "query");
			if (!query)
				return 0;

			item = xmlnode_get_child(query, "item");
			if (!item)
				return 0;
			
			subscription = xmlnode_get_attrib(item, "subscription");
			if (!subscription)
				return 0;

			jid = xmlnode_get_attrib(item, "jid");
			if (!jid)
				return 0;

			b = gaim_find_buddy(tlen->gc->account, jid);
			if (b) {
				tb = b->proto_data;
				if (tb) {
					tb->subscription = tlen_parse_subscription(subscription);
				}
			}
		}		
	/* Session stuff (login, authentication, etc.) */
	} else if (strncmp(id, tlen->session_id, strlen(tlen->session_id)) == 0) {
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "session stuff\n");
		/* We are authorized */
		if (strcmp(type, "result") == 0) {
			gaim_connection_set_state(tlen->gc, GAIM_CONNECTED);
			/* Getting user list */
			tlen_send(tlen, TLEN_GETROSTER_QUERY);
		/* Wrong usename or password */
		} else if (strcmp(type, "error") == 0) {
			gaim_connection_error(tlen->gc, _("Wrong password or username"));
		}
	/* Roster */
	} else if ((strncmp(id, "GetRoster", 9) == 0) && (strncmp(type, "result", 6) == 0)) {
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "roster stuff\n");
		xmlnode *query, *item;
		xmlnode *groupxml;
		GaimGroup *g, *tlen_group, *oldb_group;
		GaimBuddy *b;
		TlenBuddy *tb;
		int size;
		char *jid, *name, *subscription, *group;

		gaim_debug(GAIM_DEBUG_INFO, "tlen", "roster=\n%s\n", xmlnode_to_formatted_str(xml, &size));

		query = xmlnode_get_child(xml, "query");
		if (!query) {
			gaim_debug(GAIM_DEBUG_INFO, "tlen", "no query tag in GetRoster response\n");
			return 0;
		}

		/* Default group for Tlen.pl users.  If group "buddies" exists then use it.  Otherwise
		 * create our own "Tlen" group
		 */
		tlen_group = gaim_find_group("Buddies");
		if (!tlen_group) {
			tlen_group = gaim_find_group("Kontakty");
			if (!tlen_group) {
				tlen_group = gaim_find_group("Tlen");
				if (!tlen_group) {
					tlen_group = gaim_group_new("Tlen");
					gaim_blist_add_group(tlen_group, NULL);
				}
			}
                }

		/* Parsing buddies */
		for (item = xmlnode_get_child(query, "item"); item; item = xmlnode_get_next_twin(item)) {
			jid = (char *) xmlnode_get_attrib(item, "jid");
			if (!jid)
				continue;

			subscription = (char *) xmlnode_get_attrib(item, "subscription");
			if (!subscription)
				continue;

			/* Buddy name */
			name = (char *)xmlnode_get_attrib(item, "name");
			if (name)
				name = tlen_decode_and_convert(name);
			else
				name = g_strdup(jid);
			
			/* Group that buddy belongs to */
			groupxml = xmlnode_get_child(item, "group");
			if (groupxml != NULL) {
				group = (char *) xmlnode_get_data(groupxml);
				group = tlen_decode_and_convert(group);
			} else {
				group = NULL;
			}

			/* If roster entry doesn't have group set or group is "Kontakty" (default group used by original Tlen.pl client)
			 * ten put that buddy into our default group (Buddies/Tlen) pointed by tlen_group */
			if (group == NULL || strcmp(group, "Kontakty") == 0) {
				g = tlen_group;
			} else {
				g = gaim_find_group(group);
				if (g == NULL) {
					gaim_debug(GAIM_DEBUG_INFO, "tlen", "tlen_process_iq: adding new group '%s'\n", group);
					g = gaim_group_new(group);
					gaim_blist_add_group(g, NULL);
				}
			}

			b = gaim_find_buddy(tlen->gc->account, jid);
			if (b) {
				gaim_debug_info("tlen", "already have this buddy %p\n", b);

				/* Buddy from blist is in different group than roster says. Remove that buddy
				 * and add it once again */
				oldb_group = gaim_buddy_get_group(b);
				if (strcmp(oldb_group->name, g->name) != 0) {
					gaim_blist_remove_buddy(b);
					b = NULL;
				/* Different alias than one from blist?  Replace it with correct one from roster */
				} else if (b->alias && strcmp(b->alias, name) != 0) {
					gaim_blist_alias_buddy(b, name);
				} 
			}

			/* There is no such buddy (because that buddy is brand new or old one wasn't up-to-date */
			if (!b) {
				gaim_debug(GAIM_DEBUG_INFO, "tlen", "adding new user: %s\n", jid);
				b = gaim_buddy_new(tlen->gc->account, jid, name);
				gaim_blist_add_buddy(b, NULL, g, NULL);
			}

			if (b->proto_data == NULL) {
				gaim_debug_info("tlen", "adding tb\n");
				b->proto_data = g_new0(TlenBuddy, 1);
				tb = b->proto_data;
				tb->subscription = tlen_parse_subscription(subscription);
			}
			g_free(name);

			gaim_blist_alias_buddy(b, b->alias);
		}

		tlen->roster_parsed = 1;

		/* Set our status, erm presence */
		tlen_set_away(tlen->gc->account, gaim_presence_get_active_status(tlen->gc->account->presence));
	/* Pubdir info about myself */
	} else if ((strcmp(id, "tr") == 0) && (strcmp(type, "result") == 0)) {
		xmlnode *node, *item;

		node = xmlnode_get_child(xml, "query");
		if (!node)
			return 0;

		item = xmlnode_get_child(node, "item");

		tlen_pubdir_edit_user_info(tlen, item);
	/* Pubdir my details have been saved */
	} else if ((strcmp(id, "tw") == 0) && (strcmp(type, "result") == 0)) {
		gaim_notify_info(tlen->gc->account, _("Public directory ..."),
			_("Public directory information saved successfully!"), NULL);
	} else if ((strcmp(id, "src") == 0) && (strcmp(type, "get"))) {
		xmlnode *node, *item;

		node = xmlnode_get_child(xml, "query");
		if (!node)
			return 0;

		item = xmlnode_get_child(node, "item");

		tlen_pubdir_user_info(tlen, item);
	}

	return 0;
}

/* return 0 if data was parsed; otherwise returns -1 */
static int
tlen_process_data(TlenSession *tlen, xmlnode *xml)
{
	int ret = 0;

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "-> tlen_process_data\n");
	gaim_debug(GAIM_DEBUG_INFO, "tlen", "name=%s\n", xml->name);

	/* authorization, chat query responses */
	if (strncmp(xml->name, "iq", 2) == 0) {
		ret = tlen_process_iq(tlen, xml);
	} else if (strncmp(xml->name, "presence", 8) == 0) {
		ret = tlen_process_presence(tlen, xml);
	} else if (strncmp(xml->name, "message", 7) == 0) {
		ret = tlen_process_message(tlen, xml);
	} else if (strcmp(xml->name, "m") == 0) {
		ret = tlen_process_notification(tlen, xml);
	} else if (strcmp(xml->name, "n") == 0) {
		ret = tlen_email_notification(tlen, xml);
	} else if (strcmp(xml->name, "p") == 0) {
		ret = tlen_chat_process_p(tlen, xml);
	}

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_process_data\n");

	return ret;
}

static void
tlen_parser_element_end(GMarkupParseContext *context,
	const char *element_name, gpointer user_data, GError **error)
{
	TlenSession *tlen = user_data;
	//int ret;

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "-> tlen_parser_element_end\n");
	gaim_debug(GAIM_DEBUG_INFO, "tlen", "element_name=\"%s\"\n", element_name);

	if (tlen->xml == NULL) {
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "tlen->xml == NULL\n");
		return;
	}

	/* Is this a tag inside other tag? */
	if (tlen->xml->parent) {
                if(strcmp(tlen->xml->name, element_name) == 0)
                        tlen->xml = tlen->xml->parent;
        } else {
		tlen_process_data(tlen, tlen->xml);
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "tlen->xml=%p\n", tlen->xml);
		xmlnode_free(tlen->xml);
		tlen->xml = NULL;
        }

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_parser_element_end\n");
}

static void
tlen_parser_element_text(GMarkupParseContext *context, const char *text,
	gsize text_len, gpointer user_data, GError **error)
{
	TlenSession *tlen = user_data;

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "-> tlen_parser_element_text\n");
	gaim_debug(GAIM_DEBUG_INFO, "tlen", "text_len=%d text=\"%s\"\n", text_len, text);

	if (tlen->xml == NULL || text_len <= 0)
		return;

	xmlnode_insert_data(tlen->xml, text, text_len);

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_parser_element_text\n");
}

static GMarkupParser parser = {
	tlen_parser_element_start,
	tlen_parser_element_end,
	tlen_parser_element_text,
	NULL,
	NULL
};

static GList *
tlen_status_types(GaimAccount *account)
{
        GaimStatusType *type;
        GList *types = NULL;

	// gaim_debug(GAIM_DEBUG_INFO, "tlen", "-> tlen_status_types\n");

        type = gaim_status_type_new_with_attrs(GAIM_STATUS_AVAILABLE, "available",
                        _("Available"), TRUE, TRUE, FALSE, "message", _("Message"),
                        gaim_value_new(GAIM_TYPE_STRING), NULL);
        types = g_list_append(types, type);
                        
        type = gaim_status_type_new_with_attrs(GAIM_STATUS_AVAILABLE, "chat",
                        _("Chatty"), TRUE, TRUE, FALSE, "message", _("Message"),
                        gaim_value_new(GAIM_TYPE_STRING), NULL); 
        types = g_list_append(types, type);
                        
        type = gaim_status_type_new_with_attrs(GAIM_STATUS_AWAY, "away",
                        _("Away"), TRUE, TRUE, FALSE, "message", _("Message"),
                        gaim_value_new(GAIM_TYPE_STRING), NULL);
        types = g_list_append(types, type); 
                        
        type = gaim_status_type_new_with_attrs(GAIM_STATUS_EXTENDED_AWAY, "xa",
                        _("Extended Away"), TRUE, TRUE, FALSE, "message", _("Message"),
                        gaim_value_new(GAIM_TYPE_STRING), NULL);
        types = g_list_append(types, type);

        type = gaim_status_type_new_with_attrs(GAIM_STATUS_UNAVAILABLE, "dnd",
                        _("Do Not Disturb"), TRUE, TRUE, FALSE, "message", _("Message"),
                        gaim_value_new(GAIM_TYPE_STRING), NULL);
        types = g_list_append(types, type);

        type = gaim_status_type_new_with_attrs(GAIM_STATUS_INVISIBLE, "invisible",
                        _("Invisible"), TRUE, TRUE, FALSE, "message", _("Message"),
			gaim_value_new(GAIM_TYPE_STRING), NULL);
        types = g_list_append(types, type);

        type = gaim_status_type_new_with_attrs(GAIM_STATUS_OFFLINE, "offline",
                        _("Offline"), TRUE, TRUE, FALSE, "message", _("Message"),
                        gaim_value_new(GAIM_TYPE_STRING), NULL);
        types = g_list_append(types, type);
                        
	// gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_status_types\n");

	return types;
}

void
tlen_input_parse(GaimConnection *gc, const char *buf, int len)
{
	TlenSession *tlen = gc->proto_data;

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "-> tlen_input_parse\n");

	if (!g_markup_parse_context_parse(tlen->context, buf, len, NULL)) {
		g_markup_parse_context_free(tlen->context);
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "!g_markup_parse_context_parse\n");
		gaim_connection_error(gc, _("XML Parse error"));
		return;
        }

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_input_parse\n");
}

void
tlen_input_cb(gpointer data, gint source, GaimInputCondition cond)
{
	GaimConnection *gc = data;
	TlenSession    *tlen = gc->proto_data;
	/* char            buf[100]; //TLEN_BUFSIZE+1]; */
	char            buf[TLEN_BUFSIZE];
	int             len;
//	int             ret;

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "-> tlen_input_cb: fd=%d\n", tlen->fd);

	if (tlen->fd < 0) {
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "tlen->fd < 0");
		return;
	}

	len = read(tlen->fd, buf, sizeof(buf));
	if (len < 0) {
		gaim_connection_error(gc, _("Read error"));
		return;
	} else if (len == 0) {
		gaim_connection_error(gc, _("Server has disconnected"));
		return;
	}
	buf[len] = '\0';

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "data=%s\n", buf);

	tlen_input_parse(gc, buf, len);

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_input_cb()\n");
}

void
tlen_login_cb(gpointer data, gint source, const gchar *error_message)
{
	GaimConnection *gc = data;
	TlenSession    *tlen = gc->proto_data;

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "-> tlen_login_cb\n");

	if (source < 0) {
		gaim_connection_error(gc, _("Couldn't connect to host"));
		return;
	}
	
	fcntl(source, F_SETFL, 0);

	tlen->fd = source;

	gaim_connection_update_progress(tlen->gc, _("Starting session"), 2, 4);

	tlen_send(tlen, TLEN_LOGIN_QUERY);

	tlen->gc->inpa = gaim_input_add(tlen->fd, GAIM_INPUT_READ, tlen_input_cb, tlen->gc);

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_login_cb\n");
}


static void
tlen_keepalive(GaimConnection *gc)
{
	TlenSession *tlen = gc->proto_data;

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "-> tlen_keepalive\n");

	tlen_send(tlen, TLEN_KEEPALIVE);

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_keepalive\n");
}

static void
tlen_login(GaimAccount *account)
{
	GaimConnection *gc;
	TlenSession *tlen;
	GaimProxyConnectData *err;
	char *domainname;

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "-> tlen_login\n");

	gc = gaim_account_get_connection(account);
	gc->proto_data = g_new0(TlenSession, 1);

	tlen = gc->proto_data;

	tlen->user = g_strdup(gaim_account_get_username(account));
	if (!tlen->user) {
		gaim_connection_error(gc, _("Invalid Tlen.pl ID"));
		return;
	}

	domainname = strstr(tlen->user, "@tlen.pl");
	if (domainname) {
		gaim_connection_error(gc, _("Invalid Tlen.pl ID (please use just username without \"@tlen.pl\")"));
		return;
	}

	tlen->account = account;
	tlen->roster_parsed = 0;
	tlen->fd = -1;
	tlen->gc = gc;
	tlen->context = g_markup_parse_context_new(&parser, 0, tlen, NULL);
	tlen->password = g_strdup(gaim_account_get_password(account));

	/* Create a hash table for chat lookups */
	// XXX: pass a free func! or is it not needed, since the tlen_chat_leave
	// event will be fired up in tlen_close
	tlen->chat_hash = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, NULL);
	// XXX: a free func here too, l34kz!
	tlen->room_create_hash = g_hash_table_new_full(g_str_hash, g_str_equal, g_free, NULL);

	gaim_connection_update_progress(gc, _("Connecting"), 1, 4);

	err = gaim_proxy_connect(account, SERVER_ADDRESS, SERVER_PORT, tlen_login_cb, gc);
	if (!err || !gaim_account_get_connection(account)) {
		gaim_connection_error(gc, _("Couldn't create socket"));
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_login\n");
		return;
	}

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_login\n");
}


static void
tlen_close(GaimConnection *gc)
{
	TlenSession *tlen = gc->proto_data;

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "-> tlen_close\n");

	if (tlen == NULL || tlen->fd < 0) {
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_close\n");
		return;
	}

	tlen_send(tlen, "<presence type='unavailable'><show></show><status>Disconnected</status></presence>");
	tlen_send(tlen, "</s>");

	if (gc->inpa)
		gaim_input_remove(gc->inpa);

        close(tlen->fd);
	tlen->fd = -1;
        g_free(tlen->server);
        g_free(tlen->user);

	g_hash_table_destroy(tlen->chat_hash);
	g_hash_table_destroy(tlen->room_create_hash);

        g_free(tlen);

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_close\n");
}

static int
tlen_send_im(GaimConnection *gc, const char *who, const char *msg, GaimMessageFlags flags)
{
	TlenSession      *tlen = gc->proto_data;
	char             buf[4096], *tmp;
	int              r;
	char            *converted, *unescaped;

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "-> tlen_send_im\n");
	gaim_debug(GAIM_DEBUG_INFO, "tlen", "who=\"%s\", msg=\"%s\", flags=0x%x\n", who, msg, flags);


	converted = fromutf(msg);
	if (converted == NULL) {
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "cannot convert msg\n");
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_send_im\n");

		return 0;
	}

	unescaped = gaim_unescape_html(converted);
	g_free(converted);

	tmp = urlencode(unescaped);
	if (!tmp) {
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "cannot urlencode msg\n");
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_send_im\n");
		g_free(unescaped);

		return 0;
	}
	
	g_free(unescaped);

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "tmp=%s\n", tmp);

	/* Check if we want to send a private chatroom message */
	if (strstr(who, "@c") != NULL) {
		tlen_chat_send_privmsg(tlen, who, tmp);
		g_free(tmp);

		return 1;
	} else {
		r = snprintf(buf, sizeof(buf), TLEN_MESSAGE, who, tmp);
	}
	
	g_free(tmp);

	if (r <= 0 || r > sizeof(buf)) {
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "snprintf() error\n");
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_send_im\n");
		return 0;
	}

	r = tlen_send(tlen, buf);

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_send_im\n");

	return 1;
}

static void
tlen_add_buddy(GaimConnection *gc, GaimBuddy *b, GaimGroup *g)
{
	TlenSession *tlen;
	TlenBuddy *tb;
	char buf[250];

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "-> tlen_add_buddy\n");

	tlen = gc->proto_data;

	if (!tlen->roster_parsed) {
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "Roster hasn't been parsed yet.  Skipping add_buddy callback\n");
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_add_buddy\n");
		return;
	}

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "b=%p, b->proto_data=%p\n", b, b ? b->proto_data : NULL);

	if (!b->proto_data) {
		b->proto_data = g_new(TlenBuddy, 1);
		tb = b->proto_data;
		tb->subscription = SUB_NONE;
	}

	/* Adding to roster */
	if (g && g->name)
		snprintf(buf, sizeof(buf), TLEN_BUDDY_ADD, tlen->session_id, b->alias ? b->alias : b->name, b->name, g->name);
	else 
		snprintf(buf, sizeof(buf), TLEN_BUDDY_ADD_WOGRP, tlen->session_id, b->alias ? b->alias : b->name, b->name);
	gaim_debug(GAIM_DEBUG_INFO, "tlen", "buf=%s\n", buf);	
	tlen_send(tlen, buf);

	/* Asking for subscribtion */
	snprintf(buf, sizeof(buf), TLEN_PRESENCE_SUBSCRIBE, b->name);
	gaim_debug(GAIM_DEBUG_INFO, "tlen", "presence=%s\n", buf);	
	tlen_send(tlen, buf);

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_add_buddy\n");
}

static void
tlen_remove_buddy(GaimConnection *gc, GaimBuddy *b, GaimGroup *g)
{
	TlenSession *tlen;
	char buf[250];

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "-> tlen_remove_buddy\n");

	tlen = gc->proto_data;

	snprintf(buf, sizeof(buf), TLEN_PRESENCE_UNSUBSCRIBE, b->name);
	gaim_debug(GAIM_DEBUG_INFO, "tlen", "buf=%s\n", buf);	

	tlen_send(tlen, buf);

	snprintf(buf, sizeof(buf), TLEN_BUDDY_REMOVE, b->name);
	gaim_debug(GAIM_DEBUG_INFO, "tlen", "buf=%s\n", buf);	

	tlen_send(tlen, buf);

	/* XXX Free b->data ? */

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_remove_buddy\n");
}

void
tlen_alias_buddy(GaimConnection *gc, const char *who, const char *alias)
{
	int ret;
	char buf[4096];
	TlenSession *tlen;
	GaimBuddy *buddy;
	GaimGroup *group;
	char *encoded;

	tlen = gc->proto_data;

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "tlen_alias_buddy: who=%s, alias=%s\n", who, alias);

	buddy = gaim_find_buddy(tlen->gc->account, who);
	if (!buddy) {
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "cannot find budy %s\n", who);
		return;
	}

	group = gaim_buddy_get_group(buddy);

	/* User wants to remove an alias */
	if (alias == NULL) {
		ret = snprintf(buf, sizeof(buf), TLEN_BUDDY_UNALIAS, who, group->name);
		if (ret < 0 || ret >= sizeof(buf)) {
			gaim_debug(GAIM_DEBUG_INFO, "tlen", "tlen_alias_buddy: snprintf failed\n");
			return;
		}

		ret = tlen_send(tlen, buf);

		return;
	}

	encoded = tlen_encode_and_convert(alias);
	if (encoded == NULL) {
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "tlen_alias_buddy: can't encode alias\n");
		return;
	}

	ret = snprintf(buf, sizeof(buf), TLEN_BUDDY_SET, who, encoded, group->name);
	g_free(encoded);

	if (ret < 0 || ret >= sizeof(buf)) {
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "tlen_alias_buddy: snprintf failed\n");
		return;
	}

	tlen_send(tlen, buf);
}

static void
tlen_group_buddy(GaimConnection *gc, const char *who, const char *old_group, const char *new_group)
{
	GaimBuddy *buddy;
	TlenSession *tlen;
	char *group = NULL;
	char *alias = NULL;
	char buf[1024];
	int ret;

	tlen = gc->proto_data;

	buddy = gaim_find_buddy(tlen->gc->account, who);

	gaim_debug_info("tlen", "tlen_group_buddy: who=%s old_group=%s new_group=%s\n", who, old_group, new_group);

	group = tlen_encode_and_convert(new_group);
	if (group == NULL) {
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "tlen_group_buddy: can't encode group '%s'\n", new_group);
		return;
	}

	alias = tlen_encode_and_convert(buddy->alias);
	if (alias == NULL) {
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "tlen_group_buddy: can't encode alias '%s'\n", buddy->alias);
		goto end;
	}

	ret = snprintf(buf, sizeof(buf), TLEN_BUDDY_SET, who, alias, group);
	if (ret < 0 || ret >= sizeof(buf)) {
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "tlen_group_buddy: snprintf failed\n");
		goto end;
	}

	ret = tlen_send(tlen, buf);
	if (ret < 0) {
		gaim_debug(GAIM_DEBUG_INFO, "tlen", "tlen_group_buddy: tlen_send failed\n");
		goto end;
	}
end:
	g_free(group);
	g_free(alias);
}

static void
tlen_tooltip_text(GaimBuddy *b, GString *ret, gboolean full)
{
	GaimStatus *status;
	GaimPresence *presence;
	const char *tmp;
	char *tmp2;
	TlenBuddy *tb;

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "-> tlen_tooltip_text: %s\n", b->name);

	if (full) {
		g_string_append_printf(ret, "\n<b>%s:</b> ", _("Subscription"));

		tb = (TlenBuddy *) b->proto_data;

		if (!tb) {
			g_string_append(ret, _("Unknown"));
		} else if (tb->subscription == SUB_BOTH) {
			g_string_append(ret, _("Both"));
		} else if (tb->subscription == SUB_NONE) {
			g_string_append(ret, _("None"));
		} else if (tb->subscription == SUB_TO) {
			g_string_append(ret, _("To"));
		} else  {
			g_string_append(ret, _("Unknown"));
		}
	}

	presence = gaim_buddy_get_presence(b);
	g_string_append_printf(ret, "\n<b>Status:</b> %s", _(tlen_status2str(presence)));

        status = gaim_presence_get_active_status(presence);
	tmp = gaim_status_get_attr_string(status, "message");
	if (tmp && strlen(tmp) > 0) {
		tmp2 = g_markup_escape_text(tmp, -1);
		g_string_append_printf(ret, ": %s", tmp2);
		g_free(tmp2);
	}

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_tooltip_text: %s\n", ret->str);
}

static const char *
tlen_list_icon(GaimAccount *a, GaimBuddy *b)
{
	gaim_debug_info("tlen", "tlen_list_icon\n");
	return "tlen";
}

static void
tlen_buddy_rerequest_auth(GaimBlistNode *node, gpointer data)
{
	GaimBuddy *b;
	GaimConnection *gc;

	if (!GAIM_BLIST_NODE_IS_BUDDY(node)) {
		return;
	}

	b = (GaimBuddy *) node;
	gc = gaim_account_get_connection(b->account);

	/* Asking for subscribtion */
	tlen_request_auth(gc, b->name);
}

static GList *
tlen_blist_node_menu(GaimBlistNode *node)
{
        GaimConnection *gc;
	TlenBuddy *tb;
	GaimBuddy *b;
	GaimMenuAction *act;
	GList *m = NULL;

	if (GAIM_BLIST_NODE_IS_BUDDY(node)) {
		b = (GaimBuddy *) node;
		tb = (TlenBuddy *) b->proto_data;
		gc = gaim_account_get_connection(b->account);

		act = gaim_menu_action_new(_("Start a conference"), GAIM_CALLBACK(tlen_chat_start_conference), gc, NULL);
		m = g_list_append(m, act);
		
        	if (!tb || tb->subscription == SUB_NONE || tb->subscription == SUB_TO) {
                	act = gaim_menu_action_new(_("(Re-)Request authorization"),
				GAIM_CALLBACK(tlen_buddy_rerequest_auth),
				NULL, NULL);
			m = g_list_append(m, act);
		}

		act = gaim_menu_action_new(_("Whiteboard"), GAIM_CALLBACK(tlen_wb_send_request), gc, NULL);
		m = g_list_append(m, act);
	}

        return m;
}

static unsigned int
tlen_send_typing(GaimConnection *gc, const char *who, GaimTypingState typing)
{
	TlenSession *tlen;
	char buf[100];

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "-> tlen_send_typing: who=%s typing=%d\n", who, typing);

	tlen = gc->proto_data;

	/* t - starts typing
	   u - stops typing
	 */
	snprintf(buf, sizeof(buf), TLEN_NOTIF_TYPING, who, typing == GAIM_TYPING ? 't' : 'u');

	tlen_send(tlen, buf);

	gaim_debug(GAIM_DEBUG_INFO, "tlen", "<- tlen_send_typing\n");

	return 0;
}

static void
tlen_pubdir_find_buddies_cb(GaimConnection *gc, GaimRequestFields *fields)
{
	TlenSession *tlen = gc->proto_data;
	GString *tubabuf;
	char *q, buf[512];

	tubabuf = tlen_pubdir_process_fields(gc, fields, TlenUIE_MODE_SEARCH);

	q = g_string_free(tubabuf, FALSE);
	snprintf(buf, sizeof(buf), "%s%s%s", TLEN_SEARCH_PUBDIR_HEADER, q, TLEN_SEARCH_PUBDIR_FOOTER);

	tlen_send(tlen, buf);

	g_free(q);
}

static void
tlen_pubdir_find_buddies(GaimPluginAction *action)
{
	GaimConnection *gc = (GaimConnection *) action->context;
	//TlenSession *tlen = gc->proto_data;
	int i, j;

	GaimRequestFields *fields;
	GaimRequestFieldGroup *group;
	GaimRequestField *field;

	gaim_debug_info("tlen", "tlen_pubdir_find_buddies\n");

	fields = gaim_request_fields_new();
	group = gaim_request_field_group_new(NULL);

	for (i = 0; i < sizeof(tlen_user_template)/sizeof(tlen_user_template[0]); i++) {
		if (tlen_user_template[i].display == TlenUIE_DONTSHOW)
			continue; 
		
		if (strcmp(tlen_user_template[i].tag, "s") == 0) {
			field = gaim_request_field_choice_new(tlen_user_template[i].tag, 
					_(tlen_user_template[i].desc), 0);
			for (j = 0; j < tlen_gender_list_size; j++) {
				gaim_request_field_choice_add(field, _(tlen_gender_list[j]));
			}
		} else {
			field = gaim_request_field_string_new(tlen_user_template[i].tag,
					_(tlen_user_template[i].desc), "", FALSE);
		}

		//field = gaim_request_field_string_new(tlen_user_template[i].tag, _(tlen_user_template[i].desc), NULL, FALSE);
		gaim_request_field_group_add_field(group, field);
	}

	gaim_request_fields_add_group(fields, group);

	gaim_request_fields(gc,
		_("Find buddies"),
		_("Find buddies"),
		_("Please, enter your search criteria below"),
		fields,
		_("OK"), G_CALLBACK(tlen_pubdir_find_buddies_cb),
		_("Cancel"), NULL,
		gc);

	gaim_debug_info("tlen", "tlen_pubdir_find_buddies\n");
}

static void
tlen_get_info(GaimConnection *gc, const char *name)
{
	TlenSession *tlen = gc->proto_data;	
	char buf[256], *namecpy, *tmp;

	namecpy = strdup(name);
	tmp = strchr(namecpy, '@');
	if (tmp) {
		*tmp = '\0';
	}

	snprintf(buf, sizeof(buf), "%s<i>%s</i>%s", TLEN_SEARCH_PUBDIR_HEADER, namecpy, TLEN_SEARCH_PUBDIR_FOOTER);
	tlen_send(tlen, buf);

	free(namecpy);
}

static char *
tlen_status_text(GaimBuddy *b)
{
        GaimStatus *status;
	TlenBuddy *tb;
        const char *msg;
        char *text = NULL;
        char *tmp;

	tb = (TlenBuddy *) b->proto_data;

	gaim_debug_info("tlen", "-> tlen_status_text: %s tb=%p\n", b->name, tb);

	if (!tb || (tb && tb->subscription == SUB_NONE)) {
		text = g_strdup(_("Not Authorized"));
	} else {
		status = gaim_presence_get_active_status(gaim_buddy_get_presence(b));
		msg = gaim_status_get_attr_string(status, "message");
		if (msg != NULL) {
			tmp = gaim_markup_strip_html(msg);
			text = g_markup_escape_text(tmp, -1);
			g_free(tmp);
		} else if (!gaim_status_is_available(status)) {
			tmp = g_strdup(gaim_status_get_name(status));
			text = g_markup_escape_text(tmp, -1);
			g_free(tmp);
		}
	}

	gaim_debug_info("tlen", "<- tlen_status_text: ret=%s\n", text ? text : "NULL");

	return text;
}

static void
tlen_pubdir_get_user_info(GaimPluginAction *action)
{       
        GaimConnection *gc = (GaimConnection *) action->context;
	TlenSession *tlen = gc->proto_data;

	tlen_send(tlen, TLEN_GET_PUBDIR_MYSELF);
}

static GaimPluginProtocolInfo prpl_info =
{
	OPT_PROTO_CHAT_TOPIC | OPT_PROTO_MAIL_CHECK | OPT_PROTO_UNIQUE_CHATNAME,
	NULL,			/* user_splits */
	NULL,			/* protocol_options */
	NO_BUDDY_ICONS,		/* icon_spec */
	tlen_list_icon,		/* list_icon */
	tlen_list_emblems,	/* list_emblems */
	tlen_status_text,	/* status_text */
	tlen_tooltip_text,	/* tooltip_text */
	tlen_status_types,	/* status_types */
	tlen_blist_node_menu,	/* blist_node_menu */
	tlen_chat_info,		/* chat_info */
	tlen_chat_info_defaults,/* chat_info_defaults */
	tlen_login,		/* login */
	tlen_close,		/* close */
	tlen_send_im,		/* send_im */
	NULL,			/* set_info */
	tlen_send_typing,	/* send_typing */
	tlen_get_info,		/* get_info */
	tlen_set_away,		/* set_away */
	NULL,			/* set_idle */
	NULL,			/* change_passwd */
	tlen_add_buddy,		/* add_buddy */
	NULL,			/* add_buddies */
	tlen_remove_buddy, 	/* remove_buddy */
	NULL,			/* remove_buddies */
	NULL,			/* add_permit */
	NULL,			/* add_deny */
	NULL,			/* rem_permit */
	NULL,			/* rem_deny */
	NULL,			/* set_permit_deny */
	tlen_join_chat,		/* join_chat */
	NULL,			/* reject_chat */
	NULL,			/* get_chat_name */
	tlen_chat_invite,	/* chat_invite */
	tlen_chat_leave,	/* chat_leave */
	tlen_chat_whisper,	/* chat_whisper */
	tlen_chat_send,		/* chat_send */
	tlen_keepalive,		/* keepalive */
	NULL,			/* register_user */
	NULL,			/* get_cb_info */
	NULL,			/* get_cb_away */
	tlen_alias_buddy,	/* alias_buddy */
	tlen_group_buddy,	/* group_buddy */
	NULL,			/* rename_group */
	NULL,			/* buddy_free */
	NULL,			/* convo_closed */
	NULL,			/* normalize */
	NULL,			/* set_buddy_icon */
	NULL,			/* remove_group */
	tlen_chat_get_cb_real_name, /* get_cb_real_name */
	NULL,			/* set_chat_topic */
	NULL,			/* find_blist_chat */
	tlen_roomlist_get_list,	/* roomlist_get_list */
	tlen_roomlist_cancel,	/* roomlist_cancel */
	tlen_roomlist_expand_category, /* roomlist_expand_category */
	NULL,			/* can_receive_file */
	NULL,			/* send_file */
	NULL,			/* new xfer */
        NULL,			/* offline_message */
        &tlen_wb_ops		/* whiteboard_prpl_ops */
};

static GList *
tlen_actions(GaimPlugin *plugin, gpointer context)
{
	GList *list = NULL;
	GaimPluginAction *act;

	act = gaim_plugin_action_new(_("Set user info..."), tlen_pubdir_get_user_info);
	list = g_list_append(list, act);

	act = gaim_plugin_action_new(_("Find buddies..."), tlen_pubdir_find_buddies);
	list = g_list_append(list, act);

	return list;
}

static GaimPluginInfo info =
{
	GAIM_PLUGIN_MAGIC,
	GAIM_MAJOR_VERSION,
	GAIM_MINOR_VERSION,
	GAIM_PLUGIN_PROTOCOL,                        /* type           */
	NULL,                                        /* ui_requirement */
	0,                                           /* flags          */
	NULL,                                        /* dependencies   */
	GAIM_PRIORITY_DEFAULT,                       /* priority       */

	"prpl-tlen",                                 /* id             */
	"Tlen.pl",                                   /* name           */
	TLEN_VERSION,                                /* version        */

	N_("Tlen.pl Protocol Plugin"),               /* summary        */
	N_("The Tlen.pl Protocol Plugin"),           /* description    */
	"Aleksander Piotrowski <alek@nic.com.pl>",   /* author         */
	"http://nic.com.pl/~alek/gaim-tlen",         /* homepage       */

	NULL,                                        /* load           */
	NULL,                                        /* unload         */
	NULL,                                        /* destroy        */

	NULL,                                        /* ui_info        */
	&prpl_info,                                  /* extra_info     */
	NULL,                                        /* prefs_info     */
	tlen_actions                                 /* actions        */
};

static void
init_plugin(GaimPlugin *plugin)
{
	printf("tlen: init_plugin\n");
	my_protocol = plugin;
}

GAIM_INIT_PLUGIN(tlen, init_plugin, info);
