/*
 * imgsav_dlg.c
 *
 * Code for the image save parameters dialog
 *
 * (C) 1997 Randall Hopper
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met: 1. Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer. 2.
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

/*      ******************** Include Files                ************** */

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/Toggle.h>
#include "tvdefines.h"
#include "glob.h"
#include "actions.h"
#include "xutil.h"
#include "imgsav_dlg.h"

/*      ******************** Local defines                ************** */

#define PIX_FNAME_SELECT   "tickbox_on.xbm"
#define PIX_FNAME_UNSELECT "tickbox_off.xbm"

/*      ******************** Private variables            ************** */

static Widget Dialog_wgt = NULL,
              Text_wgt   = NULL,
              Suffix_wgt = NULL,
              Menu_btn   = NULL;
/*
static Pixmap Select_pixmap   = None,
              Unselect_pixmap = None;
*/

static TV_IFILEFMT_ITEM_DEF Fmtitem_def[] = {
    { TV_STILL_FMT_TIFF   ,  "tiff", "tiffCmd" },
    { TV_STILL_FMT_PPM    ,  "ppm" , "ppmCmd"  }
};

#define FBTEXT_ACCEPT_ACTION "tv-freezefilebase-accept"

static void FileBaseTextAcceptAction( Widget, XEvent *, String *, Cardinal *);

static XtActionsRec Tool_actions[1] = {
           { FBTEXT_ACCEPT_ACTION, FileBaseTextAcceptAction }
};

static char Filebase_text_transl[] =  "\
<Key>Return:   select-all() beginning-of-line() "FBTEXT_ACCEPT_ACTION"()\n\
<Key>KP_Enter: select-all() beginning-of-line() "FBTEXT_ACCEPT_ACTION"()\n\
<Key>Linefeed: select-all() beginning-of-line() "FBTEXT_ACCEPT_ACTION"()\n\
Ctrl<Key>J:    select-all() beginning-of-line() "FBTEXT_ACCEPT_ACTION"()\n\
Ctrl<Key>M:    select-all() beginning-of-line() "FBTEXT_ACCEPT_ACTION"()\n\
";
/*      ******************** Forward declarations         ************** */
/*      ******************** Function Definitions         ************** */

#ifdef OLDSTUFF

static void LoadRadioPixmaps( void )
{
    /*  Load select/unselect pixmaps if we haven't already  */
    if ( Select_pixmap == None ) {
        if ( !XUTILBitmapLoad( PIX_FNAME_SELECT, TVTOPLEVEL,
                               &Select_pixmap ) ) {
            fprintf( stderr, "Can't load '%s' pixmap\n", PIX_FNAME_SELECT );
            exit(1);
        }
        if ( !XUTILBitmapLoad( PIX_FNAME_UNSELECT, TVTOPLEVEL,
                               &Unselect_pixmap ) ) {
            fprintf( stderr, "Can't load '%s' pixmap\n", PIX_FNAME_UNSELECT );
            exit(1);
        }
    }
}

#endif


static void SetToggleState( Widget toggle_wgt, TV_BOOL state )
{
    /*Pixmap pix = state ? Select_pixmap : Unselect_pixmap;*/

    /*LoadRadioPixmaps();*/

    XtVaSetValues( toggle_wgt, /*XtNbitmap, pix,*/
                               XtNstate,  state,
                               NULL);
}


/*  OkCmdCB - Dismiss the dialog  */
static void OkCmdCB( Widget w, XtPointer cl, XtPointer cb )
{
    TV_DISK *d = &G_glob.disk;
    Boolean  state;
    String   fn_base;
    TV_BOOL  ok = TRUE;
    TV_INT32 i;
    String   fmt_label,
             fmt_label2;

    if ( Dialog_wgt == NULL )
        return;

    /*  Grab values off dialog  */

    /*  ...Filename base  */
    XtVaGetValues( Text_wgt, XtNstring, &fn_base,
                             NULL );
    if ( fn_base == NULL )
        fn_base = "";
    if ( strlen( fn_base ) == 0 )
        ok = FALSE;

    d->fn_freeze_base[0] = '\0';
    strncat( d->fn_freeze_base, fn_base, sizeof( d->fn_freeze_base ) - 1 );

    /*  ...Do Suffix Y/N  */
    XtVaGetValues( Suffix_wgt, XtNstate,  &state,
                               NULL);
    d->freeze_use_suffix = state;

    /*  ...File Format  */
    XtVaGetValues( Menu_btn, XtNlabel, &fmt_label,
                             NULL );
    for ( i = 0; i < XtNumber( Fmtitem_def ); i++ ) {
        XtVaGetValues( Fmtitem_def[i].wgt, XtNlabel, &fmt_label2, 
                                           NULL );
        if ( strcmp( fmt_label, fmt_label2 ) == 0 )
            break;
    }
    if ( i >= XtNumber( Fmtitem_def ) ) {
        fprintf( stderr, 
           "TVIMGSAV:OkCmdCB: Format string not found '%s'\n", 
           fmt_label ? fmt_label : "" );
        exit(1);
    }
    d->freeze_fmt = Fmtitem_def[i].fmt;

    /*  Reset suffix number  */
    d->freeze_next_suffix = 1;

    /*  Get rid of dialog  */
    XtPopdown( Dialog_wgt );

    /*  And save the image (or just beep if no filename)  */
    if ( ok )
        TVActionSaveImage();
    else
        XBell( TVDISPLAY, 100 );
}


/*  FileBaseTextAcceptAction - Action rtn to accept new Text wgt file base  */
static void FileBaseTextAcceptAction( 
                Widget    wgt,
                XEvent   *xevent,
                String   *params,
                Cardinal *num_params )
{
    /*  Treat this as a dialog dismiss  */
    OkCmdCB( NULL, NULL, NULL );
}

static void TextValUpdate( Widget text_wgt, char *str )
{
    XawTextBlock     tblk;
    char            *old_str;
    int              old_len;

    assert( text_wgt != NULL );

    memset( &tblk, '\0', sizeof( tblk ) );
    tblk.firstPos = 0;
    tblk.length   = strlen( str );
    tblk.ptr      = str;
    tblk.format   = XawFmt8Bit;

    XtVaGetValues( text_wgt, XtNstring, &old_str, 
                             NULL );
    old_len = (old_str == NULL) ? 0 : strlen( old_str );
    XawTextReplace( text_wgt, 0, old_len, &tblk );
}

/*  FmtMenuCB - Update menu button text  */
static void FmtMenuItemCB( Widget w, XtPointer cl, XtPointer cb )
{
    TV_INT32 i;
    String   label,
             fn_base,
             p;
    char     path[ MAXPATHLEN ];

    for ( i = 0; i < XtNumber( Fmtitem_def ); i++ )
        if ( Fmtitem_def[i].wgt == w )
            break;
    if ( i >= XtNumber( Fmtitem_def ) ) {
        fprintf( stderr, 
           "TVIMGSAV:FmtMenuItemCB: Can't find widget\n" );
        exit(1);
    }

    XtVaGetValues( Fmtitem_def[i].wgt, XtNlabel, &label, 
                                       NULL );
    XtVaSetValues( Menu_btn, XtNlabel, label,
                             NULL );

    /*  Tweak the extension of the on-dialog filename, as a convenience  */
    XtVaGetValues( Text_wgt, XtNstring, &fn_base,
                             NULL );
    if ( fn_base == NULL )
        fn_base = "";
    if ( (p = strrchr( fn_base, '/' )) == NULL )
        p = fn_base;
    if ( (p = strchr( p, '.' )) != NULL ) {
        path[0] = '\0';
        strncat( path, fn_base, MIN( sizeof(path)-1, p-fn_base+1 ) );
        strncat( path, Fmtitem_def[i].file_ext,
                 sizeof(path)-1 - strlen(path) );
        TextValUpdate( Text_wgt, path );
    }
}

static void TVImgSavDialogBuild( Widget *dialog_wgt )
{
    Position  x, y;
    Dimension width;
    Widget    w, box, gbox, cbox, form, menu_shell;
    TV_INT32  i;
    XtTranslations   transl;

    /*  Create the dialog widgets  */
    *dialog_wgt = XtVaCreatePopupShell( "imageSaveDialog",
                      transientShellWidgetClass, TVTOPLEVEL,
                      NULL );

    box = XtVaCreateManagedWidget( "mainBox", boxWidgetClass, *dialog_wgt, 
                                   XtNorientation, XtorientVertical,
                                   NULL );

    gbox = XtVaCreateManagedWidget( "groupBox", boxWidgetClass, box, 
                                   XtNorientation, XtorientVertical,
                                   NULL );

    cbox = XtVaCreateManagedWidget( "fnameBox", boxWidgetClass, gbox, 
                                   XtNorientation, XtorientHorizontal,
                                   NULL );

    w = XtVaCreateManagedWidget( "fileBaseLabel", labelWidgetClass, cbox,
                                 NULL );

    Text_wgt = XtVaCreateManagedWidget( "fileBaseText", asciiTextWidgetClass, 
                                       cbox,
                                       XtNtype            , XawAsciiString,
                                       XtNuseStringInPlace, False,
                                       XtNscrollHorizontal, XawtextScrollNever,
                                       XtNscrollVertical  , XawtextScrollNever,
                                       XtNdisplayCaret    , False,
                                       XtNeditType        , XawtextEdit,
                                       XtNresize          , XawtextResizeNever,
                                       NULL );

    /*  Text widget translation overrides  */
    XtAppAddActions( TVAPPCTX, Tool_actions, XtNumber( Tool_actions ) );
    transl = XtParseTranslationTable( G_transl_ovr_ascii_text );
    XtOverrideTranslations( Text_wgt, transl );
    transl = XtParseTranslationTable( G_transl_ovr_ascii_text_1line );
    XtOverrideTranslations( Text_wgt, transl );
    transl = XtParseTranslationTable( Filebase_text_transl );
    XtOverrideTranslations( Text_wgt, transl );

    form = XtVaCreateManagedWidget( "suffixForm", formWidgetClass, gbox, 
                                   XtNorientation, XtorientHorizontal,
                                   NULL );

    w = XtVaCreateManagedWidget( "spacerLabel", labelWidgetClass, form,
                                 XtNlabel, " ", 
                                 XtNresize, False,
                                 XtNencoding, XawTextEncoding8bit,
                                 XtNwidth, 70,
                                 NULL );

    Suffix_wgt = XtVaCreateManagedWidget( "suffixToggle", toggleWidgetClass, 
                                          form, 
                                          XtNfromHoriz, w,
                                          XtNlabel, " ", 
                                          NULL );

    w = XtVaCreateManagedWidget( "suffixLabel", labelWidgetClass, form,
                                 XtNresize, False,
                                 XtNfromHoriz, Suffix_wgt,
                                 XtNencoding, XawTextEncoding8bit,
                                 NULL );

    cbox = XtVaCreateManagedWidget( "formatBox", boxWidgetClass, gbox, 
                                   XtNorientation, XtorientHorizontal,
                                   NULL );


    w = XtVaCreateManagedWidget( "spacerLabel", labelWidgetClass, cbox,
                                 XtNlabel, " ", 
                                 XtNresize, False,
                                 XtNencoding, XawTextEncoding8bit,
                                 XtNwidth, 43,
                                 NULL );

    w = XtVaCreateManagedWidget( "formatLabel", labelWidgetClass, cbox,
                                 XtNresize, False,
                                 XtNencoding, XawTextEncoding8bit,
                                 NULL );

    w = XtVaCreateManagedWidget( "formatMenuBox", boxWidgetClass, cbox, 
                                 XtNorientation, XtorientHorizontal,
                                 NULL );

    Menu_btn = XtVaCreateManagedWidget( "formatMenu",
                                        menuButtonWidgetClass, w,
                                        XtNresize, XawtextResizeNever,
                                        NULL );

    menu_shell = XtVaCreatePopupShell( "menu", 
                                       simpleMenuWidgetClass, Menu_btn,
                                       NULL );

    /*  Create all format items for this menu  */
    for ( i = 0; i < XtNumber( Fmtitem_def ); i++ ) {
        Widget item;

        item = XtVaCreateManagedWidget( Fmtitem_def[i].wgt_name, 
                                        smeBSBObjectClass,
                                        menu_shell,
                                        NULL );
        Fmtitem_def[i].wgt = item;

        XtAddCallback( item, XtNcallback, FmtMenuItemCB, 
                       (XtPointer) Fmtitem_def[i].fmt );
    }

    form = XtVaCreateManagedWidget( "actionForm", formWidgetClass, box, 
                                 NULL );

    w = XtVaCreateManagedWidget( "spacerLabel", labelWidgetClass, form,
                                 XtNlabel, " ", 
                                 XtNresize, False,
                                 XtNencoding, XawTextEncoding8bit,
                                 XtNwidth, 54,
                                 NULL );

    w = XtVaCreateManagedWidget( "okCmd", commandWidgetClass, form,
                                 XtNfromHoriz, w,
                                 NULL );
    XtAddCallback( w, XtNcallback, OkCmdCB, NULL );

    /*  By default, position dialog just to right of main TV window  */
    XtVaGetValues( TVTOPLEVEL, XtNwidth, &width,
                               NULL);

    XtTranslateCoords( TVTOPLEVEL, width, 0, &x, &y );

    XtVaSetValues( *dialog_wgt, XtNx, x,
                                XtNy, y,
                                NULL );
}


void TVIMGSAVDIALOGPopUp()
{
    TV_DISPLAY *dpy = &G_glob.display;

    /*  If we don't have a frozen img to save, this is pointless  */
    if ( dpy->image.buf == NULL ) {
        XUTILDialogPause( TVTOPLEVEL, "Error", "No frozen image to save.",
                          TV_DIALOG_TYPE_OK );
        return;
    }

    /*  Do dialog  */
    if ( Dialog_wgt == NULL )
        TVImgSavDialogBuild( &Dialog_wgt );

    TVIMGSAVDIALOGResync();

    XUTILXtPopup( Dialog_wgt, XtGrabNonexclusive, TVTOPLEVEL );
}

void TVIMGSAVDIALOGResync()
{
    TV_DISK *d = &G_glob.disk;
    String   label;
    TV_INT32 i;

    /*  FIXME:  Also install EnterNotify handler for this dialog to  */
    /*    resync values on entry of it's shell.                      */

    if ( Dialog_wgt == NULL )
        return;

    /*  Set text field to current base filename  */
    TextValUpdate( Text_wgt, d->fn_freeze_base );

    /*  Set suffix toggle state  */
    SetToggleState( Suffix_wgt, d->freeze_use_suffix );

    /*  Set active format box label based on active still image format  */
    for ( i = 0; i < XtNumber( Fmtitem_def ); i++ )
        if ( Fmtitem_def[i].fmt == d->freeze_fmt )
            break;
    if ( i >= XtNumber( Fmtitem_def ) ) {
        fprintf( stderr, 
           "TVIMGSAVDIALOGResync: Unsupported fmt %d\n", d->freeze_fmt );
        exit(1);
    }

    XtVaGetValues( Fmtitem_def[i].wgt, XtNlabel, &label, 
                                       NULL );
    XtVaSetValues( Menu_btn, XtNlabel, label,
                             NULL );
}

