/*
 * remotetrans.c
 *
 * Module for translating mouse remote events for FXTV.
 *
 * (C) 1998 Randall Hopper
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met: 1. Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer. 2.
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

/*      ******************** Include Files                ************** */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <X11/Intrinsic.h>
#include "glob.h"
#include "tvtypes.h"
#include "tvdebug.h"

/*      ******************** Local defines                ************** */

typedef struct {
    char    *key_str;
    String   action_rtn;
    String   action_parms[5];
    Cardinal num_parms;
} TV_REMOTE_TRANS;


/*      ******************** Private variables            ************** */

static TV_REMOTE_TRANS *Rem_trans     = NULL;
static TV_INT32         Rem_trans_len = 0;

/*      ******************** Forward declarations         ************** */
/*      ******************** Function Definitions         ************** */

/**@BEGINFUNC**************************************************************

    Prototype  : void TVREMOTETRANSParse(
                      Display *display )

    Purpose    : Parse the translations specified for the remote device.

    Programmer : 31-May-98  Randall Hopper

    Parameters : display - I: X display

    Returns    : None.

    Globals    : None.

 **@ENDFUNC*****************************************************************/

void TVREMOTETRANSParse( Display *display )
{
    XrmDatabase      db = XtScreenDatabase( DefaultScreenOfDisplay(display) );
    XrmValue         value;
    char            *str_type[ 20 ],
                    *rsrc_val = NULL,
                    *p, *p_end, *p_d1, *p_d2, *p_d3,
                     str[160],
                    *arg;
    TV_INT32         i,j,
                     len;

    /*  Grab the translation list  */
    if ( !XrmGetResource( db, "Fxtv.remoteTranslations", NULL, 
                          str_type, &value ) )
        return;

    if ( (rsrc_val = malloc( value.size+1 )) == NULL )
        TVUTILOutOfMemory();
    rsrc_val[0] = '\0';
    strncat ( rsrc_val, value.addr, value.size );

    /*  How many translations?  */
    p = rsrc_val;
    for ( i = 0; (p != NULL) && (*p != '\0'); i++ )
        p = strchr( p+1, '\n' );
    if ( i == 0 ) {
        fprintf( stderr, "Zero remoteTranslations found.\n" );
        return;
    }

    if ( (Rem_trans = calloc( i, sizeof(Rem_trans[0]) )) == NULL )
        TVUTILOutOfMemory();

    /*  Parse translations  */
    for ( i = 0, p = rsrc_val;
          (i < len) && (*p != '\0'); 
          p = (*p_end ? p_end+1 : p_end), i++ ) {

        /*  Grab a translation: <Remote>[keyname] : [action]([args])\n  */
        if ( (p_end = strchr( p, '\n' )) == NULL )
             p_end = p + strlen(p);
        str[0] = '\0';
        strncat( str, p, MIN( sizeof(str)-1, p - p_end ) );
        TVUTILStrStrip( str, NULL, TRUE,TRUE,TRUE );

        p_d1 = strchr( str, ':' );
        p_d2 = p_d3 = NULL;
        if ( p_d1 )
            p_d2 = strchr( p_d1+1, '(' );
        if ( p_d2 )
            p_d3 = strchr( p_d2+1, ')' );

        /*  Basic validation  */
        if (( strncmp(str, "<Remote>", 8) != 0 ) || !p_d1 || !p_d2 || !p_d3 ) {
            fprintf( stderr, "Invalid remote translation encountered:\n  %s\n",
                     str );
            continue;
        }

        /*  Extract the pieces  */
        *p_d1 = *p_d2 = *p_d3 = '\0';
        if (( (Rem_trans[i].key_str         = strdup( str+8  )) == NULL ) ||
            ( (Rem_trans[i].action_rtn      = strdup( p_d1+1 )) == NULL ))
            TVUTILOutOfMemory();
        TVUTILstrupr( Rem_trans[i].key_str );

        j = 0;
        for ( p_d2++; (arg = strsep( &p_d2, "," )) != NULL; ) {
            if ( *arg == '\0' )
                continue;
            if ( (Rem_trans[i].action_parms[j] = strdup( arg )) == NULL )
                TVUTILOutOfMemory();
            j++;
        }
        Rem_trans[i].num_parms = j;
    }
    Rem_trans_len = i;

    RMPRINTF(( "\nTVREMOTE:  Parsed %d user translations:\n",
               Rem_trans_len ));
    for ( i = 0; i < Rem_trans_len; i++ ) {
        RMPRINTF(( "   <Remote>%-10s : %s(",
                   Rem_trans[i].key_str, Rem_trans[i].action_rtn ));
        for ( j = 0; j < Rem_trans[i].num_parms; j++ )
            RMPRINTF(( "%s%s", ( j > 0 ? "," : "" ),
                       Rem_trans[i].action_parms[j] ));
        RMPRINTF(( ")\n" ));
    }
    RMPRINTF(( "\n" ));

    free( rsrc_val );
}


/**@BEGINFUNC**************************************************************

    Prototype  : TV_BOOL TVREMOTETRANSHandleKey(
                     const char key[] )

    Purpose    : Called when a remote control event comes in.  We pass this
                 event through the user's translations to see if they
                 wanted it mapped to some action procedure in particular.
                 If so, invoke that action and return TRUE.  Else FALSE.

    Programmer : 31-May-98  Randall Hopper

    Parameters : key - I: key event strin

    Returns    : T = we handled the event; F = we didn't

    Globals    : None.

 **@ENDFUNC*****************************************************************/

TV_BOOL TVREMOTETRANSHandleKey( const char key[] )
{
    TV_DISPLAY  *d = &G_glob.display;
    Widget       wgt = d->video_wgt;
    TV_INT32     i;
    char         key_str[20];
    XEvent       xev;

    key_str[0] = '\0';
    strncat( key_str, key, sizeof(key_str)-1 );
    TVUTILstrupr( key_str );

    /*  Loop through and see if we have a match  */
    for ( i = 0; i < Rem_trans_len; i++ )
        if ( STREQ( key_str, Rem_trans[i].key_str ) )
            break;

    if ( i >= Rem_trans_len ) {
        RMPRINTF(( "TVREMOTE:  Key '%s' - No matching user translation.\n",
                   key_str ));
        return FALSE;
    }

    /*  Yep.  Invoke action proc.  */
    /*  Set up a dummy xevent  */
    memset( &xev, '\0', sizeof(xev) );
    xev.xany.type       = ClientMessage;
    xev.xclient.display = TVDISPLAY;
    xev.xclient.window  = d->win;

    XtCallActionProc( wgt, Rem_trans[i].action_rtn, &xev, 
                           Rem_trans[i].action_parms,
                           Rem_trans[i].num_parms );

    RMPRINTF(( "TVREMOTE:  Key '%s' - Invoked action rtn '%s'\n",
               key_str, Rem_trans[i].action_rtn ));

    return TRUE;
}
                      
