/* 

                          Firewall Builder

                 Copyright (C) 2003 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@fwbuilder.org

  $Id: SSHSession.cpp,v 1.10 2005/02/04 07:19:55 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/



#include "config.h"
#include "global.h"
#include "utils.h"

#include "SSHSession.h"
#include "instConf.h"

#include <qobject.h>
#include <qtimer.h>
#include <qregexp.h>
#include <qmessagebox.h>
#include <qapplication.h>
#include <qeventloop.h>

#include <iostream>

#ifdef _WIN32
#  include <windows.h>
#endif


using namespace std;

char  *SSHSession::newKeyOpenSSH  ="Are you sure you want to continue connecting (yes/no)?";
char  *SSHSession::newKeyPlink    ="Store key in cache? (y/n)";
char  *SSHSession::newKeyVsh      ="Accept and save? (y/n)";
char  *SSHSession::fingerprintPrompt="key fingerprint is";



SSHSession::SSHSession(const QString &_h,
                       const QStringList &_args,
                       const QString &_p,
                       const QString &_ep,
                       const QStringList &_in)
{
    host = _h;
    args = _args;
    pwd  = _p;
    epwd = _ep;
    input   = _in;
    quiet   = false;
    verbose = false;
    closeStdin = false;
    error = false;

    proc = NULL;

    newKeyMsg = tr("You are connecting to the firewall <b>'%1'</b> for the first time. It has provided you its identification in a form of its host public key. The fingerprint of the host public key is: \"%2\" You can save the host key to the local database by pressing YES, or you can cancel connection by pressing NO. You should press YES only if you are sure you are really connected to the firewall <b>'%3'</b>.");


    fwb_prompt="";
    quiet=false;
    verbose=false;
    backup=false;
    incremental=false;
    dry_run=false;
    testRun=false;
    stripComments=false;
    wdir="";
    conffile="";
    backupFile="";
    save_diff="";
    diff_pgm="";
    diff_file="";

}

void SSHSession::startSession()
{
    proc = new QProcess();
    proc->setCommunication(
        QProcess::Stdin|QProcess::Stdout|QProcess::Stderr|QProcess::DupStderr);

    connect(proc,SIGNAL(readyReadStdout()), this,  SLOT(readFromStdout() ) );
    connect(proc,SIGNAL(readyReadStderr()), this,  SLOT(readFromStderr() ) );
    connect(proc,SIGNAL(processExited()),   this,  SLOT(processExited() ) );

    for (QStringList::const_iterator i=args.begin(); i!=args.end(); ++i)
    {
        proc->addArgument( *i );
        cmd += *i;
    }

    QStringList env;

#ifdef _WIN32
    env.push_back( QString("APPDATA=")+getenv("APPDATA") );
    env.push_back( QString("HOMEPATH=")+getenv("HOMEPATH") );
    env.push_back( QString("HOMEDRIVE=")+getenv("HOMEDRIVE") );
    env.push_back( QString("ProgramFiles=")+getenv("ProgramFiles") );
/* NB: putty absolutely needs SystemRoot env. var. */
    env.push_back( QString("SystemRoot=")+getenv("SystemRoot") );
    env.push_back( QString("TEMP=")+getenv("TEMP") );
    env.push_back( QString("USERNAME=")+getenv("USERNAME") );
    env.push_back( QString("USERPROFILE=")+getenv("USERPROFILE") );

    env.push_back( QString("HOME=")+getenv("HOMEPATH") );
    env.push_back( QString("USER=")+getenv("USERNAME") );
#else
    env.push_back( QString("HOME=")+getenv("HOME") );
    env.push_back( QString("USER=")+getenv("USER") );
#endif

    env.push_back( QString("TMP=")+getenv("TMP") );
    env.push_back( QString("PATH=")+getenv("PATH") );
    env.push_back( QString("SSH_AUTH_SOCK=")+getenv("SSH_AUTH_SOCK") );

//    emit printStdout_sign( tr("Running command %1\n").arg(cmd) );

    if ( ! proc->start(&env))
    {
        emit printStdout_sign( tr("Failed to start ssh") );
        return;
    }

    if (fwbdebug)
        qDebug("SSHSession::startSession   started child process pid=%ld",
               proc->processIdentifier());


    logged_in = false;
    enable    = false;
    configure = false;
    state = NONE;
}

SSHSession::~SSHSession()
{
    terminate();
}

/*
 * this is redundant and wrong. Should just copy a pointer to instConf
 * object and use that instead of making local copy of each flag.
 */
void SSHSession::setOptions(instConf *cnf)
{
    setQuiet(cnf->quiet);
    setVerbose(cnf->verbose);
    setBackup(cnf->backup);
    setBackupFile(cnf->backup_file);
    setIncr(cnf->incremental);
    setDryRun(cnf->dry_run);
    setSaveStandby(cnf->saveStandby);
    setTestRun(cnf->testRun);
    setStripComments(cnf->stripComments);
    setWDir(cnf->wdir);
    setConfFile(cnf->conffile);
    setSaveDiff(cnf->save_diff);
    setDiffPgm(cnf->diff_pgm);
    setDiffFile(cnf->diff_file);
}

void SSHSession::terminate()
{
    if (proc!=NULL)
    {
        disconnect(proc,SIGNAL(readyReadStdout()),
                   this,SLOT(readFromStdout() ) );
        disconnect(proc,SIGNAL(readyReadStderr()),
                   this,SLOT(readFromStderr() ) );
        disconnect(proc,SIGNAL(processExited()),
                   this,SLOT(processExited() ) );

        if (fwbdebug)
            qDebug("SSHSession::terminate   terminating child process pid=%ld",
                   proc->processIdentifier());

        proc->kill();
        delete proc;
        proc=NULL;
//        QTimer::singleShot( 5000, proc, SLOT(kill()) );
    }
}

void SSHSession::stateMachine()
{
}

void SSHSession::allDataSent()
{
    if (fwbdebug)
        qDebug("SSHSession::allDataSent   closing stdin");

    disconnect(proc,SIGNAL(wroteToStdin()),this,SLOT(allDataSent()));
#ifdef _WIN32
    Sleep(2000);
#endif
    proc->closeStdin();
}

void SSHSession::readFromStdout()
{
    QString s=QString(proc->readStdout());
    stdoutBuffer=stdoutBuffer + s;

//    if (verbose && !quiet)
    if (!quiet)
    {
        s.replace('\r',"");    
        emit printStdout_sign(s);
    }

/* state machine operates on stdoutBuffer directly */
    stateMachine();
}

void SSHSession::readFromStderr()
{
    QString s=QString(proc->readStderr());
    emit printStdout_sign(s);
    stderrBuffer=stderrBuffer + QString(s);
}

void SSHSession::processExited()
{
    int retcode=proc->exitStatus();
    QString exitStatus = (retcode)?QObject::tr("ERROR"):QObject::tr("OK");
    
    emit printStdout_sign(tr("SSH session terminated, exit status: %1").arg(exitStatus));
    if (retcode) error=true;
    emit sessionFinished_sign();
}

bool SSHSession::cmpPrompt(const QString &str,const QString &prompt)
{
    if (str.isEmpty()) return false;

    bool res=false;
    if (str.length()>=prompt.length())
    {
        res=(unsigned(str.findRev(prompt,-1))==str.length()-prompt.length());
        if (!res)
        {
            QString s=str.stripWhiteSpace();
            res=(unsigned(s.findRev(prompt,-1))==s.length()-prompt.length());
        }
    }
    return res;
}



