/* 

                          Firewall Builder

                 Copyright (C) 2003 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@fwbuilder.org

  $Id: FWObjectPropertiesFactory.cpp,v 1.23 2005/01/24 09:29:45 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "config.h"
#include "global.h"
#include "utils.h"

#include <qobject.h>
#include <qtextstream.h>

#include "FWObjectPropertiesFactory.h"
#include "platforms.h"

#include "fwbuilder/FWException.h"

#include "fwbuilder/Library.h"
#include "fwbuilder/Group.h"

#include "fwbuilder/Firewall.h"
#include "fwbuilder/Host.h"
#include "fwbuilder/Network.h"
#include "fwbuilder/IPv4.h"
#include "fwbuilder/physAddress.h"
#include "fwbuilder/AddressRange.h"
#include "fwbuilder/ObjectGroup.h"

#include "fwbuilder/FWReference.h"
#include "fwbuilder/Interface.h"

#include "fwbuilder/CustomService.h"
#include "fwbuilder/IPService.h"
#include "fwbuilder/ICMPService.h"
#include "fwbuilder/TCPService.h"
#include "fwbuilder/UDPService.h"

#include "fwbuilder/Interval.h"

#include "fwbuilder/Resources.h"

#include <sstream>
#include <iostream>

using namespace std;
using namespace libfwbuilder;

/*
 * API methods return STL string, so it is easier to use STL class
 * ostringstream to assemble text and then convert it to QString
 * rather than convert piece by piece.
 */
QString FWObjectPropertiesFactory::getObjectProperties(FWObject *obj)
{
    QString res;
    QTextStream str(&res, IO_WriteOnly);

    try {

        if (IPv4::isA(obj))
        {
            str <<  Address::cast(obj)->getAddress().toString().c_str();
            str << "/";
            str << Address::cast(obj)->getNetmask().toString().c_str();

        } else if (physAddress::isA(obj))
        {
            str <<  physAddress::cast(obj)->getPhysAddress().c_str();

        } else if (AddressRange::isA(obj))
        {
            AddressRange *ar=AddressRange::cast(obj);
            str << ar->getRangeStart().toString().c_str();
            str << " - ";
            str << ar->getRangeEnd().toString().c_str();
        } else if (Firewall::isA(obj))
        {
            QString platform = obj->getStr("platform").c_str();
            QString version  = obj->getStr("version").c_str();
            QMap<QString,QString> vl = getVersionsForPlatform(platform);
            QString readableVersion = vl[version];
            QString hostOS = obj->getStr("host_OS").c_str();

            str <<  platform << "(" << readableVersion << ") / " << hostOS;

        } else if (Host::isA(obj))
        {
            str <<  Address::cast(obj)->getAddress().toString().c_str();

            FWObject *co=obj->getFirstByType("Interface");
            if (co!=NULL) 
            {
                physAddress *paddr=(Interface::cast(co))->getPhysicalAddress();
                if (paddr!=NULL) 
                    str << "    " <<  paddr->getPhysAddress().c_str();
            }

        } else if (Network::isA(obj))
        {
            Network *n=Network::cast(obj);
            str << n->getAddress().toString().c_str();
            str << "/";
            str << n->getNetmask().toString().c_str();

        } else if (Group::cast(obj)!=NULL)   // just any group
        {
            Group *g=Group::cast(obj);
            str << g->size() << " " << QObject::tr(" objects");

        } else if (Firewall::isA(obj))
        {

        } else if (Interface::isA(obj))
        {
            physAddress *paddr=(Interface::cast(obj))->getPhysicalAddress();
            if (paddr!=NULL) 
            {
                str << "    ";
                str <<  paddr->getPhysAddress().c_str();
            }

        } else if (IPService::isA(obj))
        {
            str << QObject::tr("protocol: %1").arg(obj->getStr("protocol_num").c_str());

        } else if (ICMPService::isA(obj))
        {
            str << QObject::tr("type: %1").arg(obj->getStr("type").c_str())
                << "  "
                << QObject::tr("code: %1").arg(obj->getStr("code").c_str());

        } else if (TCPService::isA(obj) || UDPService::isA(obj))
        {
            int sps,spe,dps,dpe;

            sps=obj->getInt("src_range_start");
            spe=obj->getInt("src_range_end");
            dps=obj->getInt("dst_range_start");
            dpe=obj->getInt("dst_range_end");

            str << sps << ":" << spe << " / ";
            str << dps << ":" << dpe;

        } else if (Interval::isA(obj))
        {

        }
    } catch (FWException &ex)
    {
        cerr << ex.toString() << endl;
    }

    return res;
}



QString FWObjectPropertiesFactory::getObjectPropertiesDetailed(FWObject *obj,
                                                               bool showPath,
                                                               bool tooltip,
                                                               bool accentName)
{
    QString str;

    QString path = obj->getPath().c_str();
    path = path.section('/',2,-1);

    if (showPath)
    {
        str += QObject::tr("<b>Library:</b> ");
        str += QString::fromUtf8(obj->getLibrary()->getName().c_str()) + "<br>\n";

        if (!tooltip)
        {
            str += QObject::tr("<b>Object Id:</b> ");
            str += QString(obj->getId().c_str()) + "<br>\n";
        }
    }

    str += QObject::tr("<b>Object Type:</b> ");
    string d = Resources::global_res->getObjResourceStr(obj,"description");
    str += QString(d.c_str()) + "<br>\n";

    str += QObject::tr("<b>Object Name:</b> ");
    if (accentName) str += "<font color=\"red\">";
    str += QString::fromUtf8(obj->getName().c_str());
    if (accentName) str += "</font>";
    str += "<br>\n";

    try {

        if (IPv4::isA(obj))
        {
            if (showPath && !tooltip) str += path + "<br>\n";
            str +=  Address::cast(obj)->getAddress().toString().c_str();
            str += "/";
            str += Address::cast(obj)->getNetmask().toString().c_str();

        } else if (physAddress::isA(obj))
        {
            if (showPath && !tooltip) str += path + "<br>\n";
            str += physAddress::cast(obj)->getPhysAddress().c_str();

        } else if (AddressRange::isA(obj))
        {
            if (showPath && !tooltip) str += path + "<br>\n";
            AddressRange *ar=AddressRange::cast(obj);
            str += ar->getRangeStart().toString().c_str();
            str += " - ";
            str += ar->getRangeEnd().toString().c_str();
        } else if (Host::isA(obj))
        {
            if (showPath && !tooltip) str += path + "<br>\n";

            str +=  Address::cast(obj)->getAddress().toString().c_str() ;

            FWObject *co=obj->getFirstByType("Interface");
            if (co!=NULL) 
            {
                physAddress *paddr=(Interface::cast(co))->getPhysicalAddress();
                if (paddr!=NULL) 
                    str += QString("    ") +  paddr->getPhysAddress().c_str() ;
            }

        } else if (Network::isA(obj))
        {
            if (showPath && !tooltip) str += path + "<br>\n";
            Network *n=Network::cast(obj);
            str += n->getAddress().toString().c_str();
            str += "/";
            str += n->getNetmask().toString().c_str();

        } else if (Group::cast(obj)!=NULL)   // just any group
        {
            if (showPath && !tooltip) str += path + "<br>\n";
            Group *g=Group::cast(obj);
            str += QObject::tr("%1 objects<br>\n").arg(g->size());
            for (FWObject::iterator i=g->begin(); i!=g->end(); ++i)
            {
                FWObject *o1=*i;
                if (FWReference::cast(o1)!=NULL)
                    o1=FWReference::cast(o1)->getPointer();
                str += QString(o1->getTypeName().c_str())
                    + "  <b>" + QString::fromUtf8(o1->getName().c_str()) + "</b><br>\n";
            }
        } else if (Firewall::isA(obj))
        {
            QString platform = obj->getStr("platform").c_str();
            QString version  = obj->getStr("version").c_str();
            QMap<QString,QString> vl = getVersionsForPlatform(platform);
            QString readableVersion = vl[version];
            QString hostOS = obj->getStr("host_OS").c_str();

            if (showPath && !tooltip) str += path + "<br>\n";
            str += "<table cellspacing=\"0\" cellpadding=\"0\">";
            str += QString("<tr><td>Platform:</td><td>") +
                platform + "</td></tr>\n";
            str += QString("<tr><td>Version:</td><td>")  +
                readableVersion + "</td></tr>\n";
            str += QString("<tr><td>Host OS:</td><td>")  +
                hostOS + "</td></tr>\n";
            
            str += "</table>";
        } else if (Interface::isA(obj))
        {
            QString q;
            if (Interface::constcast(obj)->isDyn())        q=" dyn";
            if (Interface::constcast(obj)->isUnnumbered()) q=" unnum";
            
            FWObject *p=obj;
            while (p!=NULL && !Firewall::isA(p)) p=p->getParent();
            if (p!=NULL && p->getStr("platform")=="pix")
            {
                int sl=Interface::constcast(obj)->getSecurityLevel();
                q=q+QString("sec.level %1").arg(sl);
            } else
            {
                if (Interface::constcast(obj)->isExt())        q=q+" ext";
            }
            if (q!="") str += " (" + q + ")";
            str += "<br>\n";
            if (showPath && !tooltip) str += path + "<br>\n";

            physAddress *paddr=(Interface::cast(obj))->getPhysicalAddress();
            if (paddr!=NULL) 
            {
                str += "    ";
                str +=  paddr->getPhysAddress().c_str() ;
            }

        } else if (CustomService::isA(obj))
        {

            if (showPath && !tooltip) str += path + "<br>\n";

            CustomService *s = dynamic_cast<CustomService*>(obj);
            bool first=true;

            map<string,string> platforms = Resources::getPlatforms();
            for (map<string,string>::iterator i=platforms.begin(); i!=platforms.end(); i++)
            {
                string c=s->getCodeForPlatform( (*i).first );
                if ( c!="" )
                {
                    if (first)
                    {
                        str += "<table cellspacing=\"0\" cellpadding=\"0\">";
                        first=false;
                    }
                    str += QString("<tr><td>%1</td><td>%2</td></tr>\n")
                        .arg((*i).second.c_str()).arg(c.c_str());
                }
            }
            if (!first) str += "</table>";

        } else if (IPService::isA(obj))
        {
            if (showPath && !tooltip) str += path + "<br>\n";
            str += QObject::tr("protocol ") + obj->getStr("protocol_num").c_str();

        } else if (ICMPService::isA(obj))
        {
            if (showPath && !tooltip) str += path + "<br>\n";
            str += QObject::tr("type: ") + obj->getStr("type").c_str()
                + "  "
                + QObject::tr("code: ") + obj->getStr("code").c_str();

        } else if (TCPService::isA(obj) || UDPService::isA(obj))
        {
            int sps,spe,dps,dpe;

            sps=obj->getInt("src_range_start");
            spe=obj->getInt("src_range_end");
            dps=obj->getInt("dst_range_start");
            dpe=obj->getInt("dst_range_end");

            if (showPath && !tooltip) str += path + "<br>\n";
            str += "<table cellspacing=\"0\" cellpadding=\"0\">";
            str += QString("<tr><td>source port range</td><td>%1:%2</td></tr>")
                .arg(sps).arg(spe);
            str += QString("<tr><td>destination port range</td><td>%1:%2</td></tr>")
                .arg(dps).arg(dpe);
            str += "</table>";
        } else if (Interval::isA(obj))
        {

        }
    } catch (FWException &ex)
    {
        cerr << ex.toString() << endl;
    }

    return str;
}


