/* spectrum.c: Generic Spectrum routines
   Copyright (c) 1999-2003 Philip Kendall, Darren Salt

   $Id: spectrum.c,v 1.48 2003/12/14 12:34:36 pak21 Exp $

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

   Author contact information:

   E-mail: pak21-fuse@srcf.ucam.org
   Postal address: 15 Crescent Road, Wokingham, Berks, RG40 2DB, England

*/

#include <config.h>

#include <libspectrum.h>

#include "debugger/debugger.h"
#include "display.h"
#include "event.h"
#include "fuse.h"
#include "keyboard.h"
#include "printer.h"
#include "rzx.h"
#include "settings.h"
#include "sound.h"
#include "spec128.h"
#include "spec48.h"
#include "specplus2.h"
#include "specplus3.h"
#include "spectrum.h"
#include "tape.h"
#include "timer.h"
#include "ui/ui.h"
#include "z80/z80.h"

/* The number of ROMs we have allocated space for; they might not all be
   in use at the moment */
size_t spectrum_rom_count = 0;

/* The ROMs themselves */
libspectrum_byte **ROM = NULL;

/* And the RAM */
libspectrum_byte RAM[8][0x4000];

/* How many tstates have elapsed since the last interrupt? (or more
   precisely, since the ULA last pulled the /INT line to the Z80 low) */
libspectrum_dword tstates;

/* The last byte written to the ULA */
libspectrum_byte spectrum_last_ula;

/* Set these every time we change machine to avoid having to do a
   structure lookup too often */
spectrum_screen_read_function read_screen_memory;

spectrum_port_contention_function contend_port;
spectrum_contention_delay_function contend_delay;

libspectrum_byte spectrum_contention[ 80000 ];

/* Level data from .slt files */

libspectrum_byte *slt[256];
size_t slt_length[256];

libspectrum_byte *slt_screen;	/* The screenshot from the .slt file */
int slt_screen_level;		/* The level of the screenshot.
				   Not used for anything AFAIK */

int
spectrum_frame( void )
{
  libspectrum_dword frame_length;
  int error;

  /* Reduce the t-state count of both the processor and all the events
     scheduled to occur. Done slightly differently if RZX playback is
     occuring */
  frame_length = rzx_playback ? tstates
			      : machine_current->timings.tstates_per_frame;

  error = event_frame( frame_length ); if( error ) return error;
  tstates -= frame_length;
  if( z80.interrupts_enabled_at >= 0 )
    z80.interrupts_enabled_at -= frame_length;

#ifdef ALWAYS_USE_TIMER /* Some sound routines do not provide speed control */
  if( sound_enabled ) sound_frame();
  timer_sleep();
#else			/* #ifdef ALWAYS_USE_TIMER */
  if(sound_enabled) {
    sound_frame();
  } else {
    timer_sleep();
  }
#endif			/* #ifdef ALWAYS_USE_TIMER */

  if(display_frame()) return 1;
  printer_frame();

  /* Add an interrupt unless they're being generated by .rzx playback */
  if( !rzx_playback ) {
    if( event_add( machine_current->timings.tstates_per_frame,
		   EVENT_TYPE_FRAME ) ) return 1;
  }

  return 0;
}

libspectrum_byte
readport( libspectrum_word port )
{
  spectrum_port_info *ptr;

  libspectrum_byte return_value = 0xff;
  int attached = 0;		/* Is this port attached to anything? */

  /* Trigger the debugger if wanted */
  if( debugger_mode != DEBUGGER_MODE_INACTIVE &&
      debugger_check( DEBUGGER_BREAKPOINT_TYPE_PORT_READ, port ) )
    debugger_mode = DEBUGGER_MODE_HALTED;

  /* If we're doing RZX playback, get a byte from the RZX file */
  if( rzx_playback ) {

    libspectrum_error error;

    error = libspectrum_rzx_playback( rzx, &return_value );
    if( error ) { rzx_stop_playback( 1 ); return readport( port ); }

    return return_value;
  }

  /* If we're not doing RZX playback, get the byte normally */
  for( ptr = machine_current->peripherals; ptr->mask; ptr++ ) {
    if( ( port & ptr->mask ) == ptr->data ) {
      return_value &= ptr->read(port); attached = 1;
    }
  }

  if( !attached ) return_value = machine_current->unattached_port();

  /* If we're RZX recording, store this byte */
  if( rzx_recording ) rzx_store_byte( return_value );

  return return_value;

}

void
writeport( libspectrum_word port, libspectrum_byte b )
{
  spectrum_port_info *ptr;

  /* Trigger the debugger if wanted */
  if( debugger_mode != DEBUGGER_MODE_INACTIVE &&
      debugger_check( DEBUGGER_BREAKPOINT_TYPE_PORT_WRITE, port ) )
    debugger_mode = DEBUGGER_MODE_HALTED;

  for( ptr = machine_current->peripherals; ptr->mask; ptr++ ) {
    if( ( port & ptr->mask ) == ptr->data ) {
      ptr->write(port, b);
    }
  }

}

/* A dummy function for non-readable ports */
libspectrum_byte
spectrum_port_noread( libspectrum_word port GCC_UNUSED )
{
  /* FIXME: should this return the floating bus value? */
  return 0xff;
}

/* And one for non-writable ports */
void
spectrum_port_nowrite( libspectrum_word port GCC_UNUSED,
		       libspectrum_byte value GCC_UNUSED )
{
  return;
}

/* What do we get if we read from the ULA? */
libspectrum_byte
spectrum_ula_read( libspectrum_word port )
{
  return ( keyboard_read( port >> 8 ) ^ ( tape_microphone ? 0x40 : 0x00 ) );
}

/* What happens when we write to the ULA? */
void
spectrum_ula_write( libspectrum_word port GCC_UNUSED, libspectrum_byte b )
{
  spectrum_last_ula = b;

  display_set_lores_border( b & 0x07 );
  sound_beeper( 0, b & 0x10 );

  if( machine_current->timex ) {
      keyboard_default_value=0x5f;
      return;
  }

  if( settings_current.issue2 ) {
    if( b & 0x18 ) {
      keyboard_default_value=0xff;
    } else {
      keyboard_default_value=0xbf;
    }
  } else {
    if( b & 0x10 ) {
      keyboard_default_value=0xff;
    } else {
      keyboard_default_value=0xbf;
    }
  }
}

/* What happens if we read from an unattached port? */
libspectrum_byte
spectrum_unattached_port( int offset )
{
  int line, tstates_through_line, column;

  /* Return 0xff (idle bus) if we're in the top border */
  if( tstates < machine_current->line_times[ DISPLAY_BORDER_HEIGHT ] )
    return 0xff;

  /* Work out which line we're on, relative to the top of the screen */
  line = ( (libspectrum_signed_dword)tstates -
	   machine_current->line_times[ DISPLAY_BORDER_HEIGHT ] ) /
    machine_current->timings.tstates_per_line;

  /* Idle bus if we're in the lower or upper borders */
  if( line >= DISPLAY_HEIGHT ) return 0xff;

  /* Work out where we are in this line */
  tstates_through_line = tstates -
    machine_current->line_times[ DISPLAY_BORDER_HEIGHT + line ];

  /* Idle bus if we're in the left border */
  if( tstates_through_line < machine_current->timings.left_border - offset )
    return 0xff;

  /* Or the right border or retrace */
  if( tstates_through_line >= machine_current->timings.left_border +
                              machine_current->timings.horizontal_screen -
                              offset )
    return 0xff;

  column = ( ( tstates_through_line + 1 -
	       machine_current->timings.left_border ) / 8 ) * 2;

  switch( tstates_through_line % 8 ) {

    /* FIXME: 25% of these should be screen data, 25% attribute bytes
       and 50% idle bus, but the actual distribution is unknown. Also,
       in each 8 T-state block, 16 pixels are displayed; when each of
       these is read is also unknown. Thanks to Ian Greenway for this
       information */

    /* FIXME: Arkanoid doesn't work properly with the below */

    /* Attribute bytes */
    case 1: column++;
    case 0:
      return read_screen_memory( display_attr_start[line] + column );

    /* Screen data */
    case 3: column++;
    case 2:
      return read_screen_memory( display_line_start[line] + column );

    /* Idle bus */
    case 4: case 5: case 6: case 7:
      return 0xff;

  }

  return 0;		/* Keep gcc happy */
}
