/* 
   ipmi-udm.c: IPMI Unified Driver Model (API interface for all
   IPMI drivers)

   Copyright (C) 2005 FreeIPMI Core Team

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.  

*/

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#ifdef STDC_HEADERS
#include <string.h>
#endif /* STDC_HEADERS */
#if HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */
#if TIME_WITH_SYS_TIME
#include <sys/time.h>
#include <time.h>
#else /* !TIME_WITH_SYS_TIME */
#if HAVE_SYS_TIME_H
#include <sys/time.h>
#else /* !HAVE_SYS_TIME_H */
#include <time.h>
#endif /* !HAVE_SYS_TIME_H */
#endif  /* !TIME_WITH_SYS_TIME */
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <assert.h>
#include <errno.h>

#include "freeipmi/udm/ipmi-udm.h"
#include "freeipmi/ipmi-authentication-type-spec.h"
#include "freeipmi/ipmi-debug.h"
#include "freeipmi/ipmi-locate.h"
#include "freeipmi/ipmi-kcs.h"
#include "freeipmi/ipmi-kcs-api.h"
#include "freeipmi/ipmi-messaging-support-cmds.h"
#include "freeipmi/ipmi-lan-interface.h"
#include "freeipmi/ipmi-openipmi-api.h"
#include "freeipmi/ipmi-privilege-level-spec.h"
#include "freeipmi/ipmi-ssif-api.h"
#include "freeipmi/ipmi-utils.h"
#include "freeipmi/rmcp.h"
#include "freeipmi/udm/ipmi-kcs-api-udm.h"
#include "freeipmi/udm/ipmi-lan-interface-udm.h"
#include "freeipmi/udm/ipmi-messaging-support-cmds-udm.h"
#include "freeipmi/udm/ipmi-openipmi-api-udm.h"
#include "freeipmi/udm/ipmi-ssif-api-udm.h"

#include "ipmi-udm-device.h"

#include "freeipmi-portability.h"
#include "ipmi-common.h"

#include "udm-err-wrappers.h"
#include "udm-fiid-wrappers.h"

#define IPMI_SESSION_TIMEOUT     20000
#define IPMI_RETRY_TIMEOUT       1000
#define IPMI_POLL_INTERVAL_USECS 60

#define GETHOSTBYNAME_AUX_BUFLEN 1024
extern int h_errno;

static char *ipmi_udm_errmsg[] =
  {
    "success",			                                    /* 0 */
    "null device pointer", 	                                    /* 1 */
    "invalid magic",		                                    /* 2 */
    "permission denied",	                                    /* 3 */
    "invalid username",		                                    /* 4 */
    "invalid password",		                                    /* 5 */
    "insufficient privilege",	                                    /* 6 */
    "invalid authentication type",                                  /* 7 */
    "password verification timeout",                                /* 8 */
    "session timeout",		                                    /* 9 */
    "device already open",                                          /* 10 */
    "device not open",		                                    /* 11 */
    "device not supported",	                                    /* 12 */
    "device not found",                                             /* 13 */
    "bad completion code: command invalid/not supported",           /* 14 */
    "bad completion code: request data/parameter invalid", 	    /* 15 */
    "bad completion code: insufficient privilege for this command", /* 16 */
    "bad completion code",	                                    /* 17 */
    "bmc busy",			                                    /* 18 */
    "out of memory",		                                    /* 19 */
    "invalid hostname",                                             /* 20 */
    "invalid parameters",	                                    /* 21 */
    "driver path required",                                         /* 22 */
    "internal ipmi error",	                                    /* 23 */
    "internal system error",	                                    /* 24 */
    "internal library error",	                                    /* 25 */
    "internal error",		                                    /* 26 */
    "out of range",		                                    /* 27 */
  };

static void
_ipmi_device_init(struct ipmi_device *dev)
{
  assert(dev);

  memset(dev, '\0', sizeof(struct ipmi_device));
  dev->magic = IPMI_UDM_DEVICE_MAGIC;
  dev->type = IPMI_DEVICE_UNKNOWN;
}

ipmi_device_t
ipmi_device_create(void)
{
  struct ipmi_device *dev;

  if (!(dev = (struct ipmi_device *)malloc(sizeof(struct ipmi_device))))
    return NULL;

  _ipmi_device_init(dev);
  dev->errnum = IPMI_ERR_SUCCESS;

  return dev;
}

char *
ipmi_device_strerror(int errnum)
{
  if (errnum >= IPMI_ERR_SUCCESS && errnum <= IPMI_ERR_OUTOFRANGE)
    return ipmi_udm_errmsg[errnum];
  else
    return ipmi_udm_errmsg[IPMI_ERR_OUTOFRANGE];
}

int
ipmi_device_errnum(ipmi_device_t dev)
{
  if (!dev)
    return (IPMI_ERR_NULL_DEVICE);
  else if (dev->magic != IPMI_UDM_DEVICE_MAGIC)
    return (IPMI_ERR_DEVICE_MAGIC);
  else
    return (dev->errnum);
}

static void 
_ipmi_outofband_free (ipmi_device_t dev)
{
  /* Function Note: No need to set UDM errnum - just return */
  assert(dev && dev->magic == IPMI_UDM_DEVICE_MAGIC);

  UDM_FIID_OBJ_DESTROY (dev->io.outofband.rq.obj_rmcp_hdr);
  UDM_FIID_OBJ_DESTROY (dev->io.outofband.rq.obj_lan_session_hdr);
  UDM_FIID_OBJ_DESTROY (dev->io.outofband.rq.obj_lan_msg_hdr);

  UDM_FIID_OBJ_DESTROY (dev->io.outofband.rs.obj_rmcp_hdr);
  UDM_FIID_OBJ_DESTROY (dev->io.outofband.rs.obj_lan_session_hdr);
  UDM_FIID_OBJ_DESTROY (dev->io.outofband.rs.obj_lan_msg_hdr);
  UDM_FIID_OBJ_DESTROY (dev->io.outofband.rs.obj_lan_msg_trlr);
}

static void 
_ipmi_inband_free (ipmi_device_t dev)
{
  /* Function Note: No need to set UDM errnum - just return */
  assert(dev && dev->magic == IPMI_UDM_DEVICE_MAGIC);
  
  if (dev->type == IPMI_DEVICE_KCS && dev->io.inband.kcs_ctx)
    ipmi_kcs_ctx_destroy(dev->io.inband.kcs_ctx);
  if (dev->type == IPMI_DEVICE_SSIF && dev->io.inband.ssif_ctx)
    ipmi_ssif_ctx_destroy(dev->io.inband.ssif_ctx);
  if (dev->type == IPMI_DEVICE_OPENIPMI && dev->io.inband.openipmi_ctx)
    ipmi_openipmi_ctx_destroy(dev->io.inband.openipmi_ctx);

  UDM_FIID_OBJ_DESTROY (dev->io.inband.rq.obj_hdr);
  UDM_FIID_OBJ_DESTROY (dev->io.inband.rs.obj_hdr);
}

int
ipmi_open_outofband (ipmi_device_t dev,
		     ipmi_driver_type_t driver_type, 
		     const char *hostname,
		     const char *username, 
		     const char *password, 
		     uint8_t authentication_type, 
		     uint8_t privilege_level,
                     unsigned int session_timeout,
                     unsigned int retry_timeout,
                     uint32_t flags)
{
  struct sockaddr_in addr;
#ifdef HAVE_FUNC_GETHOSTBYNAME_R_6
  struct hostent hent;
  int h_errnop;
  char buf[GETHOSTBYNAME_AUX_BUFLEN];
#endif /* HAVE_FUNC_GETHOSTBYNAME_R_6 */
  struct hostent *hptr;
  
  UDM_ERR_DEV_CHECK (dev && dev->magic == IPMI_UDM_DEVICE_MAGIC);

  UDM_ERR_DEVICE_ALREADY_OPEN(dev->type == IPMI_DEVICE_UNKNOWN);

  UDM_ERR_INVALID_PARAMETERS(driver_type == IPMI_DEVICE_LAN
			     && hostname
			     && !(username && strlen (username) > IPMI_MAX_USER_NAME_LENGTH)
			     && !(password && strlen (password) > IPMI_MAX_AUTHENTICATION_CODE_LENGTH)
			     && IPMI_1_5_AUTHENTICATION_TYPE_VALID (authentication_type)
			     && IPMI_PRIVILEGE_LEVEL_VALID (privilege_level));
   
  dev->type = driver_type;
  dev->flags = flags;

#ifdef HAVE_FUNC_GETHOSTBYNAME_R_6
  memset(&hent, '\0', sizeof(struct hostent));
  UDM_ERR_INVALID_HOSTNAME_CLEANUP(!gethostbyname_r(hostname,
						      &hent,
						      buf,
						      GETHOSTBYNAME_AUX_BUFLEN,
						      &hptr,
						      &h_errnop));
  UDM_ERR_INVALID_HOSTNAME_CLEANUP(hptr);
#else  /* !HAVE_FUNC_GETHOSTBYNAME_R */
#error Additional threadsafe gethostbyname support needed
#endif /* !HAVE_FUNC_GETHOSTBYNAME_R */

  dev->io.outofband.remote_host.sin_family = AF_INET;
  dev->io.outofband.remote_host.sin_port = htons(RMCP_AUX_BUS_SHUNT);
  dev->io.outofband.remote_host.sin_addr = *(struct in_addr *) hptr->h_addr;
  
  dev->io.outofband.session_timeout = (session_timeout ? session_timeout : IPMI_SESSION_TIMEOUT);
  dev->io.outofband.retry_timeout = (retry_timeout ? retry_timeout : IPMI_RETRY_TIMEOUT);
  memset(&dev->io.outofband.last_received, '\0', sizeof(struct timeval));
  dev->io.outofband.authentication_type = authentication_type;
  memset(dev->io.outofband.username, '\0', IPMI_MAX_USER_NAME_LENGTH);
  if (username != NULL)
    memcpy (dev->io.outofband.username, 
	    username, 
	    strlen (username));
  memset(dev->io.outofband.password, '\0', IPMI_MAX_AUTHENTICATION_CODE_LENGTH);
  if (password != NULL)
    memcpy (dev->io.outofband.password, 
	    password, 
	    strlen (password));
  dev->io.outofband.privilege_level = privilege_level;

  /* achu: per_msg_auth_disabled and lan_session_state used and
   * initialized in ipmi_lan_open_session()
   */
  
  UDM_FIID_OBJ_CREATE_CLEANUP (dev->io.outofband.rq.obj_rmcp_hdr, tmpl_rmcp_hdr);
  UDM_FIID_OBJ_CREATE_CLEANUP (dev->io.outofband.rq.obj_lan_session_hdr, tmpl_lan_session_hdr);
  UDM_FIID_OBJ_CREATE_CLEANUP (dev->io.outofband.rq.obj_lan_msg_hdr, tmpl_lan_msg_hdr_rq);
  
  UDM_FIID_OBJ_CREATE_CLEANUP (dev->io.outofband.rs.obj_rmcp_hdr, tmpl_rmcp_hdr);
  UDM_FIID_OBJ_CREATE_CLEANUP (dev->io.outofband.rs.obj_lan_session_hdr, tmpl_lan_session_hdr);
  UDM_FIID_OBJ_CREATE_CLEANUP (dev->io.outofband.rs.obj_lan_msg_hdr, tmpl_lan_msg_hdr_rs);
  UDM_FIID_OBJ_CREATE_CLEANUP (dev->io.outofband.rs.obj_lan_msg_trlr, tmpl_lan_msg_trlr);
  
  /* Open client (local) UDP socket */
  /* achu: ephemeral ports are > 1023, so no way we will bind to an IPMI port */
  
  UDM_ERR_INTERNAL_SYSTEM_ERROR_CLEANUP (!((dev->io.outofband.local_sockfd = socket (AF_INET, SOCK_DGRAM, 0)) < 0));

  memset (&addr, 0, sizeof (struct sockaddr_in));
  addr.sin_family = AF_INET;
  addr.sin_port   = htons (0);
  addr.sin_addr.s_addr = htonl (INADDR_ANY);

  UDM_ERR_INTERNAL_SYSTEM_ERROR_CLEANUP (!(bind(dev->io.outofband.local_sockfd, 
						(struct sockaddr *)&addr,
						sizeof(struct sockaddr_in)) < 0));
  
  /* Note that ipmi_lan_open_session itself calls ipmi_lan_cmd many
     times internally, at this point everything must be set to go
     -- Anand Babu */
  /* UDM errnum set in ipmi_lan_open_session */
  if (ipmi_lan_open_session (dev) < 0)
    goto cleanup;

  dev->errnum = IPMI_ERR_SUCCESS;
  return (0);

 cleanup:
  if (dev->io.outofband.local_sockfd)
    close (dev->io.outofband.local_sockfd);
  _ipmi_outofband_free (dev);
  dev->type = IPMI_DEVICE_UNKNOWN;
  return (-1);
}

int
ipmi_open_inband (ipmi_device_t dev,
		  ipmi_driver_type_t driver_type, 
		  int disable_auto_probe, 
		  uint16_t driver_address, 
		  uint8_t reg_space,
		  char *driver_device, 
		  uint32_t flags)
{
  struct ipmi_locate_info locate_info;
  uint32_t temp_flags = 0;

  UDM_ERR_DEV_CHECK (dev && dev->magic == IPMI_UDM_DEVICE_MAGIC);

  UDM_ERR_DEVICE_ALREADY_OPEN(dev->type == IPMI_DEVICE_UNKNOWN);

  UDM_ERR_INVALID_PARAMETERS(driver_type == IPMI_DEVICE_KCS
			     || driver_type == IPMI_DEVICE_SMIC
			     || driver_type == IPMI_DEVICE_BT
			     || driver_type == IPMI_DEVICE_SSIF
			     || driver_type == IPMI_DEVICE_OPENIPMI);

  UDM_ERR_DRIVER_PATH_REQUIRED(!(driver_type == IPMI_DEVICE_SSIF
                                 && !driver_device));

  dev->io.inband.kcs_ctx = NULL;
  dev->io.inband.ssif_ctx = NULL;
  dev->io.inband.openipmi_ctx = NULL;

  switch (driver_type)
    {
    case IPMI_DEVICE_KCS:
      if (disable_auto_probe)
	{
	  memset(&locate_info, '\0', sizeof(struct ipmi_locate_info));

	  locate_info.ipmi_ver_major = 1;
	  locate_info.ipmi_ver_minor = 5;
	  locate_info.locate_driver_type = IPMI_LOCATE_DRIVER_NONE;
	  locate_info.interface_type = IPMI_INTERFACE_KCS;
	  if (driver_device)
	    {
	      strncpy(locate_info.driver_device, driver_device, IPMI_LOCATE_PATH_MAX);
	      locate_info.driver_device[IPMI_LOCATE_PATH_MAX - 1] = '\0';
	    }
	  locate_info.address_space_id = IPMI_ADDRESS_SPACE_ID_SYSTEM_IO;
	  locate_info.driver_address = driver_address;
	  locate_info.reg_space = reg_space;
	}
      else 
	{
	  UDM_ERR_CLEANUP (ipmi_locate (IPMI_INTERFACE_KCS, &locate_info) == 0);
	  if (driver_device)
	    {
	      strncpy(locate_info.driver_device, driver_device, IPMI_LOCATE_PATH_MAX);
	      locate_info.driver_device[IPMI_LOCATE_PATH_MAX - 1] = '\0';
	    }
	  if (driver_address)
	    locate_info.driver_address = driver_address;
	  if (reg_space)
	    locate_info.reg_space = reg_space;
	}
      dev->type = driver_type;
      dev->flags = flags;
      
      /* At this point we only support SYSTEM_IO, i.e. inb/outb style IO. 
	 If we cant find the bass address, we better exit. -- Anand Babu */
      UDM_ERR_DEVICE_NOT_SUPPORTED_CLEANUP (locate_info.address_space_id == IPMI_ADDRESS_SPACE_ID_SYSTEM_IO);
      
      UDM_ERR_CLEANUP ((dev->io.inband.kcs_ctx = ipmi_kcs_ctx_create()));
      
      UDM_ERR_KCS_CLEANUP (!(ipmi_kcs_ctx_set_driver_address(dev->io.inband.kcs_ctx, 
							     locate_info.driver_address) < 0));

      UDM_ERR_KCS_CLEANUP (!(ipmi_kcs_ctx_set_register_space(dev->io.inband.kcs_ctx, 
							     locate_info.reg_space) < 0));
      
      UDM_ERR_KCS_CLEANUP (!(ipmi_kcs_ctx_set_poll_interval(dev->io.inband.kcs_ctx, 
							    IPMI_POLL_INTERVAL_USECS) < 0));
      
      if (dev->flags & IPMI_FLAGS_NONBLOCKING)
        temp_flags |= IPMI_KCS_FLAGS_NONBLOCKING;
      
      UDM_ERR_KCS_CLEANUP (!(ipmi_kcs_ctx_set_flags(dev->io.inband.kcs_ctx, temp_flags) < 0));
      
      UDM_ERR_KCS_CLEANUP (!(ipmi_kcs_ctx_io_init(dev->io.inband.kcs_ctx) < 0));

      break;
    case IPMI_DEVICE_SMIC:
      dev->errnum = IPMI_ERR_DEVICE_NOT_SUPPORTED;
      goto cleanup;
    case IPMI_DEVICE_BT:
      dev->errnum = IPMI_ERR_DEVICE_NOT_SUPPORTED;
      goto cleanup;
    case IPMI_DEVICE_SSIF:
      if (disable_auto_probe)
	{
	  memset(&locate_info, '\0', sizeof(struct ipmi_locate_info));

	  locate_info.ipmi_ver_major = 1;
	  locate_info.ipmi_ver_minor = 5;
	  locate_info.locate_driver_type = IPMI_LOCATE_DRIVER_NONE;
	  locate_info.interface_type = IPMI_INTERFACE_SSIF;
	  strncpy(locate_info.driver_device, driver_device, IPMI_LOCATE_PATH_MAX);
	  locate_info.driver_device[IPMI_LOCATE_PATH_MAX - 1] = '\0';
	  locate_info.address_space_id = IPMI_ADDRESS_SPACE_ID_SMBUS;
	  locate_info.driver_address = driver_address;
	  locate_info.reg_space = reg_space;
	}
      else 
	{
	  UDM_ERR_CLEANUP (ipmi_locate (IPMI_INTERFACE_SSIF, &locate_info) == 0);
	  if (driver_device)
	    {
	      strncpy(locate_info.driver_device, driver_device, IPMI_LOCATE_PATH_MAX);
	      locate_info.driver_device[IPMI_LOCATE_PATH_MAX - 1] = '\0';
	    }
	  if (driver_address)
	    locate_info.driver_address = driver_address;
	  if (reg_space)
	    locate_info.reg_space = reg_space;
	}
      dev->type = driver_type;
      dev->flags = flags;

      UDM_ERR_CLEANUP ((dev->io.inband.ssif_ctx = ipmi_ssif_ctx_create()));
      
      UDM_ERR_SSIF_CLEANUP (!(ipmi_ssif_ctx_set_driver_device(dev->io.inband.ssif_ctx, 
                                                              locate_info.driver_device) < 0));
 
      UDM_ERR_SSIF_CLEANUP (!(ipmi_ssif_ctx_set_driver_address(dev->io.inband.ssif_ctx, 
                                                               locate_info.driver_address) < 0));

      if (dev->flags & IPMI_FLAGS_NONBLOCKING)
        temp_flags |= IPMI_SSIF_FLAGS_NONBLOCKING;
      
      UDM_ERR_SSIF_CLEANUP (!(ipmi_ssif_ctx_set_flags(dev->io.inband.ssif_ctx, temp_flags) < 0));

      UDM_ERR_SSIF_CLEANUP (!(ipmi_ssif_ctx_io_init(dev->io.inband.ssif_ctx) < 0));

      break;

    case IPMI_DEVICE_OPENIPMI:
      dev->type = driver_type;
      dev->flags = flags;

      UDM_ERR_CLEANUP ((dev->io.inband.openipmi_ctx = ipmi_openipmi_ctx_create()));
      
      if (driver_device)
        UDM_ERR_OPENIPMI_CLEANUP (!(ipmi_openipmi_ctx_set_driver_device(dev->io.inband.openipmi_ctx,
                                                                        driver_device) < 0));
      
      UDM_ERR_OPENIPMI_CLEANUP (!(ipmi_openipmi_ctx_io_init(dev->io.inband.openipmi_ctx) < 0));

      break;

    default:
      goto cleanup;
    }
  
  /* Prepare in-band headers */
  UDM_FIID_OBJ_CREATE_CLEANUP (dev->io.inband.rq.obj_hdr, tmpl_hdr_kcs);
  UDM_FIID_OBJ_CREATE_CLEANUP (dev->io.inband.rs.obj_hdr, tmpl_hdr_kcs);
  
  dev->errnum = IPMI_ERR_SUCCESS;
  return (0);

 cleanup:
  _ipmi_inband_free (dev);
  dev->type = IPMI_DEVICE_UNKNOWN;
  return (-1);
}

int 
ipmi_cmd (ipmi_device_t dev, 
	  uint8_t lun, 
	  uint8_t net_fn, 
	  fiid_obj_t obj_cmd_rq, 
	  fiid_obj_t obj_cmd_rs)
{
  int8_t status = 0;

  UDM_ERR_DEV_CHECK (dev && dev->magic == IPMI_UDM_DEVICE_MAGIC);

  UDM_ERR_DEVICE_NOT_OPEN(dev->type != IPMI_DEVICE_UNKNOWN);

  UDM_ERR_INTERNAL_ERROR(dev->type == IPMI_DEVICE_LAN
			 || dev->type == IPMI_DEVICE_KCS
			 || dev->type == IPMI_DEVICE_SSIF
			 || dev->type == IPMI_DEVICE_OPENIPMI);

  UDM_FIID_OBJ_PACKET_VALID(obj_cmd_rq);

  dev->lun = lun;
  dev->net_fn = net_fn;
  
  if (dev->flags & IPMI_FLAGS_DEBUG_DUMP)
    {
      /* lan packets are dumped in ipmi lan code */
      if (dev->type != IPMI_DEVICE_LAN)
	ipmi_obj_dump(STDERR_FILENO, obj_cmd_rq);
    }

  if (dev->type == IPMI_DEVICE_LAN)
    status = ipmi_lan_cmd (dev, obj_cmd_rq, obj_cmd_rs);
  else if (dev->type == IPMI_DEVICE_KCS)
    status = ipmi_kcs_cmd_udm (dev, obj_cmd_rq, obj_cmd_rs);
  else if (dev->type == IPMI_DEVICE_SSIF)
    status = ipmi_ssif_cmd_udm (dev, obj_cmd_rq, obj_cmd_rs);
  else /* dev->type == IPMI_DEVICE_OPENIPMI */
    status = ipmi_openipmi_cmd_udm (dev, obj_cmd_rq, obj_cmd_rs);
  
  if (dev->flags & IPMI_FLAGS_DEBUG_DUMP)
    {
      /* lan packets are dumped in ipmi lan code */
      if (dev->type != IPMI_DEVICE_LAN)
	ipmi_obj_dump(STDERR_FILENO, obj_cmd_rs);
    }

  /* errnum set in ipmi_*_cmd functions */
  return (status);
}

int 
ipmi_cmd_raw (ipmi_device_t dev, 
              uint8_t lun,
              uint8_t net_fn,
	      uint8_t *in, 
	      size_t in_len, 
	      uint8_t *out, 
	      size_t out_len)
{
  int8_t status = 0;

  UDM_ERR_DEV_CHECK (dev && dev->magic == IPMI_UDM_DEVICE_MAGIC);

  UDM_ERR_DEVICE_NOT_OPEN(dev->type != IPMI_DEVICE_UNKNOWN);

  if (dev->type != IPMI_DEVICE_LAN
      && dev->type != IPMI_DEVICE_KCS
      && dev->type != IPMI_DEVICE_SSIF
      && dev->type != IPMI_DEVICE_OPENIPMI)
    {
      dev->errnum = IPMI_ERR_INTERNAL_ERROR;
      return -1;
    }
  
  UDM_ERR_INVALID_PARAMETERS(in
			     && in_len > 0
			     && out
			     && out_len > 0);

  dev->lun = lun;
  dev->net_fn = net_fn;

 if (dev->type == IPMI_DEVICE_LAN)
   status = ipmi_lan_cmd_raw (dev, in, in_len, out, out_len);
 else if (dev->type == IPMI_DEVICE_KCS)
   status = ipmi_kcs_cmd_raw_udm (dev, in, in_len, out, out_len);
 else if (dev->type == IPMI_DEVICE_SSIF)
   status = ipmi_ssif_cmd_raw_udm (dev, in, in_len, out, out_len);
 else /* dev->type == IPMI_DEVICE_OPENIPMI */
   status = ipmi_openipmi_cmd_raw_udm (dev, in, in_len, out, out_len);

  /* errnum set in ipmi_*_cmd_raw functions */
  return (status);
}

static void
_ipmi_outofband_close (ipmi_device_t dev)
{
  /* Function Note: No need to set UDM errnum - just return */
  assert(dev 
	 && dev->magic == IPMI_UDM_DEVICE_MAGIC
	 && dev->type == IPMI_DEVICE_LAN);
 
  /* No need to set udm errnum - if the anything in close session
   * fails, session will eventually timeout anyways
   */

  dev->io.outofband.lan_session_state = IPMI_LAN_SESSION_STATE_CLOSE_SESSION;
  if (ipmi_lan_close_session (dev) < 0)
    goto cleanup;

 cleanup:
  if (dev->io.outofband.local_sockfd)
    close (dev->io.outofband.local_sockfd);
  _ipmi_outofband_free (dev);
}

static void
_ipmi_inband_close (ipmi_device_t dev)
{
  /* Function Note: No need to set UDM errnum - just return */
  assert(dev 
	 && dev->magic == IPMI_UDM_DEVICE_MAGIC
	 && (dev->type == IPMI_DEVICE_KCS
	     || dev->type == IPMI_DEVICE_SMIC
	     || dev->type == IPMI_DEVICE_BT
	     || dev->type == IPMI_DEVICE_SSIF
	     || dev->type == IPMI_DEVICE_OPENIPMI));
  
  _ipmi_inband_free (dev);
}

int 
ipmi_close_device (ipmi_device_t dev)
{
  UDM_ERR_DEV_CHECK (dev && dev->magic == IPMI_UDM_DEVICE_MAGIC);

  UDM_ERR_DEVICE_NOT_OPEN(dev->type != IPMI_DEVICE_UNKNOWN);

  if (dev->type != IPMI_DEVICE_LAN
      && dev->type != IPMI_DEVICE_KCS
      && dev->type != IPMI_DEVICE_SMIC
      && dev->type != IPMI_DEVICE_BT
      && dev->type != IPMI_DEVICE_SSIF
      && dev->type != IPMI_DEVICE_OPENIPMI)
    {
      dev->errnum = IPMI_ERR_INTERNAL_ERROR;
      return -1;
    }

  if (dev->type == IPMI_DEVICE_LAN)
    _ipmi_outofband_close (dev);
  else
    _ipmi_inband_close (dev);

  dev->type = IPMI_DEVICE_UNKNOWN;
  dev->errnum = IPMI_ERR_SUCCESS;
  return (0);
}

void
ipmi_device_destroy (ipmi_device_t dev)
{
  ERR_VOID_RETURN (dev && dev->magic == IPMI_UDM_DEVICE_MAGIC);

  if (dev->type != IPMI_DEVICE_UNKNOWN)
    ipmi_close_device(dev);
  
  free(dev);
}
