/*
   argp-common.h: common work for argp for freeipmi tools.
   Copyright (C) 2005 FreeIPMI Core Team

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#ifndef _ARGP_COMMON_H
#define _ARGP_COMMON_H

#if HAVE_CONFIG_H
#include "config.h"
#endif

#include <argp.h>

#include "freeipmi/udm/ipmi-udm.h"

enum argp_common_option_keys
  { 
    DRIVER_TYPE_KEY = 'D', 
    DUMMY_KEY = 129, 
    NO_PROBING_KEY = 130, 
    DRIVER_ADDRESS_KEY = 131, 
    DRIVER_DEVICE_KEY = 132, 
    RETRY_TIMEOUT_KEY = 133, 
    SESSION_TIMEOUT_KEY = 134,
    REGISTER_SPACING_KEY = 'r',
    HOSTNAME_KEY = 'h', 
    USERNAME_KEY = 'u', 
    PASSWORD_KEY = 'p', 
    PASSWORD_PROMPT_KEY = 'P',
    AUTHENTICATION_TYPE_KEY = 'a', 
    PRIVILEGE_LEVEL_KEY = 'l',
    FLUSH_CACHE_KEY = 'f',
    QUIET_CACHE_KEY = 'Q',
    SDR_CACHE_DIR_KEY = 135,
    BUFFER_KEY = 'B',
    CONSOLIDATE_KEY = 'C',
    FANOUT_KEY = 'F',
    ELIMINATE_KEY = 'E',
    DEBUG_KEY = 136
  };

#define ARGP_COMMON_OPTIONS_INBAND                                         \
    {"no-probing",     NO_PROBING_KEY, 0, 0, 	                           \
     "Do not probe IPMI devices.", 0},		                           \
    {"driver-type",    DRIVER_TYPE_KEY, "IPMIDRIVER", 0, 	           \
     "Use this IPMIDRIVER instead of auto selection.  "		           \
     "Allowed values are KCS, SMIC, SSIF, LAN, and OPENIPMI.", 1},         \
    {"driver-address", DRIVER_ADDRESS_KEY, "DRIVERADDR", 0,                \
     "Use this DRIVERADDR address instead of probed one.", 2}, 	           \
    {"driver-device",  DRIVER_DEVICE_KEY, "DEVICE", 0,                     \
     "Use this DEVICE for IPMI driver.", 3},                               \
    {"register-spacing", REGISTER_SPACING_KEY, "REGISTERSPACING", 0,       \
     "Use this REGISTERSPACING instead of the probed one", 4}

#define ARGP_COMMON_OPTIONS_OUTOFBAND_COMMON                                      \
    {"username",       USERNAME_KEY, "USERNAME", 0, 			          \
     "Use USERNAME instead of NULL.  Maximum USERNAME length is 16.", 6},         \
    {"password",       PASSWORD_KEY, "PASSWORD", 0,	                          \
     "Use PASSWORD instead of NULL.  Maximum PASSWORD length is 16.", 7},         \
    {"password-prompt", PASSWORD_PROMPT_KEY, 0, 0,	                          \
     "Prompt for PASSWORD instead of NULL.  Maximum PASSWORD length is 16.", 8},  \
    {"retry-timeout", RETRY_TIMEOUT_KEY, "RETRY_TIMEOUT", 0,                      \
     "Use RETRY_TIMEOUT milliseconds before re-sending LAN packets.", 9},         \
    {"session-timeout", SESSION_TIMEOUT_KEY, "SESSION_TIMEOUT", 0,                \
     "Use SESSION_TIMEOUT milliseconds before ending a session.", 10}

#define ARGP_COMMON_OPTIONS_OUTOFBAND                                             \
    {"hostname",       HOSTNAME_KEY, "IPMIHOST", 0, 			          \
     "Connect to IPMIHOST.", 5},					          \
    ARGP_COMMON_OPTIONS_OUTOFBAND_COMMON

#define ARGP_COMMON_OPTIONS_OUTOFBAND_HOSTRANGED                                  \
    {"hostname",       HOSTNAME_KEY, "IPMIHOST", 0, 			          \
     "Connect to IPMIHOST.  IPMIHOST may include host ranges", 5},                \
    ARGP_COMMON_OPTIONS_OUTOFBAND_COMMON

#define ARGP_COMMON_OPTIONS_AUTHTYPE                                       \
    {"auth-type",      AUTHENTICATION_TYPE_KEY, "AUTHTYPE", 0, 		   \
     "Use AUTHTYPE instead of MD5.  "				           \
     "Allowed values are NONE, MD2, MD5, and PLAIN.", 11}	           \

#define ARGP_COMMON_OPTIONS_PRIVLEVEL_USER                                 \
    {"priv-level",     PRIVILEGE_LEVEL_KEY, "PRIVILEGE-LEVEL", 0, 	   \
     "Use this PRIVILEGE-LEVEL instead of USER.  "		           \
     "Allowed values are CALLBACK, USER, OPERATOR, ADMIN and OEM.", 12}     

#define ARGP_COMMON_OPTIONS_PRIVLEVEL_ADMIN                                \
    {"priv-level",     PRIVILEGE_LEVEL_KEY, "PRIVILEGE-LEVEL", 0, 	   \
     "Use this PRIVILEGE-LEVEL instead of ADMIN.  "		           \
     "Allowed values are CALLBACK, USER, OPERATOR, ADMIN and OEM.", 12}     

#define ARGP_COMMON_SDR_OPTIONS                                            \
    {"flush-cache", FLUSH_CACHE_KEY,  0, 0,                                \
     "Flush sensor SDR cache.", 13},                                       \
    {"quiet-cache", QUIET_CACHE_KEY,  0, 0,                                \
     "Do not output cache creation information.", 14},                     \
    {"sdr-cache-directory", SDR_CACHE_DIR_KEY, "DIRECTORY", 0,             \
     "Use DIRECTORY for sensor cache.", 15} 

#define ARGP_COMMON_HOSTRANGED_OPTIONS                                     \
    {"buffer-output", BUFFER_KEY, 0, 0,                                    \
      "Buffer hostranged output.", 16},                                    \
    {"consolidate-output", CONSOLIDATE_KEY, 0, 0,                          \
     "Consolidate hostranged output.", 17},                                \
    {"fanout", FANOUT_KEY, "NUM", 0,                                       \
     "Set multiple host fanout.", 18},                                     \
    {"eliminate", ELIMINATE_KEY, 0, 0,                                     \
     "Eliminate undetected nodes.", 19}

#ifndef NDEBUG

#define ARGP_COMMON_OPTIONS_DEBUG                                          \
    {"debug",     DEBUG_KEY, 0, 0, 	                                   \
     "Turn on debugging.", 20}                                             

#endif


struct common_cmd_args 
{
  int disable_auto_probe;
  ipmi_driver_type_t driver_type;
  unsigned int driver_address;
  char *driver_device;
  unsigned int register_spacing;
  unsigned int session_timeout;
  unsigned int retry_timeout;
  char *host;
  char *username;
  char *password;
  int authentication_type;
  int privilege_level;
#ifndef NDEBUG
  int debug;
#endif /* NDEBUG */
};

struct sdr_cmd_args
{  
  int flush_cache_wanted;
  int quiet_cache_wanted;
  int sdr_cache_dir_wanted;
  char *sdr_cache_dir;
};

struct hostrange_cmd_args
{
  int buffer_hostrange_output;
  int consolidate_hostrange_output;
  int fanout;
  int eliminate;
};

error_t common_parse_opt (int key, 
			  char *arg, 
			  struct argp_state *state, 
			  struct common_cmd_args *common_cmd_args);

error_t sdr_parse_opt (int key, 
                       char *arg, 
                       struct argp_state *state, 
                       struct sdr_cmd_args *sdr_cmd_args);

error_t hostrange_parse_opt (int key, 
                             char *arg, 
                             struct argp_state *state, 
                             struct hostrange_cmd_args *hostrange_cmd_args);

void init_common_cmd_args (struct common_cmd_args *cmd_args);
void free_common_cmd_args (struct common_cmd_args *cmd_args);

void init_sdr_cmd_args (struct sdr_cmd_args *cmd_args);
void free_sdr_cmd_args (struct sdr_cmd_args *cmd_args);

void init_hostrange_cmd_args (struct hostrange_cmd_args *cmd_args);
void free_hostrange_cmd_args (struct hostrange_cmd_args *cmd_args);

#endif
