/* Copyright (C) 2000-2006 by George Williams */
/*
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.

 * The name of the author may not be used to endorse or promote products
 * derived from this software without specific prior written permission.

 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "pfaeditui.h"
#include "ustring.h"
#include <utype.h>

static struct psaltnames {
    char *name;
    int unicode;
    int provenance;		/* 1=> Adobe PUA, 2=>AMS PUA, 3=>TeX */
} psaltnames[];

static NameList agl_sans, agl, adobepua, greeksc, tex, ams;
NameList *namelist_for_new_fonts = &agl;

	/* Adobe's standard names are wrong for: */
	/* 0x0162 is named Tcommaaccent, 0x21A should be */
	/* 0x0163 is named tcommaaccent, 0x21B should be */
/* The following have been corrected (at least by intent) or removed */
	/* 0xf6be is named dotlessj, 0x237 should be */
	/* 0x2206 is named Delta, 0x394 should be */
	/* 0x2126 is named Omega, 0x3A9 should be */
	/* 0x00b5 is named mu, 0x3BC should be */

static int psnamesinited=false;
#define HASH_SIZE	257
struct psbucket { const char *name; int uni; struct psbucket *prev; } *psbuckets[HASH_SIZE];

static int hashname(const char *_name) {
    const unsigned char *name = (const unsigned char *) _name;
    int hash=0;

    while ( *name ) {
	if ( *name<=' ' )
    break;
	hash = (hash<<3)|((hash>>29)&0x7);
	hash ^= *name++-(' '+1);
    }
    hash ^= (hash>>16);
    hash &= 0xffff;
return( hash%HASH_SIZE );
}

static void psaddbucket(const char *name, int uni) {
    int hash = hashname(name);
    struct psbucket *buck = gcalloc(1,sizeof(struct psbucket));

    buck->name = name;
    buck->uni = uni;
    buck->prev = psbuckets[hash];
    psbuckets[hash] = buck;
}

static void NameListHash(NameList *nl) {
    int i,j,k;
    for ( i=0; i<17; ++i ) if ( nl->unicode[i]!=NULL ) {
	for ( j=0; j<256; ++j ) if ( nl->unicode[i][j]!=NULL ) {
	    for ( k=0; k<256; ++k ) if ( nl->unicode[i][j][k]!=NULL )
		psaddbucket(nl->unicode[i][j][k],(i<<16)|(j<<8)|k );
	}
    }
}

static void psinitnames(void) {
    int i;
    NameList *nl;

    agl.next = &agl_sans;
    agl_sans.next = &adobepua;
    adobepua.next = &greeksc;
    greeksc.next = &tex;
    tex.next = &ams;

    for ( i=0; psaltnames[i].name!=NULL ; ++i )
	psaddbucket(psaltnames[i].name,psaltnames[i].unicode);
    for ( nl=&agl; nl!=NULL; nl=nl->next )
	NameListHash(nl);
    psnamesinited = true;
}

static void psreinitnames(void) {
    /* If we reread a (loaded) namelist file, then we must remove the old defn*/
    /*  which means we must remove all the old hash entries before we can put */
    /*  in the new ones */
    int i;
    struct psbucket *cur, *prev;
    NameList *nl;

    for ( i=0; i<HASH_SIZE; ++i ) {
	for ( cur = psbuckets[i]; cur!=NULL; cur=prev ) {
	    prev = cur->prev;
	    chunkfree(cur,sizeof(struct psbucket));
	}
	psbuckets[i] = NULL;
    }

    for ( i=0; psaltnames[i].name!=NULL ; ++i )
	psaddbucket(psaltnames[i].name,psaltnames[i].unicode);
    for ( nl=&agl; nl!=NULL; nl=nl->next )
	NameListHash(nl);
}

int UniFromName(const char *name,enum uni_interp interp,Encoding *encname) {
    int i = -1;
    char *end;
    struct psbucket *buck;

    if ( strncmp(name,"uni",3)==0 ) {
	i = strtol(name+3,&end,16);
	if ( *end || end-name!=7 )	/* uniXXXXXXXX means a ligature of uniXXXX and uniXXXX */
	    i = -1;
    } else if ( (name[0]=='U' || name[0]=='u') && name[1]=='+' &&
	    (strlen(name)==6 || strlen(name)==7)) {
	/* Unifont uses this convention */
	i = strtol(name+2,&end,16);
	if ( *end )
	    i = -1;
    } else if ( name[0]=='u' && strlen(name)>=5 ) {
	i = strtol(name+1,&end,16);
	if ( *end )
	    i = -1;
	else if ( !encname->is_unicodefull &&
		(interp==ui_ams || interp==ui_trad_chinese)) {
	    int j;
	    extern const int cns14pua[], amspua[];
	    const int *pua = interp==ui_ams ? amspua : cns14pua;
	    for ( j=0xf8ff-0xe000; j>=0; --j )
		if ( pua[j]==i ) {
		    i = j+0xe000;
	    break;
		}
	}
    } else if ( name[0]!='\0' && name[1]=='\0' )
	i = ((unsigned char *) name)[0];
    if ( i==-1 ) {
	if ( !psnamesinited )
	    psinitnames();
	for ( buck = psbuckets[hashname(name)]; buck!=NULL; buck=buck->prev )
	    if ( strcmp(buck->name,name)==0 )
	break;
	if ( buck!=NULL )
	    i = buck->uni;
    }
return( i );
}

const char *StdGlyphName(char *buffer, int uni,enum uni_interp interp,NameList *for_this_font) {
    const char *name = NULL;
    NameList *nl;
    int up, ub, uc;

    if ( for_this_font==NULL )
	for_this_font = namelist_for_new_fonts;
    else if ( for_this_font==(NameList *) -1 )
	for_this_font = &agl;
    if ( (uni>=0 && uni<' ') ||
	    (uni>=0x7f && uni<0xa0) )
	/* standard controls */;
    else if ( uni!=-1  ) {
	if ( uni>=0xe000 && uni<=0xf8ff &&
		(interp==ui_trad_chinese || for_this_font==&ams)) {
	    extern const int cns14pua[], amspua[];
	    const int *pua = interp==ui_trad_chinese ? cns14pua : amspua;
	    if ( pua[uni-0xe000]!=0 )
		uni = pua[uni-0xe000];
	}
	up = uni>>16;
	ub = (uni&0xff00)>>8;
	uc = (uni&0xff);
	if ( up<17 )
	    for ( nl=for_this_font; nl!=NULL; nl=nl->basedon ) {
		if ( nl->unicode[up]!=NULL && nl->unicode[up][ub]!=NULL &&
			(name = nl->unicode[up][ub][uc])!=NULL )
	    break;
	}
    }
    if ( name==NULL ) {
	if ( uni>=0x10000 )
	    sprintf( buffer, "u%04X", uni);
	else
	    sprintf( buffer, "uni%04X", uni);
	name = buffer;
    }
return( name );
}

#define RefMax	40

static int transcmp(RefChar *r1, RefChar *r2) {
    double d1, d2;

    if ( r1->transform[4]<r2->transform[4] )
return( -1 );
    else if ( r1->transform[4]>r2->transform[4] )
return(  1 );
    if ( (d1 = r1->transform[5])<0 ) d1 = -d1;
    if ( (d2 = r2->transform[5])<0 ) d2 = -d2;
    if ( d1<d2 )
return( -1 );
    else if ( d1==d2 )
return( 0 );
    else
return( 1 );
}

static int FindAllRefs(SplineChar *sc,SplineChar *rsc[RefMax], int *au) {
    RefChar *refs[RefMax], *alp[RefMax], *out[RefMax];
    RefChar *ref;
    int layer, rcnt, acnt, ocnt, i,j;
    int alluni;
    /* We also order the reference. The order stored in the splinechar doesn't*/
    /*  mean anything, so try and guess what is intended semantically. */

    if ( sc==NULL )
return( 0 );
    for ( layer=ly_fore; layer<sc->layer_cnt; ++layer )
	if ( sc->layers[layer].splines!=NULL || sc->layers[layer].images!=NULL )
return( 0 );
    rcnt = 0;
    for ( layer=ly_fore; layer<sc->layer_cnt; ++layer ) {
	for ( ref = sc->layers[layer].refs; ref!=NULL; ref = ref->next ) {
	    if ( rcnt>=RefMax )
return( 0 );
	    refs[rcnt++] = ref;
	}
    }
    alluni = true;
    for ( i=0; i<rcnt; ++i ) {
	if ( refs[i]->sc->unicodeenc==-1 ) {
	    alluni = false;
    break;
	}
    }
    if ( !alluni ) {
	/* If not all unicode we can't make any guesses about meaning, so */
	/*  order by transformation */
	for ( i=0; i<rcnt; ++i ) for ( j=i+1; j<rcnt; ++j ) {
	    if ( transcmp(refs[i],refs[j])>0 ) {
		ref = refs[i];
		refs[i] = refs[j];
		refs[j] = ref;
	    }
	}
    } else {
	acnt = 0;
	for ( i=0; i<rcnt; ++i ) {
	    if ( isalpha(refs[i]->sc->unicodeenc )) {
		alp[acnt++] = refs[i];
		--rcnt;
		for ( j=i; j<rcnt; ++j )
		    refs[j] = refs[j+1];
		--i;
	    }
	}
	for ( i=0; i<acnt; ++i ) for ( j=i+1; j<acnt; ++j ) {
	    if ( transcmp(alp[i],alp[j])>0 ) {
		ref = alp[i];
		alp[i] = alp[j];
		alp[j] = ref;
	    }
	}
	for ( i=0; i<rcnt; ++i ) for ( j=i+1; j<rcnt; ++j ) {
	    if ( transcmp(refs[i],refs[j])>0 ) {
		ref = refs[i];
		refs[i] = refs[j];
		refs[j] = ref;
	    }
	}
	if ( acnt!=0 ) {
	    int a=0, r=0;
	    real cutpoint;
	    ocnt = 0;
	    out[ocnt++] = alp[a++];
	    forever {
		if ( a<acnt ) cutpoint = (alp[a]->transform[4]+3*alp[a-1]->transform[4])/4;
		else		cutpoint = 1e30;
		while ( r<rcnt && refs[r]->transform[4]<cutpoint )
		    out[ocnt++] = refs[r++];
		if ( a>=acnt )
	    break;
		out[ocnt++] = alp[a++];
	    }
	    memcpy(refs,out,ocnt*sizeof(RefChar *));
	    rcnt = ocnt;
	}
    }
    for ( i=0; i<rcnt; ++i )
	rsc[i] = refs[i]->sc;
    /* alluni now means can be written as uniXXXX.XXXX.XXXX... */
    for ( i=0; i<rcnt; ++i ) {
	if ( refs[i]->sc->unicodeenc>0x10000 ) {
	    alluni = false;
    break;
	}
    }
    *au = alluni;
return( rcnt );
}

/* Return a list of all alternate or standard glyph names for this encoding */
char **AllGlyphNames(int uni, NameList *for_this_font, SplineChar *sc) {
    int cnt, k, j, i, len;
    NameList *nl, *nl2, *nl3;
    char **names = NULL;
    const char *name;
    int up, ub, uc;
    char buffer[40], *pt;
    SplineChar *refs[RefMax];
    int rcnt, alluni;

    rcnt = FindAllRefs(sc,refs,&alluni);

    up = uni>>16;
    ub = (uni&0xff00)>>8;
    uc = (uni&0xff);

    for ( k=0; k<2; ++k ) {
	cnt = 0;
	/* try the default namelist first to put that at the head of the list */
	name = NULL;
	nl = nl3 = NULL;
	if ( uni>=0 && up<17 ) {
	    if ( for_this_font!=NULL ) {
		for ( nl3=for_this_font; nl3!=NULL; nl3=nl3->basedon ) {
		    if ( nl3->unicode[up]!=NULL && nl3->unicode[up][ub]!=NULL &&
			    (name = nl3->unicode[up][ub][uc])!=NULL )
		break;
		}
		if ( name!=NULL ) {
		    if ( names )
			names[cnt] = copy(name);
		    ++cnt;
		}
	    }
	    if ( for_this_font!=namelist_for_new_fonts ) {
		for ( nl=namelist_for_new_fonts; nl!=NULL; nl=nl->basedon ) if ( nl!=nl3 ) {
		    if ( nl->unicode[up]!=NULL && nl->unicode[up][ub]!=NULL &&
			    (name = nl->unicode[up][ub][uc])!=NULL )
		break;
		}
		if ( name!=NULL ) {
		    if ( names )
			names[cnt] = copy(name);
		    ++cnt;
		}
	    }
	    for ( nl2 = &agl; nl2!=NULL; nl2=nl2->next ) if ( nl2!=nl && nl2!=nl3) {
		if ( nl2->unicode[up]!=NULL && nl2->unicode[up][ub]!=NULL &&
			(name = nl2->unicode[up][ub][uc])!=NULL ) {
		    if ( names )
			names[cnt] = copy(name);
		    ++cnt;
		}
	    }
	    for ( i=0; psaltnames[i].name!=NULL ; ++i ) {
		if ( psaltnames[i].unicode==uni ) {
		    if ( names )
			names[cnt] = copy(psaltnames[i].name);
		    ++cnt;
		}
	    }
	    if ( uni<0x10000 ) {
		if ( names ) {
		    sprintf( buffer, "uni%04X", uni);
		    names[cnt] = copy(buffer);
		}
		++cnt;
	    }
	    if ( names ) {
		sprintf( buffer, "u%04X", uni);
		names[cnt] = copy(buffer);
	    }
	    ++cnt;
	}
	if ( rcnt>1 && alluni && (uni<0 || (uni>=0xe000 && uni<0xf900) || uni>=0xf0000 ) ) {
	    if ( names ) {
		names[cnt] = galloc(4+4*rcnt);
		strcpy(names[cnt],"uni");
		pt = names[cnt]+3;
		for ( i=0; i<rcnt; ++i ) {
		    if ( refs[i]->unicodeenc==0x131 || refs[i]->unicodeenc==0x237 ||
			    refs[i]->unicodeenc==0xf6be )
			sprintf( pt,"%04X", refs[i]->unicodeenc==0x131?'i':'j' );
		    else
			sprintf( pt,"%04X", CanonicalCombiner(refs[i]->unicodeenc));
		    pt += 4;
		}
	    }
	    ++cnt;
	}
	if ( rcnt>1 ) {
	    if ( names ) {
		for ( i=len=0; i<rcnt; ++i )
		    len += strlen( refs[i]->name )+1;
		names[cnt] = pt = galloc(len);
		for ( i=len=0; i<rcnt; ++i ) {
		    strcpy(pt,refs[i]->name);
		    pt += strlen(pt);
		    *pt++ = '_';
		}
		pt[-1] = '\0';
	    }
	    ++cnt;
	}
	if ( uni<0 || up>=17 ) {
	    if ( names )
		names[cnt] = copy(".notdef");
	    ++cnt;
	}
	if ( k==0 ) {
	    names = galloc((cnt+1)*sizeof(char *));
	    names[cnt] = NULL;
	}
    }
    /* Remove any names from multiiple namelists */
    for ( i=0; i<cnt; ++i ) for ( j=i+1; j<cnt; ++j ) {
	if ( strcmp(names[i],names[j])==0 ) {
	    for ( k=j+1; k<cnt; ++k )
		names[k-1] = names[k];
	    names[--cnt] = NULL;
	    --j;
	}
    }
return( names );
}

char **AllNamelistNames(void) {
    NameList *nl;
    int cnt;
    char **names;

    for ( nl = &agl, cnt=0; nl!=NULL; nl=nl->next, ++cnt );
    names = galloc((cnt+1) *sizeof(char *));
    for ( nl = &agl, cnt=0; nl!=NULL; nl=nl->next, ++cnt )
	names[cnt] = copy(_(nl->title));
    names[cnt] = NULL;
return( names );
}

#if 0
uint8 *AllNamelistUnicodes(void) {
    NameList *nl;
    int cnt;
    uint8 *uses;

    for ( nl = &agl, cnt=0; nl!=NULL; nl=nl->next, ++cnt );
    uses = galloc((cnt+1) *sizeof(uint8));
    for ( nl = &agl, cnt=0; nl!=NULL; nl=nl->next, ++cnt )
	uses[cnt] = nl->uses_unicode;
    uses[cnt] = 0xff;
return( uses );
}
#endif

NameList *DefaultNameListForNewFonts(void) {
return( namelist_for_new_fonts );
}

NameList *NameListByName(char *name) {
    NameList *nl;
    for ( nl = &agl; nl!=NULL; nl=nl->next ) {
	if ( strcmp(_(nl->title),name)==0 || strcmp(nl->title,name)==0 )
return( nl );
    }
return( NULL );
}

static void NameListFreeContents(NameList *nl) {
    int np, nb, nc, i;

    for ( np=0; np<17; ++np ) if ( nl->unicode[np]!=NULL ) {
	for ( nb=0; nb<256; ++nb ) if ( nl->unicode[np][nb]!=NULL ) {
	    for ( nc=0; nc<256; ++nc ) if ( nl->unicode[np][nb][nc]!=NULL )
		free((char *)nl->unicode[np][nb][nc] );
	    free( (char **) nl->unicode[np][nb]);
	}
	free( (char ***) nl->unicode[np]);
    }
    free( nl->title );
    if ( nl->renames!=NULL ) {
	for ( i=0; nl->renames[i].from!=NULL; ++i ) {
	    free(nl->renames[i].from);
	    free(nl->renames[i].to);
	}
	free(nl->renames);
    }
    free(nl->a_utf8_name);
}

static void NameListFree(NameList *nl) {
    NameListFreeContents(nl);
    chunkfree(nl,sizeof(NameList));
}
/* ************************************************************************** */

#include <sys/types.h>
#include <dirent.h>

NameList *LoadNamelist(char *filename) {
    FILE *file = fopen(filename,"r");
    NameList *nl, *nl2;
    char buffer[400];
    char *pt, *end, *test;
    int uni;
    int len;
    int up, ub, uc;
    int rn_cnt=0, rn_max = 0;
    int uses_unicode = false;
# if defined(FONTFORGE_CONFIG_GTK)
    gsize read, written;
# endif

    if ( file==NULL )
return( NULL );

    if ( !psnamesinited )
	psinitnames();

    nl = chunkalloc(sizeof(NameList));
    pt = strrchr(filename,'/');
    if ( pt==NULL ) pt = filename; else ++pt;
# if defined(FONTFORGE_CONFIG_GTK)
    nl->title = g_filename_to_utf8(pt,-1,&read,&written,NULL);
# else
    nl->title = def2utf8_copy(pt);
# endif
    pt = strrchr(nl->title,'.');
    if ( pt!=NULL ) *pt = '\0';

    while ( fgets(buffer,sizeof(buffer),file)!=NULL ) {
	if ( buffer[0]=='#' || buffer[0]=='\n' || buffer[0]=='\r' )
    continue;
	len = strlen( buffer );
	if ( buffer[len-1]=='\n' ) buffer[--len] = '\0';
	if ( buffer[len-1]=='\r' ) buffer[--len] = '\0';
	if ( strncmp(buffer,"Based:",6)==0 ) {
	    for ( pt=buffer+6; *pt==' ' || *pt=='\t'; ++pt);
	    for ( nl2 = &agl; nl2!=NULL; nl2 = nl2->next )
		if ( strcmp( nl2->title,pt )==0 )
	    break;
	    if ( nl2==NULL ) {
		gwwv_post_error(_("NameList base missing"),_("NameList %s based on %s which could not be found"), nl->title, pt );
		NameListFree(nl);
return( NULL );
	    } else if ( nl->basedon!=NULL ) {
		gwwv_post_error(_("NameList based twice"),_("NameList %s based on two NameLists"), nl->title );
		NameListFree(nl);
return( NULL );
	    }
	    nl->basedon = nl2;
	} else if ( strncmp(buffer,"Rename:",7)==0 ) {
	    for ( pt=buffer+7; *pt==' ' || *pt=='\t'; ++pt);
	    for ( test=pt; *test!=' ' && *test!='\t' && *test!='\0'; ++test );
	    if ( *test=='\0' ) {
		gwwv_post_error(_("NameList parsing error"),_("Missing rename \"to\" name %s\n%s"), nl->title, buffer );
		NameListFree(nl);
return( NULL );
	    }
	    *test='\0';
	    for ( ++test; *test==' ' || *test=='\t'; ++test);
	    if ( (test[0]=='-' || test[0]=='=') && test[1]=='>' )
		for ( test+=2; *test==' ' || *test=='\t'; ++test);
	    if ( *test=='\0' ) {
		gwwv_post_error(_("NameList parsing error"),_("Missing rename \"to\" name %s\n%s"), nl->title, buffer );
		NameListFree(nl);
return( NULL );
	    }
	    if ( rn_cnt>=rn_max-1 )
		nl->renames = grealloc(nl->renames,(rn_max+=20)*sizeof(struct renames));
	    nl->renames[rn_cnt].from   = copy(pt);
	    nl->renames[rn_cnt].to     = copy(test);
	    nl->renames[++rn_cnt].from = NULL;		/* End mark */
	} else {
	    pt = buffer;
	    if ( *pt=='0' && (pt[1]=='x' || pt[1]=='X'))
		pt += 2;
	    else if (( *pt=='u' || *pt=='U') && pt[1]=='+' )
		pt += 2;
	    uni = strtol(pt,&end,16);
	    if ( end==pt || uni<0 || uni>=unicode4_size ) {
		gwwv_post_error(_("NameList parsing error"),_("Bad unicode value when parsing %s\n%s"), nl->title, buffer );
		NameListFree(nl);
return( NULL );
	    }
	    pt = end;
	    while ( *pt==' ' || *pt==';' || *pt=='\t' ) ++pt;
	    if ( *pt=='\0' ) {
		gwwv_post_error(_("NameList parsing error"),_("Missing name when parsing %s for unicode %x"), nl->title, uni );
		NameListFree(nl);
return( NULL );
	    }
	    for ( test=pt; *test; ++test ) {
		if ( (*test<=' ' && *test>=0) ||
		    *test=='(' || *test=='[' || *test=='{' || *test=='<' ||
		    *test==')' || *test==']' || *test=='}' || *test=='>' ||
		    *test=='%' || *test=='/' ) {
		    gwwv_post_error(_("NameList parsing error"),_("Bad name when parsing %s for unicode %x"), nl->title, uni );
		    NameListFree(nl);
return( NULL );
		}
		if ( *test&0x80 ) {
		    uses_unicode = true;
		    if ( nl->a_utf8_name==NULL )
			nl->a_utf8_name = copy(pt);
		}
	    }
	    up = uni>>16;
	    ub = (uni&0xff00)>>8;
	    uc = uni&0xff;
	    if ( nl->unicode[up]==NULL )
		nl->unicode[up] = gcalloc(256,sizeof(char **));
	    if ( nl->unicode[up][ub]==NULL )
		nl->unicode[up][ub] = gcalloc(256,sizeof(char *));
	    if ( nl->unicode[up][ub][uc]==NULL )
		nl->unicode[up][ub][uc]=copy(pt);
	    else {
		gwwv_post_error(_("NameList parsing error"),_("Multiple names when parsing %s for unicode %x"), nl->title, uni );
		NameListFree(nl);
return( NULL );
	    }
	}
    }

    nl->uses_unicode = uses_unicode;
    if ( nl->basedon!=NULL && nl->basedon->uses_unicode )
	nl->uses_unicode = true;
    fclose(file);
    for ( nl2 = &agl; nl2->next!=NULL; nl2=nl2->next ) {
	if ( strcmp(nl2->title,nl->title)==0 ) {	/* Replace old version */
	    NameList *next = nl2->next;
	    NameListFreeContents(nl2);
	    *nl2 = *nl;
	    nl2->next = next;
	    chunkfree(nl,sizeof(NameList));
	    psreinitnames();
return( nl2 );
	}
    }
    NameListHash(nl);
    nl2->next = nl;
return( nl );
}

static int isnamelist(char *filename) {
    char *pt;

    pt = strrchr(filename,'.');
    if ( pt==NULL )
return( false );
    if ( strcmp(pt,".nam")==0 )
return( true );

return( false );
}

void LoadNamelistDir(char *dir) {
    char prefdir[1024];
    DIR *diro;
    struct dirent *ent;
    char buffer[1025];

    if ( dir == NULL )
	dir = getPfaEditDir(prefdir);

    diro = opendir(dir);
    if ( diro==NULL )		/* It's ok not to have any */
return;
    
    while ( (ent = readdir(diro))!=NULL ) {
	if ( isnamelist(ent->d_name) ) {
	    sprintf( buffer, "%s/%s", dir, ent->d_name );
	    LoadNamelist(buffer);
	}
    }
    closedir(diro);
}
/* ************************************************************************** */
const char *RenameGlyphToNamelist(char *buffer, SplineChar *sc,NameList *old,NameList *new) {
    int i, up, ub, uc, ch;
    char space[80];		/* glyph names are supposed to be less<=31 chars */
    char tempbuf[32];
    char *pt, *start, *opt, *oend; const char *newsubname;
    SplineChar *tempsc;
    NameList *nl;

    if ( sc->unicodeenc!=-1 ) {
	up = sc->unicodeenc>>16;
	ub = (sc->unicodeenc>>8)&0xff;
	uc = (sc->unicodeenc&0xff);
	for ( nl=new; nl!=NULL; nl=nl->basedon )
	    if ( nl->unicode[up]!=NULL && nl->unicode[up][ub]!=NULL && nl->unicode[up][ub][uc]!=NULL )
return( nl->unicode[up][ub][uc] );
	if ( up==0 )
	    sprintf( buffer, "uni%04X", sc->unicodeenc );
	else
	    sprintf( buffer, "u%04X", sc->unicodeenc );
return( buffer );
    } else {
	if ( old!=NULL && old->renames!=NULL ) {
	    for ( i=0; old->renames[i].from!=NULL; ++i )
		if ( strcmp(sc->name,old->renames[i].from)==0 )
return( old->renames[i].to );
	}
	if ( new->renames!=NULL ) {
	    for ( i=0; new->renames[i].from!=NULL; ++i )
		if ( strcmp(sc->name,new->renames[i].to)==0 )
return( new->renames[i].from );
	}
	if ( strlen(sc->name)>sizeof(space)-1 )
return( sc->name );
	strcpy(space,sc->name);
	opt = buffer; oend = buffer+31;
	start = space;
	/* Check for composite names f_i, A.small */
	while ( *start ) {
	    for ( pt=start; *pt!='\0' && *pt!='.' && *pt!='_'; ++pt );
	    if ( *pt=='\0' && start==space )
return( sc->name );
	    ch = *pt;
	    *pt = '\0';
	    tempsc = SFGetChar(sc->parent,-1,start);
	    if ( tempsc==NULL )
return( sc->name );
	    newsubname = RenameGlyphToNamelist(tempbuf,tempsc,old,new);
	    while ( opt<oend && *newsubname )
		*opt++ = *newsubname++;
	    if ( *newsubname )
return( sc->name );
	    if ( ch=='\0' ) {
		*opt = '\0';
return( buffer );
	    } else if ( ch=='.' ) {
		/* don't attempt to translate anything after a '.' just copy that litterally */
		while ( opt<oend && *pt )
		    *opt++ = *pt++;
		if ( *pt )
return( sc->name );
		*opt = '\0';
return( buffer );
	    } else {		/* _ */
		*opt++ = '_';
		start = pt+1;
	    }
	}
	*opt = '\0';
return( buffer );
    }
}

void SFRenameGlyphsToNamelist(SplineFont *sf,NameList *new) {
    int gid;
    char buffer[40]; const char *name;
    SplineChar *sc;

    if ( new==NULL )
return;

    for ( gid = 0; gid<sf->glyphcnt; ++gid ) if ( (sc=sf->glyphs[gid])!=NULL ) {
	name = RenameGlyphToNamelist(buffer,sc,sf->for_new_glyphs,new);
	if ( name!=sc->name ) {
	    free(sc->name);
	    sc->name = copy(name);
	}
    }
    sf->for_new_glyphs = new;
}

char **SFTemporaryRenameGlyphsToNamelist(SplineFont *sf,NameList *new) {
    int gid;
    char buffer[40]; const char *name;
    SplineChar *sc;
    char **ret;

    if ( new==NULL )
return( NULL );

    ret = gcalloc(sf->glyphcnt,sizeof(char *));
    for ( gid = 0; gid<sf->glyphcnt; ++gid ) if ( (sc=sf->glyphs[gid])!=NULL ) {
	name = RenameGlyphToNamelist(buffer,sc,sf->for_new_glyphs,new);
	if ( name!=sc->name ) {
	    ret[gid] = sc->name;
	    sc->name = copy(name);
	}
    }
return( ret );
}

void SFTemporaryRestoreGlyphNames(SplineFont *sf,char **former) {
    int gid;
    SplineChar *sc;

    for ( gid = 0; gid<sf->glyphcnt; ++gid ) if ( (sc=sf->glyphs[gid])!=NULL ) {
	if ( former[gid]!=NULL ) {
	    free(sc->name);
	    sc->name = former[gid];
	}
    }
    free(former);
}
/* ************************************************************************** */
static const char *agl_sans_p0_b0[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"space",
	"exclam",
	"quotedbl",
	"numbersign",
	"dollar",
	"percent",
	"ampersand",
	"quotesingle",
	"parenleft",
	"parenright",
	"asterisk",
	"plus",
	"comma",
	"hyphen",
	"period",
	"slash",
	"zero",
	"one",
	"two",
	"three",
	"four",
	"five",
	"six",
	"seven",
	"eight",
	"nine",
	"colon",
	"semicolon",
	"less",
	"equal",
	"greater",
	"question",
	"at",
	"A",
	"B",
	"C",
	"D",
	"E",
	"F",
	"G",
	"H",
	"I",
	"J",
	"K",
	"L",
	"M",
	"N",
	"O",
	"P",
	"Q",
	"R",
	"S",
	"T",
	"U",
	"V",
	"W",
	"X",
	"Y",
	"Z",
	"bracketleft",
	"backslash",
	"bracketright",
	"asciicircum",
	"underscore",
	"grave",
	"a",
	"b",
	"c",
	"d",
	"e",
	"f",
	"g",
	"h",
	"i",
	"j",
	"k",
	"l",
	"m",
	"n",
	"o",
	"p",
	"q",
	"r",
	"s",
	"t",
	"u",
	"v",
	"w",
	"x",
	"y",
	"z",
	"braceleft",
	"bar",
	"braceright",
	"asciitilde",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"exclamdown",
	"cent",
	"sterling",
	"currency",
	"yen",
	"brokenbar",
	"section",
	"dieresis",
	"copyright",
	"ordfeminine",
	"guillemotleft",
	"logicalnot",
	NULL,
	"registered",
	"macron",
	"degree",
	"plusminus",
	NULL,
	NULL,
	"acute",
	NULL,
	"paragraph",
	"periodcentered",
	"cedilla",
	NULL,
	"ordmasculine",
	"guillemotright",
	"onequarter",
	"onehalf",
	"threequarters",
	"questiondown",
	"Agrave",
	"Aacute",
	"Acircumflex",
	"Atilde",
	"Adieresis",
	"Aring",
	"AE",
	"Ccedilla",
	"Egrave",
	"Eacute",
	"Ecircumflex",
	"Edieresis",
	"Igrave",
	"Iacute",
	"Icircumflex",
	"Idieresis",
	"Eth",
	"Ntilde",
	"Ograve",
	"Oacute",
	"Ocircumflex",
	"Otilde",
	"Odieresis",
	"multiply",
	"Oslash",
	"Ugrave",
	"Uacute",
	"Ucircumflex",
	"Udieresis",
	"Yacute",
	"Thorn",
	"germandbls",
	"agrave",
	"aacute",
	"acircumflex",
	"atilde",
	"adieresis",
	"aring",
	"ae",
	"ccedilla",
	"egrave",
	"eacute",
	"ecircumflex",
	"edieresis",
	"igrave",
	"iacute",
	"icircumflex",
	"idieresis",
	"eth",
	"ntilde",
	"ograve",
	"oacute",
	"ocircumflex",
	"otilde",
	"odieresis",
	"divide",
	"oslash",
	"ugrave",
	"uacute",
	"ucircumflex",
	"udieresis",
	"yacute",
	"thorn",
	"ydieresis"
};

static const char *agl_sans_p0_b1[] = {
	"Amacron",
	"amacron",
	"Abreve",
	"abreve",
	"Aogonek",
	"aogonek",
	"Cacute",
	"cacute",
	"Ccircumflex",
	"ccircumflex",
	"Cdotaccent",
	"cdotaccent",
	"Ccaron",
	"ccaron",
	"Dcaron",
	"dcaron",
	"Dcroat",
	"dcroat",
	"Emacron",
	"emacron",
	"Ebreve",
	"ebreve",
	"Edotaccent",
	"edotaccent",
	"Eogonek",
	"eogonek",
	"Ecaron",
	"ecaron",
	"Gcircumflex",
	"gcircumflex",
	"Gbreve",
	"gbreve",
	"Gdotaccent",
	"gdotaccent",
	"Gcommaaccent",
	"gcommaaccent",
	"Hcircumflex",
	"hcircumflex",
	"Hbar",
	"hbar",
	"Itilde",
	"itilde",
	"Imacron",
	"imacron",
	"Ibreve",
	"ibreve",
	"Iogonek",
	"iogonek",
	"Idotaccent",
	"dotlessi",
	"IJ",
	"ij",
	"Jcircumflex",
	"jcircumflex",
	"Kcommaaccent",
	"kcommaaccent",
	"kgreenlandic",
	"Lacute",
	"lacute",
	"Lcommaaccent",
	"lcommaaccent",
	"Lcaron",
	"lcaron",
	"Ldot",
	"ldot",
	"Lslash",
	"lslash",
	"Nacute",
	"nacute",
	"Ncommaaccent",
	"ncommaaccent",
	"Ncaron",
	"ncaron",
	"napostrophe",
	"Eng",
	"eng",
	"Omacron",
	"omacron",
	"Obreve",
	"obreve",
	"Ohungarumlaut",
	"ohungarumlaut",
	"OE",
	"oe",
	"Racute",
	"racute",
	"Rcommaaccent",
	"rcommaaccent",
	"Rcaron",
	"rcaron",
	"Sacute",
	"sacute",
	"Scircumflex",
	"scircumflex",
	"Scedilla",
	"scedilla",
	"Scaron",
	"scaron",
	NULL,
	NULL,
	"Tcaron",
	"tcaron",
	"Tbar",
	"tbar",
	"Utilde",
	"utilde",
	"Umacron",
	"umacron",
	"Ubreve",
	"ubreve",
	"Uring",
	"uring",
	"Uhungarumlaut",
	"uhungarumlaut",
	"Uogonek",
	"uogonek",
	"Wcircumflex",
	"wcircumflex",
	"Ycircumflex",
	"ycircumflex",
	"Ydieresis",
	"Zacute",
	"zacute",
	"Zdotaccent",
	"zdotaccent",
	"Zcaron",
	"zcaron",
	"longs",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"florin",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"Ohorn",
	"ohorn",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"Uhorn",
	"uhorn",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"Gcaron",
	"gcaron",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"Aringacute",
	"aringacute",
	"AEacute",
	"aeacute",
	"Oslashacute",
	"oslashacute"
};

static const char *agl_sans_p0_b2[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"Scommaaccent",
	"scommaaccent",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"circumflex",
	"caron",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"breve",
	"dotaccent",
	"ring",
	"ogonek",
	"tilde",
	"hungarumlaut",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *agl_sans_p0_b3[] = {
	"gravecomb",
	"acutecomb",
	NULL,
	"tildecomb",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"hookabovecomb",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"dotbelowcomb",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"tonos",
	"dieresistonos",
	"Alphatonos",
	"anoteleia",
	"Epsilontonos",
	"Etatonos",
	"Iotatonos",
	NULL,
	"Omicrontonos",
	NULL,
	"Upsilontonos",
	"Omegatonos",
	"iotadieresistonos",
	"Alpha",
	"Beta",
	"Gamma",
	"Delta",
	"Epsilon",
	"Zeta",
	"Eta",
	"Theta",
	"Iota",
	"Kappa",
	"Lambda",
	"Mu",
	"Nu",
	"Xi",
	"Omicron",
	"Pi",
	"Rho",
	NULL,
	"Sigma",
	"Tau",
	"Upsilon",
	"Phi",
	"Chi",
	"Psi",
	"Omega",
	"Iotadieresis",
	"Upsilondieresis",
	"alphatonos",
	"epsilontonos",
	"etatonos",
	"iotatonos",
	"upsilondieresistonos",
	"alpha",
	"beta",
	"gamma",
	"delta",
	"epsilon",
	"zeta",
	"eta",
	"theta",
	"iota",
	"kappa",
	"lambda",
	"mu",
	"nu",
	"xi",
	"omicron",
	"pi",
	"rho",
	"sigma1",
	"sigma",
	"tau",
	"upsilon",
	"phi",
	"chi",
	"psi",
	"omega",
	"iotadieresis",
	"upsilondieresis",
	"omicrontonos",
	"upsilontonos",
	"omegatonos",
	NULL,
	NULL,
	"theta1",
	"Upsilon1",
	NULL,
	NULL,
	"phi1",
	"omega1",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *agl_sans_p0_b1e[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"Wgrave",
	"wgrave",
	"Wacute",
	"wacute",
	"Wdieresis",
	"wdieresis",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"Ygrave",
	"ygrave",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *agl_sans_p0_b20[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"figuredash",
	"endash",
	"emdash",
	NULL,
	NULL,
	"underscoredbl",
	"quoteleft",
	"quoteright",
	"quotesinglbase",
	"quotereversed",
	"quotedblleft",
	"quotedblright",
	"quotedblbase",
	NULL,
	"dagger",
	"daggerdbl",
	"bullet",
	NULL,
	"onedotenleader",
	"twodotenleader",
	"ellipsis",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"perthousand",
	NULL,
	"minute",
	"second",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"guilsinglleft",
	"guilsinglright",
	NULL,
	"exclamdbl",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"fraction",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"colonmonetary",
	NULL,
	"franc",
	"lira",
	NULL,
	NULL,
	"peseta",
	NULL,
	NULL,
	NULL,
	"dong",
	"Euro",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *agl_sans_p0_b21[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"Ifraktur",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"weierstrass",
	NULL,
	NULL,
	NULL,
	"Rfraktur",
	NULL,
	"prescription",
	NULL,
	NULL,
	NULL,
	"trademark",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"estimated",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"aleph",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"onethird",
	"twothirds",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"oneeighth",
	"threeeighths",
	"fiveeighths",
	"seveneighths",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"arrowleft",
	"arrowup",
	"arrowright",
	"arrowdown",
	"arrowboth",
	"arrowupdn",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"arrowupdnbse",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"carriagereturn",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"arrowdblleft",
	"arrowdblup",
	"arrowdblright",
	"arrowdbldown",
	"arrowdblboth",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *agl_sans_p0_b22[] = {
	"universal",
	NULL,
	"partialdiff",
	"existential",
	NULL,
	"emptyset",
	NULL,
	"gradient",
	"element",
	"notelement",
	NULL,
	"suchthat",
	NULL,
	NULL,
	NULL,
	"product",
	NULL,
	"summation",
	"minus",
	NULL,
	NULL,
	NULL,
	NULL,
	"asteriskmath",
	NULL,
	NULL,
	"radical",
	NULL,
	NULL,
	"proportional",
	"infinity",
	"orthogonal",
	"angle",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"logicaland",
	"logicalor",
	"intersection",
	"union",
	"integral",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"therefore",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"similar",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"congruent",
	NULL,
	NULL,
	"approxequal",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"notequal",
	"equivalence",
	NULL,
	NULL,
	"lessequal",
	"greaterequal",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"propersubset",
	"propersuperset",
	"notsubset",
	NULL,
	"reflexsubset",
	"reflexsuperset",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"circleplus",
	NULL,
	"circlemultiply",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"perpendicular",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"dotmath",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *agl_sans_p0_b23[] = {
	NULL,
	NULL,
	"house",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"revlogicalnot",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"integraltp",
	"integralbt",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"angleleft",
	"angleright",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *agl_sans_p0_b25[] = {
	"SF100000",
	NULL,
	"SF110000",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"SF010000",
	NULL,
	NULL,
	NULL,
	"SF030000",
	NULL,
	NULL,
	NULL,
	"SF020000",
	NULL,
	NULL,
	NULL,
	"SF040000",
	NULL,
	NULL,
	NULL,
	"SF080000",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"SF090000",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"SF060000",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"SF070000",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"SF050000",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"SF430000",
	"SF240000",
	"SF510000",
	"SF520000",
	"SF390000",
	"SF220000",
	"SF210000",
	"SF250000",
	"SF500000",
	"SF490000",
	"SF380000",
	"SF280000",
	"SF270000",
	"SF260000",
	"SF360000",
	"SF370000",
	"SF420000",
	"SF190000",
	"SF200000",
	"SF230000",
	"SF470000",
	"SF480000",
	"SF410000",
	"SF450000",
	"SF460000",
	"SF400000",
	"SF540000",
	"SF530000",
	"SF440000",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"upblock",
	NULL,
	NULL,
	NULL,
	"dnblock",
	NULL,
	NULL,
	NULL,
	"block",
	NULL,
	NULL,
	NULL,
	"lfblock",
	NULL,
	NULL,
	NULL,
	"rtblock",
	"ltshade",
	"shade",
	"dkshade",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"filledbox",
	"H22073",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"H18543",
	"H18551",
	"filledrect",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"triagup",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"triagrt",
	NULL,
	"triagdn",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"triaglf",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"lozenge",
	"circle",
	NULL,
	NULL,
	NULL,
	"H18533",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"invbullet",
	"invcircle",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"openbullet",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *agl_sans_p0_b26[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"smileface",
	"invsmileface",
	"sun",
	NULL,
	NULL,
	NULL,
	"female",
	NULL,
	"male",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"spade",
	NULL,
	NULL,
	"club",
	NULL,
	"heart",
	"diamond",
	NULL,
	NULL,
	NULL,
	"musicalnote",
	"musicalnotedbl",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char **agl_sans_p0[] = {
	agl_sans_p0_b0,
	agl_sans_p0_b1,
	agl_sans_p0_b2,
	agl_sans_p0_b3,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	agl_sans_p0_b1e,
	NULL,
	agl_sans_p0_b20,
	agl_sans_p0_b21,
	agl_sans_p0_b22,
	agl_sans_p0_b23,
	NULL,
	agl_sans_p0_b25,
	agl_sans_p0_b26,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static NameList agl_sans = {
	NULL,
	"AGL without afii",
	{ agl_sans_p0, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL }
};
/* ************************************************************************** */
static const char *agl_p0_b1[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"Tcommaaccent",
	"tcommaaccent",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *agl_p0_b2[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57929",
	"afii64937",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *agl_p0_b4[] = {
	NULL,
	"afii10023",
	"afii10051",
	"afii10052",
	"afii10053",
	"afii10054",
	"afii10055",
	"afii10056",
	"afii10057",
	"afii10058",
	"afii10059",
	"afii10060",
	"afii10061",
	NULL,
	"afii10062",
	"afii10145",
	"afii10017",
	"afii10018",
	"afii10019",
	"afii10020",
	"afii10021",
	"afii10022",
	"afii10024",
	"afii10025",
	"afii10026",
	"afii10027",
	"afii10028",
	"afii10029",
	"afii10030",
	"afii10031",
	"afii10032",
	"afii10033",
	"afii10034",
	"afii10035",
	"afii10036",
	"afii10037",
	"afii10038",
	"afii10039",
	"afii10040",
	"afii10041",
	"afii10042",
	"afii10043",
	"afii10044",
	"afii10045",
	"afii10046",
	"afii10047",
	"afii10048",
	"afii10049",
	"afii10065",
	"afii10066",
	"afii10067",
	"afii10068",
	"afii10069",
	"afii10070",
	"afii10072",
	"afii10073",
	"afii10074",
	"afii10075",
	"afii10076",
	"afii10077",
	"afii10078",
	"afii10079",
	"afii10080",
	"afii10081",
	"afii10082",
	"afii10083",
	"afii10084",
	"afii10085",
	"afii10086",
	"afii10087",
	"afii10088",
	"afii10089",
	"afii10090",
	"afii10091",
	"afii10092",
	"afii10093",
	"afii10094",
	"afii10095",
	"afii10096",
	"afii10097",
	NULL,
	"afii10071",
	"afii10099",
	"afii10100",
	"afii10101",
	"afii10102",
	"afii10103",
	"afii10104",
	"afii10105",
	"afii10106",
	"afii10107",
	"afii10108",
	"afii10109",
	NULL,
	"afii10110",
	"afii10193",
	NULL,
	NULL,
	"afii10146",
	"afii10194",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii10147",
	"afii10195",
	"afii10148",
	"afii10196",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii10050",
	"afii10098",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii10846",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *agl_p0_b5[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57799",
	"afii57801",
	"afii57800",
	"afii57802",
	"afii57793",
	"afii57794",
	"afii57795",
	"afii57798",
	"afii57797",
	"afii57806",
	NULL,
	"afii57796",
	"afii57807",
	"afii57839",
	"afii57645",
	"afii57841",
	"afii57842",
	"afii57804",
	"afii57803",
	"afii57658",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57664",
	"afii57665",
	"afii57666",
	"afii57667",
	"afii57668",
	"afii57669",
	"afii57670",
	"afii57671",
	"afii57672",
	"afii57673",
	"afii57674",
	"afii57675",
	"afii57676",
	"afii57677",
	"afii57678",
	"afii57679",
	"afii57680",
	"afii57681",
	"afii57682",
	"afii57683",
	"afii57684",
	"afii57685",
	"afii57686",
	"afii57687",
	"afii57688",
	"afii57689",
	"afii57690",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57716",
	"afii57717",
	"afii57718",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *agl_p0_b6[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57388",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57403",
	NULL,
	NULL,
	NULL,
	"afii57407",
	NULL,
	"afii57409",
	"afii57410",
	"afii57411",
	"afii57412",
	"afii57413",
	"afii57414",
	"afii57415",
	"afii57416",
	"afii57417",
	"afii57418",
	"afii57419",
	"afii57420",
	"afii57421",
	"afii57422",
	"afii57423",
	"afii57424",
	"afii57425",
	"afii57426",
	"afii57427",
	"afii57428",
	"afii57429",
	"afii57430",
	"afii57431",
	"afii57432",
	"afii57433",
	"afii57434",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57440",
	"afii57441",
	"afii57442",
	"afii57443",
	"afii57444",
	"afii57445",
	"afii57446",
	"afii57470",
	"afii57448",
	"afii57449",
	"afii57450",
	"afii57451",
	"afii57452",
	"afii57453",
	"afii57454",
	"afii57455",
	"afii57456",
	"afii57457",
	"afii57458",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57392",
	"afii57393",
	"afii57394",
	"afii57395",
	"afii57396",
	"afii57397",
	"afii57398",
	"afii57399",
	"afii57400",
	"afii57401",
	"afii57381",
	NULL,
	NULL,
	"afii63167",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57511",
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57506",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57507",
	NULL,
	"afii57512",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57513",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57508",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57505",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57509",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57514",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57519",
	NULL,
	NULL,
	"afii57534",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *agl_p0_b20[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii61664",
	"afii301",
	"afii299",
	"afii300",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii00208",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii61573",
	"afii61574",
	"afii61575",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii57636",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *agl_p0_b21[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii61248",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii61289",
	NULL,
	NULL,
	"afii61352",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char **agl_p0[] = {
	NULL,
	agl_p0_b1,
	agl_p0_b2,
	NULL,
	agl_p0_b4,
	agl_p0_b5,
	agl_p0_b6,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	agl_p0_b20,
	agl_p0_b21,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static NameList agl = {
	&agl_sans,
	N_("Adobe Glyph List"),
	{ agl_p0, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL }
};
/* ************************************************************************** */
static const char *adobepua_p0_bf6[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"perthousand.oldstyle",
	"cent.denominator",
	"dollar.denominator",
	"hyphen.denominator",
	"parenleft.denominator",
	"parenright.denominator",
	"cent.numerator",
	"dollar.numerator",
	"hyphen.numerator",
	"parenleft.numerator",
	"parenright.numerator",
	"at.cap",
	"commaaccent.cap",
	"commaaccent.small",
	"pi.math",
	"h.superior",
	"zero.slash",
	"zero.fitted",
	"two.fitted",
	"three.fitted",
	"four.fitted",
	"five.fitted",
	"six.fitted",
	"seven.fitted",
	"eight.fitted",
	"nine.fitted",
	"percent.oldstyle",
	"zero.taboldstyle",
	"one.taboldstyle",
	"two.taboldstyle",
	"three.taboldstyle",
	"four.taboldstyle",
	"five.taboldstyle",
	"six.taboldstyle",
	"seven.taboldstyle",
	"eight.taboldstyle",
	"nine.taboldstyle",
	"colonmonetary.taboldstyle",
	"Euro.taboldstyle",
	"florin.taboldstyle",
	"numbersign.taboldstyle",
	"sterling.taboldstyle",
	"yen.taboldstyle",
	"dollar.taboldstyle",
	"cent.taboldstyle",
	"zero.denominator",
	"one.denominator",
	"two.denominator",
	"three.denominator",
	"four.denominator",
	"five.denominator",
	"six.denominator",
	"seven.denominator",
	"eight.denominator",
	"nine.denominator",
	"comma.denominator",
	"period.denominator",
	"zero.numerator",
	"one.numerator",
	"two.numerator",
	"three.numerator",
	"four.numerator",
	"five.numerator",
	"six.numerator",
	"seven.numerator",
	"eight.numerator",
	"nine.numerator",
	"comma.numerator",
	"period.numerator",
	"Abreve.small",
	"Amacron.small",
	"Aogonek.small",
	"AEacute.small",
	"Cacute.small",
	"Ccaron.small",
	"Ccircumflex.small",
	"Cdotaccent.small",
	"Dcaron.small",
	"Dcroat.small",
	"Ebreve.small",
	"Ecaron.small",
	"Edotaccent.small",
	"Emacron.small",
	"Eng.small",
	"Eogonek.small",
	"Gbreve.small",
	"Gcircumflex.small",
	"Gcommaaccent.small",
	"Gdotaccent.small",
	"Hbar.small",
	"Hcircumflex.small",
	"Ibreve.small",
	"IJ.small",
	"Imacron.small",
	"Iogonek.small",
	"Itilde.small",
	"Jcircumflex.small",
	"Kcommaaccent.small",
	"Lacute.small",
	"Lcaron.small",
	"Lcommaaccent.small",
	"Ldot.small",
	"Nacute.small",
	"Ncaron.small",
	"Ncommaaccent.small",
	"Obreve.small",
	"Ohungarumlaut.small",
	"Omacron.small",
	"Oslashacute.small",
	"Racute.small",
	"Rcaron.small",
	"Rcommaaccent.small",
	"Sacute.small",
	"Scedilla.small",
	"Scircumflex.small",
	"Scommaaccent.small",
	"Tbar.small",
	"Tcaron.small",
	"Tcommaaccent.small",
	"Ubreve.small",
	"Uhungarumlaut.small",
	"Umacron.small",
	"Uogonek.small",
	"Uring.small",
	"Utilde.small",
	"Wacute.small",
	"Wcircumflex.small",
	"Wdieresis.small",
	"Wgrave.small",
	"Ycircumflex.small",
	"Ygrave.small",
	"Zacute.small",
	"Zdotaccent.small",
	"Idotaccent.small",
	"parenleft.cap",
	"parenright.cap",
	"bracketleft.cap",
	"bracketright.cap",
	"braceleft.cap",
	"braceright.cap",
	"exclamdown.cap",
	"questiondown.cap",
	"guillemotleft.cap",
	"guillemotright.cap",
	"guilsinglleft.cap",
	"guilsinglright.cap",
	"hyphen.cap",
	"endash.cap",
	"emdash.cap",
	"periodcentered.cap",
	"j.dotless",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"afii10068.ital",
	"afii10066.ital",
	"afii10069.ital",
	"afii10081.ital",
	"afii10084.ital",
	"acute.cap",
	"caron.cap",
	"dieresis.cap",
	"space_uni0308_uni0301.cap",
	"space_uni0308_uni0300.cap",
	"grave.cap",
	"hungarumlaut.cap",
	"macron.cap",
	"breve.cyrcap",
	"circumflex.cyrcap",
	"space_uni030F.cap",
	"breve.cyr",
	"circumflex.cyr",
	"space_uni030F",
	"space_uni0308_uni0301",
	"space_uni0308_uni0300",
	"copyright.serif",
	"registered.serif",
	"trademark.serif",
	"one.fitted",
	"R_p",
	"uni2014.alt",
	"cent.inferior",
	"cent.superior",
	"comma.inferior",
	"comma.superior",
	"dollar.inferior",
	"dollar.superior",
	"hyphen.inferior",
	"hyphen.superior",
	"period.inferior",
	"period.superior",
	"a.superior",
	"b.superior",
	"d.superior",
	"e.superior",
	"i.superior",
	"l.superior",
	"m.superior",
	"o.superior",
	"r.superior",
	"s.superior",
	"t.superior",
	"breve.sc",
	"caron.sc",
	"circumflex.sc",
	"dotaccent.sc",
	"hungarumlaut.sc",
	"lslash.sc",
	"oe.sc",
	"ogonek.sc",
	"ring.sc",
	"scaron.sc",
	"tilde.sc",
	"zcaron.sc"
};

static const char *adobepua_p0_bf7[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"exclam.sc",
	NULL,
	NULL,
	"dollar.oldstyle",
	NULL,
	"ampersand.sc",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"zero.oldstyle",
	"one.oldstyle",
	"two.oldstyle",
	"three.oldstyle",
	"four.oldstyle",
	"five.oldstyle",
	"six.oldstyle",
	"seven.oldstyle",
	"eight.oldstyle",
	"nine.oldstyle",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"question.sc",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"grave.sc",
	"a.sc",
	"b.sc",
	"c.sc",
	"d.sc",
	"e.sc",
	"f.sc",
	"g.sc",
	"h.sc",
	"i.sc",
	"j.sc",
	"k.sc",
	"l.sc",
	"m.sc",
	"n.sc",
	"o.sc",
	"p.sc",
	"q.sc",
	"r.sc",
	"s.sc",
	"t.sc",
	"u.sc",
	"v.sc",
	"w.sc",
	"x.sc",
	"y.sc",
	"z.sc",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"exclamdown.sc",
	"cent.oldstyle",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"dieresis.sc",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"macron.sc",
	NULL,
	NULL,
	NULL,
	NULL,
	"acute.sc",
	NULL,
	NULL,
	NULL,
	"cedilla.sc",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"questiondown.sc",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"agrave.sc",
	"aacute.sc",
	"acircumflex.sc",
	"atilde.sc",
	"adieresis.sc",
	"aring.sc",
	"ae.sc",
	"ccedilla.sc",
	"egrave.sc",
	"eacute.sc",
	"ecircumflex.sc",
	"edieresis.sc",
	"igrave.sc",
	"iacute.sc",
	"icircumflex.sc",
	"idieresis.sc",
	"eth.sc",
	"ntilde.sc",
	"ograve.sc",
	"oacute.sc",
	"ocircumflex.sc",
	"otilde.sc",
	"odieresis.sc",
	NULL,
	"oslash.sc",
	"ugrave.sc",
	"uacute.sc",
	"ucircumflex.sc",
	"udieresis.sc",
	"yacute.sc",
	"thorn.sc",
	"ydieresis.sc"
};

static const char *adobepua_p0_bf8[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"registered.sans",
	"copyright.sans",
	"trademark.sans",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *adobepua_p0_bfb[] = {
	"f_f",
	"f_i",
	"f_l",
	"f_f_i",
	"f_f_l",
	"longs_t",
	"s_t",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char **adobepua_p0[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	adobepua_p0_bf6,
	adobepua_p0_bf7,
	adobepua_p0_bf8,
	NULL,
	NULL,
	adobepua_p0_bfb,
	NULL,
	NULL,
	NULL,
	NULL
};

static NameList adobepua = {
	&agl,
	N_("AGL with PUA"),
	{ adobepua_p0, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL }
};
/* ************************************************************************** */
static const char *greeksc_p0_bf5[] = {
	"alpha.sc",
	"beta.sc",
	"gamma.sc",
	"delta.sc",
	"epsilon.sc",
	"zeta.sc",
	"eta.sc",
	"theta.sc",
	"iota.sc",
	"kappa.sc",
	"lambda.sc",
	"mu.sc",
	"nu.sc",
	"xi.sc",
	"omicron.sc",
	"pi.sc",
	"rho.sc",
	NULL,
	"sigma.sc",
	"tau.sc",
	"upsilon.sc",
	"phi.sc",
	"chi.sc",
	"psi.sc",
	"omega.sc",
	"iotadieresis.sc",
	"upsilondieresis.sc",
	"alphaiota.sc",
	"etaiota.sc",
	"omegaiota.sc",
	"prosgegrammeni.sc",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char **greeksc_p0[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	greeksc_p0_bf5,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static NameList greeksc = {
	&adobepua,
	N_("Greek small caps"),
	{ greeksc_p0, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL }
};
/* ************************************************************************** */
static const char *tex_p0_b20[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"closure",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"leftharpoonaccent",
	"rightharpoonaccent",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *tex_p0_b21[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"shortleftarrow",
	"shortuparrow",
	"shortrightarrow",
	"shortdownarrow",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"leftwavyarrow",
	"rightwavyarrow",
	NULL,
	"twoheaduparrow",
	NULL,
	"twoheaddownarrow",
	NULL,
	NULL,
	NULL,
	"mapsup",
	NULL,
	"mapsdown",
	"updownarrowbar",
	NULL,
	NULL,
	NULL,
	NULL,
	"leftrightwavyarrow",
	NULL,
	"downzigzagarrow",
	NULL,
	NULL,
	"Ldsh",
	"Rdsh",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"acwopencirclearrow",
	"cwopencirclearrow",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"updownarrows",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"Nwarrow",
	"Nearrow",
	"Searrow",
	"Swarrow",
	NULL,
	NULL,
	"longleftzigzagarrow",
	"rightzigzagarrow",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"barleftarrow",
	"rightarrowbar",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"downuparrows",
	"rightthreearrows",
	"nvleftarrow",
	"nvrightarrow",
	"nvleftrightarrow",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *tex_p0_b22[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"ltrpar",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"difference",
	NULL,
	"ac",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"approxident",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"arceq",
	NULL,
	"veeeq",
	"stareq",
	NULL,
	"eqdef",
	"measeq",
	NULL,
	NULL,
	NULL,
	NULL,
	"Equiv",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"prurel",
	"scurel",
	NULL,
	NULL,
	NULL,
	NULL,
	"origof",
	"imageof",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"eqless",
	"eqgtr",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"disin",
	NULL,
	"isins",
	"isindot",
	NULL,
	NULL,
	"isinvb",
	"isinE",
	"nisd",
	NULL,
	"nis",
	NULL,
	NULL,
	NULL
};

static const char *tex_p0_b23[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"invnot",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"turnednot",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"solbar",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"rangledownzigzagarrow",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *tex_p0_b27[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"longrightzigzagarrow"
};

static const char *tex_p0_b29[] = {
	NULL,
	NULL,
	"nvLeftarrow",
	"nvRightarrow",
	"nvLeftrightarrow",
	"twoheadmapsto",
	NULL,
	NULL,
	"downarrowbarred",
	"uparrowbarred",
	"Uuparrow",
	"Ddownarrow",
	"leftbkarrow",
	"rightbkarrow",
	"leftdbkarrow",
	NULL,
	NULL,
	"rightdotarrow",
	"baruparrow",
	"downarrowbar",
	NULL,
	NULL,
	"twoheadrightarrowtail",
	NULL,
	NULL,
	"lefttail",
	"righttail",
	"leftdbltail",
	"rightdbltail",
	"diamondleftarrow",
	"rightarrowdiamond",
	"diamondleftarrowbar",
	"barrightarrowdiamond",
	"nwsearrow",
	"neswarrow",
	"hknwarrow",
	"hknearrow",
	"hksearrow",
	"hkswarrow",
	NULL,
	NULL,
	NULL,
	NULL,
	"rdiagovfdiag",
	"fdiagovrdiag",
	"seovnearrow",
	"neovsearrow",
	"fdiagovnearrow",
	"rdiagovsearrow",
	"neovnwarrow",
	"nwovnearrow",
	"rightcurvedarrow",
	NULL,
	NULL,
	"leftdowncurvedarrow",
	"rightdowncurvedarrow",
	"cwhalfcirclearrow",
	"acwhalfcirclearrow",
	NULL,
	"acwundercurvearrow",
	"curvearrowrightminus",
	"curvearrowleftplus",
	"cwundercurvearrow",
	NULL,
	NULL,
	NULL,
	"rightarrowshortleftarrow",
	"leftarrowshortrightarrow",
	"shortrightarrowleftarrow",
	"rightarrowplus",
	"leftarrowplus",
	"rightarrowx",
	"leftrightarrowcircle",
	"twoheaduparrowcircle",
	"leftrightharpoonupdown",
	"leftrightharpoondownup",
	"updownharpoonrightleft",
	"updownharpoonleftright",
	"leftrightharpoonupup",
	"updownharpoonrightright",
	"leftrightharpoondowndown",
	"updownharpoonleftleft",
	"barleftharpoonup",
	"rightharpoonupbar",
	"barupharpoonright",
	"downharpoonrightbar",
	"barleftharpoondown",
	"rightharpoondownbar",
	"barupharpoonleft",
	"downharpoonleftbar",
	"leftharpoonupbar",
	"barrightharpoonup",
	"upharpoonrightbar",
	"bardownharpoonright",
	"leftharpoondownbar",
	"barrightharpoondown",
	"upharpoonleftbar",
	"bardownharpoonleft",
	"leftharpoonsupdown",
	"upharpoonsleftright",
	"rightharpoonsupdown",
	"downharpoonsleftright",
	"leftrightharpoonsup",
	"leftrightharpoonsdown",
	"rightleftharpoonsup",
	"rightleftharpoonsdown",
	"leftharpoonupdash",
	"dashleftharpoondown",
	"rightharpoonupdash",
	"dashrightharpoondown",
	"updownharpoonsleftright",
	"downupharpoonsleftright",
	"rightimply",
	"equalrightarrow",
	"similarrightarrow",
	"leftarrowsimilar",
	"rightarrowsimilar",
	"rightarrowapprox",
	"ltlarr",
	"leftarrowless",
	"gtrarr",
	"subrarr",
	"leftarrowsubset",
	"suplarr",
	"leftfishtail",
	"rightfishtail",
	"upfishtail",
	"downfishtail",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"rtriltri",
	"ltrivb",
	"vbrtri",
	"lfbowtie",
	"rfbowtie",
	"fbowtie",
	"lftimes",
	"rftimes",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"eqvparsl",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"ruledelayed",
	NULL,
	"dsol",
	"rsolbar",
	"xsol",
	"xbsol",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *tex_p0_b2a[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"congdot",
	NULL,
	NULL,
	"approxeqq",
	NULL,
	NULL,
	"eqqsim",
	"Coloneq",
	"eqeq",
	NULL,
	NULL,
	"equivDD",
	"ltcir",
	"gtcir",
	"ltquest",
	"gtquest",
	NULL,
	NULL,
	"lesdot",
	"gesdot",
	"lesdoto",
	"gesdoto",
	"lesdotor",
	"gesdotol",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"lsime",
	"gsime",
	"lsimg",
	"gsiml",
	"lgE",
	"glE",
	"lesges",
	"gesles",
	NULL,
	NULL,
	"elsdot",
	"egsdot",
	NULL,
	NULL,
	NULL,
	NULL,
	"simless",
	"simgtr",
	"simlE",
	"simgE",
	"Lt",
	"Gt",
	NULL,
	"glj",
	"gla",
	"ltcc",
	"gtcc",
	"lescc",
	"gescc",
	"smt",
	"lat",
	"smte",
	"late",
	"bumpeqq",
	NULL,
	NULL,
	NULL,
	NULL,
	"prE",
	"scE",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"Pr",
	"Sc",
	"subsetdot",
	"supsetdot",
	"subsetplus",
	"supsetplus",
	"submult",
	"supmult",
	"subedot",
	"supedot",
	NULL,
	NULL,
	"subsim",
	"supsim",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"csub",
	"csup",
	"csube",
	"csupe",
	"subsup",
	"supsub",
	"subsub",
	"supsup",
	"suphsub",
	"supdsub",
	"forkv",
	"topfork",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"Barv",
	"vBar",
	"vBarv",
	"barV",
	"Vbar",
	"Not",
	"bNot",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *tex_p0_be2[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"rightdbkarrow",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"nrightwavyarrow",
	NULL,
	"nrightcurvedarrow",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"race",
	"acE",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *tex_p0_be3[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"lesg",
	"gesl",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"smtes",
	"lates",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"bsolhsub",
	"suphsol",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"congruence",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *tex_p0_be4[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"mostpos",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *tex_p0_be6[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"varrightfishtail",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char **tex_p0[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	tex_p0_b20,
	tex_p0_b21,
	tex_p0_b22,
	tex_p0_b23,
	NULL,
	NULL,
	NULL,
	tex_p0_b27,
	NULL,
	tex_p0_b29,
	tex_p0_b2a,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	tex_p0_be2,
	tex_p0_be3,
	tex_p0_be4,
	NULL,
	tex_p0_be6,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static NameList tex = {
	&agl,
	NU_("ΤεΧ Names"),
	{ tex_p0, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL }
};
/* ************************************************************************** */
static const char *ams_p0_be2[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni219D0338",
	NULL,
	"uni29330338",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2192.short",
	"uni2190.short",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni222AFE00",
	"uni2229FE00",
	"uni2294FE00",
	"uni2293FE00",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni223E0332.reversed",
	"uni223E0333",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2269FE00",
	NULL,
	NULL,
	"uni2268FE00",
	"uni22670338",
	"uni2A7E0338",
	"uni2A7D0338",
	"uni22660338",
	NULL,
	"uni2224.short",
	"uni2226.short",
	"uni228220D2",
	"uni228620D2",
	"uni2AC50338",
	"uni228320D2",
	"uni2AC60338",
	"uni228720D2",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2ACBFE00",
	"uni228AFE00",
	"uni228BFE00",
	"uni2ACCFE00",
	"uni224B0338",
	NULL,
	NULL,
	NULL,
	NULL,
	"uni226420D2",
	"uni226520D2",
	"uni003C20D2",
	"uni003E20D2",
	"uni2A6D0338",
	"uni224820D2",
	"uni2A700338",
	NULL,
	"uni2AA120D2",
	"uni2AA220D2",
	"uni226A0338",
	"uni226B0338",
	"uni22D80338",
	"uni22D90338",
	"uni22B520D2",
	"uni22B420D2",
	NULL,
	NULL,
	NULL,
	"uni006A.dotless",
	"uni210F.var",
	NULL,
	NULL,
	"uni222020D2",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *ams_p0_be3[] = {
	NULL,
	"uni2223.short",
	"uni2225.short",
	NULL,
	NULL,
	NULL,
	"uni2248.bold",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni22DAFE00",
	"uni22DBFE00",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2AACFE00",
	"uni2AADFE00",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni220B20D2",
	"uni220D0338",
	NULL,
	NULL,
	"uni220820D2",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2A15.up",
	"uni2A16.up",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni226120E5",
	"uni003D20E5",
	"uni2AFD20E5",
	"uni22500338",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni22020338",
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2A10.up",
	NULL,
	"uni2A12.up",
	"uni2A13.up",
	"uni2A14.up",
	"uni2A17.up",
	"uni2A11.up",
	NULL,
	"uni22F50338",
	NULL,
	"uni22F90338",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2026.em",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni220A0338",
	"uni227320D2",
	"uni227220D2",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni223E.reversed",
	NULL,
	NULL,
	NULL,
	NULL,
	"uni0265.superscript",
	"uni0252.superscript",
	NULL,
	NULL,
	NULL,
	"uni2A0F.up",
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2A9B20D2",
	"uni2A9C20D2",
	"uni2A9B0338",
	"uni2A9C0338",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *ams_p0_be4[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni223E.var",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni223C20D2",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *ams_p0_be5[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2A3CFE00",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2AAF0338",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni224220D2",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2AB00338",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *ams_p0_be6[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	"uni228F0338",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni22900338",
	"uni224E0338",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2205.var",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni0077.subscript",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni1D0B.reversed",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni025102DE",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni025B02DE",
	"uni025402DE",
	NULL,
	"uni03C9.inverted",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni297D.var",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni223C.bold",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *ams_p0_be8[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni227E0338",
	"uni227F0338",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2216.var",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni224F0338",
	"uni22420338",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *ams_p0_bea[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni00730336",
	NULL,
	"uni006C0321",
	NULL,
	NULL,
	"uni03040304",
	NULL,
	NULL,
	"uni229CFE00",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni03020302",
	NULL,
	"uni03030303",
	"uni033103310331",
	"uni0331033103310331",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni03070302",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni03040303",
	"uni03310330",
	"uni03040308",
	"uni03300331",
	NULL,
	"uni00640321",
	"uni00680321",
	"uni006B0321",
	"uni00780321",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni00730321",
	"uni007A0321",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2034.notsup",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2A0B.up",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni227620D2",
	"uni227720D2",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *ams_p0_bec[] = {
	NULL,
	NULL,
	"uni2A3DFE00",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *ams_p0_bed[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni220A20D2",
	"uni220D20D2",
	"uni22F60338",
	"uni22FD0338",
	"uni22F620D2",
	"uni22FD20D2",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2295FE00",
	"uni2297FE00",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni224320D2",
	"uni224520D2",
	NULL,
	"uni2A6C0338",
	"uni2A6C20D2",
	"uni2A7020D2",
	"uni224D20D2",
	"uni223F.reversed",
	"uni003D20D2",
	NULL,
	"uni226120D2",
	"uni22630338",
	"uni226320D2",
	NULL,
	NULL,
	"uni2A7D20D2",
	"uni2A7E20D2",
	"uni226620D2",
	"uni226720D2",
	"uni2A950338",
	"uni2A960338",
	"uni2A9520D2",
	"uni2A9620D2",
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2A990338",
	"uni2A9A0338",
	"uni2A9920D2",
	"uni2A9A20D2",
	"uni2272FE00",
	"uni2273FE00",
	"uni2A9DFE00",
	"uni2A9EFE00",
	NULL,
	NULL,
	"uni227A20D2",
	"uni227B20D2",
	"uni227C20D2",
	"uni227D20D2",
	"uni22DE20D2",
	"uni22DF20D2",
	"uni22DE0338",
	"uni22DF0338",
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2AC520D2",
	"uni2AC620D2",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2AF40338",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2A0A.lgdisplay",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni220A.narrow",
	"uni220D.narrow",
	NULL,
	"u1D6FB.narrow",
	"uni002820090029",
	"uni002822C50029",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *ams_p0_bee[] = {
	"stixEE00",
	"stixEE01",
	"stixEE02",
	"stixEE03",
	"stixEE04",
	"stixEE05",
	"stixEE06",
	"stixEE07",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"stixEE10",
	"stixEE11",
	"stixEE12",
	"stixEE13",
	"stixEE14",
	"stixEE15",
	"stixEE16",
	"stixEE17",
	"stixEE18",
	"stixEE19",
	"stixEE1A",
	"stixEE1B",
	"stixEE1C",
	"stixEE1D",
	"stixEE1E",
	"stixEE1F",
	"stixEE20",
	"stixEE21",
	"stixEE22",
	"stixEE23",
	"stixEE24",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"stixEE30",
	"stixEE31",
	"stixEE32",
	"stixEE33",
	"stixEE34",
	"stixEE35",
	"stixEE36",
	"stixEE37",
	"stixEE38",
	"stixEE39",
	"stixEE3A",
	"stixEE3B",
	"stixEE3C",
	"stixEE3D",
	"stixEE3E",
	NULL,
	"stixEE40",
	"stixEE41",
	"stixEE42",
	"stixEE43",
	"stixEE44",
	"stixEE45",
	"stixEE46",
	"stixEE47",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"stixEE50",
	"stixEE51",
	"stixEE52",
	"stixEE53",
	"stixEE54",
	"stixEE55",
	"stixEE56",
	"stixEE57",
	"stixEE58",
	"stixEE59",
	"stixEE5A",
	"stixEE5B",
	"stixEE5C",
	"stixEE5D",
	"stixEE5E",
	"stixEE5F",
	"stixEE60",
	"stixEE61",
	"stixEE62",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"stixEE70",
	"stixEE71",
	"stixEE72",
	"stixEE73",
	"stixEE74",
	"stixEE75",
	"stixEE76",
	"stixEE77",
	"stixEE78",
	"stixEE79",
	"stixEE7A",
	"stixEE7B",
	"stixEE7C",
	"stixEE7D",
	"stixEE7E",
	"stixEE7F",
	"stixEE80",
	"stixEE81",
	"stixEE82",
	"stixEE83",
	"stixEE84",
	"stixEE85",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"stixEE90",
	"stixEE91",
	"stixEE92",
	"stixEE93",
	"stixEE94",
	"stixEE95",
	"stixEE96",
	"stixEE97",
	"stixEE98",
	"stixEE99",
	"stixEE9A",
	"stixEE9B",
	"stixEE9C",
	"stixEE9D",
	"stixEE9E",
	"stixEE9F",
	"stixEEA0",
	"stixEEA1",
	"stixEEA2",
	"stixEEA3",
	"stixEEA4",
	"stixEEA5",
	"stixEEA6",
	"stixEEA7",
	"stixEEA8",
	"stixEEA9",
	"stixEEAA",
	"stixEEAB",
	"stixEEAC",
	NULL,
	NULL,
	NULL,
	"stixEEB0",
	"stixEEB1",
	"stixEEB2",
	"stixEEB3",
	"stixEEB4",
	"stixEEB5",
	"stixEEB6",
	"stixEEB7",
	"stixEEB8",
	"stixEEB9",
	"stixEEBA",
	"stixEEBB",
	"stixEEBC",
	"stixEEBD",
	"stixEEBE",
	"stixEEBF",
	"stixEEC0",
	"stixEEC1",
	"stixEEC2",
	"stixEEC3",
	"stixEEC4",
	"stixEEC5",
	"stixEEC6",
	"stixEEC7",
	"stixEEC8",
	"stixEEC9",
	"stixEECA",
	"stixEECB",
	NULL,
	"stixEECD",
	"stixEECE",
	"stixEECF",
	"stixEED0",
	"stixEED1",
	"stixEED2",
	"stixEED3",
	"stixEED4",
	"stixEED5",
	"stixEED6",
	"stixEED7",
	"stixEED8",
	"stixEED9",
	"stixEEDA",
	"stixEEDB",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *ams_p0_bf4[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni29CF0338",
	"uni29D00338",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2AA10338",
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2AA20338",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char *ams_p0_bf5[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	"uni2191.short",
	"uni2193.short",
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static const char **ams_p0[] = {
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	ams_p0_be2,
	ams_p0_be3,
	ams_p0_be4,
	ams_p0_be5,
	ams_p0_be6,
	NULL,
	ams_p0_be8,
	NULL,
	ams_p0_bea,
	NULL,
	ams_p0_bec,
	ams_p0_bed,
	ams_p0_bee,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	ams_p0_bf4,
	ams_p0_bf5,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL
};

static NameList ams = {
	&tex,
	N_("AMS Names"),
	{ ams_p0, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL }
};
/* ************************************************************************** */
static struct psaltnames psaltnames[] = {
	{ "AEmacron", 0x01e2 },
	{ "AEsmall", 0xf7e6 },
	{ "Aacutesmall", 0xf7e1 },
	{ "Abreveacute", 0x1eae },
	{ "Abrevecyrillic", 0x04d0 },
	{ "Abrevedotbelow", 0x1eb6 },
	{ "Abrevegrave", 0x1eb0 },
	{ "Abrevehookabove", 0x1eb2 },
	{ "Abrevetilde", 0x1eb4 },
	{ "Acaron", 0x01cd },
	{ "Acircle", 0x24b6 },
	{ "Acircumflexacute", 0x1ea4 },
	{ "Acircumflexdotbelow", 0x1eac },
	{ "Acircumflexgrave", 0x1ea6 },
	{ "Acircumflexhookabove", 0x1ea8 },
	{ "Acircumflexsmall", 0xf7e2 },
	{ "Acircumflextilde", 0x1eaa },
	{ "Acute", 0xf6c9 },
	{ "Acutesmall", 0xf7b4 },
	{ "Acyrillic", 0x0410 },
	{ "Adblgrave", 0x0200 },
	{ "Adieresiscyrillic", 0x04d2 },
	{ "Adieresismacron", 0x01de },
	{ "Adieresissmall", 0xf7e4 },
	{ "Adotbelow", 0x1ea0 },
	{ "Adotmacron", 0x01e0 },
	{ "Agravesmall", 0xf7e0 },
	{ "Ahookabove", 0x1ea2 },
	{ "Aiecyrillic", 0x04d4 },
	{ "Ainvertedbreve", 0x0202 },
	{ "Amonospace", 0xff21 },
	{ "Aringbelow", 0x1e00 },
	{ "Aringsmall", 0xf7e5 },
	{ "Asmall", 0xf761 },
	{ "Atildesmall", 0xf7e3 },
	{ "Aybarmenian", 0x0531 },
	{ "Bcircle", 0x24b7 },
	{ "Bdotaccent", 0x1e02 },
	{ "Bdotbelow", 0x1e04 },
	{ "Becyrillic", 0x0411 },
	{ "Benarmenian", 0x0532 },
	{ "Bhook", 0x0181 },
	{ "Blinebelow", 0x1e06 },
	{ "Bmonospace", 0xff22 },
	{ "Brevesmall", 0xf6f4 },
	{ "Bsmall", 0xf762 },
	{ "Btopbar", 0x0182 },
	{ "Caarmenian", 0x053e },
	{ "Caron", 0xf6ca },
	{ "Caronsmall", 0xf6f5 },
	{ "Ccedillaacute", 0x1e08 },
	{ "Ccedillasmall", 0xf7e7 },
	{ "Ccircle", 0x24b8 },
	{ "Cdot", 0x010a },
	{ "Cedillasmall", 0xf7b8 },
	{ "Chaarmenian", 0x0549 },
	{ "Cheabkhasiancyrillic", 0x04bc },
	{ "Checyrillic", 0x0427 },
	{ "Chedescenderabkhasiancyrillic", 0x04be },
	{ "Chedescendercyrillic", 0x04b6 },
	{ "Chedieresiscyrillic", 0x04f4 },
	{ "Cheharmenian", 0x0543 },
	{ "Chekhakassiancyrillic", 0x04cb },
	{ "Cheverticalstrokecyrillic", 0x04b8 },
	{ "Chook", 0x0187 },
	{ "Circumflexsmall", 0xf6f6 },
	{ "Cmonospace", 0xff23 },
	{ "Coarmenian", 0x0551 },
	{ "Csmall", 0xf763 },
	{ "DZ", 0x01f1 },
	{ "DZcaron", 0x01c4 },
	{ "Daarmenian", 0x0534 },
	{ "Dafrican", 0x0189 },
	{ "Dcedilla", 0x1e10 },
	{ "Dcircle", 0x24b9 },
	{ "Dcircumflexbelow", 0x1e12 },
	{ "Ddotaccent", 0x1e0a },
	{ "Ddotbelow", 0x1e0c },
	{ "Decyrillic", 0x0414 },
	{ "Deicoptic", 0x03ee },
	{ "Deltagreek", 0x0394 },
	{ "Dhook", 0x018a },
	{ "Dieresis", 0xf6cb },
	{ "DieresisAcute", 0xf6cc },
	{ "DieresisGrave", 0xf6cd },
	{ "Dieresissmall", 0xf7a8 },
	{ "Digammagreek", 0x03dc },
	{ "Djecyrillic", 0x0402 },
	{ "Dlinebelow", 0x1e0e },
	{ "Dmonospace", 0xff24 },
	{ "Dotaccentsmall", 0xf6f7 },
	{ "Dslash", 0x0110 },
	{ "Dsmall", 0xf764 },
	{ "Dtopbar", 0x018b },
	{ "Dz", 0x01f2 },
	{ "Dzcaron", 0x01c5 },
	{ "Dzeabkhasiancyrillic", 0x04e0 },
	{ "Dzecyrillic", 0x0405 },
	{ "Dzhecyrillic", 0x040f },
	{ "Eacutesmall", 0xf7e9 },
	{ "Ecedillabreve", 0x1e1c },
	{ "Echarmenian", 0x0535 },
	{ "Ecircle", 0x24ba },
	{ "Ecircumflexacute", 0x1ebe },
	{ "Ecircumflexbelow", 0x1e18 },
	{ "Ecircumflexdotbelow", 0x1ec6 },
	{ "Ecircumflexgrave", 0x1ec0 },
	{ "Ecircumflexhookabove", 0x1ec2 },
	{ "Ecircumflexsmall", 0xf7ea },
	{ "Ecircumflextilde", 0x1ec4 },
	{ "Ecyrillic", 0x0404 },
	{ "Edblgrave", 0x0204 },
	{ "Edieresissmall", 0xf7eb },
	{ "Edot", 0x0116 },
	{ "Edotbelow", 0x1eb8 },
	{ "Efcyrillic", 0x0424 },
	{ "Egravesmall", 0xf7e8 },
	{ "Eharmenian", 0x0537 },
	{ "Ehookabove", 0x1eba },
	{ "Eightroman", 0x2167 },
	{ "Einvertedbreve", 0x0206 },
	{ "Eiotifiedcyrillic", 0x0464 },
	{ "Elcyrillic", 0x041b },
	{ "Elevenroman", 0x216a },
	{ "Emacronacute", 0x1e16 },
	{ "Emacrongrave", 0x1e14 },
	{ "Emcyrillic", 0x041c },
	{ "Emonospace", 0xff25 },
	{ "Encyrillic", 0x041d },
	{ "Endescendercyrillic", 0x04a2 },
	{ "Enghecyrillic", 0x04a4 },
	{ "Enhookcyrillic", 0x04c7 },
	{ "Eopen", 0x0190 },
	{ "Ercyrillic", 0x0420 },
	{ "Ereversed", 0x018e },
	{ "Ereversedcyrillic", 0x042d },
	{ "Escyrillic", 0x0421 },
	{ "Esdescendercyrillic", 0x04aa },
	{ "Esh", 0x01a9 },
	{ "Esmall", 0xf765 },
	{ "Etarmenian", 0x0538 },
	{ "Ethsmall", 0xf7f0 },
	{ "Etilde", 0x1ebc },
	{ "Etildebelow", 0x1e1a },
	{ "Ezh", 0x01b7 },
	{ "Ezhcaron", 0x01ee },
	{ "Ezhreversed", 0x01b8 },
	{ "Fcircle", 0x24bb },
	{ "Fdotaccent", 0x1e1e },
	{ "Feharmenian", 0x0556 },
	{ "Feicoptic", 0x03e4 },
	{ "Fhook", 0x0191 },
	{ "Fitacyrillic", 0x0472 },
	{ "Fiveroman", 0x2164 },
	{ "Fmonospace", 0xff26 },
	{ "Fourroman", 0x2163 },
	{ "Fsmall", 0xf766 },
	{ "GBsquare", 0x3387 },
	{ "Gacute", 0x01f4 },
	{ "Gammaafrican", 0x0194 },
	{ "Gangiacoptic", 0x03ea },
	{ "Gcedilla", 0x0122 },
	{ "Gcircle", 0x24bc },
	{ "Gdot", 0x0120 },
	{ "Gecyrillic", 0x0413 },
	{ "Ghadarmenian", 0x0542 },
	{ "Ghemiddlehookcyrillic", 0x0494 },
	{ "Ghestrokecyrillic", 0x0492 },
	{ "Gheupturncyrillic", 0x0490 },
	{ "Ghook", 0x0193 },
	{ "Gimarmenian", 0x0533 },
	{ "Gjecyrillic", 0x0403 },
	{ "Gmacron", 0x1e20 },
	{ "Gmonospace", 0xff27 },
	{ "Grave", 0xf6ce },
	{ "Gravesmall", 0xf760 },
	{ "Gsmall", 0xf767 },
	{ "Gsmallhook", 0x029b },
	{ "Gstroke", 0x01e4 },
	{ "HPsquare", 0x33cb },
	{ "Haabkhasiancyrillic", 0x04a8 },
	{ "Hadescendercyrillic", 0x04b2 },
	{ "Hardsigncyrillic", 0x042a },
	{ "Hbrevebelow", 0x1e2a },
	{ "Hcedilla", 0x1e28 },
	{ "Hcircle", 0x24bd },
	{ "Hdieresis", 0x1e26 },
	{ "Hdotaccent", 0x1e22 },
	{ "Hdotbelow", 0x1e24 },
	{ "Hmonospace", 0xff28 },
	{ "Hoarmenian", 0x0540 },
	{ "Horicoptic", 0x03e8 },
	{ "Hsmall", 0xf768 },
	{ "Hungarumlaut", 0xf6cf },
	{ "Hungarumlautsmall", 0xf6f8 },
	{ "Hzsquare", 0x3390 },
	{ "IAcyrillic", 0x042f },
	{ "IUcyrillic", 0x042e },
	{ "Iacutesmall", 0xf7ed },
	{ "Icaron", 0x01cf },
	{ "Icircle", 0x24be },
	{ "Icircumflexsmall", 0xf7ee },
	{ "Icyrillic", 0x0406 },
	{ "Idblgrave", 0x0208 },
	{ "Idieresisacute", 0x1e2e },
	{ "Idieresiscyrillic", 0x04e4 },
	{ "Idieresissmall", 0xf7ef },
	{ "Idot", 0x0130 },
	{ "Idotbelow", 0x1eca },
	{ "Iebrevecyrillic", 0x04d6 },
	{ "Iecyrillic", 0x0415 },
	{ "Igravesmall", 0xf7ec },
	{ "Ihookabove", 0x1ec8 },
	{ "Iicyrillic", 0x0418 },
	{ "Iinvertedbreve", 0x020a },
	{ "Iishortcyrillic", 0x0419 },
	{ "Imacroncyrillic", 0x04e2 },
	{ "Imonospace", 0xff29 },
	{ "Iniarmenian", 0x053b },
	{ "Iocyrillic", 0x0401 },
	{ "Iotaafrican", 0x0196 },
	{ "Ismall", 0xf769 },
	{ "Istroke", 0x0197 },
	{ "Itildebelow", 0x1e2c },
	{ "Izhitsacyrillic", 0x0474 },
	{ "Izhitsadblgravecyrillic", 0x0476 },
	{ "Jaarmenian", 0x0541 },
	{ "Jcircle", 0x24bf },
	{ "Jecyrillic", 0x0408 },
	{ "Jheharmenian", 0x054b },
	{ "Jmonospace", 0xff2a },
	{ "Jsmall", 0xf76a },
	{ "KBsquare", 0x3385 },
	{ "KKsquare", 0x33cd },
	{ "Kabashkircyrillic", 0x04a0 },
	{ "Kacute", 0x1e30 },
	{ "Kacyrillic", 0x041a },
	{ "Kadescendercyrillic", 0x049a },
	{ "Kahookcyrillic", 0x04c3 },
	{ "Kastrokecyrillic", 0x049e },
	{ "Kaverticalstrokecyrillic", 0x049c },
	{ "Kcaron", 0x01e8 },
	{ "Kcedilla", 0x0136 },
	{ "Kcircle", 0x24c0 },
	{ "Kdotbelow", 0x1e32 },
	{ "Keharmenian", 0x0554 },
	{ "Kenarmenian", 0x053f },
	{ "Khacyrillic", 0x0425 },
	{ "Kheicoptic", 0x03e6 },
	{ "Khook", 0x0198 },
	{ "Kjecyrillic", 0x040c },
	{ "Klinebelow", 0x1e34 },
	{ "Kmonospace", 0xff2b },
	{ "Koppacyrillic", 0x0480 },
	{ "Koppagreek", 0x03de },
	{ "Ksicyrillic", 0x046e },
	{ "Ksmall", 0xf76b },
	{ "LJ", 0x01c7 },
	{ "LL", 0xf6bf },
	{ "Lcedilla", 0x013b },
	{ "Lcircle", 0x24c1 },
	{ "Lcircumflexbelow", 0x1e3c },
	{ "Ldotaccent", 0x013f },
	{ "Ldotbelow", 0x1e36 },
	{ "Ldotbelowmacron", 0x1e38 },
	{ "Liwnarmenian", 0x053c },
	{ "Lj", 0x01c8 },
	{ "Ljecyrillic", 0x0409 },
	{ "Llinebelow", 0x1e3a },
	{ "Lmonospace", 0xff2c },
	{ "Lslashsmall", 0xf6f9 },
	{ "Lsmall", 0xf76c },
	{ "MBsquare", 0x3386 },
	{ "Macron", 0xf6d0 },
	{ "Macronsmall", 0xf7af },
	{ "Macute", 0x1e3e },
	{ "Mcircle", 0x24c2 },
	{ "Mdotaccent", 0x1e40 },
	{ "Mdotbelow", 0x1e42 },
	{ "Menarmenian", 0x0544 },
	{ "Mmonospace", 0xff2d },
	{ "Msmall", 0xf76d },
	{ "Mturned", 0x019c },
	{ "NJ", 0x01ca },
	{ "Ncedilla", 0x0145 },
	{ "Ncircle", 0x24c3 },
	{ "Ncircumflexbelow", 0x1e4a },
	{ "Ndotaccent", 0x1e44 },
	{ "Ndotbelow", 0x1e46 },
	{ "Nhookleft", 0x019d },
	{ "Nineroman", 0x2168 },
	{ "Nj", 0x01cb },
	{ "Njecyrillic", 0x040a },
	{ "Nlinebelow", 0x1e48 },
	{ "Nmonospace", 0xff2e },
	{ "Nowarmenian", 0x0546 },
	{ "Nsmall", 0xf76e },
	{ "Ntildesmall", 0xf7f1 },
	{ "OEsmall", 0xf6fa },
	{ "Oacutesmall", 0xf7f3 },
	{ "Obarredcyrillic", 0x04e8 },
	{ "Obarreddieresiscyrillic", 0x04ea },
	{ "Ocaron", 0x01d1 },
	{ "Ocenteredtilde", 0x019f },
	{ "Ocircle", 0x24c4 },
	{ "Ocircumflexacute", 0x1ed0 },
	{ "Ocircumflexdotbelow", 0x1ed8 },
	{ "Ocircumflexgrave", 0x1ed2 },
	{ "Ocircumflexhookabove", 0x1ed4 },
	{ "Ocircumflexsmall", 0xf7f4 },
	{ "Ocircumflextilde", 0x1ed6 },
	{ "Ocyrillic", 0x041e },
	{ "Odblacute", 0x0150 },
	{ "Odblgrave", 0x020c },
	{ "Odieresiscyrillic", 0x04e6 },
	{ "Odieresissmall", 0xf7f6 },
	{ "Odotbelow", 0x1ecc },
	{ "Ogoneksmall", 0xf6fb },
	{ "Ogravesmall", 0xf7f2 },
	{ "Oharmenian", 0x0555 },
	{ "Ohm", 0x2126 },
	{ "Ohookabove", 0x1ece },
	{ "Ohornacute", 0x1eda },
	{ "Ohorndotbelow", 0x1ee2 },
	{ "Ohorngrave", 0x1edc },
	{ "Ohornhookabove", 0x1ede },
	{ "Ohorntilde", 0x1ee0 },
	{ "Oi", 0x01a2 },
	{ "Oinvertedbreve", 0x020e },
	{ "Omacronacute", 0x1e52 },
	{ "Omacrongrave", 0x1e50 },
	{ "Omegacyrillic", 0x0460 },
	{ "Omegagreek", 0x03a9 },
	{ "Omegaroundcyrillic", 0x047a },
	{ "Omegatitlocyrillic", 0x047c },
	{ "Omonospace", 0xff2f },
	{ "Oneroman", 0x2160 },
	{ "Oogonek", 0x01ea },
	{ "Oogonekmacron", 0x01ec },
	{ "Oopen", 0x0186 },
	{ "Oslashsmall", 0xf7f8 },
	{ "Osmall", 0xf76f },
	{ "Ostrokeacute", 0x01fe },
	{ "Otcyrillic", 0x047e },
	{ "Otildeacute", 0x1e4c },
	{ "Otildedieresis", 0x1e4e },
	{ "Otildesmall", 0xf7f5 },
	{ "Pacute", 0x1e54 },
	{ "Pcircle", 0x24c5 },
	{ "Pdotaccent", 0x1e56 },
	{ "Pecyrillic", 0x041f },
	{ "Peharmenian", 0x054a },
	{ "Pemiddlehookcyrillic", 0x04a6 },
	{ "Phook", 0x01a4 },
	{ "Piwrarmenian", 0x0553 },
	{ "Pmonospace", 0xff30 },
	{ "Psicyrillic", 0x0470 },
	{ "Psmall", 0xf770 },
	{ "Qcircle", 0x24c6 },
	{ "Qmonospace", 0xff31 },
	{ "Qsmall", 0xf771 },
	{ "Raarmenian", 0x054c },
	{ "Rcedilla", 0x0156 },
	{ "Rcircle", 0x24c7 },
	{ "Rdblgrave", 0x0210 },
	{ "Rdotaccent", 0x1e58 },
	{ "Rdotbelow", 0x1e5a },
	{ "Rdotbelowmacron", 0x1e5c },
	{ "Reharmenian", 0x0550 },
	{ "Ringsmall", 0xf6fc },
	{ "Rinvertedbreve", 0x0212 },
	{ "Rlinebelow", 0x1e5e },
	{ "Rmonospace", 0xff32 },
	{ "Rsmall", 0xf772 },
	{ "Rsmallinverted", 0x0281 },
	{ "Rsmallinvertedsuperior", 0x02b6 },
	{ "Sacutedotaccent", 0x1e64 },
	{ "Sampigreek", 0x03e0 },
	{ "Scarondotaccent", 0x1e66 },
	{ "Scaronsmall", 0xf6fd },
	{ "Schwa", 0x018f },
	{ "Schwacyrillic", 0x04d8 },
	{ "Schwadieresiscyrillic", 0x04da },
	{ "Scircle", 0x24c8 },
	{ "Sdotaccent", 0x1e60 },
	{ "Sdotbelow", 0x1e62 },
	{ "Sdotbelowdotaccent", 0x1e68 },
	{ "Seharmenian", 0x054d },
	{ "Sevenroman", 0x2166 },
	{ "Shaarmenian", 0x0547 },
	{ "Shacyrillic", 0x0428 },
	{ "Shchacyrillic", 0x0429 },
	{ "Sheicoptic", 0x03e2 },
	{ "Shhacyrillic", 0x04ba },
	{ "Shimacoptic", 0x03ec },
	{ "Sixroman", 0x2165 },
	{ "Smonospace", 0xff33 },
	{ "Softsigncyrillic", 0x042c },
	{ "Ssmall", 0xf773 },
	{ "Stigmagreek", 0x03da },
	{ "Tcedilla", 0x0162 },
	{ "Tcircle", 0x24c9 },
	{ "Tcircumflexbelow", 0x1e70 },
	{ "Tdotaccent", 0x1e6a },
	{ "Tdotbelow", 0x1e6c },
	{ "Tecyrillic", 0x0422 },
	{ "Tedescendercyrillic", 0x04ac },
	{ "Tenroman", 0x2169 },
	{ "Tetsecyrillic", 0x04b4 },
	{ "Thook", 0x01ac },
	{ "Thornsmall", 0xf7fe },
	{ "Threeroman", 0x2162 },
	{ "Tildesmall", 0xf6fe },
	{ "Tiwnarmenian", 0x054f },
	{ "Tlinebelow", 0x1e6e },
	{ "Tmonospace", 0xff34 },
	{ "Toarmenian", 0x0539 },
	{ "Tonefive", 0x01bc },
	{ "Tonesix", 0x0184 },
	{ "Tonetwo", 0x01a7 },
	{ "Tretroflexhook", 0x01ae },
	{ "Tsecyrillic", 0x0426 },
	{ "Tshecyrillic", 0x040b },
	{ "Tsmall", 0xf774 },
	{ "Twelveroman", 0x216b },
	{ "Tworoman", 0x2161 },
	{ "Uacutesmall", 0xf7fa },
	{ "Ucaron", 0x01d3 },
	{ "Ucircle", 0x24ca },
	{ "Ucircumflexbelow", 0x1e76 },
	{ "Ucircumflexsmall", 0xf7fb },
	{ "Ucyrillic", 0x0423 },
	{ "Udblacute", 0x0170 },
	{ "Udblgrave", 0x0214 },
	{ "Udieresisacute", 0x01d7 },
	{ "Udieresisbelow", 0x1e72 },
	{ "Udieresiscaron", 0x01d9 },
	{ "Udieresiscyrillic", 0x04f0 },
	{ "Udieresisgrave", 0x01db },
	{ "Udieresismacron", 0x01d5 },
	{ "Udieresissmall", 0xf7fc },
	{ "Udotbelow", 0x1ee4 },
	{ "Ugravesmall", 0xf7f9 },
	{ "Uhookabove", 0x1ee6 },
	{ "Uhornacute", 0x1ee8 },
	{ "Uhorndotbelow", 0x1ef0 },
	{ "Uhorngrave", 0x1eea },
	{ "Uhornhookabove", 0x1eec },
	{ "Uhorntilde", 0x1eee },
	{ "Uhungarumlautcyrillic", 0x04f2 },
	{ "Uinvertedbreve", 0x0216 },
	{ "Ukcyrillic", 0x0478 },
	{ "Umacroncyrillic", 0x04ee },
	{ "Umacrondieresis", 0x1e7a },
	{ "Umonospace", 0xff35 },
	{ "Upsilonacutehooksymbolgreek", 0x03d3 },
	{ "Upsilonafrican", 0x01b1 },
	{ "Upsilondieresishooksymbolgreek", 0x03d4 },
	{ "Upsilonhooksymbol", 0x03d2 },
	{ "Ushortcyrillic", 0x040e },
	{ "Usmall", 0xf775 },
	{ "Ustraightcyrillic", 0x04ae },
	{ "Ustraightstrokecyrillic", 0x04b0 },
	{ "Utildeacute", 0x1e78 },
	{ "Utildebelow", 0x1e74 },
	{ "Vcircle", 0x24cb },
	{ "Vdotbelow", 0x1e7e },
	{ "Vecyrillic", 0x0412 },
	{ "Vewarmenian", 0x054e },
	{ "Vhook", 0x01b2 },
	{ "Vmonospace", 0xff36 },
	{ "Voarmenian", 0x0548 },
	{ "Vsmall", 0xf776 },
	{ "Vtilde", 0x1e7c },
	{ "Wcircle", 0x24cc },
	{ "Wdotaccent", 0x1e86 },
	{ "Wdotbelow", 0x1e88 },
	{ "Wmonospace", 0xff37 },
	{ "Wsmall", 0xf777 },
	{ "Xcircle", 0x24cd },
	{ "Xdieresis", 0x1e8c },
	{ "Xdotaccent", 0x1e8a },
	{ "Xeharmenian", 0x053d },
	{ "Xmonospace", 0xff38 },
	{ "Xsmall", 0xf778 },
	{ "Yacutesmall", 0xf7fd },
	{ "Yatcyrillic", 0x0462 },
	{ "Ycircle", 0x24ce },
	{ "Ydieresissmall", 0xf7ff },
	{ "Ydotaccent", 0x1e8e },
	{ "Ydotbelow", 0x1ef4 },
	{ "Yericyrillic", 0x042b },
	{ "Yerudieresiscyrillic", 0x04f8 },
	{ "Yhook", 0x01b3 },
	{ "Yhookabove", 0x1ef6 },
	{ "Yiarmenian", 0x0545 },
	{ "Yicyrillic", 0x0407 },
	{ "Yiwnarmenian", 0x0552 },
	{ "Ymonospace", 0xff39 },
	{ "Ysmall", 0xf779 },
	{ "Ytilde", 0x1ef8 },
	{ "Yusbigcyrillic", 0x046a },
	{ "Yusbigiotifiedcyrillic", 0x046c },
	{ "Yuslittlecyrillic", 0x0466 },
	{ "Yuslittleiotifiedcyrillic", 0x0468 },
	{ "Zaarmenian", 0x0536 },
	{ "Zcaronsmall", 0xf6ff },
	{ "Zcircle", 0x24cf },
	{ "Zcircumflex", 0x1e90 },
	{ "Zdot", 0x017b },
	{ "Zdotbelow", 0x1e92 },
	{ "Zecyrillic", 0x0417 },
	{ "Zedescendercyrillic", 0x0498 },
	{ "Zedieresiscyrillic", 0x04de },
	{ "Zhearmenian", 0x053a },
	{ "Zhebrevecyrillic", 0x04c1 },
	{ "Zhecyrillic", 0x0416 },
	{ "Zhedescendercyrillic", 0x0496 },
	{ "Zhedieresiscyrillic", 0x04dc },
	{ "Zlinebelow", 0x1e94 },
	{ "Zmonospace", 0xff3a },
	{ "Zsmall", 0xf77a },
	{ "Zstroke", 0x01b5 },
	{ "aabengali", 0x0986 },
	{ "aadeva", 0x0906 },
	{ "aagujarati", 0x0a86 },
	{ "aagurmukhi", 0x0a06 },
	{ "aamatragurmukhi", 0x0a3e },
	{ "aarusquare", 0x3303 },
	{ "aavowelsignbengali", 0x09be },
	{ "aavowelsigndeva", 0x093e },
	{ "aavowelsigngujarati", 0x0abe },
	{ "abbreviationmarkarmenian", 0x055f },
	{ "abbreviationsigndeva", 0x0970 },
	{ "abengali", 0x0985 },
	{ "abopomofo", 0x311a },
	{ "abreveacute", 0x1eaf },
	{ "abrevecyrillic", 0x04d1 },
	{ "abrevedotbelow", 0x1eb7 },
	{ "abrevegrave", 0x1eb1 },
	{ "abrevehookabove", 0x1eb3 },
	{ "abrevetilde", 0x1eb5 },
	{ "acaron", 0x01ce },
	{ "acircle", 0x24d0 },
	{ "acircumflexacute", 0x1ea5 },
	{ "acircumflexdotbelow", 0x1ead },
	{ "acircumflexgrave", 0x1ea7 },
	{ "acircumflexhookabove", 0x1ea9 },
	{ "acircumflextilde", 0x1eab },
	{ "acutebelowcmb", 0x0317 },
	{ "acutecmb", 0x0301 },
	{ "acutedeva", 0x0954 },
	{ "acutelowmod", 0x02cf },
	{ "acutetonecmb", 0x0341 },
	{ "acyrillic", 0x0430 },
	{ "adblgrave", 0x0201 },
	{ "addakgurmukhi", 0x0a71 },
	{ "adeva", 0x0905 },
	{ "adieresiscyrillic", 0x04d3 },
	{ "adieresismacron", 0x01df },
	{ "adotbelow", 0x1ea1 },
	{ "adotmacron", 0x01e1 },
	{ "aekorean", 0x3150 },
	{ "aemacron", 0x01e3 },
	{ "afii08941", 0x20a4 },
	{ "afii10063", 0xf6c4 },
	{ "afii10064", 0xf6c5 },
	{ "afii10192", 0xf6c6 },
	{ "afii10831", 0xf6c7 },
	{ "afii10832", 0xf6c8 },
	{ "afii57694", 0xfb2a },
	{ "afii57695", 0xfb2b },
	{ "afii57700", 0xfb4b },
	{ "afii57705", 0xfb1f },
	{ "afii57723", 0xfb35 },
	{ "agujarati", 0x0a85 },
	{ "agurmukhi", 0x0a05 },
	{ "ahiragana", 0x3042 },
	{ "ahookabove", 0x1ea3 },
	{ "aibengali", 0x0990 },
	{ "aibopomofo", 0x311e },
	{ "aideva", 0x0910 },
	{ "aiecyrillic", 0x04d5 },
	{ "aigujarati", 0x0a90 },
	{ "aigurmukhi", 0x0a10 },
	{ "aimatragurmukhi", 0x0a48 },
	{ "ainarabic", 0x0639 },
	{ "ainfinalarabic", 0xfeca },
	{ "aininitialarabic", 0xfecb },
	{ "ainmedialarabic", 0xfecc },
	{ "ainvertedbreve", 0x0203 },
	{ "aivowelsignbengali", 0x09c8 },
	{ "aivowelsigndeva", 0x0948 },
	{ "aivowelsigngujarati", 0x0ac8 },
	{ "akatakana", 0x30a2 },
	{ "akatakanahalfwidth", 0xff71 },
	{ "akorean", 0x314f },
	{ "alef", 0x05d0 },
	{ "alefarabic", 0x0627 },
	{ "alefdageshhebrew", 0xfb30 },
	{ "aleffinalarabic", 0xfe8e },
	{ "alefhamzaabovearabic", 0x0623 },
	{ "alefhamzaabovefinalarabic", 0xfe84 },
	{ "alefhamzabelowarabic", 0x0625 },
	{ "alefhamzabelowfinalarabic", 0xfe88 },
	{ "alefhebrew", 0x05d0 },
	{ "aleflamedhebrew", 0xfb4f },
	{ "alefmaddaabovearabic", 0x0622 },
	{ "alefmaddaabovefinalarabic", 0xfe82 },
	{ "alefmaksuraarabic", 0x0649 },
	{ "alefmaksurafinalarabic", 0xfef0 },
	{ "alefmaksurainitialarabic", 0xfef3 },
	{ "alefmaksuramedialarabic", 0xfef4 },
	{ "alefpatahhebrew", 0xfb2e },
	{ "alefqamatshebrew", 0xfb2f },
	{ "allequal", 0x224c },
	{ "amonospace", 0xff41 },
	{ "ampersandmonospace", 0xff06 },
	{ "ampersandsmall", 0xf726 },
	{ "amsquare", 0x33c2 },
	{ "anbopomofo", 0x3122 },
	{ "angbopomofo", 0x3124 },
	{ "angkhankhuthai", 0x0e5a },
	{ "anglebracketleft", 0x3008 },
	{ "anglebracketleftvertical", 0xfe3f },
	{ "anglebracketright", 0x3009 },
	{ "anglebracketrightvertical", 0xfe40 },
	{ "angstrom", 0x212b },
	{ "anudattadeva", 0x0952 },
	{ "anusvarabengali", 0x0982 },
	{ "anusvaradeva", 0x0902 },
	{ "anusvaragujarati", 0x0a82 },
	{ "apaatosquare", 0x3300 },
	{ "aparen", 0x249c },
	{ "apostrophearmenian", 0x055a },
	{ "apostrophemod", 0x02bc },
	{ "apple", 0xf8ff },
	{ "approaches", 0x2250 },
	{ "approxequalorimage", 0x2252 },
	{ "approximatelyequal", 0x2245 },
	{ "araeaekorean", 0x318e },
	{ "araeakorean", 0x318d },
	{ "arc", 0x2312 },
	{ "arighthalfring", 0x1e9a },
	{ "aringbelow", 0x1e01 },
	{ "arrowdashdown", 0x21e3 },
	{ "arrowdashleft", 0x21e0 },
	{ "arrowdashright", 0x21e2 },
	{ "arrowdashup", 0x21e1 },
	{ "arrowdownleft", 0x2199 },
	{ "arrowdownright", 0x2198 },
	{ "arrowdownwhite", 0x21e9 },
	{ "arrowheaddownmod", 0x02c5 },
	{ "arrowheadleftmod", 0x02c2 },
	{ "arrowheadrightmod", 0x02c3 },
	{ "arrowheadupmod", 0x02c4 },
	{ "arrowhorizex", 0xf8e7 },
	{ "arrowleftdbl", 0x21d0 },
	{ "arrowleftdblstroke", 0x21cd },
	{ "arrowleftoverright", 0x21c6 },
	{ "arrowleftwhite", 0x21e6 },
	{ "arrowrightdblstroke", 0x21cf },
	{ "arrowrightheavy", 0x279e },
	{ "arrowrightoverleft", 0x21c4 },
	{ "arrowrightwhite", 0x21e8 },
	{ "arrowtableft", 0x21e4 },
	{ "arrowtabright", 0x21e5 },
	{ "arrowupdownbase", 0x21a8 },
	{ "arrowupleft", 0x2196 },
	{ "arrowupleftofdown", 0x21c5 },
	{ "arrowupright", 0x2197 },
	{ "arrowupwhite", 0x21e7 },
	{ "arrowvertex", 0xf8e6 },
	{ "asciicircummonospace", 0xff3e },
	{ "asciitildemonospace", 0xff5e },
	{ "ascript", 0x0251 },
	{ "ascriptturned", 0x0252 },
	{ "asmallhiragana", 0x3041 },
	{ "asmallkatakana", 0x30a1 },
	{ "asmallkatakanahalfwidth", 0xff67 },
	{ "asteriskaltonearabic", 0x066d },
	{ "asteriskarabic", 0x066d },
	{ "asteriskmonospace", 0xff0a },
	{ "asterisksmall", 0xfe61 },
	{ "asterism", 0x2042 },
	{ "asuperior", 0xf6e9 },
	{ "asymptoticallyequal", 0x2243 },
	{ "atmonospace", 0xff20 },
	{ "atsmall", 0xfe6b },
	{ "aturned", 0x0250 },
	{ "aubengali", 0x0994 },
	{ "aubopomofo", 0x3120 },
	{ "audeva", 0x0914 },
	{ "augujarati", 0x0a94 },
	{ "augurmukhi", 0x0a14 },
	{ "aulengthmarkbengali", 0x09d7 },
	{ "aumatragurmukhi", 0x0a4c },
	{ "auvowelsignbengali", 0x09cc },
	{ "auvowelsigndeva", 0x094c },
	{ "auvowelsigngujarati", 0x0acc },
	{ "avagrahadeva", 0x093d },
	{ "aybarmenian", 0x0561 },
	{ "ayin", 0x05e2 },
	{ "ayinaltonehebrew", 0xfb20 },
	{ "ayinhebrew", 0x05e2 },
	{ "babengali", 0x09ac },
	{ "backslashmonospace", 0xff3c },
	{ "badeva", 0x092c },
	{ "bagujarati", 0x0aac },
	{ "bagurmukhi", 0x0a2c },
	{ "bahiragana", 0x3070 },
	{ "bahtthai", 0x0e3f },
	{ "bakatakana", 0x30d0 },
	{ "barmonospace", 0xff5c },
	{ "bbopomofo", 0x3105 },
	{ "bcircle", 0x24d1 },
	{ "bdotaccent", 0x1e03 },
	{ "bdotbelow", 0x1e05 },
	{ "beamedsixteenthnotes", 0x266c },
	{ "because", 0x2235 },
	{ "becyrillic", 0x0431 },
	{ "beharabic", 0x0628 },
	{ "behfinalarabic", 0xfe90 },
	{ "behinitialarabic", 0xfe91 },
	{ "behiragana", 0x3079 },
	{ "behmedialarabic", 0xfe92 },
	{ "behmeeminitialarabic", 0xfc9f },
	{ "behmeemisolatedarabic", 0xfc08 },
	{ "behnoonfinalarabic", 0xfc6d },
	{ "bekatakana", 0x30d9 },
	{ "benarmenian", 0x0562 },
	{ "bet", 0x05d1 },
	{ "betasymbolgreek", 0x03d0 },
	{ "betdagesh", 0xfb31 },
	{ "betdageshhebrew", 0xfb31 },
	{ "bethebrew", 0x05d1 },
	{ "betrafehebrew", 0xfb4c },
	{ "bhabengali", 0x09ad },
	{ "bhadeva", 0x092d },
	{ "bhagujarati", 0x0aad },
	{ "bhagurmukhi", 0x0a2d },
	{ "bhook", 0x0253 },
	{ "bihiragana", 0x3073 },
	{ "bikatakana", 0x30d3 },
	{ "bilabialclick", 0x0298 },
	{ "bindigurmukhi", 0x0a02 },
	{ "birusquare", 0x3331 },
	{ "blackcircle", 0x25cf },
	{ "blackdiamond", 0x25c6 },
	{ "blackdownpointingtriangle", 0x25bc },
	{ "blackleftpointingpointer", 0x25c4 },
	{ "blackleftpointingtriangle", 0x25c0 },
	{ "blacklenticularbracketleft", 0x3010 },
	{ "blacklenticularbracketleftvertical", 0xfe3b },
	{ "blacklenticularbracketright", 0x3011 },
	{ "blacklenticularbracketrightvertical", 0xfe3c },
	{ "blacklowerlefttriangle", 0x25e3 },
	{ "blacklowerrighttriangle", 0x25e2 },
	{ "blackrectangle", 0x25ac },
	{ "blackrightpointingpointer", 0x25ba },
	{ "blackrightpointingtriangle", 0x25b6 },
	{ "blacksmallsquare", 0x25aa },
	{ "blacksmilingface", 0x263b },
	{ "blacksquare", 0x25a0 },
	{ "blackstar", 0x2605 },
	{ "blackupperlefttriangle", 0x25e4 },
	{ "blackupperrighttriangle", 0x25e5 },
	{ "blackuppointingsmalltriangle", 0x25b4 },
	{ "blackuppointingtriangle", 0x25b2 },
	{ "blank", 0x2423 },
	{ "blinebelow", 0x1e07 },
	{ "bmonospace", 0xff42 },
	{ "bobaimaithai", 0x0e1a },
	{ "bohiragana", 0x307c },
	{ "bokatakana", 0x30dc },
	{ "bparen", 0x249d },
	{ "bqsquare", 0x33c3 },
	{ "braceex", 0xf8f4 },
	{ "braceleftbt", 0xf8f3 },
	{ "braceleftmid", 0xf8f2 },
	{ "braceleftmonospace", 0xff5b },
	{ "braceleftsmall", 0xfe5b },
	{ "bracelefttp", 0xf8f1 },
	{ "braceleftvertical", 0xfe37 },
	{ "bracerightbt", 0xf8fe },
	{ "bracerightmid", 0xf8fd },
	{ "bracerightmonospace", 0xff5d },
	{ "bracerightsmall", 0xfe5c },
	{ "bracerighttp", 0xf8fc },
	{ "bracerightvertical", 0xfe38 },
	{ "bracketleftbt", 0xf8f0 },
	{ "bracketleftex", 0xf8ef },
	{ "bracketleftmonospace", 0xff3b },
	{ "bracketlefttp", 0xf8ee },
	{ "bracketrightbt", 0xf8fb },
	{ "bracketrightex", 0xf8fa },
	{ "bracketrightmonospace", 0xff3d },
	{ "bracketrighttp", 0xf8f9 },
	{ "brevebelowcmb", 0x032e },
	{ "brevecmb", 0x0306 },
	{ "breveinvertedbelowcmb", 0x032f },
	{ "breveinvertedcmb", 0x0311 },
	{ "breveinverteddoublecmb", 0x0361 },
	{ "bridgebelowcmb", 0x032a },
	{ "bridgeinvertedbelowcmb", 0x033a },
	{ "bstroke", 0x0180 },
	{ "bsuperior", 0xf6ea },
	{ "btopbar", 0x0183 },
	{ "buhiragana", 0x3076 },
	{ "bukatakana", 0x30d6 },
	{ "bulletinverse", 0x25d8 },
	{ "bulletoperator", 0x2219 },
	{ "bullseye", 0x25ce },
	{ "caarmenian", 0x056e },
	{ "cabengali", 0x099a },
	{ "cadeva", 0x091a },
	{ "cagujarati", 0x0a9a },
	{ "cagurmukhi", 0x0a1a },
	{ "calsquare", 0x3388 },
	{ "candrabindubengali", 0x0981 },
	{ "candrabinducmb", 0x0310 },
	{ "candrabindudeva", 0x0901 },
	{ "candrabindugujarati", 0x0a81 },
	{ "capslock", 0x21ea },
	{ "careof", 0x2105 },
	{ "caronbelowcmb", 0x032c },
	{ "caroncmb", 0x030c },
	{ "cbopomofo", 0x3118 },
	{ "ccedillaacute", 0x1e09 },
	{ "ccircle", 0x24d2 },
	{ "ccurl", 0x0255 },
	{ "cdot", 0x010b },
	{ "cdsquare", 0x33c5 },
	{ "cedillacmb", 0x0327 },
	{ "centigrade", 0x2103 },
	{ "centinferior", 0xf6df },
	{ "centmonospace", 0xffe0 },
	{ "centoldstyle", 0xf7a2 },
	{ "centsuperior", 0xf6e0 },
	{ "chaarmenian", 0x0579 },
	{ "chabengali", 0x099b },
	{ "chadeva", 0x091b },
	{ "chagujarati", 0x0a9b },
	{ "chagurmukhi", 0x0a1b },
	{ "chbopomofo", 0x3114 },
	{ "cheabkhasiancyrillic", 0x04bd },
	{ "checkmark", 0x2713 },
	{ "checyrillic", 0x0447 },
	{ "chedescenderabkhasiancyrillic", 0x04bf },
	{ "chedescendercyrillic", 0x04b7 },
	{ "chedieresiscyrillic", 0x04f5 },
	{ "cheharmenian", 0x0573 },
	{ "chekhakassiancyrillic", 0x04cc },
	{ "cheverticalstrokecyrillic", 0x04b9 },
	{ "chieuchacirclekorean", 0x3277 },
	{ "chieuchaparenkorean", 0x3217 },
	{ "chieuchcirclekorean", 0x3269 },
	{ "chieuchkorean", 0x314a },
	{ "chieuchparenkorean", 0x3209 },
	{ "chochangthai", 0x0e0a },
	{ "chochanthai", 0x0e08 },
	{ "chochingthai", 0x0e09 },
	{ "chochoethai", 0x0e0c },
	{ "chook", 0x0188 },
	{ "cieucacirclekorean", 0x3276 },
	{ "cieucaparenkorean", 0x3216 },
	{ "cieuccirclekorean", 0x3268 },
	{ "cieuckorean", 0x3148 },
	{ "cieucparenkorean", 0x3208 },
	{ "cieucuparenkorean", 0x321c },
	{ "circleot", 0x2299 },
	{ "circlepostalmark", 0x3036 },
	{ "circlewithlefthalfblack", 0x25d0 },
	{ "circlewithrighthalfblack", 0x25d1 },
	{ "circumflexbelowcmb", 0x032d },
	{ "circumflexcmb", 0x0302 },
	{ "clear", 0x2327 },
	{ "clickalveolar", 0x01c2 },
	{ "clickdental", 0x01c0 },
	{ "clicklateral", 0x01c1 },
	{ "clickretroflex", 0x01c3 },
	{ "clubsuitblack", 0x2663 },
	{ "clubsuitwhite", 0x2667 },
	{ "cmcubedsquare", 0x33a4 },
	{ "cmonospace", 0xff43 },
	{ "cmsquaredsquare", 0x33a0 },
	{ "coarmenian", 0x0581 },
	{ "colonmonospace", 0xff1a },
	{ "colonsign", 0x20a1 },
	{ "colonsmall", 0xfe55 },
	{ "colontriangularhalfmod", 0x02d1 },
	{ "colontriangularmod", 0x02d0 },
	{ "commaabovecmb", 0x0313 },
	{ "commaaboverightcmb", 0x0315 },
	{ "commaaccent", 0xf6c3 },
	{ "commaarabic", 0x060c },
	{ "commaarmenian", 0x055d },
	{ "commainferior", 0xf6e1 },
	{ "commamonospace", 0xff0c },
	{ "commareversedabovecmb", 0x0314 },
	{ "commareversedmod", 0x02bd },
	{ "commasmall", 0xfe50 },
	{ "commasuperior", 0xf6e2 },
	{ "commaturnedabovecmb", 0x0312 },
	{ "commaturnedmod", 0x02bb },
	{ "compass", 0x263c },
	{ "contourintegral", 0x222e },
	{ "control", 0x2303 },
	{ "controlACK", 0x0006 },
	{ "controlBEL", 0x0007 },
	{ "controlBS", 0x0008 },
	{ "controlCAN", 0x0018 },
	{ "controlCR", 0x000d },
	{ "controlDC1", 0x0011 },
	{ "controlDC2", 0x0012 },
	{ "controlDC3", 0x0013 },
	{ "controlDC4", 0x0014 },
	{ "controlDEL", 0x007f },
	{ "controlDLE", 0x0010 },
	{ "controlEM", 0x0019 },
	{ "controlENQ", 0x0005 },
	{ "controlEOT", 0x0004 },
	{ "controlESC", 0x001b },
	{ "controlETB", 0x0017 },
	{ "controlETX", 0x0003 },
	{ "controlFF", 0x000c },
	{ "controlFS", 0x001c },
	{ "controlGS", 0x001d },
	{ "controlHT", 0x0009 },
	{ "controlLF", 0x000a },
	{ "controlNAK", 0x0015 },
	{ "controlRS", 0x001e },
	{ "controlSI", 0x000f },
	{ "controlSO", 0x000e },
	{ "controlSOT", 0x0002 },
	{ "controlSTX", 0x0001 },
	{ "controlSUB", 0x001a },
	{ "controlSYN", 0x0016 },
	{ "controlUS", 0x001f },
	{ "controlVT", 0x000b },
	{ "copyrightsans", 0xf8e9 },
	{ "copyrightserif", 0xf6d9 },
	{ "cornerbracketleft", 0x300c },
	{ "cornerbracketlefthalfwidth", 0xff62 },
	{ "cornerbracketleftvertical", 0xfe41 },
	{ "cornerbracketright", 0x300d },
	{ "cornerbracketrighthalfwidth", 0xff63 },
	{ "cornerbracketrightvertical", 0xfe42 },
	{ "corporationsquare", 0x337f },
	{ "cosquare", 0x33c7 },
	{ "coverkgsquare", 0x33c6 },
	{ "cparen", 0x249e },
	{ "cruzeiro", 0x20a2 },
	{ "cstretched", 0x0297 },
	{ "curlyand", 0x22cf },
	{ "curlyor", 0x22ce },
	{ "cyrBreve", 0xf6d1 },
	{ "cyrFlex", 0xf6d2 },
	{ "cyrbreve", 0xf6d4 },
	{ "cyrflex", 0xf6d5 },
	{ "daarmenian", 0x0564 },
	{ "dabengali", 0x09a6 },
	{ "dadarabic", 0x0636 },
	{ "dadeva", 0x0926 },
	{ "dadfinalarabic", 0xfebe },
	{ "dadinitialarabic", 0xfebf },
	{ "dadmedialarabic", 0xfec0 },
	{ "dagesh", 0x05bc },
	{ "dageshhebrew", 0x05bc },
	{ "dagujarati", 0x0aa6 },
	{ "dagurmukhi", 0x0a26 },
	{ "dahiragana", 0x3060 },
	{ "dakatakana", 0x30c0 },
	{ "dalarabic", 0x062f },
	{ "dalet", 0x05d3 },
	{ "daletdagesh", 0xfb33 },
	{ "daletdageshhebrew", 0xfb33 },
	{ "dalethatafpatah", 0x05d3 },
	{ "dalethatafpatahhebrew", 0x05d3 },
	{ "dalethatafsegol", 0x05d3 },
	{ "dalethatafsegolhebrew", 0x05d3 },
	{ "dalethebrew", 0x05d3 },
	{ "dalethiriq", 0x05d3 },
	{ "dalethiriqhebrew", 0x05d3 },
	{ "daletholam", 0x05d3 },
	{ "daletholamhebrew", 0x05d3 },
	{ "daletpatah", 0x05d3 },
	{ "daletpatahhebrew", 0x05d3 },
	{ "daletqamats", 0x05d3 },
	{ "daletqamatshebrew", 0x05d3 },
	{ "daletqubuts", 0x05d3 },
	{ "daletqubutshebrew", 0x05d3 },
	{ "daletsegol", 0x05d3 },
	{ "daletsegolhebrew", 0x05d3 },
	{ "daletsheva", 0x05d3 },
	{ "daletshevahebrew", 0x05d3 },
	{ "dalettsere", 0x05d3 },
	{ "dalettserehebrew", 0x05d3 },
	{ "dalfinalarabic", 0xfeaa },
	{ "dammaarabic", 0x064f },
	{ "dammalowarabic", 0x064f },
	{ "dammatanaltonearabic", 0x064c },
	{ "dammatanarabic", 0x064c },
	{ "danda", 0x0964 },
	{ "dargahebrew", 0x05a7 },
	{ "dargalefthebrew", 0x05a7 },
	{ "dasiapneumatacyrilliccmb", 0x0485 },
	{ "dblGrave", 0xf6d3 },
	{ "dblanglebracketleft", 0x300a },
	{ "dblanglebracketleftvertical", 0xfe3d },
	{ "dblanglebracketright", 0x300b },
	{ "dblanglebracketrightvertical", 0xfe3e },
	{ "dblarchinvertedbelowcmb", 0x032b },
	{ "dblarrowleft", 0x21d4 },
	{ "dblarrowright", 0x21d2 },
	{ "dbldanda", 0x0965 },
	{ "dblgrave", 0xf6d6 },
	{ "dblgravecmb", 0x030f },
	{ "dblintegral", 0x222c },
	{ "dbllowline", 0x2017 },
	{ "dbllowlinecmb", 0x0333 },
	{ "dbloverlinecmb", 0x033f },
	{ "dblprimemod", 0x02ba },
	{ "dblverticalbar", 0x2016 },
	{ "dblverticallineabovecmb", 0x030e },
	{ "dbopomofo", 0x3109 },
	{ "dbsquare", 0x33c8 },
	{ "dcedilla", 0x1e11 },
	{ "dcircle", 0x24d3 },
	{ "dcircumflexbelow", 0x1e13 },
	{ "ddabengali", 0x09a1 },
	{ "ddadeva", 0x0921 },
	{ "ddagujarati", 0x0aa1 },
	{ "ddagurmukhi", 0x0a21 },
	{ "ddalarabic", 0x0688 },
	{ "ddalfinalarabic", 0xfb89 },
	{ "dddhadeva", 0x095c },
	{ "ddhabengali", 0x09a2 },
	{ "ddhadeva", 0x0922 },
	{ "ddhagujarati", 0x0aa2 },
	{ "ddhagurmukhi", 0x0a22 },
	{ "ddotaccent", 0x1e0b },
	{ "ddotbelow", 0x1e0d },
	{ "decimalseparatorarabic", 0x066b },
	{ "decimalseparatorpersian", 0x066b },
	{ "decyrillic", 0x0434 },
	{ "dehihebrew", 0x05ad },
	{ "dehiragana", 0x3067 },
	{ "deicoptic", 0x03ef },
	{ "dekatakana", 0x30c7 },
	{ "deleteleft", 0x232b },
	{ "deleteright", 0x2326 },
	{ "deltaturned", 0x018d },
	{ "denominatorminusonenumeratorbengali", 0x09f8 },
	{ "dezh", 0x02a4 },
	{ "dhabengali", 0x09a7 },
	{ "dhadeva", 0x0927 },
	{ "dhagujarati", 0x0aa7 },
	{ "dhagurmukhi", 0x0a27 },
	{ "dhook", 0x0257 },
	{ "dialytikatonos", 0x0385 },
	{ "dialytikatonoscmb", 0x0344 },
	{ "diamondsuitwhite", 0x2662 },
	{ "dieresisacute", 0xf6d7 },
	{ "dieresisbelowcmb", 0x0324 },
	{ "dieresiscmb", 0x0308 },
	{ "dieresisgrave", 0xf6d8 },
	{ "dihiragana", 0x3062 },
	{ "dikatakana", 0x30c2 },
	{ "dittomark", 0x3003 },
	{ "divides", 0x2223 },
	{ "divisionslash", 0x2215 },
	{ "djecyrillic", 0x0452 },
	{ "dlinebelow", 0x1e0f },
	{ "dlsquare", 0x3397 },
	{ "dmacron", 0x0111 },
	{ "dmonospace", 0xff44 },
	{ "dochadathai", 0x0e0e },
	{ "dodekthai", 0x0e14 },
	{ "dohiragana", 0x3069 },
	{ "dokatakana", 0x30c9 },
	{ "dollarinferior", 0xf6e3 },
	{ "dollarmonospace", 0xff04 },
	{ "dollaroldstyle", 0xf724 },
	{ "dollarsmall", 0xfe69 },
	{ "dollarsuperior", 0xf6e4 },
	{ "dorusquare", 0x3326 },
	{ "dotaccentcmb", 0x0307 },
	{ "dotbelowcmb", 0x0323 },
	{ "dotkatakana", 0x30fb },
	{ "dotlessj", 0xf6be },
	{ "dotlessjstrokehook", 0x0284 },
	{ "dottedcircle", 0x25cc },
	{ "doubleyodpatah", 0xfb1f },
	{ "doubleyodpatahhebrew", 0xfb1f },
	{ "downtackbelowcmb", 0x031e },
	{ "downtackmod", 0x02d5 },
	{ "dparen", 0x249f },
	{ "dsuperior", 0xf6eb },
	{ "dtail", 0x0256 },
	{ "dtopbar", 0x018c },
	{ "duhiragana", 0x3065 },
	{ "dukatakana", 0x30c5 },
	{ "dz", 0x01f3 },
	{ "dzaltone", 0x02a3 },
	{ "dzcaron", 0x01c6 },
	{ "dzcurl", 0x02a5 },
	{ "dzeabkhasiancyrillic", 0x04e1 },
	{ "dzecyrillic", 0x0455 },
	{ "dzhecyrillic", 0x045f },
	{ "earth", 0x2641 },
	{ "ebengali", 0x098f },
	{ "ebopomofo", 0x311c },
	{ "ecandradeva", 0x090d },
	{ "ecandragujarati", 0x0a8d },
	{ "ecandravowelsigndeva", 0x0945 },
	{ "ecandravowelsigngujarati", 0x0ac5 },
	{ "ecedillabreve", 0x1e1d },
	{ "echarmenian", 0x0565 },
	{ "echyiwnarmenian", 0x0587 },
	{ "ecircle", 0x24d4 },
	{ "ecircumflexacute", 0x1ebf },
	{ "ecircumflexbelow", 0x1e19 },
	{ "ecircumflexdotbelow", 0x1ec7 },
	{ "ecircumflexgrave", 0x1ec1 },
	{ "ecircumflexhookabove", 0x1ec3 },
	{ "ecircumflextilde", 0x1ec5 },
	{ "ecyrillic", 0x0454 },
	{ "edblgrave", 0x0205 },
	{ "edeva", 0x090f },
	{ "edot", 0x0117 },
	{ "edotbelow", 0x1eb9 },
	{ "eegurmukhi", 0x0a0f },
	{ "eematragurmukhi", 0x0a47 },
	{ "efcyrillic", 0x0444 },
	{ "egujarati", 0x0a8f },
	{ "eharmenian", 0x0567 },
	{ "ehbopomofo", 0x311d },
	{ "ehiragana", 0x3048 },
	{ "ehookabove", 0x1ebb },
	{ "eibopomofo", 0x311f },
	{ "eightarabic", 0x0668 },
	{ "eightbengali", 0x09ee },
	{ "eightcircle", 0x2467 },
	{ "eightcircleinversesansserif", 0x2791 },
	{ "eightdeva", 0x096e },
	{ "eighteencircle", 0x2471 },
	{ "eighteenparen", 0x2485 },
	{ "eighteenperiod", 0x2499 },
	{ "eightgujarati", 0x0aee },
	{ "eightgurmukhi", 0x0a6e },
	{ "eighthackarabic", 0x0668 },
	{ "eighthangzhou", 0x3028 },
	{ "eighthnotebeamed", 0x266b },
	{ "eightideographicparen", 0x3227 },
	{ "eightinferior", 0x2088 },
	{ "eightmonospace", 0xff18 },
	{ "eightoldstyle", 0xf738 },
	{ "eightparen", 0x247b },
	{ "eightperiod", 0x248f },
	{ "eightpersian", 0x06f8 },
	{ "eightroman", 0x2177 },
	{ "eightsuperior", 0x2078 },
	{ "eightthai", 0x0e58 },
	{ "einvertedbreve", 0x0207 },
	{ "eiotifiedcyrillic", 0x0465 },
	{ "ekatakana", 0x30a8 },
	{ "ekatakanahalfwidth", 0xff74 },
	{ "ekonkargurmukhi", 0x0a74 },
	{ "ekorean", 0x3154 },
	{ "elcyrillic", 0x043b },
	{ "elevencircle", 0x246a },
	{ "elevenparen", 0x247e },
	{ "elevenperiod", 0x2492 },
	{ "elevenroman", 0x217a },
	{ "ellipsisvertical", 0x22ee },
	{ "emacronacute", 0x1e17 },
	{ "emacrongrave", 0x1e15 },
	{ "emcyrillic", 0x043c },
	{ "emdashvertical", 0xfe31 },
	{ "emonospace", 0xff45 },
	{ "emphasismarkarmenian", 0x055b },
	{ "enbopomofo", 0x3123 },
	{ "encyrillic", 0x043d },
	{ "endashvertical", 0xfe32 },
	{ "endescendercyrillic", 0x04a3 },
	{ "engbopomofo", 0x3125 },
	{ "enghecyrillic", 0x04a5 },
	{ "enhookcyrillic", 0x04c8 },
	{ "enspace", 0x2002 },
	{ "eokorean", 0x3153 },
	{ "eopen", 0x025b },
	{ "eopenclosed", 0x029a },
	{ "eopenreversed", 0x025c },
	{ "eopenreversedclosed", 0x025e },
	{ "eopenreversedhook", 0x025d },
	{ "eparen", 0x24a0 },
	{ "equalmonospace", 0xff1d },
	{ "equalsmall", 0xfe66 },
	{ "equalsuperior", 0x207c },
	{ "erbopomofo", 0x3126 },
	{ "ercyrillic", 0x0440 },
	{ "ereversed", 0x0258 },
	{ "ereversedcyrillic", 0x044d },
	{ "escyrillic", 0x0441 },
	{ "esdescendercyrillic", 0x04ab },
	{ "esh", 0x0283 },
	{ "eshcurl", 0x0286 },
	{ "eshortdeva", 0x090e },
	{ "eshortvowelsigndeva", 0x0946 },
	{ "eshreversedloop", 0x01aa },
	{ "eshsquatreversed", 0x0285 },
	{ "esmallhiragana", 0x3047 },
	{ "esmallkatakana", 0x30a7 },
	{ "esmallkatakanahalfwidth", 0xff6a },
	{ "esuperior", 0xf6ec },
	{ "etarmenian", 0x0568 },
	{ "etilde", 0x1ebd },
	{ "etildebelow", 0x1e1b },
	{ "etnahtafoukhhebrew", 0x0591 },
	{ "etnahtafoukhlefthebrew", 0x0591 },
	{ "etnahtahebrew", 0x0591 },
	{ "etnahtalefthebrew", 0x0591 },
	{ "eturned", 0x01dd },
	{ "eukorean", 0x3161 },
	{ "euro", 0x20ac },
	{ "evowelsignbengali", 0x09c7 },
	{ "evowelsigndeva", 0x0947 },
	{ "evowelsigngujarati", 0x0ac7 },
	{ "exclamarmenian", 0x055c },
	{ "exclamdownsmall", 0xf7a1 },
	{ "exclammonospace", 0xff01 },
	{ "exclamsmall", 0xf721 },
	{ "ezh", 0x0292 },
	{ "ezhcaron", 0x01ef },
	{ "ezhcurl", 0x0293 },
	{ "ezhreversed", 0x01b9 },
	{ "ezhtail", 0x01ba },
	{ "fadeva", 0x095e },
	{ "fagurmukhi", 0x0a5e },
	{ "fahrenheit", 0x2109 },
	{ "fathaarabic", 0x064e },
	{ "fathalowarabic", 0x064e },
	{ "fathatanarabic", 0x064b },
	{ "fbopomofo", 0x3108 },
	{ "fcircle", 0x24d5 },
	{ "fdotaccent", 0x1e1f },
	{ "feharabic", 0x0641 },
	{ "feharmenian", 0x0586 },
	{ "fehfinalarabic", 0xfed2 },
	{ "fehinitialarabic", 0xfed3 },
	{ "fehmedialarabic", 0xfed4 },
	{ "feicoptic", 0x03e5 },
	{ "ff", 0xfb00 },
	{ "ffi", 0xfb03 },
	{ "ffl", 0xfb04 },
	{ "fi", 0xfb01 },
	{ "fifteencircle", 0x246e },
	{ "fifteenparen", 0x2482 },
	{ "fifteenperiod", 0x2496 },
	{ "finalkaf", 0x05da },
	{ "finalkafdagesh", 0xfb3a },
	{ "finalkafdageshhebrew", 0xfb3a },
	{ "finalkafhebrew", 0x05da },
	{ "finalkafqamats", 0x05da },
	{ "finalkafqamatshebrew", 0x05da },
	{ "finalkafsheva", 0x05da },
	{ "finalkafshevahebrew", 0x05da },
	{ "finalmem", 0x05dd },
	{ "finalmemhebrew", 0x05dd },
	{ "finalnun", 0x05df },
	{ "finalnunhebrew", 0x05df },
	{ "finalpe", 0x05e3 },
	{ "finalpehebrew", 0x05e3 },
	{ "finaltsadi", 0x05e5 },
	{ "finaltsadihebrew", 0x05e5 },
	{ "firsttonechinese", 0x02c9 },
	{ "fisheye", 0x25c9 },
	{ "fitacyrillic", 0x0473 },
	{ "fivearabic", 0x0665 },
	{ "fivebengali", 0x09eb },
	{ "fivecircle", 0x2464 },
	{ "fivecircleinversesansserif", 0x278e },
	{ "fivedeva", 0x096b },
	{ "fivegujarati", 0x0aeb },
	{ "fivegurmukhi", 0x0a6b },
	{ "fivehackarabic", 0x0665 },
	{ "fivehangzhou", 0x3025 },
	{ "fiveideographicparen", 0x3224 },
	{ "fiveinferior", 0x2085 },
	{ "fivemonospace", 0xff15 },
	{ "fiveoldstyle", 0xf735 },
	{ "fiveparen", 0x2478 },
	{ "fiveperiod", 0x248c },
	{ "fivepersian", 0x06f5 },
	{ "fiveroman", 0x2174 },
	{ "fivesuperior", 0x2075 },
	{ "fivethai", 0x0e55 },
	{ "fl", 0xfb02 },
	{ "fmonospace", 0xff46 },
	{ "fmsquare", 0x3399 },
	{ "fofanthai", 0x0e1f },
	{ "fofathai", 0x0e1d },
	{ "fongmanthai", 0x0e4f },
	{ "forall", 0x2200 },
	{ "fourarabic", 0x0664 },
	{ "fourbengali", 0x09ea },
	{ "fourcircle", 0x2463 },
	{ "fourcircleinversesansserif", 0x278d },
	{ "fourdeva", 0x096a },
	{ "fourgujarati", 0x0aea },
	{ "fourgurmukhi", 0x0a6a },
	{ "fourhackarabic", 0x0664 },
	{ "fourhangzhou", 0x3024 },
	{ "fourideographicparen", 0x3223 },
	{ "fourinferior", 0x2084 },
	{ "fourmonospace", 0xff14 },
	{ "fournumeratorbengali", 0x09f7 },
	{ "fouroldstyle", 0xf734 },
	{ "fourparen", 0x2477 },
	{ "fourperiod", 0x248b },
	{ "fourpersian", 0x06f4 },
	{ "fourroman", 0x2173 },
	{ "foursuperior", 0x2074 },
	{ "fourteencircle", 0x246d },
	{ "fourteenparen", 0x2481 },
	{ "fourteenperiod", 0x2495 },
	{ "fourthai", 0x0e54 },
	{ "fourthtonechinese", 0x02cb },
	{ "fparen", 0x24a1 },
	{ "gabengali", 0x0997 },
	{ "gacute", 0x01f5 },
	{ "gadeva", 0x0917 },
	{ "gafarabic", 0x06af },
	{ "gaffinalarabic", 0xfb93 },
	{ "gafinitialarabic", 0xfb94 },
	{ "gafmedialarabic", 0xfb95 },
	{ "gagujarati", 0x0a97 },
	{ "gagurmukhi", 0x0a17 },
	{ "gahiragana", 0x304c },
	{ "gakatakana", 0x30ac },
	{ "gammalatinsmall", 0x0263 },
	{ "gammasuperior", 0x02e0 },
	{ "gangiacoptic", 0x03eb },
	{ "gbopomofo", 0x310d },
	{ "gcedilla", 0x0123 },
	{ "gcircle", 0x24d6 },
	{ "gdot", 0x0121 },
	{ "gecyrillic", 0x0433 },
	{ "gehiragana", 0x3052 },
	{ "gekatakana", 0x30b2 },
	{ "geometricallyequal", 0x2251 },
	{ "gereshaccenthebrew", 0x059c },
	{ "gereshhebrew", 0x05f3 },
	{ "gereshmuqdamhebrew", 0x059d },
	{ "gershayimaccenthebrew", 0x059e },
	{ "gershayimhebrew", 0x05f4 },
	{ "getamark", 0x3013 },
	{ "ghabengali", 0x0998 },
	{ "ghadarmenian", 0x0572 },
	{ "ghadeva", 0x0918 },
	{ "ghagujarati", 0x0a98 },
	{ "ghagurmukhi", 0x0a18 },
	{ "ghainarabic", 0x063a },
	{ "ghainfinalarabic", 0xfece },
	{ "ghaininitialarabic", 0xfecf },
	{ "ghainmedialarabic", 0xfed0 },
	{ "ghemiddlehookcyrillic", 0x0495 },
	{ "ghestrokecyrillic", 0x0493 },
	{ "gheupturncyrillic", 0x0491 },
	{ "ghhadeva", 0x095a },
	{ "ghhagurmukhi", 0x0a5a },
	{ "ghook", 0x0260 },
	{ "ghzsquare", 0x3393 },
	{ "gihiragana", 0x304e },
	{ "gikatakana", 0x30ae },
	{ "gimarmenian", 0x0563 },
	{ "gimel", 0x05d2 },
	{ "gimeldagesh", 0xfb32 },
	{ "gimeldageshhebrew", 0xfb32 },
	{ "gimelhebrew", 0x05d2 },
	{ "gjecyrillic", 0x0453 },
	{ "glottalinvertedstroke", 0x01be },
	{ "glottalstop", 0x0294 },
	{ "glottalstopinverted", 0x0296 },
	{ "glottalstopmod", 0x02c0 },
	{ "glottalstopreversed", 0x0295 },
	{ "glottalstopreversedmod", 0x02c1 },
	{ "glottalstopreversedsuperior", 0x02e4 },
	{ "glottalstopstroke", 0x02a1 },
	{ "glottalstopstrokereversed", 0x02a2 },
	{ "gmacron", 0x1e21 },
	{ "gmonospace", 0xff47 },
	{ "gohiragana", 0x3054 },
	{ "gokatakana", 0x30b4 },
	{ "gparen", 0x24a2 },
	{ "gpasquare", 0x33ac },
	{ "gravebelowcmb", 0x0316 },
	{ "gravecmb", 0x0300 },
	{ "gravedeva", 0x0953 },
	{ "gravelowmod", 0x02ce },
	{ "gravemonospace", 0xff40 },
	{ "gravetonecmb", 0x0340 },
	{ "greaterequalorless", 0x22db },
	{ "greatermonospace", 0xff1e },
	{ "greaterorequivalent", 0x2273 },
	{ "greaterorless", 0x2277 },
	{ "greateroverequal", 0x2267 },
	{ "greatersmall", 0xfe65 },
	{ "gscript", 0x0261 },
	{ "gstroke", 0x01e5 },
	{ "guhiragana", 0x3050 },
	{ "gukatakana", 0x30b0 },
	{ "guramusquare", 0x3318 },
	{ "gysquare", 0x33c9 },
	{ "haabkhasiancyrillic", 0x04a9 },
	{ "haaltonearabic", 0x06c1 },
	{ "habengali", 0x09b9 },
	{ "hadescendercyrillic", 0x04b3 },
	{ "hadeva", 0x0939 },
	{ "hagujarati", 0x0ab9 },
	{ "hagurmukhi", 0x0a39 },
	{ "haharabic", 0x062d },
	{ "hahfinalarabic", 0xfea2 },
	{ "hahinitialarabic", 0xfea3 },
	{ "hahiragana", 0x306f },
	{ "hahmedialarabic", 0xfea4 },
	{ "haitusquare", 0x332a },
	{ "hakatakana", 0x30cf },
	{ "hakatakanahalfwidth", 0xff8a },
	{ "halantgurmukhi", 0x0a4d },
	{ "hamzaarabic", 0x0621 },
	{ "hamzadammaarabic", 0x0621 },
	{ "hamzadammatanarabic", 0x0621 },
	{ "hamzafathaarabic", 0x0621 },
	{ "hamzafathatanarabic", 0x0621 },
	{ "hamzalowarabic", 0x0621 },
	{ "hamzalowkasraarabic", 0x0621 },
	{ "hamzalowkasratanarabic", 0x0621 },
	{ "hamzasukunarabic", 0x0621 },
	{ "hangulfiller", 0x3164 },
	{ "hardsigncyrillic", 0x044a },
	{ "harpoonleftbarbup", 0x21bc },
	{ "harpoonrightbarbup", 0x21c0 },
	{ "hasquare", 0x33ca },
	{ "hatafpatah", 0x05b2 },
	{ "hatafpatah16", 0x05b2 },
	{ "hatafpatah23", 0x05b2 },
	{ "hatafpatah2f", 0x05b2 },
	{ "hatafpatahhebrew", 0x05b2 },
	{ "hatafpatahnarrowhebrew", 0x05b2 },
	{ "hatafpatahquarterhebrew", 0x05b2 },
	{ "hatafpatahwidehebrew", 0x05b2 },
	{ "hatafqamats", 0x05b3 },
	{ "hatafqamats1b", 0x05b3 },
	{ "hatafqamats28", 0x05b3 },
	{ "hatafqamats34", 0x05b3 },
	{ "hatafqamatshebrew", 0x05b3 },
	{ "hatafqamatsnarrowhebrew", 0x05b3 },
	{ "hatafqamatsquarterhebrew", 0x05b3 },
	{ "hatafqamatswidehebrew", 0x05b3 },
	{ "hatafsegol", 0x05b1 },
	{ "hatafsegol17", 0x05b1 },
	{ "hatafsegol24", 0x05b1 },
	{ "hatafsegol30", 0x05b1 },
	{ "hatafsegolhebrew", 0x05b1 },
	{ "hatafsegolnarrowhebrew", 0x05b1 },
	{ "hatafsegolquarterhebrew", 0x05b1 },
	{ "hatafsegolwidehebrew", 0x05b1 },
	{ "hbopomofo", 0x310f },
	{ "hbrevebelow", 0x1e2b },
	{ "hcedilla", 0x1e29 },
	{ "hcircle", 0x24d7 },
	{ "hdieresis", 0x1e27 },
	{ "hdotaccent", 0x1e23 },
	{ "hdotbelow", 0x1e25 },
	{ "he", 0x05d4 },
	{ "heartsuitblack", 0x2665 },
	{ "heartsuitwhite", 0x2661 },
	{ "hedagesh", 0xfb34 },
	{ "hedageshhebrew", 0xfb34 },
	{ "hehaltonearabic", 0x06c1 },
	{ "heharabic", 0x0647 },
	{ "hehebrew", 0x05d4 },
	{ "hehfinalaltonearabic", 0xfba7 },
	{ "hehfinalalttwoarabic", 0xfeea },
	{ "hehfinalarabic", 0xfeea },
	{ "hehhamzaabovefinalarabic", 0xfba5 },
	{ "hehhamzaaboveisolatedarabic", 0xfba4 },
	{ "hehinitialaltonearabic", 0xfba8 },
	{ "hehinitialarabic", 0xfeeb },
	{ "hehiragana", 0x3078 },
	{ "hehmedialaltonearabic", 0xfba9 },
	{ "hehmedialarabic", 0xfeec },
	{ "heiseierasquare", 0x337b },
	{ "hekatakana", 0x30d8 },
	{ "hekatakanahalfwidth", 0xff8d },
	{ "hekutaarusquare", 0x3336 },
	{ "henghook", 0x0267 },
	{ "herutusquare", 0x3339 },
	{ "het", 0x05d7 },
	{ "hethebrew", 0x05d7 },
	{ "hhook", 0x0266 },
	{ "hhooksuperior", 0x02b1 },
	{ "hieuhacirclekorean", 0x327b },
	{ "hieuhaparenkorean", 0x321b },
	{ "hieuhcirclekorean", 0x326d },
	{ "hieuhkorean", 0x314e },
	{ "hieuhparenkorean", 0x320d },
	{ "hihiragana", 0x3072 },
	{ "hikatakana", 0x30d2 },
	{ "hikatakanahalfwidth", 0xff8b },
	{ "hiriq", 0x05b4 },
	{ "hiriq14", 0x05b4 },
	{ "hiriq21", 0x05b4 },
	{ "hiriq2d", 0x05b4 },
	{ "hiriqhebrew", 0x05b4 },
	{ "hiriqnarrowhebrew", 0x05b4 },
	{ "hiriqquarterhebrew", 0x05b4 },
	{ "hiriqwidehebrew", 0x05b4 },
	{ "hlinebelow", 0x1e96 },
	{ "hmonospace", 0xff48 },
	{ "hoarmenian", 0x0570 },
	{ "hohipthai", 0x0e2b },
	{ "hohiragana", 0x307b },
	{ "hokatakana", 0x30db },
	{ "hokatakanahalfwidth", 0xff8e },
	{ "holam", 0x05b9 },
	{ "holam19", 0x05b9 },
	{ "holam26", 0x05b9 },
	{ "holam32", 0x05b9 },
	{ "holamhebrew", 0x05b9 },
	{ "holamnarrowhebrew", 0x05b9 },
	{ "holamquarterhebrew", 0x05b9 },
	{ "holamwidehebrew", 0x05b9 },
	{ "honokhukthai", 0x0e2e },
	{ "hookcmb", 0x0309 },
	{ "hookpalatalizedbelowcmb", 0x0321 },
	{ "hookretroflexbelowcmb", 0x0322 },
	{ "hoonsquare", 0x3342 },
	{ "horicoptic", 0x03e9 },
	{ "horizontalbar", 0x2015 },
	{ "horncmb", 0x031b },
	{ "hotsprings", 0x2668 },
	{ "hparen", 0x24a3 },
	{ "hsuperior", 0x02b0 },
	{ "hturned", 0x0265 },
	{ "huhiragana", 0x3075 },
	{ "huiitosquare", 0x3333 },
	{ "hukatakana", 0x30d5 },
	{ "hukatakanahalfwidth", 0xff8c },
	{ "hungarumlautcmb", 0x030b },
	{ "hv", 0x0195 },
	{ "hypheninferior", 0xf6e5 },
	{ "hyphenmonospace", 0xff0d },
	{ "hyphensmall", 0xfe63 },
	{ "hyphensuperior", 0xf6e6 },
	{ "hyphentwo", 0x2010 },
	{ "iacyrillic", 0x044f },
	{ "ibengali", 0x0987 },
	{ "ibopomofo", 0x3127 },
	{ "icaron", 0x01d0 },
	{ "icircle", 0x24d8 },
	{ "icyrillic", 0x0456 },
	{ "idblgrave", 0x0209 },
	{ "ideographearthcircle", 0x328f },
	{ "ideographfirecircle", 0x328b },
	{ "ideographicallianceparen", 0x323f },
	{ "ideographiccallparen", 0x323a },
	{ "ideographiccentrecircle", 0x32a5 },
	{ "ideographicclose", 0x3006 },
	{ "ideographiccomma", 0x3001 },
	{ "ideographiccommaleft", 0xff64 },
	{ "ideographiccongratulationparen", 0x3237 },
	{ "ideographiccorrectcircle", 0x32a3 },
	{ "ideographicearthparen", 0x322f },
	{ "ideographicenterpriseparen", 0x323d },
	{ "ideographicexcellentcircle", 0x329d },
	{ "ideographicfestivalparen", 0x3240 },
	{ "ideographicfinancialcircle", 0x3296 },
	{ "ideographicfinancialparen", 0x3236 },
	{ "ideographicfireparen", 0x322b },
	{ "ideographichaveparen", 0x3232 },
	{ "ideographichighcircle", 0x32a4 },
	{ "ideographiciterationmark", 0x3005 },
	{ "ideographiclaborcircle", 0x3298 },
	{ "ideographiclaborparen", 0x3238 },
	{ "ideographicleftcircle", 0x32a7 },
	{ "ideographiclowcircle", 0x32a6 },
	{ "ideographicmedicinecircle", 0x32a9 },
	{ "ideographicmetalparen", 0x322e },
	{ "ideographicmoonparen", 0x322a },
	{ "ideographicnameparen", 0x3234 },
	{ "ideographicperiod", 0x3002 },
	{ "ideographicprintcircle", 0x329e },
	{ "ideographicreachparen", 0x3243 },
	{ "ideographicrepresentparen", 0x3239 },
	{ "ideographicresourceparen", 0x323e },
	{ "ideographicrightcircle", 0x32a8 },
	{ "ideographicsecretcircle", 0x3299 },
	{ "ideographicselfparen", 0x3242 },
	{ "ideographicsocietyparen", 0x3233 },
	{ "ideographicspace", 0x3000 },
	{ "ideographicspecialparen", 0x3235 },
	{ "ideographicstockparen", 0x3231 },
	{ "ideographicstudyparen", 0x323b },
	{ "ideographicsunparen", 0x3230 },
	{ "ideographicsuperviseparen", 0x323c },
	{ "ideographicwaterparen", 0x322c },
	{ "ideographicwoodparen", 0x322d },
	{ "ideographiczero", 0x3007 },
	{ "ideographmetalcircle", 0x328e },
	{ "ideographmooncircle", 0x328a },
	{ "ideographnamecircle", 0x3294 },
	{ "ideographsuncircle", 0x3290 },
	{ "ideographwatercircle", 0x328c },
	{ "ideographwoodcircle", 0x328d },
	{ "ideva", 0x0907 },
	{ "idieresisacute", 0x1e2f },
	{ "idieresiscyrillic", 0x04e5 },
	{ "idotbelow", 0x1ecb },
	{ "iebrevecyrillic", 0x04d7 },
	{ "iecyrillic", 0x0435 },
	{ "ieungacirclekorean", 0x3275 },
	{ "ieungaparenkorean", 0x3215 },
	{ "ieungcirclekorean", 0x3267 },
	{ "ieungkorean", 0x3147 },
	{ "ieungparenkorean", 0x3207 },
	{ "igujarati", 0x0a87 },
	{ "igurmukhi", 0x0a07 },
	{ "ihiragana", 0x3044 },
	{ "ihookabove", 0x1ec9 },
	{ "iibengali", 0x0988 },
	{ "iicyrillic", 0x0438 },
	{ "iideva", 0x0908 },
	{ "iigujarati", 0x0a88 },
	{ "iigurmukhi", 0x0a08 },
	{ "iimatragurmukhi", 0x0a40 },
	{ "iinvertedbreve", 0x020b },
	{ "iishortcyrillic", 0x0439 },
	{ "iivowelsignbengali", 0x09c0 },
	{ "iivowelsigndeva", 0x0940 },
	{ "iivowelsigngujarati", 0x0ac0 },
	{ "ikatakana", 0x30a4 },
	{ "ikatakanahalfwidth", 0xff72 },
	{ "ikorean", 0x3163 },
	{ "ilde", 0x02dc },
	{ "iluyhebrew", 0x05ac },
	{ "imacroncyrillic", 0x04e3 },
	{ "imageorapproximatelyequal", 0x2253 },
	{ "imatragurmukhi", 0x0a3f },
	{ "imonospace", 0xff49 },
	{ "increment", 0x2206 },
	{ "iniarmenian", 0x056b },
	{ "integralbottom", 0x2321 },
	{ "integralex", 0xf8f5 },
	{ "integraltop", 0x2320 },
	{ "intisquare", 0x3305 },
	{ "iocyrillic", 0x0451 },
	{ "iotalatin", 0x0269 },
	{ "iparen", 0x24a4 },
	{ "irigurmukhi", 0x0a72 },
	{ "ismallhiragana", 0x3043 },
	{ "ismallkatakana", 0x30a3 },
	{ "ismallkatakanahalfwidth", 0xff68 },
	{ "issharbengali", 0x09fa },
	{ "istroke", 0x0268 },
	{ "isuperior", 0xf6ed },
	{ "iterationhiragana", 0x309d },
	{ "iterationkatakana", 0x30fd },
	{ "itildebelow", 0x1e2d },
	{ "iubopomofo", 0x3129 },
	{ "iucyrillic", 0x044e },
	{ "ivowelsignbengali", 0x09bf },
	{ "ivowelsigndeva", 0x093f },
	{ "ivowelsigngujarati", 0x0abf },
	{ "izhitsacyrillic", 0x0475 },
	{ "izhitsadblgravecyrillic", 0x0477 },
	{ "jaarmenian", 0x0571 },
	{ "jabengali", 0x099c },
	{ "jadeva", 0x091c },
	{ "jagujarati", 0x0a9c },
	{ "jagurmukhi", 0x0a1c },
	{ "jbopomofo", 0x3110 },
	{ "jcaron", 0x01f0 },
	{ "jcircle", 0x24d9 },
	{ "jcrossedtail", 0x029d },
	{ "jdotlessstroke", 0x025f },
	{ "jecyrillic", 0x0458 },
	{ "jeemarabic", 0x062c },
	{ "jeemfinalarabic", 0xfe9e },
	{ "jeeminitialarabic", 0xfe9f },
	{ "jeemmedialarabic", 0xfea0 },
	{ "jeharabic", 0x0698 },
	{ "jehfinalarabic", 0xfb8b },
	{ "jhabengali", 0x099d },
	{ "jhadeva", 0x091d },
	{ "jhagujarati", 0x0a9d },
	{ "jhagurmukhi", 0x0a1d },
	{ "jheharmenian", 0x057b },
	{ "jis", 0x3004 },
	{ "jmonospace", 0xff4a },
	{ "jparen", 0x24a5 },
	{ "jsuperior", 0x02b2 },
	{ "kabashkircyrillic", 0x04a1 },
	{ "kabengali", 0x0995 },
	{ "kacute", 0x1e31 },
	{ "kacyrillic", 0x043a },
	{ "kadescendercyrillic", 0x049b },
	{ "kadeva", 0x0915 },
	{ "kaf", 0x05db },
	{ "kafarabic", 0x0643 },
	{ "kafdagesh", 0xfb3b },
	{ "kafdageshhebrew", 0xfb3b },
	{ "kaffinalarabic", 0xfeda },
	{ "kafhebrew", 0x05db },
	{ "kafinitialarabic", 0xfedb },
	{ "kafmedialarabic", 0xfedc },
	{ "kafrafehebrew", 0xfb4d },
	{ "kagujarati", 0x0a95 },
	{ "kagurmukhi", 0x0a15 },
	{ "kahiragana", 0x304b },
	{ "kahookcyrillic", 0x04c4 },
	{ "kakatakana", 0x30ab },
	{ "kakatakanahalfwidth", 0xff76 },
	{ "kappasymbolgreek", 0x03f0 },
	{ "kapyeounmieumkorean", 0x3171 },
	{ "kapyeounphieuphkorean", 0x3184 },
	{ "kapyeounpieupkorean", 0x3178 },
	{ "kapyeounssangpieupkorean", 0x3179 },
	{ "karoriisquare", 0x330d },
	{ "kashidaautoarabic", 0x0640 },
	{ "kashidaautonosidebearingarabic", 0x0640 },
	{ "kasmallkatakana", 0x30f5 },
	{ "kasquare", 0x3384 },
	{ "kasraarabic", 0x0650 },
	{ "kasratanarabic", 0x064d },
	{ "kastrokecyrillic", 0x049f },
	{ "katahiraprolongmarkhalfwidth", 0xff70 },
	{ "kaverticalstrokecyrillic", 0x049d },
	{ "kbopomofo", 0x310e },
	{ "kcalsquare", 0x3389 },
	{ "kcaron", 0x01e9 },
	{ "kcedilla", 0x0137 },
	{ "kcircle", 0x24da },
	{ "kdotbelow", 0x1e33 },
	{ "keharmenian", 0x0584 },
	{ "kehiragana", 0x3051 },
	{ "kekatakana", 0x30b1 },
	{ "kekatakanahalfwidth", 0xff79 },
	{ "kenarmenian", 0x056f },
	{ "kesmallkatakana", 0x30f6 },
	{ "khabengali", 0x0996 },
	{ "khacyrillic", 0x0445 },
	{ "khadeva", 0x0916 },
	{ "khagujarati", 0x0a96 },
	{ "khagurmukhi", 0x0a16 },
	{ "khaharabic", 0x062e },
	{ "khahfinalarabic", 0xfea6 },
	{ "khahinitialarabic", 0xfea7 },
	{ "khahmedialarabic", 0xfea8 },
	{ "kheicoptic", 0x03e7 },
	{ "khhadeva", 0x0959 },
	{ "khhagurmukhi", 0x0a59 },
	{ "khieukhacirclekorean", 0x3278 },
	{ "khieukhaparenkorean", 0x3218 },
	{ "khieukhcirclekorean", 0x326a },
	{ "khieukhkorean", 0x314b },
	{ "khieukhparenkorean", 0x320a },
	{ "khokhaithai", 0x0e02 },
	{ "khokhonthai", 0x0e05 },
	{ "khokhuatthai", 0x0e03 },
	{ "khokhwaithai", 0x0e04 },
	{ "khomutthai", 0x0e5b },
	{ "khook", 0x0199 },
	{ "khorakhangthai", 0x0e06 },
	{ "khzsquare", 0x3391 },
	{ "kihiragana", 0x304d },
	{ "kikatakana", 0x30ad },
	{ "kikatakanahalfwidth", 0xff77 },
	{ "kiroguramusquare", 0x3315 },
	{ "kiromeetorusquare", 0x3316 },
	{ "kirosquare", 0x3314 },
	{ "kiyeokacirclekorean", 0x326e },
	{ "kiyeokaparenkorean", 0x320e },
	{ "kiyeokcirclekorean", 0x3260 },
	{ "kiyeokkorean", 0x3131 },
	{ "kiyeokparenkorean", 0x3200 },
	{ "kiyeoksioskorean", 0x3133 },
	{ "kjecyrillic", 0x045c },
	{ "klinebelow", 0x1e35 },
	{ "klsquare", 0x3398 },
	{ "kmcubedsquare", 0x33a6 },
	{ "kmonospace", 0xff4b },
	{ "kmsquaredsquare", 0x33a2 },
	{ "kohiragana", 0x3053 },
	{ "kohmsquare", 0x33c0 },
	{ "kokaithai", 0x0e01 },
	{ "kokatakana", 0x30b3 },
	{ "kokatakanahalfwidth", 0xff7a },
	{ "kooposquare", 0x331e },
	{ "koppacyrillic", 0x0481 },
	{ "koreanstandardsymbol", 0x327f },
	{ "koroniscmb", 0x0343 },
	{ "kparen", 0x24a6 },
	{ "kpasquare", 0x33aa },
	{ "ksicyrillic", 0x046f },
	{ "ktsquare", 0x33cf },
	{ "kturned", 0x029e },
	{ "kuhiragana", 0x304f },
	{ "kukatakana", 0x30af },
	{ "kukatakanahalfwidth", 0xff78 },
	{ "kvsquare", 0x33b8 },
	{ "kwsquare", 0x33be },
	{ "labengali", 0x09b2 },
	{ "ladeva", 0x0932 },
	{ "lagujarati", 0x0ab2 },
	{ "lagurmukhi", 0x0a32 },
	{ "lakkhangyaothai", 0x0e45 },
	{ "lamaleffinalarabic", 0xfefc },
	{ "lamalefhamzaabovefinalarabic", 0xfef8 },
	{ "lamalefhamzaaboveisolatedarabic", 0xfef7 },
	{ "lamalefhamzabelowfinalarabic", 0xfefa },
	{ "lamalefhamzabelowisolatedarabic", 0xfef9 },
	{ "lamalefisolatedarabic", 0xfefb },
	{ "lamalefmaddaabovefinalarabic", 0xfef6 },
	{ "lamalefmaddaaboveisolatedarabic", 0xfef5 },
	{ "lamarabic", 0x0644 },
	{ "lambdastroke", 0x019b },
	{ "lamed", 0x05dc },
	{ "lameddagesh", 0xfb3c },
	{ "lameddageshhebrew", 0xfb3c },
	{ "lamedhebrew", 0x05dc },
	{ "lamedholam", 0x05dc },
	{ "lamedholamdagesh", 0x05dc },
	{ "lamedholamdageshhebrew", 0x05dc },
	{ "lamedholamhebrew", 0x05dc },
	{ "lamfinalarabic", 0xfede },
	{ "lamhahinitialarabic", 0xfcca },
	{ "laminitialarabic", 0xfedf },
	{ "lamjeeminitialarabic", 0xfcc9 },
	{ "lamkhahinitialarabic", 0xfccb },
	{ "lamlamhehisolatedarabic", 0xfdf2 },
	{ "lammedialarabic", 0xfee0 },
	{ "lammeemhahinitialarabic", 0xfd88 },
	{ "lammeeminitialarabic", 0xfccc },
	{ "lammeemjeeminitialarabic", 0xfedf },
	{ "lammeemkhahinitialarabic", 0xfedf },
	{ "largecircle", 0x25ef },
	{ "lbar", 0x019a },
	{ "lbelt", 0x026c },
	{ "lbopomofo", 0x310c },
	{ "lcedilla", 0x013c },
	{ "lcircle", 0x24db },
	{ "lcircumflexbelow", 0x1e3d },
	{ "ldotaccent", 0x0140 },
	{ "ldotbelow", 0x1e37 },
	{ "ldotbelowmacron", 0x1e39 },
	{ "leftangleabovecmb", 0x031a },
	{ "lefttackbelowcmb", 0x0318 },
	{ "lessequalorgreater", 0x22da },
	{ "lessmonospace", 0xff1c },
	{ "lessorequivalent", 0x2272 },
	{ "lessorgreater", 0x2276 },
	{ "lessoverequal", 0x2266 },
	{ "lesssmall", 0xfe64 },
	{ "lezh", 0x026e },
	{ "lhookretroflex", 0x026d },
	{ "liwnarmenian", 0x056c },
	{ "lj", 0x01c9 },
	{ "ljecyrillic", 0x0459 },
	{ "ll", 0xf6c0 },
	{ "lladeva", 0x0933 },
	{ "llagujarati", 0x0ab3 },
	{ "llinebelow", 0x1e3b },
	{ "llladeva", 0x0934 },
	{ "llvocalicbengali", 0x09e1 },
	{ "llvocalicdeva", 0x0961 },
	{ "llvocalicvowelsignbengali", 0x09e3 },
	{ "llvocalicvowelsigndeva", 0x0963 },
	{ "lmiddletilde", 0x026b },
	{ "lmonospace", 0xff4c },
	{ "lmsquare", 0x33d0 },
	{ "lochulathai", 0x0e2c },
	{ "logicalnotreversed", 0x2310 },
	{ "lolingthai", 0x0e25 },
	{ "lowlinecenterline", 0xfe4e },
	{ "lowlinecmb", 0x0332 },
	{ "lowlinedashed", 0xfe4d },
	{ "lparen", 0x24a7 },
	{ "lsquare", 0x2113 },
	{ "lsuperior", 0xf6ee },
	{ "luthai", 0x0e26 },
	{ "lvocalicbengali", 0x098c },
	{ "lvocalicdeva", 0x090c },
	{ "lvocalicvowelsignbengali", 0x09e2 },
	{ "lvocalicvowelsigndeva", 0x0962 },
	{ "lxsquare", 0x33d3 },
	{ "mabengali", 0x09ae },
	{ "macronbelowcmb", 0x0331 },
	{ "macroncmb", 0x0304 },
	{ "macronlowmod", 0x02cd },
	{ "macronmonospace", 0xffe3 },
	{ "macute", 0x1e3f },
	{ "madeva", 0x092e },
	{ "magujarati", 0x0aae },
	{ "magurmukhi", 0x0a2e },
	{ "mahapakhhebrew", 0x05a4 },
	{ "mahapakhlefthebrew", 0x05a4 },
	{ "mahiragana", 0x307e },
	{ "maichattawalowleftthai", 0xf895 },
	{ "maichattawalowrightthai", 0xf894 },
	{ "maichattawathai", 0x0e4b },
	{ "maichattawaupperleftthai", 0xf893 },
	{ "maieklowleftthai", 0xf88c },
	{ "maieklowrightthai", 0xf88b },
	{ "maiekthai", 0x0e48 },
	{ "maiekupperleftthai", 0xf88a },
	{ "maihanakatleftthai", 0xf884 },
	{ "maihanakatthai", 0x0e31 },
	{ "maitaikhuleftthai", 0xf889 },
	{ "maitaikhuthai", 0x0e47 },
	{ "maitholowleftthai", 0xf88f },
	{ "maitholowrightthai", 0xf88e },
	{ "maithothai", 0x0e49 },
	{ "maithoupperleftthai", 0xf88d },
	{ "maitrilowleftthai", 0xf892 },
	{ "maitrilowrightthai", 0xf891 },
	{ "maitrithai", 0x0e4a },
	{ "maitriupperleftthai", 0xf890 },
	{ "maiyamokthai", 0x0e46 },
	{ "makatakana", 0x30de },
	{ "makatakanahalfwidth", 0xff8f },
	{ "mansyonsquare", 0x3347 },
	{ "maqafhebrew", 0x05be },
	{ "mars", 0x2642 },
	{ "masoracirclehebrew", 0x05af },
	{ "masquare", 0x3383 },
	{ "mbopomofo", 0x3107 },
	{ "mbsquare", 0x33d4 },
	{ "mcircle", 0x24dc },
	{ "mcubedsquare", 0x33a5 },
	{ "mdotaccent", 0x1e41 },
	{ "mdotbelow", 0x1e43 },
	{ "meemarabic", 0x0645 },
	{ "meemfinalarabic", 0xfee2 },
	{ "meeminitialarabic", 0xfee3 },
	{ "meemmedialarabic", 0xfee4 },
	{ "meemmeeminitialarabic", 0xfcd1 },
	{ "meemmeemisolatedarabic", 0xfc48 },
	{ "meetorusquare", 0x334d },
	{ "mehiragana", 0x3081 },
	{ "meizierasquare", 0x337e },
	{ "mekatakana", 0x30e1 },
	{ "mekatakanahalfwidth", 0xff92 },
	{ "mem", 0x05de },
	{ "memdagesh", 0xfb3e },
	{ "memdageshhebrew", 0xfb3e },
	{ "memhebrew", 0x05de },
	{ "menarmenian", 0x0574 },
	{ "merkhahebrew", 0x05a5 },
	{ "merkhakefulahebrew", 0x05a6 },
	{ "merkhakefulalefthebrew", 0x05a6 },
	{ "merkhalefthebrew", 0x05a5 },
	{ "mhook", 0x0271 },
	{ "mhzsquare", 0x3392 },
	{ "middledotkatakanahalfwidth", 0xff65 },
	{ "middot", 0x00b7 },
	{ "mieumacirclekorean", 0x3272 },
	{ "mieumaparenkorean", 0x3212 },
	{ "mieumcirclekorean", 0x3264 },
	{ "mieumkorean", 0x3141 },
	{ "mieumpansioskorean", 0x3170 },
	{ "mieumparenkorean", 0x3204 },
	{ "mieumpieupkorean", 0x316e },
	{ "mieumsioskorean", 0x316f },
	{ "mihiragana", 0x307f },
	{ "mikatakana", 0x30df },
	{ "mikatakanahalfwidth", 0xff90 },
	{ "minusbelowcmb", 0x0320 },
	{ "minuscircle", 0x2296 },
	{ "minusmod", 0x02d7 },
	{ "minusplus", 0x2213 },
	{ "miribaarusquare", 0x334a },
	{ "mirisquare", 0x3349 },
	{ "mlonglegturned", 0x0270 },
	{ "mlsquare", 0x3396 },
	{ "mmcubedsquare", 0x33a3 },
	{ "mmonospace", 0xff4d },
	{ "mmsquaredsquare", 0x339f },
	{ "mohiragana", 0x3082 },
	{ "mohmsquare", 0x33c1 },
	{ "mokatakana", 0x30e2 },
	{ "mokatakanahalfwidth", 0xff93 },
	{ "molsquare", 0x33d6 },
	{ "momathai", 0x0e21 },
	{ "moverssquare", 0x33a7 },
	{ "moverssquaredsquare", 0x33a8 },
	{ "mparen", 0x24a8 },
	{ "mpasquare", 0x33ab },
	{ "mssquare", 0x33b3 },
	{ "msuperior", 0xf6ef },
	{ "mturned", 0x026f },
	{ "mu1", 0x00b5 },
	{ "muasquare", 0x3382 },
	{ "muchgreater", 0x226b },
	{ "muchless", 0x226a },
	{ "mufsquare", 0x338c },
	{ "mugreek", 0x03bc },
	{ "mugsquare", 0x338d },
	{ "muhiragana", 0x3080 },
	{ "mukatakana", 0x30e0 },
	{ "mukatakanahalfwidth", 0xff91 },
	{ "mulsquare", 0x3395 },
	{ "mumsquare", 0x339b },
	{ "munahhebrew", 0x05a3 },
	{ "munahlefthebrew", 0x05a3 },
	{ "musicflatsign", 0x266d },
	{ "musicsharpsign", 0x266f },
	{ "mussquare", 0x33b2 },
	{ "muvsquare", 0x33b6 },
	{ "muwsquare", 0x33bc },
	{ "mvmegasquare", 0x33b9 },
	{ "mvsquare", 0x33b7 },
	{ "mwmegasquare", 0x33bf },
	{ "mwsquare", 0x33bd },
	{ "nabengali", 0x09a8 },
	{ "nabla", 0x2207 },
	{ "nadeva", 0x0928 },
	{ "nagujarati", 0x0aa8 },
	{ "nagurmukhi", 0x0a28 },
	{ "nahiragana", 0x306a },
	{ "nakatakana", 0x30ca },
	{ "nakatakanahalfwidth", 0xff85 },
	{ "nasquare", 0x3381 },
	{ "nbopomofo", 0x310b },
	{ "nbspace", 0x00a0 },
	{ "ncedilla", 0x0146 },
	{ "ncircle", 0x24dd },
	{ "ncircumflexbelow", 0x1e4b },
	{ "ndotaccent", 0x1e45 },
	{ "ndotbelow", 0x1e47 },
	{ "nehiragana", 0x306d },
	{ "nekatakana", 0x30cd },
	{ "nekatakanahalfwidth", 0xff88 },
	{ "newsheqelsign", 0x20aa },
	{ "nfsquare", 0x338b },
	{ "ngabengali", 0x0999 },
	{ "ngadeva", 0x0919 },
	{ "ngagujarati", 0x0a99 },
	{ "ngagurmukhi", 0x0a19 },
	{ "ngonguthai", 0x0e07 },
	{ "nhiragana", 0x3093 },
	{ "nhookleft", 0x0272 },
	{ "nhookretroflex", 0x0273 },
	{ "nieunacirclekorean", 0x326f },
	{ "nieunaparenkorean", 0x320f },
	{ "nieuncieuckorean", 0x3135 },
	{ "nieuncirclekorean", 0x3261 },
	{ "nieunhieuhkorean", 0x3136 },
	{ "nieunkorean", 0x3134 },
	{ "nieunpansioskorean", 0x3168 },
	{ "nieunparenkorean", 0x3201 },
	{ "nieunsioskorean", 0x3167 },
	{ "nieuntikeutkorean", 0x3166 },
	{ "nihiragana", 0x306b },
	{ "nikatakana", 0x30cb },
	{ "nikatakanahalfwidth", 0xff86 },
	{ "nikhahitleftthai", 0xf899 },
	{ "nikhahitthai", 0x0e4d },
	{ "ninearabic", 0x0669 },
	{ "ninebengali", 0x09ef },
	{ "ninecircle", 0x2468 },
	{ "ninecircleinversesansserif", 0x2792 },
	{ "ninedeva", 0x096f },
	{ "ninegujarati", 0x0aef },
	{ "ninegurmukhi", 0x0a6f },
	{ "ninehackarabic", 0x0669 },
	{ "ninehangzhou", 0x3029 },
	{ "nineideographicparen", 0x3228 },
	{ "nineinferior", 0x2089 },
	{ "ninemonospace", 0xff19 },
	{ "nineoldstyle", 0xf739 },
	{ "nineparen", 0x247c },
	{ "nineperiod", 0x2490 },
	{ "ninepersian", 0x06f9 },
	{ "nineroman", 0x2178 },
	{ "ninesuperior", 0x2079 },
	{ "nineteencircle", 0x2472 },
	{ "nineteenparen", 0x2486 },
	{ "nineteenperiod", 0x249a },
	{ "ninethai", 0x0e59 },
	{ "nj", 0x01cc },
	{ "njecyrillic", 0x045a },
	{ "nkatakana", 0x30f3 },
	{ "nkatakanahalfwidth", 0xff9d },
	{ "nlegrightlong", 0x019e },
	{ "nlinebelow", 0x1e49 },
	{ "nmonospace", 0xff4e },
	{ "nmsquare", 0x339a },
	{ "nnabengali", 0x09a3 },
	{ "nnadeva", 0x0923 },
	{ "nnagujarati", 0x0aa3 },
	{ "nnagurmukhi", 0x0a23 },
	{ "nnnadeva", 0x0929 },
	{ "nohiragana", 0x306e },
	{ "nokatakana", 0x30ce },
	{ "nokatakanahalfwidth", 0xff89 },
	{ "nonbreakingspace", 0x00a0 },
	{ "nonenthai", 0x0e13 },
	{ "nonuthai", 0x0e19 },
	{ "noonarabic", 0x0646 },
	{ "noonfinalarabic", 0xfee6 },
	{ "noonghunnaarabic", 0x06ba },
	{ "noonghunnafinalarabic", 0xfb9f },
	{ "noonhehinitialarabic", 0xfee7 },
	{ "nooninitialarabic", 0xfee7 },
	{ "noonjeeminitialarabic", 0xfcd2 },
	{ "noonjeemisolatedarabic", 0xfc4b },
	{ "noonmedialarabic", 0xfee8 },
	{ "noonmeeminitialarabic", 0xfcd5 },
	{ "noonmeemisolatedarabic", 0xfc4e },
	{ "noonnoonfinalarabic", 0xfc8d },
	{ "notcontains", 0x220c },
	{ "notelementof", 0x2209 },
	{ "notgreater", 0x226f },
	{ "notgreaternorequal", 0x2271 },
	{ "notgreaternorless", 0x2279 },
	{ "notidentical", 0x2262 },
	{ "notless", 0x226e },
	{ "notlessnorequal", 0x2270 },
	{ "notparallel", 0x2226 },
	{ "notprecedes", 0x2280 },
	{ "notsucceeds", 0x2281 },
	{ "notsuperset", 0x2285 },
	{ "nowarmenian", 0x0576 },
	{ "nparen", 0x24a9 },
	{ "nssquare", 0x33b1 },
	{ "nsuperior", 0x207f },
	{ "nuhiragana", 0x306c },
	{ "nukatakana", 0x30cc },
	{ "nukatakanahalfwidth", 0xff87 },
	{ "nuktabengali", 0x09bc },
	{ "nuktadeva", 0x093c },
	{ "nuktagujarati", 0x0abc },
	{ "nuktagurmukhi", 0x0a3c },
	{ "numbersignmonospace", 0xff03 },
	{ "numbersignsmall", 0xfe5f },
	{ "numeralsigngreek", 0x0374 },
	{ "numeralsignlowergreek", 0x0375 },
	{ "numero", 0x2116 },
	{ "nun", 0x05e0 },
	{ "nundagesh", 0xfb40 },
	{ "nundageshhebrew", 0xfb40 },
	{ "nunhebrew", 0x05e0 },
	{ "nvsquare", 0x33b5 },
	{ "nwsquare", 0x33bb },
	{ "nyabengali", 0x099e },
	{ "nyadeva", 0x091e },
	{ "nyagujarati", 0x0a9e },
	{ "nyagurmukhi", 0x0a1e },
	{ "oangthai", 0x0e2d },
	{ "obarred", 0x0275 },
	{ "obarredcyrillic", 0x04e9 },
	{ "obarreddieresiscyrillic", 0x04eb },
	{ "obengali", 0x0993 },
	{ "obopomofo", 0x311b },
	{ "ocandradeva", 0x0911 },
	{ "ocandragujarati", 0x0a91 },
	{ "ocandravowelsigndeva", 0x0949 },
	{ "ocandravowelsigngujarati", 0x0ac9 },
	{ "ocaron", 0x01d2 },
	{ "ocircle", 0x24de },
	{ "ocircumflexacute", 0x1ed1 },
	{ "ocircumflexdotbelow", 0x1ed9 },
	{ "ocircumflexgrave", 0x1ed3 },
	{ "ocircumflexhookabove", 0x1ed5 },
	{ "ocircumflextilde", 0x1ed7 },
	{ "ocyrillic", 0x043e },
	{ "odblacute", 0x0151 },
	{ "odblgrave", 0x020d },
	{ "odeva", 0x0913 },
	{ "odieresiscyrillic", 0x04e7 },
	{ "odotbelow", 0x1ecd },
	{ "oekorean", 0x315a },
	{ "ogonekcmb", 0x0328 },
	{ "ogujarati", 0x0a93 },
	{ "oharmenian", 0x0585 },
	{ "ohiragana", 0x304a },
	{ "ohookabove", 0x1ecf },
	{ "ohornacute", 0x1edb },
	{ "ohorndotbelow", 0x1ee3 },
	{ "ohorngrave", 0x1edd },
	{ "ohornhookabove", 0x1edf },
	{ "ohorntilde", 0x1ee1 },
	{ "oi", 0x01a3 },
	{ "oinvertedbreve", 0x020f },
	{ "okatakana", 0x30aa },
	{ "okatakanahalfwidth", 0xff75 },
	{ "okorean", 0x3157 },
	{ "olehebrew", 0x05ab },
	{ "omacronacute", 0x1e53 },
	{ "omacrongrave", 0x1e51 },
	{ "omdeva", 0x0950 },
	{ "omegacyrillic", 0x0461 },
	{ "omegalatinclosed", 0x0277 },
	{ "omegaroundcyrillic", 0x047b },
	{ "omegatitlocyrillic", 0x047d },
	{ "omgujarati", 0x0ad0 },
	{ "omonospace", 0xff4f },
	{ "onearabic", 0x0661 },
	{ "onebengali", 0x09e7 },
	{ "onecircle", 0x2460 },
	{ "onecircleinversesansserif", 0x278a },
	{ "onedeva", 0x0967 },
	{ "onefitted", 0xf6dc },
	{ "onegujarati", 0x0ae7 },
	{ "onegurmukhi", 0x0a67 },
	{ "onehackarabic", 0x0661 },
	{ "onehangzhou", 0x3021 },
	{ "oneideographicparen", 0x3220 },
	{ "oneinferior", 0x2081 },
	{ "onemonospace", 0xff11 },
	{ "onenumeratorbengali", 0x09f4 },
	{ "oneoldstyle", 0xf731 },
	{ "oneparen", 0x2474 },
	{ "oneperiod", 0x2488 },
	{ "onepersian", 0x06f1 },
	{ "oneroman", 0x2170 },
	{ "onesuperior", 0x00b9 },
	{ "onethai", 0x0e51 },
	{ "oogonek", 0x01eb },
	{ "oogonekmacron", 0x01ed },
	{ "oogurmukhi", 0x0a13 },
	{ "oomatragurmukhi", 0x0a4b },
	{ "oopen", 0x0254 },
	{ "oparen", 0x24aa },
	{ "option", 0x2325 },
	{ "oshortdeva", 0x0912 },
	{ "oshortvowelsigndeva", 0x094a },
	{ "osmallhiragana", 0x3049 },
	{ "osmallkatakana", 0x30a9 },
	{ "osmallkatakanahalfwidth", 0xff6b },
	{ "ostrokeacute", 0x01ff },
	{ "osuperior", 0xf6f0 },
	{ "otcyrillic", 0x047f },
	{ "otildeacute", 0x1e4d },
	{ "otildedieresis", 0x1e4f },
	{ "oubopomofo", 0x3121 },
	{ "overline", 0x203e },
	{ "overlinecenterline", 0xfe4a },
	{ "overlinecmb", 0x0305 },
	{ "overlinedashed", 0xfe49 },
	{ "overlinedblwavy", 0xfe4c },
	{ "overlinewavy", 0xfe4b },
	{ "overscore", 0x00af },
	{ "ovowelsignbengali", 0x09cb },
	{ "ovowelsigndeva", 0x094b },
	{ "ovowelsigngujarati", 0x0acb },
	{ "paampssquare", 0x3380 },
	{ "paasentosquare", 0x332b },
	{ "pabengali", 0x09aa },
	{ "pacute", 0x1e55 },
	{ "padeva", 0x092a },
	{ "pagedown", 0x21df },
	{ "pageup", 0x21de },
	{ "pagujarati", 0x0aaa },
	{ "pagurmukhi", 0x0a2a },
	{ "pahiragana", 0x3071 },
	{ "paiyannoithai", 0x0e2f },
	{ "pakatakana", 0x30d1 },
	{ "palatalizationcyrilliccmb", 0x0484 },
	{ "palochkacyrillic", 0x04c0 },
	{ "pansioskorean", 0x317f },
	{ "parallel", 0x2225 },
	{ "parenleftaltonearabic", 0xfd3e },
	{ "parenleftbt", 0xf8ed },
	{ "parenleftex", 0xf8ec },
	{ "parenleftinferior", 0x208d },
	{ "parenleftmonospace", 0xff08 },
	{ "parenleftsmall", 0xfe59 },
	{ "parenleftsuperior", 0x207d },
	{ "parenlefttp", 0xf8eb },
	{ "parenleftvertical", 0xfe35 },
	{ "parenrightaltonearabic", 0xfd3f },
	{ "parenrightbt", 0xf8f8 },
	{ "parenrightex", 0xf8f7 },
	{ "parenrightinferior", 0x208e },
	{ "parenrightmonospace", 0xff09 },
	{ "parenrightsmall", 0xfe5a },
	{ "parenrightsuperior", 0x207e },
	{ "parenrighttp", 0xf8f6 },
	{ "parenrightvertical", 0xfe36 },
	{ "paseqhebrew", 0x05c0 },
	{ "pashtahebrew", 0x0599 },
	{ "pasquare", 0x33a9 },
	{ "patah", 0x05b7 },
	{ "patah11", 0x05b7 },
	{ "patah1d", 0x05b7 },
	{ "patah2a", 0x05b7 },
	{ "patahhebrew", 0x05b7 },
	{ "patahnarrowhebrew", 0x05b7 },
	{ "patahquarterhebrew", 0x05b7 },
	{ "patahwidehebrew", 0x05b7 },
	{ "pazerhebrew", 0x05a1 },
	{ "pbopomofo", 0x3106 },
	{ "pcircle", 0x24df },
	{ "pdotaccent", 0x1e57 },
	{ "pe", 0x05e4 },
	{ "pecyrillic", 0x043f },
	{ "pedagesh", 0xfb44 },
	{ "pedageshhebrew", 0xfb44 },
	{ "peezisquare", 0x333b },
	{ "pefinaldageshhebrew", 0xfb43 },
	{ "peharabic", 0x067e },
	{ "peharmenian", 0x057a },
	{ "pehebrew", 0x05e4 },
	{ "pehfinalarabic", 0xfb57 },
	{ "pehinitialarabic", 0xfb58 },
	{ "pehiragana", 0x307a },
	{ "pehmedialarabic", 0xfb59 },
	{ "pekatakana", 0x30da },
	{ "pemiddlehookcyrillic", 0x04a7 },
	{ "perafehebrew", 0xfb4e },
	{ "percentarabic", 0x066a },
	{ "percentmonospace", 0xff05 },
	{ "percentsmall", 0xfe6a },
	{ "periodarmenian", 0x0589 },
	{ "periodhalfwidth", 0xff61 },
	{ "periodinferior", 0xf6e7 },
	{ "periodmonospace", 0xff0e },
	{ "periodsmall", 0xfe52 },
	{ "periodsuperior", 0xf6e8 },
	{ "perispomenigreekcmb", 0x0342 },
	{ "pfsquare", 0x338a },
	{ "phabengali", 0x09ab },
	{ "phadeva", 0x092b },
	{ "phagujarati", 0x0aab },
	{ "phagurmukhi", 0x0a2b },
	{ "phieuphacirclekorean", 0x327a },
	{ "phieuphaparenkorean", 0x321a },
	{ "phieuphcirclekorean", 0x326c },
	{ "phieuphkorean", 0x314d },
	{ "phieuphparenkorean", 0x320c },
	{ "philatin", 0x0278 },
	{ "phinthuthai", 0x0e3a },
	{ "phisymbolgreek", 0x03d5 },
	{ "phook", 0x01a5 },
	{ "phophanthai", 0x0e1e },
	{ "phophungthai", 0x0e1c },
	{ "phosamphaothai", 0x0e20 },
	{ "pieupacirclekorean", 0x3273 },
	{ "pieupaparenkorean", 0x3213 },
	{ "pieupcieuckorean", 0x3176 },
	{ "pieupcirclekorean", 0x3265 },
	{ "pieupkiyeokkorean", 0x3172 },
	{ "pieupkorean", 0x3142 },
	{ "pieupparenkorean", 0x3205 },
	{ "pieupsioskiyeokkorean", 0x3174 },
	{ "pieupsioskorean", 0x3144 },
	{ "pieupsiostikeutkorean", 0x3175 },
	{ "pieupthieuthkorean", 0x3177 },
	{ "pieuptikeutkorean", 0x3173 },
	{ "pihiragana", 0x3074 },
	{ "pikatakana", 0x30d4 },
	{ "pisymbolgreek", 0x03d6 },
	{ "piwrarmenian", 0x0583 },
	{ "plusbelowcmb", 0x031f },
	{ "pluscircle", 0x2295 },
	{ "plusmod", 0x02d6 },
	{ "plusmonospace", 0xff0b },
	{ "plussmall", 0xfe62 },
	{ "plussuperior", 0x207a },
	{ "pmonospace", 0xff50 },
	{ "pmsquare", 0x33d8 },
	{ "pohiragana", 0x307d },
	{ "pointingindexdownwhite", 0x261f },
	{ "pointingindexleftwhite", 0x261c },
	{ "pointingindexrightwhite", 0x261e },
	{ "pointingindexupwhite", 0x261d },
	{ "pokatakana", 0x30dd },
	{ "poplathai", 0x0e1b },
	{ "postalmark", 0x3012 },
	{ "postalmarkface", 0x3020 },
	{ "pparen", 0x24ab },
	{ "precedes", 0x227a },
	{ "primemod", 0x02b9 },
	{ "primereversed", 0x2035 },
	{ "projective", 0x2305 },
	{ "prolongedkana", 0x30fc },
	{ "propellor", 0x2318 },
	{ "proportion", 0x2237 },
	{ "psicyrillic", 0x0471 },
	{ "psilipneumatacyrilliccmb", 0x0486 },
	{ "pssquare", 0x33b0 },
	{ "puhiragana", 0x3077 },
	{ "pukatakana", 0x30d7 },
	{ "pvsquare", 0x33b4 },
	{ "pwsquare", 0x33ba },
	{ "qadeva", 0x0958 },
	{ "qadmahebrew", 0x05a8 },
	{ "qafarabic", 0x0642 },
	{ "qaffinalarabic", 0xfed6 },
	{ "qafinitialarabic", 0xfed7 },
	{ "qafmedialarabic", 0xfed8 },
	{ "qamats", 0x05b8 },
	{ "qamats10", 0x05b8 },
	{ "qamats1a", 0x05b8 },
	{ "qamats1c", 0x05b8 },
	{ "qamats27", 0x05b8 },
	{ "qamats29", 0x05b8 },
	{ "qamats33", 0x05b8 },
	{ "qamatsde", 0x05b8 },
	{ "qamatshebrew", 0x05b8 },
	{ "qamatsnarrowhebrew", 0x05b8 },
	{ "qamatsqatanhebrew", 0x05b8 },
	{ "qamatsqatannarrowhebrew", 0x05b8 },
	{ "qamatsqatanquarterhebrew", 0x05b8 },
	{ "qamatsqatanwidehebrew", 0x05b8 },
	{ "qamatsquarterhebrew", 0x05b8 },
	{ "qamatswidehebrew", 0x05b8 },
	{ "qarneyparahebrew", 0x059f },
	{ "qbopomofo", 0x3111 },
	{ "qcircle", 0x24e0 },
	{ "qhook", 0x02a0 },
	{ "qmonospace", 0xff51 },
	{ "qof", 0x05e7 },
	{ "qofdagesh", 0xfb47 },
	{ "qofdageshhebrew", 0xfb47 },
	{ "qofhatafpatah", 0x05e7 },
	{ "qofhatafpatahhebrew", 0x05e7 },
	{ "qofhatafsegol", 0x05e7 },
	{ "qofhatafsegolhebrew", 0x05e7 },
	{ "qofhebrew", 0x05e7 },
	{ "qofhiriq", 0x05e7 },
	{ "qofhiriqhebrew", 0x05e7 },
	{ "qofholam", 0x05e7 },
	{ "qofholamhebrew", 0x05e7 },
	{ "qofpatah", 0x05e7 },
	{ "qofpatahhebrew", 0x05e7 },
	{ "qofqamats", 0x05e7 },
	{ "qofqamatshebrew", 0x05e7 },
	{ "qofqubuts", 0x05e7 },
	{ "qofqubutshebrew", 0x05e7 },
	{ "qofsegol", 0x05e7 },
	{ "qofsegolhebrew", 0x05e7 },
	{ "qofsheva", 0x05e7 },
	{ "qofshevahebrew", 0x05e7 },
	{ "qoftsere", 0x05e7 },
	{ "qoftserehebrew", 0x05e7 },
	{ "qparen", 0x24ac },
	{ "quarternote", 0x2669 },
	{ "qubuts", 0x05bb },
	{ "qubuts18", 0x05bb },
	{ "qubuts25", 0x05bb },
	{ "qubuts31", 0x05bb },
	{ "qubutshebrew", 0x05bb },
	{ "qubutsnarrowhebrew", 0x05bb },
	{ "qubutsquarterhebrew", 0x05bb },
	{ "qubutswidehebrew", 0x05bb },
	{ "questionarabic", 0x061f },
	{ "questionarmenian", 0x055e },
	{ "questiondownsmall", 0xf7bf },
	{ "questiongreek", 0x037e },
	{ "questionmonospace", 0xff1f },
	{ "questionsmall", 0xf73f },
	{ "quotedblmonospace", 0xff02 },
	{ "quotedblprime", 0x301e },
	{ "quotedblprimereversed", 0x301d },
	{ "quoteleftreversed", 0x201b },
	{ "quoterightn", 0x0149 },
	{ "quotesinglemonospace", 0xff07 },
	{ "raarmenian", 0x057c },
	{ "rabengali", 0x09b0 },
	{ "radeva", 0x0930 },
	{ "radicalex", 0xf8e5 },
	{ "radoverssquare", 0x33ae },
	{ "radoverssquaredsquare", 0x33af },
	{ "radsquare", 0x33ad },
	{ "rafe", 0x05bf },
	{ "rafehebrew", 0x05bf },
	{ "ragujarati", 0x0ab0 },
	{ "ragurmukhi", 0x0a30 },
	{ "rahiragana", 0x3089 },
	{ "rakatakana", 0x30e9 },
	{ "rakatakanahalfwidth", 0xff97 },
	{ "ralowerdiagonalbengali", 0x09f1 },
	{ "ramiddlediagonalbengali", 0x09f0 },
	{ "ramshorn", 0x0264 },
	{ "ratio", 0x2236 },
	{ "rbopomofo", 0x3116 },
	{ "rcedilla", 0x0157 },
	{ "rcircle", 0x24e1 },
	{ "rdblgrave", 0x0211 },
	{ "rdotaccent", 0x1e59 },
	{ "rdotbelow", 0x1e5b },
	{ "rdotbelowmacron", 0x1e5d },
	{ "referencemark", 0x203b },
	{ "registersans", 0xf8e8 },
	{ "registerserif", 0xf6da },
	{ "reharabic", 0x0631 },
	{ "reharmenian", 0x0580 },
	{ "rehfinalarabic", 0xfeae },
	{ "rehiragana", 0x308c },
	{ "rehyehaleflamarabic", 0x0631 },
	{ "rekatakana", 0x30ec },
	{ "rekatakanahalfwidth", 0xff9a },
	{ "resh", 0x05e8 },
	{ "reshdageshhebrew", 0xfb48 },
	{ "reshhatafpatah", 0x05e8 },
	{ "reshhatafpatahhebrew", 0x05e8 },
	{ "reshhatafsegol", 0x05e8 },
	{ "reshhatafsegolhebrew", 0x05e8 },
	{ "reshhebrew", 0x05e8 },
	{ "reshhiriq", 0x05e8 },
	{ "reshhiriqhebrew", 0x05e8 },
	{ "reshholam", 0x05e8 },
	{ "reshholamhebrew", 0x05e8 },
	{ "reshpatah", 0x05e8 },
	{ "reshpatahhebrew", 0x05e8 },
	{ "reshqamats", 0x05e8 },
	{ "reshqamatshebrew", 0x05e8 },
	{ "reshqubuts", 0x05e8 },
	{ "reshqubutshebrew", 0x05e8 },
	{ "reshsegol", 0x05e8 },
	{ "reshsegolhebrew", 0x05e8 },
	{ "reshsheva", 0x05e8 },
	{ "reshshevahebrew", 0x05e8 },
	{ "reshtsere", 0x05e8 },
	{ "reshtserehebrew", 0x05e8 },
	{ "reversedtilde", 0x223d },
	{ "reviahebrew", 0x0597 },
	{ "reviamugrashhebrew", 0x0597 },
	{ "rfishhook", 0x027e },
	{ "rfishhookreversed", 0x027f },
	{ "rhabengali", 0x09dd },
	{ "rhadeva", 0x095d },
	{ "rhook", 0x027d },
	{ "rhookturned", 0x027b },
	{ "rhookturnedsuperior", 0x02b5 },
	{ "rhosymbolgreek", 0x03f1 },
	{ "rhotichookmod", 0x02de },
	{ "rieulacirclekorean", 0x3271 },
	{ "rieulaparenkorean", 0x3211 },
	{ "rieulcirclekorean", 0x3263 },
	{ "rieulhieuhkorean", 0x3140 },
	{ "rieulkiyeokkorean", 0x313a },
	{ "rieulkiyeoksioskorean", 0x3169 },
	{ "rieulkorean", 0x3139 },
	{ "rieulmieumkorean", 0x313b },
	{ "rieulpansioskorean", 0x316c },
	{ "rieulparenkorean", 0x3203 },
	{ "rieulphieuphkorean", 0x313f },
	{ "rieulpieupkorean", 0x313c },
	{ "rieulpieupsioskorean", 0x316b },
	{ "rieulsioskorean", 0x313d },
	{ "rieulthieuthkorean", 0x313e },
	{ "rieultikeutkorean", 0x316a },
	{ "rieulyeorinhieuhkorean", 0x316d },
	{ "rightangle", 0x221f },
	{ "righttackbelowcmb", 0x0319 },
	{ "righttriangle", 0x22bf },
	{ "rihiragana", 0x308a },
	{ "rikatakana", 0x30ea },
	{ "rikatakanahalfwidth", 0xff98 },
	{ "ringbelowcmb", 0x0325 },
	{ "ringcmb", 0x030a },
	{ "ringhalfleft", 0x02bf },
	{ "ringhalfleftarmenian", 0x0559 },
	{ "ringhalfleftbelowcmb", 0x031c },
	{ "ringhalfleftcentered", 0x02d3 },
	{ "ringhalfright", 0x02be },
	{ "ringhalfrightbelowcmb", 0x0339 },
	{ "ringhalfrightcentered", 0x02d2 },
	{ "rinvertedbreve", 0x0213 },
	{ "rittorusquare", 0x3351 },
	{ "rlinebelow", 0x1e5f },
	{ "rlongleg", 0x027c },
	{ "rlonglegturned", 0x027a },
	{ "rmonospace", 0xff52 },
	{ "rohiragana", 0x308d },
	{ "rokatakana", 0x30ed },
	{ "rokatakanahalfwidth", 0xff9b },
	{ "roruathai", 0x0e23 },
	{ "rparen", 0x24ad },
	{ "rrabengali", 0x09dc },
	{ "rradeva", 0x0931 },
	{ "rragurmukhi", 0x0a5c },
	{ "rreharabic", 0x0691 },
	{ "rrehfinalarabic", 0xfb8d },
	{ "rrvocalicbengali", 0x09e0 },
	{ "rrvocalicdeva", 0x0960 },
	{ "rrvocalicgujarati", 0x0ae0 },
	{ "rrvocalicvowelsignbengali", 0x09c4 },
	{ "rrvocalicvowelsigndeva", 0x0944 },
	{ "rrvocalicvowelsigngujarati", 0x0ac4 },
	{ "rsuperior", 0xf6f1 },
	{ "rturned", 0x0279 },
	{ "rturnedsuperior", 0x02b4 },
	{ "ruhiragana", 0x308b },
	{ "rukatakana", 0x30eb },
	{ "rukatakanahalfwidth", 0xff99 },
	{ "rupeemarkbengali", 0x09f2 },
	{ "rupeesignbengali", 0x09f3 },
	{ "rupiah", 0xf6dd },
	{ "ruthai", 0x0e24 },
	{ "rvocalicbengali", 0x098b },
	{ "rvocalicdeva", 0x090b },
	{ "rvocalicgujarati", 0x0a8b },
	{ "rvocalicvowelsignbengali", 0x09c3 },
	{ "rvocalicvowelsigndeva", 0x0943 },
	{ "rvocalicvowelsigngujarati", 0x0ac3 },
	{ "sabengali", 0x09b8 },
	{ "sacutedotaccent", 0x1e65 },
	{ "sadarabic", 0x0635 },
	{ "sadeva", 0x0938 },
	{ "sadfinalarabic", 0xfeba },
	{ "sadinitialarabic", 0xfebb },
	{ "sadmedialarabic", 0xfebc },
	{ "sagujarati", 0x0ab8 },
	{ "sagurmukhi", 0x0a38 },
	{ "sahiragana", 0x3055 },
	{ "sakatakana", 0x30b5 },
	{ "sakatakanahalfwidth", 0xff7b },
	{ "sallallahoualayhewasallamarabic", 0xfdfa },
	{ "samekh", 0x05e1 },
	{ "samekhdagesh", 0xfb41 },
	{ "samekhdageshhebrew", 0xfb41 },
	{ "samekhhebrew", 0x05e1 },
	{ "saraaathai", 0x0e32 },
	{ "saraaethai", 0x0e41 },
	{ "saraaimaimalaithai", 0x0e44 },
	{ "saraaimaimuanthai", 0x0e43 },
	{ "saraamthai", 0x0e33 },
	{ "saraathai", 0x0e30 },
	{ "saraethai", 0x0e40 },
	{ "saraiileftthai", 0xf886 },
	{ "saraiithai", 0x0e35 },
	{ "saraileftthai", 0xf885 },
	{ "saraithai", 0x0e34 },
	{ "saraothai", 0x0e42 },
	{ "saraueeleftthai", 0xf888 },
	{ "saraueethai", 0x0e37 },
	{ "saraueleftthai", 0xf887 },
	{ "sarauethai", 0x0e36 },
	{ "sarauthai", 0x0e38 },
	{ "sarauuthai", 0x0e39 },
	{ "sbopomofo", 0x3119 },
	{ "scarondotaccent", 0x1e67 },
	{ "schwa", 0x0259 },
	{ "schwacyrillic", 0x04d9 },
	{ "schwadieresiscyrillic", 0x04db },
	{ "schwahook", 0x025a },
	{ "scircle", 0x24e2 },
	{ "sdotaccent", 0x1e61 },
	{ "sdotbelow", 0x1e63 },
	{ "sdotbelowdotaccent", 0x1e69 },
	{ "seagullbelowcmb", 0x033c },
	{ "secondtonechinese", 0x02ca },
	{ "seenarabic", 0x0633 },
	{ "seenfinalarabic", 0xfeb2 },
	{ "seeninitialarabic", 0xfeb3 },
	{ "seenmedialarabic", 0xfeb4 },
	{ "segol", 0x05b6 },
	{ "segol13", 0x05b6 },
	{ "segol1f", 0x05b6 },
	{ "segol2c", 0x05b6 },
	{ "segolhebrew", 0x05b6 },
	{ "segolnarrowhebrew", 0x05b6 },
	{ "segolquarterhebrew", 0x05b6 },
	{ "segoltahebrew", 0x0592 },
	{ "segolwidehebrew", 0x05b6 },
	{ "seharmenian", 0x057d },
	{ "sehiragana", 0x305b },
	{ "sekatakana", 0x30bb },
	{ "sekatakanahalfwidth", 0xff7e },
	{ "semicolonarabic", 0x061b },
	{ "semicolonmonospace", 0xff1b },
	{ "semicolonsmall", 0xfe54 },
	{ "semivoicedmarkkana", 0x309c },
	{ "semivoicedmarkkanahalfwidth", 0xff9f },
	{ "sentisquare", 0x3322 },
	{ "sentosquare", 0x3323 },
	{ "sevenarabic", 0x0667 },
	{ "sevenbengali", 0x09ed },
	{ "sevencircle", 0x2466 },
	{ "sevencircleinversesansserif", 0x2790 },
	{ "sevendeva", 0x096d },
	{ "sevengujarati", 0x0aed },
	{ "sevengurmukhi", 0x0a6d },
	{ "sevenhackarabic", 0x0667 },
	{ "sevenhangzhou", 0x3027 },
	{ "sevenideographicparen", 0x3226 },
	{ "seveninferior", 0x2087 },
	{ "sevenmonospace", 0xff17 },
	{ "sevenoldstyle", 0xf737 },
	{ "sevenparen", 0x247a },
	{ "sevenperiod", 0x248e },
	{ "sevenpersian", 0x06f7 },
	{ "sevenroman", 0x2176 },
	{ "sevensuperior", 0x2077 },
	{ "seventeencircle", 0x2470 },
	{ "seventeenparen", 0x2484 },
	{ "seventeenperiod", 0x2498 },
	{ "seventhai", 0x0e57 },
	{ "sfthyphen", 0x00ad },
	{ "shaarmenian", 0x0577 },
	{ "shabengali", 0x09b6 },
	{ "shacyrillic", 0x0448 },
	{ "shaddaarabic", 0x0651 },
	{ "shaddadammaarabic", 0xfc61 },
	{ "shaddadammatanarabic", 0xfc5e },
	{ "shaddafathaarabic", 0xfc60 },
	{ "shaddafathatanarabic", 0x0651 },
	{ "shaddakasraarabic", 0xfc62 },
	{ "shaddakasratanarabic", 0xfc5f },
	{ "shadedark", 0x2593 },
	{ "shadelight", 0x2591 },
	{ "shademedium", 0x2592 },
	{ "shadeva", 0x0936 },
	{ "shagujarati", 0x0ab6 },
	{ "shagurmukhi", 0x0a36 },
	{ "shalshelethebrew", 0x0593 },
	{ "shbopomofo", 0x3115 },
	{ "shchacyrillic", 0x0449 },
	{ "sheenarabic", 0x0634 },
	{ "sheenfinalarabic", 0xfeb6 },
	{ "sheeninitialarabic", 0xfeb7 },
	{ "sheenmedialarabic", 0xfeb8 },
	{ "sheicoptic", 0x03e3 },
	{ "sheqel", 0x20aa },
	{ "sheqelhebrew", 0x20aa },
	{ "sheva", 0x05b0 },
	{ "sheva115", 0x05b0 },
	{ "sheva15", 0x05b0 },
	{ "sheva22", 0x05b0 },
	{ "sheva2e", 0x05b0 },
	{ "shevahebrew", 0x05b0 },
	{ "shevanarrowhebrew", 0x05b0 },
	{ "shevaquarterhebrew", 0x05b0 },
	{ "shevawidehebrew", 0x05b0 },
	{ "shhacyrillic", 0x04bb },
	{ "shimacoptic", 0x03ed },
	{ "shin", 0x05e9 },
	{ "shindagesh", 0xfb49 },
	{ "shindageshhebrew", 0xfb49 },
	{ "shindageshshindot", 0xfb2c },
	{ "shindageshshindothebrew", 0xfb2c },
	{ "shindageshsindot", 0xfb2d },
	{ "shindageshsindothebrew", 0xfb2d },
	{ "shindothebrew", 0x05c1 },
	{ "shinhebrew", 0x05e9 },
	{ "shinshindot", 0xfb2a },
	{ "shinshindothebrew", 0xfb2a },
	{ "shinsindot", 0xfb2b },
	{ "shinsindothebrew", 0xfb2b },
	{ "shook", 0x0282 },
	{ "sigmafinal", 0x03c2 },
	{ "sigmalunatesymbolgreek", 0x03f2 },
	{ "sihiragana", 0x3057 },
	{ "sikatakana", 0x30b7 },
	{ "sikatakanahalfwidth", 0xff7c },
	{ "siluqhebrew", 0x05bd },
	{ "siluqlefthebrew", 0x05bd },
	{ "sindothebrew", 0x05c2 },
	{ "siosacirclekorean", 0x3274 },
	{ "siosaparenkorean", 0x3214 },
	{ "sioscieuckorean", 0x317e },
	{ "sioscirclekorean", 0x3266 },
	{ "sioskiyeokkorean", 0x317a },
	{ "sioskorean", 0x3145 },
	{ "siosnieunkorean", 0x317b },
	{ "siosparenkorean", 0x3206 },
	{ "siospieupkorean", 0x317d },
	{ "siostikeutkorean", 0x317c },
	{ "sixarabic", 0x0666 },
	{ "sixbengali", 0x09ec },
	{ "sixcircle", 0x2465 },
	{ "sixcircleinversesansserif", 0x278f },
	{ "sixdeva", 0x096c },
	{ "sixgujarati", 0x0aec },
	{ "sixgurmukhi", 0x0a6c },
	{ "sixhackarabic", 0x0666 },
	{ "sixhangzhou", 0x3026 },
	{ "sixideographicparen", 0x3225 },
	{ "sixinferior", 0x2086 },
	{ "sixmonospace", 0xff16 },
	{ "sixoldstyle", 0xf736 },
	{ "sixparen", 0x2479 },
	{ "sixperiod", 0x248d },
	{ "sixpersian", 0x06f6 },
	{ "sixroman", 0x2175 },
	{ "sixsuperior", 0x2076 },
	{ "sixteencircle", 0x246f },
	{ "sixteencurrencydenominatorbengali", 0x09f9 },
	{ "sixteenparen", 0x2483 },
	{ "sixteenperiod", 0x2497 },
	{ "sixthai", 0x0e56 },
	{ "slashmonospace", 0xff0f },
	{ "slong", 0x017f },
	{ "slongdotaccent", 0x1e9b },
	{ "smonospace", 0xff53 },
	{ "sofpasuqhebrew", 0x05c3 },
	{ "softhyphen", 0x00ad },
	{ "softsigncyrillic", 0x044c },
	{ "sohiragana", 0x305d },
	{ "sokatakana", 0x30bd },
	{ "sokatakanahalfwidth", 0xff7f },
	{ "soliduslongoverlaycmb", 0x0338 },
	{ "solidusshortoverlaycmb", 0x0337 },
	{ "sorusithai", 0x0e29 },
	{ "sosalathai", 0x0e28 },
	{ "sosothai", 0x0e0b },
	{ "sosuathai", 0x0e2a },
	{ "spacehackarabic", 0x0020 },
	{ "spadesuitblack", 0x2660 },
	{ "spadesuitwhite", 0x2664 },
	{ "sparen", 0x24ae },
	{ "squarebelowcmb", 0x033b },
	{ "squarecc", 0x33c4 },
	{ "squarecm", 0x339d },
	{ "squarediagonalcrosshatchfill", 0x25a9 },
	{ "squarehorizontalfill", 0x25a4 },
	{ "squarekg", 0x338f },
	{ "squarekm", 0x339e },
	{ "squarekmcapital", 0x33ce },
	{ "squareln", 0x33d1 },
	{ "squarelog", 0x33d2 },
	{ "squaremg", 0x338e },
	{ "squaremil", 0x33d5 },
	{ "squaremm", 0x339c },
	{ "squaremsquared", 0x33a1 },
	{ "squareorthogonalcrosshatchfill", 0x25a6 },
	{ "squareupperlefttolowerrightfill", 0x25a7 },
	{ "squareupperrighttolowerleftfill", 0x25a8 },
	{ "squareverticalfill", 0x25a5 },
	{ "squarewhitewithsmallblack", 0x25a3 },
	{ "srsquare", 0x33db },
	{ "ssabengali", 0x09b7 },
	{ "ssadeva", 0x0937 },
	{ "ssagujarati", 0x0ab7 },
	{ "ssangcieuckorean", 0x3149 },
	{ "ssanghieuhkorean", 0x3185 },
	{ "ssangieungkorean", 0x3180 },
	{ "ssangkiyeokkorean", 0x3132 },
	{ "ssangnieunkorean", 0x3165 },
	{ "ssangpieupkorean", 0x3143 },
	{ "ssangsioskorean", 0x3146 },
	{ "ssangtikeutkorean", 0x3138 },
	{ "ssuperior", 0xf6f2 },
	{ "sterlingmonospace", 0xffe1 },
	{ "strokelongoverlaycmb", 0x0336 },
	{ "strokeshortoverlaycmb", 0x0335 },
	{ "subset", 0x2282 },
	{ "subsetnotequal", 0x228a },
	{ "subsetorequal", 0x2286 },
	{ "succeeds", 0x227b },
	{ "suhiragana", 0x3059 },
	{ "sukatakana", 0x30b9 },
	{ "sukatakanahalfwidth", 0xff7d },
	{ "sukunarabic", 0x0652 },
	{ "superset", 0x2283 },
	{ "supersetnotequal", 0x228b },
	{ "supersetorequal", 0x2287 },
	{ "svsquare", 0x33dc },
	{ "syouwaerasquare", 0x337c },
	{ "tabengali", 0x09a4 },
	{ "tackdown", 0x22a4 },
	{ "tackleft", 0x22a3 },
	{ "tadeva", 0x0924 },
	{ "tagujarati", 0x0aa4 },
	{ "tagurmukhi", 0x0a24 },
	{ "taharabic", 0x0637 },
	{ "tahfinalarabic", 0xfec2 },
	{ "tahinitialarabic", 0xfec3 },
	{ "tahiragana", 0x305f },
	{ "tahmedialarabic", 0xfec4 },
	{ "taisyouerasquare", 0x337d },
	{ "takatakana", 0x30bf },
	{ "takatakanahalfwidth", 0xff80 },
	{ "tatweelarabic", 0x0640 },
	{ "tav", 0x05ea },
	{ "tavdages", 0xfb4a },
	{ "tavdagesh", 0xfb4a },
	{ "tavdageshhebrew", 0xfb4a },
	{ "tavhebrew", 0x05ea },
	{ "tbopomofo", 0x310a },
	{ "tccurl", 0x02a8 },
	{ "tcedilla", 0x0163 },
	{ "tcheharabic", 0x0686 },
	{ "tchehfinalarabic", 0xfb7b },
	{ "tchehinitialarabic", 0xfb7c },
	{ "tchehmedialarabic", 0xfb7d },
	{ "tchehmeeminitialarabic", 0xfb7c },
	{ "tcircle", 0x24e3 },
	{ "tcircumflexbelow", 0x1e71 },
	{ "tdieresis", 0x1e97 },
	{ "tdotaccent", 0x1e6b },
	{ "tdotbelow", 0x1e6d },
	{ "tecyrillic", 0x0442 },
	{ "tedescendercyrillic", 0x04ad },
	{ "teharabic", 0x062a },
	{ "tehfinalarabic", 0xfe96 },
	{ "tehhahinitialarabic", 0xfca2 },
	{ "tehhahisolatedarabic", 0xfc0c },
	{ "tehinitialarabic", 0xfe97 },
	{ "tehiragana", 0x3066 },
	{ "tehjeeminitialarabic", 0xfca1 },
	{ "tehjeemisolatedarabic", 0xfc0b },
	{ "tehmarbutaarabic", 0x0629 },
	{ "tehmarbutafinalarabic", 0xfe94 },
	{ "tehmedialarabic", 0xfe98 },
	{ "tehmeeminitialarabic", 0xfca4 },
	{ "tehmeemisolatedarabic", 0xfc0e },
	{ "tehnoonfinalarabic", 0xfc73 },
	{ "tekatakana", 0x30c6 },
	{ "tekatakanahalfwidth", 0xff83 },
	{ "telephone", 0x2121 },
	{ "telephoneblack", 0x260e },
	{ "telishagedolahebrew", 0x05a0 },
	{ "telishaqetanahebrew", 0x05a9 },
	{ "tencircle", 0x2469 },
	{ "tenideographicparen", 0x3229 },
	{ "tenparen", 0x247d },
	{ "tenperiod", 0x2491 },
	{ "tenroman", 0x2179 },
	{ "tesh", 0x02a7 },
	{ "tet", 0x05d8 },
	{ "tetdagesh", 0xfb38 },
	{ "tetdageshhebrew", 0xfb38 },
	{ "tethebrew", 0x05d8 },
	{ "tetsecyrillic", 0x04b5 },
	{ "tevirhebrew", 0x059b },
	{ "tevirlefthebrew", 0x059b },
	{ "thabengali", 0x09a5 },
	{ "thadeva", 0x0925 },
	{ "thagujarati", 0x0aa5 },
	{ "thagurmukhi", 0x0a25 },
	{ "thalarabic", 0x0630 },
	{ "thalfinalarabic", 0xfeac },
	{ "thanthakhatlowleftthai", 0xf898 },
	{ "thanthakhatlowrightthai", 0xf897 },
	{ "thanthakhatthai", 0x0e4c },
	{ "thanthakhatupperleftthai", 0xf896 },
	{ "theharabic", 0x062b },
	{ "thehfinalarabic", 0xfe9a },
	{ "thehinitialarabic", 0xfe9b },
	{ "thehmedialarabic", 0xfe9c },
	{ "thereexists", 0x2203 },
	{ "thetasymbolgreek", 0x03d1 },
	{ "thieuthacirclekorean", 0x3279 },
	{ "thieuthaparenkorean", 0x3219 },
	{ "thieuthcirclekorean", 0x326b },
	{ "thieuthkorean", 0x314c },
	{ "thieuthparenkorean", 0x320b },
	{ "thirteencircle", 0x246c },
	{ "thirteenparen", 0x2480 },
	{ "thirteenperiod", 0x2494 },
	{ "thonangmonthothai", 0x0e11 },
	{ "thook", 0x01ad },
	{ "thophuthaothai", 0x0e12 },
	{ "thothahanthai", 0x0e17 },
	{ "thothanthai", 0x0e10 },
	{ "thothongthai", 0x0e18 },
	{ "thothungthai", 0x0e16 },
	{ "thousandcyrillic", 0x0482 },
	{ "thousandsseparatorarabic", 0x066c },
	{ "thousandsseparatorpersian", 0x066c },
	{ "threearabic", 0x0663 },
	{ "threebengali", 0x09e9 },
	{ "threecircle", 0x2462 },
	{ "threecircleinversesansserif", 0x278c },
	{ "threedeva", 0x0969 },
	{ "threegujarati", 0x0ae9 },
	{ "threegurmukhi", 0x0a69 },
	{ "threehackarabic", 0x0663 },
	{ "threehangzhou", 0x3023 },
	{ "threeideographicparen", 0x3222 },
	{ "threeinferior", 0x2083 },
	{ "threemonospace", 0xff13 },
	{ "threenumeratorbengali", 0x09f6 },
	{ "threeoldstyle", 0xf733 },
	{ "threeparen", 0x2476 },
	{ "threeperiod", 0x248a },
	{ "threepersian", 0x06f3 },
	{ "threequartersemdash", 0xf6de },
	{ "threeroman", 0x2172 },
	{ "threesuperior", 0x00b3 },
	{ "threethai", 0x0e53 },
	{ "thzsquare", 0x3394 },
	{ "tihiragana", 0x3061 },
	{ "tikatakana", 0x30c1 },
	{ "tikatakanahalfwidth", 0xff81 },
	{ "tikeutacirclekorean", 0x3270 },
	{ "tikeutaparenkorean", 0x3210 },
	{ "tikeutcirclekorean", 0x3262 },
	{ "tikeutkorean", 0x3137 },
	{ "tikeutparenkorean", 0x3202 },
	{ "tildebelowcmb", 0x0330 },
	{ "tildecmb", 0x0303 },
	{ "tildedoublecmb", 0x0360 },
	{ "tildeoperator", 0x223c },
	{ "tildeoverlaycmb", 0x0334 },
	{ "tildeverticalcmb", 0x033e },
	{ "timescircle", 0x2297 },
	{ "tipehahebrew", 0x0596 },
	{ "tipehalefthebrew", 0x0596 },
	{ "tippigurmukhi", 0x0a70 },
	{ "titlocyrilliccmb", 0x0483 },
	{ "tiwnarmenian", 0x057f },
	{ "tlinebelow", 0x1e6f },
	{ "tmonospace", 0xff54 },
	{ "toarmenian", 0x0569 },
	{ "tohiragana", 0x3068 },
	{ "tokatakana", 0x30c8 },
	{ "tokatakanahalfwidth", 0xff84 },
	{ "tonebarextrahighmod", 0x02e5 },
	{ "tonebarextralowmod", 0x02e9 },
	{ "tonebarhighmod", 0x02e6 },
	{ "tonebarlowmod", 0x02e8 },
	{ "tonebarmidmod", 0x02e7 },
	{ "tonefive", 0x01bd },
	{ "tonesix", 0x0185 },
	{ "tonetwo", 0x01a8 },
	{ "tonsquare", 0x3327 },
	{ "topatakthai", 0x0e0f },
	{ "tortoiseshellbracketleft", 0x3014 },
	{ "tortoiseshellbracketleftsmall", 0xfe5d },
	{ "tortoiseshellbracketleftvertical", 0xfe39 },
	{ "tortoiseshellbracketright", 0x3015 },
	{ "tortoiseshellbracketrightsmall", 0xfe5e },
	{ "tortoiseshellbracketrightvertical", 0xfe3a },
	{ "totaothai", 0x0e15 },
	{ "tpalatalhook", 0x01ab },
	{ "tparen", 0x24af },
	{ "trademarksans", 0xf8ea },
	{ "trademarkserif", 0xf6db },
	{ "tretroflexhook", 0x0288 },
	{ "ts", 0x02a6 },
	{ "tsadi", 0x05e6 },
	{ "tsadidagesh", 0xfb46 },
	{ "tsadidageshhebrew", 0xfb46 },
	{ "tsadihebrew", 0x05e6 },
	{ "tsecyrillic", 0x0446 },
	{ "tsere", 0x05b5 },
	{ "tsere12", 0x05b5 },
	{ "tsere1e", 0x05b5 },
	{ "tsere2b", 0x05b5 },
	{ "tserehebrew", 0x05b5 },
	{ "tserenarrowhebrew", 0x05b5 },
	{ "tserequarterhebrew", 0x05b5 },
	{ "tserewidehebrew", 0x05b5 },
	{ "tshecyrillic", 0x045b },
	{ "tsuperior", 0xf6f3 },
	{ "ttabengali", 0x099f },
	{ "ttadeva", 0x091f },
	{ "ttagujarati", 0x0a9f },
	{ "ttagurmukhi", 0x0a1f },
	{ "tteharabic", 0x0679 },
	{ "ttehfinalarabic", 0xfb67 },
	{ "ttehinitialarabic", 0xfb68 },
	{ "ttehmedialarabic", 0xfb69 },
	{ "tthabengali", 0x09a0 },
	{ "tthadeva", 0x0920 },
	{ "tthagujarati", 0x0aa0 },
	{ "tthagurmukhi", 0x0a20 },
	{ "tturned", 0x0287 },
	{ "tuhiragana", 0x3064 },
	{ "tukatakana", 0x30c4 },
	{ "tukatakanahalfwidth", 0xff82 },
	{ "tusmallhiragana", 0x3063 },
	{ "tusmallkatakana", 0x30c3 },
	{ "tusmallkatakanahalfwidth", 0xff6f },
	{ "twelvecircle", 0x246b },
	{ "twelveparen", 0x247f },
	{ "twelveperiod", 0x2493 },
	{ "twelveroman", 0x217b },
	{ "twentycircle", 0x2473 },
	{ "twentyhangzhou", 0x5344 },
	{ "twentyparen", 0x2487 },
	{ "twentyperiod", 0x249b },
	{ "twoarabic", 0x0662 },
	{ "twobengali", 0x09e8 },
	{ "twocircle", 0x2461 },
	{ "twocircleinversesansserif", 0x278b },
	{ "twodeva", 0x0968 },
	{ "twodotleader", 0x2025 },
	{ "twodotleadervertical", 0xfe30 },
	{ "twogujarati", 0x0ae8 },
	{ "twogurmukhi", 0x0a68 },
	{ "twohackarabic", 0x0662 },
	{ "twohangzhou", 0x3022 },
	{ "twoideographicparen", 0x3221 },
	{ "twoinferior", 0x2082 },
	{ "twomonospace", 0xff12 },
	{ "twonumeratorbengali", 0x09f5 },
	{ "twooldstyle", 0xf732 },
	{ "twoparen", 0x2475 },
	{ "twoperiod", 0x2489 },
	{ "twopersian", 0x06f2 },
	{ "tworoman", 0x2171 },
	{ "twostroke", 0x01bb },
	{ "twosuperior", 0x00b2 },
	{ "twothai", 0x0e52 },
	{ "ubar", 0x0289 },
	{ "ubengali", 0x0989 },
	{ "ubopomofo", 0x3128 },
	{ "ucaron", 0x01d4 },
	{ "ucircle", 0x24e4 },
	{ "ucircumflexbelow", 0x1e77 },
	{ "ucyrillic", 0x0443 },
	{ "udattadeva", 0x0951 },
	{ "udblacute", 0x0171 },
	{ "udblgrave", 0x0215 },
	{ "udeva", 0x0909 },
	{ "udieresisacute", 0x01d8 },
	{ "udieresisbelow", 0x1e73 },
	{ "udieresiscaron", 0x01da },
	{ "udieresiscyrillic", 0x04f1 },
	{ "udieresisgrave", 0x01dc },
	{ "udieresismacron", 0x01d6 },
	{ "udotbelow", 0x1ee5 },
	{ "ugujarati", 0x0a89 },
	{ "ugurmukhi", 0x0a09 },
	{ "uhiragana", 0x3046 },
	{ "uhookabove", 0x1ee7 },
	{ "uhornacute", 0x1ee9 },
	{ "uhorndotbelow", 0x1ef1 },
	{ "uhorngrave", 0x1eeb },
	{ "uhornhookabove", 0x1eed },
	{ "uhorntilde", 0x1eef },
	{ "uhungarumlautcyrillic", 0x04f3 },
	{ "uinvertedbreve", 0x0217 },
	{ "ukatakana", 0x30a6 },
	{ "ukatakanahalfwidth", 0xff73 },
	{ "ukcyrillic", 0x0479 },
	{ "ukorean", 0x315c },
	{ "umacroncyrillic", 0x04ef },
	{ "umacrondieresis", 0x1e7b },
	{ "umatragurmukhi", 0x0a41 },
	{ "umonospace", 0xff55 },
	{ "underscoremonospace", 0xff3f },
	{ "underscorevertical", 0xfe33 },
	{ "underscorewavy", 0xfe4f },
	{ "uparen", 0x24b0 },
	{ "upperdothebrew", 0x05c4 },
	{ "upsilonlatin", 0x028a },
	{ "uptackbelowcmb", 0x031d },
	{ "uptackmod", 0x02d4 },
	{ "uragurmukhi", 0x0a73 },
	{ "ushortcyrillic", 0x045e },
	{ "usmallhiragana", 0x3045 },
	{ "usmallkatakana", 0x30a5 },
	{ "usmallkatakanahalfwidth", 0xff69 },
	{ "ustraightcyrillic", 0x04af },
	{ "ustraightstrokecyrillic", 0x04b1 },
	{ "utildeacute", 0x1e79 },
	{ "utildebelow", 0x1e75 },
	{ "uubengali", 0x098a },
	{ "uudeva", 0x090a },
	{ "uugujarati", 0x0a8a },
	{ "uugurmukhi", 0x0a0a },
	{ "uumatragurmukhi", 0x0a42 },
	{ "uuvowelsignbengali", 0x09c2 },
	{ "uuvowelsigndeva", 0x0942 },
	{ "uuvowelsigngujarati", 0x0ac2 },
	{ "uvowelsignbengali", 0x09c1 },
	{ "uvowelsigndeva", 0x0941 },
	{ "uvowelsigngujarati", 0x0ac1 },
	{ "vadeva", 0x0935 },
	{ "vagujarati", 0x0ab5 },
	{ "vagurmukhi", 0x0a35 },
	{ "vakatakana", 0x30f7 },
	{ "vav", 0x05d5 },
	{ "vavdagesh", 0xfb35 },
	{ "vavdagesh65", 0xfb35 },
	{ "vavdageshhebrew", 0xfb35 },
	{ "vavhebrew", 0x05d5 },
	{ "vavholam", 0xfb4b },
	{ "vavholamhebrew", 0xfb4b },
	{ "vavvavhebrew", 0x05f0 },
	{ "vavyodhebrew", 0x05f1 },
	{ "vcircle", 0x24e5 },
	{ "vdotbelow", 0x1e7f },
	{ "vecyrillic", 0x0432 },
	{ "veharabic", 0x06a4 },
	{ "vehfinalarabic", 0xfb6b },
	{ "vehinitialarabic", 0xfb6c },
	{ "vehmedialarabic", 0xfb6d },
	{ "vekatakana", 0x30f9 },
	{ "venus", 0x2640 },
	{ "verticalbar", 0x007c },
	{ "verticallineabovecmb", 0x030d },
	{ "verticallinebelowcmb", 0x0329 },
	{ "verticallinelowmod", 0x02cc },
	{ "verticallinemod", 0x02c8 },
	{ "vewarmenian", 0x057e },
	{ "vhook", 0x028b },
	{ "vikatakana", 0x30f8 },
	{ "viramabengali", 0x09cd },
	{ "viramadeva", 0x094d },
	{ "viramagujarati", 0x0acd },
	{ "visargabengali", 0x0983 },
	{ "visargadeva", 0x0903 },
	{ "visargagujarati", 0x0a83 },
	{ "vmonospace", 0xff56 },
	{ "voarmenian", 0x0578 },
	{ "voicediterationhiragana", 0x309e },
	{ "voicediterationkatakana", 0x30fe },
	{ "voicedmarkkana", 0x309b },
	{ "voicedmarkkanahalfwidth", 0xff9e },
	{ "vokatakana", 0x30fa },
	{ "vparen", 0x24b1 },
	{ "vtilde", 0x1e7d },
	{ "vturned", 0x028c },
	{ "vuhiragana", 0x3094 },
	{ "vukatakana", 0x30f4 },
	{ "waekorean", 0x3159 },
	{ "wahiragana", 0x308f },
	{ "wakatakana", 0x30ef },
	{ "wakatakanahalfwidth", 0xff9c },
	{ "wakorean", 0x3158 },
	{ "wasmallhiragana", 0x308e },
	{ "wasmallkatakana", 0x30ee },
	{ "wattosquare", 0x3357 },
	{ "wavedash", 0x301c },
	{ "wavyunderscorevertical", 0xfe34 },
	{ "wawarabic", 0x0648 },
	{ "wawfinalarabic", 0xfeee },
	{ "wawhamzaabovearabic", 0x0624 },
	{ "wawhamzaabovefinalarabic", 0xfe86 },
	{ "wbsquare", 0x33dd },
	{ "wcircle", 0x24e6 },
	{ "wdotaccent", 0x1e87 },
	{ "wdotbelow", 0x1e89 },
	{ "wehiragana", 0x3091 },
	{ "wekatakana", 0x30f1 },
	{ "wekorean", 0x315e },
	{ "weokorean", 0x315d },
	{ "whitebullet", 0x25e6 },
	{ "whitecircle", 0x25cb },
	{ "whitecircleinverse", 0x25d9 },
	{ "whitecornerbracketleft", 0x300e },
	{ "whitecornerbracketleftvertical", 0xfe43 },
	{ "whitecornerbracketright", 0x300f },
	{ "whitecornerbracketrightvertical", 0xfe44 },
	{ "whitediamond", 0x25c7 },
	{ "whitediamondcontainingblacksmalldiamond", 0x25c8 },
	{ "whitedownpointingsmalltriangle", 0x25bf },
	{ "whitedownpointingtriangle", 0x25bd },
	{ "whiteleftpointingsmalltriangle", 0x25c3 },
	{ "whiteleftpointingtriangle", 0x25c1 },
	{ "whitelenticularbracketleft", 0x3016 },
	{ "whitelenticularbracketright", 0x3017 },
	{ "whiterightpointingsmalltriangle", 0x25b9 },
	{ "whiterightpointingtriangle", 0x25b7 },
	{ "whitesmallsquare", 0x25ab },
	{ "whitesmilingface", 0x263a },
	{ "whitesquare", 0x25a1 },
	{ "whitestar", 0x2606 },
	{ "whitetelephone", 0x260f },
	{ "whitetortoiseshellbracketleft", 0x3018 },
	{ "whitetortoiseshellbracketright", 0x3019 },
	{ "whiteuppointingsmalltriangle", 0x25b5 },
	{ "whiteuppointingtriangle", 0x25b3 },
	{ "wihiragana", 0x3090 },
	{ "wikatakana", 0x30f0 },
	{ "wikorean", 0x315f },
	{ "wmonospace", 0xff57 },
	{ "wohiragana", 0x3092 },
	{ "wokatakana", 0x30f2 },
	{ "wokatakanahalfwidth", 0xff66 },
	{ "won", 0x20a9 },
	{ "wonmonospace", 0xffe6 },
	{ "wowaenthai", 0x0e27 },
	{ "wparen", 0x24b2 },
	{ "wring", 0x1e98 },
	{ "wsuperior", 0x02b7 },
	{ "wturned", 0x028d },
	{ "wynn", 0x01bf },
	{ "xabovecmb", 0x033d },
	{ "xbopomofo", 0x3112 },
	{ "xcircle", 0x24e7 },
	{ "xdieresis", 0x1e8d },
	{ "xdotaccent", 0x1e8b },
	{ "xeharmenian", 0x056d },
	{ "xmonospace", 0xff58 },
	{ "xparen", 0x24b3 },
	{ "xsuperior", 0x02e3 },
	{ "yaadosquare", 0x334e },
	{ "yabengali", 0x09af },
	{ "yadeva", 0x092f },
	{ "yaekorean", 0x3152 },
	{ "yagujarati", 0x0aaf },
	{ "yagurmukhi", 0x0a2f },
	{ "yahiragana", 0x3084 },
	{ "yakatakana", 0x30e4 },
	{ "yakatakanahalfwidth", 0xff94 },
	{ "yakorean", 0x3151 },
	{ "yamakkanthai", 0x0e4e },
	{ "yasmallhiragana", 0x3083 },
	{ "yasmallkatakana", 0x30e3 },
	{ "yasmallkatakanahalfwidth", 0xff6c },
	{ "yatcyrillic", 0x0463 },
	{ "ycircle", 0x24e8 },
	{ "ydotaccent", 0x1e8f },
	{ "ydotbelow", 0x1ef5 },
	{ "yeharabic", 0x064a },
	{ "yehbarreearabic", 0x06d2 },
	{ "yehbarreefinalarabic", 0xfbaf },
	{ "yehfinalarabic", 0xfef2 },
	{ "yehhamzaabovearabic", 0x0626 },
	{ "yehhamzaabovefinalarabic", 0xfe8a },
	{ "yehhamzaaboveinitialarabic", 0xfe8b },
	{ "yehhamzaabovemedialarabic", 0xfe8c },
	{ "yehinitialarabic", 0xfef3 },
	{ "yehmedialarabic", 0xfef4 },
	{ "yehmeeminitialarabic", 0xfcdd },
	{ "yehmeemisolatedarabic", 0xfc58 },
	{ "yehnoonfinalarabic", 0xfc94 },
	{ "yehthreedotsbelowarabic", 0x06d1 },
	{ "yekorean", 0x3156 },
	{ "yenmonospace", 0xffe5 },
	{ "yeokorean", 0x3155 },
	{ "yeorinhieuhkorean", 0x3186 },
	{ "yerahbenyomohebrew", 0x05aa },
	{ "yerahbenyomolefthebrew", 0x05aa },
	{ "yericyrillic", 0x044b },
	{ "yerudieresiscyrillic", 0x04f9 },
	{ "yesieungkorean", 0x3181 },
	{ "yesieungpansioskorean", 0x3183 },
	{ "yesieungsioskorean", 0x3182 },
	{ "yetivhebrew", 0x059a },
	{ "yhook", 0x01b4 },
	{ "yhookabove", 0x1ef7 },
	{ "yiarmenian", 0x0575 },
	{ "yicyrillic", 0x0457 },
	{ "yikorean", 0x3162 },
	{ "yinyang", 0x262f },
	{ "yiwnarmenian", 0x0582 },
	{ "ymonospace", 0xff59 },
	{ "yod", 0x05d9 },
	{ "yoddagesh", 0xfb39 },
	{ "yoddageshhebrew", 0xfb39 },
	{ "yodhebrew", 0x05d9 },
	{ "yodyodhebrew", 0x05f2 },
	{ "yodyodpatahhebrew", 0xfb1f },
	{ "yohiragana", 0x3088 },
	{ "yoikorean", 0x3189 },
	{ "yokatakana", 0x30e8 },
	{ "yokatakanahalfwidth", 0xff96 },
	{ "yokorean", 0x315b },
	{ "yosmallhiragana", 0x3087 },
	{ "yosmallkatakana", 0x30e7 },
	{ "yosmallkatakanahalfwidth", 0xff6e },
	{ "yotgreek", 0x03f3 },
	{ "yoyaekorean", 0x3188 },
	{ "yoyakorean", 0x3187 },
	{ "yoyakthai", 0x0e22 },
	{ "yoyingthai", 0x0e0d },
	{ "yparen", 0x24b4 },
	{ "ypogegrammeni", 0x037a },
	{ "ypogegrammenigreekcmb", 0x0345 },
	{ "yr", 0x01a6 },
	{ "yring", 0x1e99 },
	{ "ysuperior", 0x02b8 },
	{ "ytilde", 0x1ef9 },
	{ "yturned", 0x028e },
	{ "yuhiragana", 0x3086 },
	{ "yuikorean", 0x318c },
	{ "yukatakana", 0x30e6 },
	{ "yukatakanahalfwidth", 0xff95 },
	{ "yukorean", 0x3160 },
	{ "yusbigcyrillic", 0x046b },
	{ "yusbigiotifiedcyrillic", 0x046d },
	{ "yuslittlecyrillic", 0x0467 },
	{ "yuslittleiotifiedcyrillic", 0x0469 },
	{ "yusmallhiragana", 0x3085 },
	{ "yusmallkatakana", 0x30e5 },
	{ "yusmallkatakanahalfwidth", 0xff6d },
	{ "yuyekorean", 0x318b },
	{ "yuyeokorean", 0x318a },
	{ "yyabengali", 0x09df },
	{ "yyadeva", 0x095f },
	{ "zaarmenian", 0x0566 },
	{ "zadeva", 0x095b },
	{ "zagurmukhi", 0x0a5b },
	{ "zaharabic", 0x0638 },
	{ "zahfinalarabic", 0xfec6 },
	{ "zahinitialarabic", 0xfec7 },
	{ "zahiragana", 0x3056 },
	{ "zahmedialarabic", 0xfec8 },
	{ "zainarabic", 0x0632 },
	{ "zainfinalarabic", 0xfeb0 },
	{ "zakatakana", 0x30b6 },
	{ "zaqefgadolhebrew", 0x0595 },
	{ "zaqefqatanhebrew", 0x0594 },
	{ "zarqahebrew", 0x0598 },
	{ "zayin", 0x05d6 },
	{ "zayindagesh", 0xfb36 },
	{ "zayindageshhebrew", 0xfb36 },
	{ "zayinhebrew", 0x05d6 },
	{ "zbopomofo", 0x3117 },
	{ "zcircle", 0x24e9 },
	{ "zcircumflex", 0x1e91 },
	{ "zcurl", 0x0291 },
	{ "zdot", 0x017c },
	{ "zdotbelow", 0x1e93 },
	{ "zecyrillic", 0x0437 },
	{ "zedescendercyrillic", 0x0499 },
	{ "zedieresiscyrillic", 0x04df },
	{ "zehiragana", 0x305c },
	{ "zekatakana", 0x30bc },
	{ "zeroarabic", 0x0660 },
	{ "zerobengali", 0x09e6 },
	{ "zerodeva", 0x0966 },
	{ "zerogujarati", 0x0ae6 },
	{ "zerogurmukhi", 0x0a66 },
	{ "zerohackarabic", 0x0660 },
	{ "zeroinferior", 0x2080 },
	{ "zeromonospace", 0xff10 },
	{ "zerooldstyle", 0xf730 },
	{ "zeropersian", 0x06f0 },
	{ "zerosuperior", 0x2070 },
	{ "zerothai", 0x0e50 },
	{ "zerowidthjoiner", 0xfeff },
	{ "zerowidthnonjoiner", 0x200c },
	{ "zerowidthspace", 0x200b },
	{ "zhbopomofo", 0x3113 },
	{ "zhearmenian", 0x056a },
	{ "zhebrevecyrillic", 0x04c2 },
	{ "zhecyrillic", 0x0436 },
	{ "zhedescendercyrillic", 0x0497 },
	{ "zhedieresiscyrillic", 0x04dd },
	{ "zihiragana", 0x3058 },
	{ "zikatakana", 0x30b8 },
	{ "zinorhebrew", 0x05ae },
	{ "zlinebelow", 0x1e95 },
	{ "zmonospace", 0xff5a },
	{ "zohiragana", 0x305e },
	{ "zokatakana", 0x30be },
	{ "zparen", 0x24b5 },
	{ "zretroflexhook", 0x0290 },
	{ "zstroke", 0x01b6 },
	{ "zuhiragana", 0x305a },
	{ "zukatakana", 0x30ba },
	{ "dotaccent.cap", 0xefed },
	{ "breve.cap", 0xefee },
	{ "ogonek.cap", 0xeff1 },
	{ "cedilla.cap", 0xeff2 },
	{ "ring.cap", 0xeff3 },
	{ "tilde.cap", 0xeff5 },
	{ "circumflex.cap", 0xeff7 },
	{ "a1", 0x2701 },
	{ "a2", 0x2702 },
	{ "a202", 0x2703 },
	{ "a3", 0x2704 },
	{ "a4", 0x260e },
	{ "a5", 0x2706 },
	{ "a119", 0x2707 },
	{ "a118", 0x2708 },
	{ "a117", 0x2709 },
	{ "a11", 0x261b },
	{ "a12", 0x261e },
	{ "a13", 0x270c },
	{ "a14", 0x270d },
	{ "a15", 0x270e },
	{ "a16", 0x270f },
	{ "a105", 0x2710 },
	{ "a17", 0x2711 },
	{ "a18", 0x2712 },
	{ "a19", 0x2713 },
	{ "a20", 0x2714 },
	{ "a21", 0x2715 },
	{ "a22", 0x2716 },
	{ "a23", 0x2717 },
	{ "a24", 0x2718 },
	{ "a25", 0x2719 },
	{ "a26", 0x271a },
	{ "a27", 0x271b },
	{ "a28", 0x271c },
	{ "a6", 0x271d },
	{ "a7", 0x271e },
	{ "a8", 0x271f },
	{ "a9", 0x2720 },
	{ "a10", 0x2721 },
	{ "a29", 0x2722 },
	{ "a30", 0x2723 },
	{ "a31", 0x2724 },
	{ "a32", 0x2725 },
	{ "a33", 0x2726 },
	{ "a34", 0x2727 },
	{ "a35", 0x2605 },
	{ "a36", 0x2729 },
	{ "a37", 0x272a },
	{ "a38", 0x272b },
	{ "a39", 0x272c },
	{ "a40", 0x272d },
	{ "a41", 0x272e },
	{ "a42", 0x272f },
	{ "a43", 0x2730 },
	{ "a44", 0x2731 },
	{ "a45", 0x2732 },
	{ "a46", 0x2733 },
	{ "a47", 0x2734 },
	{ "a48", 0x2735 },
	{ "a49", 0x2736 },
	{ "a50", 0x2737 },
	{ "a51", 0x2738 },
	{ "a52", 0x2739 },
	{ "a53", 0x273a },
	{ "a54", 0x273b },
	{ "a55", 0x273c },
	{ "a56", 0x273d },
	{ "a57", 0x273e },
	{ "a58", 0x273f },
	{ "a59", 0x2740 },
	{ "a60", 0x2741 },
	{ "a61", 0x2742 },
	{ "a62", 0x2743 },
	{ "a63", 0x2744 },
	{ "a64", 0x2745 },
	{ "a65", 0x2746 },
	{ "a66", 0x2747 },
	{ "a67", 0x2748 },
	{ "a68", 0x2749 },
	{ "a69", 0x274a },
	{ "a70", 0x274b },
	{ "a71", 0x25cf },
	{ "a72", 0x274d },
	{ "a73", 0x25a0 },
	{ "a74", 0x274f },
	{ "a203", 0x2750 },
	{ "a75", 0x2751 },
	{ "a204", 0x2752 },
	{ "a76", 0x25b2 },
	{ "a77", 0x25bc },
	{ "a78", 0x25c6 },
	{ "a79", 0x2756 },
	{ "a81", 0x25d7 },
	{ "a82", 0x2758 },
	{ "a83", 0x2759 },
	{ "a84", 0x275a },
	{ "a97", 0x275b },
	{ "a98", 0x275c },
	{ "a99", 0x275d },
	{ "a100", 0x275e },
	{ "a89", 0x2768 },
	{ "a90", 0x2769 },
	{ "a93", 0x276a },
	{ "a94", 0x276b },
	{ "a91", 0x276c },
	{ "a92", 0x276d },
	{ "a205", 0x276e },
	{ "a85", 0x276f },
	{ "a206", 0x2770 },
	{ "a86", 0x2771 },
	{ "a87", 0x2772 },
	{ "a88", 0x2773 },
	{ "a95", 0x2774 },
	{ "a96", 0x2775 },
	{ "a101", 0x2761 },
	{ "a102", 0x2762 },
	{ "a103", 0x2763 },
	{ "a104", 0x2764 },
	{ "a106", 0x2765 },
	{ "a107", 0x2766 },
	{ "a108", 0x2767 },
	{ "a112", 0x2663 },
	{ "a111", 0x2666 },
	{ "a110", 0x2665 },
	{ "a109", 0x2660 },
	{ "a120", 0x2460 },
	{ "a121", 0x2461 },
	{ "a122", 0x2462 },
	{ "a123", 0x2463 },
	{ "a124", 0x2464 },
	{ "a125", 0x2465 },
	{ "a126", 0x2466 },
	{ "a127", 0x2467 },
	{ "a128", 0x2468 },
	{ "a129", 0x2469 },
	{ "a130", 0x2776 },
	{ "a131", 0x2777 },
	{ "a132", 0x2778 },
	{ "a133", 0x2779 },
	{ "a134", 0x277a },
	{ "a135", 0x277b },
	{ "a136", 0x277c },
	{ "a137", 0x277d },
	{ "a138", 0x277e },
	{ "a139", 0x277f },
	{ "a140", 0x2780 },
	{ "a141", 0x2781 },
	{ "a142", 0x2782 },
	{ "a143", 0x2783 },
	{ "a144", 0x2784 },
	{ "a145", 0x2785 },
	{ "a146", 0x2786 },
	{ "a147", 0x2787 },
	{ "a148", 0x2788 },
	{ "a149", 0x2789 },
	{ "a150", 0x278a },
	{ "a151", 0x278b },
	{ "a152", 0x278c },
	{ "a153", 0x278d },
	{ "a154", 0x278e },
	{ "a155", 0x278f },
	{ "a156", 0x2790 },
	{ "a157", 0x2791 },
	{ "a158", 0x2792 },
	{ "a159", 0x2793 },
	{ "a160", 0x2794 },
	{ "a161", 0x2192 },
	{ "a163", 0x2194 },
	{ "a164", 0x2195 },
	{ "a196", 0x2798 },
	{ "a165", 0x2799 },
	{ "a192", 0x279a },
	{ "a166", 0x279b },
	{ "a167", 0x279c },
	{ "a168", 0x279d },
	{ "a169", 0x279e },
	{ "a170", 0x279f },
	{ "a171", 0x27a0 },
	{ "a172", 0x27a1 },
	{ "a173", 0x27a2 },
	{ "a162", 0x27a3 },
	{ "a174", 0x27a4 },
	{ "a175", 0x27a5 },
	{ "a176", 0x27a6 },
	{ "a177", 0x27a7 },
	{ "a178", 0x27a8 },
	{ "a179", 0x27a9 },
	{ "a193", 0x27aa },
	{ "a180", 0x27ab },
	{ "a199", 0x27ac },
	{ "a181", 0x27ad },
	{ "a200", 0x27ae },
	{ "a182", 0x27af },
	{ "a201", 0x27b1 },
	{ "a183", 0x27b2 },
	{ "a184", 0x27b3 },
	{ "a197", 0x27b4 },
	{ "a185", 0x27b5 },
	{ "a194", 0x27b6 },
	{ "a198", 0x27b7 },
	{ "a186", 0x27b8 },
	{ "a195", 0x27b9 },
	{ "a187", 0x27ba },
	{ "a188", 0x27bb },
	{ "a189", 0x27bc },
	{ "a190", 0x27bd },
	{ "a191", 0x27be },
	{ "register.sans", 0xf8e8 },
	{ "register.serif", 0xf6da },
	{ "one.superior", 0x00b9 },
	{ "two.superior", 0x00b2 },
	{ "three.superior", 0x00b3 },
	{ "parenleft.superior", 0x207d },
	{ "parenright.superior", 0x207e },
	{ "n.superior", 0x207f },
	{ "parenleft.inferior", 0x208d },
	{ "parenright.inferior", 0x208e },
	{ "zero.superior", 0x2070 },
	{ "four.superior", 0x2074 },
	{ "five.superior", 0x2075 },
	{ "six.superior", 0x2076 },
	{ "seven.superior", 0x2077 },
	{ "eight.superior", 0x2078 },
	{ "nine.superior", 0x2079 },
	{ "zero.inferior", 0x2080 },
	{ "one.inferior", 0x2081 },
	{ "two.inferior", 0x2082 },
	{ "three.inferior", 0x2083 },
	{ "four.inferior", 0x2084 },
	{ "five.inferior", 0x2085 },
	{ "six.inferior", 0x2086 },
	{ "seven.inferior", 0x2087 },
	{ "eight.inferior", 0x2088 },
	{ "nine.inferior", 0x2089 },
	{ "mu.greek", 0x03bc },
	{ "Delta.greek", 0x0394 },
	{ "Omega.greek", 0x03a9 },
	{ "sigma.end", 0x03c2 },
	{ "beta.alt", 0x03d0 },
	{ "kappa.alt", 0x03f0 },
	{ "theta.alt", 0x03d1 },
	{ "Upsilon.alt", 0x03d2 },
	{ "phi.alt", 0x03d5 },
	{ "pi.alt", 0x03d6 },
	{ "A.cyr", 0x0410 },
	{ "Ghe.up", 0x0490 },
	{ "I.cyr", 0x0418 },
	{ "I.short", 0x0419 },
	{ "O.cyr", 0x041e },
	{ "U.cyr", 0x0423 },
	{ "E.cyr", 0x042d },
	{ "Ie.ukran", 0x0404 },
	{ "I.ukran", 0x0406 },
	{ "U.short", 0x040e },
	{ "a.cyr", 0x0430 },
	{ "ghe.up", 0x0491 },
	{ "i.cyr", 0x0438 },
	{ "i.short", 0x0439 },
	{ "o.cyr", 0x043e },
	{ "u.cyr", 0x0443 },
	{ "e.cyr", 0x044d },
	{ "ie.ukran", 0x0454 },
	{ "i.ukran", 0x0456 },
	{ "u.short", 0x045e },
	{ "ghe.ital", 0xf6c4 },
	{ "afii10068.ital", 0xf6c4 },
	{ "afii10066.ital", 0xf6c5 },
	{ "be.alt", 0xf6c5 },
	{ "afii10069.ital", 0xf6c6 },
	{ "de.ital", 0xf6c6 },
	{ "afii10081.ital", 0xf6c7 },
	{ "pe.ital", 0xf6c7 },
	{ "afii10084.ital", 0xf6c8 },
	{ "te.ital", 0xf6c8 },
	{ "Omega.math", 0x2126 },
	{ "Delta.math", 0x2206 },
	{ "afii57689_afii57804", 0xfb2a },
	{ "afii57689_afii57803", 0xfb2b },
	{ "afii57669_afii57806", 0xfb4b },
	{ "afii57718_afii57798", 0xfb1f },
	{ "afii57669_afii57807", 0xfb35 },
	{ ".null", 0x0000 },
	{ "dialytika_perispomeni", 0x1fc1 },
	{ "psili_varia", 0x1fcd },
	{ "psili_oxia", 0x1fce },
	{ "psili_perispomeni", 0x1fcf },
	{ "dasia_varia", 0x1fdd },
	{ "dasia_oxia", 0x1fde },
	{ "dasia_perispomeni", 0x1fdf },
	{ "dialytika_varia", 0x1fed },
	{ "dialytika_oxia", 0x1fee },
	{ "Alphaacute", 0x1fbb },
	{ "Alphaasper", 0x1f09 },
	{ "Alphaasperacute", 0x1f0d },
	{ "Alphaaspergrave", 0x1f0b },
	{ "Alphaaspertilde", 0x1f0f },
	{ "Alphabreve", 0x1fb8 },
	{ "Alphagrave", 0x1fba },
	{ "Alphaiotasub", 0x1fbc },
	{ "Alphaiotasubasper", 0x1f89 },
	{ "Alphaiotasubasperacute", 0x1f8d },
	{ "Alphaiotasubaspergrave", 0x1f8b },
	{ "Alphaiotasubaspertilde", 0x1f8f },
	{ "Alphaiotasublenis", 0x1f88 },
	{ "Alphaiotasublenisacute", 0x1f8c },
	{ "Alphaiotasublenisgrave", 0x1f8a },
	{ "Alphaiotasublenistilde", 0x1f8e },
	{ "Alphalenis", 0x1f08 },
	{ "Alphalenisacute", 0x1f0c },
	{ "Alphalenisgrave", 0x1f0a },
	{ "Alphalenistilde", 0x1f0e },
	{ "Alphamacron", 0x1fb9 },
	{ "Epsilonacute", 0x1fc9 },
	{ "Epsilonasper", 0x1f19 },
	{ "Epsilonasperacute", 0x1f1d },
	{ "Epsilonaspergrave", 0x1f1b },
	{ "Epsilongrave", 0x1fc8 },
	{ "Epsilonlenis", 0x1f18 },
	{ "Epsilonlenisacute", 0x1f1c },
	{ "Epsilonlenisgrave", 0x1f1a },
	{ "Etaacute", 0x1fcb },
	{ "Etaasper", 0x1f29 },
	{ "Etaasperacute", 0x1f2d },
	{ "Etaaspergrave", 0x1f2b },
	{ "Etaaspertilde", 0x1f2f },
	{ "Etagrave", 0x1fca },
	{ "Etaiotasub", 0x1fcc },
	{ "Etaiotasubasper", 0x1f99 },
	{ "Etaiotasubasperacute", 0x1f9d },
	{ "Etaiotasubaspergrave", 0x1f9b },
	{ "Etaiotasubaspertilde", 0x1f9f },
	{ "Etaiotasublenis", 0x1f98 },
	{ "Etaiotasublenisacute", 0x1f9c },
	{ "Etaiotasublenisgrave", 0x1f9a },
	{ "Etaiotasublenistilde", 0x1f9e },
	{ "Etalenis", 0x1f28 },
	{ "Etalenisacute", 0x1f2c },
	{ "Etalenisgrave", 0x1f2a },
	{ "Etalenistilde", 0x1f2e },
	{ "Iotaacute", 0x1fdb },
	{ "Iotaasper", 0x1f39 },
	{ "Iotaasperacute", 0x1f3d },
	{ "Iotaaspergrave", 0x1f3b },
	{ "Iotaaspertilde", 0x1f3f },
	{ "Iotabreve", 0x1fd8 },
	{ "Iotagrave", 0x1fda },
	{ "Iotalenis", 0x1f38 },
	{ "Iotalenisacute", 0x1f3c },
	{ "Iotalenisgrave", 0x1f3a },
	{ "Iotalenistilde", 0x1f3e },
	{ "Iotamacron", 0x1fd9 },
	{ "Omegaacute", 0x1ffb },
	{ "Omegaasper", 0x1f69 },
	{ "Omegaasperacute", 0x1f6d },
	{ "Omegaaspergrave", 0x1f6b },
	{ "Omegaaspertilde", 0x1f6f },
	{ "Omegagrave", 0x1ffa },
	{ "Omegaiotasub", 0x1ffc },
	{ "Omegaiotasubasper", 0x1fa9 },
	{ "Omegaiotasubasperacute", 0x1fad },
	{ "Omegaiotasubaspergrave", 0x1fab },
	{ "Omegaiotasubaspertilde", 0x1faf },
	{ "Omegaiotasublenis", 0x1fa8 },
	{ "Omegaiotasublenisacute", 0x1fac },
	{ "Omegaiotasublenisgrave", 0x1faa },
	{ "Omegaiotasublenistilde", 0x1fae },
	{ "Omegalenis", 0x1f68 },
	{ "Omegalenisacute", 0x1f6c },
	{ "Omegalenisgrave", 0x1f6a },
	{ "Omegalenistilde", 0x1f6e },
	{ "Omicronacute", 0x1ff9 },
	{ "Omicronasper", 0x1f49 },
	{ "Omicronasperacute", 0x1f4d },
	{ "Omicronaspergrave", 0x1f4b },
	{ "Omicrongrave", 0x1ff8 },
	{ "Omicronlenis", 0x1f48 },
	{ "Omicronlenisacute", 0x1f4c },
	{ "Omicronlenisgrave", 0x1f4a },
	{ "Rhoasper", 0x1fec },
	{ "Upsilonacute", 0x1feb },
	{ "Upsilonasper", 0x1f59 },
	{ "Upsilonasperacute", 0x1f5d },
	{ "Upsilonaspergrave", 0x1f5b },
	{ "Upsilonaspertilde", 0x1f5f },
	{ "Upsilonbreve", 0x1fe8 },
	{ "Upsilongrave", 0x1fea },
	{ "Upsilonmacron", 0x1fe9 },
	{ "numeralsign", 0x0374 },
	{ "lownumeralsign", 0x0375 },
	{ "lowcomma", 0x037a },
	{ "stigma", 0x03da },
	{ "koppa", 0x03de },
	{ "sampi", 0x03e0 },
	{ "lenis", 0x1fbd },
	{ "iotasubscript", 0x1fbe },
	{ "dieresistilde", 0x1fc1 },
	{ "lenisgrave", 0x1fcd },
	{ "lenisacute", 0x1fce },
	{ "lenistilde", 0x1fcf },
	{ "aspergrave", 0x1fdd },
	{ "asperacute", 0x1fde },
	{ "aspertilde", 0x1fdf },
	{ "dialytikagrave", 0x1fed },
	{ "dialytikaacute", 0x1fee },
	{ "asper", 0x1ffe },
	{ "alphalenis", 0x1f00 },
	{ "alphaasper", 0x1f01 },
	{ "alphalenisgrave", 0x1f02 },
	{ "alphaaspergrave", 0x1f03 },
	{ "alphalenisacute", 0x1f04 },
	{ "alphaasperacute", 0x1f05 },
	{ "alphalenistilde", 0x1f06 },
	{ "alphaaspertilde", 0x1f07 },
	{ "epsilonlenis", 0x1f10 },
	{ "epsilonasper", 0x1f11 },
	{ "epsilonlenisgrave", 0x1f12 },
	{ "epsilonaspergrave", 0x1f13 },
	{ "epsilonlenisacute", 0x1f14 },
	{ "epsilonasperacute", 0x1f15 },
	{ "etalenis", 0x1f20 },
	{ "etaasper", 0x1f21 },
	{ "etalenisgrave", 0x1f22 },
	{ "etaaspergrave", 0x1f23 },
	{ "etalenisacute", 0x1f24 },
	{ "etaasperacute", 0x1f25 },
	{ "etalenistilde", 0x1f26 },
	{ "etaaspertilde", 0x1f27 },
	{ "iotalenis", 0x1f30 },
	{ "iotaasper", 0x1f31 },
	{ "iotalenisgrave", 0x1f32 },
	{ "iotaaspergrave", 0x1f33 },
	{ "iotalenisacute", 0x1f34 },
	{ "iotaasperacute", 0x1f35 },
	{ "iotalenistilde", 0x1f36 },
	{ "iotaaspertilde", 0x1f37 },
	{ "omicronlenis", 0x1f40 },
	{ "omicronasper", 0x1f41 },
	{ "omicronlenisgrave", 0x1f42 },
	{ "omicronaspergrave", 0x1f43 },
	{ "omicronlenisacute", 0x1f44 },
	{ "omicronasperacute", 0x1f45 },
	{ "upsilonlenis", 0x1f50 },
	{ "upsilonasper", 0x1f51 },
	{ "upsilonlenisgrave", 0x1f52 },
	{ "upsilonaspergrave", 0x1f53 },
	{ "upsilonlenisacute", 0x1f54 },
	{ "upsilonasperacute", 0x1f55 },
	{ "upsilonlenistilde", 0x1f56 },
	{ "upsilonaspertilde", 0x1f57 },
	{ "omegalenis", 0x1f60 },
	{ "omegaasper", 0x1f61 },
	{ "omegalenisgrave", 0x1f62 },
	{ "omegaaspergrave", 0x1f63 },
	{ "omegalenisacute", 0x1f64 },
	{ "omegaasperacute", 0x1f65 },
	{ "omegalenistilde", 0x1f66 },
	{ "omegaaspertilde", 0x1f67 },
	{ "alphagrave", 0x1f70 },
	{ "alphaacute", 0x1f71 },
	{ "epsilongrave", 0x1f72 },
	{ "epsilonacute", 0x1f73 },
	{ "etagrave", 0x1f74 },
	{ "etaacute", 0x1f75 },
	{ "iotagrave", 0x1f76 },
	{ "iotaacute", 0x1f77 },
	{ "omicrongrave", 0x1f78 },
	{ "omicronacute", 0x1f79 },
	{ "upsilongrave", 0x1f7a },
	{ "upsilonacute", 0x1f7b },
	{ "omegagrave", 0x1f7c },
	{ "omegaacute", 0x1f7d },
	{ "alphaiotasublenis", 0x1f80 },
	{ "alphaiotasubasper", 0x1f81 },
	{ "alphaiotasublenisgrave", 0x1f82 },
	{ "alphaiotasubaspergrave", 0x1f83 },
	{ "alphaiotasublenisacute", 0x1f84 },
	{ "alphaiotasubasperacute", 0x1f85 },
	{ "alphaiotasublenistilde", 0x1f86 },
	{ "alphaiotasubaspertilde", 0x1f87 },
	{ "etaiotasublenis", 0x1f90 },
	{ "etaiotasubasper", 0x1f91 },
	{ "etaiotasublenisgrave", 0x1f92 },
	{ "etaiotasubaspergrave", 0x1f93 },
	{ "etaiotasublenisacute", 0x1f94 },
	{ "etaiotasubasperacute", 0x1f95 },
	{ "etaiotasublenistilde", 0x1f96 },
	{ "etaiotasubaspertilde", 0x1f97 },
	{ "omegaiotasublenis", 0x1fa0 },
	{ "omegaiotasubasper", 0x1fa1 },
	{ "omegaiotasublenisgrave", 0x1fa2 },
	{ "omegaiotasubaspergrave", 0x1fa3 },
	{ "omegaiotasublenisacute", 0x1fa4 },
	{ "omegaiotasubasperacute", 0x1fa5 },
	{ "omegaiotasublenistilde", 0x1fa6 },
	{ "omegaiotasubaspertilde", 0x1fa7 },
	{ "alphabreve", 0x1fb0 },
	{ "alphamacron", 0x1fb1 },
	{ "alphaiotasubgrave", 0x1fb2 },
	{ "alphaiotasub", 0x1fb3 },
	{ "alphatilde", 0x1fb6 },
	{ "alphaiotasubtilde", 0x1fb7 },
	{ "etaiotasubgrave", 0x1fc2 },
	{ "etaiotasub", 0x1fc3 },
	{ "etaiotasubacute", 0x1fc4 },
	{ "etatilde", 0x1fc6 },
	{ "etaiotasubtilde", 0x1fc7 },
	{ "iotabreve", 0x1fd0 },
	{ "iotamacron", 0x1fd1 },
	{ "iotadieresisgrave", 0x1fd2 },
	{ "iotadieresisacute", 0x1fd3 },
	{ "iotatilde", 0x1fd6 },
	{ "iotadieresistilde", 0x1fd7 },
	{ "upsilonbreve", 0x1fe0 },
	{ "upsilonmacron", 0x1fe1 },
	{ "upsilondieresisgrave", 0x1fe2 },
	{ "upsilondieresisacute", 0x1fe3 },
	{ "rholenis", 0x1fe4 },
	{ "rhoasper", 0x1fe5 },
	{ "upsilontilde", 0x1fe6 },
	{ "omegaiotasubgrave", 0x1ff2 },
	{ "omegaiotasub", 0x1ff3 },
	{ "omegaiotasubacute", 0x1ff4 },
	{ "omegatilde", 0x1ff6 },
	{ "omegaiotasubtilde", 0x1ff7 },
	{ "alphaiotasubacute", 0x1fb4 },
	{ "upsilondieresistilde", 0x1fe7 },
	{ "digamma", 0xefd5 },
	{ "Be", 0x0411 },
	{ "Ve", 0x0412 },
	{ "Ghe", 0x0413 },
	{ "De", 0x0414 },
	{ "Ie", 0x0415 },
	{ "Zhe", 0x0416 },
	{ "Ze", 0x0417 },
	{ "Ka", 0x041a },
	{ "El", 0x041b },
	{ "Em", 0x041c },
	{ "En", 0x041d },
	{ "Pe", 0x041f },
	{ "Er", 0x0420 },
	{ "Es", 0x0421 },
	{ "Te", 0x0422 },
	{ "Ef", 0x0424 },
	{ "Ha", 0x0425 },
	{ "Tse", 0x0426 },
	{ "Che", 0x0427 },
	{ "Sha", 0x0428 },
	{ "Shcha", 0x0429 },
	{ "Hard", 0x042a },
	{ "Yeru", 0x042b },
	{ "Soft", 0x042c },
	{ "Yu", 0x042e },
	{ "Ya", 0x042f },
	{ "Io", 0x0401 },
	{ "Dje", 0x0402 },
	{ "Gje", 0x0403 },
	{ "Dze", 0x0405 },
	{ "Yi", 0x0407 },
	{ "Je", 0x0408 },
	{ "Lje", 0x0409 },
	{ "Nje", 0x040a },
	{ "Tshe", 0x040b },
	{ "Kje", 0x040c },
	{ "Dzhe", 0x040f },
	{ "Yat", 0x0462 },
	{ "Fita", 0x0472 },
	{ "Izhitsa", 0x0474 },
	{ "be", 0x0431 },
	{ "ve", 0x0432 },
	{ "ghe", 0x0433 },
	{ "de", 0x0434 },
	{ "ie", 0x0435 },
	{ "zhe", 0x0436 },
	{ "ze", 0x0437 },
	{ "ka", 0x043a },
	{ "el", 0x043b },
	{ "em", 0x043c },
	{ "en", 0x043d },
	{ "er", 0x0440 },
	{ "es", 0x0441 },
	{ "te", 0x0442 },
	{ "ef", 0x0444 },
	{ "ha", 0x0445 },
	{ "tse", 0x0446 },
	{ "che", 0x0447 },
	{ "sha", 0x0448 },
	{ "shcha", 0x0449 },
	{ "hard", 0x044a },
	{ "yeru", 0x044b },
	{ "soft", 0x044c },
	{ "yu", 0x044e },
	{ "ya", 0x044f },
	{ "io", 0x0451 },
	{ "dje", 0x0452 },
	{ "gje", 0x0453 },
	{ "dze", 0x0455 },
	{ "yi", 0x0457 },
	{ "je", 0x0458 },
	{ "lje", 0x0459 },
	{ "nje", 0x045a },
	{ "tshe", 0x045b },
	{ "kje", 0x045c },
	{ "dzhe", 0x045f },
	{ "yat", 0x0463 },
	{ "fita", 0x0473 },
	{ "izhitsa", 0x0475 },
	{ "litre", 0x2113 },
	{ "Dotaccent", 0xefed },
	{ "Breve", 0xefee },
	{ "Ogonek", 0xeff1 },
	{ "Cedilla", 0xeff2 },
	{ "Ring", 0xeff3 },
	{ "Tilde", 0xeff5 },
	{ "Circumflex", 0xeff7 },
	{ "dbar", 0x0111 },
	{ "st", 0xfb06 },
	{ "TeXtext32", 0x0337 },
	{ "DD070", 0x2014 },
	{ "monospacedash", 0x2014 },
	{ "Dash1cent", 0x2015 },
	{ "apostrophereverse", 0x201b },
	{ "Pts", 0x20a7 },
	{ "SM760000", 0x2195 },
	{ "SM770000", 0x21a8 },
	{ "SP320000", 0x2219 },
	{ "SD630000", 0x22c5 },
	{ "SM790000", 0x2302 },
	{ "caretinverted", 0x2304 },
	{ "SM680000", 0x2310 },
	{ "SA420000", 0x2319 },
	{ "BorderULCorner1", 0x231c },
	{ "DD010", 0x231c },
	{ "DD101", 0x231c },
	{ "BorderURCorner1", 0x231d },
	{ "DD012", 0x231d },
	{ "DD104", 0x231d },
	{ "BorderLLCorner1", 0x231e },
	{ "DD014", 0x231e },
	{ "DD109", 0x231e },
	{ "BorderLRCorner1", 0x231f },
	{ "DD015", 0x231f },
	{ "DD112", 0x231f },
	{ "SS260000", 0x2320 },
	{ "SS270000", 0x2321 },
	{ "hook", 0x2440 },
	{ "chair", 0x2441 },
	{ "fork", 0x2442 },
	{ "SF600000", 0x2580 },
	{ "BorderLower1", 0x2581 },
	{ "DD011", 0x2581 },
	{ "DD021", 0x2581 },
	{ "DD110", 0x2581 },
	{ "Flower1l", 0x2581 },
	{ "SF570000", 0x2584 },
	{ "SF610000", 0x2588 },
	{ "Tile", 0x2588 },
	{ "HalfTile", 0x258c },
	{ "SF580000", 0x258c },
	{ "BorderLeftVertical1", 0x258f },
	{ "DD013", 0x258f },
	{ "DD034", 0x258f },
	{ "DD105", 0x258f },
	{ "Flower1d", 0x258f },
	{ "SF590000", 0x2590 },
	{ "SF140000", 0x2591 },
	{ "SF150000", 0x2592 },
	{ "SF160000", 0x2593 },
	{ "BorderUpper1", 0x2594 },
	{ "DD024", 0x2594 },
	{ "DD102", 0x2594 },
	{ "BorderRightVertical1", 0x2595 },
	{ "DD031", 0x2595 },
	{ "DD106", 0x2595 },
	{ "Flower2dr", 0x2595 },
	{ "SM600000", 0x25b2 },
	{ "SM590000", 0x25b6 },
	{ "SV040000", 0x25bc },
	{ "SM630000", 0x25c0 },
	{ "SM750000", 0x25cb },
	{ "SM570001", 0x25d8 },
	{ "SM750002", 0x25d9 },
	{ "ShootingStar", 0x2604 },
	{ "DD130", 0x2605 },
	{ "StarDingbat1", 0x2605 },
	{ "TheSun", 0x2609 },
	{ "FleuronLeft", 0x2619 },
	{ "cross", 0x2629 },
	{ "SS000000", 0x263a },
	{ "SS010000", 0x263b },
	{ "SM690000", 0x263c },
	{ "TheMoon", 0x263d },
	{ "Moon", 0x263e },
	{ "Mercury", 0x263f },
	{ "Venus", 0x2640 },
	{ "Earth", 0x2641 },
	{ "Mars", 0x2642 },
	{ "Jupiter", 0x2643 },
	{ "Saturn", 0x2644 },
	{ "Uranus", 0x2645 },
	{ "Neptune", 0x2646 },
	{ "Pluto", 0x2647 },
	{ "Aries", 0x2648 },
	{ "Taurus", 0x2649 },
	{ "Gemini", 0x264a },
	{ "Cancer", 0x264b },
	{ "Leo", 0x264c },
	{ "Virgo", 0x264d },
	{ "Libra", 0x264e },
	{ "Scorpio", 0x264f },
	{ "Sagittarius", 0x2650 },
	{ "Capricorn", 0x2651 },
	{ "Aquarius", 0x2652 },
	{ "Pisces", 0x2653 },
	{ "diamondopen", 0x2662 },
	{ "FleuronCenter", 0x2766 },
	{ "FleuronRight", 0x2767 },
	{ "st1", 0xfb06 },
	{ "zeroslash", 0xf638 },
	{ "pi1", 0x03d6 },
	{ "hyphen-minus", 0x002d },
	{ "hyphenminus", 0x002d },
	{ "nonmarkingreturn", 0x000c },
	{ "micro", 0x00b5 },
	{ "Dmacron", 0x0110 },
	{ "kra", 0x0138 },
	{ "bbar", 0x0180 },
	{ "Dbar", 0x0189 },
	{ "deltaturn", 0x018d },
	{ "Eturn", 0x018e },
	{ "Epsilonlatin", 0x0190 },
	{ "Gammalatin", 0x0194 },
	{ "Iotalatin", 0x0196 },
	{ "Ibar", 0x0197 },
	{ "lambdabar", 0x019b },
	{ "mcapturn", 0x019c },
	{ "Nhook", 0x019d },
	{ "nleg", 0x019e },
	{ "Obar", 0x019f },
	{ "OI", 0x01a2 },
	{ "YR", 0x01a6 },
	{ "eshlooprev", 0x01aa },
	{ "Trthook", 0x01ae },
	{ "Upsilonlatin", 0x01b1 },
	{ "Vcursive", 0x01b2 },
	{ "Zbar", 0x01b5 },
	{ "zbar", 0x01b6 },
	{ "Yogh", 0x01b7 },
	{ "Yoghrev", 0x01b8 },
	{ "yoghrev", 0x01b9 },
	{ "yoghtail", 0x01ba },
	{ "twobar", 0x01bb },
	{ "glottalstopbarinv", 0x01be },
	{ "pipe", 0x01c0 },
	{ "pipedbl", 0x01c1 },
	{ "pipedblbar", 0x01c2 },
	{ "exclamlatin", 0x01c3 },
	{ "DZhacek", 0x01c4 },
	{ "Dzhacek", 0x01c5 },
	{ "dzhacek", 0x01c6 },
	{ "Ahacek", 0x01cd },
	{ "ahacek", 0x01ce },
	{ "Ihacek", 0x01cf },
	{ "ihacek", 0x01d0 },
	{ "Ohacek", 0x01d1 },
	{ "ohacek", 0x01d2 },
	{ "Uhacek", 0x01d3 },
	{ "uhacek", 0x01d4 },
	{ "Udieresishacek", 0x01d9 },
	{ "udieresishacek", 0x01da },
	{ "eturn", 0x01dd },
	{ "Gbar", 0x01e4 },
	{ "gbar", 0x01e5 },
	{ "Ghacek", 0x01e6 },
	{ "ghacek", 0x01e7 },
	{ "Khacek", 0x01e8 },
	{ "khacek", 0x01e9 },
	{ "Yoghhacek", 0x01ee },
	{ "yoghhacek", 0x01ef },
	{ "jhacek", 0x01f0 },
	{ "aturn", 0x0250 },
	{ "ascriptturn", 0x0252 },
	{ "cturn", 0x0254 },
	{ "drthook", 0x0256 },
	{ "erev", 0x0258 },
	{ "epsilonlatin", 0x025b },
	{ "epsilonlatinrev", 0x025c },
	{ "epsilonlatinrevhook", 0x025d },
	{ "epsilonlatinrevclosed", 0x025e },
	{ "jdotlessbar", 0x025f },
	{ "gcursive", 0x0261 },
	{ "Gsmallcap", 0x0262 },
	{ "gammalatin", 0x0263 },
	{ "babygamma", 0x0264 },
	{ "hturn", 0x0265 },
	{ "ibar", 0x0268 },
	{ "Ismallcap", 0x026a },
	{ "lmidtilde", 0x026b },
	{ "lrthook", 0x026d },
	{ "lyogh", 0x026e },
	{ "mturn", 0x026f },
	{ "mturndescend", 0x0270 },
	{ "nlftlfthook", 0x0272 },
	{ "nrthook", 0x0273 },
	{ "Nsmallcap", 0x0274 },
	{ "obar", 0x0275 },
	{ "OEsmallcap", 0x0276 },
	{ "omegaclosed", 0x0277 },
	{ "rturn", 0x0279 },
	{ "rturnascend", 0x027a },
	{ "rturnrthook", 0x027b },
	{ "rdescend", 0x027c },
	{ "rrthook", 0x027d },
	{ "rfishhookrev", 0x027f },
	{ "Rsmallcap", 0x0280 },
	{ "Rsmallcapinv", 0x0281 },
	{ "srthook", 0x0282 },
	{ "jhookdblbar", 0x0284 },
	{ "eshshortrev", 0x0285 },
	{ "tturn", 0x0287 },
	{ "trthook", 0x0288 },
	{ "vscript", 0x028b },
	{ "vturn", 0x028c },
	{ "wturn", 0x028d },
	{ "yturn", 0x028e },
	{ "Ysmallcap", 0x028f },
	{ "zrthook", 0x0290 },
	{ "yogh", 0x0292 },
	{ "yoghcurl", 0x0293 },
	{ "glottalstoprevinv", 0x0295 },
	{ "glottalstopinv", 0x0296 },
	{ "cstretch", 0x0297 },
	{ "kiss", 0x0298 },
	{ "Bsmallcap", 0x0299 },
	{ "epsilonclosed", 0x029a },
	{ "Gsmallcaphook", 0x029b },
	{ "Hsmallcap", 0x029c },
	{ "jcrosstail", 0x029d },
	{ "kturn", 0x029e },
	{ "Lsmallcap", 0x029f },
	{ "glottalstopbar", 0x02a1 },
	{ "glottalstopbarrev", 0x02a2 },
	{ "dyogh", 0x02a4 },
	{ "hsuper", 0x02b0 },
	{ "hhooksuper", 0x02b1 },
	{ "jsuper", 0x02b2 },
	{ "rsuper", 0x02b3 },
	{ "rturnsuper", 0x02b4 },
	{ "rturnrthooksuper", 0x02b5 },
	{ "Rturnsuper", 0x02b6 },
	{ "wsuper", 0x02b7 },
	{ "ysuper", 0x02b8 },
	{ "primedblmod", 0x02ba },
	{ "quoteleftmod", 0x02bb },
	{ "apostrophe", 0x02bc },
	{ "apostropherev", 0x02bd },
	{ "ringrighthalfsuper", 0x02be },
	{ "ringlefthalfsuper", 0x02bf },
	{ "glottal", 0x02c0 },
	{ "glottalrev", 0x02c1 },
	{ "fronted", 0x02c2 },
	{ "backed", 0x02c3 },
	{ "raised", 0x02c4 },
	{ "lowered", 0x02c5 },
	{ "linevert", 0x02c8 },
	{ "macronmodifier", 0x02c9 },
	{ "acutemodifier", 0x02ca },
	{ "gravemodifier", 0x02cb },
	{ "linevertsub", 0x02cc },
	{ "macronsub", 0x02cd },
	{ "gravesub", 0x02ce },
	{ "acutesub", 0x02cf },
	{ "length", 0x02d0 },
	{ "halflength", 0x02d1 },
	{ "ringrighthalfcenter", 0x02d2 },
	{ "ringlefthalfsup", 0x02d3 },
	{ "tackupmid", 0x02d4 },
	{ "tackdownmid", 0x02d5 },
	{ "rhotichook", 0x02de },
	{ "gammasuper", 0x02e0 },
	{ "lsuper", 0x02e1 },
	{ "ssuper", 0x02e2 },
	{ "xsuper", 0x02e3 },
	{ "glottalrevsuper", 0x02e4 },
	{ "toneextrahigh", 0x02e5 },
	{ "tonehigh", 0x02e6 },
	{ "tonemid", 0x02e7 },
	{ "tonelow", 0x02e8 },
	{ "toneextralow", 0x02e9 },
	{ "gravenosp", 0x0300 },
	{ "acutenosp", 0x0301 },
	{ "circumflexnosp", 0x0302 },
	{ "tildenosp", 0x0303 },
	{ "macronnosp", 0x0304 },
	{ "overscorenosp", 0x0305 },
	{ "brevenosp", 0x0306 },
	{ "dotnosp", 0x0307 },
	{ "dieresisnosp", 0x0308 },
	{ "hooksupnosp", 0x0309 },
	{ "ringnosp", 0x030a },
	{ "acutedblnosp", 0x030b },
	{ "haceknosp", 0x030c },
	{ "linevertnosp", 0x030d },
	{ "linevertdblnosp", 0x030e },
	{ "gravedblnosp", 0x030f },
	{ "candrabindunosp", 0x0310 },
	{ "breveinvnosp", 0x0311 },
	{ "commaturnsupnosp", 0x0312 },
	{ "apostrophesupnosp", 0x0313 },
	{ "commasuprevnosp", 0x0314 },
	{ "commasuprightnosp", 0x0315 },
	{ "gravesubnosp", 0x0316 },
	{ "acutesubnosp", 0x0317 },
	{ "tackleftsubnosp", 0x0318 },
	{ "tackrightsubnosp", 0x0319 },
	{ "anglesupnosp", 0x031a },
	{ "hornnosp", 0x031b },
	{ "ringlefthalfsubnosp", 0x031c },
	{ "tackupsubnosp", 0x031d },
	{ "tackdownsubnosp", 0x031e },
	{ "plussubnosp", 0x031f },
	{ "minussubnosp", 0x0320 },
	{ "hooksubpalatnosp", 0x0321 },
	{ "hooksubretronosp", 0x0322 },
	{ "dotsubnosp", 0x0323 },
	{ "dotdblsubnosp", 0x0324 },
	{ "ringsubnosp", 0x0325 },
	{ "commasubnosp", 0x0326 },
	{ "cedillanosp", 0x0327 },
	{ "ogoneknosp", 0x0328 },
	{ "linevertsubnosp", 0x0329 },
	{ "bridgesubnosp", 0x032a },
	{ "archdblsubnosp", 0x032b },
	{ "haceksubnosp", 0x032c },
	{ "circumflexsubnosp", 0x032d },
	{ "brevesubnosp", 0x032e },
	{ "breveinvsubnosp", 0x032f },
	{ "tildesubnosp", 0x0330 },
	{ "macronsubnosp", 0x0331 },
	{ "underscorenosp", 0x0332 },
	{ "underscoredblnosp", 0x0333 },
	{ "tildemidnosp", 0x0334 },
	{ "barmidshortnosp", 0x0335 },
	{ "barmidlongnosp", 0x0336 },
	{ "slashshortnosp", 0x0337 },
	{ "slashlongnosp", 0x0338 },
	{ "ringrighthalfsubnosp", 0x0339 },
	{ "bridgeinvsubnosp", 0x033a },
	{ "squaresubnosp", 0x033b },
	{ "seagullsubnosp", 0x033c },
	{ "xsupnosp", 0x033d },
	{ "tildevertsupnosp", 0x033e },
	{ "overscoredblnosp", 0x033f },
	{ "graveleftnosp", 0x0340 },
	{ "acuterightnosp", 0x0341 },
	{ "wavyoverscorenosp", 0x0342 },
	{ "zigzagoverscorenosp", 0x0343 },
	{ "diaeresistonosnosp", 0x0344 },
	{ "iotasubnosp", 0x0345 },
	{ "iotasub", 0x037a },
	{ "diaeresistonos", 0x0385 },
	{ "iotadiaeresistonos", 0x0390 },
	{ "Iotadiaeresis", 0x03aa },
	{ "Upsilondiaeresis", 0x03ab },
	{ "upsilondiaeresistonos", 0x03b0 },
	{ "iotadiaeresis", 0x03ca },
	{ "upsilondiaeresis", 0x03cb },
	{ "betacurled", 0x03d0 },
	{ "thetascript", 0x03d1 },
	{ "Upsilonhook", 0x03d2 },
	{ "Upsilonhooktonos", 0x03d3 },
	{ "Upsilonhookdiaeresis", 0x03d4 },
	{ "phiscript", 0x03d5 },
	{ "omegapi", 0x03d6 },
	{ "Stigma", 0x03da },
	{ "stigma", 0x03db },
	{ "Digamma", 0x03dc },
	{ "digamma", 0x03dd },
	{ "Koppa", 0x03de },
	{ "koppa", 0x03df },
	{ "Sampi", 0x03e0 },
	{ "sampi", 0x03e1 },
	{ "Shei", 0x03e2 },
	{ "shei", 0x03e3 },
	{ "Fei", 0x03e4 },
	{ "fei", 0x03e5 },
	{ "Khei", 0x03e6 },
	{ "khei", 0x03e7 },
	{ "Hori", 0x03e8 },
	{ "hori", 0x03e9 },
	{ "Gangia", 0x03ea },
	{ "gangia", 0x03eb },
	{ "Shima", 0x03ec },
	{ "shima", 0x03ed },
	{ "Dei", 0x03ee },
	{ "dei", 0x03ef },
	{ "kappascript", 0x03f0 },
	{ "sigmalunate", 0x03f2 },
	{ "Io", 0x0401 },
	{ "Dje", 0x0402 },
	{ "Gje", 0x0403 },
	{ "Ecyril", 0x0404 },
	{ "Dze", 0x0405 },
	{ "Icyril", 0x0406 },
	{ "Yi", 0x0407 },
	{ "Je", 0x0408 },
	{ "Lje", 0x0409 },
	{ "Nje", 0x040a },
	{ "Tshe", 0x040b },
	{ "Kje", 0x040c },
	{ "Ucyrilbreve", 0x040e },
	{ "Dzhe", 0x040f },
	{ "Acyril", 0x0410 },
	{ "Be", 0x0411 },
	{ "Ve", 0x0412 },
	{ "Ge", 0x0413 },
	{ "De", 0x0414 },
	{ "Ie", 0x0415 },
	{ "Zhe", 0x0416 },
	{ "Ze", 0x0417 },
	{ "Ii", 0x0418 },
	{ "Iibreve", 0x0419 },
	{ "Ka", 0x041a },
	{ "El", 0x041b },
	{ "Em", 0x041c },
	{ "En", 0x041d },
	{ "Ocyril", 0x041e },
	{ "Pecyril", 0x041f },
	{ "Er", 0x0420 },
	{ "Es", 0x0421 },
	{ "Te", 0x0422 },
	{ "Ucyril", 0x0423 },
	{ "Ef", 0x0424 },
	{ "Kha", 0x0425 },
	{ "Tse", 0x0426 },
	{ "Che", 0x0427 },
	{ "Sha", 0x0428 },
	{ "Shcha", 0x0429 },
	{ "Hard", 0x042a },
	{ "Yeri", 0x042b },
	{ "Soft", 0x042c },
	{ "Ecyrilrev", 0x042d },
	{ "Iu", 0x042e },
	{ "Ia", 0x042f },
	{ "acyril", 0x0430 },
	{ "be", 0x0431 },
	{ "ve", 0x0432 },
	{ "ge", 0x0433 },
	{ "de", 0x0434 },
	{ "ie", 0x0435 },
	{ "zhe", 0x0436 },
	{ "ze", 0x0437 },
	{ "ii", 0x0438 },
	{ "iibreve", 0x0439 },
	{ "ka", 0x043a },
	{ "el", 0x043b },
	{ "em", 0x043c },
	{ "en", 0x043d },
	{ "ocyril", 0x043e },
	{ "pecyril", 0x043f },
	{ "er", 0x0440 },
	{ "es", 0x0441 },
	{ "te", 0x0442 },
	{ "ucyril", 0x0443 },
	{ "ef", 0x0444 },
	{ "kha", 0x0445 },
	{ "tse", 0x0446 },
	{ "che", 0x0447 },
	{ "sha", 0x0448 },
	{ "shcha", 0x0449 },
	{ "hard", 0x044a },
	{ "yeri", 0x044b },
	{ "soft", 0x044c },
	{ "ecyrilrev", 0x044d },
	{ "iu", 0x044e },
	{ "ia", 0x044f },
	{ "io", 0x0451 },
	{ "dje", 0x0452 },
	{ "gje", 0x0453 },
	{ "ecyril", 0x0454 },
	{ "dze", 0x0455 },
	{ "icyril", 0x0456 },
	{ "yi", 0x0457 },
	{ "je", 0x0458 },
	{ "lje", 0x0459 },
	{ "nje", 0x045a },
	{ "tshe", 0x045b },
	{ "kje", 0x045c },
	{ "ucyrilbreve", 0x045e },
	{ "dzhe", 0x045f },
	{ "Yat", 0x0462 },
	{ "yat", 0x0463 },
	{ "Yusbig", 0x046a },
	{ "yusbig", 0x046b },
	{ "Psicyril", 0x0470 },
	{ "psicyril", 0x0471 },
	{ "Fita", 0x0472 },
	{ "fita", 0x0473 },
	{ "Izhitsa", 0x0474 },
	{ "izhitsa", 0x0475 },
	{ "Izhitsagravedbl", 0x0476 },
	{ "izhitsagravedbl", 0x0477 },
	{ "Digraphuk", 0x0478 },
	{ "digraphuk", 0x0479 },
	{ "Omegaround", 0x047a },
	{ "omegaround", 0x047b },
	{ "Omegatitlo", 0x047c },
	{ "omegatitlo", 0x047d },
	{ "OT", 0x047e },
	{ "ot", 0x047f },
	{ "Geupturn", 0x0490 },
	{ "geupturn", 0x0491 },
	{ "Gebar", 0x0492 },
	{ "gebar", 0x0493 },
	{ "Gehook", 0x0494 },
	{ "gehook", 0x0495 },
	{ "Zhertdes", 0x0496 },
	{ "zhertdes", 0x0497 },
	{ "Zecedilla", 0x0498 },
	{ "zecedilla", 0x0499 },
	{ "Kartdes", 0x049a },
	{ "kartdes", 0x049b },
	{ "Kavertbar", 0x049c },
	{ "kavertbar", 0x049d },
	{ "Kabar", 0x049e },
	{ "kabar", 0x049f },
	{ "GeKarev", 0x04a0 },
	{ "gekarev", 0x04a1 },
	{ "Enrtdes", 0x04a2 },
	{ "enrtdes", 0x04a3 },
	{ "EnGe", 0x04a4 },
	{ "enge", 0x04a5 },
	{ "Pehook", 0x04a6 },
	{ "pehook", 0x04a7 },
	{ "Ohook", 0x04a8 },
	{ "ohook", 0x04a9 },
	{ "Escedilla", 0x04aa },
	{ "escedilla", 0x04ab },
	{ "Tertdes", 0x04ac },
	{ "tertdes", 0x04ad },
	{ "Ustrt", 0x04ae },
	{ "ustrt", 0x04af },
	{ "Ustrtbar", 0x04b0 },
	{ "ustrtbar", 0x04b1 },
	{ "Khartdes", 0x04b2 },
	{ "khartdes", 0x04b3 },
	{ "TeTse", 0x04b4 },
	{ "tetse", 0x04b5 },
	{ "Chertdes", 0x04b6 },
	{ "chertdes", 0x04b7 },
	{ "Chevertbar", 0x04b8 },
	{ "chevertbar", 0x04b9 },
	{ "Hcyril", 0x04ba },
	{ "hcyril", 0x04bb },
	{ "Iehook", 0x04bc },
	{ "iehook", 0x04bd },
	{ "Iehookogonek", 0x04be },
	{ "iehookogonek", 0x04bf },
	{ "Icyril1", 0x04c0 },
	{ "Zhebreve", 0x04c1 },
	{ "zhebreve", 0x04c2 },
	{ "Kahook", 0x04c3 },
	{ "kahook", 0x04c4 },
	{ "Enhook", 0x04c7 },
	{ "enhook", 0x04c8 },
	{ "Cheleftdes", 0x04cb },
	{ "cheleftdes", 0x04cc },
	{ "qibuts", 0x05bb },
	{ "meteg", 0x05bd },
	{ "maqaf", 0x05be },
	{ "paseq", 0x05c0 },
	{ "shindot", 0x05c1 },
	{ "sindot", 0x05c2 },
	{ "sofpasuq", 0x05c3 },
	{ "kaffinal", 0x05da },
	{ "memfinal", 0x05dd },
	{ "nunfinal", 0x05df },
	{ "pefinal", 0x05e3 },
	{ "tsadifinal", 0x05e5 },
	{ "vavdbl", 0x05f0 },
	{ "vavyod", 0x05f1 },
	{ "yoddbl", 0x05f2 },
	{ "geresh", 0x05f3 },
	{ "gershayim", 0x05f4 },
	{ "varika", 0x05f5 },
	{ "enquad", 0x2000 },
	{ "emquad", 0x2001 },
	{ "emspace", 0x2003 },
	{ "threeperemspace", 0x2004 },
	{ "fourperemspace", 0x2005 },
	{ "sixperemspace", 0x2006 },
	{ "figurespace", 0x2007 },
	{ "punctuationspace", 0x2008 },
	{ "thinspace", 0x2009 },
	{ "hairspace", 0x200a },
	{ "zerospace", 0x200b },
	{ "zeronojoin", 0x200c },
	{ "zerojoin", 0x200d },
	{ "hyphennobreak", 0x2011 },
	{ "quotedash", 0x2015 },
	{ "dashdbl", 0x2016 },
	{ "quotesinglrev", 0x201b },
	{ "quotedblrev", 0x201f },
	{ "trianglebullet", 0x2023 },
	{ "onedotlead", 0x2024 },
	{ "twodotlead", 0x2025 },
	{ "hyphendot", 0x2027 },
	{ "lineseparator", 0x2028 },
	{ "paragraphseparator", 0x2029 },
	{ "lre", 0x202a },
	{ "rle", 0x202b },
	{ "pdf", 0x202c },
	{ "lro", 0x202d },
	{ "rlo", 0x202e },
	{ "pertenthousand", 0x2031 },
	{ "prime", 0x2032 },
	{ "primedbl", 0x2033 },
	{ "primetriple", 0x2034 },
	{ "primerev", 0x2035 },
	{ "primedblrev", 0x2036 },
	{ "primetriplerev", 0x2037 },
	{ "caret", 0x2038 },
	{ "refmark", 0x203b },
	{ "interrobang", 0x203d },
	{ "tie", 0x2040 },
	{ "caretinsert", 0x2041 },
	{ "hyphenbullet", 0x2043 },
	{ "minussuperior", 0x207b },
	{ "plusinferior", 0x208a },
	{ "equalinferior", 0x208c },
	{ "eurocurrency", 0x20a0 },
	{ "coloncurrency", 0x20a1 },
	{ "mill", 0x20a5 },
	{ "naira", 0x20a6 },
	{ "pesetas", 0x20a7 },
	{ "rupee", 0x20a8 },
	{ "newsheqel", 0x20aa },
	{ "accountof", 0x2100 },
	{ "addresssubject", 0x2101 },
	{ "Cbb", 0x2102 },
	{ "degreecentigrade", 0x2103 },
	{ "CL", 0x2104 },
	{ "cadauna", 0x2106 },
	{ "Euler", 0x2107 },
	{ "scruple", 0x2108 },
	{ "degreefarenheit", 0x2109 },
	{ "Hscript", 0x210b },
	{ "Hblackletter", 0x210c },
	{ "Hbb", 0x210d },
	{ "planck", 0x210e },
	{ "planckover2pi", 0x210f },
	{ "Iscript", 0x2110 },
	{ "Lscript", 0x2112 },
	{ "lscript", 0x2113 },
	{ "lbbar", 0x2114 },
	{ "Nbb", 0x2115 },
	{ "recordright", 0x2117 },
	{ "Pbb", 0x2119 },
	{ "Qbb", 0x211a },
	{ "Rscript", 0x211b },
	{ "Rfractur", 0x211c },
	{ "Rbb", 0x211d },
	{ "Rx", 0x211e },
	{ "response", 0x211f },
	{ "servicemark", 0x2120 },
	{ "tel", 0x2121 },
	{ "versicle", 0x2123 },
	{ "Zbb", 0x2124 },
	{ "ounce", 0x2125 },
	{ "ohm", 0x2126 },
	{ "mho", 0x2127 },
	{ "Zblackletter", 0x2128 },
	{ "iotaturn", 0x2129 },
	{ "degreekelvin", 0x212a },
	{ "Bscript", 0x212c },
	{ "Cblackletter", 0x212d },
	{ "escript", 0x212f },
	{ "Escript", 0x2130 },
	{ "Fscript", 0x2131 },
	{ "Fturn", 0x2132 },
	{ "Mscript", 0x2133 },
	{ "u0scrip", 0x2134 },
	{ "alephmath", 0x2135 },
	{ "gimelmath", 0x2137 },
	{ "dalethmath", 0x2138 },
	{ "twothird", 0x2154 },
	{ "onefifth", 0x2155 },
	{ "twofifths", 0x2156 },
	{ "threefifths", 0x2157 },
	{ "fourfifths", 0x2158 },
	{ "onesixth", 0x2159 },
	{ "fivesixths", 0x215a },
	{ "onenumerator", 0x215f },
	{ "arrowlongboth", 0x2194 },
	{ "arrowlongbothv", 0x2195 },
	{ "arrownorthwest", 0x2196 },
	{ "arrownortheast", 0x2197 },
	{ "arrowsoutheast", 0x2198 },
	{ "arrowsouthwest", 0x2199 },
	{ "arrowleftnot", 0x219a },
	{ "arrowrightnot", 0x219b },
	{ "arrowwaveleft", 0x219c },
	{ "arrowwaveright", 0x219d },
	{ "dblarrowheadleft", 0x219e },
	{ "dblarrowheadup", 0x219f },
	{ "dblarrowheadright", 0x21a0 },
	{ "dblarrowheaddown", 0x21a1 },
	{ "arrowtailleft", 0x21a2 },
	{ "arrowtailright", 0x21a3 },
	{ "arrowbarleft", 0x21a4 },
	{ "arrowbarup", 0x21a5 },
	{ "arrowbarright", 0x21a6 },
	{ "arrowbardown", 0x21a7 },
	{ "arrowbothvbase", 0x21a8 },
	{ "arrowhookleft", 0x21a9 },
	{ "arrowhookright", 0x21aa },
	{ "arrowloopleft", 0x21ab },
	{ "arrowloopright", 0x21ac },
	{ "arrowwaveboth", 0x21ad },
	{ "arrowlongbothnot", 0x21ae },
	{ "arrowzigzag", 0x21af },
	{ "arrowrightdown", 0x21b4 },
	{ "carriagerreturn", 0x21b5 },
	{ "arrowsemanticlockw", 0x21b6 },
	{ "arrowsemclockw", 0x21b7 },
	{ "home", 0x21b8 },
	{ "tableftright", 0x21b9 },
	{ "arrowanticlockw", 0x21ba },
	{ "arrowclockw", 0x21bb },
	{ "arrowlefttophalf", 0x21bc },
	{ "arrowleftbothalf", 0x21bd },
	{ "harpoonupright", 0x21be },
	{ "harpoonupleft", 0x21bf },
	{ "arrowrighttophalf", 0x21c0 },
	{ "arrowrightbothalf", 0x21c1 },
	{ "harpoondownright", 0x21c2 },
	{ "harpoondownleft", 0x21c3 },
	{ "arrowparrrightleft", 0x21c4 },
	{ "dblarrowupdown", 0x21c5 },
	{ "arrowparrleftright", 0x21c6 },
	{ "dblarrowup", 0x21c8 },
	{ "dblarrowdown", 0x21ca },
	{ "harpoonleftright", 0x21cb },
	{ "harpoonrightleft", 0x21cc },
	{ "arrowdblleftnot", 0x21cd },
	{ "arrowdbllongbothnot", 0x21ce },
	{ "arrowdblrightnot", 0x21cf },
	{ "arrowdbllongboth", 0x21d4 },
	{ "arrowdbllongbothv", 0x21d5 },
	{ "arrowdblnw", 0x21d6 },
	{ "arrowdblne", 0x21d7 },
	{ "arrowdblse", 0x21d8 },
	{ "arrowdblsw", 0x21d9 },
	{ "arrowtripleleft", 0x21da },
	{ "arrowtripleright", 0x21db },
	{ "arrowsquiggleleft", 0x21dc },
	{ "arrowsquiggleright", 0x21dd },
	{ "arrowopenleft", 0x21e6 },
	{ "arrowopenup", 0x21e7 },
	{ "arrowopenright", 0x21e8 },
	{ "arrowopendown", 0x21e9 },
	{ "complement", 0x2201 },
	{ "notexistential", 0x2204 },
	{ "elementsmall", 0x220a },
	{ "owner", 0x220b },
	{ "notowner", 0x220c },
	{ "ownersmall", 0x220d },
	{ "eop", 0x220e },
	{ "coproduct", 0x2210 },
	{ "dotplus", 0x2214 },
	{ "slashmath", 0x2215 },
	{ "backslashmath", 0x2216 },
	{ "ringoperator", 0x2218 },
	{ "bulletmath", 0x2219 },
	{ "cuberoot", 0x221b },
	{ "fourthroot", 0x221c },
	{ "measuredangle", 0x2221 },
	{ "sphericalangle", 0x2222 },
	{ "notbar", 0x2224 },
	{ "parallelto", 0x2225 },
	{ "notbardbl", 0x2226 },
	{ "integraldbl", 0x222c },
	{ "integraltrpl", 0x222d },
	{ "contintegral", 0x222e },
	{ "surfintegral", 0x222f },
	{ "volintegral", 0x2230 },
	{ "clwintegral", 0x2231 },
	{ "clwcontintegral", 0x2232 },
	{ "cclwcontintegral", 0x2233 },
	{ "dotminus", 0x2238 },
	{ "excess", 0x2239 },
	{ "geomproportion", 0x223a },
	{ "homothetic", 0x223b },
	{ "revsimilar", 0x223d },
	{ "lazysinv", 0x223e },
	{ "sine", 0x223f },
	{ "wreathproduct", 0x2240 },
	{ "notsimilar", 0x2241 },
	{ "minustilde", 0x2242 },
	{ "asymptequal", 0x2243 },
	{ "notasymptequal", 0x2244 },
	{ "approxorequal", 0x2245 },
	{ "approxnotequal", 0x2246 },
	{ "notapproxequal", 0x2247 },
	{ "almostequal", 0x2248 },
	{ "notalmostequal", 0x2249 },
	{ "almostorequal", 0x224a },
	{ "tildetrpl", 0x224b },
	{ "equivasymptotic", 0x224d },
	{ "geomequivalent", 0x224e },
	{ "difference", 0x224f },
	{ "approachlimit", 0x2250 },
	{ "geomequal", 0x2251 },
	{ "imageorapproxequal", 0x2253 },
	{ "colonequal", 0x2254 },
	{ "equalcolon", 0x2255 },
	{ "ringinequal", 0x2256 },
	{ "ringequal", 0x2257 },
	{ "corresponds", 0x2258 },
	{ "estimates", 0x2259 },
	{ "equiangular", 0x225a },
	{ "starequal", 0x225b },
	{ "deltaequal", 0x225c },
	{ "definequal", 0x225d },
	{ "measurequal", 0x225e },
	{ "questionequal", 0x225f },
	{ "notequivalence", 0x2262 },
	{ "strictequivalence", 0x2263 },
	{ "lessdblequal", 0x2266 },
	{ "greaterdblequal", 0x2267 },
	{ "lessnotdblequal", 0x2268 },
	{ "greaternotdblequal", 0x2269 },
	{ "lessmuch", 0x226a },
	{ "greatermuch", 0x226b },
	{ "between", 0x226c },
	{ "notequivasymptotic", 0x226d },
	{ "notlessequal", 0x2270 },
	{ "notgreaterequal", 0x2271 },
	{ "lessequivlnt", 0x2272 },
	{ "greaterequivlnt", 0x2273 },
	{ "notlessequivlnt", 0x2274 },
	{ "notgreaterequivlnt", 0x2275 },
	{ "notlessgreater", 0x2278 },
	{ "notgreaterless", 0x2279 },
	{ "follows", 0x227b },
	{ "precedesequal", 0x227c },
	{ "followsequal", 0x227d },
	{ "precedequivlnt", 0x227e },
	{ "followsequivlnt", 0x227f },
	{ "notpreceeds", 0x2280 },
	{ "notfollows", 0x2281 },
	{ "notpropersubset", 0x2284 },
	{ "notpropersuperset", 0x2285 },
	{ "notreflexsubset", 0x2288 },
	{ "notreflexsuperset", 0x2289 },
	{ "multiset", 0x228c },
	{ "multiplymultiset", 0x228d },
	{ "unionmulti", 0x228e },
	{ "squareimage", 0x228f },
	{ "squareoriginal", 0x2290 },
	{ "subsetsqequal", 0x2291 },
	{ "supersetsqequal", 0x2292 },
	{ "intersectionsq", 0x2293 },
	{ "unionsq", 0x2294 },
	{ "circleminus", 0x2296 },
	{ "circledivide", 0x2298 },
	{ "circledot", 0x2299 },
	{ "circlering", 0x229a },
	{ "circleasterisk", 0x229b },
	{ "circleequal", 0x229c },
	{ "circlevertbar", 0x229d },
	{ "squareplus", 0x229e },
	{ "squareminus", 0x229f },
	{ "squaremultiply", 0x22a0 },
	{ "squaredot", 0x22a1 },
	{ "turnstileleft", 0x22a2 },
	{ "turnstileright", 0x22a3 },
	{ "latticetop", 0x22a4 },
	{ "assertion", 0x22a6 },
	{ "truestate", 0x22a7 },
	{ "satisfy", 0x22a8 },
	{ "force", 0x22a9 },
	{ "tacktrpl", 0x22aa },
	{ "forceextr", 0x22ab },
	{ "notturnstileleft", 0x22ac },
	{ "notsatisfy", 0x22ad },
	{ "notforce", 0x22ae },
	{ "notforceextr", 0x22af },
	{ "lowerrank", 0x22b0 },
	{ "higherrank", 0x22b1 },
	{ "triangleright", 0x22b2 },
	{ "triangleleft", 0x22b3 },
	{ "triangleftequal", 0x22b4 },
	{ "triangrightequal", 0x22b5 },
	{ "original", 0x22b6 },
	{ "image", 0x22b7 },
	{ "multimap", 0x22b8 },
	{ "hermitconjmatrix", 0x22b9 },
	{ "intercal", 0x22ba },
	{ "xor", 0x22bb },
	{ "nand", 0x22bc },
	{ "nor", 0x22bd },
	{ "rightanglearc", 0x22be },
	{ "narylogicaland", 0x22c0 },
	{ "narylogicalor", 0x22c1 },
	{ "naryintersection", 0x22c2 },
	{ "naryunion", 0x22c3 },
	{ "diamondmath", 0x22c4 },
	{ "divideonmultiply", 0x22c7 },
	{ "bowtie", 0x22c8 },
	{ "multicloseleft", 0x22c9 },
	{ "multicloseright", 0x22ca },
	{ "multiopenleft", 0x22cb },
	{ "multiopenright", 0x22cc },
	{ "revasymptequal", 0x22cd },
	{ "curlor", 0x22ce },
	{ "curland", 0x22cf },
	{ "subsetdbl", 0x22d0 },
	{ "supersetdbl", 0x22d1 },
	{ "uniondbl", 0x22d2 },
	{ "intersectiondbl", 0x22d3 },
	{ "fork", 0x22d4 },
	{ "equalparallel", 0x22d5 },
	{ "lessdot", 0x22d6 },
	{ "greaterdot", 0x22d7 },
	{ "verymuchless", 0x22d8 },
	{ "verymuchgreater", 0x22d9 },
	{ "lessequalgreater", 0x22da },
	{ "greaterequalless", 0x22db },
	{ "equalless", 0x22dc },
	{ "equalgreater", 0x22dd },
	{ "equalprecedes", 0x22de },
	{ "equalfollows", 0x22df },
	{ "preceedsnotequal", 0x22e0 },
	{ "followsnotequal", 0x22e1 },
	{ "notsubsetsqequal", 0x22e2 },
	{ "notsupersetsqequal", 0x22e3 },
	{ "sqimageornotequal", 0x22e4 },
	{ "sqoriginornotequal", 0x22e5 },
	{ "lessnotequivlnt", 0x22e6 },
	{ "greaternotequivlnt", 0x22e7 },
	{ "preceedsnotsimilar", 0x22e8 },
	{ "followsnotequivlnt", 0x22e9 },
	{ "nottriangleleft", 0x22ea },
	{ "nottriangleright", 0x22eb },
	{ "nottriangleleftequal", 0x22ec },
	{ "nottrianglerightequal", 0x22ed },
	{ "vertellipsis", 0x22ee },
	{ "midhorizellipsis", 0x22ef },
	{ "upslopeellipsis", 0x22f0 },
	{ "downslopeellipsis", 0x22f1 },
	{ "perspcorrespond", 0x2306 },
	{ "ceilingleft", 0x2308 },
	{ "ceilingright", 0x2309 },
	{ "floorleft", 0x230a },
	{ "floorright", 0x230b },
	{ "slurabove", 0x2322 },
	{ "slurbelow", 0x2323 },
	{ "null", 0x2400 },
	{ "startofhead", 0x2401 },
	{ "starttext", 0x2402 },
	{ "endtext", 0x2403 },
	{ "endtrans", 0x2404 },
	{ "enquiry", 0x2405 },
	{ "acknowledge", 0x2406 },
	{ "bell", 0x2407 },
	{ "backspace", 0x2408 },
	{ "horiztab", 0x2409 },
	{ "linefeed", 0x240a },
	{ "verttab", 0x240b },
	{ "formfeed", 0x240c },
	{ "shiftout", 0x240e },
	{ "shiftin", 0x240f },
	{ "datalinkescape", 0x2410 },
	{ "devcon1", 0x2411 },
	{ "devcon2", 0x2412 },
	{ "devcon3", 0x2413 },
	{ "devcon4", 0x2414 },
	{ "negacknowledge", 0x2415 },
	{ "synch", 0x2416 },
	{ "endtransblock", 0x2417 },
	{ "cancel", 0x2418 },
	{ "endmedium", 0x2419 },
	{ "substitute", 0x241a },
	{ "escape", 0x241b },
	{ "fileseparator", 0x241c },
	{ "groupseparator", 0x241d },
	{ "recordseparator", 0x241e },
	{ "unitseparator", 0x241f },
	{ "spaceliteral", 0x2420 },
	{ "delete", 0x2421 },
	{ "blankb", 0x2422 },
	{ "spaceopenbox", 0x2423 },
	{ "newline", 0x2424 },
	{ "lthorizform", 0x2500 },
	{ "hvhorizform", 0x2501 },
	{ "ltvertform", 0x2502 },
	{ "hvvertform", 0x2503 },
	{ "ltdashtriphorizform", 0x2504 },
	{ "hvdashtriphorizform", 0x2505 },
	{ "ltdashtripvertform", 0x2506 },
	{ "hvdashtripvertform", 0x2507 },
	{ "ltdashquadhorizform", 0x2508 },
	{ "hvdashquadhorizform", 0x2509 },
	{ "ltdashquadvertform", 0x250a },
	{ "hvdashquadvertform", 0x250b },
	{ "ltdnrtform", 0x250c },
	{ "dnltrthvform", 0x250d },
	{ "dnhvrtltform", 0x250e },
	{ "hvdnrtform", 0x250f },
	{ "ltdnleftform", 0x2510 },
	{ "dnltlefthvform", 0x2511 },
	{ "dnhvleftltform", 0x2512 },
	{ "hvdnleftform", 0x2513 },
	{ "ltuprtform", 0x2514 },
	{ "upltrthvform", 0x2515 },
	{ "uphvrtltform", 0x2516 },
	{ "hvuprtform", 0x2517 },
	{ "ltupleftform", 0x2518 },
	{ "upltlefthvform", 0x2519 },
	{ "uphvleftltform", 0x251a },
	{ "hvupleftform", 0x251b },
	{ "ltvertrightform", 0x251c },
	{ "vertltrthvform", 0x251d },
	{ "uphvrtdnltform", 0x251e },
	{ "dnhvrtupltform", 0x251f },
	{ "verthvrtltform", 0x2520 },
	{ "dnltrtuphvform", 0x2521 },
	{ "upltrtdnhvform", 0x2522 },
	{ "hvvertrtform", 0x2523 },
	{ "ltvertleftform", 0x2524 },
	{ "vtltlefthvform", 0x2525 },
	{ "uphvleftdnltform", 0x2526 },
	{ "dnhvleftupltform", 0x2527 },
	{ "verthvleftltform", 0x2528 },
	{ "dnltleftuphvform", 0x2529 },
	{ "upltleftdnhvform", 0x252a },
	{ "hvvertleftform", 0x252b },
	{ "ltdnhorizform", 0x252c },
	{ "lefthvrtdnltform", 0x252d },
	{ "rthvleftdnltform", 0x252e },
	{ "dnlthorizhvform", 0x252f },
	{ "dnhvhorizltform", 0x2530 },
	{ "rtltrtdnhvform", 0x2531 },
	{ "leftltrtdnhvform", 0x2532 },
	{ "hvdnhorizform", 0x2533 },
	{ "ltuphorizform", 0x2534 },
	{ "lefthvrtupltform", 0x2535 },
	{ "rthvleftupltform", 0x2536 },
	{ "uplthorizhvform", 0x2537 },
	{ "uphvhorizltform", 0x2538 },
	{ "rtltleftuphvform", 0x2539 },
	{ "leftltrtuphvform", 0x253a },
	{ "hvuphorizform", 0x253b },
	{ "ltverthorizform", 0x253c },
	{ "lefthvrtvertltform", 0x253d },
	{ "rthvleftvertltform", 0x253e },
	{ "vertlthorizhvform", 0x253f },
	{ "uphvdnhorizltform", 0x2540 },
	{ "dnhvuphorizltform", 0x2541 },
	{ "verthvhorizltform", 0x2542 },
	{ "leftuphvrtdnltform", 0x2543 },
	{ "rtuphvleftdnltform", 0x2544 },
	{ "leftdnhvrtupltform", 0x2545 },
	{ "rtdnhvleftupltform", 0x2546 },
	{ "dnltuphorizhvform", 0x2547 },
	{ "upltdnhorizhvform", 0x2548 },
	{ "rtltleftverthvform", 0x2549 },
	{ "leftltrtverthvform", 0x254a },
	{ "hvverthorizform", 0x254b },
	{ "ltdashdblhorizform", 0x254c },
	{ "hvdashdblhorizform", 0x254d },
	{ "ltdashdblvertform", 0x254e },
	{ "hvdashdblvertform", 0x254f },
	{ "horizdblbar", 0x2550 },
	{ "vertdblbar", 0x2551 },
	{ "dnrtdblform", 0x2552 },
	{ "dndblrtform", 0x2553 },
	{ "dbldnrtform", 0x2554 },
	{ "dnleftdblform", 0x2555 },
	{ "dndblleftform", 0x2556 },
	{ "dbldnleftform", 0x2557 },
	{ "uprtdblform", 0x2558 },
	{ "updblrtform", 0x2559 },
	{ "dbluprtform", 0x255a },
	{ "upleftdblform", 0x255b },
	{ "updblleftform", 0x255c },
	{ "dblupleftform", 0x255d },
	{ "vertrtdblform", 0x255e },
	{ "vertdblrtform", 0x255f },
	{ "dblvertrtform", 0x2560 },
	{ "vertleftdblform", 0x2561 },
	{ "vertdblleftform", 0x2562 },
	{ "dblvertleftform", 0x2563 },
	{ "dnhorizdblform", 0x2564 },
	{ "dndblhorizform", 0x2565 },
	{ "dbldnhorizform", 0x2566 },
	{ "uphorizdblform", 0x2567 },
	{ "updblhorizform", 0x2568 },
	{ "dbluphorizform", 0x2569 },
	{ "verthorizdblform", 0x256a },
	{ "vertdblhorizform", 0x256b },
	{ "dblverthorizform", 0x256c },
	{ "ltarcdnrtform", 0x256d },
	{ "ltarcdnleftform", 0x256e },
	{ "ltarcupleftform", 0x256f },
	{ "ltarcuprtform", 0x2570 },
	{ "forwarddiagonal", 0x2571 },
	{ "backwarddiagonal", 0x2572 },
	{ "ltdiagonalcross", 0x2573 },
	{ "dneighthblock", 0x2581 },
	{ "dnquarterblock", 0x2582 },
	{ "dnthreeeighthblock", 0x2583 },
	{ "dnfiveeighthblock", 0x2585 },
	{ "dnthreequarterblock", 0x2586 },
	{ "dnseveneighthblock", 0x2587 },
	{ "lfseveneighthblock", 0x2589 },
	{ "lfthreequarterblock", 0x258a },
	{ "lffiveeighthblock", 0x258b },
	{ "lfthreeeighthblock", 0x258d },
	{ "lfquarterblock", 0x258e },
	{ "lfeighthblock", 0x258f },
	{ "upeighthblock", 0x2594 },
	{ "rteighthblock", 0x2595 },
	{ "box", 0x25a1 },
	{ "boxrounded", 0x25a2 },
	{ "boxnested", 0x25a3 },
	{ "boxhorizhatch", 0x25a4 },
	{ "boxverthatch", 0x25a5 },
	{ "boxcrosshatch", 0x25a6 },
	{ "boxleftdiaghatch", 0x25a7 },
	{ "boxrtdiaghatch", 0x25a8 },
	{ "boxcrossdiaghatch", 0x25a9 },
	{ "smallboxfilled", 0x25aa },
	{ "smallbox", 0x25ab },
	{ "rectangle", 0x25ad },
	{ "filledvertrect", 0x25ae },
	{ "vertrectangle", 0x25af },
	{ "filledparallelogram", 0x25b0 },
	{ "parallelogram", 0x25b1 },
	{ "triangle", 0x25b3 },
	{ "smalltrianglesld", 0x25b4 },
	{ "smalltriangle", 0x25b5 },
	{ "trianglerightsld1", 0x25b6 },
	{ "triangleright1", 0x25b7 },
	{ "smalltrianglerightsld", 0x25b8 },
	{ "smalltriangleright", 0x25b9 },
	{ "triagrtopen", 0x25bb },
	{ "triangleinv", 0x25bd },
	{ "smalltriangleinvsld", 0x25be },
	{ "smalltriangleinv", 0x25bf },
	{ "triangleleftsld1", 0x25c0 },
	{ "triangleleft1", 0x25c1 },
	{ "smalltriangleleftsld", 0x25c2 },
	{ "smalltriangleleft", 0x25c3 },
	{ "triaglfopen", 0x25c5 },
	{ "diamondrhombsolid", 0x25c6 },
	{ "diamondrhomb", 0x25c7 },
	{ "diamondrhombnested", 0x25c8 },
	{ "circledash", 0x25cc },
	{ "circleverthatch", 0x25cd },
	{ "circlesolid", 0x25cf },
	{ "circleleftsld", 0x25d0 },
	{ "circlerightsld", 0x25d1 },
	{ "circlebottomsld", 0x25d2 },
	{ "circletopsld", 0x25d3 },
	{ "circlenesld", 0x25d4 },
	{ "circlenwopen", 0x25d5 },
	{ "semicircleleftsld", 0x25d6 },
	{ "semicirclelertsld", 0x25d7 },
	{ "invsemicircleup", 0x25da },
	{ "invsemicircledn", 0x25db },
	{ "nwquadarc", 0x25dc },
	{ "nequadarc", 0x25dd },
	{ "sequadarc", 0x25de },
	{ "swquadarc", 0x25df },
	{ "toparc", 0x25e0 },
	{ "bottomarc", 0x25e1 },
	{ "trianglesesld", 0x25e2 },
	{ "triangleswsld", 0x25e3 },
	{ "tranglenwsld", 0x25e4 },
	{ "trianglenesld", 0x25e5 },
	{ "squareleftsld", 0x25e7 },
	{ "squarerightsld", 0x25e8 },
	{ "squarenwsld", 0x25e9 },
	{ "squaresesld", 0x25ea },
	{ "squarevertbisect", 0x25eb },
	{ "triangledot", 0x25ec },
	{ "triangleleftsld", 0x25ed },
	{ "trianglerightsld", 0x25ee },
	{ "heartopen", 0x2661 },
	{ "diamondopen", 0x2662 },
	{ "spadeopen", 0x2664 },
	{ "Omegaiotasubleniscircumflex", 0x1fae },
	{ "Omegaleniscircumflex", 0x1f6e },
	{ "Upsilonaspercircumflex", 0x1f5f },
	{ "dieresiscircumflex", 0x1fc1 },
	{ "leniscircumflex", 0x1fcf },
	{ "aspercircumflex", 0x1fdf },
	{ "alphaleniscircumflex", 0x1f06 },
	{ "alphaaspercircumflex", 0x1f07 },
	{ "etaleniscircumflex", 0x1f26 },
	{ "etaaspercircumflex", 0x1f27 },
	{ "iotaleniscircumflex", 0x1f36 },
	{ "iotaaspercircumflex", 0x1f37 },
	{ "upsilonleniscircumflex", 0x1f56 },
	{ "upsilonaspercircumflex", 0x1f57 },
	{ "omegaleniscircumflex", 0x1f66 },
	{ "omegaaspercircumflex", 0x1f67 },
	{ "alphaiotasubleniscircumflex", 0x1f86 },
	{ "alphaiotasubaspercircumflex", 0x1f87 },
	{ "etaiotasubleniscircumflex", 0x1f96 },
	{ "etaiotasubaspercircumflex", 0x1f97 },
	{ "omegaiotasubleniscircumflex", 0x1fa6 },
	{ "omegaiotasubaspercircumflex", 0x1fa7 },
	{ "alphacircumflex", 0x1fb6 },
	{ "alphaiotasubcircumflex", 0x1fb7 },
	{ "etacircumflex", 0x1fc6 },
	{ "etaiotasubcircumflex", 0x1fc7 },
	{ "iotacircumflex", 0x1fd6 },
	{ "iotadieresiscircumflex", 0x1fd7 },
	{ "upsiloncircumflex", 0x1fe6 },
	{ "omegacircumflex", 0x1ff6 },
	{ "omegaiotasubcircumflex", 0x1ff7 },
	{ "upsilondieresiscircumflex", 0x1fe7 },
	{ "dialytika", 0x0308 },
	{ "koronis", 0x1fbd },
	{ "prosgegrammeni", 0x1fbe },
	{ "psili", 0x1fbf },
	{ "perispomeni", 0x1fc0 },
	{ "varia", 0x1fef },
	{ "oxia", 0x1ffd },
	{ "dasia", 0x1ffe },
	{ "Alphasmall", 0xf500 },
	{ "Betasmall", 0xf501 },
	{ "Gammasmall", 0xf502 },
	{ "Deltasmall", 0xf503 },
	{ "Epsilonsmall", 0xf504 },
	{ "Zetasmall", 0xf505 },
	{ "Etasmall", 0xf506 },
	{ "Thetasmall", 0xf507 },
	{ "Iotasmall", 0xf508 },
	{ "Kappasmall", 0xf509 },
	{ "Lambdasmall", 0xf50a },
	{ "Musmall", 0xf50b },
	{ "Nusmall", 0xf50c },
	{ "Xismall", 0xf50d },
	{ "Omicronsmall", 0xf50e },
	{ "Pismall", 0xf50f },
	{ "Rhosmall", 0xf510 },
	{ "Sigmasmall", 0xf512 },
	{ "Tausmall", 0xf513 },
	{ "Upsilonsmall", 0xf514 },
	{ "Phismall", 0xf515 },
	{ "Chismall", 0xf516 },
	{ "Psismall", 0xf517 },
	{ "Omegasmall", 0xf518 },
	{ "Iotadieresissmall", 0xf519 },
	{ "Upsilondieresissmall", 0xf51a },
	{ "uni2A0B.lgdisplay", 0xea57 },
	{ "uni2A0B.uplgdisplay", 0xea57 },
	{ "uni2A15.lgdisplay", 0xe376 },
	{ "uni2A15.uplgdisplay", 0xe376 },
	{ "uni2A16.lgdisplay", 0xe377 },
	{ "uni2A16.uplgdisplay", 0xe377 },
	{ "uni2A10.lgdisplay", 0xe395 },
	{ "uni2A10.uplgdisplay", 0xe395 },
	{ "uni2A12.lgdisplay", 0xe397 },
	{ "uni2A12.uplgdisplay", 0xe397 },
	{ "uni2A13.lgdisplay", 0xe398 },
	{ "uni2A13.uplgdisplay", 0xe398 },
	{ "uni2A14.lgdisplay", 0xe399 },
	{ "uni2A14.uplgdisplay", 0xe399 },
	{ "uni2A17.lgdisplay", 0xe39a },
	{ "uni2A17.uplgdisplay", 0xe39a },
	{ "uni2A11.uplgdisplay", 0xe39b },
	{ "uni2A11.lgdisplay", 0xe39b },
	{ "uni2A0F.lgdisplay", 0xe3d3 },
	{ "uni2A0F.uplgdisplay", 0xe3d3 },
/* From Richard Kinch's TeX list of glyph aliases */
	{ "fscript", 0x192 },
	{ "fraction1", 0x2215 },
	{ "negationslash", 0x2215 },
	{ "circleR", 0xae },
	{ "circlecopyrt", 0xa9 },
	{ "smile", 0x263a },
	{ "Ifractur", 0x2111 },
	{ "Rfractur", 0x211C },
	{ "Omegainv", 0x2127 },
	{ "mho", 0x2127 },
	{ "alephmath", 0x2135 },
	{ "beth", 0x2136 },
	{ "bethmath", 0x2136 },
	{ "gimelmath", 0x2137 },
	{ "daleth", 0x2138 },
	{ "daletmath", 0x2138 },
	{ "arrowbothv", 0x2195 },
	{ "prime1", 0x2032 },
	{ "primerev1", 0x2035 },
	{ "primereverse", 0x2035 },
	{ "followsequal1", 0x227d },
	{ "similarequal", 0x2243 },
	{ "square", 0x25a1 },
	{ "squaresolid", 0x25a0 },
	{ "squaresmallsolid", 0x25aa },
	{ "diamondsolid", 0x25c6 },
	{ "clockwise", 0x21bb },
	{ "anticlockwise", 0x21ba },
	{ "forces", 0x22a9 },
	{ "forcesbar", 0x22aa },
	{ "satisfies", 0x22a8 },
	{ "dblarrowdwn", 0x21ca },
	{ "shiftleft", 0x2196 },
	{ "shiftright", 0x2197 },
	{ "squiggleright", 0x21dd },
	{ "squiggleleft", 0x21dc },
	{ "squiggleleftright", 0x21ad },
	{ "curlyleft", 0x21ab },
	{ "curlyright", 0x21ac },
	{ "followsorequal", 0x227d },
	{ "equalsdots", 0x2251 },
	{ "defines", 0x225c },
	{ "ng", 0x014b },
	{ "Ng", 0x014a },
	{ "visiblespace", 0x2420 },
	{ "dslash", 0x0111 },
	{ "tie1", 0x2040 },
	{ "arrowdblbothv", 0x21d5 },
	{ "precedesequal1", 0x227c },
	{ "greaterorsimilar", 0x2273 },
	{ "precedesorequal", 0x227e },
	{ "lessorsimilar", 0x2272 },
	{ "equalorgreater", 0x22dd },
	{ "lessorequalslant", 0x2264 },
	{ "equaldotrightleft", 0x2253 },
	{ "equaldotleftright", 0x2252 },
	{ "followsorcurly", 0x227d },
	{ "greaterorequalslant", 0x2265 },
	{ "trianglerightequal", 0x22b5 },
	{ "triangleleftequal", 0x22b4 },
	{ "triangledownsld", 0x25bc },
	{ "arrowaxisright", 0x2192 },
	{ "arrowaxisleft", 0x2190 },
	{ "trianglesolid", 0x25b2 },
	{ "greaterlessequal", 0x22db },
	{ "orunderscore", 0x22bb },
	{ "frown", 0x2322 },
	{ "uprise", 0x22cf },
	{ "downfall", 0x22ce },
	{ "subsetdblequal", 0x2286 },
	{ "supersetdblequal", 0x2287 },
	{ "Finv", 0x2132 },
	{ "notarrowboth", 0x21ae },
	{ "archleftdown", 0x21b6 },
	{ "archrightdown", 0x21b7 },
	{ "notdblarrowleft", 0x21cd },
	{ "notdblarrowboth", 0x21ce },
	{ "notdblarrowright", 0x21cf },
	{ "epsiloninv", 0x220a },
	{ "equalorsimilar", 0x2242 },
	{ "notprecedes", 0x2280 },
	{ "notsubseteql", 0x2288 },
	{ "notsuperseteql", 0x2289 },
	{ "subsetnoteql", 0x228a },
	{ "supersetnoteql", 0x228b },
	{ "notturnstile", 0x22ac },
	{ "notforcesextra", 0x22af },
	{ "dividemultiply", 0x22c7 },
	{ "notprecedesoreql", 0x22e0 },
	{ "notfollowsoreql", 0x22e1 },
	{ "lessornotsimilar", 0x22e6 },
	{ "greaterornotsimilar", 0x22e7 },
	{ "precedeornoteqvlnt", 0x22e8 },
	{ "followornoteqvlnt", 0x22e9 },
	{ "nottriangeqlleft", 0x22ec },
	{ "nottriangeqlright", 0x22ed },
	{ "angbracketleft", 0x3008 },
	{ "angbracketright", 0x3009 },
	{ "check", 0x2713 },
	{ "circleS", 0x24c8 },
	{ "rightanglenw", 0x250c },
	{ "rightanglene", 0x2510 },
	{ "rightanglesw", 0x2514 },
	{ "rightanglese", 0x2518 },
	{ "Yen", 0x00a5 },
	{ "permill", 0x2030 },
	{ "recipe", 0x211e },
	{ "pertenmill", 0x2031 },
	{ "lnot", 0x00ac },
	{ "circleP", 0x2117 },
	{ "surd", 0x221a },
	{ "asciigrave", 0x60 },
	{ "asciigrave", 0x60 },
	{ "asciiacute", 0xb4 },
	{ "asciidieresis", 0xa8 },
	{ "pilcrow", 0xb6 },
/* Useful names to use to find glyphs */
	{ "circumflexcomb", 0x302 },
	{ "macroncomb", 0x304 },
	{ "brevecomb", 0x306 },
	{ "diaeresiscomb", 0x308 },
	{ "caroncomb", 0x30c },
	{ "vrachycomb", 0x306 },
	{ "psilicomb", 0x313 },
	{ "dasiacomb", 0x314 },
	{ "psilivaria", 0x1fcd },
	{ "psilioxia", 0x1fce },
	{ "psiliperispomeni", 0x1fcf },
	{ "dasiavaria", 0x1fdd },
	{ "dasiaoxia", 0x1fde },
	{ "dasiaperispomeni", 0x1fdf },
	{ "dialytikavaria", 0x1fed },
	{ "dialytikaoxia", 0x1fee },
	{ "dialytikaperispomeni", 0x1fc1 },
	NULL
};
