//
// File: <usb_desc.c>
//
// Written by: David M. Stanhope [voip@fobbit.com]
//
// Under NetBSD and others that use the same stack, could use the file
// '/usr/include/dev/usb/usb.h' but under Linux the only standard
// system file that has this info is <linux/usb.h> but only includes the
// info if __KERNEL__ is defined so must dup it here, and since have it
// may as well use it for all operating systems, at least for these functions.
//

#include "vblast.h"

struct usb_device_descriptor
{
    u_int8_t  bLength            ;
    u_int8_t  bDescriptorType    ;
    u_int16_t bcdUSB             ;
    u_int8_t  bDeviceClass       ;
    u_int8_t  bDeviceSubClass    ;
    u_int8_t  bDeviceProtocol    ;
    u_int8_t  bMaxPacketSize     ;
    u_int16_t idVendor           ;
    u_int16_t idProduct          ;
    u_int16_t bcdDevice          ;
    u_int8_t  iManufacturer      ;
    u_int8_t  iProduct           ;
    u_int8_t  iSerialNumber      ;
    u_int8_t  bNumConfigurations ;
};

struct usb_config_descriptor
{
    u_int8_t  bLength            ;
    u_int8_t  bDescriptorType    ;
    u_int16_t wTotalLength       ;
    u_int8_t  bNumInterface      ;
    u_int8_t  bConfigurationValue;
    u_int8_t  iConfiguration     ;
    u_int8_t  bmAttributes       ;
    u_int8_t  bMaxPower          ;
};

struct usb_interface_descriptor
{
    u_int8_t  bLength            ;
    u_int8_t  bDescriptorType    ;
    u_int8_t  bInterfaceNumber   ;
    u_int8_t  bAlternateSetting  ;
    u_int8_t  bNumEndpoints      ;
    u_int8_t  bInterfaceClass    ;
    u_int8_t  bInterfaceSubClass ;
    u_int8_t  bInterfaceProtocol ;
    u_int8_t  iInterface         ;
};

struct usb_endpoint_descriptor
{
    u_int8_t  bLength            ;
    u_int8_t  bDescriptorType    ;
    u_int8_t  bEndpointAddress   ;
    u_int8_t  bmAttributes       ;
    u_int16_t wMaxPacketSize     ;
    u_int8_t  bInterval          ;
};


// validate minimal device descriptor
int
check_is_vblaster(struct usb_device_descriptor *dp)
{
    // validate minimum info in device descriptor to check if voip-blaster

    if((dp->bLength         !=     18) ||
       (dp->bDescriptorType !=      1) ||
       (dp->idVendor        != 0x1292) ||
       (dp->idProduct       != 0x0258))
    {
        return 0; // not a voip-blaster
    }

    return 1; // is a voip-blaster
}

static void
bad_value(char *type, char *vname, int got, int wanted)
{
    ERR(("For <%s> in <%s> descriptor, got<%d> wanted<%x>\n",
                                   vname, type, got, wanted))
}

#define check_desc(a,b) \
    if(p->a != b) { bad_value(type, #a, p->a, b); return 0; }

// validate full device descriptor
int
check_desc_device(void *v)
{
    char *type = "device";

    struct usb_device_descriptor *p = (struct usb_device_descriptor *) v;

    check_desc(bLength           ,     18)
    check_desc(bDescriptorType   ,      1)
    check_desc(bcdUSB            , 0x0100) // USB 1.0
    check_desc(bDeviceClass      ,      0) // no class here
    check_desc(bDeviceSubClass   ,      0) // no sub-class here
    check_desc(bDeviceProtocol   ,      0) // no specific protocol
    check_desc(bMaxPacketSize    ,     16) // size for endpoing 0
    check_desc(idVendor          , 0x1292) // vendor id
    check_desc(idProduct         , 0x0258) // vendor product id
    check_desc(bcdDevice         , 0x0100) // device release 1.0
    check_desc(iManufacturer     ,      0) // no strings
    check_desc(iProduct          ,      0) // no strings
    check_desc(iSerialNumber     ,      0) // no strings
    check_desc(bNumConfigurations,      1) // only 1 configuration

    MSG(("Device Ok!\n"))

    return p->bLength; // is a voip-blaster
}

int
check_desc_config(void *v)
{
    char *type = "config";

    struct usb_config_descriptor *p = (struct usb_config_descriptor *) v;

    check_desc(bLength            ,    9)
    check_desc(bDescriptorType    ,    2)
    check_desc(wTotalLength       ,   46)
    check_desc(bNumInterface      ,    1) // only 1 interface
    check_desc(bConfigurationValue,    1) // for SetConfig
    check_desc(iConfiguration     ,    0) // no strings
    check_desc(bmAttributes       , 0x80) // self powered
    check_desc(bMaxPower          ,  240) // (240 * 2) = 480 MA

    MSG(("Configuration Ok!\n"))

    return p->bLength; // is a voip-blaster
}

int
check_desc_interface(void *v)
{
    char *type = "interface";

    struct usb_interface_descriptor *p = (struct usb_interface_descriptor *) v;

    check_desc(bLength           , 9)
    check_desc(bDescriptorType   , 4)
    check_desc(bInterfaceNumber  , 0) // interface number
    check_desc(bAlternateSetting , 0) // no alternate
    check_desc(bNumEndpoints     , 4) // 4 endpoints
    check_desc(bInterfaceClass   , 0) // no class info
    check_desc(bInterfaceSubClass, 0) // no subclass info
    check_desc(bInterfaceProtocol, 0) // no class protocol
    check_desc(iInterface        , 0) // no strings

    MSG(("Interface Ok!\n"))

    return p->bLength; // is a voip-blaster
}

int
check_desc_endpoint(void *v, int i)
{
    char *type = "endpoint";

    struct usb_endpoint_descriptor *p = (struct usb_endpoint_descriptor *) v;

    u_int8_t addr[4] = { 0x02, 0x82, 0x01, 0x81 };
    u_int8_t size[4] = {   64,   20,    1,    1 };

    check_desc(bLength         ,       7)
    check_desc(bDescriptorType ,       5)
    check_desc(bEndpointAddress, addr[i])
    check_desc(bmAttributes    ,       3) // all are 'interrupt'
    check_desc(wMaxPacketSize  , size[i])
    check_desc(bInterval       ,      30)

    MSG(("Addr(0x%02x:%s) Size(%04d)\n", p->bEndpointAddress,
                  p->bEndpointAddress & 0x80 ? "inp" : "out",
                                           p->wMaxPacketSize))

    return p->bLength; // is a voip-blaster
}

//
// The End!
//
