/*
 * Copyright (c) 2001 Mark Fullmer and The Ohio State University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *      $Id: flow-xlate.c,v 1.2 2001/07/02 05:01:55 maf Exp $
 */

#if HAVE_CONFIG_H
 #include <config.h>
#endif

#include <sys/types.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#if HAVE_STRINGS_H
 #include <strings.h>
#endif
#if HAVE_STRING_H
  #include <string.h>
#endif
#include <fcntl.h>
#include "ftlib.h"
#include "support.h"

int debug;
void usage();

#define XLATE_AS0                    0x1
#define XLATE_TO_LEGACY_NETWORK      0x2
#define XLATE_TO_FLOW_NETWORK        0x4
#define XLATE_PRIVACY_MASK           0x8
#define XLATE_SCALE_FLOWS            0x10
#define XLATE_SCALE_OCTETS           0x20
#define XLATE_SCALE_PACKETS          0x40
#define XLATE_SCALE_DURATION         0x80
#define XLATE_SCALE_ALL              (0x10 | 0x20 | 0x40 | 0x80)
#define XLATE_VERSION                0x100

struct xlate {
  struct ftipmask ftipmask;
  u_int32 flags;
  u_int32 scale;
  u_int32 privacy_mask;
  u_int16 as;
};

int main(int argc, char **argv)
{
  extern char *optarg;
  extern int optind, debug;
  struct ftio ftio_in, ftio_out;
  struct ftprof ftp;
  struct ftver ftv_in, ftv_out;
  struct ftset ftset;
  struct fts3rec_v5 *rec;
  struct xlate xlate;
  char out_rec[FT_IO_MAXREC];
  unsigned int v1, v2;
  int i, n;
  u_int32 total_flows, cap_start, cap_end;

  /* init fterr */
  fterr_setid(argv[0]);

  bzero(&ftv_in, sizeof ftv_in);
  bzero(&ftv_out, sizeof ftv_out);
  total_flows = 0;
  xlate.flags = 0;

  /* profile */
  ftprof_start (&ftp);

  /* defaults */
  ftset_init(&ftset, -1);

  while ((i = getopt(argc, argv, "0:b:C:d:fh?lm:s:V:z:")) != -1)

    switch (i) {

    case '0': /* AS */
      xlate.flags |= XLATE_AS0;
      xlate.as = atoi(optarg);
      break;

    case 'b': /* output byte order */
      if (!strcasecmp(optarg, "little"))
        ftset.byte_order = FT_HEADER_LITTLE_ENDIAN;
      else if (!strcasecmp(optarg, "big"))
        ftset.byte_order = FT_HEADER_BIG_ENDIAN;
      else
        fterr_errx(1, "expecting \"big\" or \"little\"");
      break;

    case 'C': /* comment field */
      ftset.comments = optarg;
      break;

    case 'd': /* debug */
      debug = atoi(optarg);
      break;

    case 'f': /* flow netmask */
      xlate.flags |= XLATE_TO_FLOW_NETWORK;
      break;

    case 'l': /* legacy netmask */
      xlate.flags |= XLATE_TO_LEGACY_NETWORK;
      break;

    case 'm': /* privacy_mask */
      xlate.privacy_mask = scan_ip(optarg);
      xlate.flags |= XLATE_PRIVACY_MASK;
      break;

    case 's': /* scale factor */
      xlate.flags |= XLATE_SCALE_FLOWS | XLATE_SCALE_OCTETS |
        XLATE_SCALE_PACKETS | XLATE_SCALE_DURATION;
      xlate.scale = atoi(optarg);
      break;

    case 'V': /* version */
     n = sscanf(optarg, "%u.%u", &v1, &v2);
      if (n == 1) {
        ftv_out.s_version = FT_IO_SVERSION;
        ftv_out.d_version = v1;
        ftv_out.set = 1;
      } else if (n == 2) {
        ftv_out.s_version = FT_IO_SVERSION;
        ftv_out.d_version = v1;
        ftv_out.agg_method = v2;
        ftv_out.agg_version = 2;
        ftv_out.set = 1;
      } else {
        fterr_errx(1, "Version scan failed");
      }
      xlate.flags |= XLATE_VERSION;
      break;


    case 'z': /* compress level */
      ftset.z_level = atoi(optarg);
      if ((ftset.z_level < 0) || (ftset.z_level > 9))
        fterr_errx(1, "Compression level must be between 0 and 9");
      break;

    case 'h': /* help */
    case '?': 
    default:
      usage();
      exit (1);
      break;

    } /* switch */

  i = optind;

  if (xlate.flags & XLATE_PRIVACY_MASK)
    /* setup for ftrec_mask_ip */
    ftrec_compute_mask(&xlate.ftipmask, xlate.privacy_mask, xlate.privacy_mask,
      ftset.byte_order);

  /* input is stdin */
  if (ftio_init(&ftio_in, 0, FT_IO_FLAG_READ) < 0)
    fterr_errx(1, "ftio_init(): failed");

  if (ftio_check_generic(&ftio_in) < 0) {
    fterr_errx(1, "Unsupported version");
  }

  /* output is stdout */
  if (ftio_init(&ftio_out, 1, FT_IO_FLAG_WRITE |
    ((ftset.z_level) ? FT_IO_FLAG_ZINIT : 0) ) < 0)
    fterr_errx(1, "ftio_init(): failed");

  ftio_get_ver(&ftio_in, &ftv_in);

  ftv_in.s_version = FT_IO_SVERSION;

  if (!ftv_out.set)
    bcopy(&ftv_in, &ftv_out, sizeof ftv_in);

  /* set the version information in the io stream */
  if (ftio_set_ver(&ftio_out, &ftv_out) < 0)
    fterr_errx(1, "ftio_set_ver(): failed");

  ftio_set_comment(&ftio_out, ftset.comments);
  ftio_set_byte_order(&ftio_out, ftset.byte_order);
  ftio_set_z_level(&ftio_out, ftset.z_level);
  ftio_set_streaming(&ftio_out, 1);
  ftio_set_debug(&ftio_out, debug);
  ftio_set_cap_time(&ftio_out, cap_start, cap_end);
  ftio_set_flows_count(&ftio_out, total_flows);

  if (ftio_write_header(&ftio_out) < 0)
    fterr_errx(1, "ftio_write_header(): failed");

  while ((rec = ftio_read(&ftio_in))) {

    if (xlate.flags & XLATE_TO_LEGACY_NETWORK) {

      if ((rec->srcaddr & 0x80000000) == 0)
        rec->srcaddr &= 0xFF000000;
      else if ((rec->srcaddr & 0xC0000000) == 0x80000000)
        rec->srcaddr &= 0xFFFF0000;
      else if ((rec->srcaddr & 0xC0000000) == 0xC0000000)
        rec->srcaddr &= 0xFFFFFF00;

      if ((rec->dstaddr & 0x80000000) == 0)
        rec->dstaddr &= 0xFF000000;
      else if ((rec->dstaddr & 0xC0000000) == 0x80000000)
        rec->dstaddr &= 0xFFFF0000;
      else if ((rec->dstaddr & 0xC0000000) == 0xC0000000)
        rec->dstaddr &= 0xFFFFFF00;

    }

    if (xlate.flags & XLATE_TO_FLOW_NETWORK) {

      if ((ftv_in.d_version == 5) || (ftv_in.d_version == 6) ||
        (ftv_in.d_version == 7)) {

        rec->srcaddr &= ipv4_len2mask(rec->src_mask);
        rec->dstaddr &= ipv4_len2mask(rec->dst_mask);

      }
    }

    if (xlate.flags & XLATE_AS0) {

      if ((ftv_in.d_version == 5) || (ftv_in.d_version == 6) ||
        (ftv_in.d_version == 7)) {

        if (rec->src_as == 0)
          rec->src_as = xlate.as;

        if (rec->dst_as == 0)
          rec->dst_as = xlate.as;

      }
    }

    if (xlate.flags & XLATE_PRIVACY_MASK)
      ftrec_mask_ip(rec, &ftv_in, &xlate.ftipmask);

    if (xlate.flags & XLATE_SCALE_OCTETS)
      rec->dOctets *= xlate.scale;

    if (xlate.flags & XLATE_SCALE_PACKETS)
      rec->dPkts *= xlate.scale;

    if (xlate.flags & XLATE_VERSION) {

      ftrec_xlate(rec, &ftv_in, &out_rec, &ftv_out);

      if (ftio_write(&ftio_out, &out_rec) < 0)
        fterr_errx(1, "ftio_write(): failed");

    } else {

      if (ftio_write(&ftio_out, rec) < 0)
        fterr_errx(1, "ftio_write(): failed");

    }

  } /* while */

  if (ftio_close(&ftio_out) < 0)
    fterr_errx(1, "ftio_close(): failed");

  if (ftio_close(&ftio_in) < 0)
    fterr_errx(1, "ftio_close(): failed");

  if (debug > 0) {
    ftprof_end (&ftp, total_flows);
    ftprof_print(&ftp, argv[0], stderr);
  }

  return 0;

} /* main */

void usage() {

  fprintf(stderr, "flow-xlate:\n\n");
  fprintf(stderr, " -0 #  replace AS0 with AS #\n");
  fprintf(stderr, " -b s  set output byte order (little or big)\n");
  fprintf(stderr, " -C s  set comment field in output\n");
  fprintf(stderr, " -d #  set debug level.\n");
  fprintf(stderr, " -l    mask src and dst addresses to legacy classful network addresses\n");
  fprintf(stderr, " -f    use flow netmask length to mask src and dst ip to network addresses\n");
  fprintf(stderr, " -s #  scale packets and octets by #\n");
  fprintf(stderr, " -m m  privacy mask, ie 255.255.255.0\n");
  fprintf(stderr, " -V #  version of export to translate to.\n");

  fprintf(stderr, " -z #  set compression level (0..9).\n");
  fprintf(stderr, "\n\n");
}

