/*
 * Copyright (c) 2001 Mark Fullmer and The Ohio State University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *      $Id: flow-print.c,v 1.14 2001/07/06 14:24:38 maf Exp $
 */

#if HAVE_CONFIG_H
 #include <config.h>
#endif

#include <sys/time.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <netinet/in.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#if HAVE_STRINGS_H
 #include <strings.h>
#endif
#if HAVE_STRING_H
  #include <string.h>
#endif
#include <time.h>
#include <fcntl.h>
#include "ftlib.h"
#include "fmt.h"

struct jump {
    int (*where)(struct ftio *ftio, int options);
};

int format0(struct ftio *ftio, int options);
int format1(struct ftio *ftio, int options);
int format2(struct ftio *ftio, int options);
int format3(struct ftio *ftio, int options);
int format4(struct ftio *ftio, int options);
int format5(struct ftio *ftio, int options);
int format6(struct ftio *ftio, int options);
int format7(struct ftio *ftio, int options);
int format8(struct ftio *ftio, int options);
int format9(struct ftio *ftio, int options);
int format10(struct ftio *ftio, int options);
int format11(struct ftio *ftio, int options);
int format12(struct ftio *ftio, int options);
int format13(struct ftio *ftio, int options);
int format14(struct ftio *ftio, int options);

struct jump format[] = {{format0}, {format1}, {format2},
          {format3}, {format4}, {format5}, {format6}, {format7},
          {format8}, {format9}, {format10}, {format11}, {format12},
          {format13}, {format14}, };

#define NFORMATS 15

void usage();

int main(argc, argv)
int argc;
char **argv;
{
  char cc; /* comment character */
  int i, format_index, set_format, ret;
  extern char *optarg;
  extern struct jump format[];
  int print_header;
  struct ftio ftio;
  struct ftprof ftp;
  int options, debug;

  /* init fterr */
  fterr_setid(argv[0]);

  options = 0;
  debug = 0;

  /* profile */
  ftprof_start (&ftp);

  set_format = 0;
  print_header = 0;
  cc = '#';

  while ((i = getopt(argc, argv, "ph?d:f:c:lnw")) != -1)
    switch (i) {

    case 'c': /* comment character */
      cc = optarg[0];
      break;

    case 'd': /* debug */
      debug = atoi(optarg);
      break;

    case 'f': /* format */
      format_index = atoi(optarg);
      set_format = 1;
      break;

    case 'h': /* help */
    case '?':
      usage();
      exit (0);
      break;

    case 'l': /* turn off buffered output */
      options |= FT_OPT_NOBUF;
      break;

    case 'n': /* symbolic names */
      options |= FT_OPT_NAMES;
      break;

    case 'p': /* print header */
      print_header = 1;
      break;

    case 'w': /* wide */
      options |= FT_OPT_WIDE;
      break;

    default:
      usage();
      exit (1);
      break;

    } /* switch */

  if (set_format && (format_index >= NFORMATS)) 
    fterr_errx(1, "No such format, %d\n", format_index);

  /* read from stdin */
  if (ftio_init(&ftio, 0, FT_IO_FLAG_READ) < 0)
    fterr_errx(1, "ftio_init(): failed");

  /* if the format was not set on the command line use a reasonable default */
  if (!set_format) {
    if (ftio.fth.d_version == 8) {
      if (ftio.fth.agg_method == 1)
        format_index = 10;
      else if (ftio.fth.agg_method == 2)
        format_index = 11;
      else if (ftio.fth.agg_method == 3)
        format_index = 12;
      else if (ftio.fth.agg_method == 4)
        format_index = 13;
      else if (ftio.fth.agg_method == 5)
        format_index = 14;
    } else if (ftio.fth.d_version == 7) {
      format_index = 7;
    } else if (ftio.fth.d_version == 6) {
      format_index = 8;
    } else
      format_index = 3;
  } /* !set_format */

  if (print_header) {
    ftio_header_print(&ftio, stdout, cc);
  }

  ret = format[format_index].where(&ftio, options);

  if ((!ret) && (debug > 0)) {
    ftprof_end(&ftp, ftio_get_rec_total(&ftio));
    ftprof_print(&ftp, argv[0], stderr);
  }

  return ret;

} /* main */

void usage() {

  fprintf(stderr, "flow-print:\n\n");
  fprintf(stderr, " -d # set debug level.\n");
  fprintf(stderr, " -f # format\n");
  fprintf(stderr, "    0 {1,5,7} 1 line, interfaces, hex ports\n");
  fprintf(stderr, "    1 {1,5,7} 2 line (includes timing and flags)\n");
  fprintf(stderr, "    2 {1,5,7} 2 line candidate TCP syn attack flows\n");
  fprintf(stderr, "    3 {1,5,7} 1 line, no interfaces, decimal ports\n");
  fprintf(stderr, "    4 {5,7}   1 line with AS number\n");
  fprintf(stderr, "    5 {1,5,7} 1 line, 132 column\n");
  fprintf(stderr, "    6 {1,5,7} show ip accounting emulation\n");
  fprintf(stderr, "    7 {7}     1 line, 132 column +router_id\n");
  fprintf(stderr, "    8 {6}     1 line, 132 column +encapsulation\n");
  fprintf(stderr, "   10 {8.1}   AS aggregation\n");
  fprintf(stderr, "   11 {8.2}   Protocol Port aggregation\n");
  fprintf(stderr, "   12 {8.3}   Source Prefix aggregation\n");
  fprintf(stderr, "   13 {8.4}   Destination Prefix aggregation\n");
  fprintf(stderr, "   14 {8.5}   Prefix aggregation\n");
  fprintf(stderr, " -l no buffered output\n");
  fprintf(stderr, " -n use symbolic names\n");
  fprintf(stderr, " -p print header\n");
  fprintf(stderr, " -w wide output\n");
  fprintf(stderr, "\n\n");

} /* usage */


/*
 * function: format0
 *
 * 1 line summary
*/
int format0(struct ftio *ftio, int options)
{
  struct fts3rec_gen *rec;
  char fmt_buf1[64], fmt_buf2[64];

  if (ftio_check_generic(ftio) < 0)
    return -1;

  puts("Sif  SrcIPaddress     Dif  DstIPaddress      Pr SrcP DstP  Pkts       Octets");

  while ((rec = ftio_read(ftio))) {

    fmt_ipv4(fmt_buf1, rec->srcaddr, FMT_PAD_RIGHT);
    fmt_ipv4(fmt_buf2, rec->dstaddr, FMT_PAD_RIGHT);

    printf("%4.4x %-15.15s  %4.4x %-15.15s   %2.2x %-4x %-4x  %-10lu %-10lu\n",
      (int)rec->input, fmt_buf1, (int)rec->output, fmt_buf2,
      (int)rec->prot, (int)rec->srcport, (int)rec->dstport,
      (u_long)rec->dPkts, (u_long)rec->dOctets);

    if (options & FT_OPT_NOBUF)
      fflush(stdout);

  } /* while */

  return 0;

} /* format0 */


/*
 * function: format1
 *
 * 2 line summary
*/
int format1(struct ftio *ftio, int options)
{
  struct fts3rec_gen *rec;
  struct fttime ftt;
  char fmt_buf1[64], fmt_buf2[64];
  struct tm *tm;
  u_long active_secs, active_msecs;
  u_long bpp;

  if (ftio_check_generic(ftio) < 0)
    return -1;

  puts(
    "Sif  SrcIPaddress     DIf  DstIPaddress      Pr SrcP DstP  Pkts  Octets");
  puts(
    " StartTime          EndTime             Active   B/Pk Ts Fl\n");

  while ((rec = ftio_read(ftio))) {

    fmt_ipv4(fmt_buf1, rec->srcaddr, FMT_PAD_RIGHT);
    fmt_ipv4(fmt_buf2, rec->dstaddr, FMT_PAD_RIGHT);

    printf("%4.4x %-15.15s  %4.4x %-15.15s   %2.2x %-4x %-4x  %-10lu %-10lu\n",
      (int)rec->input, fmt_buf1, (int)rec->output, fmt_buf2,
      (int)rec->prot, (int)rec->srcport, (int)rec->dstport,
      (u_long)rec->dPkts, (u_long)rec->dOctets);

    ftt = ftltime(rec->sysUpTime, rec->unix_secs, rec->unix_nsecs, rec->First);
    tm = localtime((time_t*)&ftt.secs);

    printf(" %-2.2d%-2.2d.%-2.2d:%-2.2d:%-2.2d.%-3lu  ",
      (int)tm->tm_mon+1, (int)tm->tm_mday, (int)tm->tm_hour,
      (int)tm->tm_min, (int)tm->tm_sec, (u_long)ftt.msecs);

    ftt = ftltime(rec->sysUpTime, rec->unix_secs, rec->unix_nsecs, rec->Last);
    tm = localtime((time_t*)&ftt.secs);

    active_secs = (rec->Last - rec->First) / 1000;
    active_msecs = (rec->Last - rec->First) % 1000;

    bpp = rec->dOctets / rec->dPkts;

    printf("%-2.2d%-2.2d.%-2.2d:%-2.2d:%-2.2d.%-3lu  %5lu.%-3lu %-3lu %2.2x %2.2x\n\n",
      (int)tm->tm_mon+1, (int)tm->tm_mday, (int)tm->tm_hour,
      (int)tm->tm_min, (int)tm->tm_sec, (u_long)ftt.msecs,
      active_secs, active_msecs, bpp, (int)rec->tos,
      (int)rec->tcp_flags);

    if (options & FT_OPT_NOBUF)
      fflush(stdout);

  } /* while */

  return 0;

} /* format1 */


/*
 * function: format2
 *
 * only print flows that are TCP with only a SYN bit set and
 * a single packet
*/
int format2(struct ftio *ftio, int options)
{
  struct fts3rec_gen *rec;
  struct fttime ftt;
  char fmt_buf1[64], fmt_buf2[64];
  struct tm *tm;
  u_long active_secs, active_msecs;
  u_long bpp;

  if (ftio_check_generic(ftio) < 0)
    return -1;

  puts("Sif SrcIPaddress     DIf DstIPaddress    Pr SrcP DstP Pkts       Octets");
  puts(" StartTime          EndTime             Active   B/Pk Ts Fl\n");

  while ((rec = ftio_read(ftio))) {

    /* If it's not TCP */
    if (rec->prot != IPPROTO_TCP)
      continue;

    /* If more than the SYN bit is set */
    if (rec->tcp_flags != 2)
      continue;

    /* many SYN bit only packets per flow are suspect */
    if (rec->dPkts != 2)
      continue;

    /* 40 byte datagrams are the output of the current tool */
    if (rec->dOctets != (rec->dPkts * 40))
      continue;

    fmt_ipv4(fmt_buf1, rec->srcaddr, FMT_PAD_RIGHT);
    fmt_ipv4(fmt_buf2, rec->dstaddr, FMT_PAD_RIGHT);

    printf("%2.2x  %-15.15s  %2.2x  %-15.15s %2.2x %-4x %-4x %-10lu %-10lu\n",
      (int)rec->input, fmt_buf1, (int)rec->output, fmt_buf2,
      (int)rec->prot, (int)rec->srcport, (int)rec->dstport,
      (u_long)rec->dPkts, (u_long)rec->dOctets);

    ftt = ftltime(rec->sysUpTime, rec->unix_secs, rec->unix_nsecs, rec->First);
    tm = localtime((time_t*)&ftt.secs);

    printf(" %-2.2d%-2.2d.%-2.2d:%-2.2d:%-2.2d.%-3lu  ",
      (int)tm->tm_mon+1, (int)tm->tm_mday, (int)tm->tm_hour, (int)tm->tm_min,
      (int)tm->tm_sec, (u_long)ftt.msecs);

    ftt = ftltime(rec->sysUpTime, rec->unix_secs, rec->unix_nsecs, rec->Last);
    tm = localtime((time_t*)&ftt.secs);

    active_secs = (rec->Last - rec->First) / 1000;
    active_msecs = (rec->Last - rec->First) % 1000;

    bpp = rec->dOctets / rec->dPkts;

    printf("%-2.2d%-2.2d.%-2.2d:%-2.2d:%-2.2d.%-3lu  %5lu.%-3lu %-3lu %2.2x %2.2x\n\n",
      (int)tm->tm_mon+1, (int)tm->tm_mday, (int)tm->tm_hour,
      (int)tm->tm_min, (int)tm->tm_sec, (u_long)ftt.msecs, active_secs,
      active_msecs, bpp, (int)rec->tos, (int)rec->tcp_flags);

    if (options & FT_OPT_NOBUF)
      fflush(stdout);

  } /* while */

  return 0;

} /* format2 */


/*
 * function: format3
 *
 * another 1 line format
 */
int format3(struct ftio *ftio, int options)
{
  struct fts3rec_gen *rec;
  struct ftsym *sym_tcp, *sym_prot;
  char fmt_buf1[64], fmt_buf2[64], fmt_buf3[64], fmt_buf4[64], fmt_buf5[64];

  if (ftio_check_generic(ftio) < 0)
    return -1;

  sym_tcp = sym_prot = (struct ftsym*)0L;

  if (options & FT_OPT_NAMES) {
    sym_tcp = ftsym_new(FT_FILE_TCP_PORT);
    sym_prot = ftsym_new(FT_FILE_IP_PROT);
  }

  if (options & FT_OPT_WIDE)
    puts("srcIP            dstIP            prot   srcPort           dstPort           octets      packets");
  else
    puts("srcIP            dstIP            prot  srcPort  dstPort  octets      packets");

  while ((rec = ftio_read(ftio))) {

    fmt_ipv4(fmt_buf1, rec->srcaddr, FMT_PAD_RIGHT);

    fmt_ipv4(fmt_buf2, rec->dstaddr, FMT_PAD_RIGHT);

    fmt_uint16s(sym_prot, 5, fmt_buf3, (u_int16)rec->prot, FMT_PAD_RIGHT);

    fmt_uint16s(sym_tcp, 16, fmt_buf4, (u_int16)rec->srcport, FMT_PAD_RIGHT);

    fmt_uint16s(sym_tcp, 16, fmt_buf5, (u_int16)rec->dstport, FMT_PAD_RIGHT);

    if (options & FT_OPT_WIDE)
      printf("%-15.15s  %-15.15s  %-5.5s  %-16.16s  %-16.16s  %-10lu  %-10lu\n",
        fmt_buf1, fmt_buf2, fmt_buf3, fmt_buf4, fmt_buf5,
        (u_long)rec->dOctets, (u_long)rec->dPkts);
    else
      printf("%-15.15s  %-15.15s  %-4.4s  %-7.7s  %-7.7s  %-10lu  %-10lu\n",
        fmt_buf1, fmt_buf2, fmt_buf3, fmt_buf4, fmt_buf5,
        (u_long)rec->dOctets, (u_long)rec->dPkts);

    if (options & FT_OPT_NOBUF)
      fflush(stdout);

  } /* while */

  ftsym_free(sym_tcp);
  ftsym_free(sym_prot);

  return 0;

} /* format3 */


/*
 * function: format4
 *
*/
int format4(struct ftio *ftio, int options)
{
  struct fts3rec_v5 *rec;
  struct ftsym *sym_prot, *sym_asn;
  char fmt_buf1[64], fmt_buf2[64], fmt_buf3[64], fmt_buf4[64], fmt_buf5[64];
  struct ftver ftv;

  ftio_get_ver(ftio, &ftv);

  if ((ftv.d_version != 5) && (ftv.d_version != 7) && (ftv.d_version != 6)) {
    fterr_warnx("AS information not present in export version");
    return -1;
  }

  sym_prot = sym_asn = (struct ftsym*)0L;

  if (options & FT_OPT_NAMES) {
    sym_prot = ftsym_new(FT_FILE_IP_PROT);
    sym_asn = ftsym_new(FT_FILE_ASN);
  }


  if (options & FT_OPT_WIDE)
    puts("srcIP               dstIP               prot   srcAS             dstAS             octets      packets");
  else
    puts("srcIP              dstIP              prot  srcAS  dstAS  octets      packets");

  while ((rec = ftio_read(ftio))) {

    fmt_ipv4prefix(fmt_buf1, rec->srcaddr, rec->src_mask, FMT_JUST_LEFT);

    fmt_ipv4prefix(fmt_buf2, rec->dstaddr, rec->dst_mask, FMT_JUST_LEFT);

    fmt_uint16s(sym_prot, 5, fmt_buf3, (u_int16)rec->prot, FMT_PAD_RIGHT);

    fmt_uint16s(sym_asn, 18, fmt_buf4, (u_int16)rec->src_as, FMT_JUST_LEFT);

    fmt_uint16s(sym_asn, 18, fmt_buf5, (u_int16)rec->dst_as, FMT_JUST_LEFT);

    if (options & FT_OPT_WIDE)
      printf("%-18.18s  %-18.18s  %-5.5s  %-16.16s  %-16.16s  %-10lu  %-10lu\n",
        fmt_buf1, fmt_buf2, fmt_buf3, fmt_buf4, fmt_buf5,
        (u_long)rec->dOctets, (u_long)rec->dPkts);
    else
      printf("%-18.18s %-18.18s %-4.4s  %-5.5s  %-5.5s  %-10lu  %-10lu\n",
        fmt_buf1, fmt_buf2, fmt_buf3, fmt_buf4, fmt_buf5,
        (u_long)rec->dOctets, (u_long)rec->dPkts);

    if (options & FT_OPT_NOBUF)
      fflush(stdout);

  } /* while */

  ftsym_free(sym_prot);
  ftsym_free(sym_asn);

  return 0;

} /* format4 */


/*
 * function: format5
 *
 * 1 line summary, steve's favorite
*/
int format5(struct ftio *ftio, int options)
{
  struct fts3rec_gen *rec;
  struct fttime ftt;
  char fmt_buf1[64], fmt_buf2[64];
  struct tm *tm;

  if (ftio_check_generic(ftio) < 0)
    return -1;

puts("Start             End               Sif   SrcIPaddress    SrcP  DIf   DstIPaddress    DstP    P Fl Pkts       Octets\n");

  while ((rec = ftio_read(ftio))) {

    ftt = ftltime(rec->sysUpTime, rec->unix_secs, rec->unix_nsecs, rec->First);
    tm = localtime((time_t*)&ftt.secs);

    printf("%-2.2d%-2.2d.%-2.2d:%-2.2d:%-2.2d.%-3lu ",
      (int)tm->tm_mon+1, (int)tm->tm_mday, (int)tm->tm_hour,
      (int)tm->tm_min, (int)tm->tm_sec, (u_long)ftt.msecs);

    ftt = ftltime(rec->sysUpTime, rec->unix_secs, rec->unix_nsecs, rec->Last);
    tm = localtime((time_t*)&ftt.secs);

    printf("%-2.2d%-2.2d.%-2.2d:%-2.2d:%-2.2d.%-3lu ",
      (int)tm->tm_mon+1, (int)tm->tm_mday, (int)tm->tm_hour,
      (int)tm->tm_min, (int)tm->tm_sec, (u_long)ftt.msecs);

    /* other info */
    fmt_ipv4(fmt_buf1, rec->srcaddr, FMT_PAD_RIGHT);
    fmt_ipv4(fmt_buf2, rec->dstaddr, FMT_PAD_RIGHT);

    printf("%-5d %-15.15s %-5d %-5d %-15.15s %-5d %-3d %-2d %-10lu %-10lu\n",
/*    printf("%4d %-15.15s %6d %4d %-15.15s %6d %3d %2d %10lu %10lu\n", */

           (int)rec->input, fmt_buf1, (int)rec->srcport, 
           (int)rec->output, fmt_buf2, (int)rec->dstport,
           (int)rec->prot, 
           (int)rec->tcp_flags & 0x7,
           (u_long)rec->dPkts, 
           (u_long)rec->dOctets);

    if (options & FT_OPT_NOBUF)
      fflush(stdout);

  } /* while */

  return 0;

} /* format5 */

/*
 * function: format6
 *
 * 1 line summary, similar to `show ip accounting`
*/
int format6(struct ftio *ftio, int options)
{
  struct fts3rec_gen *rec;
  char fmt_buf1[64], fmt_buf2[64];

  if (ftio_check_generic(ftio) < 0)
    return -1;

  puts(
    "   Source           Destination              Packets               Bytes");

  while ((rec = ftio_read(ftio))) {

    fmt_ipv4(fmt_buf1, rec->srcaddr, FMT_PAD_RIGHT);

    fmt_ipv4(fmt_buf2, rec->dstaddr, FMT_PAD_RIGHT);

    printf(" %-15.15s  %-15.15s  %17lu  %18lu\n",
      fmt_buf1, fmt_buf2, (u_long)rec->dPkts, (u_long)rec->dOctets);

    if (options & FT_OPT_NOBUF)
      fflush(stdout);

  } /* while */

  return 0;

} /* format6 */

/*
 * function: format7
 *
 * 1 line, 132 column -- includes router_sc from v7 format
 */
int format7(struct ftio *ftio, int options)
{
  struct fts3rec_v7 *rec;
  char fmt_buf1[64], fmt_buf2[64], fmt_buf3[64], fmt_buf4[64], fmt_buf5[64];
  char fmt_buf6[64];
  struct ftsym *sym_tcp, *sym_prot;

  /* check for version 7 format */
  if (ftio->fth.d_version != 7) {
    fterr_warnx("PDU Version 7 flows only");
    return -1;
  }

  sym_tcp = sym_prot = (struct ftsym*)0L;
 
  if (options & FT_OPT_NAMES) {
    sym_tcp = ftsym_new(FT_FILE_TCP_PORT);
    sym_prot = ftsym_new(FT_FILE_IP_PROT);
  }


  puts("srcIP               dstIP               router_sc        prot   srcPort         dstPort         octets      packets");

  while ((rec = ftio_read(ftio))) {

    fmt_ipv4prefix(fmt_buf1, rec->srcaddr, rec->src_mask, FMT_PAD_RIGHT);

    fmt_ipv4prefix(fmt_buf2, rec->dstaddr, rec->dst_mask, FMT_PAD_RIGHT);

    fmt_ipv4(fmt_buf3, rec->router_sc, FMT_PAD_RIGHT);

    fmt_uint16s(sym_prot, 5, fmt_buf4, (u_int16)rec->prot, FMT_PAD_RIGHT);

    fmt_uint16s(sym_tcp, 16, fmt_buf5, (u_int16)rec->srcport, FMT_PAD_RIGHT);

    fmt_uint16s(sym_tcp, 16, fmt_buf6, (u_int16)rec->dstport, FMT_PAD_RIGHT);

    printf("%-18.18s  %-18.18s  %-15.15s  %-5.5s  %-14.14s  %-14.14s  %-10lu  %-10lu\n",
      fmt_buf1, fmt_buf2, fmt_buf3, fmt_buf4, fmt_buf5, fmt_buf6,
      (u_long)rec->dOctets, (u_long)rec->dPkts);

    if (options & FT_OPT_NOBUF)
      fflush(stdout);

  } /* while */

  ftsym_free(sym_tcp);
  ftsym_free(sym_prot);

  return 0;

} /* format7 */

/*
 * function: format8
 *
 * 1 line, 132 column -- includes encapsulation size from v6 format
 */
int format8(struct ftio *ftio, int options)
{
  struct fts3rec_v6 *rec;
  struct ftsym *sym_tcp, *sym_prot;
  char fmt_buf1[64], fmt_buf2[64], fmt_buf3[64], fmt_buf4[64], fmt_buf5[64];
  char fmt_buf6[64];

  /* check for version 6 format */
  if (ftio->fth.d_version != 6) {
    fterr_warnx("PDU Version 6 flows only");
    return -1;
  }

  sym_tcp = sym_prot = (struct ftsym*)0L;
  
  if (options & FT_OPT_NAMES) {
    sym_tcp = ftsym_new(FT_FILE_TCP_PORT);
    sym_prot = ftsym_new(FT_FILE_IP_PROT);
  } 

  puts("srcIP               dstIP               peer_nexthop     encap i/o  prot   srcPort         dstPort         octets      packets");

  while ((rec = ftio_read(ftio))) {
    fmt_ipv4prefix(fmt_buf1, rec->srcaddr, rec->src_mask, FMT_PAD_RIGHT);

    fmt_ipv4prefix(fmt_buf2, rec->dstaddr, rec->dst_mask, FMT_PAD_RIGHT);

    fmt_ipv4(fmt_buf3, rec->peer_nexthop, FMT_PAD_RIGHT);

    fmt_uint16s(sym_prot, 5, fmt_buf4, (u_int16)rec->prot, FMT_PAD_RIGHT);

    fmt_uint16s(sym_tcp, 16, fmt_buf5, (u_int16)rec->srcport, FMT_PAD_RIGHT);

    fmt_uint16s(sym_tcp, 16, fmt_buf6, (u_int16)rec->dstport, FMT_PAD_RIGHT);

    printf("%-18.18s  %-18.18s  %-15.15s  %-4d %-4d  %-5.5s  %-14.14s  %-14.14s  %-10lu  %-10lu\n",
      fmt_buf1, fmt_buf2, fmt_buf3, (int)rec->in_encaps, (int)rec->out_encaps,
      fmt_buf4, fmt_buf5, fmt_buf6, (u_long)rec->dOctets, (u_long)rec->dPkts);

    if (options & FT_OPT_NOBUF)
      fflush(stdout);

  } /* while */

  ftsym_free(sym_tcp);
  ftsym_free(sym_prot);

  return 0;

} /* format8 */

/*
 * reserved
*/
int format9(struct ftio *ftio, int options)
{
  fterr_warnx("reserved");
  return -1;
}

/*
 * AS aggregation
*/
int format10(struct ftio *ftio, int options)
{
  struct fts3rec_v8_1 *rec;
  struct ftsym *sym_asn;
  char fmt_buf1[64], fmt_buf2[64];
  char fmt_buf3[32], fmt_buf4[32];
  char fmt_buf5[32], fmt_buf6[32];
  char fmt_buf7[32], fmt_buf8[32];

  if ((ftio->fth.d_version != 8) || (ftio->fth.agg_method != 1) || 
    (ftio->fth.agg_version != 2)) {
    fterr_warnx("PDU Version 8, Aggregation Method 1 Version 2 flows only");
    return -1;
  }

  sym_asn = (struct ftsym*)0L;
        
  if (options & FT_OPT_NAMES) {
    sym_asn = ftsym_new(FT_FILE_ASN);
  }

  if (options & FT_OPT_WIDE)
    puts("srcAS             dstAS             in     out    flows       octets      packets     duration");
  else
    puts(
"srcAS  dstAS  in     out    flows       octets      packets     duration");

   while ((rec = ftio_read(ftio))) {

    fmt_uint16s(sym_asn, 18, fmt_buf1, (u_int16)rec->src_as, FMT_JUST_LEFT);
    fmt_uint16s(sym_asn, 18, fmt_buf2, (u_int16)rec->dst_as, FMT_JUST_LEFT);
    fmt_uint16(fmt_buf3, rec->input, FMT_JUST_LEFT);
    fmt_uint16(fmt_buf4, rec->output, FMT_JUST_LEFT);
    fmt_uint32(fmt_buf5, rec->dFlows, FMT_JUST_LEFT);
    fmt_uint32(fmt_buf6, rec->dOctets, FMT_JUST_LEFT);
    fmt_uint32(fmt_buf7, rec->dPkts, FMT_JUST_LEFT);
    fmt_uint32(fmt_buf8, rec->Last - rec->First, FMT_JUST_LEFT);

    if (options & FT_OPT_WIDE)
      printf(
        "%-16.16s  %-16.16s  %-7.7s%-7.7s%-12.12s%-12.12s%-12.12s%-12.12s\n",
        fmt_buf1, fmt_buf2, fmt_buf3, fmt_buf4, fmt_buf5, fmt_buf6, fmt_buf7,
        fmt_buf8);
    else
      printf("%-5.5s  %-5.5s  %-7.7s%-7.7s%-12.12s%-12.12s%-12.12s%-12.12s\n",
        fmt_buf1, fmt_buf2, fmt_buf3, fmt_buf4, fmt_buf5, fmt_buf6, fmt_buf7,
        fmt_buf8);

    if (options & FT_OPT_NOBUF)
      fflush(stdout);

  }

  ftsym_free(sym_asn);

  return 0;

} /* format10 */

/*
 * Protocol Port aggregation
*/
int format11(struct ftio *ftio, int options)
{
  struct fts3rec_v8_2 *rec;
  struct ftsym *sym_tcp, *sym_prot;
  char fmt_buf1[32], fmt_buf2[32];
  char fmt_buf3[32], fmt_buf4[32];
  char fmt_buf5[32], fmt_buf6[32];
  char fmt_buf7[32];

  if ((ftio->fth.d_version != 8) || (ftio->fth.agg_method != 2) ||
    (ftio->fth.agg_version != 2)) {
    fterr_warnx("PDU Version 8, Aggregation Method 2 Version 2 flows only");
    return -1;
  }

  sym_tcp = sym_prot = (struct ftsym*)0L;

  if (options & FT_OPT_NAMES) {
    sym_tcp = ftsym_new(FT_FILE_TCP_PORT);
    sym_prot = ftsym_new(FT_FILE_IP_PROT);
  }

  if (options & FT_OPT_WIDE)
    puts(
"srcPort           dstPort           prot   flows       octets      packets     duration");
  else
    puts(
"srcPort  dstPort  prot   flows       octets      packets     duration");

   while ((rec = ftio_read(ftio))) {

    fmt_uint16s(sym_tcp, 16, fmt_buf1, (u_int16)rec->srcport, FMT_PAD_RIGHT);
  
    fmt_uint16s(sym_tcp, 16, fmt_buf2, (u_int16)rec->dstport, FMT_PAD_RIGHT);

    fmt_uint16s(sym_prot, 5, fmt_buf3, (u_int16)rec->prot, FMT_PAD_RIGHT);

    fmt_uint32(fmt_buf4, rec->dFlows, FMT_JUST_LEFT);

    fmt_uint32(fmt_buf5, rec->dOctets, FMT_JUST_LEFT);

    fmt_uint32(fmt_buf6, rec->dPkts, FMT_JUST_LEFT);

    fmt_uint32(fmt_buf7, rec->Last - rec->First, FMT_JUST_LEFT);

    if (options & FT_OPT_WIDE)
      printf("%-16.16s  %-16.16s  %-5.5s  %-12.12s%-12.12s%-12.12s%-12.12s\n",
        fmt_buf1, fmt_buf2, fmt_buf3, fmt_buf4, fmt_buf5, fmt_buf6, fmt_buf7);
    else
      printf("%-7.7s  %-7.7s  %-5.5s  %-12.12s%-12.12s%-12.12s%-12.12s\n",
        fmt_buf1, fmt_buf2, fmt_buf3, fmt_buf4, fmt_buf5, fmt_buf6, fmt_buf7);

    if (options & FT_OPT_NOBUF)
      fflush(stdout);

  }

  ftsym_free(sym_tcp);
  ftsym_free(sym_prot);

  return 0;

}

/*
 * Source Prefix aggregation
 */
int format12(struct ftio *ftio, int options)
{
  struct fts3rec_v8_3 *rec;
  struct ftsym *sym_asn;
  char fmt_buf1[32];
  char fmt_buf3[32], fmt_buf4[32];
  char fmt_buf5[32], fmt_buf6[32];
  char fmt_buf7[32], fmt_buf8[32];

  if ((ftio->fth.d_version != 8) || (ftio->fth.agg_method != 3) ||
    (ftio->fth.agg_version != 2)) {
    fterr_warnx("PDU Version 8, Aggregation Method 3 Version 2 flows only");
    return -1;
  }

  sym_asn = (struct ftsym*)0L;

  if (options & FT_OPT_NAMES) {
    sym_asn = ftsym_new(FT_FILE_ASN);
  }

  if (options & FT_OPT_WIDE)
    puts(
"srcPrefix           srcAS             input  flows       octets      packets     duration");
  else
    puts(
"src/mask            srcAS  input  flows       octets      packets     duration");

   while ((rec = ftio_read(ftio))) {

    fmt_ipv4prefix(fmt_buf1, rec->src_prefix, rec->src_mask, FMT_JUST_LEFT);

    fmt_uint16s(sym_asn, 18, fmt_buf3, rec->src_as, FMT_JUST_LEFT);

    fmt_uint16(fmt_buf4, rec->input, FMT_JUST_LEFT);

    fmt_uint32(fmt_buf5, rec->dFlows, FMT_JUST_LEFT);

    fmt_uint32(fmt_buf6, rec->dOctets, FMT_JUST_LEFT);

    fmt_uint32(fmt_buf7, rec->dPkts, FMT_JUST_LEFT);

    fmt_uint32(fmt_buf8, rec->Last - rec->First, FMT_JUST_LEFT);

    if (options & FT_OPT_WIDE)
      printf("%-18.18s  %-16.16s  %-7.7s%-12.12s%-12.12s%-12.12s%-12.12s\n",
        fmt_buf1, fmt_buf3, fmt_buf4, fmt_buf5, fmt_buf6, fmt_buf7, fmt_buf8);
    else
      printf("%-18.18s  %-5.5s  %-7.7s%-12.12s%-12.12s%-12.12s%-12.12s\n",
        fmt_buf1, fmt_buf3, fmt_buf4, fmt_buf5, fmt_buf6, fmt_buf7, fmt_buf8);

    if (options & FT_OPT_NOBUF)
      fflush(stdout);

  } /* while */

  ftsym_free(sym_asn);

  return 0;

}

/*
 * Destination Prefix aggregation
*/
int format13(struct ftio *ftio, int options)
{
  struct fts3rec_v8_4 *rec;
  struct ftsym *sym_asn;
  char fmt_buf1[32];
  char fmt_buf3[32], fmt_buf4[32];
  char fmt_buf5[32], fmt_buf6[32];
  char fmt_buf7[32], fmt_buf8[32];

  if ((ftio->fth.d_version != 8) || (ftio->fth.agg_method != 4) ||
    (ftio->fth.agg_version != 2)) {
    fterr_warnx("PDU Version 8, Aggregation Method 4 Version 2 flows only");
    return -1;
  }

  sym_asn = (struct ftsym*)0L;

  if (options & FT_OPT_NAMES) {
    sym_asn = ftsym_new(FT_FILE_ASN);
  }

  if (options & FT_OPT_WIDE)
    puts(
"dstPrefix           dstAS             input  flows       octets      packets     duration");
  else
    puts(
"dst/mask            dstAS  input  flows       octets      packets     duration");

  while ((rec = ftio_read(ftio))) {

    fmt_ipv4prefix(fmt_buf1, rec->dst_prefix, rec->dst_mask, FMT_JUST_LEFT);

    fmt_uint16s(sym_asn, 18, fmt_buf3, rec->dst_as, FMT_JUST_LEFT);

    fmt_uint16(fmt_buf4, rec->output, FMT_JUST_LEFT);

    fmt_uint32(fmt_buf5, rec->dFlows, FMT_JUST_LEFT);

    fmt_uint32(fmt_buf6, rec->dOctets, FMT_JUST_LEFT);

    fmt_uint32(fmt_buf7, rec->dPkts, FMT_JUST_LEFT);

    fmt_uint32(fmt_buf8, rec->Last - rec->First, FMT_JUST_LEFT);

    if (options & FT_OPT_WIDE)
      printf("%-18.18s  %-16.16s  %-7.7s%-12.12s%-12.12s%-12.12s%-12.12s\n",
        fmt_buf1, fmt_buf3, fmt_buf4, fmt_buf5, fmt_buf6, fmt_buf7, fmt_buf8);
    else
      printf("%-18.18s  %-5.5s  %-7.7s%-12.12s%-12.12s%-12.12s%-12.12s\n",
        fmt_buf1, fmt_buf3, fmt_buf4, fmt_buf5, fmt_buf6, fmt_buf7, fmt_buf8);

    if (options & FT_OPT_NOBUF)
      fflush(stdout);

  } /* while */

  ftsym_free(sym_asn);

  return 0;
}

/*
 * Prefix aggregation
 */
int format14(struct ftio *ftio, int options)
{
  struct fts3rec_v8_5 *rec;
  struct ftsym *sym_asn;
  char fmt_buf1[32], fmt_buf2[32];
  char fmt_buf3[32], fmt_buf4[32];
  char fmt_buf5[32], fmt_buf6[32];
  char fmt_buf7[32], fmt_buf8[32];
  char fmt_buf9[32], fmt_buf10[32];

  if ((ftio->fth.d_version != 8) || (ftio->fth.agg_method != 5) ||
    (ftio->fth.agg_version != 2)) {
    fterr_warnx("PDU Version 8, Aggregation Method 5 Version 2 flows only");
    return -1;
  }

  sym_asn = (struct ftsym*)0L;

  if (options & FT_OPT_NAMES) {
    sym_asn = ftsym_new(FT_FILE_ASN);
  }

  if (options & FT_OPT_WIDE)
    puts(
"srcPrefix           srcAS             dstPrefix           dstAS             input  output flows       octets      packets     duration");
  else
    puts(
"srcPrefix           srcAS  dstPrefix           dstAS  input  output flows       octets      packets     duration");

  while ((rec = ftio_read(ftio))) {

    fmt_ipv4prefix(fmt_buf1, rec->src_prefix, rec->src_mask, FMT_JUST_LEFT);

    fmt_ipv4prefix(fmt_buf3, rec->dst_prefix, rec->dst_mask, FMT_JUST_LEFT);

    fmt_uint16s(sym_asn, 18, fmt_buf2, rec->src_as, FMT_JUST_LEFT);

    fmt_uint16s(sym_asn, 18, fmt_buf4, rec->dst_as, FMT_JUST_LEFT);

    fmt_uint16(fmt_buf5, rec->input, FMT_JUST_LEFT);
    fmt_uint16(fmt_buf6, rec->output, FMT_JUST_LEFT);

    fmt_uint32(fmt_buf7, rec->dFlows, FMT_JUST_LEFT);
    fmt_uint32(fmt_buf8, rec->dOctets, FMT_JUST_LEFT);
    fmt_uint32(fmt_buf9, rec->dPkts, FMT_JUST_LEFT);
    fmt_uint32(fmt_buf10, rec->Last - rec->First, FMT_JUST_LEFT);

    if (options & FT_OPT_WIDE)
    printf("%-18.18s  %-16.16s  %-18.18s  %-16.16s  %-7.7s%-7.7s%-12.12s%-12.12s%-12.12s%-12.12s\n",
      fmt_buf1, fmt_buf2, fmt_buf3, fmt_buf4, fmt_buf5, fmt_buf6,
      fmt_buf7, fmt_buf8, fmt_buf9, fmt_buf10);
    else
    printf("%-18.18s  %-5.5s  %-18.18s  %-5.5s  %-7.7s%-7.7s%-12.12s%-12.12s%-12.12s%-12.12s\n",
      fmt_buf1, fmt_buf2, fmt_buf3, fmt_buf4, fmt_buf5, fmt_buf6,
      fmt_buf7, fmt_buf8, fmt_buf9, fmt_buf10);

    if (options & FT_OPT_NOBUF)
      fflush(stdout);

  } /* while */

  ftsym_free(sym_asn);

  return 0;

}

