/*
 * Copyright (c) 2001 Mark Fullmer and The Ohio State University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *      $Id: flow-fanout.c,v 1.18 2001/07/15 19:16:16 maf Exp $
 */

#if HAVE_CONFIG_H
 #include <config.h>
#endif

#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/errno.h>
#include <sys/uio.h>
#include <signal.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <syslog.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#if HAVE_STRINGS_H
 #include <strings.h>
#endif
#if HAVE_STRING_H
  #include <string.h>
#endif
#include <sys/time.h>
#include "ftlib.h"
#include "support.h"

#define FANOUT_PIDFILE    "/var/run/flow-fanout.pid"

#define SELECT_TIMEOUT 5   /* 5 seconds */


int debug;

void usage();

struct peer {
  int fd;
  struct sockaddr_in loc_addr; /* us */
  struct sockaddr_in rem_addr; /* them */
  int port;
  int ttl;
};

void fterr_exit_handler(int code);

int sig_quit_flag, sig_hup_flag;
void sig_quit(), sig_hup();
pid_t pid;
u_int16 listen_port;

int main(int argc, char **argv)
{
  extern char *optarg;
  extern int optind, errno;
  struct sockaddr_in tmp_addr;
#ifdef IP_ADD_MEMBERSHIP
  struct ip_mreq mr;
#ifdef IP_ADD_SOURCE_MEMBERSHIP
  struct ip_mreq_source mrs;
#endif
#endif
  struct timeval tv;
  fd_set rfd;
  struct ftpeeri ftpi;
  struct ftpdu ftpdu;
  struct ftver ftv;
  struct ftseq ftseq;
  int i, len, n, detach;
  int npeers, tx_delay;
  struct peer *peers;
  u_int32 flows_corrupt, flows_lost, flows_reset;

  /* init fterr */
  fterr_setid(argv[0]);
  fterr_setexit(fterr_exit_handler);

  bzero(&ftpdu, sizeof ftpdu);
  bzero(&ftv, sizeof ftv);
  bzero(&ftseq, sizeof ftseq);

  flows_corrupt = flows_lost = flows_reset = 0;

  debug = 0;
  tx_delay = 0;
  detach = 1;

  while ((i = getopt(argc, argv, "d:Dhx:")) != -1)
    switch (i) {

    case 'd': /* debug */
      debug = atoi(optarg);
      break;

    case 'D': /* detach */
      detach = 0;
      break;

    case 'h': /* help */
      usage();
      exit (0);
      break;

    case 'x': /* transmit delay */
      tx_delay = atoi(optarg);
      break;

    default:
      usage();
      exit (1);
      break;

    } /* switch */

  /* allocate argc - optind peer entries */
  npeers = argc - optind;

  if (npeers <= 1)
    fterr_errx(1, "Must define at least source and one destination.");

  /* pre-scan so write_pidfile has the port */
  ftpi = scan_peeri(argv[optind]);

  listen_port = (ftpi.dst_port) ? ftpi.dst_port : FT_PORT;

  if (signal(SIGQUIT, sig_quit) < 0)
    fterr_err(1, "signal()");

 /* daemonize */
  if (detach) {  
    if ((pid = fork()) == -1) {
      fterr_err(1, "fork()");
    } else if (pid) {
      write_pidfile(pid, FANOUT_PIDFILE, listen_port);
      exit (0); /* parent */
    }
   
    chdir ("/"); 
    umask(0022);
    setsid();
    for (n = 0; n < 16; ++n) /* XXX dynamically get NOFILE */
      close (n);
    
    /* enable syslog */
    fterr_setsyslog(1, LOG_PID|LOG_NDELAY, LOG_LOCAL6);

    /* disable stderr */
    fterr_setfile(0, (void*)0L);

  }

  if (!(peers = (struct peer*)malloc(npeers * sizeof (struct peer))))
    fterr_err(1, "malloc()");

  /* zero out malloc'd memory */
  bzero(peers, npeers*sizeof(struct peer));

  /* fill in peer entries */
  /* listen on entry 0, send on all other entries */
  for (i = optind, n = 0; i < argc; ++i, ++n) {

    /* parse loc_ip/rem_ip/port/ttl */
    ftpi = scan_peeri(argv[i]);

    if (n == 0) { /* listener */
      if (ftpi.dst_port)
        peers[n].loc_addr.sin_port = htons(ftpi.dst_port);
      else
        peers[n].loc_addr.sin_port = htons(FT_PORT);
    } else {
      if (ftpi.dst_port)
        peers[n].rem_addr.sin_port = htons(ftpi.dst_port);
      else
        peers[n].rem_addr.sin_port = htons(FT_PORT);
    }

    peers[n].loc_addr.sin_family = AF_INET;
    peers[n].rem_addr.sin_family = AF_INET;
    peers[n].ttl = ftpi.ttl;

    /* reverse remote and local if multicast */
    if ((n == 0) && (IN_CLASSD(ftpi.rem_ip))) {

      peers[n].loc_addr.sin_addr.s_addr = htonl(ftpi.rem_ip);
      peers[n].rem_addr.sin_addr.s_addr = htonl(ftpi.loc_ip);

    } else {

      peers[n].rem_addr.sin_addr.s_addr = htonl(ftpi.rem_ip);
      peers[n].loc_addr.sin_addr.s_addr = htonl(ftpi.loc_ip);

    }

    if ((peers[n].fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
      fterr_err(1, "socket()");

    if (bigsockbuf(peers[n].fd, (n ? SO_SNDBUF : SO_RCVBUF), 224*1024) < 0)
      fterr_err(1, "bigsockbuf()");

  }

  /* setup to receive flows */

/* multicast capable? */
#ifdef IP_ADD_MEMBERSHIP

  if (IN_CLASSD(ntohl(peers[0].loc_addr.sin_addr.s_addr))) {

    int itmp = 1;

    /* socket API usually requires INADDR_ANY
     * and s/g/port identifier does not have a source interface field
     * to use here
     */

    bzero(&tmp_addr, sizeof tmp_addr);
    tmp_addr.sin_family = AF_INET;
    tmp_addr.sin_port = peers[0].loc_addr.sin_port;

    /* Multicast streams may have multiple receivers */
    if (setsockopt(peers[0].fd, SOL_SOCKET, SO_REUSEADDR, (char *)&itmp,
      sizeof(itmp)) < 0)
      fterr_err(1, "setsockopt(SO_REUSEADDR)");

    if (bind(peers[0].fd, (struct sockaddr*)&tmp_addr,
      sizeof(struct sockaddr)) < 0)
      fterr_err(1, "bind(mcast-rcv)");

/* ssm capable? */
#ifdef IP_ADD_SOURCE_MEMBERSHIP

    /* ssm address? */
    if (IN_CLASSD_SSM(ntohl(peers[0].loc_addr.sin_addr.s_addr))) {

      mrs.imr_sourceaddr.s_addr = peers[0].rem_addr.sin_addr.s_addr;
      mrs.imr_multiaddr.s_addr = peers[0].loc_addr.sin_addr.s_addr;
      mrs.imr_interface.s_addr = INADDR_ANY;

      if (setsockopt(peers[0].fd, IPPROTO_IP, IP_ADD_SOURCE_MEMBERSHIP,
        (char*)&mrs, sizeof(mrs)) < 0)
        fterr_err(1, "setsockopt(IP_ADD_SOURCE_MEMBERSHIP)");

    }

    goto mcast_done;
                  
#endif /* IP_ADD_SOURCE_MEMBERSHIP */

    mr.imr_multiaddr.s_addr = peers[0].loc_addr.sin_addr.s_addr;
    mr.imr_interface.s_addr = INADDR_ANY;

    if (setsockopt(peers[0].fd, IPPROTO_IP, IP_ADD_MEMBERSHIP,
      (char *)&mr, sizeof(mr)) < 0)
      fterr_err(1, "setsockopt(IP_ADD_MEMBERSHIP)");

  
  } else { /* is a multicast group */

    /* unicast bind -- multicast support */
    if (bind(peers[0].fd, (struct sockaddr*)&peers[0].loc_addr,
      sizeof(struct sockaddr)) < 0)
      fterr_err(1, "bind(rcv)");

  } /* not multicast group */

#ifdef IP_ADD_SOURCE_MEMBERSHIP
mcast_done:
#endif /* IP_ADD_SOURCE_MEMBERSHIP */

#else /* IP_ADD_MEMBERSHIP */

  /* unicast bind -- no multicast support */
  if (bind(peers[0].fd, (struct sockaddr*)&peers[0].loc_addr,
    sizeof(struct sockaddr)) < 0)
    fterr_err(1, "bind(rcv)");

#endif /* IP_ADD_MEMBERSHIP */

  len = sizeof(struct sockaddr);

  /* connect to destinations */
  for (n = 1; n < npeers; ++n) {

#ifdef IP_ADD_MEMBERSHIP

    /* multicast destination? */
    if (IN_CLASSD(ntohl(peers[n].rem_addr.sin_addr.s_addr))) {

      u_char ttl = peers[n].ttl;

      /* set the ttl */
      if (setsockopt(peers[n].fd, IPPROTO_IP, IP_MULTICAST_TTL,
        (char*)&ttl, sizeof(ttl)) < 0) {
        fterr_err(1, "setsockopt(IP_MULTICAST_TTL=%d)", ttl);
      }

    } /* multicast */

#endif /* IP_ADD_MEMBERSHIP */

    if (bind(peers[n].fd, (struct sockaddr*)&peers[n].loc_addr,
      sizeof(struct sockaddr)) < 0)
      fterr_err(1, "bind(xmit)");

    if (connect(peers[n].fd, (struct sockaddr*)&peers[n].rem_addr,
      sizeof(struct sockaddr)) < 0)
      fterr_err(1, "connect(xmit)");

  }

  bzero (&tv, sizeof tv);
  tv.tv_sec = SELECT_TIMEOUT;

  while (1) {

    if (sig_quit_flag) {
      fterr_info("SIGQUIT");
      break;
    }

    if (sig_hup_flag) {
      fterr_info("SIGHUP");
      sig_hup_flag = 0;
    }

    FD_ZERO (&rfd);
    FD_SET (peers[0].fd, &rfd);

    if (select (peers[0].fd+1, &rfd, (fd_set *)0, (fd_set *)0, &tv) < 0)
      fterr_err(1, "select()");

    bzero (&tv, sizeof tv);
    tv.tv_sec = SELECT_TIMEOUT;

    if (sig_quit_flag)
      goto main_exit;

    if (FD_ISSET(peers[0].fd, &rfd)) {

      if ((ftpdu.bused = recvfrom(peers[0].fd, (char*)&ftpdu.buf,
        sizeof ftpdu.buf, 0, (struct sockaddr*) &tmp_addr, &len)) == -1)
        fterr_err(1, "recvfrom()");

      /* verify source of sender */
      if (peers[0].rem_addr.sin_addr.s_addr &&
        (peers[0].rem_addr.sin_addr.s_addr != tmp_addr.sin_addr.s_addr)) {
        fterr_warnx("src IP (%s) not configured.  Skipping PDU",
          inet_ntoa(tmp_addr.sin_addr));
        continue;
      }

      /* verify integrity, get version */
      if (ftpdu_verify(&ftpdu) < 0) {
        fterr_warnx("ftpdu_verify(): failed.  Skipping PDU");
        flows_corrupt ++;
        continue;
      }

      /* first time? */
      if (!ftv.set) {

        /* copy to compare next time */
        bcopy(&ftpdu.ftv, &ftv, sizeof ftv);

        /* flag struct as configured */
        ftv.set = 1;

        fterr_info("PDU format detected: d=%d agg=%d",
            (int)ftv.d_version, (int)ftv.agg_method);

      } else {

        /* check previous version == current decoded pdu version */
        if ((ftv.d_version != ftpdu.ftv.d_version) ||
            (ftv.agg_method != ftpdu.ftv.agg_method)) {
          
          fterr_info("PDU format changed: od=%d d=%d oagg=%d agg=%d",
            (int)ftv.d_version, (int)ftpdu.ftv.d_version,
            (int)ftv.agg_method, (int)ftpdu.ftv.agg_method);

          /* update it */
          bcopy(&ftpdu.ftv, &ftv, sizeof ftv);

        }

      }

      if (ftpdu_check_seq(&ftpdu, &ftseq) < 0) {
        fterr_warnx(
          "ftpdu_seq_check(): fail, expecting=%lu received=%lu lost=%lu",
          (u_long)ftseq.seq_exp, (u_long)ftseq.seq_rcv,
          (u_long)ftseq.seq_lost);
    
        /* only count these if lost if "lost" is a reasonable number */
        if (ftseq.seq_lost < FT_SEQ_RESET)
          flows_lost += ftseq.seq_lost;
        else
          flows_reset ++;
      }

      for (i = 1; i < npeers; ++i) {

        if (send(peers[i].fd, &ftpdu.buf, ftpdu.bused, 0) < 0)
          if (errno != ECONNREFUSED)
            fterr_warn("send(i=%d)", i);

        if (tx_delay)
          usleep((unsigned)tx_delay);

      }

    } /* if FD_ISSET */

  } /* while 1 */

main_exit:

  unlink_pidfile(pid, FANOUT_PIDFILE, listen_port);

  for (n = 0; n < npeers; ++n)
    close(peers[0].fd);

  return 0;

} /* main */

void usage() {

  fprintf(stderr, "usage: flow-fanout [-p #] [-h] [-x #] A.B.C.D/A.B.C.D/port\n   A.B.C.D/A.B.C.D/port/ttl ...\n\n");
  fprintf(stderr, "    -d #  set debug level.\n");
  fprintf(stderr, "    -h help.\n");
  fprintf(stderr, "    -x #  transmit delay in microseconds.\n");
  fprintf(stderr, "\n");

} /* usage */

void sig_quit()
{
  sig_quit_flag = 1;
}

void sig_hup()
{
  sig_hup_flag = 1;
}

void fterr_exit_handler(int code)
{
  if (pid)
    unlink_pidfile(pid, FANOUT_PIDFILE, listen_port);
  exit (code);
} /* fterr_exit_handler */
 

