/*
 * Copyright (c) 2001 Mark Fullmer and The Ohio State University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *      $Id: support.c,v 1.20 2001/07/15 19:19:30 maf Exp $
 */

#if HAVE_CONFIG_H
 #include <config.h>
#endif

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/errno.h>
#include <ctype.h>
#include <limits.h>
#include <netdb.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <syslog.h>
#include <time.h>
#include <fcntl.h>
#if HAVE_STRINGS_H
 #include <strings.h>
#endif
#if HAVE_STRING_H
  #include <string.h>
#endif
#include "support.h"
#include "bytes.h"
#include "ftlib.h"

/* max stack size for sorting */
#define SORT_STACK  65536*2

/*
 * lookup table for mask length to mask
 *
 *               (first 8)
 *  128.0.0.0 192.0.0.0 224.0.0.0 240.0.0.0
 *  248.0.0.0 252.0.0.0 254.0.0.0 255.0.0.0
 *
 */
static u_int32 mask_lookup[] = { 0xffffffff,
     0x80000000, 0xc0000000, 0xe0000000, 0xf0000000,
     0xf8000000, 0xfc000000, 0xfe000000, 0xff000000,
     0xff800000, 0xffc00000, 0xffe00000, 0xfff00000,
     0xfff80000, 0xfffc0000, 0xfffe0000, 0xffff0000,
     0xffff8000, 0xffffc000, 0xffffe000, 0xfffff000,
     0xfffff800, 0xfffffc00, 0xfffffe00, 0xffffff00,
     0xffffff80, 0xffffffc0, 0xffffffe0, 0xfffffff0,
     0xfffffff8, 0xfffffffc, 0xfffffffe, 0xffffffff };

/*
 * function ipv4_len2mask
 *
 * returns the 32 bit network mask given a length
 *
*/
u_int32 ipv4_len2mask(u_int8 len)
{
  return mask_lookup[(len > 32) ? 0 : len];
}

/*
 * function: load_lookup
 *
 *  loads a list of , seperated numbers into an array
 *  ! will invert the list
 *  - can be used as a range operator
 *
 *  example
 *   1,5-10   == 1,5,6,7,8,9,10
 *   !1       == all numbers in the range except for 1
 */
int load_lookup(char *s, int size, char *list)
{
  char *p;
  int j, k;
  unsigned i, i2;

  p = s;

  while ((*p == ' ') || (*p == '\t')) ++p;
  if (*p == '!') {
    for (k = 0; k < size; ++k)
      list[k] = 1;
    k = 0;
    ++p;
  } else {
    for (k = 0; k < size; ++k)
      list[k] = 0;
    k = 1;
  }

  while (*p) {

    i = (unsigned)strtol(p, (char**)0L, 0);
    if (i > size) return -1;
    list[i] = k;

    /* skip to , or - */
    while (*p && (*p != ',') && (*p != '-')) ++p;

    if (*p == '-') {

      ++p;
      i2 = (unsigned)strtol(p, (char**)0L, 0);
      if (i2 > size) return -1;
      for (j = i; j <= i2; ++j) list[j] = k;

      /* skip to , or - */
      while (*p && (*p != ',') && (*p != '-')) ++p;
    }

    /* skip past , and - */
    while (*p && ((*p == ',') || (*p == '-'))) ++p;

  } /* *p */

  return 0;

} /* load_lookup */

/*
 * function: scan_peeri
 *
 * scan peer identifier
 *
 * scan 1.2.3.4/1.2.3.4/nn[/nnl]
 *      locip   remip   port  ttl
 * into ftpeer struct
 */
struct ftpeeri scan_peeri(char *input)
{
  struct ftpeeri ftpi;
  char *s, *s2, *locip, *remip, *dstport, *ttl;

  bzero (&ftpi, sizeof ftpi);
  ftpi.dst_port = FT_PORT;

  locip = remip = dstport = ttl = (char*)0L;

  if (!(s = malloc(strlen(input+1)))) {
    fterr_warn("malloc");
    return ftpi;
  }

  /* keep track of original pointer to free */
  s2 = s;

  strcpy(s, input);

  locip = s;
  if (*s) {
  }

  for (; *s && *s != '/'; ++s);
  if (*s) {
    *s = 0;
    remip = ++s;
  }

  for (; *s && *s != '/'; ++s);
  if (*s) {
    *s = 0;
    dstport = ++s;
  }

  for (; *s && *s != '/'; ++s);
  if (*s) {
    *s = 0;
    ttl = ++s;
  }

  if (locip)
    ftpi.loc_ip = scan_ip(locip);
  if (remip)
    ftpi.rem_ip = scan_ip(remip);
  if (dstport)
    ftpi.dst_port = atoi(dstport);
  if (ttl)
    ftpi.ttl = atoi(ttl);

  free (s2);

  return ftpi;
  
} /* scan_peer */

/*
 * function: scan_ip
 *
 *  IP address in string S is converted to a u_long
 *  (borrowed from tcpdump)
 *
 */
u_long scan_ip(char *s)
{
  struct hostent *he;
  struct in_addr *ina;
  u_long addr = 0;
  u_int n;
  int dns;
  char *t;

  /* if there is anything ascii in here, this may be a hostname */
  for (dns = 0, t = s; *t; ++t) {
    if (islower((int)*t) || isupper((int)*t)) {
      dns = 1;
      break;
    }
  }

  if (dns) {

    if (!(he = gethostbyname(s)))
      goto numeric;

    if (he->h_addrtype != AF_INET)
      goto numeric;

    if (he->h_length != sizeof (u_int32))
      goto numeric;

    ina = (struct in_addr*)*he->h_addr_list;
    return (ntohl(ina->s_addr));

  } /* dns */

numeric:
  while (1) {

    /* n is the nibble */
    n = 0;

    /* nibble's are . bounded */
    while (*s && (*s != '.') && (*s != ' ') && (*s != '\t'))
      n = n * 10 + *s++ - '0';

    /* shift in the nibble */
    addr <<=8;
    addr |= n & 0xff;

    /* return on end of string */
    if ((!*s) || (*s == ' ') || (*s == '\t'))
      return addr;

    /* skip the . */
    ++s;
  } /* forever */
} /* scan_ip */


/*
 * function: print_3float
 *
 *  format a floating point # to stdout w. 1 trailing space
 *
 */
void print_3float(float f)
{

  char s[10], *c;
  sprintf(s, "%-3.3f", f);
  c = s + 1;
  printf("%s ", c);

} /* print_3float */

/*
 * function: print_3float2
 *
 *  format a floating point # to stdout w. 2 trailing spaces
 *
 */
void print_3float2(float f)
{

  char s[10], *c;
  sprintf(s, "%-3.3f", f);
  c = s + 1;
  printf("%s  ", c);

} /* print_3float */

/*
 *  function: sort_64uint64
 *
 *  a   an array of up to 65536 u_int64's
 *  index an array of up to 65536 indexes
 *  elem  # of elements in a and index
 *
 *  uses quicker sort to arrange the index by comparing
 *  the values the index points to in a.
 *
 */
int sort_64uint64(u_int64 *a, u_int *index, u_int elem)
{
  long i, l, r, j;
  u_int stack[SORT_STACK];
  u_int sp;
  u_int t;
  u_int64 v;

  l = 0;
  r = elem - 1;
  sp = SORT_STACK;

  for (;;) {

    while (r > l) {

      /* partition */
      v = a[index[r]];
      i = l-1;
      j = r;

      for (;;) {
        while (a[index[++i]] < v);
        while (a[index[--j]] > v);
        if (i >= j) break;
        t = index[i]; index[i] = index[j]; index[j] = t;
      }

      t = index[i]; index[i] = index[r]; index[r] = t;

      if (i-l > r-i) {
        stack[--sp] = l;
        stack[--sp] = i-1;
        l = i+1;
      } else {
        stack[--sp] = i+1;
        stack[--sp] = r;
        r = i-1;
      }
    } /* while */

    if (sp == SORT_STACK)
      break;

    if (!sp) {
      fterr_errx(1, "sort_64uint64(): stack overflow");
    }

    r = stack[sp++];
    l = stack[sp++];

  } /* for */

  return 0;

} /* sort_64uint64 */


/* adapted from dd */

int64 scan_size(char *val)
{
  u_int64 num, t;
  char *expr;

  if ((num = strtoul(val, &expr, 0)) == ULONG_MAX)
    goto erange;

  switch(*expr) {

    case 0:
      break;

    case 'b':
      t = num;
      num *= 512;
      if (t > num)
        goto erange;
      break;
    case 'G':
      t = num;
      num *= 1024;
      num *= 1024;
      num *= 1024;
      if (t > num)
        goto erange;
      break;
    case 'K':
      t = num;
      num *= 1024;
      if (t > num)
        goto erange;
      break;
    case 'M':
      t = num;
      num *= 1024;
      num *= 1024;
      if (t > num)
        goto erange;
      break;
    default:
      goto erange;
  }

  return num;

erange: 

  return (int64)-1;

} /* scan_size */

#ifdef POSIX_SIGNALS

/*
 * Function: mysignal()
 *  POSIX style signals.
 *
 *  signal() has different semantics over different versions of unix.
 *  this emulates signal() with sigaction() to behave like BSD.
 *
 * From Stevens Advanced Programming in the UNIX environment 
 *
 */
void *mysignal(int signo, void *func)
{
  struct sigaction act, oact;
  
  act.sa_handler = (void*)func;
  sigemptyset(&act.sa_mask);
  act.sa_flags = 0;

  if (signo == SIGALRM) {
#ifdef  SA_INTERRUPT
  act.sa_flags |= SA_INTERRUPT; /* SunOS */
#endif
  } else {
#ifdef SA_RESTART
  act.sa_flags |= SA_RESTART; /* SVR4, 4.3+BSD */
#endif
  }

  if (sigaction(signo, &act, &oact) < 0)
    return SIG_ERR;

  return oact.sa_handler;
} /* signal */

#endif /* POSIX_SIGNALS */


int unlink_pidfile(int pid, char *file, u_int16 port)
{
  char *c;
  int ret;

  if (!(c = (char*)malloc(strlen(file)+16)))
    return -1;

  sprintf(c, "%s.%d", file, (int)port);

  if ((ret = unlink(c)) < 0)
    fterr_warn("unlink(%s)", c);

  free (c);

  return ret;

} /* unlink_pidfile */
 

int write_pidfile(int pid, char *file, u_int16 port)
{
  int fd, len;
  char str[16], *c;
  
  if (!(c = (char*)malloc(strlen(file)+16)))
    return -1;

  sprintf(c, "%s.%d", file, (int)port);
    
  len = sprintf(str, "%u\n", (unsigned)pid);

  if ((fd = open(c, O_WRONLY|O_CREAT|O_TRUNC, 0644)) < 0 ) {
    fterr_warn("open(%s)", c);
    free (c);
    return -1; 
  }


  if (write(fd, str, len) != len) {
    fterr_warn("write(%s)", c);
    close (fd);
    free (c);
    return -1;
  }

  return close (fd);

} /* write_pidfile */


/*
 * function get_gmtoff
 *
 * return offset from GMT in seconds
 *
 * based on compute_tz() code by Michael R. Elkins
 */
int get_gmtoff(time_t t) 
{
  struct tm *tmp, local, gmt;
  time_t t2;
  int yday;
  tmp = gmtime(&t);
  bcopy(tmp, &gmt, sizeof gmt);

  tmp = localtime(&t);
  bcopy(tmp, &local, sizeof local);

  /* calculate difference in seconds */
  t2 = (local.tm_hour - gmt.tm_hour) * 60; /* to minutes */
  t2 += (local.tm_min - gmt.tm_min); /* minutes */
  t2 *= 60; /* to seconds */

  /* diff day */
  yday = (local.tm_yday - gmt.tm_yday);

  if ((yday == -1) || (yday > 1))
    t2 -= 86400; /* sub one day */
  else if (yday != 0)
    t2 += 86400; /* add one day */

  return t2;

} /* get_gmtoff */

/*
 * function: bigsockbuf
 *
 * There is no portable way to determine the max send and receive buffers
 * that can be set for a socket, so guess then decrement that guess by
 * 512 until the call succeeds.
 *
 * returns size or -1 for error
*/

int bigsockbuf(int fd, int dir, int size)
{
  extern int errno;
  int n;

  /* initial size */
  n = size;

  while (n > 512) {
    if (setsockopt(fd, SOL_SOCKET, dir, (char*)&n, sizeof (n)) < 0) {

      /* anything other than no buffers available is fatal */
      if (errno != ENOBUFS) {
        fterr_warn("setsockopt(size=%d)", n);
        return -1;
      }

      /* try a smaller value */
      n -= 512;

    } else
      break;
  }

  return n;

} /* bigsockbuf */

