#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "hash_ip2.h"
#include "fmt.h"

#define OPT_PERCENT 1 /* use percent's */

/* init data structures */
int hash_ip2_init(ht)
struct hash_ip2_table *ht;
{

  struct hash_ip2_rec *rec;

  ht->head.seg = 1; /* 0 is for the end of chain marker */
  ht->head.off = 0;

  /* calculate the offset that the last record can go on */
  ht->lastoff = 65536 - (65536 % sizeof(struct hash_ip2_rec)) -
    sizeof(struct hash_ip2_rec);

  /* allocate the hash table */
  if (!(ht->table = (struct hash_ip2_addr*)malloc(
    sizeof(struct hash_ip2_addr)*65536))) {
    return -1;
  }

  bzero(ht->table, (65536*sizeof(struct hash_ip2_addr)));

  /* the 0 index is wasted here, the address seg=0, off=0 is a marker */
  if (!(ht->data = (char**)malloc(sizeof(char*)*2))) {
    free (ht->table);
    return -1;
  }

  /* allocate the first block of data records */
  if (!(ht->data[1] = (char*)malloc(65536))) {
    free (ht->table);
    free (ht->data);
    return -1;
  }

  /* only need to make sure the first block doesn't have invalid
    next pointer */
  rec = (struct hash_ip2_rec*)(ht->data[1]);
  rec->next.seg = 0;


  return 0;

} /* hash_ip2_table */

/* free data structures */
int hash_ip2_free(ht)
struct hash_ip2_table *ht;
{
  int x;

  if (ht->table)
    free (ht->table);

  if (ht->data) {
    for (x = 0; x < ht->head.seg; ++x)
      free (ht->data[x]);
    free (ht->data);
  }

  return 0;

} /* hash_ip2_free */

/* update or add entry */
/* 0  =  allready existed, updated
/* 1  =  collision, added new
/* 2  =  no collision, added new
/* <0 =  error
*/
int hash_ip2_update(ht, newrec)
struct hash_ip2_table *ht;
struct hash_ip2_rec *newrec;
{

  u_long hash;
  struct hash_ip2_rec *rec;
  struct hash_ip2_addr addr, addr2;

  /* check if the record exists in the table */
  hash =  (newrec->src_addr>>16) ^ (newrec->src_addr & 0xFFFF) ^
      (newrec->dst_addr>>16) ^ (newrec->dst_addr & 0xFFFF);

  if (ht->table[hash].seg) { /* hash key exists */

    addr.seg = ht->table[hash].seg;
    addr.off = ht->table[hash].off;

    while (1) {

      rec = (struct hash_ip2_rec*)(ht->data[addr.seg] + addr.off);

      if ((rec->src_addr == newrec->src_addr) &&
        (rec->dst_addr == newrec->dst_addr)) { /* found it */
        rec->nflows += newrec->nflows;
        rec->noctets += newrec->noctets;
        rec->npackets += newrec->npackets;
        rec->etime += newrec->etime;
        return 0;
      }

      if (rec->next.seg) { /* not the last record in the chain? */
        addr.seg = rec->next.seg;
        addr.off = rec->next.off;
      } else
        break;
    
    }

    /* didn't find it, allocate a new one and add it to the chain */
#ifdef HASH_IP2_STAT
  ht->table[hash].hits ++;
#endif /* HASH_IP2_STAT */

    rec->next.seg = ht->head.seg;
    rec->next.off = ht->head.off;

    rec = (struct hash_ip2_rec*)(ht->data[ht->head.seg] + ht->head.off);
    rec->src_addr = newrec->src_addr;
    rec->dst_addr = newrec->dst_addr;
    rec->nflows = newrec->nflows;
    rec->noctets = newrec->noctets;
    rec->npackets = newrec->npackets;
    rec->etime = newrec->etime;
    rec->next.seg = 0;
    rec->next.off = 0;

    if (ht->head.off != ht->lastoff)
      ht->head.off += sizeof(struct hash_ip2_rec);
    else {
      if (!(ht->data = (char**)realloc(ht->data,
        sizeof(char*)*(ht->head.seg+2))))
        return -1;
      if (!(ht->data[ht->head.seg+1] = (char*)malloc(65536))) 
        return -1;
      ++ ht->head.seg;
      ht->head.off = 0;
    }
    return 1;
  }

  /* wasn't in the hash table, allocate a block and link it in */
#ifdef HASH_IP2_STAT
  ht->table[hash].hits ++;
#endif /* HASH_IP2_STAT */

  ht->table[hash].seg = ht->head.seg;
  ht->table[hash].off = ht->head.off;

  rec = (struct hash_ip2_rec*)(ht->data[ht->head.seg] + ht->head.off);
  rec->src_addr = newrec->src_addr;
  rec->dst_addr = newrec->dst_addr;
  rec->nflows = newrec->nflows;
  rec->noctets = newrec->noctets;
  rec->npackets = newrec->npackets;
  rec->next.seg = 0;

  if (ht->head.off != ht->lastoff)
    ht->head.off += sizeof(struct hash_ip2_rec);
  else {
    if (!(ht->data = (char**)realloc(ht->data,
      sizeof(char*)*(ht->head.seg+2))))
      return -1;
    if (!(ht->data[ht->head.seg+1] = (char*)malloc(65536))) 
      return -1;
    ++ ht->head.seg;
    ht->head.off = 0;
  }

  return 2;

} /* hash_ip2_update */

hash_ip2_dump (ht, cc, sort_order, options, nflows, noctets,
  npackets, nduration)
struct hash_ip2_table *ht;
char cc;
int sort_order, options;
u_int64 nflows, noctets, npackets, nduration;
{
  struct hash_ip2_rec *rec;
  int x, len;
  struct hash_ip2_addr addr;
  char fmt_buf[256];

  if (options & OPT_PERCENT) 
    printf("%c\n%c src IPaddr     dst IPaddr       flows    octets   packets  duration\n%c\n",
    cc, cc, cc);
  else 
    printf("%c\n%c src IPaddr     dst IPaddr       flows                 octets                packets               duration\n%c\n", cc, cc, cc);

  for (x = 0; x < 65536; ++x) {

    if (ht->table[x].seg) { /* hash key exists */

      addr.seg = ht->table[x].seg;
      addr.off = ht->table[x].off;

      while (1) {

        rec = (struct hash_ip2_rec*)(ht->data[addr.seg] + addr.off);

        if (options & OPT_PERCENT) {

          len = fmt_ipv4(fmt_buf, rec->src_addr, FMT_PAD_RIGHT);
          fmt_buf[len++] = ' '; fmt_buf[len++] = ' ';

          fmt_ipv4(fmt_buf+len, rec->dst_addr, FMT_PAD_RIGHT);

          printf("%-32.32s  %-6.3f   %-6.3f   %-6.3f   %-6.3f\n",
            fmt_buf,
            ((double)rec->nflows/(double)nflows)*100,
            ((double)rec->noctets/(double)noctets)*100,
            ((double)rec->npackets/(double)npackets)*100,
            ((double)rec->etime/(double)nduration)*100);

        } else {
          
          len = fmt_ipv4(fmt_buf, rec->src_addr, FMT_PAD_RIGHT);
          fmt_buf[len++] = ' '; fmt_buf[len++] = ' ';

          len += fmt_ipv4(fmt_buf+len, rec->dst_addr, FMT_PAD_RIGHT);
          fmt_buf[len++] = ' '; fmt_buf[len++] = ' ';

          len += fmt_uint64(fmt_buf+len, rec->nflows, FMT_PAD_RIGHT);
          fmt_buf[len++] = ' '; fmt_buf[len++] = ' ';
        
          len += fmt_uint64(fmt_buf+len, rec->noctets, FMT_PAD_RIGHT);
          fmt_buf[len++] = ' '; fmt_buf[len++] = ' ';

          len += fmt_uint64(fmt_buf+len, rec->npackets,
            FMT_PAD_RIGHT);
          fmt_buf[len++] = ' '; fmt_buf[len++] = ' ';

          len += fmt_uint64(fmt_buf+len, rec->etime, FMT_JUST_LEFT);

          puts(fmt_buf);
        }

        if (!rec->next.seg)
          break;

        addr.seg = rec->next.seg;
        addr.off = rec->next.off;

      }
    }
  }

  return 0;

} /* hash_ip2_dump */

#ifdef HASH_IP2_STAT

int hash_ip2_stat_dump(ht)
struct hash_ip2_table *ht;
{
  int x;
  int maxdepth, m;
  struct hash_ip2_rec *rec;
  struct hash_ip2_addr addr;

  maxdepth = 0;

  for (x = 0; x < 65536; ++x) {

    if (ht->table[x].seg) { /* hash key exists */

      addr.seg = ht->table[x].seg;
      addr.off = ht->table[x].off;

      m = 1;

      while (1) {

        rec = (struct hash_ip2_rec*)(ht->data[addr.seg] + addr.off);

        if (!rec->next.seg)
          break;

        addr.seg = rec->next.seg;
        addr.off = rec->next.off;

        ++m;
      }
      if (m > maxdepth)
        maxdepth = m;
    }
  }

  printf("# maxdepth = %d\n", maxdepth);

  for (x = 0; x < 65536; ++x) {

    if (!ht->table[x].hits)
      continue;

    printf("%-10d    %-10ld\n", x, ht->table[x].hits);
  }

} /* hash_ip2_stat_dump */

#endif /* HASH_IP2_STAT */
