#include <sys/time.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <fcntl.h>
#include "flow.h"
#include "fmt.h"
#include "hash_ip.h"
#include "hash_ip2.h"
#include "sym.h"

/* output options */
#define OPT_PERCENT 1 /* use percent's */
#define OPT_NAMES 2 /* use symbolic names */
#define OPT_TALLY 4 /* use tally lines when outputing percents */


int debug;

int tally; /* the n in print a tally every n lines */

int ip_net_only;

int sort_64uint64();
int tbl_out1();

int flow_stat0(); int flow_stat1(); int flow_stat2(); int flow_stat3();
int flow_stat4(); int flow_stat5(); int flow_stat6(); int flow_stat7();
int flow_stat8(); int flow_stat9(); int flow_stat10(); int flow_stat11();
int flow_stat12(); int flow_stat13(); int flow_stat14(); int flow_stat15();
int flow_stat16(); int flow_stat17(); int flow_stat18(); int flow_stat19();
int flow_stat20();

struct jump flow_stat[] = {{flow_stat0}, {flow_stat1}, {flow_stat2},
          {flow_stat3}, {flow_stat4}, {flow_stat5},
          {flow_stat6}, {flow_stat7}, {flow_stat8},
          {flow_stat9}, {flow_stat10}, {flow_stat11},
          {flow_stat12}, {flow_stat13}, {flow_stat14},
          {flow_stat15}, {flow_stat16}, {flow_stat17},
          {flow_stat18}, {flow_stat19}, {flow_stat20},};

#define FLOW_STAT_FORMATS 21  /* # of types of output */

void usage();
void print_3float();
void print_3float2();

char cc; /* comment character */

int main(argc, argv)
int argc;
char **argv;
{

  extern char *optarg;
  extern struct jump flow_stat[];
  struct flow_stream fs;
  extern char cc;
  int x, y, ret, i, format;
  int sort_order, print_header;
  u_int64 total_flows;
  struct flow_profile fp;
  int options;

  /* profile */
  if (profile_start (&fp) == -1) {
    fprintf(stderr, "profile_start(): failed\n");
    exit (1);
  }

  options = 0;
  format = 0;
  ip_net_only = 0;
  print_header = 0;
  bzero(&fs, sizeof fs);
  cc = '#';
  sort_order = 0;

  while ((i = getopt(argc, argv, "c:d:f:h?nCpPs:S:t:")) != -1)
    switch (i) {

    case 'c': /* comment character */
      cc = optarg[0];
      break;

    case 'C': /* summarize ip to classful networks */
      ip_net_only = 1;
      break;

    case 'd': /* debug */
      debug = atoi(optarg);
      break;

    case 'f': /* format */
      format = atoi(optarg);
      break;

    case 'h': /* help */
    case '?': /* help */
      usage();
      exit (0);
      break;

    case 'n': /* use names */
      options |= OPT_NAMES;
      break;

    case 'p': /* print header */
      print_header = 1;
      break;

    case 'P': /* percent's */
      options |= OPT_PERCENT;
      break;

    case 's': /* sort low to high on field n */
      sort_order = (atoi(optarg)+1) * -1;
      break;

    case 'S': /* sort high to low on field n */
      sort_order = (atoi(optarg)+1);
      break;

    case 't': /* tallies */
      options |= OPT_TALLY;
      tally = atoi(optarg);
      break;

    default:
      usage();
      exit (1);
      break;

    } /* switch */

  if (format >= FLOW_STAT_FORMATS) {
    fprintf(stderr, "no such format, %d\n", format);
    exit (1);
  }

  /* read from stdin */
  fs.fd = 0;

  if (flow_read_header(&fs) == -1) {
    fprintf(stderr, "flow_read_header(): failed.\n");
    exit (1);
  }

  /* print cmdl args */
  printf("%c\n%c output from: ", cc, cc);
  for (x = 0; x < argc; ++x) {
    for (y = 0; y < strlen(argv[x]); ++y) {
      if (isprint((int)argv[x][y])) putc(argv[x][y], stdout);
    }
    putc (' ', stdout);
  }
  putc ('\n', stdout);

  if (print_header) {
    flow_print_header(&fs.fh, cc);
  }

  ret = flow_stat[format].where(&fs, sort_order, &total_flows, options);

  fp.nflows = total_flows;

  if ((!ret) && (debug > 0))
    if (profile_end(argv[0], &fp) == -1) {
      fprintf(stderr, "profile_end(): failed\n");
      exit (1);
    }

  return ret;

} /* main */

void usage() {

  fprintf(stderr, "flow-stat:\n\n");
  fprintf(stderr, " -dn  set debug level to n.\n");
  fprintf(stderr, " -fn  format n.\n");
  fprintf(stderr, " -n   use symbolic names for ports and protocols\n");
  fprintf(stderr, " -p   print flow header.\n");
  fprintf(stderr, " -P   use percents instead of counters\n");
  fprintf(stderr, " -C   summarize IP host addreses to classful net addresses.\n");
  fprintf(stderr, " -Sn  sort descending on field n.\n");
  fprintf(stderr, " -sn  sort ascending on field n.\n");
  fprintf(stderr, " -tn  tally totals every n output lines when used with -P.\n");
  fprintf(stderr, "\n formats:\n");
  fprintf(stderr, "    0 Summary\n");
  fprintf(stderr, "    1 Average packet size distribution histogram\n");
  fprintf(stderr, "    2 Packets per flow distribution histogram\n");
  fprintf(stderr, "    3 Octets per flow flow distribution histogram\n");
  fprintf(stderr, "    5 UDP/TCP destination port flows,octets,packets,duration histogram\n");
  fprintf(stderr, "    6 UDP/TCP source port flows,octets,packets,duration histogram\n");
  fprintf(stderr, "    7 UDP/TCP port flows,octets,packets,duration histogram\n");
  fprintf(stderr, "    8 Destination IP flows,octets,packets,duration\n");
  fprintf(stderr, "    9 Source IP flows,octets,packets,duration\n");
  fprintf(stderr, "   10 Source/Destination IP flows,octets,packets,duration\n");
  fprintf(stderr, "   11 IP flows,octets,packets,duration\n");
  fprintf(stderr, "   12 flows,octets,packets,duration by IP protocol\n");
  fprintf(stderr, "   13 octets for flow duration plot data\n");
  fprintf(stderr, "   14 packets for flow duration plot data\n");
  fprintf(stderr, "   15 short summary\n");
  fprintf(stderr, "   16 Next hop flows,octets,packets,duration\n");
  fprintf(stderr, "   17 Input interface flows,octets,packets,duration\n");
  fprintf(stderr, "   18 Output interface flows,octets,packets,duration\n");
  fprintf(stderr, "   19 Source AS flows,octets,packets,duration\n");
  fprintf(stderr, "   20 Destination AS flows,octets,packets,duration\n");
  fprintf(stderr, "   20 Source/Destination AS flows,octets,packets,duration\n");
  fprintf(stderr, "\n\n");

} /* usage */


/*
/* function: flow_stat0
/*
/*  prints long summary
/*
/* returns 0 for okay.
*/
int flow_stat0(fs, sort_order, total_flows, options)
struct flow_stream *fs;
int sort_order, options;
u_int64 *total_flows;
{
  struct flow_stat0 fs0;
  struct flow_data *fdata;
  u_long p;
  char fmt_buf[256];

  bzero(&fs0, sizeof fs0);
  fs0.start = 0xFFFFFFFF;
  fs0.end = 0;

  while ((fdata = flow_read(fs))) {

    fs0.nflows ++;
    fs0.noctets += fdata->dOctets;
    fs0.npackets += fdata->dPkts;

    if (fdata->First < fs0.start)
      fs0.start = fdata->First;

    if (fdata->Last > fs0.end)
      fs0.end = fdata->Last;

    p = fdata->dOctets / fdata->dPkts;

    if (p <= 32) ++ fs0.psize32;
    else if (p <= 64) ++ fs0.psize64;
    else if (p <= 96) ++ fs0.psize96;
    else if (p <= 128) ++ fs0.psize128;
    else if (p <= 160) ++ fs0.psize160;
    else if (p <= 192) ++ fs0.psize192;
    else if (p <= 224) ++ fs0.psize224;
    else if (p <= 256) ++ fs0.psize256;
    else if (p <= 288) ++ fs0.psize288;
    else if (p <= 320) ++ fs0.psize320;
    else if (p <= 352) ++ fs0.psize352;
    else if (p <= 384) ++ fs0.psize384;
    else if (p <= 416) ++ fs0.psize416;
    else if (p <= 448) ++ fs0.psize448;
    else if (p <= 480) ++ fs0.psize480;
    else if (p <= 512) ++ fs0.psize512;
    else if (p <= 544) ++ fs0.psize544;
    else if (p <= 576) ++ fs0.psize576;
    else if (p <= 1024) ++ fs0.psize1024;
    else if (p <= 1536) ++ fs0.psize1536;
    else if (p <= 2048) ++ fs0.psize2048;
    else if (p <= 2560) ++ fs0.psize2560;
    else if (p <= 3072) ++ fs0.psize3072;
    else if (p <= 3584) ++ fs0.psize3584;
    else if (p <= 4096) ++ fs0.psize4096;
    else if (p <= 4608) ++ fs0.psize4608;

    p = fdata->dPkts;
    if (p <= 1) ++ fs0.fpsize1;
    else if (p <= 2) ++ fs0.fpsize2;
    else if (p <= 4) ++ fs0.fpsize4;
    else if (p <= 8) ++ fs0.fpsize8;
    else if (p <= 12) ++ fs0.fpsize12;
    else if (p <= 16) ++ fs0.fpsize16;
    else if (p <= 20) ++ fs0.fpsize20;
    else if (p <= 24) ++ fs0.fpsize24;
    else if (p <= 28) ++ fs0.fpsize28;
    else if (p <= 32) ++ fs0.fpsize32;
    else if (p <= 36) ++ fs0.fpsize36;
    else if (p <= 40) ++ fs0.fpsize40;
    else if (p <= 44) ++ fs0.fpsize44;
    else if (p <= 48) ++ fs0.fpsize48;
    else if (p <= 52) ++ fs0.fpsize52;
    else if (p <= 60) ++ fs0.fpsize60;
    else if (p <= 100) ++ fs0.fpsize100;
    else if (p <= 200) ++ fs0.fpsize200;
    else if (p <= 300) ++ fs0.fpsize300;
    else if (p <= 400) ++ fs0.fpsize400;
    else if (p <= 500) ++ fs0.fpsize500;
    else if (p <= 600) ++ fs0.fpsize600;
    else if (p <= 700) ++ fs0.fpsize700;
    else if (p <= 800) ++ fs0.fpsize800;
    else if (p <= 900) ++ fs0.fpsize900;
    else ++ fs0.fpsize_other;

    p = fdata->dOctets;
    if (p <= 32) ++ fs0.fosize32;
    else if (p <= 64) ++ fs0.fosize32;
    else if (p <= 128) ++ fs0.fosize128;
    else if (p <= 256) ++ fs0.fosize256;
    else if (p <= 512) ++ fs0.fosize512;
    else if (p <= 1280) ++ fs0.fosize1280;
    else if (p <= 2048) ++ fs0.fosize2048;
    else if (p <= 2816) ++ fs0.fosize2816;
    else if (p <= 3584) ++ fs0.fosize3584;
    else if (p <= 4352) ++ fs0.fosize4352;
    else if (p <= 5120) ++ fs0.fosize5120;
    else if (p <= 5888) ++ fs0.fosize5888;
    else if (p <= 6656) ++ fs0.fosize6656;
    else if (p <= 7424) ++ fs0.fosize7424;
    else if (p <= 8192) ++ fs0.fosize8192;
    else if (p <= 8960) ++ fs0.fosize8960;
    else if (p <= 9728) ++ fs0.fosize9728;
    else if (p <= 10496) ++ fs0.fosize10496;
    else if (p <= 11264) ++ fs0.fosize11264;
    else if (p <= 12032) ++ fs0.fosize12032;
    else if (p <= 12800) ++ fs0.fosize12800;
    else if (p <= 13568) ++ fs0.fosize13568;
    else if (p <= 14336) ++ fs0.fosize14336;
    else if (p <= 15104) ++ fs0.fosize15104;
    else if (p <= 15872) ++ fs0.fosize15872;
    else ++ fs0.fosize_other;


    p = fdata->Last - fdata->First;
    fs0.time += p;

    if (p <= 10) ++ fs0.ftime10;
    else if (p <= 50) ++ fs0.ftime50;
    else if (p <= 100) ++ fs0.ftime100;
    else if (p <= 200) ++ fs0.ftime200;
    else if (p <= 500) ++ fs0.ftime500;
    else if (p <= 1000) ++ fs0.ftime1000;
    else if (p <= 2000) ++ fs0.ftime2000;
    else if (p <= 3000) ++ fs0.ftime3000;
    else if (p <= 4000) ++ fs0.ftime4000;
    else if (p <= 5000) ++ fs0.ftime5000;
    else if (p <= 6000) ++ fs0.ftime6000;
    else if (p <= 7000) ++ fs0.ftime7000;
    else if (p <= 8000) ++ fs0.ftime8000;
    else if (p <= 9000) ++ fs0.ftime9000;
    else if (p <= 10000) ++ fs0.ftime10000;
    else if (p <= 12000) ++ fs0.ftime12000;
    else if (p <= 14000) ++ fs0.ftime14000;
    else if (p <= 16000) ++ fs0.ftime16000;
    else if (p <= 18000) ++ fs0.ftime18000;
    else if (p <= 20000) ++ fs0.ftime20000;
    else if (p <= 22000) ++ fs0.ftime22000;
    else if (p <= 24000) ++ fs0.ftime24000;
    else if (p <= 26000) ++ fs0.ftime26000;
    else if (p <= 28000) ++ fs0.ftime28000;
    else if (p <= 30000) ++ fs0.ftime30000;
    else ++ fs0.ftime_other;

  }

  fs0.aflowtime = fs0.time / fs0.nflows;
  fs0.aps = fs0.noctets / fs0.npackets;
  fs0.afs = fs0.noctets / fs0.nflows;
  fs0.apf = fs0.npackets / fs0.nflows;
  fs0.fps = (float)fs0.nflows / ((fs0.end - fs0.start) / 1000);
  fs0.aos = ((float)(fs0.noctets*8) / 1024) / ((fs0.end - fs0.start) / 1000);

  strcpy(fmt_buf, "Total Flows                     : ");
  fmt_uint64(fmt_buf+34, fs0.nflows, FMT_JUST_LEFT);
  puts(fmt_buf);

  strcpy(fmt_buf, "Total Octets                    : ");
  fmt_uint64(fmt_buf+34, fs0.noctets, FMT_JUST_LEFT);
  puts(fmt_buf);

  strcpy(fmt_buf, "Total Packets                   : ");
  fmt_uint64(fmt_buf+34, fs0.npackets, FMT_JUST_LEFT);
  puts(fmt_buf);

  strcpy(fmt_buf, "Total Time (1/1000 secs)        : ");
  fmt_uint64(fmt_buf+34, fs0.time, FMT_JUST_LEFT);
  puts(fmt_buf);

  strcpy(fmt_buf, "Duration of data (1/1000 secs)  : ");
  fmt_uint64(fmt_buf+34, (fs0.end - fs0.start), FMT_JUST_LEFT);
  puts(fmt_buf);

  printf("Average flow time (1/1000 secs) : %4.4f\n", fs0.aflowtime);
  printf("Average packet size (octets)    : %4.4f\n", fs0.aps);
  printf("Average flow size (octets)      : %4.4f\n", fs0.afs);
  printf("Average packets per flow        : %4.4f\n", fs0.apf);
  printf("Average flows / second          : %4.4f\n", fs0.fps);
  printf("Average Kbits / second          : %4.4f\n", fs0.aos);

  printf("\n\n");

  printf("IP packet size distribution:\n");
  printf("   1-32   64   96  128  160  192  224  256  288  320  352  384  416  448  480\n   ");
  print_3float((float)fs0.psize32 / fs0.nflows);
  print_3float((float)fs0.psize64 / fs0.nflows);
  print_3float((float)fs0.psize96 / fs0.nflows);
  print_3float((float)fs0.psize128 / fs0.nflows);
  print_3float((float)fs0.psize160 / fs0.nflows);
  print_3float((float)fs0.psize192 / fs0.nflows);
  print_3float((float)fs0.psize224 / fs0.nflows);
  print_3float((float)fs0.psize256 / fs0.nflows);
  print_3float((float)fs0.psize288 / fs0.nflows);
  print_3float((float)fs0.psize320 / fs0.nflows);
  print_3float((float)fs0.psize352 / fs0.nflows);
  print_3float((float)fs0.psize384 / fs0.nflows);
  print_3float((float)fs0.psize416 / fs0.nflows);
  print_3float((float)fs0.psize448 / fs0.nflows);
  print_3float((float)fs0.psize480 / fs0.nflows);
  printf("\n\n");

  printf("    512  544  576 1024 1536 2048 2560 3072 3584 4096 4608\n   ");
  print_3float((float)fs0.psize512 / fs0.nflows);
  print_3float((float)fs0.psize544 / fs0.nflows);
  print_3float((float)fs0.psize576 / fs0.nflows);
  print_3float((float)fs0.psize1024 / fs0.nflows);
  print_3float((float)fs0.psize1536 / fs0.nflows);
  print_3float((float)fs0.psize2048 / fs0.nflows);
  print_3float((float)fs0.psize2560 / fs0.nflows);
  print_3float((float)fs0.psize3072 / fs0.nflows);
  print_3float((float)fs0.psize3584 / fs0.nflows);
  print_3float((float)fs0.psize4096 / fs0.nflows);
  print_3float((float)fs0.psize4608 / fs0.nflows);
  printf("\n\n");

  printf("Packets per flow distribution:\n");
  printf("      1    2    4    8   12   16   20   24   28   32   36   40   44   48   52\n   ");

  print_3float((float)fs0.fpsize1 / fs0.nflows);
  print_3float((float)fs0.fpsize2 / fs0.nflows);
  print_3float((float)fs0.fpsize4 / fs0.nflows);
  print_3float((float)fs0.fpsize8 / fs0.nflows);
  print_3float((float)fs0.fpsize12 / fs0.nflows);
  print_3float((float)fs0.fpsize16 / fs0.nflows);
  print_3float((float)fs0.fpsize20 / fs0.nflows);
  print_3float((float)fs0.fpsize24 / fs0.nflows);
  print_3float((float)fs0.fpsize28 / fs0.nflows);
  print_3float((float)fs0.fpsize32 / fs0.nflows);
  print_3float((float)fs0.fpsize36 / fs0.nflows);
  print_3float((float)fs0.fpsize40 / fs0.nflows);
  print_3float((float)fs0.fpsize44 / fs0.nflows);
  print_3float((float)fs0.fpsize48 / fs0.nflows);
  print_3float((float)fs0.fpsize52 / fs0.nflows);
  printf("\n\n     60  100  200  300  400  500  600  700  800  900 >900\n   ");
  print_3float((float)fs0.fpsize60 / fs0.nflows);
  print_3float((float)fs0.fpsize100 / fs0.nflows);
  print_3float((float)fs0.fpsize200 / fs0.nflows);
  print_3float((float)fs0.fpsize300 / fs0.nflows);
  print_3float((float)fs0.fpsize400 / fs0.nflows);
  print_3float((float)fs0.fpsize500 / fs0.nflows);
  print_3float((float)fs0.fpsize600 / fs0.nflows);
  print_3float((float)fs0.fpsize700 / fs0.nflows);
  print_3float((float)fs0.fpsize800 / fs0.nflows);
  print_3float((float)fs0.fpsize900 / fs0.nflows);
  print_3float((float)fs0.fpsize_other / fs0.nflows);
  printf("\n\n");
             
  printf("Octets per flow distribution:\n");
  printf("     32   64  128  256  512 1280 2048 2816 3584 4352 5120 5888 6656 7424 8192\n   ");
  print_3float((float)fs0.fosize32 / fs0.nflows);
  print_3float((float)fs0.fosize64 / fs0.nflows);
  print_3float((float)fs0.fosize128 / fs0.nflows);
  print_3float((float)fs0.fosize256 / fs0.nflows);
  print_3float((float)fs0.fosize512 / fs0.nflows);
  print_3float((float)fs0.fosize1280 / fs0.nflows);
  print_3float((float)fs0.fosize2048 / fs0.nflows);
  print_3float((float)fs0.fosize2816 / fs0.nflows);
  print_3float((float)fs0.fosize3584 / fs0.nflows);
  print_3float((float)fs0.fosize4352 / fs0.nflows);
  print_3float((float)fs0.fosize5120 / fs0.nflows);
  print_3float((float)fs0.fosize5888 / fs0.nflows);
  print_3float((float)fs0.fosize6656 / fs0.nflows);
  print_3float((float)fs0.fosize7424 / fs0.nflows);
  print_3float((float)fs0.fosize8192 / fs0.nflows);
  printf("\n\n   8960 9728 10496 11264 12032 12800 13568 14336 15104 15872 >15872\n   ");
  print_3float((float)fs0.fosize8960 / fs0.nflows);
  print_3float2((float)fs0.fosize9728 / fs0.nflows);
  print_3float2((float)fs0.fosize10496 / fs0.nflows);
  print_3float2((float)fs0.fosize11264 / fs0.nflows);
  print_3float2((float)fs0.fosize12032 / fs0.nflows);
  print_3float2((float)fs0.fosize12800 / fs0.nflows);
  print_3float2((float)fs0.fosize13568 / fs0.nflows);
  print_3float2((float)fs0.fosize14336 / fs0.nflows);
  print_3float2((float)fs0.fosize15104 / fs0.nflows);
  print_3float2((float)fs0.fosize15872 / fs0.nflows);
  print_3float2((float)fs0.fosize_other / fs0.nflows);
  printf("\n\n");

  printf("Flow time distribution:\n");
  printf("    10    50  100  200  500 1000 2000 3000 4000 5000 6000 7000 8000 9000 10000\n   ");
  print_3float((float)fs0.ftime10 / fs0.nflows);
  print_3float((float)fs0.ftime50 / fs0.nflows);
  print_3float((float)fs0.ftime100 / fs0.nflows);
  print_3float((float)fs0.ftime200 / fs0.nflows);
  print_3float((float)fs0.ftime500 / fs0.nflows);
  print_3float((float)fs0.ftime1000 / fs0.nflows);
  print_3float((float)fs0.ftime2000 / fs0.nflows);
  print_3float((float)fs0.ftime3000 / fs0.nflows);
  print_3float((float)fs0.ftime4000 / fs0.nflows);
  print_3float((float)fs0.ftime5000 / fs0.nflows);
  print_3float((float)fs0.ftime6000 / fs0.nflows);
  print_3float((float)fs0.ftime7000 / fs0.nflows);
  print_3float((float)fs0.ftime8000 / fs0.nflows);
  print_3float2((float)fs0.ftime9000 / fs0.nflows);
  print_3float2((float)fs0.ftime10000 / fs0.nflows);
  printf("\n\n  12000 14000 16000 18000 20000 22000 24000 26000 28000 30000 >30000\n   ");
  print_3float2((float)fs0.ftime12000 / fs0.nflows);
  print_3float2((float)fs0.ftime14000 / fs0.nflows);
  print_3float2((float)fs0.ftime16000 / fs0.nflows);
  print_3float2((float)fs0.ftime18000 / fs0.nflows);
  print_3float2((float)fs0.ftime20000 / fs0.nflows);
  print_3float2((float)fs0.ftime22000 / fs0.nflows);
  print_3float2((float)fs0.ftime24000 / fs0.nflows);
  print_3float2((float)fs0.ftime26000 / fs0.nflows);
  print_3float2((float)fs0.ftime28000 / fs0.nflows);
  print_3float2((float)fs0.ftime30000 / fs0.nflows);
  print_3float2((float)fs0.ftime_other / fs0.nflows);
  printf("\n\n");

  *total_flows = fs0.nflows;

  return 0;

} /* flow_stat0 */

/*
/* function: flow_stat1
/*
/*  Average packet size distribution histogram
/*
/* returns 0 for okay.
*/
int flow_stat1(fs, sort_order, total_flows, options)
struct flow_stream *fs;
int sort_order, options;
u_int64 *total_flows;
{
  u_int64 flows[65536], octets[65536], packets[65536], duration[65536];
  u_int64 nflows, noctets, npackets, nduration, overflow, tmp;
  u_int index[65536];
  struct flow_data *fdata;
  int32 i;

  bzero(&flows, sizeof flows);
  bzero(&octets, sizeof octets);
  bzero(&packets, sizeof packets);
  bzero(&duration, sizeof duration);
  nflows = noctets = npackets = nduration = overflow = 0;

  /* initialize index */
  for (i = 0; i < 65536; ++i)
    index[i] = i;


  while ((fdata = flow_read(fs))) {

    ++nflows;
    noctets += fdata->dOctets;
    npackets += fdata->dPkts;
    nduration += (fdata->Last - fdata->First);

    tmp = fdata->dOctets / fdata->dPkts;

    /* shouldn't happen */
    if (tmp > 65535) {
      ++ overflow;
      continue;
    }

    flows[tmp] ++;
    octets[tmp] += fdata->dOctets;
    packets[tmp] += fdata->dPkts;
    duration[tmp] += (fdata->Last - fdata->First);
  }

  /* doesn't make sense here */
  options &= ~OPT_NAMES;

  tbl_out1(sort_order, options, index, flows, octets, packets, duration,
    65536, nflows, noctets, npackets, nduration, "pkt size  ", 
    (struct sym_table*)0L);

  *total_flows = nflows;

  return 0;
} /* flow_stat1 */


/*
/* function: flow_stat2
/*
/*  Packets per flow distribution histogram
/*
/* returns 0 for okay.
*/
int flow_stat2(fs, sort_order, total_flows, options)
struct flow_stream *fs;
int sort_order, options;
u_int64 *total_flows;
{
  u_int64 flows[65536], octets[65536], packets[65536], duration[65536];
  u_int64 nflows, noctets, npackets, nduration, overflow;
  u_int index[65536];
  struct flow_data *fdata;
  int32 i;

  bzero(&flows, sizeof flows);
  bzero(&octets, sizeof octets);
  bzero(&packets, sizeof packets);
  bzero(&duration, sizeof duration);
  nflows = noctets = npackets = nduration = overflow = 0;

  /* initialize index */
  for (i = 0; i < 65536; ++i)
    index[i] = i;


  while ((fdata = flow_read(fs))) {

    ++nflows;
    noctets += fdata->dOctets;
    npackets += fdata->dPkts;
    nduration += (fdata->Last - fdata->First);

    if (fdata->dPkts > 65535) {
      ++ overflow;
      continue;
    }

    flows[fdata->dPkts] ++;
    octets[fdata->dPkts] += fdata->dOctets;
    packets[fdata->dPkts] += fdata->dPkts;
    duration[fdata->dPkts] += (fdata->Last - fdata->First);
  }

  /* doesn't make sense here */
  options &= ~OPT_NAMES;

  tbl_out1(sort_order, options, index, flows, octets, packets, duration,
    65536, nflows, noctets, npackets, nduration, "packets   ",
    (struct sym_table*)0L);

  *total_flows = nflows;

  return 0;

} /* flow_stat2 */

/*
/* function: flow_stat3
/*
/*  Octets per flow flow distribution histogram
/*
*/
int flow_stat3(fs, sort_order, total_flows, options)
struct flow_stream *fs;
int sort_order, options;
u_int64 *total_flows;
{
  u_int64 flows[65536], octets[65536], packets[65536], duration[65536];
  u_int64 nflows, noctets, npackets, nduration, overflow;
  u_int index[65536];
  struct flow_data *fdata;
  int32 i;

  bzero(&flows, sizeof flows);
  bzero(&octets, sizeof octets);
  bzero(&packets, sizeof packets);
  bzero(&duration, sizeof duration);
  nflows = noctets = npackets = nduration = overflow = 0;

  /* initialize index */
  for (i = 0; i < 65536; ++i)
    index[i] = i;


  while ((fdata = flow_read(fs))) {

    ++nflows;
    noctets += fdata->dOctets;
    npackets += fdata->dPkts;
    nduration += (fdata->Last - fdata->First);

    if (fdata->dOctets > 65535) {
      ++ overflow;
      continue;
    }

    flows[fdata->dOctets] ++;
    octets[fdata->dOctets] += fdata->dOctets;
    packets[fdata->dOctets] += fdata->dPkts;
    duration[fdata->dOctets] += (fdata->Last - fdata->First);
  }

  /* doesn't make sense here */
  options &= ~OPT_NAMES;


  tbl_out1(sort_order, options, index, flows, octets, packets, duration,
    65536, nflows, noctets, npackets, nduration, "octets    ",
    (struct sym_table*)0L);

  *total_flows = nflows;

  return 0;

} /* flow_stat3 */

/*
/* function: flow_stat4
/*
/*  ??
/*
*/
int flow_stat4(fs, sort_order, total_flows, options)
struct flow_stream *fs;
int sort_order, options;
u_int64 *total_flows;
{

  printf("Not implemented.\n");
  return 0;
}

/*
/* function: flow_stat5
/*
/*  UDP/TCP destination port flows,octets,packets,duration histogram
/*
/* returns 0 for okay.
*/
int flow_stat5(fs, sort_order, total_flows, options)
struct flow_stream *fs;
int sort_order, options;
u_int64 *total_flows;
{
  u_int64 flows[65536], octets[65536], packets[65536], duration[65536];
  u_int64 nflows, noctets, npackets, nduration;
  u_int index[65536];
  struct flow_data *fdata;
  int32 i;
  extern struct sym_table sym_tcp[];

  bzero(&flows, sizeof flows);
  bzero(&octets, sizeof octets);
  bzero(&packets, sizeof packets);
  bzero(&duration, sizeof duration);
  nflows = noctets = npackets = nduration = 0;

  /* initialize index */
  for (i = 0; i < 65536; ++i)
    index[i] = i;


  while ((fdata = flow_read(fs))) {

    /* ports only make sense for TCP and UDP */
    if ((fdata->prot != PROTO_UDP) && (fdata->prot != PROT_TCP))
      continue;

    ++nflows;
    noctets += fdata->dOctets;
    npackets += fdata->dPkts;
    nduration += (fdata->Last - fdata->First);

    flows[fdata->dstport] ++;
    octets[fdata->dstport] += fdata->dOctets;
    packets[fdata->dstport] += fdata->dPkts;
    duration[fdata->dstport] += (fdata->Last - fdata->First);
  }

  tbl_out1(sort_order, options, index, flows, octets, packets, duration,
    65536, nflows, noctets, npackets, nduration, "port      ", sym_tcp);

  *total_flows = nflows;

  return 0;
}

/*
/* function: flow_stat6
/*
/*  UDP/TCP source port flows,octets,packets,duration
/*
/* returns 0 for okay.
*/
int flow_stat6(fs, sort_order, total_flows, options)
struct flow_stream *fs;
int sort_order, options;
u_int64 *total_flows;
{
  u_int64 flows[65536], octets[65536], packets[65536], duration[65536];
  u_int64 nflows, noctets, npackets, nduration;
  u_int index[65536];
  struct flow_data *fdata;
  int32 i;
  extern struct sym_table sym_tcp[];

  /* initialize index */
  for (i = 0; i < 65536; ++i)
    index[i] = i;


  bzero(&flows, sizeof flows);
  bzero(&octets, sizeof octets);
  bzero(&packets, sizeof packets);
  bzero(&duration, sizeof duration);
  nflows = noctets = npackets = nduration = 0;

  while ((fdata = flow_read(fs))) {

    /* ports only make sense for TCP and UDP */
    if ((fdata->prot != PROTO_UDP) && (fdata->prot != PROT_TCP))
      continue;

    ++nflows;
    noctets += fdata->dOctets;
    npackets += fdata->dPkts;
    nduration += (fdata->Last - fdata->First);

    flows[fdata->srcport] ++;
    octets[fdata->srcport] += fdata->dOctets;
    packets[fdata->srcport] += fdata->dPkts;
    duration[fdata->srcport] += (fdata->Last - fdata->First);
  }

  tbl_out1(sort_order, options, index, flows, octets, packets, duration,
    65536, nflows, noctets, npackets, nduration, "port      ", sym_tcp);

  *total_flows = nflows;

  return 0;
} /* flow_stat6 */

/*
/* function: flow_stat7
/*
/*  UDP/TCP port flows,octets,packets,duration histogram
/*
/* returns 0 for okay.
*/
int flow_stat7(fs, sort_order, total_flows, options)
struct slow_stream *fs;
int sort_order, options;
u_int64 *total_flows;
{
  u_int64 flows[65536], octets[65536], packets[65536], duration[65536];
  u_int64 nflows, noctets, npackets, nduration;
  u_int index[65536];
  struct flow_data *fdata;
  int32 i;
  extern struct sym_table sym_tcp[];

  bzero(&flows, sizeof flows);
  bzero(&octets, sizeof octets);
  bzero(&packets, sizeof packets);
  bzero(&duration, sizeof duration);
  nflows = noctets = npackets = nduration = 0;

  /* initialize index */
  for (i = 0; i < 65536; ++i)
    index[i] = i;


  while ((fdata = flow_read(fs))) {

    /* ports only make sense for TCP and UDP */
    if ((fdata->prot != PROTO_UDP) && (fdata->prot != PROT_TCP))
      continue;

    ++nflows;
    noctets += fdata->dOctets;
    npackets += fdata->dPkts;
    nduration += (fdata->Last - fdata->First);

    flows[fdata->srcport] ++;
    octets[fdata->srcport] += fdata->dOctets;
    packets[fdata->srcport] += fdata->dPkts;
    duration[fdata->srcport] += (fdata->Last - fdata->First);

/*
    if (fdata->srcport != fdata->dstport) {
      flows[fdata->dstport] ++;
      octets[fdata->dstport] += fdata->dOctets;
      packets[fdata->dstport] += fdata->dPkts;
      duration[fdata->dstport] += (fdata->Last - fdata->First);
    }
*/
  }

  tbl_out1(sort_order, options, index, flows, octets, packets, duration,
    65536, nflows, noctets, npackets, nduration, "port      ", sym_tcp);

  *total_flows = nflows;

  return 0;

} /* flow_stat7 */

/*
/* function: flow_stat8
/*
/*  Destination IP flows,octets,packets,duration
/*
/* returns 0 for okay.
*/
int flow_stat8(fs, sort_order, total_flows, options)
struct flow_stream *fs;
int sort_order, options;
u_int64 *total_flows;
{
  struct flow_data *fdata;
  struct hash_ip_rec rec;
  struct hash_ip_table ht;
  extern int ip_net_only;
  extern char cc;
  u_int64 nflows, noctets, npackets, nduration;

  nflows = noctets = npackets = nduration = 0;

  if (hash_ip_init(&ht)) {
    fprintf(stderr, "hash_ip_init() failed\n");
    return -1;
  }

  while ((fdata = flow_read(fs))) {

    ++nflows;
    noctets += fdata->dOctets;
    npackets += fdata->dPkts;
    nduration += (fdata->Last - fdata->First);

    rec.addr = fdata->dstaddr;
    rec.nflows = 1;
    rec.noctets = fdata->dOctets;
    rec.npackets = fdata->dPkts;
    rec.etime = (fdata->Last - fdata->First);

    if (ip_net_only) {
      if ((rec.addr & 0x80000000) == 0)
        rec.addr &= 0xFF000000;
      else if ((rec.addr & 0xC0000000) == 0x80000000)
        rec.addr &= 0xFFFF0000;
      else if ((rec.addr & 0xC0000000) == 0xC0000000)
        rec.addr &= 0xFFFFFF00;
    }

    if (hash_ip_update(&ht, &rec) == -1) {
      fprintf(stderr, "hash_ip_update() failed\n");
      hash_ip_free(&ht);
      return -1;
    }

  }

/*  hash_ip_stat_dump(&ht); */

  hash_ip_dump(&ht, cc, sort_order, options, nflows, noctets, npackets,
    nduration);

  if (hash_ip_free(&ht)) {
    fprintf(stderr, "hash_ip_free() failed\n");
    return -1;
  }

  *total_flows = nflows;

  return 0;

} /* flow_stat8 */

/*
/* function: flow_stat9
/*
/*  Source IP flows,octets,packets,duration
/*
/* returns 0 for okay.
*/
int flow_stat9(fs, sort_order, total_flows, options)
struct flow_stream *fs;
int sort_order, options;
u_int64 *total_flows;
{
  struct flow_data *fdata;
  struct hash_ip_rec rec;
  struct hash_ip_table ht;
  extern int ip_net_only;
  extern char cc;
  u_int64 nflows, noctets, npackets, nduration;

  nflows = noctets = npackets = nduration = 0;

  if (hash_ip_init(&ht)) {
    fprintf(stderr, "hash_ip_init() failed\n");
    return -1;
  }

  while ((fdata = flow_read(fs))) {

    ++nflows;
    noctets += fdata->dOctets;
    npackets += fdata->dPkts;
    nduration += (fdata->Last - fdata->First);

    rec.addr = fdata->srcaddr;
    rec.nflows = 1;
    rec.noctets = fdata->dOctets;
    rec.npackets = fdata->dPkts;
    rec.etime = (fdata->Last - fdata->First);

    if (ip_net_only) {
      if ((rec.addr & 0x80000000) == 0)
        rec.addr &= 0xFF000000;
      else if ((rec.addr & 0xC0000000) == 0x80000000)
        rec.addr &= 0xFFFF0000;
      else if ((rec.addr & 0xC0000000) == 0xC0000000)
        rec.addr &= 0xFFFFFF00;
    }

    if (hash_ip_update(&ht, &rec) == -1) {
      fprintf(stderr, "hash_ip_update() failed\n");
      hash_ip_free(&ht);
      return -1;
    }

  }

/*  hash_ip_stat_dump(&ht); */

  hash_ip_dump(&ht, cc, sort_order, options, nflows, noctets, npackets,
    nduration);

  if (hash_ip_free(&ht)) {
    fprintf(stderr, "hash_ip_free() failed\n");
    return -1;
  }

  *total_flows = nflows;

  return 0;
}

/*
/* function: flow_stat10
/*
/*  Source/Destination IP flows,octets,packets,duration
/*
/* returns 0 for okay.
*/
int flow_stat10(fs, sort_order, total_flows, options)
struct flow_stream *fs;
int sort_order, options;
u_int64 *total_flows;
{
  struct flow_data *fdata;
  struct hash_ip2_rec rec;
  struct hash_ip2_table ht;
  extern int ip_net_only;
  extern char cc;
  u_int64 nflows, noctets, npackets, nduration;

  nflows = noctets = npackets = nduration = 0;
  
  if (hash_ip2_init(&ht)) {
    fprintf(stderr, "hash_ip2_init() failed\n");
    return -1;
  }

  while ((fdata = flow_read(fs))) {

    ++nflows;
    noctets += fdata->dOctets;
    npackets += fdata->dPkts;
    nduration += (fdata->Last - fdata->First);

    rec.src_addr = fdata->srcaddr;
    rec.dst_addr = fdata->dstaddr;
    rec.nflows = 1;
    rec.noctets = fdata->dOctets;
    rec.npackets = fdata->dPkts;
    rec.etime = (fdata->Last - fdata->First);

    if (ip_net_only) {
      if ((rec.src_addr & 0x80000000) == 0)
        rec.src_addr &= 0xFF000000;
      else if ((rec.src_addr & 0xC0000000) == 0x80000000)
        rec.src_addr &= 0xFFFF0000;
      else if ((rec.src_addr & 0xC0000000) == 0xC0000000)
        rec.src_addr &= 0xFFFFFF00;

      if ((rec.dst_addr & 0x80000000) == 0)
        rec.dst_addr &= 0xFF000000;
      else if ((rec.dst_addr & 0xC0000000) == 0x80000000)
        rec.dst_addr &= 0xFFFF0000;
      else if ((rec.dst_addr & 0xC0000000) == 0xC0000000)
        rec.dst_addr &= 0xFFFFFF00;
    }

    if (hash_ip2_update(&ht, &rec) == -1) {
      fprintf(stderr, "hash_ip2_update() failed\n");
      hash_ip2_free(&ht);
      return -1;
    }

  }

/*  hash_ip2_stat_dump(&ht); */

  hash_ip2_dump(&ht, cc, sort_order, options, nflows, noctets, npackets,
    nduration);

  if (hash_ip2_free(&ht)) {
    fprintf(stderr, "hash_ip2_free() failed\n");
    return -1;
  }

  *total_flows = nflows;

  return 0;
}

/*
/* function: flow_stat11
/*
/*  IP flows,octets,packets,duration
/*
/* returns 0 for okay.
*/
int flow_stat11(fs, sort_order, total_flows, options)
struct flow_stream *fs;
int sort_order, options;
u_int64 *total_flows;
{
  struct flow_data *fdata;
  struct hash_ip_rec rec;
  struct hash_ip_table ht;
  extern int ip_net_only;
  extern char cc;
  u_int64 nflows, noctets, npackets, nduration;

  nflows = noctets = npackets = nduration = 0;

  if (hash_ip_init(&ht)) {
    fprintf(stderr, "hash_ip_init() failed\n");
    return -1;
  }

  while ((fdata = flow_read(fs))) {

    ++nflows;
    noctets += fdata->dOctets;
    npackets += fdata->dPkts;
    nduration += (fdata->Last - fdata->First);

    rec.addr = fdata->srcaddr;
    rec.nflows = 1;
    rec.noctets = fdata->dOctets;
    rec.npackets = fdata->dPkts;
    rec.etime = (fdata->Last - fdata->First);

    if (ip_net_only) {
      if ((rec.addr & 0x80000000) == 0)
        rec.addr &= 0xFF000000;
      else if ((rec.addr & 0xC0000000) == 0x80000000)
        rec.addr &= 0xFFFF0000;
      else if ((rec.addr & 0xC0000000) == 0xC0000000)
        rec.addr &= 0xFFFFFF00;
    }

    if (hash_ip_update(&ht, &rec) == -1) {
      fprintf(stderr, "hash_ip_update() failed\n");
      hash_ip_free(&ht);
      return -1;
    }

    rec.addr = fdata->dstaddr;
    rec.nflows = 1;
    rec.noctets = fdata->dOctets;
    rec.npackets = fdata->dPkts;
    rec.etime = (fdata->Last - fdata->First);

    if (ip_net_only) {
      if ((rec.addr & 0x80000000) == 0)
        rec.addr &= 0xFF000000;
      else if ((rec.addr & 0xC0000000) == 0x80000000)
        rec.addr &= 0xFFFF0000;
      else if ((rec.addr & 0xC0000000) == 0xC0000000)
        rec.addr &= 0xFFFFFF00;
    }

    if (hash_ip_update(&ht, &rec) == -1) {
      fprintf(stderr, "hash_ip_update() failed\n");
      hash_ip_free(&ht);
      return -1;
    }

  }

/*  hash_ip_stat_dump(&ht); */

  hash_ip_dump(&ht, cc, sort_order, options, nflows, noctets, npackets,
    nduration);

  if (hash_ip_free(&ht)) {
    fprintf(stderr, "hash_ip_free() failed\n");
    return -1;
  }

  *total_flows = nflows;

  return 0;
}

/*
/* function: flow_stat12
/*
/*  flows,octets,packets,duration by IP protocol
/*
*/
int flow_stat12(fs, sort_order, total_flows, options)
struct flow_stream *fs;
int sort_order, options;
u_int64 *total_flows;
{
  u_int64 flows[256], octets[256], packets[256], duration[256];
  u_int64 nflows, noctets, npackets, nduration;
  u_int index[256];
  struct flow_data *fdata;
  int32 i;
  extern struct sym_table sym_ipprot[];

  bzero(&flows, sizeof flows);
  bzero(&octets, sizeof octets);
  bzero(&packets, sizeof packets);
  bzero(&duration, sizeof duration);
  nflows = noctets = npackets = nduration = 0;

  /* initialize index */
  for (i = 0; i < 256; ++i)
    index[i] = i;

  while ((fdata = flow_read(fs))) {

    ++nflows;
    noctets += fdata->dOctets;
    npackets += fdata->dPkts;
    nduration += (fdata->Last - fdata->First);

    flows[fdata->prot] ++;
    octets[fdata->prot] += fdata->dOctets;
    packets[fdata->prot] += fdata->dPkts;
    duration[fdata->prot] += (fdata->Last - fdata->First);
  }

  tbl_out1(sort_order, options, index, flows, octets, packets, duration,
    256, nflows, noctets, npackets, nduration, "protocol  ", sym_ipprot);

  *total_flows = nflows;

  return 0;
}

/*
/* function: flow_stat13
/*
/*  octets for flow duration plot data
/*
*/
int flow_stat13(fs, sort_order, total_flows, options)
struct flow_stream *fs;
int sort_order, options;
u_int64 *total_flows;
{
  struct flow_data *fdata;
  u_long ymin, ymax;
  u_long xmin, xmax;
  extern char cc;
  u_int64 nflows;

  nflows = 0;

  xmin = ymin = 0xFFFFFFFF;
  xmax = ymax = 0;

  printf("%c\n%c start      octets\n%c end        octets\n%c\n", cc, cc, cc, cc);

  while ((fdata = flow_read(fs))) {

    ++nflows;

    if (fdata->Last > xmax)
      xmax = fdata->Last;

    if (fdata->First < xmin)
      xmin = fdata->First;

    if (fdata->dOctets > ymax)
      ymax = fdata->dOctets;

    if (fdata->dOctets < ymin)
      ymin = fdata->dOctets;

    printf("%-10lu   %-10lu\n", (u_long)fdata->First,
      (u_long)fdata->dOctets);
    printf("%-10lu   %-10lu\n\n", (u_long)fdata->Last,
      (u_long)fdata->dOctets);

  }

  printf("%c xmin=%-10lu   ymin=%-10lu   xmax=%-10lu   ymax=%-10lu\n",
    cc, xmin, ymin, xmax, ymax);

  *total_flows = nflows;

  return 0;
}

/*
/* function: flow_stat14
/*
/*  octets for flow duration plot data
/*
*/
int flow_stat14(fs, sort_order, total_flows, options)
struct flow_stream *fs;
int sort_order, options;
u_int64 *total_flows;
{
  struct flow_data *fdata;
  u_long ymin, ymax;
  u_long xmin, xmax;
  u_int64 nflows;

  nflows = 0;

  xmin = ymin = 0xFFFFFFFF;
  xmax = ymax = 0;

  printf("%c\n%c start     packets\n%c end       packets\n%c\n", cc, cc, cc, cc);

  while ((fdata = flow_read(fs))) {

    if (fdata->Last > xmax)
      xmax = fdata->Last;

    if (fdata->First < xmin)
      xmin = fdata->First;

    if (fdata->dOctets > ymax)
      ymax = fdata->dPkts;

    if (fdata->dOctets < ymin)
      ymin = fdata->dPkts;

    printf("%-10lu    %-10lu\n", (u_long)fdata->First,
      (u_long)fdata->dPkts);
    printf("%-10lu    %-10lu\n\n", (u_long)fdata->Last,
      (u_long)fdata->dPkts);

  }

  printf("%c xmin=%-10lu   ymin=%-10lu   xmax=%-10lu   ymax=%-10lu\n",
    cc, xmin, ymin, xmax, ymax);

  *total_flows = nflows;

  return 0;
}


/*
/* function: flow_stat15
/*
/*  short summary
/*
/* returns 0 for okay.
*/
int flow_stat15(fs, sort_order, total_flows, options)
struct flow_stream *fs;
int sort_order, options;
u_int64 *total_flows;
{
  u_int64 octets, packets;
  double mbytes;
  struct flow_data *fdata;
  char fmt_buf[256];
  u_int len;
  extern char cc;
  u_int64 nflows;

  nflows = 0;

  octets = packets = 0LL;

  while ((fdata = flow_read(fs))) {
    ++nflows;
    octets += fdata->dOctets;
    packets += fdata->dPkts;
  }

  mbytes = (double)octets / (double)(1024*1024);

  printf("%c\n%c octets            packets             mbytes\n%c\n", cc, cc, cc);

  len = fmt_uint64(fmt_buf, octets, FMT_PAD_RIGHT);
  len += fmt_uint64(fmt_buf+len, packets, FMT_PAD_RIGHT);

  printf("%s%3.3f\n", fmt_buf, mbytes);

  *total_flows = nflows;

  return 0;

} /* flow_stat15 */

/*
/* function: flow_stat16
/*
/*  Next hop flows,octets,packets,duration
/*
/* returns 0 for okay.
*/
int flow_stat16(fs, sort_order, total_flows, options)
struct flow_stream *fs;
int sort_order, options;
u_int64 *total_flows;
{
  struct flow_data *fdata;
  struct hash_ip_rec rec;
  struct hash_ip_table ht;
  extern int ip_net_only;
  extern char cc;
  u_int64 nflows, noctets, npackets, nduration;

  nflows = noctets = npackets = nduration = 0;

  if (hash_ip_init(&ht)) {
    fprintf(stderr, "hash_ip_init() failed\n");
    return -1;
  }

  while ((fdata = flow_read(fs))) {

    ++nflows;
    noctets += fdata->dOctets;
    npackets += fdata->dPkts;
    nduration += (fdata->Last - fdata->First);

    rec.addr = fdata->nexthop;
    rec.nflows = 1;
    rec.noctets = fdata->dOctets;
    rec.npackets = fdata->dPkts;
    rec.etime = (fdata->Last - fdata->First);

    if (ip_net_only) {
      if ((rec.addr & 0x80000000) == 0)
        rec.addr &= 0xFF000000;
      else if ((rec.addr & 0xC0000000) == 0x80000000)
        rec.addr &= 0xFFFF0000;
      else if ((rec.addr & 0xC0000000) == 0xC0000000)
        rec.addr &= 0xFFFFFF00;
    }

    if (hash_ip_update(&ht, &rec) == -1) {
      fprintf(stderr, "hash_ip_update() failed\n");
      hash_ip_free(&ht);
      return -1;
    }

  }

/*  hash_ip_stat_dump(&ht); */

  hash_ip_dump(&ht, cc, sort_order, options, nflows, noctets, npackets,
    nduration);

  if (hash_ip_free(&ht)) {
    fprintf(stderr, "hash_ip_free() failed\n");
    return -1;
  }

  *total_flows = nflows;

  return 0;
} /* flow_stat16 */

/*
/* function: flow_stat17
/*
/*  input interface flows,octets,packets,duration
/*
/* returns 0 for okay.
*/
int flow_stat17(fs, sort_order, total_flows, options)
struct flow_stream *fs;
int sort_order, options;
u_int64 *total_flows;
{
  u_int64 flows[65536], octets[65536], packets[65536], duration[65536];
  u_int64 nflows, noctets, npackets, nduration;
  u_int index[65536];
  struct flow_data *fdata;
  int32 i;

  bzero(&flows, sizeof flows);
  bzero(&octets, sizeof octets);
  bzero(&packets, sizeof packets);
  bzero(&duration, sizeof duration);
  nflows = noctets = npackets = nduration = 0;

  /* initialize index */
  for (i = 0; i < 65536; ++i)
    index[i] = i;

  while ((fdata = flow_read(fs))) {

    nflows ++;
    noctets += fdata->dOctets;
    npackets += fdata->dPkts;
    nduration += (fdata->Last - fdata->First);

    flows[fdata->input] ++;
    octets[fdata->input] += fdata->dOctets;
    packets[fdata->input] += fdata->dPkts;
    duration[fdata->input] += (fdata->Last - fdata->First);
  }

  /* doesn't make sense here */
  options &= ~OPT_NAMES;

  tbl_out1(sort_order, options, index, flows, octets, packets, duration,
    65536, nflows, noctets, npackets, nduration, "interface ",
    (struct sym_table*)0L);

  *total_flows = nflows;

  return 0;

} /* flow_stat17 */

/*
/* function: flow_stat18
/*
/*  output interface flows,octets,packets,duration 
/*
/* returns 0 for okay.
*/
int flow_stat18(fs, sort_order, total_flows, options)
struct flow_stream *fs;
int sort_order, options;
u_int64 *total_flows;
{
  u_int64 flows[65536], octets[65536], packets[65536], duration[65536];
  u_int64 nflows, noctets, npackets, nduration;
  u_int   index[65536];
  struct flow_data *fdata;
  int32 i;

  bzero(&flows, sizeof flows);
  bzero(&octets, sizeof octets);
  bzero(&packets, sizeof packets);
  bzero(&duration, sizeof duration);
  nflows = noctets = npackets = nduration = 0;

  /* initialize index */
  for (i = 0; i < 65536; ++i)
    index[i] = i;


  while ((fdata = flow_read(fs))) {

    nflows++;
    noctets += fdata->dOctets;
    npackets += fdata->dPkts;
    nduration += (fdata->Last - fdata->First);

    flows[fdata->output] ++;
    octets[fdata->output] += fdata->dOctets;
    packets[fdata->output] += fdata->dPkts;
    duration[fdata->output] += (fdata->Last - fdata->First);
  }

  /* doesn't make sense here */
  options &= ~OPT_NAMES;

  tbl_out1(sort_order, options, index, flows, octets, packets, duration,
    65536, nflows, noctets, npackets, nduration, "interface ",
    (struct sym_table*)0L);

  *total_flows = nflows;

  return 0;

} /* flow_stat18 */

/*
/* function: flow_stat19
/*
/*  Source AS flows,octets,packets,duration
/*
/* returns 0 for okay.
*/
int flow_stat19(fs, sort_order, total_flows, options)
struct flow_stream *fs;
int sort_order, options;
u_int64 *total_flows;
{
  u_int64 flows[65536], octets[65536], packets[65536], duration[65536];
  u_int64 nflows, noctets, npackets, nduration;
  u_int index[65536];
  struct flow_data *fdata;
  int32 i;
  extern struct sym_table sym_asn[];

  /* initialize index */
  for (i = 0; i < 65536; ++i)
    index[i] = i;


  bzero(&flows, sizeof flows);
  bzero(&octets, sizeof octets);
  bzero(&packets, sizeof packets);
  bzero(&duration, sizeof duration);
  nflows = noctets = npackets = nduration = 0;

  while ((fdata = flow_read(fs))) {

    ++nflows;
    noctets += fdata->dOctets;
    npackets += fdata->dPkts;
    nduration += (fdata->Last - fdata->First);

    flows[fdata->src_as] ++;
    octets[fdata->src_as] += fdata->dOctets;
    packets[fdata->src_as] += fdata->dPkts;
    duration[fdata->src_as] += (fdata->Last - fdata->First);
  }

  tbl_out1(sort_order, options, index, flows, octets, packets, duration,
    65536, nflows, noctets, npackets, nduration, "src AS    ", 
    sym_asn);

  *total_flows = nflows;

  return 0;
} /* flow_stat19 */

/*
/* function: flow_stat20
/*
/*  Source AS flows,octets,packets,duration
/*
/* returns 0 for okay.
*/
int flow_stat20(fs, sort_order, total_flows, options)
struct flow_stream *fs;
int sort_order, options;
u_int64 *total_flows;
{
  u_int64 flows[65536], octets[65536], packets[65536], duration[65536];
  u_int64 nflows, noctets, npackets, nduration;
  u_int index[65536];
  struct flow_data *fdata;
  int32 i;
  extern struct sym_table sym_asn[];

  /* initialize index */
  for (i = 0; i < 65536; ++i)
    index[i] = i;


  bzero(&flows, sizeof flows);
  bzero(&octets, sizeof octets);
  bzero(&packets, sizeof packets);
  bzero(&duration, sizeof duration);
  nflows = noctets = npackets = nduration = 0;

  while ((fdata = flow_read(fs))) {

    ++nflows;
    noctets += fdata->dOctets;
    npackets += fdata->dPkts;
    nduration += (fdata->Last - fdata->First);

    flows[fdata->dst_as] ++;
    octets[fdata->dst_as] += fdata->dOctets;
    packets[fdata->dst_as] += fdata->dPkts;
    duration[fdata->dst_as] += (fdata->Last - fdata->First);
  }

  tbl_out1(sort_order, options, index, flows, octets, packets, duration,
    65536, nflows, noctets, npackets, nduration, "dst AS    ", 
    sym_asn);

  *total_flows = nflows;

  return 0;
} /* flow_stat20 */


int tbl_out1(sort_order, options, index, flows, octets, packets, duration,
    nindex, nflows, noctets, npackets, nduration, title, sym)
int sort_order, options;
u_int *index, nindex;
u_int64 nflows, noctets, npackets, nduration;
u_int64 *flows, *octets, *packets, *duration;
char *title;
struct sym_table sym[];
{

  char fmt_buf[256];
  int32 i, start, end, increment, x;
  int s, len;
  extern char cc;
  u_int64 t_flows, t_octets, t_packets, t_duration;
  u_int plines;

  s = abs(sort_order);
  x = 0;
  t_flows = t_octets = t_packets = t_duration = 0LL;

  if (s == 0) /* no sorting */
    ;
  else if (s == 1) /* port */
    ;
  else if (s == 2) /* flows */
    sort_64uint64(flows, index, nindex);
  else if (s == 3) /* octets */
    sort_64uint64(octets, index, nindex);
  else if (s == 4) /* packets */
    sort_64uint64(packets, index, nindex);
  else if (s == 5) /* duration */
    sort_64uint64(duration, index, nindex);
  else {
    printf("%c can't sort on field %d.\n", cc, sort_order);
    return -1;
  }

  if (sort_order >= 0)
    start = nindex - 1, end = -1, increment = -1;
  else
    start = 0, end = nindex, increment = 1;

  /* header */
  if (options & OPT_PERCENT)
    printf("%c\n%c %sflows    octets   packets  duration\n%c\n",
      cc, cc, title, cc);
  else
    printf("%c\n%c %sflows                 octets                packets               duration\n%c\n", cc, cc, title, cc);

  /*
  /* if OPT_PERCENT and we're doing tallying, then the # of lines
  /* that will be output needs to be known first
  */
  for (i = start; i != end; i += increment)
    if (flows[index[i]])
      ++plines;

  for (i = start; i != end; i += increment) {


    if (flows[index[i]]) {

      /* print a tally line ? */
      if ( (options & OPT_TALLY) && x && (!(x % tally))) {

        if (options & OPT_PERCENT) {

          printf("%cSUB %-6.3f  %-6.3f   %-6.3f   %-6.3f   %-6.3f\n", 
            cc, ((double)x/(double)plines)*100,
            ((double)t_flows/(double)nflows)*100,
            ((double)t_octets/(double)noctets)*100,
            ((double)t_packets/(double)npackets)*100,
            ((double)t_duration/(double)nduration)*100);
        } /* options & OPT_PERCENT */
      } /* print tally */

      if (options & OPT_PERCENT) {

        printf("%-10.10s  %-6.3f   %-6.3f   %-6.3f   %-6.3f\n", 
          sym_pr_uint16(sym, (u_int16)index[i], FMT_JUST_LEFT,
          options), ((double)flows[index[i]]/(double)nflows)*100,
          ((double)octets[index[i]]/(double)noctets)*100,
          ((double)packets[index[i]]/(double)npackets)*100,
          ((double)duration[index[i]]/(double)nduration)*100);

      } else {

        len = fmt_uint64(fmt_buf, flows[index[i]], FMT_PAD_RIGHT);
        fmt_buf[len++] = ' '; fmt_buf[len++] = ' '; 

        len += fmt_uint64(fmt_buf+len, octets[index[i]], FMT_PAD_RIGHT);
        fmt_buf[len++] = ' '; fmt_buf[len++] = ' '; 

        len += fmt_uint64(fmt_buf+len, packets[index[i]],
          FMT_PAD_RIGHT);
        fmt_buf[len++] = ' '; fmt_buf[len++] = ' '; 

        len += fmt_uint64(fmt_buf+len, duration[index[i]],
          FMT_JUST_LEFT);

        printf("%-10.10s  %s\n", sym_pr_uint16(sym,
          (u_int16)index[i], FMT_JUST_LEFT, options), fmt_buf);
      }

      t_flows += flows[index[i]];
      t_octets += octets[index[i]];
      t_packets += packets[index[i]];
      t_duration += duration[index[i]];
      ++x;

    } /* flows[index[i]] */
  }

  return 0;

} /* tbl_out1 */

