#include <sys/time.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <sys/signal.h>
#include <sys/errno.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "flow.h"
#include "support.h"

int debug;
int done;

void usage();
void sig_quit();

int main(argc, argv)
int argc;
char **argv;
{

  extern char *optarg;
  extern int optind, errno;
  struct flow_stream fs;
  int fport, fsockfd, len, i;
  int n, x, line_buf;
  fd_set rfd;
  struct flow_data fdata;
  struct sockaddr_in fsocaddr;
  struct flow_header fh;
  u_long j, k;
  int z_level, out_byte_order;
  extern int done;
  char hostname[MAXDNAME+1];
  char rcv_buf[FLOW_RECV_BUFSIZE];
  struct flow_pdu_v1 *fpdu_v1;
  struct flow_pdu_v5 *fpdu_v5;
  u_int16 fpdu_version;

  fport = FLOWPORT;
  bzero (&fs, sizeof fs);
  bzero(&fh, sizeof fh);
  line_buf = 1;
  z_level = 0;
  done = 0;
#if BYTE_ORDER == BIG_ENDIAN
  out_byte_order = BIG_ENDIAN;
#endif /* BYTE_ORDER == BIG_ENDIAN */
#if BYTE_ORDER == LITTLE_ENDIAN
  out_byte_order = LITTLE_ENDIAN;
#endif /* BYTE_ORDER == LITTLE_ENDIAN */

  fpdu_v1 = (struct flow_pdu_v1*)rcv_buf;
  fpdu_v5 = (struct flow_pdu_v5*)rcv_buf;

  while ((i = getopt(argc, argv, "b:C:d:h?lp:z:")) != -1)

    switch (i) {

    case 'b': /* output byte order */
      if (!strcasecmp(optarg, "little"))
        out_byte_order = LITTLE_ENDIAN;
      else if (!strcasecmp(optarg, "big"))
        out_byte_order = BIG_ENDIAN;
      else {
        fprintf(stderr, "expecting \"big\" or \"little\" at -b\n");
        exit (1);
      }
      break;

    case 'C': /* comment field */
      if (strlen(optarg) > (FH_COMMENTS_LEN-1)) {
        fprintf(stderr, "comments limited to %d bytes.\n", 
          (FH_COMMENTS_LEN-1));
        exit (1);
      }
      strcpy((char*)&fh.comments, optarg);
      break;

    case 'd': /* debug */
      debug = atoi(optarg);
      break;

    case 'h': /* help */
    case '?':
      usage();
      exit (0);
      break;

    case 'l': /* turn off buffered output */
      line_buf = 0;
      break;

    case 'p': /* udp port */
      fport = atoi(optarg);
      break;

    case 'z': /* compress level */
      z_level = atoi(optarg);
      if ((z_level < 0) || (z_level > 9)) {
        fprintf(stderr, "Compression level must be between 0 and 9\n");
        exit (1);
      }
      break;

    default:
      usage();
      exit (1);
      break;

    } /* switch */

  /* handle SIGQUIT */
  signal(SIGQUIT, sig_quit);
  signal(SIGHUP, sig_quit);
  signal(SIGINT, sig_quit);
  signal(SIGTERM, sig_quit);

  /* get hostname */
  if (gethostname((char*)&hostname, (int)MAXDNAME) == -1) {
    fprintf(stderr, "gethostname(): %s\n", strerror(errno));
    exit (1);
  }

  bcopy(hostname, fh.hostname, FH_HOSTNAME_LEN);
  fh.hostname[FH_HOSTNAME_LEN-1] = 0; /* ensure null terminated */

  fh.magic1 = FF_MAGIC1;
  fh.magic2 = FF_MAGIC2;
  fh.s_version = FSTREAM_VERSION;
  fh.d_version = FF_D_VERSION_UNKNOWN;

#if BYTE_ORDER == LITTLE_ENDIAN
  fh.byte_order = FF_LITTLE_ENDIAN;
#endif /* BYTE_ORDER == LITTLE_ENDIAN */

#if BYTE_ORDER == BIG_ENDIAN
  fh.byte_order = FF_BIG_ENDIAN;
#endif /* BYTE_ORDER == BIG_ENDIAN */


  bzero(&fsocaddr, sizeof(fsocaddr));
  fsocaddr.sin_family = AF_INET;
  fsocaddr.sin_addr.s_addr = htonl(INADDR_ANY);
  fsocaddr.sin_port = htons(fport);

  if ((fsockfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
    fprintf(stderr, "socket(): %s\n", strerror(errno));
    exit (1);
  }

  if (bind(fsockfd, (struct sockaddr*)&fsocaddr, sizeof(fsocaddr)) < 0) {
    fprintf(stderr, "bind(): %s\n", strerror(errno));
    exit (1);
  }

  len = sizeof(fsocaddr);

  /* output to stdout */
  fs.fd = 1;
  fs.z_level = z_level;

#if BYTE_ORDER == BIG_ENDIAN
  if (out_byte_order == LITTLE_ENDIAN)
    swap_flow_header(&fh);
#endif /* BYTE_ORDER == LITTLE_ENDIAN */

#if BYTE_ORDER == LITTLE_ENDIAN
  if (out_byte_order == BIG_ENDIAN)
    swap_flow_header(&fh);
#endif /* BYTE_ORDER == LITTLE_ENDIAN */


  if (writen(1, &fh, sizeof fh) == -1) {
    fprintf(stderr, "write(): %s\n", strerror(errno));
    exit (1);
  }

  while (1) {

    FD_ZERO (&rfd);
    FD_SET (fsockfd, &rfd);

    if (select (fsockfd+1, &rfd, (fd_set *)0, (fd_set *)0, 
      (struct timeval*)0L) < 0)  {
      if (errno != EINTR) {
        fprintf(stderr, "select(): %s\n", strerror(errno));
        exit (1);
      }
    }

    if (done) {
      fprintf(stderr, "Cleaning up.\n");
      break;
    }

    if (FD_ISSET(fsockfd, &rfd)) {

      if ((n = recvfrom(fsockfd, (char*)&rcv_buf, sizeof rcv_buf,
        0, (struct sockaddr*)&fsocaddr, &len)) == -1) {
        fprintf(stderr, "recvfrom(): %s\n", strerror(errno));
        exit (1);
      }

      /* the first 2 bytes are the flow version */
      bcopy(rcv_buf, (char*)&fpdu_version, 2);

#if BYTE_ORDER == LITTLE_ENDIAN
      SWAPINT16(fpdu_version);
#endif /* BYTE_ORDER == LITTLE_ENDIAN */

      if (fpdu_version == 1) {


#if BYTE_ORDER == LITTLE_ENDIAN
        if (out_byte_order == LITTLE_ENDIAN)
          swap_flow_pdu_v1(fpdu_v1, BIG_ENDIAN);
        else if (out_byte_order == BIG_ENDIAN)
          swap_flow_pdu2_v1(fpdu_v1, BIG_ENDIAN);
#endif /* BYTE_ORDER == LITTLE_ENDIAN */

        /* convert the pdu to internal format */
  
        j = fpdu_v1->sysUpTime / 1000;
        k = fpdu_v1->sysUpTime % 1000;
  
        fdata.unix_secs = fpdu_v1->unix_secs;
        fdata.unix_msecs = fpdu_v1->unix_nsecs / 1000000;
  
        fdata.unix_secs -= j;
        if (k > fdata.unix_msecs) {
          -- fdata.unix_secs;
          fdata.unix_msecs += 1000;
        }
        fdata.unix_msecs -= k;
  
        for (x = 0; x < fpdu_v1->count; ++x) {
  
          fdata.srcaddr = fpdu_v1->records[x].srcaddr;
          fdata.dstaddr = fpdu_v1->records[x].dstaddr;
          fdata.nexthop = fpdu_v1->records[x].nexthop;
          fdata.input = fpdu_v1->records[x].input;
          fdata.output = fpdu_v1->records[x].output;
          fdata.dPkts = fpdu_v1->records[x].dPkts;
          fdata.dOctets = fpdu_v1->records[x].dOctets;
          fdata.First = fpdu_v1->records[x].First;
          fdata.Last = fpdu_v1->records[x].Last;
          fdata.dstport = fpdu_v1->records[x].dstport;
          fdata.srcport = fpdu_v1->records[x].srcport;
          fdata.prot = fpdu_v1->records[x].prot;
          fdata.tos = fpdu_v1->records[x].tos;
          fdata.flags = fpdu_v1->records[x].flags;
          fdata.tcp_retx_cnt = fpdu_v1->records[x].tcp_retx_cnt;
          fdata.tcp_retx_secs = fpdu_v1->records[x].tcp_retx_secs;
          fdata.tcp_misseq_cnt = fpdu_v1->records[x].tcp_misseq_cnt;
  
#if BYTE_ORDER == BIG_ENDIAN
          if (out_byte_order == LITTLE_ENDIAN)
            swap_flow_data(&fdata);
#endif /* BYTE_ORDER == BIG_ENDIAN */
  
          if (flow_write(&fdata, &fs) == -1) {
            fprintf(stderr, "flow_write(): failed\n");
            exit (1);
          }
  
          /* flush ? */
          if (!z_level && !line_buf) {
            if (flow_flush(&fs) == -1) {
              fprintf(stderr, "flow_flush(): failed\n");
              exit (1);
            }
          }
        } /* for x */
      } else if (fpdu_version == 5) {

#if BYTE_ORDER == LITTLE_ENDIAN
        if (out_byte_order == LITTLE_ENDIAN)
          swap_flow_pdu_v5(fpdu_v5, BIG_ENDIAN);
        else if (out_byte_order == BIG_ENDIAN)
          swap_flow_pdu2_v5(fpdu_v5, BIG_ENDIAN);
#endif /* BYTE_ORDER == LITTLE_ENDIAN */
  
        /* convert the pdu to internal format */
  
        j = fpdu_v5->sysUpTime / 1000;
        k = fpdu_v5->sysUpTime % 1000;
  
        fdata.unix_secs = fpdu_v5->unix_secs;
        fdata.unix_msecs = fpdu_v5->unix_nsecs / 1000000;
  
        fdata.unix_secs -= j;
        if (k > fdata.unix_msecs) {
          -- fdata.unix_secs;
          fdata.unix_msecs += 1000;
        }
        fdata.unix_msecs -= k;
  
        for (x = 0; x < fpdu_v5->count; ++x) {
  
          fdata.srcaddr = fpdu_v5->records[x].srcaddr;
          fdata.dstaddr = fpdu_v5->records[x].dstaddr;
          fdata.nexthop = fpdu_v5->records[x].nexthop;
          fdata.input = fpdu_v5->records[x].input;
          fdata.output = fpdu_v5->records[x].output;
          fdata.dPkts = fpdu_v5->records[x].dPkts;
          fdata.dOctets = fpdu_v5->records[x].dOctets;
          fdata.First = fpdu_v5->records[x].First;
          fdata.Last = fpdu_v5->records[x].Last;
          fdata.dstport = fpdu_v5->records[x].dstport;
          fdata.srcport = fpdu_v5->records[x].srcport;
          fdata.flags = fpdu_v5->records[x].tcp_flags;
          fdata.prot = fpdu_v5->records[x].prot;
          fdata.tos = fpdu_v5->records[x].tos;
          fdata.src_as = fpdu_v5->records[x].src_as;
          fdata.dst_as = fpdu_v5->records[x].dst_as;
          fdata.src_mask = fpdu_v5->records[x].src_mask;
          fdata.dst_mask = fpdu_v5->records[x].dst_mask;
          fdata.drops = fpdu_v5->records[x].drops;
  
#if BYTE_ORDER == BIG_ENDIAN
          if (out_byte_order == LITTLE_ENDIAN)
            swap_flow_data(&fdata);
#endif /* BYTE_ORDER == BIG_ENDIAN */
  
          if (flow_write(&fdata, &fs) == -1) {
            fprintf(stderr, "flow_write(): failed\n");
            exit (1);
          }

          /* flush ? */
          if (!z_level && !line_buf) {
            if (flow_flush(&fs) == -1) {
              fprintf(stderr, "flow_flush(): failed\n");
              exit (1);
            }
          }

        } /* for x */
      } else {
        fprintf(stderr, "unknown flow pdu version (%u).\n",
          (unsigned int)fpdu_version);
          continue;
        }
    } /* if FD_ISSET */
  } /* while 1 */

  if (flow_flush(&fs) == -1) {
    fprintf(stderr, "flow_flush(): failed\n");
    exit (1);
  }

  if (flow_write_end(&fs) == -1) {
    fprintf(stderr, "flow_write_end(): failed\n");
    exit (1);
  }

  return 0;

} /* main */

void usage() {

  fprintf(stderr, "flow-receive:\n\n");
  fprintf(stderr, " -b s set output byte order (little or big)\n");
  fprintf(stderr, " -C s set comment field in output\n");
  fprintf(stderr, " -d # set debug level.\n");
  fprintf(stderr, " -p # udp port to receive on.\n");
  fprintf(stderr, "\n\n");

} /* usage */

void sig_quit(sig)
int sig;
{
  extern int done;
  done = 1;
} /* sig_quit */
