/***********************************************************************
 * fdg.c  Flow Datagram Get
 *
 * Simple client that listens to configured udp port for flow-stat
 * messages and prints out some interesting fields.
 *
 * dkerr - 3/11/96
 ***********************************************************************/

#include <sys/types.h>
#include <fcntl.h>
#include <stdio.h>
#include <sys/socket.h>
#include <netinet/in.h>

typedef unsigned long ulong;
typedef ulong ipaddrtype;
typedef unsigned char uchar;

#define PROTO_UDP 17
#define MAXFLOWS  24
#define FLOWV1     1
struct flowpkt {
    ushort version;         /* 1 for now. */
    ushort count;           /* The number of records in PDU. */
    ulong  SysUptime;       /* Current time in millisecs since router booted */
    ulong  unix_secs;       /* Current seconds since 0000 UTC 1970 */
    ulong  unix_nsecs;      /* Residual nanoseconds since 0000 UTC 1970 */
    
    struct flowrec {
	ipaddrtype srcaddr;    /* Source IP Address */
	ipaddrtype dstaddr;    /* Destination IP Address */
	ipaddrtype nexthop;    /* Next hop router's IP Address */
	ushort input;          /* Input interface index */
	ushort output;         /* Output interface index */
	
	ulong dPkts;           /* Packets sent in Duration */
	ulong dOctets;         /* Octets sent in Duration. */
	ulong First;           /* SysUptime at start of flow */
	ulong Last;            /* and of last packet of flow */
	
	ushort dstport;        /* TCP/UDP destination port number or equiv */
	ushort srcport;        /* TCP/UDP source port number or equivalent */
	ushort pad;
	uchar  prot;           /* IP protocol, e.g., 6=TCP, 17=UDP, ... */
	uchar  tos;            /* IP Type-of-Service */
	
	uchar  flags;          /* Reason flow was discarded, etc...  */
	uchar  tcp_retx_cnt;   /* Number of mis-seq with delay > 1sec */
	uchar  tcp_retx_secs;  /* Cumulative secs between mis-sequenced pkts */
	uchar  tcp_misseq_cnt; /* Number of mis-sequenced tcp pkts seen */
	ulong  reserved;
    } records[MAXFLOWS];
} flows;
    

main(argc, argv)
int argc;
char *argv[];
{
    int soc, sts, alen, cnt, len;
    ulong pkts, bytes, duration;
    struct sockaddr_in socadr;
    ushort port;
    uchar raddr[4], sadr[4], dadr[4];
    char srcstr[16], dststr[16];

    socadr.sin_family = AF_INET;
    socadr.sin_port = 9995;
    socadr.sin_addr.s_addr = INADDR_ANY;
    
    if (argc > 1) {
	if ((port = atoi(argv[1])) == 0) {
	    fprintf(stderr, "usage: fdg [port_num]\n");
	    exit(1);
	}
	socadr.sin_port = port;
    }

    if ((soc = socket(PF_INET, SOCK_DGRAM, PROTO_UDP)) < 0)
	exit(1);

    if ((sts = bind(soc, &socadr, sizeof(socadr))) < 0) {
	fprintf(stderr, "bind failed\n");
	exit(1);
    }

    while (1) {
	alen = sizeof(socadr);
	len = recvfrom(soc, &flows, sizeof(flows), 0, &socadr, &alen);
	if (len < 0) {
	    fprintf(stderr, "recvfrom failed\n");
	    exit(1);
	}
	*(ulong *)raddr = socadr.sin_addr.s_addr;
	printf("\nFlowstat from %d.%d.%d.%d, %s",
	       raddr[0], raddr[1], raddr[2], raddr[3],
	       ctime(&flows.unix_secs));

	if (flows.version != FLOWV1 || flows.count > MAXFLOWS) {
	    fprintf(stderr, "bogus version/count\n");
	    exit(1);
	}

	printf("Source          Sport Destination     Dport Prot ");
	printf("Packets B/Pkt Duration Flags\n");
	for (cnt=0; cnt<flows.count; cnt++) {
	    *(ulong *)sadr = flows.records[cnt].srcaddr;
	    *(ulong *)dadr = flows.records[cnt].dstaddr;
	    sprintf(srcstr, "%d.%d.%d.%d", sadr[0], sadr[1], sadr[2], sadr[3]);
	    sprintf(dststr, "%d.%d.%d.%d", dadr[0], dadr[1], dadr[2], dadr[3]);
	    printf("%-15s %5d %-15s %5d %4d ",
		   srcstr, flows.records[cnt].srcport,
		   dststr, flows.records[cnt].dstport,
		   flows.records[cnt].prot);
	    pkts = flows.records[cnt].dPkts;
	    bytes = flows.records[cnt].dOctets;
	    duration = flows.records[cnt].Last - flows.records[cnt].First;
	    printf("%7d %5d %4d.%03d    %02x\n",
		   pkts, (pkts) ? bytes/pkts : 0,
		   duration / 1000, duration % 1000,
		   flows.records[cnt].flags);
	}
    }
}
