/* $Id: expiretable.c,v 1.59 2006/01/18 22:47:01 gsson Exp $ */

/*
 * Copyright (c) 2005 Henrik Gustafsson <henrik.gustafsson@fnord.se>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/fcntl.h>
#include <sys/types.h>
#include <sys/limits.h>
#include <pwd.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <syslog.h>
#include <err.h>
#include <errno.h>

#include "ioctl_helpers.h"
#include "messages.h"

#define SLEEP_MIN 10
#define SLEEP_MAX 100000000 /* As defined by nanosleep(2) */

long
lmax(long a,long b) {
	return (a > b)?a:b;
}

long
lmin(long a,long b) {
	return (a < b)?a:b;
}

int
sleep_overflow(unsigned int base, unsigned int add, unsigned int factor) {
	if (ULONG_MAX / factor < add || ULONG_MAX - (add * factor) < base|| base + (add * factor) > SLEEP_MAX) {
		warnx("sleep_overflow(): Invalid sleep time.");
		return 1;
	}
	return 0;
}

unsigned long
parse_age(const char *t) {
	char *endp;
	unsigned long seconds = 0;
	unsigned long l;

	while (*t) {
		errno = 0;
        l = strtoul(t, &endp, 10);
        
		/* Invalid numeral */
        if ((l == 0 && errno == EINVAL) || (l == ULONG_MAX && errno == ERANGE)) {
        		warnx("parse_age(): Invalid numeric.");
                return -1;
        }
		/* Make sure there was a number to parse */
        if (t == endp) {
       		warnx("parse_age(): Missing numeric.");
        	return -1;
        }
        
        if (!*endp) {
	    	if (sleep_overflow(seconds, l, 1)) {
	                return -1;
	        	}
	        	else {
	        		seconds += l;
	        		return seconds;
	        	}
        }

        switch(*endp) {
                case 'd':
                case 'D': 
                	if (sleep_overflow(seconds, l, 24 * 60 * 60)) {
                        return -1;
                	}
                	else {
                		seconds += l * 24 * 60 * 60;
                	}
                	break;
                case 'h':
                case 'H': 
                	if (sleep_overflow(seconds, l, 60 * 60)) {
                        return -1;
                	}
                	else {
                		seconds += l * 60 * 60;
                	}
                	break;
                case 'm':
                case 'M': 
                	if (sleep_overflow(seconds, l, 60)) {
                        return -1;
                	}
                	else {
                		seconds += l * 60;
                	}
                	break;
                case 's':
                case 'S': 
                	if (sleep_overflow(seconds, l, 1)) {
                        return -1;
                	}
                	else {
                		seconds += l;
                	}
                	break;
                default:
					warnx("parse_age(): Invalid suffix.");
                	return -1;
        }

        t = endp + 1;
	}
	return seconds;
} 

void
print_address(u_int32_t a, u_int8_t net) {
	printf("%d.%d.%d.%d/%d", a >> 24 & 255, a >> 16 & 255, a >> 8 & 255, a & 255, net);
}

void
usage(void) {
	extern char *__progname;
	fprintf(stderr, "usage: %s [-dnvp] [-a anchor] [-t age] table\n", __progname);
	exit(1);
}

void
drop_privileges() {
	struct passwd *pw;
	
	if ((pw = getpwnam("nobody")) == NULL) {
		errx(-1, "Failed to get pw-entry for user \"nobody\".");
	}
	if (setgroups(1, &pw->pw_gid) || setegid(pw->pw_gid) || setgid(pw->pw_gid)
			|| seteuid(pw->pw_uid) || setuid(pw->pw_uid)) {
		err(1, "Failed to drop privileges");
	}

}

int
main(int argc, char *const *argv) {
	extern char *__progname;
	
	int dev;	
	struct pfr_astats *astats;
	struct pfr_table target;
	struct pfr_addr *del_addrs_list;
	int astats_count;
	int del_addrs_count;
	int del_addrs_result;
	
	unsigned long age;
	int verbose;
	int daemonize;
	int poll;

	long min_timestamp;
	long oldest_entry;
	int ch;
	int i;
	int flags;
		
	
	/* Default values for options */
	
	memset(&target, 0, sizeof(struct pfr_table));
	
	age = 60*60*3; /* 3 hours */
	verbose = 0;
	daemonize = 0;
	poll = 0;
	flags = PFR_FLAG_FEEDBACK;
	
	/* Parse options */
	
	while ((ch = getopt(argc, argv, "a:t:nvdp")) != -1) {
		switch (ch) {
		case 't': {
			/* optarg must be a positive integer within the limits of a 'long'.
			   Also less than 100 million, as defined by nanosleep(2), which is
			  used to implement sleep(3)
			*/
			
			if (optarg[0] == '\0')
				usage();
				
			age = parse_age(optarg);
			
			if (age == -1)
				usage();
			break;
		}
		case 'a': {
			strncpy(target.pfrt_anchor, optarg, sizeof(target.pfrt_anchor));
			break;
		}
		case 'v': {
			verbose++;
			break;
		}
		case 'p': {
			poll=1;
			break;
		}
		case 'n': {
			flags |= PFR_FLAG_DUMMY;
			break;
		}
		case 'd': {
			daemonize = 1;
			poll = 1;
			break;
		}
		default: {
			usage();
			break;
		}
		}
	}
	argc -= optind;
	argv += optind;
	
	if (*argv == NULL) {
		usage();
	}
	
	strncpy(target.pfrt_name, *argv, sizeof(target.pfrt_name));	
		
	dev = open("/dev/pf", O_RDWR);
	
	if (dev == -1) {
		err(-1, "Error opening \"/dev/pf\"");
	}

	drop_privileges();
	
	if(daemonize) {
		if (daemon(0, 0)) {
			err(-1, "daemon()");
		}
		/* Don't want too verbose output when running as a daemon */
		if (daemonize && verbose) {
			verbose = 1;
		}
		openlog(__progname, LOG_PID, LOG_DAEMON);
	}
	
	do {
		min_timestamp = (long)time(NULL) - age;
		oldest_entry = time(NULL);
		astats_count = radix_get_astats(dev, &astats, &target,0);
		if (astats_count > 0) {
			
			del_addrs_list = NULL;
			del_addrs_count = 0;
			for (i = 0; i < astats_count; i++) {
				if (astats[i].pfras_tzero <= min_timestamp) {
					del_addrs_count++;
				}
				else {
					oldest_entry = lmin(oldest_entry, astats[i].pfras_tzero);
				}
			}			
			
			if ((del_addrs_list = malloc(del_addrs_count * sizeof(struct pfr_addr))) == NULL) {
				error(daemonize, -1, "malloc()");
			}
			
			del_addrs_count = 0;
			for (i = 0; i < astats_count; i++) {
				if (astats[i].pfras_tzero <= min_timestamp) {
					del_addrs_list[del_addrs_count] = astats[i].pfras_a;
					del_addrs_count++;
				}
			}
			
			if (del_addrs_count > 0) {
				del_addrs_result = radix_del_addrs(dev, &target, del_addrs_list,
						del_addrs_count, flags);
				if (del_addrs_result  < 0) {
					warningx(daemonize,"%s: Failed to remove address(es).", target.pfrt_name);
				}
	
				if (verbose > 1) {
					/* This should never happen while we're daemonized. */
					for (i = 0; i < del_addrs_count; i++) {
						print_address(
							ntohl(del_addrs_list[i].pfra_ip4addr.s_addr), del_addrs_list[i].pfra_net);
							
						switch(del_addrs_list[i].pfra_fback) {
						case PFR_FB_NONE: {
							printf(": No action taken.\n");
							break;
						}
						case PFR_FB_CONFLICT: {
							printf(": Conflicting entry.\n");
							break;
						}
						case PFR_FB_DUPLICATE: {
							printf(": Already deleted.\n");
							break;
						}
						case PFR_FB_DELETED: {
							printf(": Entry deleted.\n");
							break;
						}
						}
					}
				}
				if (verbose > 0) {
					infox(daemonize, "%s: %d of %d entries deleted.",
						target.pfrt_name, del_addrs_result, astats_count);
				}
				
				free(del_addrs_list);
			}
			else {
				if (verbose > 1) {
					infox(daemonize, "%s: 0 of %d entries deleted.",
						target.pfrt_name, astats_count);
				}
			}
			free(astats);
		}
		else if (astats_count == 0) {
			if (verbose > 1) {
				infox(daemonize, "%s: Table empty.", target.pfrt_name);
			}
			free(astats);
		}
		else {
			warningx(daemonize, "%s: Failed to get table-statistics.",
				target.pfrt_name);
		}
		
		if (poll) {
			long sleeptime = (unsigned int)lmax(age - (time(NULL) - oldest_entry), SLEEP_MIN);
			if (verbose > 1) {
				printf("Sleeping for %ld seconds.\n", sleeptime);
			}
			sleep(lmax(age - (time(NULL) - oldest_entry), SLEEP_MIN) );
		}
	}
	while (poll);
	

	return 0;
}
