/*
    ettercap -- signal handler

    Copyright (C) 2001  ALoR <alor@users.sourceforge.net>, NaGA <crwm@freemail.it>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

    $Id: ec_signal.c,v 1.1.1.1 2001/08/27 19:23:43 alor Exp $
*/

#include "include/ec_main.h"

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <unistd.h>
#if defined (LINUX) && !defined (__USE_GNU)
   #define __USE_GNU    // for strsignal()
#endif
#include <string.h>


#ifdef HAVE_NCURSES
   #ifdef HAVE_NCURSES_H
      #include <ncurses.h>
   #else
      #include <curses.h>
   #endif
   #include "include/ec_interface.h"
   #include "include/ec_interface_sniff_data.h"
   #include "include/ec_interface_plugins.h"

   extern WINDOW *data_source_win;
   extern WINDOW *plugin_window;
   extern short ScreenMode;

   #ifdef HAVE_TERMIOS_H
      #include <termios.h>
      extern struct termios original_term_info;
   #endif
#endif

#include "include/ec_error.h"

#ifdef DEBUG
   #include "include/ec_debug.h"
#endif

// protos...

void Signal_SigBuster(void);
RETSIGTYPE Signal_ILL(int sig);
RETSIGTYPE Signal_SEGV(int sig);
RETSIGTYPE Signal_TERM(int sig);
RETSIGTYPE Signal_WINCH(int sig);
RETSIGTYPE Signal_USR1(int sig);
RETSIGTYPE Signal_USR2(int sig);

//-----------------------

void Signal_SigBuster(void)
{
#ifdef DEBUG
   Debug_msg("Signal_Buster");
#endif

   signal(SIGILL,  Signal_ILL);     // this is received by ettercap if one of its sons get a SIGSEGV
   signal(SIGBUS,  Signal_SEGV);
   signal(SIGSEGV,  Signal_SEGV);
   signal(SIGHUP,   Signal_TERM);
   signal(SIGTERM,  Signal_TERM);
   signal(SIGWINCH, Signal_WINCH);
   signal(SIGCHLD,  SIG_IGN);       // if I kill a forked process it doesn't become a zombie...
   signal(SIGPIPE,  Signal_TERM);   // if ettercap SIGSEGV, illithid receives this sig... (pipe_with_illithid)

   signal(SIGUSR1, Signal_USR1);
   signal(SIGUSR2, Signal_USR2);
}


RETSIGTYPE Signal_ILL(int sig)
{
#ifdef DEBUG
   Debug_msg("[%s] One of my sons got an error...", program_argv0 );
#endif
#ifdef HAVE_NCURSES
   if (ScreenMode)    // restore the cursor
      curs_set(1);

   #ifdef HAVE_TERMIOS_H
      tcsetattr(0, TCSANOW, &original_term_info);
   #endif
#endif

   exit(666);
}


RETSIGTYPE Signal_SEGV(int sig)
{
#ifdef DEBUG
   struct rlimit corelimit = {RLIM_INFINITY, RLIM_INFINITY};
   if (sig == SIGBUS)
      Debug_msg("[%s] Bus Error...", program_argv0 );
   else if (sig == SIGSEGV)
      Debug_msg("[%s] Segmentation Fault...", program_argv0 );
#endif

   if (!strstr(program_argv0, PROGRAM)) kill(getppid(), SIGILL);    // kill the main process

#ifdef HAVE_NCURSES
   if (ScreenMode)    // close the ncurses screen
      Interface_CloseScreen();
#endif


   fprintf (stderr, "\n\033[01m\033[1m Ooops !! Somewhere in the stack a pointer got crazy...\n\n");
   if (sig == SIGBUS)
      fprintf (stderr, " [%s] Bus Error...\033[0m\n\n\n", program_argv0);
   else if (sig == SIGSEGV)
      fprintf (stderr, " [%s] Segmentation Fault...\033[0m\n\n", program_argv0);

   fprintf (stderr, "===========================================================================\n");
   fprintf (stderr, " To report this error follow these steps:\n\n");
#ifndef DEBUG
   fprintf (stderr, "  1) recompile ettercap in debug mode : \n"
                    "  \t\"configure --enable-debug && make clean && make\"\n\n");
   fprintf (stderr, "  2) reproduce the critical situation\n\n");
#else
   fprintf (stderr, "  1) and 2) already done...\n\n");
#endif
   if (sig == SIGBUS)
   {
      fprintf (stderr, "  3) make a report : \"tar zcvf error.tar.gz ettercap_debug.log\"\n\n");
      fprintf (stderr, "  4) mail us the error.tar.gz\n\n\n\n");
      exit(666);
   }

   fprintf (stderr, "  3) make a report : \"tar zcvf error.tar.gz ettercap_debug.log \"\n\n");
   fprintf (stderr, "  4) get the gdb backtrace :\n"
                    "  \t - \"gdb ettercap core\"\n"
                    " \t - at the gdb prompt \"bt\"\n"
                    " \t - at the gdb prompt \"quit\" and return to the shell\n"
                    " \t - copy and paste this output.\n\n");
   fprintf (stderr, " 5) mail us the output of gdb and the error.tar.gz\n");
   fprintf (stderr, "============================================================================\n");

#ifndef DEBUG
   exit(666);
#else // DEBUGGING MODE
   fprintf(stderr, "\n\033[01m\033[1m Overriding any 'ulimit -c 0'...\n"
                   " Setting core size to RLIM_INFINITY...\n\n"
                   " Core dumping... (use the 'core' file for gdb analysis)\033[0m\n\n");
   setrlimit(RLIMIT_CORE, &corelimit);
   signal(sig, SIG_DFL);
   raise(sig);
#endif
}



RETSIGTYPE Signal_TERM(int sig)
{
#ifdef DEBUG
   #ifdef HAVE_STRSIGNAL
      Debug_msg("[%s] Signal handler... (caught SIGNAL: %d) | %s", program_argv0, sig, strsignal(sig));
   #else
      Debug_msg("[%s] Signal handler... (caught SIGNAL: %d)", program_argv0, sig);
   #endif
#endif

   if (!strstr(program_argv0, PROGRAM)) exit(0);    // is not the main process...

#ifdef HAVE_NCURSES
   if (ScreenMode)
      Interface_CloseScreen();
#endif

   #ifdef HAVE_STRSIGNAL
      printf("\n Signal handler... (caught SIGNAL: %d) | %s\n\n", sig, strsignal(sig));
   #else
      printf("\n Signal handler... (caught SIGNAL: %d)\n\n", sig);
   #endif

   kill(-1*getpid(), SIGTERM);  // kill all my forked processes
   exit(1);

}



RETSIGTYPE Signal_WINCH(int sig)
{

#ifdef HAVE_NCURSES
   if (ScreenMode)
   {
      if (data_source_win != NULL)
      {
         #ifdef DEBUG
            Debug_msg("Winching sniff data windows...");
         #endif
         Interface_Sniff_Data_Winch();
      }
#ifdef PERMIT_PLUGINS
      else if (plugin_window != NULL)
      {
         #ifdef DEBUG
            Debug_msg("Winching plugin windows...");
         #endif
         Interface_Plugins_Winch();
      }
#endif
      else
      {
         #ifdef DEBUG
            Debug_msg("Winching windows...");
         #endif
         Interface_Winch();
      }
   }
#endif
   signal(SIGWINCH, Signal_WINCH);
}



RETSIGTYPE Signal_USR1(int sig)
{
   active_dissector = (active_dissector) ? 0 : 1;     // activate/deactivate active dissector (arp based)
   #ifdef DEBUG
      Debug_msg("\tactive_dissector %d", active_dissector);
   #endif
   signal(SIGUSR1, Signal_USR1);
}



RETSIGTYPE Signal_USR2(int sig)
{
   Connection_Mode = (Connection_Mode) ? 0 : 1;
   #ifdef DEBUG
      Debug_msg("\tConnection_Mode %d", Connection_Mode);
   #endif
   signal(SIGUSR2, Signal_USR2);
}

/* EOF */
