/*
    ettercap -- Grell -- HTTPS dissector

    Copyright (C) 2001  ALoR <alor@users.sourceforge.net>, NaGA <crwm@freemail.it>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

    $Id: ec_grell.c,v 1.1.1.1 2001/08/27 19:26:31 alor Exp $
*/

#include "include/ec_main.h"

#if defined (HAVE_OPENSSL) && defined (PERMIT_HTTPS)  // don't compile if you don't have OpenSSL

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#if defined (LINUX) && !defined (__USE_GNU)  // for memmem
   #define __USE_GNU
#endif
#include <string.h>
#include <fcntl.h>

#include <openssl/ssl.h>

#include "include/ec_inet_structures.h"
#include "include/ec_inet_forge.h"
#include "include/ec_dissector.h"
#include "include/ec_buffer.h"
#include "include/ec_error.h"

#ifdef DEBUG
   #include "include/ec_debug.h"
#endif

int Grell_ProxyIP = 0;
int Grell_ProxyPort = 8080;

unsigned int    ServerIP;
unsigned short  ServerPort;
int             proxy_fd, https_fd, client_fd, server_fd, ssl_type;
SSL_CTX         *ssl_ctx_client, *ssl_ctx_server;
SSL             *ssl_client, *ssl_server;
struct          sockaddr_in client_sin, server_sin;

// protos...

void Grell_Dissector(char *payload, CONNECTION *data_to_ettercap);
void Grell_init(void);
void Grell_spawn(void);
void Grell_start(void);
int Grell_Run(void);

// ================================

void Grell_Dissector(char *payload, CONNECTION *data_to_ettercap)
{
   char *buf;

   buf = Inet_Forge_packet( 1500 );                         // prepare the packet for the HTTP dissector
   Inet_Forge_tcp( buf, data_to_ettercap->source_port,      // create a fake tcp header
                        data_to_ettercap->dest_port,
                        0xabadc0de,
                        0xabadc0de,
                        0,
                        payload,
                        data_to_ettercap->datalen);

   Dissector_http(buf, data_to_ettercap, 1, 443);

   Inet_Forge_packet_destroy( buf );

}

static void client_parse(char *buf, int len)
{
   CONNECTION data_to_ettercap;
   SNIFFED_DATA sniff_data_to_ettercap;
   struct in_addr dest;

   memset(&data_to_ettercap, 0, sizeof(CONNECTION));
   memset(&sniff_data_to_ettercap, 0, sizeof(SNIFFED_DATA));

   dest.s_addr=ServerIP;

   if (Connection_Mode)
   {
      strcpy(data_to_ettercap.source_ip, inet_ntoa(client_sin.sin_addr) );
      strcpy(data_to_ettercap.dest_ip, inet_ntoa(dest) );
      data_to_ettercap.source_port = ntohs(client_sin.sin_port);
      data_to_ettercap.dest_port = ntohs(ServerPort);
      data_to_ettercap.source_seq = 0;
      data_to_ettercap.dest_seq = 0;
      data_to_ettercap.flags = 0;
      data_to_ettercap.proto = 'T';
      data_to_ettercap.datalen = len;

      Grell_Dissector(buf, &data_to_ettercap);
      Buffer_Put(pipe_with_illithid, &data_to_ettercap, sizeof(CONNECTION));
   }
   else
   {
      strcpy(sniff_data_to_ettercap.source_ip, inet_ntoa(client_sin.sin_addr) );
      strcpy(sniff_data_to_ettercap.dest_ip, inet_ntoa(dest) );
      sniff_data_to_ettercap.source_port = ntohs(client_sin.sin_port);
      sniff_data_to_ettercap.dest_port = ntohs(ServerPort);
      sniff_data_to_ettercap.seq = 0;
      sniff_data_to_ettercap.ack_seq = 0;
      sniff_data_to_ettercap.flags = 0;
      sniff_data_to_ettercap.proto = 'T';
      len = (len > MAX_DATA) ? MAX_DATA : len;
      sniff_data_to_ettercap.datasize = len;
      memset(&sniff_data_to_ettercap.data, 0, sizeof(sniff_data_to_ettercap.data));
      memcpy(&sniff_data_to_ettercap.data, buf, len);

      Buffer_Put(pipe_with_illithid_data, &sniff_data_to_ettercap, sizeof(SNIFFED_DATA));
   }
}

static void server_parse(char *buf, int len)
{
   CONNECTION data_to_ettercap;
   SNIFFED_DATA sniff_data_to_ettercap;
   struct in_addr source;

   memset(&data_to_ettercap, 0, sizeof(CONNECTION));
   memset(&sniff_data_to_ettercap, 0, sizeof(SNIFFED_DATA));

   source.s_addr=ServerIP;
   if (Connection_Mode)
   {
      strcpy(data_to_ettercap.source_ip, inet_ntoa(source) );
      strcpy(data_to_ettercap.dest_ip, inet_ntoa(client_sin.sin_addr) );
      data_to_ettercap.source_port = ntohs(ServerPort);
      data_to_ettercap.dest_port = ntohs(client_sin.sin_port);
      data_to_ettercap.source_seq = 0;
      data_to_ettercap.dest_seq = 0;
      data_to_ettercap.flags = 0;
      data_to_ettercap.proto = 'T';
      data_to_ettercap.datalen = len;

      Buffer_Put(pipe_with_illithid, &data_to_ettercap, sizeof(CONNECTION));
   }
   else
   {
      strcpy(sniff_data_to_ettercap.source_ip, inet_ntoa(source) );
      strcpy(sniff_data_to_ettercap.dest_ip, inet_ntoa(client_sin.sin_addr) );
      sniff_data_to_ettercap.source_port = ntohs(ServerPort);
      sniff_data_to_ettercap.dest_port = ntohs(client_sin.sin_port);
      sniff_data_to_ettercap.seq = 0;
      sniff_data_to_ettercap.ack_seq = 0;
      sniff_data_to_ettercap.flags = 0;
      sniff_data_to_ettercap.proto = 'T';
      len = (len > MAX_DATA) ? MAX_DATA : len;
      sniff_data_to_ettercap.datasize = len;
      memset(&sniff_data_to_ettercap.data, 0, sizeof(sniff_data_to_ettercap.data));
      memcpy(&sniff_data_to_ettercap.data, buf, len);

      Buffer_Put(pipe_with_illithid_data, &sniff_data_to_ettercap, sizeof(SNIFFED_DATA));
   }
}


static int client_read(char *buf, int size)
{
   if (ssl_type) return (SSL_read(ssl_client, buf, size));
   return (read(client_fd, buf, size));
}

static int client_write(char *buf, int size)
{
   if (ssl_type) return (SSL_write(ssl_client, buf, size));
   return (write(client_fd, buf, size));
}

static void client_init(void)
{
   fcntl(client_fd, F_SETFL, 0);

   if (ssl_type)
   {
      ssl_client = SSL_new(ssl_ctx_client);
      SSL_set_fd(ssl_client, client_fd);
      SSL_accept(ssl_client);
   }
}

static void client_close(void)
{
   if (ssl_type) SSL_free(ssl_client);
   close(client_fd);
}

static int server_read(char *buf, int size)
{
   if (ssl_type) return (SSL_read(ssl_server, buf, size));
   return (read(server_fd, buf, size));
}

static int server_write(char *buf, int size)
{
   if (ssl_type) return (SSL_write(ssl_server, buf, size));
   return (write(server_fd, buf, size));
}

static int server_init(char *buf, int size)
{
   char vhost[500], *i=0;
   int offset=0, type_connect=0, addr, temp, j;
   struct hostent *toresolv=(struct hostent *)1;

   do
   {
      temp = client_read(buf + offset, size - offset);
      offset+=temp;

      if (Grell_ProxyIP && !strncasecmp(buf,"CONNECT",7) && !ssl_type)
      {
         type_connect=1;
         break;
      }

       if (temp==-1) offset=size+1;
   } while(size>=offset && (i=(char *)memmem(buf, size, "\r\nHost: ",8))==NULL);

   if (offset>=size) // No virtual host
   {
       client_close();
       exit(0);
   }

   memset(&server_sin, 0, sizeof(server_sin));
   server_sin.sin_family = AF_INET;

   if (!Grell_ProxyIP || ssl_type)
   {
      memcpy(vhost, i+8, 500);
      strtok(vhost, "\r");
      strtok(vhost, ":");
      server_sin.sin_port = ssl_type ? htons(443) : htons(80);

      addr=inet_addr(vhost);
      if (addr==INADDR_NONE)
      {
         toresolv=gethostbyname(vhost);
         if(toresolv) addr=*(unsigned long *)toresolv->h_addr;
      }

      if (!toresolv || addr==ntohl(INADDR_LOOPBACK))
      {
         client_close();
         exit(0);
      }
   }
   else
   {
       addr=Grell_ProxyIP;
       server_sin.sin_port=htons((short)Grell_ProxyPort);
   }

   server_sin.sin_addr.s_addr = addr;
   ServerIP = addr;
   ServerPort = server_sin.sin_port;

   server_fd = socket(AF_INET, SOCK_STREAM, 0);
   connect(server_fd, (struct sockaddr *)&server_sin, sizeof(server_sin));

   if (type_connect)
   {
      server_write(buf,offset);  // Skip readable messages
      j=server_read(buf,MAX_DATA);
      buf[j]=0; // Only for paranoid tests
      client_write(buf,j);
      ssl_type=1;       // Turn on encryption
      client_init();
      offset=client_read(buf,MAX_DATA);
      if (offset == -1) exit(0);    // if it can't handle proxy auth...
      buf[offset]=0; // Only for paranoid tests
   }

   if (ssl_type)
   {
      ssl_ctx_server = SSL_CTX_new(SSLv23_client_method());
      ssl_server = SSL_new(ssl_ctx_server);
      SSL_set_connect_state(ssl_server);
      SSL_set_fd(ssl_server, server_fd);
      SSL_connect(ssl_server);
   }

   return(offset);
}

static void server_close(void)
{
   if (ssl_type) SSL_free(ssl_server);
   close(server_fd);
}

void Grell_init(void)
{
   struct sockaddr_in sa_in;
   int i=1;
   char path_file[100];

   proxy_fd = socket(AF_INET, SOCK_STREAM, 0);
   https_fd = socket(AF_INET, SOCK_STREAM, 0);

   setsockopt(proxy_fd, SOL_SOCKET, SO_REUSEADDR, &i, sizeof(i));
   setsockopt(https_fd, SOL_SOCKET, SO_REUSEADDR, &i, sizeof(i));

   memset(&sa_in, 0, sizeof(sa_in));
   sa_in.sin_family = AF_INET;
   sa_in.sin_addr.s_addr = INADDR_ANY;

   sa_in.sin_port = htons(Proxy_Local_Port);

   if (bind(proxy_fd, (struct sockaddr *)&sa_in, sizeof(sa_in)) < 0)
       Error_msg("ec_grell: can't bind port %d \n",Proxy_Local_Port);

   sa_in.sin_port = htons(HTTPS_Local_Port);

   if (bind(https_fd, (struct sockaddr *)&sa_in, sizeof(sa_in)) < 0)
       Error_msg("ec_grell: can't bind port %d \n",HTTPS_Local_Port);

#ifdef DEBUG
   Debug_msg("Grell_init -- Listening for Proxy redirect on port %d", Proxy_Local_Port);
   Debug_msg("Grell_init -- Listening for HTTPS redirect on port %d", HTTPS_Local_Port);
#endif

   listen(proxy_fd, 50);
   listen(https_fd, 50);

   SSL_library_init();

   ssl_ctx_client = SSL_CTX_new(SSLv23_server_method());

   if (SSL_CTX_use_certificate_file(ssl_ctx_client, CERT_FILE, SSL_FILETYPE_PEM) == 0)
   {
      sprintf(path_file, "%s%s", INSTALL_PATH, CERT_FILE);
      #ifdef DEBUG
         Debug_msg("Grell_init -- SSL_CTX_use_certificate_file -- %s", path_file);
      #endif
      if (SSL_CTX_use_certificate_file(ssl_ctx_client, path_file, SSL_FILETYPE_PEM) == 0)
         Error_msg("Can't open \"%s\" file !!", CERT_FILE);
   }

   if (SSL_CTX_use_PrivateKey_file(ssl_ctx_client, CERT_FILE, SSL_FILETYPE_PEM) == 0)
   {
      sprintf(path_file, "%s%s", INSTALL_PATH, CERT_FILE);
      #ifdef DEBUG
         Debug_msg("Grell_init -- SSL_CTX_use_PrivateKey_file -- %s", path_file);
      #endif
      if (SSL_CTX_use_PrivateKey_file(ssl_ctx_client, path_file, SSL_FILETYPE_PEM) == 0)
         Error_msg("Can't open \"%s\" file !!", CERT_FILE);
   }

   if (SSL_CTX_check_private_key(ssl_ctx_client) == 0)
      Error_msg("Bad SSL Key couple !!");

}

void Grell_spawn(void)
{
   u_char buf[MAX_DATA];
   fd_set fds;
   int i;

   client_init();
   i=server_init(buf, sizeof(buf));
   server_write(buf, i);
   client_parse(buf, i);

   loop
   {
      FD_ZERO(&fds);
      FD_SET(client_fd, &fds);
      FD_SET(server_fd, &fds);

      select(FOPEN_MAX, &fds, 0, 0, 0);

      if (FD_ISSET(client_fd, &fds))
      {
         i = sizeof(buf);
         if ((i = client_read(buf, i)) <= 0) break;
         if (server_write(buf, i) != i)   break;
         client_parse(buf, i);
      }
      else if (FD_ISSET(server_fd, &fds))
      {
         i = sizeof(buf);
         if ((i = server_read(buf, i)) <= 0) break;
         if (client_write(buf, i) != i) break;
         server_parse(buf, i);
      }
   }
   server_close();
   client_close();
}

// ssl_type = 0 Proxy
// ssl_type = 1 HTTPS

void Grell_start(void)
{
   fd_set fds;
   int Act_CM=1, dummy;

   strcpy(program_argv0, "grell   ");

   Act_CM = Connection_Mode;

   Grell_init();

   fcntl(proxy_fd, F_SETFL, O_NONBLOCK);
   fcntl(https_fd, F_SETFL, O_NONBLOCK);

   loop
   {
      FD_ZERO(&fds);

      FD_SET(proxy_fd, &fds);
      FD_SET(https_fd, &fds);

      if (getppid() == 1) exit(0);

      select(FOPEN_MAX, &fds, 0, 0, (struct timeval *)0);

      if (Act_CM!=Connection_Mode) // Doesn't spawn on signal
      {
          Act_CM=Connection_Mode;
          continue;
      }

      dummy=sizeof(struct sockaddr);

      if (FD_ISSET(proxy_fd, &fds))
      {
         u_long peer;

         client_fd = accept(proxy_fd, (struct sockaddr *)&client_sin, &dummy);
         ssl_type = 0;

         memcpy(&peer, &client_sin.sin_addr, sizeof(u_long));
         peer &= htonl(0x00FFFFFF);
         peer |= inet_addr(Host_In_LAN[0].ip) & htonl(0xFF000000);
         memcpy(&client_sin.sin_addr, &peer, sizeof(u_long));
      }
      else if (FD_ISSET(https_fd, &fds))
      {
         u_long peer;

         client_fd = accept(https_fd, (struct sockaddr *)&client_sin, &dummy);
         ssl_type = 1;

         memcpy(&peer, &client_sin.sin_addr, sizeof(u_long));
         peer &= htonl(0x00FFFFFF);
         peer |= inet_addr(Host_In_LAN[0].ip) & htonl(0xFF000000);
         memcpy(&client_sin.sin_addr, &peer, sizeof(u_long));
      }
      else
      {
         usleep(1000);
         continue;
      }

      if (fork() == 0)
      {
         close(proxy_fd);
         close(https_fd);
         Grell_spawn();
         exit(0);
      }
      close(client_fd);
   }
}

int Grell_Run(void)
{
   int Grell_pid;
   extern DISSECTOR Available_Dissectors[];
   DISSECTOR *ds;

   for( ds = Available_Dissectors; ds->port != 0; ds++)
   {
      if (!strcasecmp("HTTPS", ds->name))
      {
         if (ds->active == 0)
         {
            #ifdef DEBUG
               Debug_msg("Grell was disabled by conf.file");
            #endif
            return 0;
         }
      }
   }

#ifdef DEBUG
   Debug_msg("Grell Starts");
#endif

   if (!(Grell_pid = fork()))
      Grell_start();
   else
      return Grell_pid;

   return 0;
}

#endif //HAVE_OPENSSL

/* EOF */
