/*
    ettercap -- module for logging into different files

    Copyright (C) 2001  ALoR <alor@users.sourceforge.net>, NaGA <crwm@freemail.it>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <time.h>

#include "include/ec_main.h"
#include "include/ec_error.h"
#include "include/ec_inet_structures.h"

#ifdef DEBUG
   #include "include/ec_debug.h"
#endif

typedef struct {
   char source_ip[16];
   char dest_ip[16];
   int source_port;
   int dest_port;
   char proto;
   FILE *flog;
   struct LOGGED_DATA *next;
} LOGGED_DATA;

LOGGED_DATA *head;

// protos

void LogToFile(SNIFFED_DATA *data);
FILE * LogToFile_open_collect();
void LogToFile_freelist(LOGGED_DATA *head);
void LogToFile_close();
void LogToFile_FilteredData(u_char * buf_ip);
char *LogToFile_DumpPass(void);

// -------------------------------

FILE * LogToFile_open_collect()
{
   FILE *fto;
   char filename[50];
   time_t tt = time(NULL);
   struct tm *dd = localtime(&tt);

   sprintf(filename, "%04d%02d%02d-collected-pass.log", dd->tm_year+1900, dd->tm_mon+1, dd->tm_mday);
   fto = fopen(filename, "a");

   return fto;
}


void LogToFile(SNIFFED_DATA *data)
{
   char filter[45];
   char cur_conn[45];
   char found = 0;
   LOGGED_DATA *index = head;
   LOGGED_DATA *ptr = index;
   LOGGED_DATA *current;
   time_t tt = time(NULL);
   struct tm *dd = localtime(&tt);


   sprintf(filter, "%c%s%d%s%d", data->proto, data->source_ip, data->source_port, data->dest_ip, data->dest_port);

   if (head != NULL)
   {
      for(ptr = head; ptr != NULL; ptr = (LOGGED_DATA *)ptr->next)
      {
         index = ptr;
         sprintf(cur_conn, "%c%s%d%s%d", index->proto, index->source_ip, index->source_port, index->dest_ip, index->dest_port);

         if (!strcmp(filter, cur_conn))
         {
            found = 1;

            write(fileno(index->flog), data->data, data->datasize);

            if ( (data->flags & TH_RST) || (data->flags & TH_FIN) )
               fprintf(index->flog, "\n\n| CONNECTION CLOSED ON %02d:%02d:%02d |\n\n",  dd->tm_hour, dd->tm_min, dd->tm_sec);

            fflush(index->flog);
            return;
         }
      }
   }

   if (!found)
   {
      char logname[100];

      #ifdef DEBUG
         Debug_msg("LogToFile - new log ! %c %s:%d - %s:%d ",  data->proto, data->source_ip, data->source_port, data->dest_ip, data->dest_port);
      #endif

      current = (LOGGED_DATA *)calloc(1, sizeof(LOGGED_DATA));
      if (current == NULL)
         Error_msg("ec_logtofile:%d calloc() | ERRNO : %d | %s", __LINE__, errno, strerror(errno));

      memcpy(current->source_ip, &data->source_ip, sizeof(data->source_ip));
      memcpy(current->dest_ip, &data->dest_ip, sizeof(data->dest_ip));

      current->source_port = data->source_port;
      current->dest_port = data->dest_port;
      current->proto = data->proto;

      sprintf(logname, "%04d%02d%02d-%c-%s:%d-%s:%d.log", dd->tm_year+1900, dd->tm_mon+1, dd->tm_mday,
                        data->proto,
                        data->source_ip,
                        data->source_port,
                        data->dest_ip,
                        data->dest_port
      );

      current->flog = fopen(logname, "a");
      if (current->flog == NULL)
         Error_msg("ec_logtofile:%d fopen(%s) | ERRNO : %d | %s", logname, __LINE__, errno, strerror(errno));

      write(fileno(current->flog), data->data, data->datasize);
      fflush(current->flog);

      if (head == NULL) head = current;
      if (index != NULL) index->next = (struct LOGGED_DATA *)current;
   }
}


void LogToFile_close()
{

#ifdef DEBUG
   Debug_msg("LogToFile_close");
#endif

   LogToFile_freelist(head);
   head = NULL;

}

void LogToFile_freelist(LOGGED_DATA *cur)
{
   if (!cur)   return;

   LogToFile_freelist((LOGGED_DATA *) cur->next);
   fclose(cur->flog);
   free(cur);

}


void LogToFile_FilteredData(u_char * buf_ip)
{
   IP_header *ip;
   UDP_header *udp;
   TCP_header *tcp;
   u_char *data = NULL;
   short datalen = 0;
   FILE *fto;
   char logname[100];
   time_t tt = time(NULL);
   struct tm *dd = localtime(&tt);

#ifdef DEBUG
   Debug_msg("LogToFile_FilteredData");
#endif

   ip = (IP_header *) buf_ip;

   if (ip->proto == IPPROTO_UDP)
   {
      udp = (UDP_header *) ((int)ip + ip->h_len * 4);
      data = (char *)((int)udp + UDP_HEADER);
      datalen = ntohs(udp->len) - UDP_HEADER;
      sprintf(logname, "%04d%02d%02d-Filtered-U-%s:%d-", dd->tm_year+1900, dd->tm_mon+1, dd->tm_mday,
                        int_ntoa(ip->source_ip),
                        ntohs(udp->source));
      sprintf(logname, "%s%s:%d.log", logname, int_ntoa(ip->dest_ip), ntohs(udp->dest) ); // damned static in inet_ntoa...
   }
   else if (ip->proto == IPPROTO_TCP)
   {
      tcp = (TCP_header *) ((int)ip + ip->h_len * 4);
      data = (char *)((int)tcp + tcp->doff * 4);
      datalen = (int)ip + ntohs(ip->t_len) - (int)data;
      sprintf(logname, "%04d%02d%02d-Filtered-T-%s:%d-", dd->tm_year+1900, dd->tm_mon+1, dd->tm_mday,
                        int_ntoa(ip->source_ip),
                        ntohs(tcp->source));
      sprintf(logname, "%s%s:%d.log", logname, int_ntoa(ip->dest_ip), ntohs(tcp->dest) ); // damned static in inet_ntoa...
   }

   if (data && datalen)
   {
      fto = fopen(logname, "a");
      if (fto == NULL)
         Error_msg("ec_logtofile:%d fopen(%s) | ERRNO : %d | %s", logname, __LINE__, errno, strerror(errno));
      write(fileno(fto), data, datalen);
      fflush(fto);
      fclose(fto);
   }

}


char *LogToFile_DumpPass(void)
{
   static char logname[100];
   FILE *fto;
   int i;
   time_t tt = time(NULL);
   struct tm *dd = localtime(&tt);

#ifdef DEBUG
   Debug_msg("LogToFile_DumpPass");
#endif

   sprintf(logname, "%04d%02d%02d-%02d:%02d:%02d-DumpedPassword.log", dd->tm_year+1900, dd->tm_mon+1, dd->tm_mday, dd->tm_hour, dd->tm_min, dd->tm_sec);

   fto = fopen(logname, "a");
   if (fto == NULL)
         Error_msg("ec_logtofile:%d fopen(%s) | ERRNO : %d | %s", logname, __LINE__, errno, strerror(errno));

   for (i=0; i < number_of_connections; i++)
   {
      if (Conn_Between_Hosts[i].user[0] != 0 &&  Conn_Between_Hosts[i].pass[0] != 0)
      {
         fprintf(fto, "\n\n%s:%d -> %s:%d\t\t%s\n\n", Conn_Between_Hosts[i].source_ip,
                                                      Conn_Between_Hosts[i].source_port,
                                                      Conn_Between_Hosts[i].dest_ip,
                                                      Conn_Between_Hosts[i].dest_port,
                                                      Conn_Between_Hosts[i].type
                                                      );
         fprintf(fto, "%s\n", Conn_Between_Hosts[i].user);
         fprintf(fto, "%s\n", Conn_Between_Hosts[i].pass);
         if (strlen(Conn_Between_Hosts[i].info))
            fprintf(fto, "\n%s\n", Conn_Between_Hosts[i].info);
         fflush(fto);
      }
   }

   fclose(fto);
   return logname;
}


/* EOF */
