/*
    ettercap -- ncurses interface for data sniffing-logging

    Copyright (C) 2001  ALoR <alor@users.sourceforge.net>, NaGA <crwm@freemail.it>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include "include/ec_main.h"

#ifdef HAVE_NCURSES  // don't compile if ncurses interface is not supported

#include <string.h>
#include <ncurses.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <signal.h>
#include <time.h>

#include "include/ec_interface.h"
#include "include/ec_interface_inject.h"
#include "include/ec_decodedata.h"
#include "include/ec_dissector.h"
#include "include/ec_doppleganger.h"
#include "include/ec_illithid.h"
#include "include/ec_buffer.h"
#include "include/ec_error.h"

#ifdef DEBUG
   #include "include/ec_debug.h"
#endif

#define PAD_BUFFER 500        // buffered lines for sniffing


#define BOTTOM_COLOR 1        // color schemes
#define TITLE_COLOR  2
#define MAIN_COLOR   3
#define POINT_COLOR  4
#define SEL_COLOR    5
#define HELP_COLOR   6
#define SNIFF_COLOR  7

#define KEY_TAB      '\t'     // they aren't defined in ncurses.h :(
#define KEY_RETURN   10
#define KEY_CTRL_L   12

#define ASCII_VIEW   0        // data in the sniffing windows...
#define HEX_VIEW     1


// protos...

void Interface_Sniff_Data_Run(char *ipsource, int portsource, char *ipdest, int portdest, char *macsource, char *macdest, char proto, char *app, short mode);
void Interface_Sniff_Data_Redraw(void);
void Interface_Sniff_Data_KeyTab(char *ipsource, int portsource, char *ipdest, int portdest);
void Interface_Sniff_Data_SniffData(char *source, char *dest, int portsource, int portdest, char proto);
void Interface_Sniff_Data_LogToFile(char proto, char *ipsource, int portsource, char *ipdest, int portdest);
void Interface_Sniff_Data_Winch(void);
void Interface_Sniff_Data_StopCont(void);
void Interface_Sniff_Data_View(short mode);
void Interface_Sniff_Data_Scroll(short direction);
void Interface_Sniff_Data_DrawScroller(void);
void Interface_Sniff_Data_Inject(char proto, char *app, char *ipsource, char *ipdest);
void Interface_Sniff_Data_Filter(void);
void Interface_Sniff_Data_ActiveFilter(short mode);

// global variables

extern WINDOW *main_window, *bottom_window, *top_window;
WINDOW *data_source_win, *data_dest_win, *data_source, *data_dest;
WINDOW *win_pointer;

extern int W_MAINX1, W_MAINY1, W_MAINX2, W_MAINY2;
extern int W_BOTTOMY2;

extern int Conn_Pointer;

short log_source, log_dest;
short view;
short stop = 0;
short inject = -1;

int inject_tod=0, inject_tos=0;

short scroll_yd = 0, scroll_ys = 0;

FILE *file_source, *file_dest;


//---------------------------

void Interface_Sniff_Data_Winch(void)  // TODO better handling...
{
#ifdef DEBUG
   Debug_msg("Interface_Sniff_Data_Winch\tTODO");
#endif

   Interface_Sniff_Data_Redraw();
}


void Interface_Sniff_Data_Redraw(void)
{

   Interface_Redraw();

#ifdef DEBUG
   Debug_msg("Interface_Sniff_Data_Redraw");
#endif

   redrawwin(data_dest_win);
   redrawwin(data_source_win);

   wnoutrefresh(data_dest_win);
   wnoutrefresh(data_source_win);
   pnoutrefresh(data_source, scroll_ys,0, W_MAINY1 + 3, 3, W_MAINY2 - 3 , W_MAINX2 / 2 - 2);
   pnoutrefresh(data_dest, scroll_yd,0, W_MAINY1 + 3, W_MAINX2 / 2 + 2, W_MAINY2 - 3 , W_MAINX2 - 2);

   doupdate();
}



void Interface_Sniff_Data_KeyTab(char *ipsource, int portsource, char *ipdest, int portdest)
{


   win_pointer = (win_pointer == data_source_win) ? data_dest_win : data_source_win;

   wattroff(data_source_win, A_BOLD);
   wattroff(data_dest_win, A_BOLD);

   wattron(win_pointer, A_BOLD);

   box(data_source_win,ACS_VLINE,ACS_HLINE);
   box(data_dest_win,ACS_VLINE,ACS_HLINE);

   mvwprintw(data_source_win, 0, 1, "%s:%d", ipsource, portsource);
   mvwprintw(data_dest_win, 0, 1, "%s:%d", ipdest, portdest);

   Interface_Sniff_Data_DrawScroller();

   if (log_source) mvwprintw(data_source_win, W_MAINY2 - W_MAINY1 - 4, W_MAINX2 / 2 - 10, "LOGGED");
   if (log_dest) mvwprintw(data_dest_win, W_MAINY2 - W_MAINY1 - 4, W_MAINX2 / 2 - 10, "LOGGED");

   switch(view)
   {
      case ASCII_VIEW:
                        mvwprintw(data_source_win, W_MAINY2 - W_MAINY1 - 4, 2, "ASCII");
                        mvwprintw(data_dest_win, W_MAINY2 - W_MAINY1 - 4, 2, "ASCII");
                        break;
      case HEX_VIEW:
                        mvwprintw(data_source_win, W_MAINY2 - W_MAINY1 - 4, 2, "HEX");
                        mvwprintw(data_dest_win, W_MAINY2 - W_MAINY1 - 4, 2, "HEX");
                        break;
   }

   if (stop) mvwprintw(data_source_win, W_MAINY2 - W_MAINY1 - 4, 2, "STOPPED");
   if (stop) mvwprintw(data_dest_win, W_MAINY2 - W_MAINY1 - 4, 2, "STOPPED");

   mvwprintw(win_pointer, 0, W_MAINX2 / 2 - 10, "active");

   redrawwin(data_dest_win);
   redrawwin(data_source_win);

   wnoutrefresh(data_dest_win);
   wnoutrefresh(data_source_win);
   pnoutrefresh(data_source, scroll_ys,0, W_MAINY1 + 3, 3, W_MAINY2 - 3 , W_MAINX2 / 2 - 2);
   pnoutrefresh(data_dest, scroll_yd,0, W_MAINY1 + 3, W_MAINX2 / 2 + 2, W_MAINY2 - 3 , W_MAINX2 - 2);

   doupdate();

}




void Interface_Sniff_Data_LogToFile(char proto, char *ipsource, int portsource, char *ipdest, int portdest)
{
   WINDOW *question_window;
   char question[100];
   char filename[50];
   char answer;
   time_t tt;
   struct tm *dd;
   char date[8];

#ifdef DEBUG
   Debug_msg("Interface_Sniff_Data_LogToFile");
#endif

   tt = time(NULL);
   dd = localtime(&tt);

   sprintf(date, "%04d%02d%02d", dd->tm_year+1900, dd->tm_mon+1, dd->tm_mday);

   if (win_pointer == data_source_win)
   {
      strcpy(question, "Do U want to log SOURCE-to-DEST to this file ? (y/n)");
      sprintf(filename, "%s-%c-%s:%d-%s:%d.log", date, proto, ipsource, portsource, ipdest, portdest);
   }
   else
   {
      strcpy(question, "Do U want to log DEST-to-SOURCE to this file ? (y/n)");
      sprintf(filename, "%s-%c-%s:%d-%s:%d.log", date, proto, ipdest, portdest, ipsource, portsource);
   }

   question_window = newwin(7, strlen(question) + 4,0,0);
   mvwin(question_window, W_MAINY1+(W_MAINY2-W_MAINY1)/2-2, W_MAINX2/2 - (strlen(question)+4)/2 );
   wbkgdset(question_window, COLOR_PAIR(TITLE_COLOR));
   wattron(question_window, A_BOLD);
   box(question_window,ACS_VLINE,ACS_HLINE);
   mvwprintw(question_window,  2, 2, question);
   wnoutrefresh(question_window);
   doupdate();

   wbkgdset(question_window, COLOR_PAIR(HELP_COLOR));
   mvwprintw(question_window,  4, 4, filename);
   answer = wgetch(question_window);
   delwin(question_window);

   if ( (answer == 'y') || (answer == 'Y') )
   {
      if (win_pointer == data_source_win)
      {
         file_source = fopen(filename, "a");
         if (file_source == NULL)
            Error_msg("ec_interface_sniff_data:%d fopen(%s) | ERRNO : %d | %s", filename, __LINE__, errno, strerror(errno));
         log_source = 1;
         mvwprintw(data_source_win, W_MAINY2 - W_MAINY1 - 4, W_MAINX2 / 2 - 10, "LOGGED");
      }
      else
      {
         file_dest = fopen(filename, "a");
         if (file_dest == NULL)
            Error_msg("ec_interface_sniff_data:%d fopen(%s) | ERRNO : %d | %s", filename, __LINE__, errno, strerror(errno));
         log_dest = 1;
         mvwprintw(data_dest_win, W_MAINY2 - W_MAINY1 - 4, W_MAINX2 / 2 - 10, "LOGGED");
      }
   }
   else
   {
      if (win_pointer == data_source_win)
      {
         if (log_source) fclose(file_source);
         log_source = 0;
         wmove(data_source_win, W_MAINY2 - W_MAINY1 - 4, W_MAINX2 / 2 - 10);
         whline(data_source_win, ACS_HLINE, 6);
      }
      else
      {
         if (log_dest) fclose(file_dest);
         log_dest = 0;
         wmove(data_dest_win, W_MAINY2 - W_MAINY1 - 4, W_MAINX2 / 2 - 10);
         whline(data_dest_win, ACS_HLINE, 6);
      }
   }


#ifdef DEBUG
   Debug_msg("Interface_Sniff_Data_LogToFile returns -- %c", answer);
   if ( (answer == 'y') || (answer == 'Y') ) Debug_msg("\t %s", filename);
#endif

   Interface_Sniff_Data_Redraw();

}


void Interface_Sniff_Data_StopCont(void)
{
#ifdef DEBUG
   Debug_msg("Interface_Sniff_Data_StopCont");
#endif


   if (stop == 0)
   {
      mvwprintw(data_source_win, W_MAINY2 - W_MAINY1 - 4, 2, "STOPPED");
      mvwprintw(data_dest_win, W_MAINY2 - W_MAINY1 - 4, 2, "STOPPED");
      stop = 1;
   }
   else
   {
      wmove(data_source_win, W_MAINY2 - W_MAINY1 - 4, 2);   whline(data_source_win, ACS_HLINE, 7);
      wmove(data_dest_win, W_MAINY2 - W_MAINY1 - 4, 2); whline(data_dest_win, ACS_HLINE, 7);
      Interface_Sniff_Data_View(view);
      stop = 0;
   }

   redrawwin(data_dest_win);
   redrawwin(data_source_win);

   wnoutrefresh(data_dest_win);
   wnoutrefresh(data_source_win);
   pnoutrefresh(data_source, scroll_ys,0, W_MAINY1 + 3, 3, W_MAINY2 - 3 , W_MAINX2 / 2 - 2);
   pnoutrefresh(data_dest, scroll_yd,0, W_MAINY1 + 3, W_MAINX2 / 2 + 2, W_MAINY2 - 3 , W_MAINX2 - 2);

   doupdate();

}



void Interface_Sniff_Data_View(short mode)
{
#ifdef DEBUG
   Debug_msg("Interface_Sniff_Data_View -- %d", mode);
#endif


   switch(mode)
   {
      case ASCII_VIEW:
                        wmove(data_source_win, W_MAINY2 - W_MAINY1 - 4, 2);   whline(data_source_win, ACS_HLINE, 7);
                        wmove(data_dest_win, W_MAINY2 - W_MAINY1 - 4, 2); whline(data_dest_win, ACS_HLINE, 7);
                        mvwprintw(data_source_win, W_MAINY2 - W_MAINY1 - 4, 2, "ASCII");
                        mvwprintw(data_dest_win, W_MAINY2 - W_MAINY1 - 4, 2, "ASCII");
                        break;
      case HEX_VIEW:
                        wmove(data_source_win, W_MAINY2 - W_MAINY1 - 4, 2);   whline(data_source_win, ACS_HLINE, 7);
                        wmove(data_dest_win, W_MAINY2 - W_MAINY1 - 4, 2); whline(data_dest_win, ACS_HLINE, 7);
                        mvwprintw(data_source_win, W_MAINY2 - W_MAINY1 - 4, 2, "HEX");
                        mvwprintw(data_dest_win, W_MAINY2 - W_MAINY1 - 4, 2, "HEX");
                        break;
   }

   view = mode;

   redrawwin(data_dest_win);
   redrawwin(data_source_win);

   wnoutrefresh(data_dest_win);
   wnoutrefresh(data_source_win);
   pnoutrefresh(data_source, scroll_ys,0, W_MAINY1 + 3, 3, W_MAINY2 - 3 , W_MAINX2 / 2 - 2);
   pnoutrefresh(data_dest, scroll_yd,0, W_MAINY1 + 3, W_MAINX2 / 2 + 2, W_MAINY2 - 3 , W_MAINX2 - 2);

   doupdate();

}



void Interface_Sniff_Data_SniffData(char *source, char *dest, int portsource, int portdest, char proto)
{
   SNIFFED_DATA data_from_illithid;
   int datalen;

   memset(&data_from_illithid, 0, sizeof(SNIFFED_DATA));

   datalen = Buffer_Get(pipe_with_illithid_data, &data_from_illithid, sizeof(SNIFFED_DATA));

   if (datalen<=0)
   {
       usleep(1000);
       return;
   }

   // Not from our stream
   if (!Illithid_ToBeSniffed(source, portsource, dest, portdest, &data_from_illithid) || proto!=data_from_illithid.proto)
       return;

   if (!strcmp(source, data_from_illithid.source_ip))
   {
      if (log_source) { write(fileno(file_source), &data_from_illithid.data, data_from_illithid.datasize); fflush(file_source);}
      if (!stop)
         switch (view)
         {
            case ASCII_VIEW:
                           wprintw(data_source, "%s", Decodedata_GetAsciiData(data_from_illithid.data, data_from_illithid.datasize));
                           pnoutrefresh(data_source, scroll_ys,0, W_MAINY1 + 3, 3, W_MAINY2 - 3 , W_MAINX2 / 2 - 2);
                           break;
            case HEX_VIEW:
                           if (data_from_illithid.proto == 'T')
                              wprintw(data_source, "\n> S %lx A %lx (%s) <\n%s",
                                                data_from_illithid.seq,
                                                data_from_illithid.ack_seq,
                                                Decodedata_TCPFlags(data_from_illithid.flags),
                                                Decodedata_GetHexData(data_from_illithid.data, data_from_illithid.datasize, W_MAINX2/2 - 4));
                           else
                              wprintw(data_source, "%s", Decodedata_GetHexData(data_from_illithid.data, data_from_illithid.datasize, W_MAINX2/2 - 4));

                           pnoutrefresh(data_source, scroll_ys,0, W_MAINY1 + 3, 3, W_MAINY2 - 3 , W_MAINX2 / 2 - 2);
                           break;
         }
   }
   else
   {
      if (log_dest) { write(fileno(file_dest), &data_from_illithid.data, data_from_illithid.datasize); fflush(file_dest);}
      if (!stop)
         switch (view)
         {
            case ASCII_VIEW:
                           wprintw(data_dest, "%s", Decodedata_GetAsciiData(data_from_illithid.data, data_from_illithid.datasize));
                           pnoutrefresh(data_dest, scroll_yd,0, W_MAINY1 + 3, W_MAINX2 / 2 + 2, W_MAINY2 - 3 , W_MAINX2 - 2);
                           break;
            case HEX_VIEW:
                           if (data_from_illithid.proto == 'T')
                              wprintw(data_dest, "\n> S %lx A %lx (%s) <\n%s",
                                                data_from_illithid.seq,
                                                data_from_illithid.ack_seq,
                                                Decodedata_TCPFlags(data_from_illithid.flags),
                                                Decodedata_GetHexData(data_from_illithid.data, data_from_illithid.datasize, W_MAINX2/2 - 4));
                           else
                              wprintw(data_dest, "%s", Decodedata_GetHexData(data_from_illithid.data, data_from_illithid.datasize, W_MAINX2/2 - 4));

                           pnoutrefresh(data_dest, scroll_yd,0, W_MAINY1 + 3, W_MAINX2 / 2 + 2, W_MAINY2 - 3 , W_MAINX2 - 2);
                           break;
         }
   }
   if (inject == -1) inject = 0; // ok we have taken at least one packet
   doupdate();
}


void Interface_Sniff_Data_DrawScroller(void)
{
   short sheight = (W_MAINY2-10)*(W_MAINY2-10)/PAD_BUFFER;
   short svpos = (W_MAINY2-6)*scroll_ys/PAD_BUFFER;
   short dvpos = (W_MAINY2-6)*scroll_yd/PAD_BUFFER;

   sheight = (sheight < 1) ? 1 : sheight;

   svpos = (svpos == 0) ? 1 : svpos;
   svpos = (svpos > W_MAINY2-9-sheight) ? W_MAINY2-9-sheight : svpos;

   dvpos = (dvpos == 0) ? 1 : dvpos;
   dvpos = (dvpos > W_MAINY2-9-sheight) ? W_MAINY2-9-sheight : dvpos;

   wattron(win_pointer, A_BOLD);

   mvwvline(data_source_win, 1, W_MAINX2 / 2 - 3, ACS_VLINE, W_MAINY2-10);
   wattron(data_source_win, A_REVERSE);
   mvwvline(data_source_win, svpos, W_MAINX2 / 2 - 3, ' ', sheight);
   wnoutrefresh(data_source_win);
   wattroff(data_source_win, A_REVERSE);

   mvwvline(data_dest_win, 1, W_MAINX2 / 2 - 3, ACS_VLINE, W_MAINY2-10);
   wattron(data_dest_win, A_REVERSE);
   mvwvline(data_dest_win, dvpos, W_MAINX2 / 2 - 3, ' ', sheight);
   wnoutrefresh(data_dest_win);
   wattroff(data_dest_win, A_REVERSE);

}


void Interface_Sniff_Data_Scroll(short direction)
{

   if (win_pointer == data_source_win)
   {
      scroll_ys += direction;
      scroll_ys = (scroll_ys < 0) ? 0 : scroll_ys;
      scroll_ys = (scroll_ys > PAD_BUFFER - (W_MAINY2-10)) ? PAD_BUFFER - (W_MAINY2-10) : scroll_ys;
      pnoutrefresh(data_source, scroll_ys,0, W_MAINY1 + 3, 3, W_MAINY2 - 3 , W_MAINX2 / 2 - 2);
      Interface_Sniff_Data_DrawScroller();
   }
   else
   {
      scroll_yd += direction;
      scroll_yd = (scroll_yd < 0) ? 0 : scroll_yd;
      scroll_yd = (scroll_yd > PAD_BUFFER - (W_MAINY2-10)) ? PAD_BUFFER - (W_MAINY2-10) : scroll_yd;
      pnoutrefresh(data_dest, scroll_yd,0, W_MAINY1 + 3, W_MAINX2 / 2 + 2, W_MAINY2 - 3 , W_MAINX2 - 2);
      Interface_Sniff_Data_DrawScroller();
   }

   doupdate();
}



void Interface_Sniff_Data_Inject(char proto, char *app, char *ipsource, char *ipdest)
{

   char inject_data[MAX_INJECT];
   int data_len;
   char answer;

#ifdef DEBUG
   Debug_msg("Interface_Sniff_Data_Inject");
#endif

   if (inject == -1 && proto == 'T')
   {
      Interface_PopUp("Before injecting we need to sniff at least one packet...");
      Interface_Sniff_Data_Redraw();
      return;
   }

   if (inject == 0)
   {
      answer = Interface_PopUp("Do U really want to inject character in this connection (y/n)?");
      Interface_Sniff_Data_Redraw();
      if ((answer != 'y') && (answer != 'Y'))
         return;
   }

   inject = 1;  // on exit the connection must be RSTted

   memset(inject_data, 0, MAX_INJECT);

   data_len = Interface_Inject_Run(inject_data, proto, app);

   Interface_Sniff_Data_Redraw();

   if (win_pointer == data_dest_win)
   {
      write(pipe_inject_stod[1], inject_data, data_len);
      wmove(bottom_window, 1, 30);
      wbkgdset(bottom_window, COLOR_PAIR(BOTTOM_COLOR));
      wprintw(bottom_window, "%d ", inject_tos += data_len);
      wbkgdset(bottom_window, COLOR_PAIR(SNIFF_COLOR));
      wprintw(bottom_window, "chars injected to %s", ipsource);
   }
   else
   {
      write(pipe_inject_dtos[1], inject_data, data_len);
      wmove(bottom_window, 0, 30);
      wbkgdset(bottom_window, COLOR_PAIR(BOTTOM_COLOR));
      wprintw(bottom_window, "%d ", inject_tod += data_len);
      wbkgdset(bottom_window, COLOR_PAIR(SNIFF_COLOR));
      wprintw(bottom_window, "chars injected to %s", ipdest);
   }

   wnoutrefresh(bottom_window);

   doupdate();

}


void Interface_Sniff_Data_Filter(void)
{

   DROP_FILTER filter;
   DROP_FILTER null_filter;

#ifdef DEBUG
   Debug_msg("Interface_Sniff_Data_Filter %d", inject);
#endif

   memset(&filter, 0, sizeof(DROP_FILTER));
   memset(&null_filter, 0, sizeof(DROP_FILTER));

   if (Interface_Inject_Filter(&filter))
   {
      if (filter.slen == 0 && memcmp(&filter, &null_filter, sizeof(DROP_FILTER)) )
      {
         Interface_PopUp("Setting a 0 length search string is NOT a good idea... ;)");
         Interface_Sniff_Data_Redraw();
         return;
      }

      if (win_pointer == data_dest_win)
      {
         #ifdef DEBUG
            Debug_msg("Interface_Sniff_Data_Filter -- set filter on source [%s]", filter.search);
         #endif
         write(pipe_filter_stod[1], &filter, sizeof(DROP_FILTER));
         memcpy(&ARP_Filter_Source, &filter,  sizeof(DROP_FILTER));
      }
      else
      {
         #ifdef DEBUG
            Debug_msg("Interface_Sniff_Data_Filter -- set filter on dest [%s]", filter.search);
         #endif
         write(pipe_filter_dtos[1], &filter, sizeof(DROP_FILTER));
         memcpy(&ARP_Filter_Dest, &filter,  sizeof(DROP_FILTER));
      }

      if ( !memcmp(&ARP_Filter_Source, &null_filter, sizeof(DROP_FILTER)) && !memcmp(&ARP_Filter_Dest, &null_filter, sizeof(DROP_FILTER)) )
         Interface_Sniff_Data_ActiveFilter(0);
      else
      {
         Interface_Sniff_Data_ActiveFilter(1);
         if (filter.type != 'L') inject = 1;  // on exit the connection must be RSTted
      }
   }

   Interface_Sniff_Data_Redraw();

   wnoutrefresh(bottom_window);

   doupdate();

}


void Interface_Sniff_Data_ActiveFilter(short mode)
{

#ifdef DEBUG
   Debug_msg("Interface_Sniff_Data_ActiveFilter %d", mode);
#endif

   wbkgdset(top_window, COLOR_PAIR(HELP_COLOR));
   mvwprintw(top_window, 2, 55, "Active Filter: ");

   switch(mode)
   {
      case 1:  wbkgdset(top_window, COLOR_PAIR(BOTTOM_COLOR));
               mvwprintw(top_window, 2, 70, "ON ");
               break;

      case 0:  wbkgdset(top_window, COLOR_PAIR(BOTTOM_COLOR));
               mvwprintw(top_window, 2, 70, "OFF");
               break;
   }

   wnoutrefresh(top_window);
   doupdate();

}



void Interface_Sniff_Data_Run(char *ipsource, int portsource, char *ipdest, int portdest, char *macsource, char *macdest, char proto, char *app, short mode)
{
   int KeyPress;
   int dimY = W_MAINY2 - W_MAINY1 - 3;
   int dimX = W_MAINX2 / 2 - 2;
   fd_set msk_fd;
   struct timeval TimeOut;
   int Illithid_data_pid = 0, dopple_pid = 0;

   Buffer_Flush(pipe_with_illithid_data);

   pipe(pipe_inject_stod);
   pipe(pipe_inject_dtos);
   pipe(pipe_filter_stod);
   pipe(pipe_filter_dtos);

   if ( mode <= PUBLICARP )   // PUBLICARP or ARPBASED
   {
      if ( mode == PUBLICARP )
      {
         dopple_pid = Doppleganger_Run(Options.netiface, ipsource, ipdest, macsource, macdest);
         mode = ARPBASED;
      }

      Illithid_data_pid = Illithid_ARPBased_GetData(Options.netiface, proto, ipsource, portsource, ipdest, portdest, macsource, macdest);
   }

#ifdef DEBUG
   Debug_msg("Interface_Sniff_Data_Run -- %d -- %c [%s:%d] [%s:%d] [%s] [%s]", mode, proto, ipsource, portsource, ipdest, portdest, macsource, macdest );
#endif

   init_pair(SNIFF_COLOR, COLOR_CYAN, COLOR_BLACK);

   data_source_win = newwin(dimY, dimX, W_MAINY1 + 2, 2);
   data_dest_win = newwin(dimY, dimX, W_MAINY1 + 2, W_MAINX2 / 2 + 1);
   data_source = newpad(PAD_BUFFER, dimX - 2);
   data_dest = newpad(PAD_BUFFER, dimX - 2);

   win_pointer = data_source_win;

   wbkgdset(data_source_win, COLOR_PAIR(SNIFF_COLOR));
   wbkgdset(data_dest_win, COLOR_PAIR(SNIFF_COLOR));
   wattron(data_source_win, A_BOLD);

   box(data_source_win,ACS_VLINE,ACS_HLINE);
   box(data_dest_win,ACS_VLINE,ACS_HLINE);

   mvwprintw(data_source_win, 0, 1, "%s:%d", ipsource, portsource);
   mvwprintw(data_dest_win, 0, 1, "%s:%d", ipdest, portdest);

   mvwprintw(data_source_win, 0, W_MAINX2 / 2 - 10, "active");

   scrollok(data_source, TRUE);
   scrollok(data_dest, TRUE);

   scroll_yd = PAD_BUFFER - (W_MAINY2-10);
   scroll_ys = PAD_BUFFER - (W_MAINY2-10);

   wmove(data_source, scroll_ys, 0);
   wmove(data_dest, scroll_yd, 0);

   wbkgdset(main_window, COLOR_PAIR(MAIN_COLOR));

   Interface_Sniff_Data_ActiveFilter(0);
   Interface_Sniff_Data_DrawScroller();

   werase(main_window);
   werase(bottom_window);

   wmove(bottom_window, 0, 1);
   wbkgdset(bottom_window, COLOR_PAIR(SNIFF_COLOR));
   wprintw(bottom_window, "Protocol: ");
   wbkgdset(bottom_window, COLOR_PAIR(BOTTOM_COLOR));
   if (proto == 'T')
      wprintw(bottom_window, "TCP");
   else
      wprintw(bottom_window, "UDP");
   wmove(bottom_window, 1, 1);
   wbkgdset(bottom_window, COLOR_PAIR(SNIFF_COLOR));
   wprintw(bottom_window, "Application: ");
   wbkgdset(bottom_window, COLOR_PAIR(BOTTOM_COLOR));
   wprintw(bottom_window, app);

   redrawwin(main_window);
   wnoutrefresh(main_window);
   wnoutrefresh(bottom_window);

   wnoutrefresh(data_source_win);
   wnoutrefresh(data_dest_win);
   doupdate();

   if (Options.hexview)
   {
      view = HEX_VIEW;
      Options.hexview = 0;
   }

   Interface_Sniff_Data_View(view);    // default view


   memset(&TimeOut, 0, sizeof(TimeOut));  //  timeout = 0
   FD_ZERO(&msk_fd);

   loop
   {

      FD_SET(0, &msk_fd);

      select(FOPEN_MAX, &msk_fd, (fd_set *) 0, (fd_set *) 0, &TimeOut);

      Interface_Sniff_Data_SniffData(ipsource, ipdest, portsource, portdest, proto);

      if (FD_ISSET(0, &msk_fd))
      {
         KeyPress = wgetch(main_window);

         switch (KeyPress)
         {
            case KEY_DOWN:
                  Interface_Sniff_Data_Scroll( +1 );
                  break;

            case KEY_UP:
                  Interface_Sniff_Data_Scroll( -1 );
                  break;

            case KEY_NPAGE:
                  Interface_Sniff_Data_Scroll( W_MAINY2-10 );     //PGDOWN
                  break;

            case KEY_PPAGE:
                  Interface_Sniff_Data_Scroll( -(W_MAINY2-10) );  //PGUP
                  break;

            case 'L':
            case 'l':
                  Interface_Sniff_Data_LogToFile(proto, ipsource, portsource, ipdest, portdest);
                  break;

            case 'A':
            case 'a':
                  Interface_Sniff_Data_View(ASCII_VIEW);
                  break;

            case 'X':
            case 'x':
                  Interface_Sniff_Data_View(HEX_VIEW);
                  break;

            case 'S':
            case 's':
                  Interface_Sniff_Data_StopCont();
                  break;

            case 'I':
            case 'i':
                  if ( mode == ARPBASED )
                  {
                     if (strstr(app, "ssh") || strstr(app, "SSH"))
                     {
                        Interface_PopUp("You CAN'T inject in a crypted connection !!");
                        Interface_Sniff_Data_Redraw();
                     }
                     else
                        Interface_Sniff_Data_Inject(proto, app, ipsource, ipdest);
                  }
                  else
                  {
                     Interface_PopUp("Characters injection is only supported in ARP Based mode !!");
                     Interface_Sniff_Data_Redraw();
                  }
                  break;

            case 'F':
            case 'f':
                  if ( mode == ARPBASED )
                  {
                     if (strstr(app, "ssh") || strstr(app, "SSH"))
                     {
                        Interface_PopUp("You CAN'T filter a crypted connection !!");
                        Interface_Sniff_Data_Redraw();
                     }
                     else
                        Interface_Sniff_Data_Filter();
                  }
                  else
                  {
                     Interface_PopUp("Active Filtering is only supported in ARP Based mode !!");
                     Interface_Sniff_Data_Redraw();
                  }
                  break;

            case KEY_TAB:
                  Interface_Sniff_Data_KeyTab(ipsource, portsource, ipdest, portdest);
                  break;

            case KEY_CTRL_L:  // CTRL+L refresh the screen
                  Interface_Sniff_Data_Redraw();
                  break;

            case KEY_F(1):
            case 'H':
            case 'h':{
                        static char *help[] = {
                        "[qQ][F10] - quit (and stop sniffing)",
                        "[tab]     - switch between window",
                        "[pgUp]    - scroll back (as well as UpArrow)",
                        "[pgDw]    - scroll forward (as well as DownArrow)",
                        "[iI]      - inject characters in the connection",
                        "[fF]      - filter (search for, and replace or drop) packets",
                        "[aA]      - ASCII view",
                        "[xX]      - HEX view",
                        "[sS]      - stop/cont the sniff (only visualization)",
                        "[lL]      - Log to file",
                        NULL};
                        Interface_HelpWindow(help);
                     }
                     Interface_Sniff_Data_Redraw();
                  break;

            case KEY_F(10):
            case 'Q':
            case 'q':
                     if (inject == 1 && proto == 'T')
                     {
                        char answer;
                        answer = Interface_PopUp("If U exit now the connection may be ReSeTted!! continue (y/n)?");
                        Interface_Sniff_Data_Redraw();
                        if ((answer != 'y') && (answer != 'Y'))
                           break;
                        else
                           strcpy(Conn_Between_Hosts[Conn_Pointer].status, "KILLED");
                     }
                     if (Illithid_data_pid) kill(Illithid_data_pid, SIGTERM);
                     if (dopple_pid) kill(dopple_pid, SIGTERM);
                     if (log_dest) fclose(file_dest);
                     if (log_source) fclose(file_source);
                     log_dest = log_source = 0;
                     stop = 0;
                     inject = -1;
                     inject_tod=0; inject_tos=0;  // reset counter
                     delwin(data_source_win);
                     data_source_win = NULL;    // for winching (see ec_signal.c)
                     wbkgdset(bottom_window, COLOR_PAIR(BOTTOM_COLOR));
                     mvwhline(top_window, 2, 55, ' ', 20);
                     memset(&ARP_Filter_Source, 0, sizeof(DROP_FILTER));
                     memset(&ARP_Filter_Dest, 0, sizeof(DROP_FILTER));
                     delwin(data_dest_win);
                     delwin(data_source);
                     delwin(data_dest);
                     werase(bottom_window);
                     wnoutrefresh(top_window);
                     wnoutrefresh(bottom_window);
                     wnoutrefresh(main_window);
                     doupdate();
                     #ifdef DEBUG
                        Debug_msg("Interface_Sniff_Data_End");
                     #endif
                     return;
                  break;
         }
      }

   }

}



#endif

/* EOF */
