/*
    ettercap -- module for filtering or dropping packets that match criteria

    Copyright (C) 2001  ALoR <alor@users.sourceforge.net>, NaGA <crwm@freemail.it>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include "include/ec_main.h"

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#ifdef LINUX
   #define __USE_GNU    // for memmem() on linux
#endif
#include <string.h>
#include <sys/types.h>
#ifdef HAVE_CTYPE_H
   #include <ctype.h>
#endif

#include "include/ec_inet_structures.h"
#include "include/ec_inet_forge.h"
#include "include/ec_logtofile.h"

#ifdef DEBUG
   #include "include/ec_debug.h"
#endif

DROP_FILTER ARP_Filter_Source;   // ec_main.h
DROP_FILTER ARP_Filter_Dest;

// protos

int FilterDrop_MakefilterTCP(u_char *buf_ip, int *buflen, short maxlen, DROP_FILTER *filter);
int FilterDrop_MakefilterUDP(u_char *buf_ip, int *buflen, short maxlen, DROP_FILTER *filter);
int FilterDrop_memreplace(u_char *buf, int *buflen, short maxlen, DROP_FILTER *filter);
int FilterDrop_strescape(char *dst, char *src);

// -------------------------------

int FilterDrop_MakefilterTCP(u_char *buf_ip, int *buflen, short maxlen, DROP_FILTER *filter)
{
   u_char *data;
   int datalen;
   int delta = 0;
   IP_header *ip;
   TCP_header *tcp;

   ip = (IP_header *) buf_ip;
   tcp = (TCP_header *) ((int)ip + ip->h_len * 4);
   data = (char *)((int)tcp + tcp->doff * 4);
   datalen = (int)ip + ntohs(ip->t_len) - (int)data;


   if (filter->type != 'R' && filter->type != 'D' && filter->type != 'L' )   // improper filter
      return 0;

	if (filter->type == 'L')
	{
		if (FilterDrop_memreplace(data, &datalen, maxlen, filter))
		{
			LogToFile_FilteredData(buf_ip);
		}
		return 0;
	}
	else
   	delta = FilterDrop_memreplace(data, &datalen, maxlen, filter);

   if (delta > 0)
      ip->t_len += htons(delta);
   else
      ip->t_len -= htons(-delta);

   ip->checksum = 0;
   ip->checksum = Inet_Forge_ChecksumIP( (u_short *)ip, sizeof(IP_header) );
   tcp->checksum = 0;
   tcp->checksum = Inet_Forge_Checksum( (u_short *)tcp, IPPROTO_TCP, TCP_HEADER+datalen, ip->source_ip, ip->dest_ip );

   *buflen += delta;

   return delta;
}


int FilterDrop_MakefilterUDP(u_char *buf_ip, int *buflen, short maxlen, DROP_FILTER *filter)
{
   u_char *data;
   int datalen;
   int delta = 0;
   IP_header *ip;
   UDP_header *udp;

   ip = (IP_header *) buf_ip;
   udp = (UDP_header *) ((int)ip + ip->h_len * 4);
   data = (char *)((int)udp + UDP_HEADER);
   datalen = ntohs(udp->len) - UDP_HEADER;

   if (filter->type != 'R' && filter->type != 'D' && filter->type != 'L' )   // improper filter
      return 0;


	if (filter->type == 'L')
	{
		if (FilterDrop_memreplace(data, &datalen, maxlen, filter))
		{
			LogToFile_FilteredData(buf_ip);
		}
		return 0;
	}
	else
   	delta = FilterDrop_memreplace(data, &datalen, maxlen, filter);

   if (delta > 0)
   {
      ip->t_len += htons(delta);
      udp->len += htons(delta);
   }
   else
   {
      ip->t_len -= htons(-delta);
      udp->len -= htons(-delta);
   }

   ip->checksum = 0;
   ip->checksum = Inet_Forge_ChecksumIP( (u_short *)ip, sizeof(IP_header) );
   udp->checksum = 0;
   udp->checksum = Inet_Forge_Checksum( (u_short *)udp, IPPROTO_UDP, UDP_HEADER+datalen, ip->source_ip, ip->dest_ip );

   *buflen += delta;
   return delta;

}

int FilterDrop_memreplace(u_char *buf, int *buflen, short maxlen, DROP_FILTER *filter)
{
   u_char *ptr = buf;
   int found = 0;
   int rest;
   int end = *buflen;
   int delta = 0;
   DROP_FILTER sfilter;

   sfilter.slen = FilterDrop_strescape(sfilter.search, filter->search);       // convert escape sequences...
   sfilter.rlen = FilterDrop_strescape(sfilter.replace, filter->replace);
   sfilter.type = filter->type;

   do    // first we check that after replacement the size will be within maxlen...
   {
      rest = *buflen-((u_int)ptr-(u_int)buf);
      ptr = (u_char *)memmem(ptr, rest, sfilter.search, sfilter.slen);

      if (ptr != NULL)
      {
         delta += (sfilter.rlen - sfilter.slen);
         end = *buflen + delta;
         if (end > maxlen)
         {
            #ifdef DEBUG
               Debug_msg("FilterDrop_memreplace -- maxlen reached !! %d/%d from %d [%d][%d]", end, maxlen, *buflen, found, (sfilter.rlen - sfilter.slen) );
            #endif
            return 0;
         }
         ptr = (u_char *)((int)ptr + sfilter.slen);   // move the ptr after the replaced string
         found++;
      }

   }while (ptr != NULL);

   if (!found) return 0;

   if (filter->type == 'D') return -*buflen;
   if (filter->type == 'L') return 1;	// log this packet to a file

   ptr = buf;
   end = *buflen;
   delta = 0;

   do    // then we make the replacement...
   {
      rest = end-((u_int)ptr-(u_int)buf);
      ptr = (u_char *)memmem(ptr, rest , sfilter.search, sfilter.slen);

      if (ptr != NULL)
      {
         memmove(ptr + sfilter.rlen, ptr + sfilter.slen, rest - sfilter.slen );
         memcpy(ptr, sfilter.replace, sfilter.rlen);
         ptr = (u_char *)((int)ptr + sfilter.rlen);   // move the ptr after the replaced string
         delta += (sfilter.rlen - sfilter.slen);
         end = *buflen + delta;
      }

   }while (ptr != NULL);

   *buflen = end;

   return (found) ? delta : 0;

}



// adapted from magic.c part of dsniff <dugsong@monkey.org> source code...


static int hextoint(int c)
{
   if (!isascii((u_char) c))       return (-1);
   if (isdigit((u_char) c))        return (c - '0');
   if ((c >= 'a') && (c <= 'f'))   return (c + 10 - 'a');
   if ((c >= 'A') && (c <= 'F'))   return (c + 10 - 'A');
   return (-1);
}

int FilterDrop_strescape( char *dst, char *src)
{
   char  *olddst = dst;
   int   c;
   int   val;

   while ((c = *src++) != '\0')
   {
      if (c == '\\')
      {
         switch ((c = *src++))
         {
            case '\0':
               goto strend;
            default:
               *dst++ = (char) c;
               break;
            case 'n':
               *dst++ = '\n';
               break;
            case 'r':
               *dst++ = '\r';
               break;
            case 'b':
               *dst++ = '\b';
               break;
            case 't':
               *dst++ = '\t';
               break;
            case 'f':
               *dst++ = '\f';
               break;
            case 'v':
               *dst++ = '\v';
               break;
            /* \ and up to 3 octal digits */
            case '0':
            case '1':
            case '2':
            case '3':
            case '4':
            case '5':
            case '6':
            case '7':
               val = c - '0';
               c = *src++;  /* try for 2 */
               if (c >= '0' && c <= '7') {
                  val = (val << 3) | (c - '0');
                  c = *src++;  /* try for 3 */
                  if (c >= '0' && c <= '7')
                     val = (val << 3) | (c - '0');
                  else --src;
               }
               else --src;
               *dst++ = (char) val;
               break;

            case 'x':
               val = 'x';      /* Default if no digits */
               c = hextoint(*src++);     /* Get next char */
               if (c >= 0) {
                       val = c;
                       c = hextoint(*src++);
                       if (c >= 0) val = (val << 4) + c;
                       else --src;
               }
               else --src;
               *dst++ = (char) val;
               break;
         }
      }
      else if (c == 8 || c == 263)  // the backspace
         dst--;
      else
         *dst++ = (char) c;
   }

strend:
   *dst = '\0';

   return (dst - olddst);
}



/* EOF */
