/*
    ettercap -- the protocol dissector

    Copyright (C) 2001  ALoR <alor@users.sourceforge.net>, NaGA <crwm@freemail.it>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include "include/ec_main.h"

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>

#include "include/ec_dissector.h"
#include "include/ec_inet_structures.h"
#include "include/ec_decodedata.h"
#include "include/ec_error.h"

#ifdef DEBUG
   #include "include/ec_debug.h"
#endif


typedef struct {
   char mode;
   short proto;
   short port;
   int (*dissector)(u_char *, void *, int);
} DISSECTOR;

#define ALL_P -1


static DISSECTOR Available_Dissectors[] = {

   {MACBASED,  IPPROTO_TCP,    21, &Dissector_ftp},
#ifdef HAVE_OPENSSL
   {ARPBASED,  IPPROTO_TCP,    22, &Dissector_ssh},
#endif
   {MACBASED,  IPPROTO_TCP,    23, &Dissector_telnet},
   {MACBASED,  IPPROTO_TCP,    80, &Dissector_http},
   {MACBASED,  IPPROTO_TCP,   110, &Dissector_pop},
   {MACBASED,  IPPROTO_TCP,   119, &Dissector_nntp},
   {MACBASED,  IPPROTO_TCP,   139, &Dissector_smb},
   {MACBASED,  IPPROTO_TCP,   143, &Dissector_imap},
   {MACBASED,  IPPROTO_TCP,   179, &Dissector_bgp},
   {MACBASED,  IPPROTO_TCP,   220, &Dissector_imap},
   {MACBASED,  IPPROTO_TCP,   512, &Dissector_rlogin},
   {MACBASED,  IPPROTO_TCP,   513, &Dissector_rlogin},
   {MACBASED,  IPPROTO_TCP,   514, &Dissector_rlogin},
   {MACBASED,  IPPROTO_UDP,   520, &Dissector_rip},
   {MACBASED,  IPPROTO_TCP,  1080, &Dissector_socks},
   {MACBASED,  IPPROTO_TCP,  3306, &Dissector_mysql},
   {MACBASED,  IPPROTO_UDP,  4000, &Dissector_icq},
   {MACBASED,  IPPROTO_TCP,  5900, &Dissector_vnc},
   {MACBASED,  IPPROTO_TCP,  5901, &Dissector_vnc},
   {MACBASED,  IPPROTO_TCP,  5902, &Dissector_vnc},
   {MACBASED,  IPPROTO_TCP,  5903, &Dissector_vnc},
   {MACBASED,  IPPROTO_TCP,  5904, &Dissector_vnc},
   {MACBASED,  IPPROTO_TCP,  6000, &Dissector_x11},
   {MACBASED,  IPPROTO_TCP,  6001, &Dissector_x11},
   {MACBASED,  IPPROTO_TCP,  6002, &Dissector_x11},
   {MACBASED,  IPPROTO_TCP,  6003, &Dissector_x11},
   {MACBASED,  IPPROTO_TCP,  6004, &Dissector_x11},
   {MACBASED,  IPPROTO_TCP,  6666, &Dissector_napster},
   {MACBASED,  IPPROTO_TCP,  6667, &Dissector_irc},
   {MACBASED,  IPPROTO_TCP,  6668, &Dissector_irc},
   {MACBASED,  IPPROTO_TCP,  6669, &Dissector_irc},
   {MACBASED,  IPPROTO_TCP,  7777, &Dissector_napster},
   {MACBASED,  IPPROTO_TCP,  8080, &Dissector_http},     // for proxies
   {MACBASED,  IPPROTO_TCP,  8888, &Dissector_napster},
   {MACBASED,  IPPROTO_UDP, ALL_P, &Dissector_icq},      // ICQv5 uses different server port

                                          };

// protos....

void Dissector_Connections( char mode, short proto, u_char *data, void *data_to_ettercap, int Conn_Mode  );
int Dissector_base64decode(char *bufplain, const char *bufcoded);  // stolen from ap_base64.c part of apache source code

// -------------------------------------


void Dissector_Connections( char mode, short proto, u_char *data, void *data_to_ettercap, int Conn_Mode )
{

   TCP_header *tcp;
   UDP_header *udp;
   DISSECTOR *ds;

   switch(proto)
   {
      case IPPROTO_TCP:
                        tcp = (TCP_header *) data;

                        for( ds = Available_Dissectors; ds->port != 0; ds++)
                        {
                           if ( ds->proto == IPPROTO_TCP && mode <= ds->mode && (ds->port == ALL_P || ntohs(tcp->source) == ds->port || ntohs(tcp->dest) == ds->port) )
                           {
                              if (ds->mode == ARPBASED)
                              {
                                 if (active_dissector)      // activated by user in iterface_sniff
                                    ds->dissector(data, data_to_ettercap, Conn_Mode);
                              }
                              else
                                 ds->dissector(data, data_to_ettercap, Conn_Mode);

                              break;
                           }
                        }
                        break;
      case IPPROTO_UDP:
                        udp = (UDP_header *) data;

                        for( ds = Available_Dissectors; ds->port != 0; ds++)
                        {
                           if ( ds->proto == IPPROTO_UDP && mode <= ds->mode && (ds->port == ALL_P || ntohs(udp->source) == ds->port || ntohs(udp->dest) == ds->port) )
                           {
                              if (ds->mode == ARPBASED)
                              {
                                 if (active_dissector)      // activated by user in iterface_sniff
                                    ds->dissector(data, data_to_ettercap, Conn_Mode);
                              }
                              else
                                 ds->dissector(data, data_to_ettercap, Conn_Mode);

                              break;
                           }
                        }
                        break;
   }
}

// lines below stolen from ap_base64.c ... ;)

static const unsigned char pr2six[256] =
{
    /* ASCII table */
    64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
    64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
    64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 62, 64, 64, 64, 63,
    52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 64, 64, 64, 64, 64, 64,
    64,  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14,
    15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 64, 64, 64, 64, 64,
    64, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40,
    41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 64, 64, 64, 64, 64,
    64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
    64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
    64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
    64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
    64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
    64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
    64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64,
    64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64
};


int Dissector_base64decode(char *bufplain, const char *bufcoded)
{
    int nbytesdecoded;
    register const unsigned char *bufin;
    register unsigned char *bufout;
    register int nprbytes;

    bufin = (const unsigned char *) bufcoded;
    while (pr2six[*(bufin++)] <= 63);
    nprbytes = (bufin - (const unsigned char *) bufcoded) - 1;
    nbytesdecoded = ((nprbytes + 3) / 4) * 3;

    bufout = (unsigned char *) bufplain;
    bufin = (const unsigned char *) bufcoded;

    while (nprbytes > 4)
    {
      *(bufout++) = (unsigned char) (pr2six[*bufin] << 2 | pr2six[bufin[1]] >> 4);
      *(bufout++) = (unsigned char) (pr2six[bufin[1]] << 4 | pr2six[bufin[2]] >> 2);
      *(bufout++) = (unsigned char) (pr2six[bufin[2]] << 6 | pr2six[bufin[3]]);
      bufin += 4;
      nprbytes -= 4;
    }

    /* Note: (nprbytes == 1) would be an error, so just ingore that case */
    if (nprbytes > 1)
      *(bufout++) = (unsigned char) (pr2six[*bufin] << 2 | pr2six[bufin[1]] >> 4);

    if (nprbytes > 2)
      *(bufout++) = (unsigned char) (pr2six[bufin[1]] << 4 | pr2six[bufin[2]] >> 2);

    if (nprbytes > 3)
      *(bufout++) = (unsigned char) (pr2six[bufin[2]] << 6 | pr2six[bufin[3]]);

    nbytesdecoded -= (4 - nprbytes) & 3;

    bufplain[nbytesdecoded] = '\0';
    return nbytesdecoded;
}


/* EOF */
