/* ``The contents of this file are subject to the Erlang Public License,
 * Version 1.0, (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.erlang.org/EPL1_0.txt
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 * 
 * The Original Code is Erlang-4.7.3, December, 1998.
 * 
 * The Initial Developer of the Original Code is Ericsson Telecom
 * AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
 * Telecom AB. All Rights Reserved.
 * 
 * Contributor(s): ______________________________________.''
 */
/* Copyright (C) 1993, Ellemtel Telecommunications Systems Laboratories */
/* Author: Claes Wikstrom  klacke@erix.ericsson.se */
/* Purpose:   Implement the erlang port mapper daemon  */

#include "sys.h"
#include <sys/types.h>
#ifdef VXWORKS
#include <selectLib.h>
#include <sockLib.h>
#include <rpc/rpc.h>
#else /* ! VXWORKS */
#ifndef __WIN32__
#include <sys/time.h>
#endif
#endif /* ! VXWORKS */
#ifndef __WIN32__
#include <fcntl.h>
#include <netinet/in.h>
#include <sys/socket.h>
#ifdef DEF_INADDR_LOOPBACK_IN_RPC_TYPES_H
#include <rpc/types.h>
#endif
#include <arpa/inet.h>
#include <netinet/tcp.h>
#endif

#include <ctype.h>
#include <errno.h>

#ifndef NO_SYSLOG
#include <syslog.h>
#endif

#ifdef SYS_SELECT_H
#include <sys/select.h>
#endif
/* How to get max no of file descriptors? We used to use NOFILE from
   <sys/param.h>, but that tends to have little relation to reality.
   Best is to use sysconf() (POSIX), but we'll just punt if that isn't
   available (noone wants more than 64 Erlang systems on a host, right?:-) */
#define MAX_FILES 64		/* if sysconf() isn't available, or fails */
#ifndef NO_SYSCONF
#include <unistd.h>
#endif

#ifdef __WIN32__
#include <winsock2.h>
#include <windows.h>

#define close(s) closesocket((s))
#define write(a,b,c) send((a),(b),(c),0)
#define read(a,b,c) recv((a),(char *)(b),(c),0)

#ifndef EADDRINUSE
#define EADDRINUSE WSAEADDRINUSE
#endif

#endif


/* BEGIN epmd.h (with the port number moved to the "vsn.mk" file) */

#define EPMD_ALIVE 'a'
#define EPMD_PORT_PLEASE 'p'
#define EPMD_NAMES 'n'
#define EPMD_DUMP 'd'
#define EPMD_KILL 'k'
#define EPMD_STOP 's'

/* END R2 epmd.h */


#ifndef SOMAXCONN
#define SOMAXCONN 5
#endif

#define MAXSYMLEN 255

#define TRUE 1
#define FALSE 0

#define DEBUG1(format) {if (debug && !is_daemon) fprintf(stderr,format);}
#define DEBUG2(format,s2) {if (debug && !is_daemon) fprintf(stderr,format,s2);}

struct local {
    unsigned short port;
    char symname[MAXSYMLEN];
    int fd;
    short creation;
};

#if defined(HAVE_IN6) && defined(AF_INET6) && defined(EPMD6)

#define SOCKADDR_IN sockaddr_in6
#define FAMILY      AF_INET6

#define SET_ADDR_LOOPBACK(addr, af, port) do { \
    static u_int32_t __addr[4] = IN6ADDR_LOOPBACK_INIT; \
    memset((char*)&(addr), 0, sizeof(addr)); \
    (addr).sin6_family = (af); \
    (addr).sin6_flowinfo = 0; \
    (addr).sin6_addr.s6_addr32[0] = __addr[0]; \
    (addr).sin6_addr.s6_addr32[1] = __addr[1]; \
    (addr).sin6_addr.s6_addr32[2] = __addr[2]; \
    (addr).sin6_addr.s6_addr32[3] = __addr[3]; \
    (addr).sin6_port = htons(port); \
 } while(0)

#define SET_ADDR_ANY(addr, af, port) do { \
    static u_int32_t __addr[4] = IN6ADDR_ANY_INIT; \
    memset((char*)&(addr), 0, sizeof(addr)); \
    (addr).sin6_family = (af); \
    (addr).sin6_flowinfo = 0; \
    (addr).sin6_addr.s6_addr32[0] = __addr[0]; \
    (addr).sin6_addr.s6_addr32[1] = __addr[1]; \
    (addr).sin6_addr.s6_addr32[2] = __addr[2]; \
    (addr).sin6_addr.s6_addr32[3] = __addr[3]; \
    (addr).sin6_port = htons(port); \
 } while(0)

        
#else

#define SOCKADDR_IN sockaddr_in
#define FAMILY      AF_INET

#define SET_ADDR_LOOPBACK(addr, af, port) do { \
    memset((char*)&(addr), 0, sizeof(addr)); \
    (addr).sin_family = (af); \
    (addr).sin_addr.s_addr = htonl(INADDR_LOOPBACK); \
    (addr).sin_port = htons(port); \
 } while(0)

#define SET_ADDR_ANY(addr, af, port) do { \
    memset((char*)&(addr), 0, sizeof(addr)); \
    (addr).sin_family = (af); \
    (addr).sin_addr.s_addr = htonl(INADDR_ANY); \
    (addr).sin_port = htons(port); \
 } while(0)

#endif


static int max_nodes;
static struct local *nodes;

static fd_set read_mask, orig_read_mask;
static int debug;
static int silent=0; 
static int is_daemon = 0;
static unsigned char buf[BUFSIZ];

static char* progname;

/* forward declarations */
static void check_close(), do_request();
static void run();
static void kill_epmd();
static void epmd_call();
static void usage();
static void run_daemon();

/* Fill buffer, return buffer length, 0 for EOF, < 0 for error. */
static int read_fill(fd, buf, len)
char *buf;
{
    int i;
    int got = 0;
    
    do {
	if ((i = read(fd, buf+got, len-got)) <= 0) 
	    return (i);
	got += i;
    } while (got < len);
    return (len);
}

static
void d_perror(t)
char* t;

{
#ifndef NO_SYSLOG
    if (is_daemon)
	syslog(LOG_ERR, "%s: %m", t);
    else
#endif
	perror(t);
}

static
void d_error(t)
char* t;
{
#ifndef NO_SYSLOG
    if (is_daemon)
	syslog(LOG_ERR,"%s",t);
    else
#endif
	fprintf(stderr,"%s\n",t);
}


int
#ifdef DONT_USE_MAIN
epmd(argc,argv)
#else
main(argc,argv)
#endif
int argc;
char **argv;
{
    int debug_arg = 0;
    int erlang_daemon_port = ERLANG_DAEMON_PORT;

#ifdef __WIN32__
    WORD wVersionRequested;
    WSADATA wsaData;
    int err;

    wVersionRequested = MAKEWORD(1, 1);

    err = WSAStartup(wVersionRequested, &wsaData);
    if (err != 0)
        exit(1);

    if (LOBYTE(wsaData.wVersion) != 1 || HIBYTE(wsaData.wVersion ) != 1) {
        WSACleanup();
    	exit(1);
    }
#endif
    progname = argv[0];

    argc--;
    argv++;
    while (argc > 0) {
	if (strcmp(argv[0], "-d")==0) { /* debug (trace ) on */
	    debug_arg = 1;
	    argv++;
	    argc--;
	} else if (strcmp(argv[0], "-daemon") == 0) {
	    is_daemon = 1;
	    argv++;
	    argc--;
	} else if (strcmp(argv[0], "-kill") == 0) {
	    if (argc == 1)
		kill_epmd(erlang_daemon_port);
	    else
		usage();
	    exit(0);
	} else if (strcmp(argv[0], "-port") == 0) { /* Don't use default port */
	    if (argc == 1) 
		usage();
	    if ((erlang_daemon_port = atoi(argv[1])) == 0)
		usage();
	    argv++; argv++;
	    argc--; argc--;
	} else if (strcmp(argv[0], "-names") == 0) {
	    debug = debug_arg;
	    if (argc == 1)
		epmd_call(erlang_daemon_port, EPMD_NAMES);
	    else
		usage();
	    exit(0);
	} else if (strcmp(argv[0], "-started") == 0) {
	    debug = debug_arg;
	    silent = 1;
	    if (argc == 1)
		epmd_call(erlang_daemon_port, EPMD_NAMES);
	    else
		usage();
	    exit(0);
	} else if (strcmp(argv[0], "-dump") == 0) {
	    debug = debug_arg;
	    if (argc == 1)
		epmd_call(erlang_daemon_port, EPMD_DUMP);
	    else
		usage();
	    exit(0);
	}
	else
	    usage();
    }
    debug = debug_arg;
    DEBUG2("--- epmd running --- daemon = %d \n",is_daemon);
    if (is_daemon)  {
	run_daemon(erlang_daemon_port);
    } else {
	run(erlang_daemon_port);
    }
    return 0;
}

static void
run(port)
int port;
{
    int listensock;
    int i;
    int one = 1;
    int creation;
    struct SOCKADDR_IN iserv_addr;
    unsigned short erlang_daemon_port = port;

    DEBUG1("After init_daemon\n");
    creation = (time(NULL) % 3) + 1; /* "random" in range 1-3 */

    if ((listensock = socket(FAMILY,SOCK_STREAM,0)) < 0) {
	d_perror("epmd: opening stream socket");
	exit(1);
    }

    /*
     * Note that we must not enable the SO_REUSEADDR on Windows,
     * because addresses will be reused even if they are still in use.
     */
    
#ifndef __WIN32__
    if (setsockopt(listensock,SOL_SOCKET,SO_REUSEADDR,(char* ) &one,
		   sizeof(one)) <0) {
	d_perror("epmd: Cant set sockopt");
	exit(1);
    }
#endif

    SET_ADDR_ANY(iserv_addr, FAMILY, erlang_daemon_port);

    if(bind(listensock,(struct sockaddr*) &iserv_addr,
	    sizeof(iserv_addr)) <0) {
	if (errno == EADDRINUSE) {
	    exit(0);
	} else {
	    d_perror("epmd: Failed to bind socket");
	    exit(1);
	}
    }
    
    listen(listensock, SOMAXCONN);
#ifndef NO_SYSCONF
    if ((max_nodes = sysconf(_SC_OPEN_MAX)) <= 0)
#endif
	max_nodes = MAX_FILES;

    /* max_nodes must not be greater than FD_SETSIZE. */
    /* (at least QNX crashes)                         */
    if (max_nodes > FD_SETSIZE) max_nodes = FD_SETSIZE;

    if ((nodes =
	 (struct local *)malloc(max_nodes * sizeof(struct local))) == NULL) {
	d_perror("epmd: Insufficient memory");
	exit(1);
    }
    for(i = 0; i < max_nodes; i++) {
	memzero(nodes[i].symname,MAXSYMLEN);
	nodes[i].port  = 0;
	nodes[i].creation = creation;
	nodes[i].fd = -1;
    }
    memzero(buf,BUFSIZ);

    FD_ZERO(&read_mask);
    FD_ZERO(&orig_read_mask);
    FD_SET(listensock,&orig_read_mask);
    while(1) {	
	read_mask = orig_read_mask;
	if (select (max_nodes,&read_mask,(fd_set *)0,(fd_set *)0,
		    (struct timeval *)0) < 0 ) {
	    d_perror("epmd: error in select ");
	}
	
	else {
	    if(FD_ISSET(listensock,&read_mask))
		do_request(listensock, port);
	    else
		check_close();
	}
    }
}

static void check_close()
{
    int i;

    for (i=0;i<max_nodes;i++) {
	if (nodes[i].fd < 0 ) continue;
	if ((nodes[i].fd >= 0) && (FD_ISSET(nodes[i].fd,&read_mask)) ) {
	    if (debug) {
	    	fprintf(stderr, "epmd: closed '%s', port %d\n",
			nodes[i].symname, nodes[i].port);
	    }
	    FD_CLR(nodes[i].fd,&orig_read_mask);
	    close(nodes[i].fd);
	    nodes[i].fd = -1;   /* free now */
	    nodes[i].port = 0;
	    /* Keep the name and incr the creation */
	    if (nodes[i].creation == 3)
		nodes[i].creation = 1;
	    else
		nodes[i].creation += 1;
	    return;
	}
    }
    d_error("epmd: cant find local to close");
}


static void do_request(listensock, port)
int listensock, port;
{
    int i;
    int found, msgsock, rval, ival;
    unsigned char* t, tmp[4];
    unsigned char twobytes[2];
    struct SOCKADDR_IN icli_addr; /* workaround for QNX bug - cannot */
    int icli_addr_len;            /* handle NULL pointers to accept.  */
    int one = 1;

    icli_addr_len = sizeof(icli_addr);
    msgsock = accept(listensock,(struct sockaddr*) &icli_addr,
		     (int*) &icli_addr_len);
    if (msgsock < 0 ) {
	d_perror("epmd: Error  in accept\n");
	return;
    }
    setsockopt(msgsock, IPPROTO_TCP, TCP_NODELAY, (char *) &one, sizeof(one));

    rval = read_fill(msgsock,buf,2);
    if (rval <=  0) {
	close(msgsock);
	return;
    }
    ival = get_int16(buf);
    if ((rval = read_fill(msgsock,buf,ival)) != ival) {
	close(msgsock);
	return;
    }
    if (debug) {
	int num;
	
	fprintf(stderr,"epmd: GOT: %d bytes<",rval);
	for(i=0; i<rval; i++) {
	    if (isprint(buf[i])) {
		putc(buf[i],stderr);
		putc(',',stderr);
	    }
	    else {
		num = buf[i];
		fprintf(stderr,"%d,",num);
	    }
	}
	putc('>',stderr);
	putc('\n',stderr);
    }
    buf[rval] = '\0';
    switch (*buf) {

    case EPMD_ALIVE:
	
	/* here we expect a string on the format 
	   "axxyyyyyy" where xx is port and yyyyyy is symname 
	   port is given to us in host byte order */
	
	if (rval <= 3) {
	    close(msgsock);
	    return;
	}
	t = &buf[3]; /* points to nodename */
	found = 0;
	for (i=0;i<max_nodes;i++) {
	    if (strcmp(nodes[i].symname,t) == 0 ) {
		if (nodes[i].fd > 0) {  /* allready occupied */
		    close(msgsock);
		    return;
		}
		goto l_found; /* reuse slot, i leads to old name */
	    }
	}

	for (i=0; i<max_nodes; i++) {
	    if (nodes[i].fd < 0 && nodes[i].symname[0] == 0) 
		goto l_found;  /* i leads to new slot */
	}

	/* Ok now we know , that there are no free slots,
	   and no slot with our name in it .*/
	
	for (i=0; i<max_nodes; i++) {
	    if ( nodes[i].fd < 0 ) {
		/* Reuse slot */
		memzero(nodes[i].symname, MAXSYMLEN);
		nodes[i].port  = 0;
		goto l_found;  /* clear old unused slot */
	    }
	}

	/* All slots occupied  */

	close(msgsock);
	return;

    l_found:
	/* Now i is set */
	    
	t = &buf[1];
	tmp[0] = 'Y';
	put_int16(nodes[i].creation,tmp+1);
	if (write(msgsock,tmp,3) == 3) {
	    nodes[i].fd = msgsock;
	    nodes[i].port = *t << 8 | t[1];  /* still in host byte order */
	    DEBUG2("Port number was %d\n",nodes[i].port);
	    strcpy(nodes[i].symname,&buf[3]);
	    FD_SET(msgsock,&orig_read_mask);
	    return;
	}
	else {
	  close(msgsock);
	  return;
      }
	break;

    case EPMD_PORT_PLEASE:
	if (rval <= 1) {
	    close(msgsock);
	    return;
	}
	t = &buf[1];
	for (i=0;i<max_nodes;i++) {
	    if (nodes[i].fd < 0) continue;
	    if (strcmp(nodes[i].symname,t) == 0) {
		put_int16(nodes[i].port,twobytes);  /* hbo */
		write(msgsock,twobytes,2 );
		close(msgsock);
		return;
	    }
	}
	close(msgsock);
	break;
    case EPMD_NAMES:
	i = htonl(port);
	memcpy(buf,&i,4);
	write(msgsock,buf,4);
	for(i=0;i<max_nodes;i++) {
	    if (nodes[i].fd <0 ) continue;
	    sprintf(buf,"name %s at port %d\n",nodes[i].symname,nodes[i].port);
	    write(msgsock,buf,strlen(buf));
	}
	close(msgsock);
	break;
    case EPMD_DUMP:
	i = htonl(port);
	memcpy(buf,&i,4);
	write(msgsock,buf,4);
	for(i=0;i<max_nodes;i++) {
	    if (nodes[i].fd < 0 ) 
		sprintf(buf,"old/unused name <%s>, port = %d, fd = %d \n",
			nodes[i].symname,nodes[i].port, nodes[i].fd);
	    else
		sprintf(buf,"active name     <%s> at port %d, fd = %d\n",
			nodes[i].symname, nodes[i].port, nodes[i].fd);
	    write(msgsock,buf,strlen(buf)+1);
	}
	close(msgsock);
	break;
    case EPMD_KILL:
	DEBUG1("epmd killed\n");
	write(msgsock,"OK",2);
	close(msgsock);
	exit(0);
    case EPMD_STOP:
	t = &buf[1];
	if (rval <= 1 ) {
	    close(msgsock);
	    return;
	}
	for (i=0;i<max_nodes;i++) {
	    if (nodes[i].fd < 0 ) continue;
	    if (strcmp(nodes[i].symname,t)==0) {
		close(nodes[i].fd);  /* !!!  */
		write(msgsock,"STOPPED",7);
		nodes[i].fd = -1;
		close(msgsock);
		return;
	    }
	}
	write(msgsock,"NOEXIST",7);
	close(msgsock);
	return;
    default:
	d_error("epmd: got garbage ");
	close(msgsock);
    }
}

#ifndef NO_DAEMON
static void
run_daemon(port)
int port;
{
    register int child_pid, fd;
    
    DEBUG1("Deamon setup\n");
    
    /* fork to make sure first child is not a process group leader */
    if (( child_pid = fork()) <0) {
#ifndef NO_SYSLOG
	syslog(LOG_ERR,"erlang mapper daemon cant fork %m");
#endif
	exit(1);
    }
    
    else if (child_pid > 0) {
	DEBUG2("Daemon Child is %d\n",child_pid);
	exit(0);  /*parent */
    }
    
    if (setsid() < 0) {
	d_perror("epmd: Cant setsid()");
	exit(1);
    }
    
    /* close all files ... */
    for (fd = 0; fd < max_nodes ; fd++)
	close(fd);
    
    errno = 0;  /* if set by close */

    /* move cwd to root to make sure we are not on a mounted filesystem  */
    chdir("/");
    
    umask(0);
    run(port);
}

#endif /* NO_DAEMON */    

#ifdef __WIN32__
static void
run_daemon(port)
    int port;
{
    close(0);
    close(1);
    close(2);
    run(port);
}
#endif

#ifdef VXWORKS
static void
run_daemon(port)
    int port;
{
    run(port);
}
#endif


static int 
conn_to_epmd(port)
int port;
{
    struct SOCKADDR_IN address;
    int connect_sock;
    unsigned short erlang_daemon_port = port;
    
    connect_sock = socket(FAMILY, SOCK_STREAM, 0);
    if (connect_sock<0)
	goto error;
    SET_ADDR_LOOPBACK(address, FAMILY, erlang_daemon_port);

    if (connect(connect_sock, (struct sockaddr*)&address, sizeof address) < 0) 
	goto error;
    return connect_sock;

 error:
    if (!silent) {
	fprintf(stderr, "epmd: Cannot connect to local epmd\n");
    }
    exit(1);
    return -1;
}


static void
kill_epmd(port)
int port;
{
    char buf[5];
    int fd, rval;

    fd = conn_to_epmd(port);
    put_int16(1,buf);
    buf[2] = EPMD_KILL;
    if (write(fd, buf, 3) != 3) {
	printf("epmd: Can't write to epmd\n");
	exit(1);
    }
    if ((rval = read_fill(fd,buf,2)) == 2) {
	printf("Killed\n");
	exit(0);
    } else {
	buf[rval] = buf[5] = '\0';
	printf("epmd: local epmd responded with <%s>\n", buf);
	exit(1);
    }
}

/* what == EPMD_NAMES || EPMD_DUMP */

static void
epmd_call(port, what)
int port;
int what;
{
    char buf[BUFSIZ];
    int rval,fd,i,j;
    
    fd = conn_to_epmd(port);
    put_int16(1,buf);
    buf[2] = what;
    write(fd,buf,3);
    if (read(fd,(char *)&i,4) != 4) {
	if (!silent)
	    printf("epmd: no response from local epmd\n");
	exit(1);
    }
    j = ntohl(i);
    if (!silent)
	printf("epmd: up and running on port %d with data:\n", j);
    while(1) {
	if ((rval = read(fd,buf,1)) <= 0)  {
	    close(fd);
	    exit(0);
	}
	buf[rval] = '\0';
	if (!silent)
	    printf("%s",buf);
    }
}

static void
usage()
{
    fprintf(stderr, "usage: epmd [-port Portno] [-daemon]\n");
    fprintf(stderr, "            [-port Portno] [-names|-kill]\n");
    exit(1);
}
