/* ``The contents of this file are subject to the Erlang Public License,
 * Version 1.0, (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.erlang.org/EPL1_0.txt
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 * 
 * The Original Code is Erlang-4.7.3, December, 1998.
 * 
 * The Initial Developer of the Original Code is Ericsson Telecom
 * AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
 * Telecom AB. All Rights Reserved.
 * 
 * Contributor(s): ______________________________________.''
 */
/*
 * Purpose: System-dependent time functions.
 */

#include "sys.h"
#include "assert.h"

/******************* Routines for time measurement *********************/

static struct _timeb gtv;
static struct _timeb inittv;
static struct _timeb then;
static unsigned long last;	/* The time this process spent executing
				 * in user mode.
				 */

static struct _timeb last_time;

FUNCTION(int, sys_time_remaining, (void));
FUNCTION(void, sys_deliver_time, (struct _timeb* current));
EXTERN_FUNCTION(void, increment_time, (int));
EXTERN_FUNCTION(int, next_time, (_VOID_));

/*
 * Local functions.
 */

static unsigned long filetime_to_ms(FILETIME* time);
/*
** A Correct(TM) version of ftime().
*/ 
static void sys_ftime(struct _timeb *);
#ifdef DEBUG
static void test_filetime_to_ms(void);
#endif

sys_init_time(itime)
int itime;
{
    FILETIME dummy;
    FILETIME user;		/* The time this process spent in user mode. */

#ifdef DEBUG
    test_filetime_to_ms();
#endif

    sys_ftime(&last_time);

    /* record the initial time for the statistics function */
    sys_ftime(&inittv);
    gtv = inittv;    
    then.time = then.millitm = 0;

    GetProcessTimes(GetCurrentProcess(), &dummy, &dummy, &dummy, &user);
    last = filetime_to_ms(&user);
    return 1;
}

#define EPOCH_JULIAN_DIFF 11644473600i64

static void 
sys_ftime(tb)
struct _timeb *tb;
{
    SYSTEMTIME t;
    FILETIME ft;
    LONGLONG lft;
    TIME_ZONE_INFORMATION tzi;
    DWORD ret;

    GetSystemTime(&t);
    SystemTimeToFileTime(&t, &ft);
    memcpy(&lft, &ft, sizeof(lft));

    tb->millitm = (unsigned short) ((lft / 10000) % 1000);
    tb->time = (unsigned long) ((lft / 10000000) - EPOCH_JULIAN_DIFF);

    if (( ret = GetTimeZoneInformation(&tzi)) == TIME_ZONE_ID_INVALID){
	tb->dstflag = tb->timezone = 0;
    } else {
	tb->dstflag = (ret == TIME_ZONE_ID_DAYLIGHT);
	tb->timezone = (short) tzi.Bias;
    }
}

static unsigned long
filetime_to_ms(time)
FILETIME* time;			/* A filetime to be converted to milliseconds. */
{
  /*
   * Divide the filetime with 10000 to obtain the corresponding time in
   * in milliseconds.  This is done like this in integer arithmethic.
   *
   * Let H be the value of time.dwHighDateTime and L the value
   * time.dwLowDateTime.  Thus, we want to calculate:
   *
   * (2^32*H + L)/10000 ==
   * (4294967296*H + L)/10000 ==
   * (429496+7296/10000)*H + L/10000 ==
   * 429496*H + 7296*H/10000 + L/10000
   * 429496*H + (7296*H+L)/10000
   *            ^^^^^^^^^^^^^^^^
   *	               ! Calculated in pices to not cause overflow.
   */

  unsigned long middle = 7296*time->dwHighDateTime;

  return 429496*time->dwHighDateTime +
    middle/10000 + time->dwLowDateTime/10000 + 
    (middle%10000 + time->dwLowDateTime%10000)/10000;
}

#ifdef DEBUG
static void 
test_filetime_to_ms(void)
{
  FILETIME dummy;

  #define VERIFY(high, low, result) \
  { dummy.dwHighDateTime = high; \
    dummy.dwLowDateTime = low; \
    assert(filetime_to_ms(&dummy) == result); }
    
  VERIFY(0, 9999, 0);
  VERIFY(0, 2320000, 232);
  VERIFY(0, 2321234, 232);
  VERIFY(1, 0, 429496);
  VERIFY(1, 2703, 429496);	/* 4294969999 */
  VERIFY(1, 2704, 429497);	/* 4294970000 */
  VERIFY(1, 0x0000ffff, 429503);
  VERIFY(0xF1, 0, 103508711);
  VERIFY(0xF1, 0x0000ffff,103508718);
  VERIFY(1, 0xffffffff, 858993);
  VERIFY(9999, 0xffffffff, 4294967295);
  VERIFY(0xffffffff, 0xffffffff, 429496);
  #undef VERIFY
}
#endif

void
elapsed_time_both(ms_user, ms_sys, ms_user_diff, ms_sys_diff)
     unsigned long *ms_user, *ms_sys;
     unsigned long *ms_user_diff, *ms_sys_diff;
{
    FILETIME dummy;
    FILETIME user;
    FILETIME system;

    *ms_user = 0;
    *ms_sys = 0;
    *ms_user_diff = 0;
    *ms_sys_diff = 0;

    if (GetProcessTimes(GetCurrentProcess(), &dummy, &dummy,
			&system, &user) == 0)
      return;

    *ms_user = filetime_to_ms(&user);
    *ms_sys = filetime_to_ms(&system);

    /* We always return 0 for ms_sys_diff, which isn't quite appropriate,
       since ms_sys isn't 0, but currently it doesn't matter, since the
       sys values aren't used. [arndt] */

    *ms_user_diff = *ms_user - last;
    last = *ms_user;
}

/* wall clock routines */
void wall_clock_elapsed_time_both(ms, ms_diff)
unsigned long *ms, *ms_diff;
{
    struct _timeb tv;

    sys_ftime(&tv);
    *ms = 1000 * (tv.time - inittv.time) +
	(tv.millitm - inittv.millitm);

    *ms_diff = 1000 * (tv.time - gtv.time) + (tv.millitm - gtv.millitm);
    gtv = tv;
    sys_deliver_time(&tv);
}

/* get current time */
void get_time(hour, minute, second)
int *hour, *minute, *second;
{
    time_t clock;
    struct tm *tm;

    clock = time((time_t *)0);
    tm = localtime(&clock);
    *hour = tm->tm_hour;
    *minute = tm->tm_min;
    *second = tm->tm_sec;
}

/* get current date */
void get_date(year, month, day)
int *year, *month, *day;
{
    time_t clock;
    struct tm *tm;

    clock = time((time_t *)0);
    tm = localtime(&clock);
    *year = tm->tm_year + 1900;
    *month = tm->tm_mon +1;
    *day = tm->tm_mday;
}

/* get localtime */
void get_localtime(year, month, day, hour, minute, second)
     int *year, *month, *day, *hour, *minute, *second;
{
  time_t clock;
  struct tm *tm;
  
  clock = time((time_t *)0);
  tm = localtime(&clock);
  *year = tm->tm_year + 1900;
  *month = tm->tm_mon +1;
  *day = tm->tm_mday;
  *hour = tm->tm_hour;
  *minute = tm->tm_min;
  *second = tm->tm_sec;
}

/* get universaltime */
void get_universaltime(year, month, day, hour, minute, second)
     int *year, *month, *day, *hour, *minute, *second;
{
  time_t clock;
  struct tm *tm;
  
  clock = time((time_t *)0);
  tm = gmtime(&clock);
  *year = tm->tm_year + 1900;
  *month = tm->tm_mon +1;
  *day = tm->tm_mday;
  *hour = tm->tm_hour;
  *minute = tm->tm_min;
  *second = tm->tm_sec;
}


/* days in month = 1, 2, ..., 12 */
static const int mdays[14] = {0, 31, 28, 31, 30, 31, 30,
			      31, 31, 30, 31, 30, 31};
static const int check[14] = {0, 31, 28, 31, 30, 31, 30,
			      31, 31, 30, 31, 30, 31};

#define  IN_RANGE(a,x,b)  (((a) <= (x)) && ((x) <= (b)))
#define  is_leap_year(y)  (((((y) % 4) == 0) && \
                            (((y) % 100) != 0)) || \
                           (((y) % 400) == 0))
     
#define  BASEYEAR	1970
/*
 * gregday
 *
 * Returns the number of days since Jan 1, 1600, if year is
 * greater of equal to 1600, and month [1-12] and day [1-31] 
 * are within range. Otherwise it returns -1.
 *
 * The base day is chosen to be divisable by both 4 and
 * 400, so that the leap day rule will work correctly.
 */
static int long gregday(int year, int month, int day)
{
  int long ndays = 0;
  int gyear, pyear, m;
  
  /* number of days in previous years */
  gyear = year - 1600;
  if (gyear > 0) {
    pyear = gyear - 1;
    ndays = (pyear/4) - (pyear/100) + (pyear/400) + pyear*365 + 366;
  }
  /* number of days in all months preceeding month */
  for (m = 1; m < month; m++)
    ndays += mdays[m];
  /* Extra day if leap year and March or later */
  if (is_leap_year(year) && (month > 2))
    ndays++;
  ndays += day - 1;
  return ndays - 135140;	/* 135140 = Jan 1, 1970 */
}

int local_to_univ(year, month, day, hour, minute, second)
     int *year, *month, *day, *hour, *minute, *second;
{
  time_t clock;
  struct tm *tm, t;
  
  if (!(IN_RANGE(BASEYEAR, *year, INT_MAX - 1) &&
	IN_RANGE(1, *month, 12) &&
	IN_RANGE(1, *day, (mdays[*month] + 
			   (*month == 2 
			    && (*year % 4 == 0) 
			    && (*year % 100 != 0 || *year % 400 == 0)))) &&
	IN_RANGE(0, *hour, 23) &&
	IN_RANGE(0, *minute, 59) &&
	IN_RANGE(0, *second, 59))) {
    return 0;
  }
  
  t.tm_year = *year - 1900;
  t.tm_mon = *month - 1;
  t.tm_mday = *day;
  t.tm_hour = *hour;
  t.tm_min = *minute;
  t.tm_sec = *second;
  t.tm_isdst = -1;
  clock = mktime(&t);
  tm = gmtime(&clock);
  *year = tm->tm_year + 1900;
  *month = tm->tm_mon +1;
  *day = tm->tm_mday;
  *hour = tm->tm_hour;
  *minute = tm->tm_min;
  *second = tm->tm_sec;
  return 1;
}

int univ_to_local(year, month, day, hour, minute, second)
     int *year, *month, *day, *hour, *minute, *second;
{
  time_t clock;
  struct tm *tm;
  
  if (!(IN_RANGE(BASEYEAR, *year, INT_MAX - 1) &&
	IN_RANGE(1, *month, 12) &&
	IN_RANGE(1, *day, (mdays[*month] + 
			   (*month == 2 
			    && (*year % 4 == 0) 
			    && (*year % 100 != 0 || *year % 400 == 0)))) &&
	IN_RANGE(0, *hour, 23) &&
	IN_RANGE(0, *minute, 59) &&
	IN_RANGE(0, *second, 59))) {
    return 0;
  }
  
  clock = *second + 60 * (*minute + 60 * (*hour + 24 *
					  gregday(*year, *month, *day)));
  tm = localtime(&clock);
  *year = tm->tm_year + 1900;
  *month = tm->tm_mon +1;
  *day = tm->tm_mday;
  *hour = tm->tm_hour;
  *minute = tm->tm_min;
  *second = tm->tm_sec;
  return 1;
}

/* get a timestamp
 * we are assuming that this function cannot be called more than one hundred 
 * times per millisecond - should be OK until at least the year 2002
 */

void
get_now(megasec, sec, microsec)
uint32 *megasec;
uint32 *sec;
uint32 *microsec;
{
    static int then_micro = 0;
    struct _timeb now;
  
    sys_ftime(&now);
    sys_deliver_time(&now);

    /* Make sure time is later than last */

    if (then.time > now.time ||
	(then.time == now.time && then.millitm >= now.millitm)) {
	now = then;
	then_micro += 10;	/* so we don't get exactly same answer twice */
    } else {			/* reset the number of microseconds if
				 * we've ticked at least another millisecond
				 * since the last call... */
	then_micro = 0;
    }

    *megasec = now.time / 1000000;
    *sec = (now.time % 1000000);
    *microsec = (now.millitm * 1000) + then_micro;
    then = now;
}

/* deliver elapsed ticks to the machine - takes a pointer
   to a struct timeval representing current time (to save
   a gettimeofday() where possible) or NULL */

void
sys_deliver_time(current)
struct _timeb* current;
{
    struct _timeb cur_time;
    long elapsed;
    
    if (current == NULL) {
	current = &cur_time;
	sys_ftime(current);
    }

    elapsed = 1000 * (current->time - last_time.time) +
	(current->millitm - last_time.millitm);

    /* HP machines seem to return an earlier time sometimes,
       resulting in a negative elapsed time. We compensate for
       this by simply pretend as if the time stood still. :) */
    if (elapsed < 0)
	elapsed = 0;
    increment_time(elapsed);
    last_time = *current;
}

/* get real time remaining until next timeout - if there
   isn't one, give a "long" time, that is guaranteed
   to not cause overflow when we report elapsed time later on */

int
sys_time_remaining(void)
{
    int ticks;
    struct _timeb cur_time;
    long elapsed;

    /* next_time() returns no of ticks to next timeout or -1 if none */

    if ((ticks = next_time()) == -1) { /* Timer queue empty */
	return 100000;
    } else {			/* Next timeout after tick ticks */
	sys_ftime(&cur_time);
	elapsed = 1000 * (cur_time.time - last_time.time) +
	    (cur_time.millitm - last_time.millitm);
	if (ticks <= elapsed) { /* Ooops, better hurry */
	    return 0;
	}
	return ticks - elapsed;
    }
}
