/* ``The contents of this file are subject to the Erlang Public License,
 * Version 1.0, (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.erlang.org/EPL1_0.txt
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 * 
 * The Original Code is Erlang-4.7.3, December, 1998.
 * 
 * The Initial Developer of the Original Code is Ericsson Telecom
 * AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
 * Telecom AB. All Rights Reserved.
 * 
 * Contributor(s): ______________________________________.''
 */
/* This file is copyright (c) Ellemtel in January 1991
 *
 * Author Mike Williams, Claes Wikstrom
 *
 *
 */
#include "sys.h"
#include "config.h"
#include "global.h"
#include "erl_process.h"
#include "big.h"
#include "bif.h"

/* Exported from this file, but in what header file should this be? */
EXTERN_FUNCTION(void, trace_bif,
		(Process*, BifFunction, uint32, uint32, uint32));

/* Imported from drv/gzio.c. Why not in any header file? */
EXTERN_FUNCTION(DriverBinary*, gzinflate_buffer, (char*, int));

/* Imported from sys.c */
EXTERN_FUNCTION(void*, sys_alloc2, (unsigned int));
EXTERN_FUNCTION(void*, sys_realloc2, (void*,unsigned int));
EXTERN_FUNCTION(void, sys_free2, (void*));

#define UNSAFE_MASK  0xf0000000	/* Mask for bits that must be constant
				 * in pointers.
				 */


/*
 * The following macro checks that the start and end addresses of
 * a memory block allocated by sys_alloc() are inside the allowable
 * ranges (to allow four tag bits).
 */

#if EXTRA_POINTER_BITS == 0
/*
 * In this case, memory is allocated from 0 and upwards.
 * If the end pointer is okay, the start pointer must be okay too.
 */
#define CHECK_MEMORY(ptr, size) \
  if (((((unsigned long) ptr)+size) & UNSAFE_MASK) != 0) {\
    erl_exit(1, "Got unusable memory block 0x%x, size %u\n", ptr, size); \
  }
#else
/*
 * The following test assumes that the start pointer can never be
 * below the allowed range.  If that could be the case we would have
 * to test both start and end pointers.
 */
#define CHECK_MEMORY(ptr, size) \
  if (((((unsigned long) ptr)+size) & UNSAFE_MASK) != EXTRA_POINTER_BITS) {\
    erl_exit(1, "Got unusable memory block 0x%x, size %u\n", ptr, size); \
  }

#endif

/* allocate buffer memeory and attach it to heap */
uint32* halloc(p, sz)
Process* p; uint32 sz;
{
    ErlMessageBuffer* bp = new_message_buffer(sz);

    /* Link in to process */
    bp->next = p->mbuf;
    p->mbuf = bp;
    /* Update mbuf size */
    p->mbuf_sz += sz;
    p->mbuf_struct_sz += 
	    (sizeof(ErlMessageBuffer)/sizeof(uint32) - 1); 
    /* mbuf_struct_sz is the administrative overhead caused by 
       message buffers, used together with mbuf_sz to indicate 
       that GC is needed */
    /* A complete test if GC is needed is done to prevent that execution
       of BIFs in a loop creates so many mesagebuffers that memory gets 
       exhausted. If gc is needed all reductions are bumped which causes
       the process to be rescheduled with initial gc as a result */
    if ((p->mbuf_sz + p->mbuf_struct_sz)*MBUF_GC_FACTOR >= p->heap_sz) {
	BUMP_ALL_REDS(p);
	p->flags |= F_NEED_GC;
    }
    /* We set the F_NEED_GC flag here, because then the scheduler
       will notice that gc is needed in a cheaper test than the
       full GC test which the scheduler also performs
       */
    return bp->mem;
}

/*
** Suspend a process 
** If we are to suspend on a port the busy_port is the thing
** otherwise busy_port is NIL
*/
void erl_suspend(process, busy_port)
Process* process; uint32 busy_port;
{
    process->rcount++;  /* count number of suspend */
    switch(process->status) {
    case P_SUSPENDED:
	break;
    case P_RUNABLE:
	remove_proc_from_sched_q(process);
	process->rstatus = P_RUNABLE; /* wakeup as runnable */
	break;
    case P_RUNNING:
	process->rstatus = P_RUNABLE; /* wakeup as runnable */
	break;
    case P_WAITING:
	process->rstatus = P_WAITING; /* wakeup as waiting */
	break;
    case P_EXITING:
	return; /* ignore this */
    case P_GARBING:
    case P_FREE:
	erl_exit(1, "bad state in erl_suspend\n");
    }
    process->status = P_SUSPENDED;
    if (busy_port != NIL)
	wake_process_later(busy_port, process);
}


void erl_resume(process)
Process* process;
{
    /* We may get called from trace([suspend], false) */
    if (process->status != P_SUSPENDED)
	return;
    ASSERT(process->rcount > 0);

    if (--process->rcount > 0)  /* multiple suspend i.e trace and busy port */
	return;
    switch(process->rstatus) {
    case P_RUNABLE:
	process->status = P_WAITING;  /* make add_to_schedule_q work */
	add_to_schedule_q(process);
	break;
    case P_WAITING:
	process->status = P_WAITING;
	break;
    default:
	erl_exit(1, "bad state in erl_resume\n");
    }
    process->rstatus = P_FREE;
}


/* CTYPE macros */

#define IS_DIGIT(c)  ((c) >= '0' && (c) <= '9')
#define IS_LOWER(c)  ((c) >= 'a' && (c) <= 'z')
#define IS_UPPER(c)  ((c) >= 'A' && (c) <= 'Z')
#define IS_ALNUM(c)  (IS_DIGIT(c) || IS_LOWER(c) || IS_UPPER(c))
#define IS_SPACE(c)  (c == ' ' || c == '\n' || c == '\t' || c == '\r')
#define IS_CNTRL(c)  (!IS_SPACE(c) && (c) < ' ')
#define IS_PUNCT(c)  (!IS_CNTRL(c) && !IS_ALNUM(c))
#define IS_PRINT(c)  (IS_ALNUM(c) || (!IS_CNTRL(c) && (c) <= 0177))

/*
** Generate the integer part from a double
*/
uint32 double_to_integer(p, x)
Process* p; double x;
{
    int sign;
    int ds;
    digit_t* xp;
    int i;
    uint32 res;
    uint32 sz;
    uint32* hp;

    if ((x <= (float) MAX_SMALL) && (x >= (float) MIN_SMALL)) {
	sint32 xi = x;
	return make_small(xi);
    }

    if (x < 0) {
	sign = 1;
	x = -x;
    }
    else
	sign = 0;

    /* Unscale & (calculate exponent) */
    ds = 0;
    while(x >= 1.0) {
	x /= D_BASE;         /* "shift" right */
	ds++;
    }
    sz = ((ds+1) >> 1);          /* number of words */

    /*
     * Beam note: This function is called from guard bifs (round/1 and trunc/1),
     * which are not allowed to build anything at all on the heap.
     * Therefore it is essential to use the ArithAlloc() macro instead of HAlloc()
     * (on Jam, ArithAlloc() is just an alias for HAlloc()).
     */
    hp = ArithAlloc(p, sz+1);
    res = make_big(hp);
    xp = (digit_t*) (hp + 1);

    for (i = ds-1; i >= 0; i--) {
	digit_t d;

	x *= D_BASE;      /* "shift" left */
	d = x;            /* trunc */
	xp[i] = d;        /* store digit */
	x -= d;           /* remove integer part */
    }
    if (ds & 1)  /* odd ds need to zero high word */
	xp[ds] = 0;

    if (sign)
	*hp = make_thing(sz) | BIG_SIGN_BIT;
    else
	*hp = make_thing(sz);
    hp += (sz + 1);
    return res;
}

/*
** Create a new link
*/
ErlLink* new_link(next, type, item, data)
ErlLink* next; ErlLinkType type; uint32 item; uint32 data;
{
    ErlLink* lnk = (ErlLink*) fix_alloc(link_desc);

    lnk->next = next;
    lnk->type = type;
    lnk->item = item;
    lnk->data = data;
    return lnk;
}

/*
** Delete an old link (and relink)
*/
void del_link(lnk)
ErlLink** lnk;
{
    ErlLink* tlink;

    if (lnk != NULL) {
	tlink = *lnk;
	*lnk = tlink->next;
	fix_free(link_desc, (uint32*)tlink);
    }
}

/*
** Find a link.
** Result is NULL if not found 
** otherwise a pointer to a pointer to it is returned (fit with del_link)
*/
ErlLink** find_link(first, type, item, data)
ErlLink** first; ErlLinkType type; uint32 item; uint32 data;
{
    ErlLink* lnk = *first;
    ErlLink* prev = NULL;

    while(lnk != NULL) {
	if ((lnk->type == type) && (lnk->item == item)) {
	    if ((data == NIL) || (lnk->data == data))
		return (prev == NULL) ? first : &prev->next;
	}
	prev = lnk;
	lnk = lnk->next;
    }
    return NULL;
}

/*
** Calculate length of a list 
** -1 if not a proper list i.e not terminated with NIL
*/
int list_length(list)
uint32 list;
{
    int i = 0;

    while(is_list(list)) {
	i++;
	list = CDR(ptr_val(list));
    }
    if (is_not_nil(list))
	return -1;
    return i;
}


/* make a hash index from an erlang term */

/* some prime numbers just above 2 ^ 28 */

#define FUNNY_NUMBER1 268440163
#define FUNNY_NUMBER2 268439161
#define FUNNY_NUMBER3 268435459
#define FUNNY_NUMBER4 268436141
#define FUNNY_NUMBER5 268438633
#define FUNNY_NUMBER6 268437017
#define FUNNY_NUMBER7 268438039
#define FUNNY_NUMBER8 268437511
#define FUNNY_NUMBER9 268439627


uint32 make_hash(term, hash)
uint32 term; uint32 hash;
{
    switch tag_val_def(term) {
    case ATOM_DEF:
	return hash*FUNNY_NUMBER1 + 
	    (atom_tab(unsigned_val(term))->slot.bucket.hvalue);

    case SMALL_DEF:
	return hash*FUNNY_NUMBER2 + unsigned_val(term);
#if defined(JAM)
    case NIL_DEF:
	return hash*FUNNY_NUMBER3 + 1;
#endif
    case BINARY_DEF:
	{
	    byte* ptr = ((ProcBin*) ptr_val(term))->bytes;
	    uint32 sz = ((ProcBin*) ptr_val(term))->size;
	    int i = (sz < 15) ? sz : 15;

	    while(i--)
		hash = hash*FUNNY_NUMBER1 + *ptr++;
	    return hash*FUNNY_NUMBER4 + sz;
	}

    case PID_DEF:
	return hash*FUNNY_NUMBER5 + get_number(term);

    case PORT_DEF:
    case REFER_DEF:
	return hash*FUNNY_NUMBER9 + get_number_reference(term);

    case FLOAT_DEF: 
	{
	    FloatDef ff;
	    GET_DOUBLE(term, ff);
	    return hash*FUNNY_NUMBER6 + (ff.fw[0] ^ ff.fw[1]);
	}
	break;

    case LIST_DEF:
	{
	    uint32* list = ptr_val(term);

	    while(1) {
		hash = make_hash(*list, hash);
		if (is_not_list(CDR(list)))
		    return make_hash(CDR(list),hash)*FUNNY_NUMBER8;
		list = ptr_val(CDR(list));
	    }
	}
	break;

    case BIG_DEF:
#if defined(BEAM)
      if (is_nil(term)) {
	return hash*FUNNY_NUMBER3 + 1;
      }
#endif
      {
	uint32* ptr  = ptr_val(term);
	uint32 arity = thing_arityval(*ptr);
	int sign = *ptr & BIG_SIGN_BIT;
	int i = arity;
	
	ptr++;
	while(i--)
	  hash = hash*FUNNY_NUMBER2 + *ptr++;
	
	if (sign)
	  return hash*FUNNY_NUMBER3 + arity;
	else
	  return hash*FUNNY_NUMBER2 + arity;
      }
      break;

    case TUPLE_DEF: 
	{
	    uint32* ptr = ptr_val(term);
	    uint32 arity = arityval(*ptr);
	    int i = arity;

	    ptr++;
	    while(i--)
		hash = make_hash(*ptr++, hash);
	    return hash*FUNNY_NUMBER9 + arity;
	}
	break;

    default:
	erl_exit(1, "Invalid tag in make_hash\n");
	return 0;
    }
}

/*
 * the function is used by apply to find the address into the export_list and
 * when called the undefined code handler.
 * I HAVE INLINED calls to index_get ... for faster apply !
 */
int find_function(m, f, a)
uint32 m; uint32 f; int a;
{
    Export e;
    Export* ep;

    e.module = unsigned_val(m);
    e.function = unsigned_val(f);
    e.arity = a;

    if ((ep = hash_get(&export_table.htable, (void*) &e)) == NULL)
	return -1;
#ifdef BEAM
    if (ep->address == &ep->op_error_handler)
	return -1;
#else
    if (ep->address == 0)
	return -1;
#endif
    return ep->slot.index;
}

/* look up an entry in the table, returning the index of the function or -1
   if not found.
 */
int find_bif(atom_index,arity)
int atom_index, arity;
{
    int i;

    for (i = 0; i < BIF_SIZE; i++) {
	if (bif_table[i].arity == arity &&
	    unsigned_val(*bif_table[i].name) == atom_index)
	    return i;
    }
    return -1;
}


/* Return the name of the bif with index "index" */
#if ! defined(BEAM)
const char* bif_name(indx)
int indx;
{
    if (indx >= 0 && indx < BIF_SIZE) {
	int i = bif_table[indx].name - &am_ix[0];
	return (const char*) am_nm[i];
    }
    else
	return "????";
}
#endif /* BEAM */

int send_error_to_logger(gleader)
uint32 gleader;
{
    Process* p;
    ErlMessageBuffer* bp;
    uint32* hp;
    uint32 name;
    uint32 res;
    uint32 gl;
    uint32 list;
    int i;
    
    if (gleader == 0)
	gl = am_noproc;
    else
	gl = gleader;
    if ((i = cerr_pos) == 0)
	return 0;
    cerr_pos = 0;
    name = am_error_logger;
    if ((p = whereis_process(unsigned_val(name))) == NULL)  {
	erl_printf(CERR,"%s",tmp_buf);
	return(0);
    }
    /* !!!!!!! Uhhh  */
    if (p->status == P_EXITING || p->status == P_RUNNING) {
	erl_printf(CERR,"%s",tmp_buf);
	return(0);
    }
    bp = new_message_buffer(i*2 + 4);
    hp = bp->mem;
    list = buf_to_intlist(&hp, tmp_buf, i, NIL);
    res = TUPLE3(hp, am_emulator, gl, list);
    queue_message(p, bp, res);
    return 1;
}


/* Ok it's not very clever to rename this function, but i do it any way :-) */
/* safe_malloc() -> safe_alloc() */

void *safe_alloc(len)
uint32 len;
{
    char *buf;

    if ((buf = sys_alloc(len)) == NULL)
	erl_exit(1, "Can't allocate %d bytes of memory\n", len);
    CHECK_MEMORY(buf, len);
    return(buf);
}


void *safe_realloc(ptr, len)
char* ptr; uint32 len;
{
    char *buf;

    if ((buf = sys_realloc(ptr, len)) == NULL)
	erl_exit(1, "Can't reallocate %d bytes of memory\n", len);
    CHECK_MEMORY(buf, len);
    return(buf);
}


/* eq and cmp are written as separate functions a eq is a little faster */

/*
 * Test for equality of two terms.
 * Returns 0 if not equal.
 * Returns 1 if equal.
 * NB this function is recursive, may be problems with stack size!!
 */

int eq(a, b)
uint32 a, b;
{
    int i;
    uint32 *aa;
    uint32 *bb;

    if (a == b) 
	return 1;

    if (not_eq_tags(a,b))
	return 0;
#if defined(BEAM)
    if (is_nil(a) || is_nil(b))
	return (a == b);         /* ??? must be 0 ??? */
#endif

    switch (tag_val_def(a)) {
    case BINARY_DEF: {
	ProcBin *b1, *b2;
	b1 = (ProcBin*) ptr_val(a);
	b2 = (ProcBin*) ptr_val(b);
	if ((b1->size == b2->size) &&
	    (sys_memcmp(b1->bytes,b2->bytes,
			b1->size) == 0))
	    return(1);
	return(0);
    }
    case LIST_DEF:
	aa = ptr_val(a);
	bb = ptr_val(b);
	while (1) {
	    if (!eq(*aa++, *bb++)) return(0);
	    if (*aa == *bb) return(1);
	    if (is_not_list(*aa) || is_not_list(*bb)) return(eq(*aa, *bb));
	    aa = ptr_val(*aa);
	    bb = ptr_val(*bb);
	}
    case TUPLE_DEF:
	aa = ptr_val(a);
	bb = ptr_val(b);
	if (*aa != *bb) return(0); /* different arities */
	i = arityval(*aa);	   /* get the arity*/
	while (i--) {
	    if (eq(*++aa, *++bb) == 0) return(0);
	}
	return(1);
    case BIG_DEF:
	aa = ptr_val(a);  /* get pointer to thing */
	bb = ptr_val(b);
	if (*aa != *bb) return(0);
        i = BIG_ARITY(aa);
	while(i--) {
	    if (*++aa != *++bb) return(0);
	}
	return(1);
    case FLOAT_DEF:
	{
	    FloatDef af;
	    FloatDef bf;

	    GET_DOUBLE(a, af);
	    GET_DOUBLE(b, bf);
	    if (af.fd == bf.fd) return(1);
	    return(0);
	}

    default:
	return(0);
    }
}

/* 
 * Lexically compare two strings of bytes (string s1 length l1 and s2 l2).
 *
 *	s1 < s2	return -1
 *	s1 = s2	return  0
 *	s1 > s2 return +1
 */
static int cmpbytes(s1,l1,s2,l2)
byte *s1,*s2;
int l1,l2;
{
    int i;
    i = 0;
    while((i < l1) && (i < l2)) {
	if (s1[i] < s2[i]) return(-1);
	if (s1[i] > s2[i]) return(1);
	i++;
    }
    if (l1 < l2) return(-1);
    if (l1 > l2) return(1);
    return(0);
}


/*
  Lexically compare objects
  if equal return 0
  if a < b return -1
  if a > b retun 1

  int < nil < atom < list < tuple 

  If both tuple tuple with larger arity > tumple with smaller arity,
  if equal compare term of tuple

  Lists - compare terms of lists

  Integers & floats - as you would expect

  Atoms compare names - see cmpbytes below

*/


#define float_comp(x,y)    (((x)<(y)) ? -1 : (((x)==(y)) ? 0 : 1))

static uint32  big_buf[2];

int cmp(a,b)
uint32 a,b;
{
    uint32* aa;
    uint32* bb;
    int i;
    int j;

    if (a == b)
	return 0;

#if defined(BEAM)
    /* in BEAM a < b when tag(a) > tag(b), (except NIL) */
    if (is_nil(a)) {
      if (tag_val_def(b) == LIST)
	return -1;
      return 1;
    }
    if (is_nil(b)) {
      if (tag_val_def(a) == LIST)
	return 1;
      return -1;
    }
#endif

    if (not_eq_tags(a,b)) {
	FloatDef f1, f2;
	uint32 big;

	switch(NUMBER_CODE(a, b)) {
	case SMALL_BIG:
	    big = small_to_big(signed_val(a), big_buf);
	    return big_comp(big, b);
	    break;
	case SMALL_FLOAT:
	    f1.fd = signed_val(a);
	    GET_DOUBLE(b, f2);
	    return float_comp(f1.fd, f2.fd);
	    break;
	case BIG_SMALL:
	    big = small_to_big(signed_val(b), big_buf);
	    return big_comp(a, big);
	    break;
	case BIG_FLOAT:
	    f1.fd = big_to_double(a);
	    GET_DOUBLE(b, f2);
	    if (!FP_RESULT_OK(f1.fd)) {
	       return big_sign(a) ? -1 : 1;
	    }
	    return float_comp(f1.fd, f2.fd);
	case FLOAT_SMALL:
	    GET_DOUBLE(a, f1);
	    f2.fd = signed_val(b);
	    return float_comp(f1.fd, f2.fd);
	case FLOAT_BIG:
	    GET_DOUBLE(a, f1);
	    f2.fd = big_to_double(b);
	    if (!FP_RESULT_OK(f2.fd)) {
	       return big_sign(b) ? 1 : -1;
	    }
	    return float_comp(f1.fd, f2.fd);
	    break;
	default:
#if defined(BEAM)
	    /* in BEAM a < b when tag(a) > tag(b), (except NIL) */
	    if (tag_val_def(a) > tag_val_def(b))
		return -1;
	    return 1;
#else
	    if (tag_val_def(a) < tag_val_def(b))
		return -1;
	    return 1;
#endif
	}
    }

    /* we now know that tags are the same */
    switch (tag_val_def(a)) {
    case ATOM_DEF:
	return(cmpbytes(atom_tab(unsigned_val(a))->name, atom_tab(unsigned_val(a))->len,
			atom_tab(unsigned_val(b))->name, atom_tab(unsigned_val(b))->len));
    case TUPLE_DEF:
	aa = ptr_val(a);
	bb = ptr_val(b);
	/* compare the arities */
	if (arityval(*aa) < arityval(*bb)) return(-1);
	if (arityval(*aa) > arityval(*bb)) return(1);
	i = arityval(*aa);	/* get the arity*/
	while (i--) {
	    if ((j = cmp(*++aa, *++bb)) != 0) 
		return j;
	}
	return 0;
    case LIST_DEF:
	aa = ptr_val(a);
	bb = ptr_val(b);
	while (1) {
	    if ((j = cmp(*aa++, *bb++)) != 0) 
		return j;
	    if (*aa==*bb)
		return 0;
	    if (is_not_list(*aa) || is_not_list(*bb))
		return cmp(*aa, *bb);
	    aa = ptr_val(*aa);
	    bb = ptr_val(*bb);
	}
    case SMALL_DEF:
	if ((i = (signed_val(a) - signed_val(b))) < 0)
	    return -1;
	else if (i == 0)
	    return 0;
	return 1;
    case FLOAT_DEF:
	{
	    FloatDef af;
	    FloatDef bf; 

	    GET_DOUBLE(a, af);
	    GET_DOUBLE(b, bf);
	    return float_comp(af.fd, bf.fd);
	}

    case BIG_DEF:
	return big_comp(a, b);

    case BINARY_DEF:
	aa = ptr_val(a);
	bb = ptr_val(b);
	if ( ((ProcBin*)aa)->size < ((ProcBin*)bb)->size)
	    return -1;
	else if ( ((ProcBin*)aa)->size > ((ProcBin*)bb)->size)
	    return 1;
	else {
	    i = sys_memcmp(((ProcBin*)aa)->bytes,((ProcBin*)bb)->bytes,
			   ((ProcBin*)aa)->size);
	    if (i < 0)
		return -1;
	    else if (i >  0)
		return 1;
	    else
		return 0;
	}
    case PID_DEF:
	if ((! (get_node(a) == get_node(b))) && /*  different nodes */
	    (get_number(a) == get_number(b)) &&
	    (get_serial(a) == get_serial(b))) { /* equal numbers */
	    
	    uint32 atoma, atomb;
	    i = get_node(a); /* index in atom table */
	    j = get_node(b);
	    atoma = dist_addrs[i].sysname;
	    atomb = dist_addrs[j].sysname;
	    return(cmpbytes(atom_tab(unsigned_val(atoma))->name,
			    atom_tab(unsigned_val(atoma))->len,
			    atom_tab(unsigned_val(atomb))->name,
			    atom_tab(unsigned_val(atomb))->len));
	    

	}
	/* fall through to default branch */
    default: /* this deals with the rest  */
	if (a < b)
	    return -1;
	return 1;
    }
}

Process* pid2proc(pid)
uint32 pid;
{
    Process *rp;
    int i;
    int pix = get_number(pid);

    if (is_not_pid(pid) || 
	(get_node(pid) != THIS_NODE) || 
	(pix >= max_process))
	return NULL;
    i = get_creation(pid);
    if ((i != this_creation) && (i != 0))
	return NULL;

    rp = process_tab[pix];
    if (INVALID_PID(rp, pid))
	return NULL;
    return rp;
}


static int dcount;

/* 
  display an object
  */

static int display1(obj, fd)
uint32 obj; CIO fd;
{
    int i, k;
    uint32 *nobj;
    if (dcount-- <= 0) return(1);
    switch (tag_val_def(obj)) {
    case ATOM_DEF:
	print_atom((int)unsigned_val(obj),fd);
	break;
    case SMALL_DEF:
	erl_printf(fd, "%d", signed_val(obj));
	break;

    case BIG_DEF:
#if defined(BEAM)
        if(is_nil(obj)) {
            erl_printf(fd, "[]");
            break;
        }
#endif
	nobj = ptr_val(obj);
	i = BIG_SIZE(nobj);
	if (BIG_SIGN(nobj))
	    erl_printf(fd, "-#integer(%d) = {", i);
	else
	    erl_printf(fd, "#integer(%d) = {", i);
	erl_printf(fd, "%d", BIG_DIGIT(nobj, 0));
	for (k = 1; k < i; k++)
	    erl_printf(fd, ",%d", BIG_DIGIT(nobj, k));
	erl_putc('}', fd);
	break;
#if defined(JAM)
    case NIL_DEF:
	erl_printf(fd, "[]");
	break;
#endif

    case REFER_DEF:
	erl_printf(fd, "<<%d,%d>>", get_node_reference(obj),
		get_number_reference(obj));
	break;
    case PID_DEF:
	erl_printf(fd, "<%d.%d.%d>",
		get_node(obj),get_number(obj),get_serial(obj));
	break;
    case PORT_DEF:
	erl_printf(fd, "<%d,%d>", get_node_reference(obj),
		get_number_reference(obj));
	break;
    case LIST_DEF:
	erl_putc('[', fd);
	nobj = ptr_val(obj);
	while (1) {
	    if (display1(*nobj++, fd) != 0) return(1);
	    if (is_not_list(*nobj)) break;
	    erl_putc(',',fd);
	    nobj = ptr_val(*nobj);
	}
	if (is_not_nil(*nobj)) {
	    erl_putc('|', fd);
	    if (display1(*nobj, fd) != 0) return(1);
	}
	erl_putc(']', fd);
	break;
    case TUPLE_DEF:
	nobj = ptr_val(obj);	/* pointer to arity */
	i = arityval(*nobj);	/* arity */
	erl_putc('{', fd);
	while (i--) {
	    if (display1(*++nobj,fd) != 0) return(1);
	    if (i >= 1) erl_putc(',',fd);
	}
	erl_putc('}',fd);
	break;
    case FLOAT_DEF: {
	    FloatDef ff;
	    GET_DOUBLE(obj, ff);
	    erl_printf(fd, "%.20e", ff.fd);
	}
	break;
    case BINARY_DEF:
	erl_printf(fd, "#Bin");
	break;
#if defined(BEAM)
    case CP0:
#ifndef NOT_ALIGNED
    case CP4:
    case CP8:
    case CP12:
#endif
        erl_printf(fd, "cp %d", obj);
        break;  
    case BLANK:
        erl_printf(fd, "blank");
        break;  
#endif
    default:
	erl_printf(fd, "unknown object %x", obj);
    }
    return(0);
}


/* display an object on file fd */
/* only used by debugging rountines as Erlang formatting is 
   done in the io module */

void display(obj, fd)
uint32 obj; CIO fd;
{
    dcount = 100000;
    display1(obj, fd);
}


/* as above, but limit the number of items printed */
void ldisplay(obj, fd, count)
uint32 obj; CIO fd; int count;
{
    dcount = count;
    display1(obj, fd);
    if (dcount <= 0) erl_printf(fd, "... "); /* Show that more items exit */
}


/* print a name doing what quoting is necessary */
static void print_name(s, n, fd)
byte *s; int n; CIO fd;
{
    
    int need_quote;
    int pos;
    byte *cpos;
    int c;

    if (n == 0) {
	erl_printf(fd, "''");
	return;
    }

    need_quote = 0;
    cpos = s;
    pos = n - 1;

    c = *cpos++;
    if (!IS_LOWER(c))
	need_quote++;
    else {
	while (pos--) {
	    c = *cpos++;
	    if (!IS_ALNUM(c) && (c != '_')) {
		need_quote++;
		break;
	    }
	}
    }
    cpos = s;
    pos = n;
    if (need_quote)
	erl_putc('\'',fd);
    while(pos--) {
	c = *cpos++;
	switch(c) {
	case '\'': erl_printf(fd, "\\'"); break;
	case '\\': erl_printf(fd, "\\\\"); break;
	case '\n': erl_printf(fd, "\\n"); break;
	case '\f': erl_printf(fd, "\\f"); break;
	case '\t': erl_printf(fd, "\\t"); break;
	case '\r': erl_printf(fd, "\\r"); break;
	case '\b': erl_printf(fd, "\\b"); break;
	case '\v': erl_printf(fd, "\\v"); break;
	default:
	    if (c < 32 || c >= 128)
		erl_printf(fd, "\\%03o", c);
	    else
		erl_putc(c, fd);
	    break;
	}
    }
    if (need_quote) 
	erl_putc('\'',fd);
}

/* print the text of an atom with number i on open file descriptor fd */
void print_atom(i, fd)
int i; CIO fd;
{
    if ((i < 0) || (i >= atom_table_size) ||  (atom_tab(i) == NULL))
	erl_printf(fd, "???");
    print_name(atom_tab(i)->name, atom_tab(i)->len, fd);
    dcount -= atom_tab(i)->len;
}

#define TS_SIZE(p) (((p)->flags & F_TIMESTAMP) ? 5 : 0)
/*
** Patch a timestamp into a tuple
** this is done by adding arity value by one leaving a gap in the heap
** for timestamp tuple. Its required that the tuple is allocated
** just before calling this functions.
** Return the new hp pointer
*/
static uint32* patch_ts(tuple, hp)
uint32 tuple; uint32* hp;
{
    uint32 ms, s, us;
    uint32* ptr = ptr_val(tuple);
    int arity = arityval(*ptr);

    ASSERT((ptr+arity+1) == hp);
    /* add one element to tuple */
    *ptr = make_arityval(arity+1); 

    get_now(&ms, &s, &us);
    /* build the timestamp at hp+1 i.e leave gap for tuple element
    ** insert element last in tuple 
    */
    *hp = TUPLE3(hp+1, make_small(ms),make_small(s), make_small(us));
    return hp+5;  /* 3-tuple + element */
}

/* send a trace message on the form  */
/* {trace, running, Pid, Currentfunction, Timestamp}  or */
/* {trace, running, Pid, Currentfunction} */

void trace_sched(p, what)
Process *p; uint32 what;
{
    Process *tracer;
    uint32 mess;
    uint32 tmp;
    ErlMessageBuffer* bp;
    uint32* hp;

    tracer = process_tab[get_number(p->tracer_proc)];
    if (INVALID_PID(tracer, p->tracer_proc)) {
	p->flags &= ~TRACE_FLAGS;
	p->tracer_proc = NIL;
	return;
    }
    bp = new_message_buffer(9 + TS_SIZE(p));
    hp = bp->mem;

#if defined(BEAM)
    if (p->current == NULL) {
	p->current = find_function_from_pc(p->i);
    }
    if (p->current == NULL) {
        tmp = make_small(0);
    } else {
	tmp = TUPLE3(hp, p->current[0], p->current[1], make_small(p->current[2]));
	hp += 4;
    }
#elif defined(JAM)
    if (p->cc == NULL)
	tmp = make_small(0);
    else {
	byte *tb = p->cc+1;
	tmp = TUPLE3(hp,
		     make_atom(make_16(tb[1], tb[2])),
		     make_atom(make_16(tb[3], tb[4])),
		     make_small(tb[0]));
	hp += 4;
    }
#endif
    mess = TUPLE4(hp, am_trace,p->id, what, tmp);
    hp += 5;
    if (p->flags & F_TIMESTAMP)
	patch_ts(mess, hp);
    queue_message(tracer, bp, mess);
}


void trace_send(p, to, msg)
Process *p; uint32 to; uint32 msg;
{
    Process *rp;
    Process *tracer;
    uint32 operation;
    uint32 mess;
    uint32 sz_msg;
    uint32 sz_to;
    ErlMessageBuffer* bp;
    uint32* hp;
    
    if (!(p->flags & F_TRACE_SEND))
	return;
    operation = am_send;
    tracer = process_tab[get_number(p->tracer_proc)];
    if (INVALID_PID(tracer, p->tracer_proc)) {
	p->flags &= ~TRACE_FLAGS;
	p->tracer_proc = NIL;
	return;
    }
    if (is_pid(to) && (get_node(to) == THIS_NODE)) {
	rp = process_tab[get_number(to)];
	if (INVALID_PID(rp, to)) {
	    char *s = "send_to_non_existing_process";
	    operation = am_atom_put(s, sys_strlen(s));
	}
    }
    sz_msg = size_object(msg);
    sz_to  = size_object(to);
    bp = new_message_buffer(sz_msg + sz_to + 6 + TS_SIZE(p));
    hp = bp->mem;
    to = copy_struct(to, sz_to, &hp, &tracer->mso);
    msg = copy_struct(msg, sz_msg, &hp, &tracer->mso);
    mess = TUPLE5(hp, am_trace, operation, msg, p->id, to);
    hp += 6;
    if (p->flags & F_TIMESTAMP)
	patch_ts(mess, hp);
    queue_message(tracer, bp, mess);
}


/* We know for sure that rp exists */
void trace_receive(rp, msg)
Process *rp; uint32 msg;
{
    Process *tracer;
    uint32 m2;
    uint32 mess;
    uint32 sz_msg;
    ErlMessageBuffer* bp;
    uint32* hp;

    tracer = process_tab[get_number(rp->tracer_proc)];
    if (INVALID_PID(tracer, rp->tracer_proc)) {
	rp->flags &= ~TRACE_FLAGS;
	rp->tracer_proc = NIL;
	return;
    }
    sz_msg = size_object(msg);
    bp = new_message_buffer(sz_msg + 5 + TS_SIZE(rp));
    hp = bp->mem;
    m2 = copy_struct(msg, sz_msg, &hp, &tracer->mso);
    mess = TUPLE4(hp, am_trace, am_receive, rp->id, m2);
    hp += 5;
    if (rp->flags & F_TIMESTAMP)
	patch_ts(mess, hp);
    queue_message(tracer, bp, mess);
}

#ifdef SEQ_TRACE

int seq_trace_update_send(p)
Process *p;
{
    ASSERT((is_tuple(SEQ_TRACE_TOKEN(p)) || is_nil(SEQ_TRACE_TOKEN(p))));
    if ( (p->id == system_seq_tracer) || (SEQ_TRACE_TOKEN(p) == NIL))
	return 0;

    SEQ_TRACE_TOKEN_SENDER(p) = p->id;
    SEQ_TRACE_TOKEN_SERIAL(p) = 
	make_small(++(p -> seq_trace_clock));
    SEQ_TRACE_TOKEN_LASTCNT(p) = 
	make_small(p -> seq_trace_lastcnt);
    return 1;
}


/* Send a sequential trace message to the sequential tracer.
 * p is the caller (which contains the trace token), 
 * msg is the original message, type is an atom (am_send,
 * am_receive, or am_print), and receiver is the receiver of the
 * message.
 *
 * The message to be received by the sequential tracer is:
 * 
 *    TraceMsg = 
 *   {seq_trace, Label, {Type, {Lastcnt, Serial}, Sender, Receiver, Msg} [,Timestamp] }
 *
 */
void seq_trace_output_exit(token, msg, type, receiver, exitfrom)
uint32 token; uint32 msg, type, receiver, exitfrom;
{
    Process* tracer;
    uint32 mess;
    ErlMessageBuffer* bp;
    uint32* hp;
    uint32 sz_label, sz_lastcnt_serial, sz_msg, sz_ts;
    uint32 label, lastcnt_serial, m2;
    uint32 type_atom;
    int sz_exit;

    ASSERT(is_tuple(token) || is_nil(token));
    if ( (SEQ_TRACE_T_SENDER(token) == system_seq_tracer) || (token == NIL))
	return;

    switch (type) {
    case SEQ_TRACE_SEND:
    case SEQ_TRACE_PRINT:
	if (type == SEQ_TRACE_PRINT)
	    type_atom = am_print;
	else
	    type_atom = am_send;
	break;
    case SEQ_TRACE_RECEIVE:
	type_atom = am_receive;
	break;
    default:
	erl_printf(CERR,"invalid type in seq_trace_output %d:\n",type);
	return;
    }

    if (!(unsigned_val(SEQ_TRACE_T_FLAGS(token)) & type))
	/* The flag send, receive or print is not set */
	return;

    if (is_nil(system_seq_tracer))
	return; /* no need to send anything */
    tracer = process_tab[get_number(system_seq_tracer)];
    if (INVALID_PID(tracer, tracer->id) || (receiver == system_seq_tracer)) 
	return; /* no need to send anything */

    sz_label = size_object(SEQ_TRACE_T_LABEL(token));
    sz_lastcnt_serial = 3; /* TUPLE2 */
    sz_msg = size_object(msg);

    sz_ts = ((unsigned_val(SEQ_TRACE_T_FLAGS(token)) & SEQ_TRACE_TIMESTAMP) ? 
	     5 : 0); 
    sz_exit = (exitfrom == NIL) ? 0 : 4; /* create {'EXIT',exitfrom,msg} */
    bp = new_message_buffer(4 /* TUPLE3 */ + sz_ts + 6 /* TUPLE5 */ 
			    + sz_lastcnt_serial + sz_label + sz_msg + sz_exit);
    hp = bp->mem;
    label = copy_struct(SEQ_TRACE_T_LABEL(token), sz_label, &hp, &bp->mso);
    lastcnt_serial = TUPLE2(hp,SEQ_TRACE_T_LASTCNT(token),SEQ_TRACE_T_SERIAL(token));
    hp += 3;
    m2 = copy_struct(msg, sz_msg, &hp, &bp->mso);
    if (sz_exit) {
	m2 = TUPLE3(hp, am_EXIT, exitfrom, m2);
	hp += 4;
    }
    mess = TUPLE5(hp, type_atom, lastcnt_serial, SEQ_TRACE_T_SENDER(token), receiver, m2);
    hp += 6;
    if (sz_ts) {/* timestamp should be included */
	uint32 ms,s,us,ts;
	get_now(&ms, &s, &us);
	ts = TUPLE3(hp, make_small(ms),make_small(s), make_small(us));
	hp += 4;
	mess = TUPLE4(hp, am_seq_trace, label, mess, ts);
    } else {
	mess = TUPLE3(hp, am_seq_trace, label, mess);
    }
    queue_message_tt(tracer, bp, mess, NIL); /* trace_token must be NIL here */
}

#endif

void trace_proc(p, what, data) 
Process *p; uint32 what; uint32 data;
{
    Process *tracer;
    uint32 tmp;
    uint32 mess;
    ErlMessageBuffer* bp;
    uint32* hp;
    uint32 sz_data;

    tracer = process_tab[get_number(p->tracer_proc)];
    if (INVALID_PID(tracer, p->tracer_proc)) {
	p->flags &= ~TRACE_FLAGS;
	p->tracer_proc = NIL;
	return;
    }
    sz_data = size_object(data);
    bp = new_message_buffer(sz_data + 5 + TS_SIZE(p));    
    hp = bp->mem;
    tmp = copy_struct(data, sz_data, &hp, &tracer->mso);
    mess = TUPLE4(hp, am_trace, p->id, what, tmp);
    hp += 5;
    if (p->flags & F_TIMESTAMP)
	patch_ts(mess, hp);
    queue_message(tracer, bp, mess);
}

/* usage of arg1, arg2, arg3 must be checked with arity (varargs) */

void trace_bif(p, bif, arg1, arg2, arg3)
Process* p; BifFunction bif; uint32 arg1; uint32 arg2; uint32 arg3;
{
    uint32 sz_arg1 = NIL;	/* shut up warnings about use before set */
    uint32 sz_arg2 = NIL;
    uint32 sz_arg3 = NIL;
    uint32 size;
    uint32 name;
    int arity;
    int i;
    ErlMessageBuffer* bp;
    Process* tracer;
    uint32* hp;
    uint32 mess;

    if (!(p->flags & F_TRACE_BIFS))
	return;

    /* Beam load function pointers */
    for (i = 0; i < BIF_SIZE; i++) {
	if (bif == bif_table[i].f)
	    break;
    }
    ASSERT(i < BIF_SIZE);
    name = *bif_table[i].name;    
    arity = bif_table[i].arity;    

    tracer = process_tab[get_number(p->tracer_proc)];
    if (INVALID_PID(tracer, p->tracer_proc)) {
	p->flags &= ~TRACE_FLAGS;
	p->tracer_proc = NIL;
	return;
    }
    size = 0;
    switch(arity) {
    case 3: sz_arg3 = size_object(arg3); size += sz_arg3;
    case 2: sz_arg2 = size_object(arg2); size += sz_arg2;
    case 1: sz_arg1 = size_object(arg1); size += sz_arg1;
    }

    bp = new_message_buffer(size + arity*2 + 9 + TS_SIZE(p));
    hp = bp->mem;

    mess = NIL;
    switch(arity) {
    case 3: 
	arg3 = copy_struct(arg3, sz_arg3, &hp, &tracer->mso);
	mess = CONS(hp, arg3, mess);
	hp += 2;
    case 2:
	arg2 = copy_struct(arg2, sz_arg2, &hp, &tracer->mso);
	mess = CONS(hp, arg2, mess); 
	hp += 2;
    case 1: 
	arg1 = copy_struct(arg1, sz_arg1, &hp, &tracer->mso);
	mess = CONS(hp, arg1, mess); 
	hp += 2;
    }
    mess = TUPLE3(hp, am_erlang, name, mess);
    hp += 4;
    mess = TUPLE4(hp, am_trace, p->id, am_call, mess);
    hp += 5;
    if (p->flags & F_TIMESTAMP)
	patch_ts(mess, hp);
    queue_message(tracer, bp, mess);
}

void
trace_gc(Process *p, uint32 what)
{
    Process* tracer;
    ErlMessageBuffer* bp;
    uint32* hp;
    uint32 msg = NIL;
    uint32 tuple;

#define CONS_PAIR(key, val) \
    tuple = TUPLE2(hp, key, val); hp += 3; \
    msg = CONS(hp, tuple, msg); hp += 2

    tracer = process_tab[get_number(p->tracer_proc)];
    if (INVALID_PID(tracer, p->tracer_proc)) {
	p->flags &= ~TRACE_FLAGS;
	p->tracer_proc = NIL;
	return;
    }
    bp = new_message_buffer(128);
    hp = bp->mem;

    CONS_PAIR(am_heap_size, make_small(p->htop - p->heap));
    CONS_PAIR(am_old_heap_size, make_small(p->old_htop - p->old_heap));
#if defined(BEAM)
    CONS_PAIR(am_stack_size, make_small(p->stack - p->stop));
#elif defined(JAM)
    CONS_PAIR(am_stack_size, make_small(p->stop - p->stack));
#endif
    CONS_PAIR(am_recent_size, make_small(p->high_water - p->low_water));
    CONS_PAIR(am_mbuf_size, make_small(p->mbuf_sz));

    msg = TUPLE4(hp, am_trace, p->id, what, msg);
    hp += 5;
    if (p->flags & F_TIMESTAMP)
	patch_ts(msg, hp);
    queue_message(tracer, bp, msg);
#undef CONS_PAIR
}

/* 
 *  member(X,Y)
 *  returns 0 if X is a member of list Y
 *  returns 1 if X is not a member of list Y
 *  returns 2 if Y is not a list or is a badly formed list
 */


int member(x,y)
uint32 x,y;
{
    uint32 *z;
    if (is_nil(y)) return(1); /* empty list */
    if (is_not_list(y)) return(2); /* bad argument */
    z = ptr_val(y);
    for (;;) {
	if (eq(*z, x)) return(0);
	if (is_nil(*(z + 1))) return(1); /* end of list */
	if (is_not_list(*(z + 1))) return(2); /* badly formed list */
	z = ptr_val(*(z + 1));
    }
}



/* routines for converting no aligned bytes arrays to floats (double)
   and vice versa */
double bytes_to_float(b)
byte *b;
{
    union {
	double fd;
	byte fb[sizeof(double)];
    } f;
    int i;
    for (i = 0; i < sizeof(double); i++) f.fb[i] = *b++;
    return(f.fd);
}


void float_to_bytes(b,fl)
byte *b; double fl;
{
    union {
	double fd;
	byte fb[sizeof(double)];
    } f;
    int i;

    f.fd = fl;
    for (i = 0; i < sizeof(double); i++)
	*b++ = f.fb[i];
}


void bin_write(fp,buf,sz)
CIO fp; byte* buf;
int sz;
{
    int i;

    for (i=0;i<sz;i++) {
	if (IS_DIGIT(buf[i]))
	    erl_printf(fp, "%d,", buf[i]);
	else if (IS_PRINT(buf[i])) {
	    erl_putc(buf[i],fp);
	    erl_putc(',',fp);
	}
	else
	    erl_printf(fp,"%d,", buf[i]);
    }
    erl_putc('\n',fp);
}

/* Fill buf with the contents of bytelist list 
   return number of chars in list or -1 for error */

int intlist_to_buf(list,buf,len)
uint32 list;
byte *buf;
int len;
{
    uint32 *listptr;
    int sz = 0;

    if (is_nil(list)) 
	return 0;
    if (is_not_list(list))
	return -1;
    listptr = ptr_val(list);

    while (sz < len) {
	if (!is_byte(*listptr)) 
	    return -1;
	buf[sz++] = unsigned_val(*listptr);
	if (is_nil(*(listptr + 1)))
	    return(sz);
	if (is_not_list(*(listptr + 1))) 
	    return -1;
	listptr = ptr_val(*(listptr + 1));
    }
    return -1;			/* not enough space */
}

/*
** Convert an integer to a byte list buf must have at least 12 bytes avaiable
** return pointer to converted stuff (need not to be at start of buf!)
*/
char* int_to_buf(n, buf)
int n; char* buf;
{
    char* p = buf+11;
    int sign = 0;

    *p-- = '\0'; /* null terminate */
    if (n == 0)
	*p-- = '0';
    else if (n < 0) {
	sign = 1;
	n = -n;
    }

    while (n != 0) {
	*p-- = (n % 10) + '0';
	n /= 10;
    }
    if (sign)
	*p-- = '-';
    return p+1;
}

/* Build a list of integers in some safe memory area
** Memory must be pre allocated prio call 2*len in size
** hp is a pointer to the "heap" pointer on return
** this pointer is updated to point after the list
*/

uint32 buf_to_intlist(hpp, buf, len, tail)
uint32** hpp; byte *buf; int len; uint32 tail;
{
    uint32* hp = *hpp;

    buf += (len-1);
    while(len > 0) {
	tail = CONS(hp, make_small((byte)*buf), tail);
	hp += 2;
	buf--;
	len--;
    }
    *hpp = hp;
    return tail;
}

/*
** write io list in to a buffer.
**
** A iolist is defined as:
**
** iohead ::= Binary
**        |   Byte (i.e integer in range [0..255]
**        |   iolist
**        ;
**
** iotail ::= []
**        |   Binary  (added by tony)
**        |   iolist
**        ;
**
** iolist ::= []
**        |   Binary
**        |   [ iohead | iotail]
**        ;
** 
** return (char*) 0 on overflow, 
**        (char*) 1 on error
**        pointer continue buffer otherwise
** 
*/
#define IOL_OVERFLOW ((char*) 0)
#define IOL_ERROR    ((char*) 1)

static char* iol_to_buf(list, ptr, maxptr)
uint32 list; char* ptr; char* maxptr;
{
    int i;

    while (is_list(list)) {
	uint32* cons = ptr_val(list);
	uint32 obj = CAR(cons);

	if (is_byte(obj)) {
	    if (ptr >= maxptr)
		return IOL_OVERFLOW;
	    *ptr++ = unsigned_val(obj);
	}
	else if (is_list(obj)) {
	    if ((ptr = iol_to_buf(obj, ptr, maxptr)) <= IOL_ERROR)
		return ptr;
	}
	else if (is_binary(obj)) {
	    ProcBin *pb = (ProcBin*) ptr_val(obj);
	    i = pb->size;
	    if (ptr + i >= maxptr)
		return IOL_OVERFLOW;
	    sys_memcpy(ptr, pb->bytes, i);
	    ptr += i;
	}
	else if (is_nil(obj))
	    ;
	else 
	    return IOL_ERROR;
	list = CDR(cons);
    }

    if (is_nil(list))
	return ptr;
    else if (is_binary(list)) {
	ProcBin *pb = (ProcBin*) ptr_val(list);
	i = pb->size;
	if (ptr + i >= maxptr)
	    return IOL_OVERFLOW;
	sys_memcpy(ptr, pb->bytes, i);
	return ptr + i;
    }
    return IOL_ERROR;
}


/* Fills a deep list of chars and binaries int buf */
/* Used by the port write routines                 */
/* Return 0 on success,                            */
/*        -1 on overflow                           */
/*        -2 on type error                         */

int io_list_to_buf(list, buf, cpos, max)
uint32 list; char* buf; int* cpos; int max;
{
    char* ptr = buf + *cpos;  /* start position */
    char* end = buf + max;    /* end position */
    char* cur;
    
    if ((cur = iol_to_buf(list, ptr, end)) > IOL_ERROR) {
	*cpos += (cur - ptr);
	return 0;
    }
    else if (cur == IOL_OVERFLOW)
	return -1;
    else
	return -2;
}


int io_list_len(list)
uint32 list;
{
    int len = 0;
    int i;

    while (is_list(list)) {
	uint32* cons = ptr_val(list);
	uint32 obj = CAR(cons);

	if (is_byte(obj))
	    len++;
	else if (is_list(obj)) {
	    if ((i = io_list_len(obj)) < 0)
		return i;
	    len += i;
	}
	else if (is_binary(obj))
	    len += ((ProcBin*) ptr_val(obj))->size;
	else if (is_nil(obj))
	    ;
	else 
	    return(-1);
	list = CDR(cons);
    }
    if (is_nil(list))
	return len;
    else if (is_binary(list))
	return len + ((ProcBin*) ptr_val(list))->size;
    else
	return -1;
}

/* return 0 if item is not a non-empty flat list of bytes */

int is_string(list)
uint32 list;
{
    int len = 0;

    while(is_list(list)) {
	uint32* consp = ptr_val(list);
	uint32  hd = CAR(consp);

	if (!is_byte(hd))
	    return 0;
	len++;
	list = CDR(consp);
    }
    if (is_nil(list))
	return len;
    return 0;
}


int
do_load(group_leader, mod, code, size)
    uint32 group_leader;	/* Group leader or 0 if none. */
    uint32 mod;			/* Module name as an atom. */
    byte* code;			/* Points to the code to load */
    int size;			/* Size of code to load. */
{
    DriverBinary* bin;
    int result;

    if ((bin = (DriverBinary *) gzinflate_buffer(code, size)) == NULL) {
	return -1;
    }
    result = bin_load(group_leader, mod, bin->orig_bytes, bin->orig_size);
    driver_free_binary(bin);
    return result;
}


#ifdef BEAM
void export_function(module, func, arity, lbl)
uint32 module; uint32 func; int arity; uint32 *lbl;
{
    int index = export_put(unsigned_val(module),unsigned_val(func),arity);
    export_list(index)->address = lbl;
}

Export* import_function(module, func, arity)
uint32 module; uint32 func;  int arity;
{
    int index = export_put(unsigned_val(module),unsigned_val(func),arity);
    return export_list(index);
}
#endif


/* debug routine which checks for a valid object */
int check_struct(obj)
uint32 obj;
{
    uint32 i, arity, *nobj;
    switch (tag_val_def(obj)) {
    case ATOM_DEF:
	if (unsigned_val(obj) >= atom_table_size)
	    return 0;
	return 1;
#if defined(JAM)
    case NIL_DEF:
#endif
    case SMALL_DEF:
    case PID_DEF:
    case REFER_DEF:
    case PORT_DEF:
	return 0;
    case BINARY_DEF:
	if (((ProcBin*) ptr_val(obj))->mark == 1)
	    return 1;
	if (((ProcBin*) ptr_val(obj))->size > 0xfffffff)
	    return 1;
	if (((ProcBin*) ptr_val(obj))->bytes == 0)
	    return 1;
	return 0;
    case BIG_DEF:
#if defined(BEAM)
	if(is_nil(obj))
	    return(0);
#endif
	nobj = ptr_val(obj);
	if (is_not_thing(*nobj)) return(1);
	/* A bignum on the heap may never fit into a small !!! */
	if ((arity = BIG_ARITY(nobj)) == 1) {
	    uint32 d =  BIG_DIGIT(nobj,0)+D_BASE*BIG_DIGIT(nobj,1);

	    if (IS_USMALL(BIG_SIGN(nobj), d))
		return(1);
	}
	return(0);

    case FLOAT_DEF:
	if (*ptr_val(obj) != make_thing(2)) 
	    return(1);
	return(0); /* no way to check binary data */
    case LIST_DEF:
	nobj = ptr_val(obj);
	while (1) {
	    if (check_struct(*nobj++) != 0) return(1);
	    if (is_not_list(*nobj)) return(check_struct(*nobj));
	    nobj = ptr_val(*nobj);
	}
    case TUPLE_DEF:
	if (is_not_arity_value(*(ptr_val(obj)))) return(1);
	arity = arityval(*(ptr_val(obj)));
	for (i = 0; i < arity; i++)
	    if (check_struct(*(ptr_val(obj) + i + 1)) != 0) return(1);
	return(0);
    default:
	return(1);
    }
}

#ifdef INSTRUMENT
typedef union most_strict {
    double x;
    long y;
} Most_strict;

typedef struct mem_link
{
   struct mem_link *prev, *next;
   unsigned long size;
   int type;
   Most_strict align;
} mem_link;

int alloc_who = -1;

static mem_link *mem_anchor = NULL;	/* better to set this in erl_init */

static void link_in(l, size)
mem_link *l;
unsigned size;
{
   l->next = mem_anchor;
   if (mem_anchor != NULL)
      mem_anchor->prev = l;
   l->prev = NULL;
   l->size = size;
   if (l->type == -1)
      l->type = alloc_who;
   alloc_who = -1;
   mem_anchor = l;
}

static void link_out(l)
mem_link *l;
{
   mem_link *prev, *next;

   prev = l->prev;
   next = l->next;
   if (prev != NULL)
      prev->next = next;
   else
      mem_anchor = next;

   if (next != NULL)
      next->prev = prev;
}

void* sys_alloc(size)
unsigned int size;
{
   char *p;
   mem_link *l;

   p = sys_alloc2(size + sizeof(mem_link));
   if (p == NULL)
      return NULL;

   l = (mem_link *) p;
   l->type = -1;
   link_in(l, size);

   return (void *) (p + sizeof(mem_link));
}

void* sys_realloc(ptr, size)
void* ptr; unsigned int size;
{
   char *p, *new_p;
   mem_link *l;
   unsigned old_size;

   p = ((char *) ptr) - sizeof(mem_link);

   l = (mem_link *) p;
   link_out(l);
   old_size = l->size;

   new_p = sys_realloc2(p, size + sizeof(mem_link));
   if (new_p == NULL)
      return NULL;

   l = (mem_link *) new_p;
   link_in(l, size);

   return (void *) (new_p + sizeof(mem_link));
}

void sys_free(ptr)
void* ptr;
{
   mem_link *l;
   char *p;

   p = ((char *) ptr) - sizeof(mem_link);

   l = (mem_link *) p;
   link_out(l);

   sys_free2(p);
}

int dump_memory_data(name)
const char *name;
{
   mem_link *l;
   FILE *f;

   f = fopen(name, "w");
   if (f == NULL)
      return 0;

   l = mem_anchor;

   while (l != NULL)
   {
      fprintf(f, "{%d, %lu, %lu}.\n",
		   l->type,
		   ((unsigned long) l) + sizeof(mem_link),
		   l->size);
      l = l->next;
   }

   fclose(f);
   return 1;
}

uint32 collect_memory(process)
Process *process;
{
   uint32 list, tup;
   uint32 *hp, *end_hp;
   mem_link *l;
   uint32 need;

   list = NIL;

   need = 0;
   l = mem_anchor;
   while (l != NULL)
   {
      need += 4+2;
      l = l->next;
   }

   /* The "alloc" operation itself is likely to add to the list,
      so add a little. */
   need += 20;

   hp = HAlloc(process, need);
   end_hp = hp + need;

   l = mem_anchor;
   while (l != NULL)
   {
      if (hp >= end_hp - 6)
	 break;

      tup = TUPLE3(hp,
		   make_small(l->type),
		   make_small(((int) l) + sizeof(mem_link)),
		   make_small(l->size));
      hp += 4;
      list = CONS(hp, tup, list);
      hp += 2;

      l = l->next;
   }

   while (l != NULL)
   {
      hp = HAlloc(process, 4);
      tup = TUPLE3(hp,
		   make_small(l->type),
		   make_small(((int) l) + sizeof(mem_link)),
		   make_small(l->size));
      hp = HAlloc(process, 2);
      list = CONS(hp, tup, list);

      l = l->next;
   }

   return list;
}

void alloc_from(from)
int from;
{
   if (alloc_who == -1)
      alloc_who = from;
}

#else

void* sys_alloc(size)
unsigned int size;
{
   return sys_alloc2(size);
}

void* sys_realloc(ptr, size)
void* ptr; unsigned int size;
{
   return sys_realloc2(ptr, size);
}

void sys_free(ptr)
void* ptr;
{
   sys_free2(ptr);
}
#endif

#ifdef DEBUG
/*
 * Handy functions when using a debugger - don't use in the code!
 */

void upp(buf,sz)
byte* buf;
int sz;
{
    bin_write(CERR,buf,sz);
}

/* Print an atom as an uint32 or just as an index */     
void pat(a)
uint32 a;
{
    upp(atom_tab(unsigned_val(a))->name,
	atom_tab(unsigned_val(a))->len);
}


void pinfo()
{
    process_info(COUT);
}


void pp(p)
Process *p;
{
    print_process_info(p,CERR);
}
    
void ppi(p)
uint32 p;
{
    pp(process_tab[get_number(p)]);
}

void td(x) 
uint32 x;
{
    display(x, CERR);
    erl_putc('\n', CERR);
}

#if defined(JAM)


/* print BIF arguments */
void pba(p, arity)
Process *p; int arity;
{
    int i;
    for (i=1; i<= arity; i++) {
	erl_printf(CERR,"Arg %d:",i);
	ldisplay((*(p->stop - arity + i - 1)), CERR, 1000);
	erl_putc('\n', CERR);
    }
}

/* print stack to stop */
void ps(p, stop)
Process* p; uint32* stop;
{
    uint32* sp = p->stack;

    while(sp < stop) {
	if (is_frame(*sp)) {
	    erl_printf(COUT,"%08lx: FP = %08lx\n", 
		       (uint32)(sp+FRAME_FP), frame_val(sp[FRAME_FP]));
	    erl_printf(COUT,"%08lx: AP = %08lx\n",
		       (uint32)(sp+FRAME_AP), frame_val(sp[FRAME_AP]));
	    erl_printf(COUT,"%08lx: PC   = %08lx\n",
		       (uint32)(sp), (byte*) sp[FRAME_PC]);
	    if (sp[FRAME_CC] != 0) {
		byte* cc = ((byte*) sp[FRAME_CC]) + 1;
		erl_printf(COUT,"%08lx: CC   = %08lx, FUN = ",
			   (uint32)(sp+FRAME_CC), cc);
		print_atom(make_16(cc[1], cc[2]), COUT);
		erl_printf(COUT, ":");
		print_atom(make_16(cc[3], cc[4]), COUT);
		erl_printf(COUT,"/%d\n", cc[0]);
	    }
	    else
		erl_printf(COUT,"%08lx: CC   = %08lx\n",
			   (uint32)(sp+FRAME_CC), 0);
	    sp += FRAME_SIZE;
	}
	else if (is_catch(*sp)) {
	    erl_printf(COUT,"%08lx: CATCH = %08lx\n",
		       (uint32)(sp+CATCH_PTR), frame_val(sp[CATCH_PTR]));
	    erl_printf(COUT,"%08lx: PC = %08lx\n",
		       (uint32)(sp+CATCH_PC), (byte*) sp[CATCH_PC]);
	    sp += CATCH_SIZE;
	}
	else
	{
	    erl_printf(COUT,"%08lx: ", (uint32) sp);
	    ldisplay(*sp, COUT, 75);
	    erl_putc('\n', COUT);
	    sp++;
	}
    }
}

#else

void ps(p, stop)
Process* p; uint32* stop;
{
    uint32* sp = p->stack-1;

    if (stop <= p->htop) {
	stop = p->htop + 1;
    }

    while(sp >= stop) {
	erl_printf(COUT,"%08lx: ", (uint32) sp);
	ldisplay(*sp, COUT, 75);
	erl_putc('\r', COUT);
	erl_putc('\n', COUT);
	sp--;
    }
}

#endif
#endif
