/* ``The contents of this file are subject to the Erlang Public License,
 * Version 1.0, (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.erlang.org/EPL1_0.txt
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 * 
 * The Original Code is Erlang-4.7.3, December, 1998.
 * 
 * The Initial Developer of the Original Code is Ericsson Telecom
 * AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
 * Telecom AB. All Rights Reserved.
 * 
 * Contributor(s): ______________________________________.''
 */
/*
** File: mkam.c
** Author: Tony Rogvall
**
** Generate files am.c and am.h
** 
** usage mkam file ... 
**
** The files should contain lines with one atom on each row
** the atoms should look like [a-z][a-zA-Z0-9_]*
*/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#define MAX_ATOMS 1000
#define MAX_ATOM_LEN 256

static char *am[MAX_ATOMS];	/* All atoms read from all files. */
static char atom[MAX_ATOM_LEN+1]; /* The current atom read from the file. */

static char* outdir = NULL;	/* Where to place the output files. */

static FILE* create_file();

static int cmp_string(a, b)
char** a; char** b;
{
    return strcmp(*a, *b);
}

void main(argc, argv)
int argc;
char** argv;
{
    int i;
    int j;
    int k;
    int t;
    FILE* f;

    /*
     * Check for for options.
     */

    while (argc > 1 && argv[1][0] == '-') {
      switch (argv[1][1]) {
      case 'd':
	outdir = argv[2];
	argc--, argv++;
	break;
      default:
	fprintf(stderr, "mkam: illegal option: %s\n", argv[1]);
	exit(1);
      }
      argc--, argv++;
    }

    /*
     * Read atoms from each of the files mentioned on the command line.
     */

    j = 0;
    for (i = 1; i < argc; i++) {
	if ((f = fopen(argv[i], "r")) == NULL) {
	    fprintf(stderr, "mkam: could not open %s\n", argv[i]);
	    exit(1);
	}
	while(fgets(atom, MAX_ATOM_LEN, f)) {
	    int n;
	    if (j >= MAX_ATOMS) {
		fprintf(stderr, "mkam: too many atoms MAX = %d\n", MAX_ATOMS);
		exit(1);
	    }
	    n = strlen(atom);
	    if (atom[n-1] == '\n') {
		atom[n-1] = '\0';
		n--;
	    }
	    am[j] = (char*) malloc(n+1);
	    strcpy(am[j], atom);
	    j++;
	}
	fclose(f);
    }

    /*
     * Sort the atoms and remove any duplicates.
     */

    qsort(am, j, sizeof(char*), cmp_string);
    k = 0;
    t = 0;
    while(t < j) {
	int tn = t + 1;

	while(tn < j && strcmp(am[t], am[tn]) == 0) {
	    tn++;
	}
	am[k] = am[t];
	k++;
	t = tn;
    }

    /*
     * Generate the am.c file.
     */

    f = create_file("am.c");

    fprintf(f, "/* File: am.c -- generated by mkam. */\n");
    fprintf(f, "#include \"sys.h\"\n");
    fprintf(f, "#include \"am.h\"\n\n");
    fprintf(f, "uint32 am_ix[AM_SIZE];\n\n");
    fprintf(f, "const char* am_nm[AM_SIZE+1] = {\n");

    for (i = 0; i < k; i++)
	fprintf(f, "  \"%s\",\n", am[i]);
    fprintf(f, "  NULL\n};\n");
    
    fclose(f);

    /*
     * Generate the am.h file.
     */
    
    f = create_file("am.h");

    fprintf(f, "/* File: am.h This file was generated by mkam */\n");
    fprintf(f, "#ifndef __AM_H__\n");
    fprintf(f, "#define __AM_H__\n\n");
    fprintf(f, "extern uint32 am_ix[];\n");
    fprintf(f, "extern const char* am_nm[];\n\n");

    for (i = 0; i < k; i++)
	fprintf(f, "#define am_%s am_ix[%d]\n", am[i], i);

    fprintf(f, "#define AM_SIZE %d\n", k);
    fprintf(f, "#endif\n");
    fclose(f);
    exit(0);
}

static FILE*
create_file(name)
char* name;
{
  FILE* file;
  char filename[1024];

  if (outdir)
    sprintf(filename, "%s/%s", outdir, name);
  else
    strcpy(filename, name);

  if ((file = fopen(filename, "w")) == NULL) {
    fprintf(stderr, "mkam: could not open %s for writing\n", filename);
    exit(1);
  }
  return file;
}
