/* ``The contents of this file are subject to the Erlang Public License,
 * Version 1.0, (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.erlang.org/EPL1_0.txt
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 * 
 * The Original Code is Erlang-4.7.3, December, 1998.
 * 
 * The Initial Developer of the Original Code is Ericsson Telecom
 * AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
 * Telecom AB. All Rights Reserved.
 * 
 * Contributor(s): ______________________________________.''
 */
/*  Implementation of the erlang external format 
 *  Author: Claes Wikstrom (originally Mike Williams) 
 *
 *  And a nice cache mechanism which is used just to send a
 *  index indicating a specific atom to a remote node instead of the
 *  entire atom.
 */

#include "sys.h"
#include "config.h"
#include "global.h"
#include "erl_process.h"
#include "external.h"
#include "big.h"


#define MAX_STRING_LEN 0xffff

extern int dist_buf_size;
extern byte *dist_buf;

static FUNCTION(byte*, enc_term, (int, uint32, byte*));
static FUNCTION(byte*, enc_atom, (int, uint32, byte*));
static FUNCTION(byte*, enc_pid, (int, uint32, byte*));
static FUNCTION(byte*, dec_term, (int, uint32**, byte*, ProcBin**,uint32*));
static FUNCTION(byte*, dec_atom, (int, byte*, uint32*));
static FUNCTION(byte*, dec_pid, (int, byte*, uint32*));
static FUNCTION(byte*, dec_hashed_atom, (int, byte*, uint32*));
static FUNCTION(byte*, dec_and_insert_hashed_atom, (int,byte*,uint32*));

static FUNCTION(int, decode_size2, (byte**, byte*, int*));
static FUNCTION(uint32, encode_size_struct2, (uint32));

static FUNCTION(int, to_external2, (int, uint32, byte**));


/* This function fills ext with the external format of atom 
   if it's an old atom we just supply an index, otherewise
   we insert the index _and_ the entire atom , This way the receiving side
   does not have to perform an hash on the etom to locate it, and
   we save a lot of space on the wire */


static byte* enc_atom(slot,atom,ep)
int slot; uint32 atom; byte *ep;
{
    uint32* ce;
    uint32 ix;
    int i,j;

    ASSERT(is_atom(atom));

    /*  for term_to_words and initial setup msg */     
    if ((slot < 0) || (dist_addrs[slot].cache == NULL)) { 
	i = unsigned_val(atom);
	j = atom_tab(i)->len;
	*ep++ = ATOM_EXT;
	put_int16(j, ep);  /* XXX reduce this to 8 bit in the future */
	ep += 2;
	sys_memcpy((char *) ep, (char*)atom_tab(i)->name, (int) j);
	ep += j;
	return ep;
    }
    ix = unsigned_val(atom) % MAXINDX;
    ce = &dist_addrs[slot].cache->out_arr[ix];
    
    if (*ce == 0) {   /* FREE CACHE ENTRY */
	/* Ok insert the atom in the cache */
	i = unsigned_val(atom);
	j = atom_tab(i)->len;
	*ep++ = NEW_CACHE;
	*ep++ = ix;
	put_int16(j, ep);
	ep += 2;
	sys_memcpy((char *) ep, (char*) atom_tab(i)->name, (int) j);
	*ce = atom;
	return ep + j;
    }
    else  { /*   case CACHE_OCCUPIED: */
	if (*ce == atom) {
	    *ep++ =  CACHED_ATOM;
	    *ep++ = ix;
	    return ep;
	}
	/* we try to encode an atom that hashed into an already
	   occupied slot ... owerwrite */
	i = unsigned_val(atom);
	j = atom_tab(i)->len;
	*ep++ = NEW_CACHE;
	*ep++ = ix;
	put_int16(j, ep);
	ep += 2;
	sys_memcpy((char *) ep, (char*) atom_tab(i)->name, (int) j);
	*ce = atom;
	return ep + j;
    }
}

static byte* enc_pid(slot, pid, ep)
int slot; uint32 pid; byte* ep;
{
    uint32 m;
    uint32 j;
    uint32 creation;

    *ep++ = PID_EXT;
    m = get_node(pid);
    /* insert  atom here containing host and sysname  */
    ep = enc_atom(slot, dist_addrs[m].sysname, ep);

    /* two bytes for each number and serial */
    j = get_number(pid);
    put_int32(j, ep);
    ep += 4;
    j = get_serial(pid);
    put_int32(j, ep);
    ep += 4;

    /*
     * Setting the creation is tricky.  For foreign node names, we always
     * retain the creation.  If the nodename is the same as our own,
     * we must retain the creation of the pid if it has one (it must be
     * a previous incarnation of the current node, which another node has
     * passed to us).  Otherwise, if the creation is 0, we set the
     * creation to the current creation.
     */

    creation = get_creation(pid);
    if (m == THIS_NODE && creation == 0) {
	*ep++ = this_creation;
    } else {
	*ep++ = get_creation(pid);
    }
    return ep;
}



static byte* dec_hashed_atom(slot,ep,objp)
int slot; byte *ep; uint32* objp;
{
    *objp = dist_addrs[slot].cache->in_arr[get_int8(ep)];
    return ep+1;
}


static byte* dec_and_insert_hashed_atom(slot,ep,objp) 
int slot; byte *ep; uint32* objp;
{
    uint32 ix;
    uint32 len;

    ix = get_int8(ep);
    ep++;
    len = get_int16(ep);
    ep += 2;
    *objp = dist_addrs[slot].cache->in_arr[ix] = am_atom_put((char*)ep,len);
    return ep + len;
}

/* Expect an atom (cached or new) */
static byte* dec_atom(slot, ep, objp)
int slot; byte* ep; uint32* objp;
{
    uint32 len;

    switch (*ep++) {
    case ATOM_EXT:
	len = get_int16(ep),
	ep += 2;
	*objp = am_atom_put((char*)ep, len);
	return ep + len;

    case NEW_CACHE:
	return dec_and_insert_hashed_atom(slot,ep,objp); 

    case CACHED_ATOM:
	return dec_hashed_atom(slot,ep,objp);

    default:
	return NULL;
    }
}

static byte* dec_pid(slot, ep, objp)
int slot; byte* ep; uint32* objp;
{
    uint32 i;
    uint32 j;
    uint32 k;
    uint32 n;
    int si;

    /* eat first atom */
    if ((ep = dec_atom(slot, ep, &i)) == NULL)
	return NULL;
    if ((si = find_or_insert_dist_slot(i)) < 0)
	return NULL;
    j = get_int32(ep);
    ep += 4;
    k = get_int32(ep);
    ep += 4;
    n = get_int8(ep);
    ep += 1;

    if (si == THIS_NODE)   /* local pid */
	if (n == this_creation)  /* This incarnation */
	    n = ORIG_CREATION;
    if (j >= MAX_PROCESS) /* CHECK max_process for si==0 ? */
	return NULL;
    if (k >= MAX_SERIAL)
	return NULL;
    *objp = make_pid3(k,si,j,n);
    return ep;
}


/* This function is written in this strange way because in dist.c
   we call it twice for some messages, First we insert a control message 
   in the buffer , and then we insert the actual message in the buffer
   immediataly following the control message. If the real (2) message
   makes the buffer owerflow, we must not destroy the work we have allready
   done, i.e we must not forget to copy the encoding of the 
   controll message as well, (If there is one)
   
   The buffer dist_buf initially points to tmp_buf, So if we write
   an application where the encoding of all distributed messages are <
   TMP_BUF_SIZE, all remote messages will be encoded into tmp_buf,
   and no additional buffer allocation is necessary.
   However, once a message does not fit into tmp_buf we 
   allocate a larger buffer and set dist_buf to poiny to that buffer.
   We never go back to tmp_buf situation again.

   We don't have the energy to describe the external format in words here.
   The code explains itself :-). However, All external encodings
   are predeeded with the special character VERSION_MAGIC, which makes
   it possible do distinguish encodings from incompatible erlang systems.
   Every time the external format is changed, This VERSION_MAGIC shall be 
   incremented.

*/



static byte* enc_term(slot, obj, ep)
int slot; uint32 obj; byte* ep;
{
    int val;
    uint32 n;
    uint32 i;
    uint32 j;
    uint32 m;
    uint32* ptr;
    ProcBin *bp;
    FloatDef f;
    byte* back;

    switch(tag_val_def(obj)) {
    case ATOM_DEF:
	return enc_atom(slot,obj,ep);

    case SMALL_DEF:
	if (is_byte(obj)) {
	    val = unsigned_val(obj);
	    *ep++ = SMALL_INTEGER_EXT;
	    put_int8(val, ep);
	    return ep + 1;
	}
	else {
	    val = signed_val(obj);
	    *ep++ = INTEGER_EXT;
	    put_int32(val, ep);
	    return ep + 4;
	}

    case BIG_DEF:
#if defined(BEAM)
	if (is_nil(obj)) {
            *ep++ = NIL_EXT;   /* soley empty lists */
	    return ep;
        }
#endif	
	if ((n = big_bytes(obj)) < 256) {
	    *ep++ = SMALL_BIG_EXT;
	    put_int8(n, ep);
	    ep += 1;
	}
	else {
	    *ep++ = LARGE_BIG_EXT;
	    put_int32(n, ep);
	    ep += 4;
	}
	*ep++ = big_sign(obj);
	return big_to_bytes(obj, ep);

    case PID_DEF:
	return enc_pid(slot, obj, ep);

    case REFER_DEF:
	*ep++ =  REFERENCE_EXT;
	m = get_node_reference(obj);
	ep = enc_atom(slot,dist_addrs[m].sysname,ep);
	j = get_number_reference(obj);
	put_int32(j, ep);
	ep += 4;
	if (m == THIS_NODE)     /* A local refer */
	    *ep++ = this_creation;
	else   
	    *ep++ = get_creation(obj);
	return ep;

    case PORT_DEF:
	*ep++ = PORT_EXT;
	m = get_node_reference(obj);
	ep = enc_atom(slot,dist_addrs[m].sysname,ep);
	j = get_number_reference(obj);
	put_int32(j, ep);
	ep += 4;
	if (m == THIS_NODE)     /* A local refer */
	    *ep++ = this_creation;
	else
	    *ep++ = get_creation(obj);
	return ep;

#if defined(JAM)
    case NIL_DEF:
	*ep++ = NIL_EXT;   /* soley empty lists */
	return ep;
#endif

    case LIST_DEF:
	if ((i = is_string(obj)) && (i < MAX_STRING_LEN)) {
	    *ep++ = STRING_EXT;
	    put_int16(i, ep);
	    ep += 2;
	    while (is_list(obj)) {
		uint32* cons = ptr_val(obj);
		*ep++ = unsigned_val(CAR(cons));
		obj = CDR(cons);
	    }
	    return ep;
	}
	*ep++ = LIST_EXT;
	back = ep;  /* leave length space */
	ep += 4;
	i = 0;
	while (is_list(obj)) {
	    uint32* cons = ptr_val(obj);
	    i++;  /* count number of cons cells */
	    if ((ep = enc_term(slot, CAR(cons), ep)) == NULL)
		return NULL;
	    obj = CDR(cons);
	}
	if ((ep = enc_term(slot, obj, ep)) == NULL)
	    return NULL;
	put_int32(i, back);
	return ep;

    case TUPLE_DEF:
	ptr = ptr_val(obj);
	i = arityval(*ptr);
	ptr++;
	if (i <= 0xff) {
	    *ep++ = SMALL_TUPLE_EXT;
	    put_int8(i, ep);
	    ep += 1;
	}
	else  {
	    *ep++ = LARGE_TUPLE_EXT;
	    put_int32(i, ep);
	    ep += 4;
	}
	while(i--) {
	    if ((ep = enc_term(slot, *ptr++, ep)) == NULL)
		return NULL;
	}
	return ep;

    case FLOAT_DEF:
	*ep++ = FLOAT_EXT;
	ptr = ptr_val(obj);
	GET_DOUBLE(obj, f);

	/* now the sprintf which does the work */
	i = sys_double_to_chars(f.fd, (char*) ep);

	/* Don't leave garbage after the float!  (Bad practice in general,
	 * and Purify complains.)
	 */
	sys_memset(ep+i, 0, 31-i);

	return ep + 31;

    case BINARY_DEF:
	*ep++ = BINARY_EXT;
	bp = (ProcBin*) ptr_val(obj);
	j = bp->size;
	put_int32(j, ep);
	ep += 4;
	sys_memcpy((char *) ep, (char*) bp->bytes, (int) j);
	return ep + j;
    }
    return NULL;
}

static int to_external2(slot,obj, ext)
int slot; uint32 obj; byte **ext;
{
    uint32 i, j,m ,arity, *nobj, *hp;
    ProcBin *bp;
    FloatDef f;
    byte *back;

    switch (tag_val_def(obj)) {
    case ATOM_DEF:
	*ext = enc_atom(slot,obj,*ext);
	return 0;

    case SMALL_DEF:
	if ((j = unsigned_val(obj)) < 256) { /* special case for small ints */
	    *(*ext)++ = SMALL_INTEGER_EXT;
	    *(*ext)++ = j & 0xff;
	    return(0);
	}
	j = signed_val(obj);
	*(*ext)++ = INTEGER_EXT;
	*(*ext)++ = (j >> 24) & 0xff;
	*(*ext)++ = (j >> 16) & 0xff;
	*(*ext)++ = (j >> 8) & 0xff;
	*(*ext)++ = j  & 0xff;
	return(0);
    case BIG_DEF:
#if defined(BEAM)
	if(is_nil(obj)) {
            *(*ext)++ = NIL_EXT;   /* soley empty lists */
            return(0);
        }
#endif
	if ((arity = big_bytes(obj)) < 256)
	{
	    *(*ext)++ = SMALL_BIG_EXT;
	    *(*ext)++ = arity & 0xff;
	}
	else
	{
	    *(*ext)++ = LARGE_BIG_EXT;
	    *(*ext)++ = (arity >> 24) & 0xff;
	    *(*ext)++ = (arity >> 16 ) & 0xff;
	    *(*ext)++ = (arity >> 8) & 0xff;
	    *(*ext)++ = arity & 0xff;
	}
	*(*ext)++ = big_sign(obj);
	*ext = big_to_bytes(obj, *ext);
	return(0);
#if defined(JAM)
    case NIL_DEF:
	*(*ext)++ = NIL_EXT;   /* soley empty lists */
	return(0);
#endif
    case PID_DEF:
	*ext = enc_pid(slot, obj, *ext);
	return 0;

    case REFER_DEF:
	*(*ext)++ =  REFERENCE_EXT;
	m = get_node_reference(obj);
	*ext = enc_atom(slot,dist_addrs[m].sysname,*ext);
	j = get_number_reference(obj);
	*(*ext)++ = (j >>24) &0xff;
	*(*ext)++ = (j >>16) &0xff;
	*(*ext)++ = (j >>8) &0xff;
	*(*ext)++ = j &0xff;
	if (m == THIS_NODE)     /* A local refer */
	    *(*ext)++ = this_creation;
	else   
	    *(*ext)++ = get_creation(obj);
	
	return(0);
    case PORT_DEF:
	*(*ext)++ = PORT_EXT;
	m = get_node_reference(obj);
	*ext = enc_atom(slot,dist_addrs[m].sysname,*ext);
	j = get_number_reference(obj);
	*(*ext)++ = (j >>24) &0xff;
	*(*ext)++ = (j >>16) &0xff;
	*(*ext)++ = (j >>8) &0xff;
	*(*ext)++ = j &0xff;
	if (m == THIS_NODE)     /* A local refer */
	    *(*ext)++ = this_creation;
	else   
	    *(*ext)++ = get_creation(obj);
	
	return(0);
    case LIST_DEF:
	nobj = ptr_val(obj);
	if ((i = is_string(obj)) && (i < MAX_STRING_LEN)) {
	    *(*ext)++ = STRING_EXT;
	    *(*ext)++ = (i >>8) &0xff;
	    *(*ext)++ = i &0xff;
	    while (1) {
		 *(*ext)++ = unsigned_val(*nobj++);
		 if (is_nil(*nobj)) return(0);
		 nobj = ptr_val(*nobj);
	     }
	}
	*(*ext)++ = LIST_EXT;
	back = *ext;
	i = 0;
	*ext += 4;
	if (!is_nil(obj)) {
	    while (1) {
		i++;
		to_external2(slot,*nobj, ext);
		nobj++; 
		if (is_not_list(*nobj)) {
		    /* non well formed list or nil */
		    to_external2(slot,*nobj, ext);
		    break;
		}
		nobj = ptr_val(*nobj);
	    }
	}
	*back++ = (i >> 24) & 0xff;
	*back++ = (i >> 16 ) & 0xff;
	*back++ = (i >> 8) & 0xff;
	*back   = i & 0xff;
	return(0); 
    case TUPLE_DEF:
	i = *(ptr_val(obj)); /* arity value */
	if (is_not_arity_value(i)) {
	    erl_exit(1,"Got 0x%08x instead of arityvalue in to_external\n", i);
	}
	/* two bytes for arity */
	arity = arityval(i);
	if (arity <= 0xff) {
	    *(*ext)++ = SMALL_TUPLE_EXT;
	    *(*ext)++ = arity & 0xff;
	}
	else  {
	     *(*ext)++ = LARGE_TUPLE_EXT;
	     *(*ext)++ = (arity >> 24) & 0xff;
	     *(*ext)++ = (arity >> 16 ) & 0xff;
	     *(*ext)++ = (arity >> 8) & 0xff;
	     *(*ext)++ = arity & 0xff;
	 }
	for (i = 0; i < arity; i++) 
	    to_external2(slot,*(ptr_val(obj) + i + 1), ext);
	return(0);
    case FLOAT_DEF:
	*(*ext)++ = FLOAT_EXT;
	hp = ptr_val(obj);
	if (*hp != make_thing(2)) {
	    erl_exit(1,"Got 0x%08x instead of thing value 2 in to_external\n", 
		     *(ptr_val(obj)));
	}
	GET_DOUBLE(obj, f);
     
	/* now the sprintf which does the work */
	i = sys_double_to_chars(f.fd, (char*)*ext);

	/* Don't leave garbage after the float!  (Bad practice in general,
	 * and Purify complains.)
	 */
	sys_memset(*ext+i, 0, 31-i);

	*ext += 31;
	return(0);
    case BINARY_DEF:
	*(*ext)++ = BINARY_EXT;
	bp = (ProcBin*) ptr_val(obj);
	j = bp->size;
	*(*ext)++ = (j >> 24) & 0xff;
	*(*ext)++ = (j >> 16) & 0xff;
	*(*ext)++ = (j >> 8) & 0xff;
	*(*ext)++ = j  & 0xff;
	
	sys_memcpy((char *) *ext, (char*) bp->bytes, (int) j);
	*ext += j;
	return(0);
    default:
	erl_exit(1, "Internal data structure error (in to_external) 0x%08x\n",
		 obj);
    }
    return(-1);  /* Pedantic (lint does not know about erl_exit) */
}



int to_external(slot,obj,ext) 
int slot; uint32 obj; byte **ext;
{
    byte* ptr = *ext;
    byte* dist_end = dist_buf + dist_buf_size;

    /* check if we are using dist_buf !! */
    if ((slot > 0) && (ptr >= dist_buf) && (ptr < dist_end)) {
	int size = 50 + encode_size_struct(obj);

	/* check if distbuf must grow */
	if ((ptr + size) > dist_end) {
	    int len = ptr - dist_buf;
	    char* buf;
	    
	    size += (1000 + len);
	    buf = (byte*) safe_alloc(20+size);  /* REMOVE THIS SLOPPY !!! */
	    
	    /* We need to restore the old contetnts of dist_buf
	       before we can proceed */
	    sys_memcpy(buf,dist_buf,len);
	    if (dist_buf != tmp_buf)
		sys_free(dist_buf);
	    dist_buf_size = size;
	    dist_buf = buf;
	    ptr =  dist_buf + len;
	    *ext = ptr;
	}
    }
    *ptr++ = VERSION_MAGIC;
    if ((ptr = enc_term(slot, obj, ptr)) == NULL)
	erl_exit(1, "Internal data structure error (in to_external)\n");
    *ext = ptr;
    return 0;
}

/*
** hpp is set to either a &p->htop or
** a pointer to a memory pointer (form message buffers)
** on return hpp is updated to point after allocated data
*/
uint32 from_external(slot, hpp, ext, msoh)
int slot; uint32** hpp; byte **ext; ProcBin** msoh;
{
    uint32 obj;
    byte* ep = *ext;

    if (*ep++ != VERSION_MAGIC) {
	if (slot >= 0)
	    erl_printf(CBUF,
		       "** Got message from noncompatible erlang on slot %d\n",
		       slot);
	else 
	    erl_printf(CBUF,
		       "** Attempt to convert old non compatible binary %d\n",
		       *ep);
	send_error_to_logger(0);
	return 0;
    }
    if ((ep = dec_term(slot,hpp,ep,msoh,&obj)) == NULL) {
	if (slot >= 0) {	/* Don't print this for binary_to_term */
	    erl_printf(CBUF,"Got corrupted message on slot %d\n", slot);
	    send_error_to_logger(0);
	}
#ifdef DEBUG
	bin_write(CERR,*ext,500);
#endif
	return 0;
    }
#ifdef DEBUG
    check_struct(obj);
#endif
    *ext = ep;
    return obj;
}


static byte* dec_term(slot, hpp, ep, msoh, objp)
int slot; uint32** hpp; byte *ep; ProcBin** msoh; uint32* objp;
{
    uint32 i;
    uint32 j;
    uint32 k;
    uint32 n;
    uint32* ptr = NULL;		/* suppress warnings about use before set */
    int si;
    FloatDef ff;
    ProcBin* pb;
    uint32* hp = *hpp;  /* updated by done_hp */

    switch (*ep++) {
    case INTEGER_EXT:
	n = get_int32(ep); 
	ep += 4;
	*objp = make_small(n);
	return ep;

    case SMALL_INTEGER_EXT:
	n = get_int8(ep);
	ep++;
	*objp = make_small(n);
	return ep;

    case SMALL_BIG_EXT:
	n = get_int8(ep);
	ep++;
	goto big_loop;

    case LARGE_BIG_EXT:
	n = get_int32(ep);
	ep += 4;

    big_loop:
	k = get_int8(ep);
	ep++;
	i = bytes_to_big(ep, n, k, hp);
	if (is_big(i))
	    hp += (big_arity(i)+1);
	*objp = i;
	ep += n;
	*hpp = hp;
	return ep;

    case ATOM_EXT:
	n = get_int16(ep);
	ep += 2;
	*objp = am_atom_put((char*)ep, n);
	return ep + n;

    case NEW_CACHE:
	return dec_and_insert_hashed_atom(slot,ep,objp);

    case CACHED_ATOM:
	return dec_hashed_atom(slot,ep,objp);

    case PID_EXT:
	return dec_pid(slot, ep, objp);

    case REFERENCE_EXT:
	if ((ep = dec_atom(slot,ep, &i)) == NULL)
	    return 0;
	if ((si = find_or_insert_dist_slot(i)) < 0)
	    return 0;
	j = get_int32(ep);
	ep += 4;
	n = get_int8(ep);
	ep += 1;
	*objp = make_refer(si,j);
	return ep;

    case PORT_EXT:
	if ((ep = dec_atom(slot,ep, &i)) == NULL)
	    return 0;
	if ((si = find_or_insert_dist_slot(i)) < 0)
	    return 0;
	j = get_int32(ep);
	ep += 4;
	n = get_int8(ep);
	ep += 1;
	*objp = make_port2(si,j);
	return ep;

    case NIL_EXT:
	*objp = NIL;
	return ep;

    case LIST_EXT:
	n = get_int32(ep);
	ep += 4;
	if (n == 0) {
	    *objp = NIL;
	    return ep;
	}
	*objp = make_list(hp);
	for (k = 0; k < n; k++) {
	    ptr = hp;
	    hp += 2;
	    if ((ep = dec_term(slot,&hp,ep,msoh,&ptr[0])) == NULL)
		return NULL;
	    ptr[1] = make_list(hp);
	}
	if (*ep == NIL_EXT) {
	    ep++;
	    ptr[1] = NIL;
	}
	else {
	    if ((ep = dec_term(slot,&hp,ep,msoh,&ptr[1])) == NULL)
		return NULL;
	}
	*hpp = hp;
	return ep;

    case STRING_EXT:
	n = get_int16(ep);
	ep += 2;
	if (n == 0) {
	    *objp = NIL;
	    return ep;
	}
	*objp = make_list(hp);
	for (k = 0; k < n; k++) {
	    ptr = hp;
	    hp += 2;
	    ptr[0] = make_small(*ep);
	    ptr[1] = make_list(hp);
	    ep++;
	}
	ptr[1] = NIL;
	*hpp = hp;
	return ep;

    case SMALL_TUPLE_EXT:
	n = get_int8(ep);
	ep++;
	goto tuple_loop;

    case LARGE_TUPLE_EXT:
	n = get_int32(ep);
	ep += 4;

    tuple_loop:
	*objp = make_tuple(hp);
	ptr = hp;
	*ptr = make_arityval(n);
	hp += (n+1);
	for (j = 1; j <= n; j++) {
	    if ((ep = dec_term(slot,&hp,ep,msoh, &ptr[j])) == NULL)
		return NULL;
	}
	*hpp = hp;
	return ep;

    case FLOAT_EXT:
	if (sys_chars_to_double((char*)ep, &ff.fd) != 0)
	    return 0;
	ep += 31;
	*objp = make_float(hp);
	PUT_DOUBLE(ff, hp);
	hp += 3;
	*hpp = hp;
	return ep;

    case BINARY_EXT:
	n = get_int32(ep);
	ep += 4;
	pb = alloc_binary(msoh, ep, n);
	ep += n;
	*objp = make_binary(pb);
	return ep;

    default:
	return NULL;
    }
}


/* returns the number of bytes needed to encode an object
   to a sequence of bytes
   N.B. That this must agree with to_external2() above!!!
   (except for cached atoms) */

uint32 encode_size_struct(obj)
uint32 obj;
{
    return (1 + encode_size_struct2(obj));  /* 1 for the VERSION_MAGIC */
}

static uint32 encode_size_struct2(obj)
uint32 obj;
{
    uint32 m,i, arity, *nobj, sum;
    ProcBin *bp;

    switch (tag_val_def(obj)) {
    case ATOM_DEF:
	/* Make sure NEW_CACHE ix l1 l0 a1 a2 .. an fits */
	return (1 + 1 + 2 + atom_tab(unsigned_val(obj))->len);
    case SMALL_DEF:
	if (unsigned_val(obj) < 256 ) 
	    return(1 + 1);
	else 
	    return(1 + 4);
    case BIG_DEF:
#if defined(BEAM)
	if (is_nil(obj)) 
	    return 1;
#endif
	if ((i = big_bytes(obj)) < 256)
	    return 1 + 1 + 1 + i;  /* tag,size,sign,digits */
	else
	    return 1 + 4 + 1 + i;  /* tag,size,sign,digits */
#if defined(JAM)
    case NIL_DEF:
	return(1);
#endif
    case PID_DEF:
	m = get_node(obj);
	return (1 + encode_size_struct2(dist_addrs[m].sysname) + 4 + 4 + 1);
    case REFER_DEF:
    case PORT_DEF:
	m = get_node_reference(obj);
	return (1 + encode_size_struct2(dist_addrs[m].sysname) + 4 + 1);
    case LIST_DEF:
	if ((m = is_string(obj)) && (m < MAX_STRING_LEN))
	    return m + 2 + 1;
	nobj = ptr_val(obj);
	sum = 5;
	while (1) {
	    sum += encode_size_struct2(*nobj++);
	    if (is_not_list(*nobj)) {
		if (is_nil(*nobj))
		    return sum + 1;
		return(sum + encode_size_struct2(*nobj));
	    }
	    nobj = ptr_val(*nobj);
	}
    case TUPLE_DEF:
	arity = arityval(*(ptr_val(obj)));
	if (arity <= 0xff) 
	    sum = 1 + 1;
	else
	    sum = 1 + 4;
	for (i = 0; i < arity; i++)
	    sum += encode_size_struct2(*(ptr_val(obj) + i + 1));
	return sum;
    case FLOAT_DEF:
	return 32;   /* Yes  including the tag */
    case BINARY_DEF:
	bp = (ProcBin*) ptr_val(obj);
	return 1 + 4 + bp->size;
    default:
	erl_exit(1,"Internal data structure error (in encode_size_struct2)%x\n",
		 obj);
    }
    return -1; /* Pedantic (lint does not know about erl_exit) */
}

/*
** Return the number of words to hold the unpacked version of
** the encoded data
** Returns -1 on error
**          0 for constant objects
**          N for structured objects
*/
static byte* dec_size_atom(ep, ep_end)
byte* ep; byte* ep_end;
{
    uint32 i;

    if (ep >= ep_end) return NULL;

    switch(*ep++) {
    case ATOM_EXT:
	if (ep + 2 > ep_end) return NULL;
	i = get_int16(ep);
	return ep + 2 + i;
    case NEW_CACHE:
	if (ep + 3 > ep_end) return NULL;
	ep++;
	i = get_int16(ep);
	return ep + 2 + i;
    case CACHED_ATOM:
	return ep + 1;
    default:
	return NULL;
    }
}

/* dec_size:
**
**  Calculate number of words required to represent an externally
**  represented term on the heap.
**
**  Current checks include:
**       a) Byte buffer must be of precise length
**       b) Total number of elements must match
**
** NOTE. A change is needed to take care of other type of errors
*/

#define DEC_CHECK(p, sz, p_end) do { \
  if ((p)+(sz) > (p_end)) \
     return -1; \
 } while(0)

int dec_size(ep, length)
byte* ep; uint32 length;
{
    int size = 0;
    int elems = 1;  /* require at least one element */
    uint32 i;
    byte* ep_end = ep + length;

    if (length == 0 || *ep++ != VERSION_MAGIC) {
	return -1;
    }

    while(ep < ep_end) {
	elems--;
	switch(*ep++) {
	case INTEGER_EXT:
	    ep += 4; 
	    break;

	case SMALL_INTEGER_EXT:
	    ep += 1; 
	    break;

	case SMALL_BIG_EXT:
	    DEC_CHECK(ep, 1, ep_end);
	    i = get_int8(ep);
	    ep += (1+1+i);
	    size += 1+(i+3)/4;  /* thing + 32-bit words */
	    break;

	case LARGE_BIG_EXT:
	    DEC_CHECK(ep, 4, ep_end);
	    i = get_int32(ep);
	    ep += (4+1+i); 
	    size += 1+(i+3)/4;  /* thing + 32-bit words */
	    break;

	case ATOM_EXT: 
	    DEC_CHECK(ep, 2, ep_end);
	    i = get_int16(ep); 
	    ep += (2 + i);
	    break;

	case NEW_CACHE:
	    DEC_CHECK(ep, 3, ep_end);
	    ep++;
	    i = get_int16(ep);
	    ep += (2 + i);
	    break;

	case CACHED_ATOM:
	    ep++; 
	    break;

	case PID_EXT:
	    if ((ep = dec_size_atom(ep, ep_end)) == NULL)
		return -1;
	    ep += 9;
	    break;

	case PORT_EXT:
	case REFERENCE_EXT:
	    if ((ep = dec_size_atom(ep, ep_end)) == NULL)
		return -1;
	    ep += 5;
	    break;

	case FLOAT_EXT:
	    ep += 31;
	    size += 3;
	    break;

	case BINARY_EXT:
	    DEC_CHECK(ep, 4, ep_end);
	    i = get_int32(ep);
	    ep += (4 + i);
	    break;

	case NIL_EXT:
	    break;

	case STRING_EXT:
	    DEC_CHECK(ep, 2, ep_end);
	    i = get_int16(ep);
	    ep += (2 + i);
	    size += (2 * i);
	    break;

	case LIST_EXT:
	    DEC_CHECK(ep, 4, ep_end);
	    i = get_int32(ep);
	    elems += (i+1);   /* expect at least i elements & a tail */
	    ep += 4;
	    size += (2 * i);
	    break;
	    
	case SMALL_TUPLE_EXT:
	    DEC_CHECK(ep, 1, ep_end);
	    i = get_int8(ep);
	    elems += i;   /* expect at least i elements */
	    ep += 1;
	    size += (1 + i);
	    break;
	    
	case LARGE_TUPLE_EXT:
	    DEC_CHECK(ep, 4, ep_end);
	    i = get_int32(ep);
	    elems += i;   /* expect at least i elements */
	    ep += 4;
	    size += (1 + i);
	    break;
	    
	default:
	    return -1;
	}

    }
    if ((ep == ep_end) && (elems == 0))
	return size;
    return -1;
}
#undef DEC_CHECK

static int
decode_size2(ext, endp, okp)
    byte **ext;
    byte* endp;
    int* okp;
{
    uint32 sum,j,k,i = 0;

#define SKIP(sz) \
    do { \
	*ext += (sz); \
	if ((*ext) > endp) { return *okp = 0; } \
    } while (0)

#define CHKSIZE(sz) \
    do { \
	 if ((*ext)+(sz) > endp) { return *okp = 0; } \
    } while (0)

    SKIP(1);
    switch ((*ext)[-1]) {
    case INTEGER_EXT:
	SKIP(4);
	return(1);
    case SMALL_INTEGER_EXT:
	SKIP(1);
	return(1);
    case SMALL_BIG_EXT:
	CHKSIZE(1);
	i = **ext;		/* number of bytes */
	SKIP(1+1+i);		/* skip size,sign,digits */
	return 1+1+(i+3)/4;
    case LARGE_BIG_EXT:
	CHKSIZE(4);
	i = (**ext << 24) | ((*ext)[1] << 16) |((*ext)[2] << 8) | (*ext)[3];
	SKIP(4+1+i);		/* skip, size,sign,digits */
	return 1+1+(i+3)/4;
    case ATOM_EXT:
	CHKSIZE(2);
	i = (**ext << 8) | (*ext)[1];
	SKIP(i+2);
	return(1);
    case NEW_CACHE:
	CHKSIZE(3);
	i = get_int16(*ext+1);
	SKIP(i+3);
	return(1);
    case CACHED_ATOM:
	SKIP(1);
	return(1);
    case PID_EXT:
	if (decode_size2(ext, endp, okp) == 0) /* Eat first atom */
	    return 0;
	SKIP(9);
	return(1);
    case REFERENCE_EXT:
    case PORT_EXT:
	/* eat first atom */
	if (decode_size2(ext, endp, okp) == 0)
	    return 0;
	SKIP(5);		/* One int and the creation field */
	return(1);
    case NIL_EXT:
	return(1);
    case LIST_EXT:
	i = j = sum = 0;
	CHKSIZE(4);
	j = (**ext << 24) | ((*ext)[1] << 16) |((*ext)[2] << 8) | (*ext)[3];
	SKIP(4);
	for(k=0; k<j; k++) {
	    if ((i = decode_size2(ext, endp, okp)) == 0)
		return(0);
	    sum += 1 + i;
	}
	CHKSIZE(1);
	if (**ext == NIL_EXT) {
	    SKIP(1);
	    return sum + 1;
	}
	if ((i = decode_size2(ext, endp, okp)) == 0)
	    return 0;
	return(sum + i);
    case STRING_EXT:
	CHKSIZE(2);
	i = **ext << 8 | (*ext)[1];
	SKIP(i+2);
	return (2 * i) + 1;
    case SMALL_TUPLE_EXT:
	CHKSIZE(1);
	i = *(*ext);
	SKIP(1);
	sum = 2;
	for (j = 0; j < i; j++) {
	    if ((k = decode_size2(ext, endp, okp)) == 0)
		return(0);
	    sum += k;
	}
	return(sum);
    case LARGE_TUPLE_EXT:
	CHKSIZE(4);
	i = (**ext << 24) | ((*ext)[1] << 16) |((*ext)[2] << 8) | (*ext)[3];
	SKIP(4);
	sum = 2;
	for (j = 0; j < i; j++) {
	    if ((k = decode_size2(ext, endp, okp)) == 0)
		return(0);
	    sum += k;
	}
	return(sum);
    case FLOAT_EXT:
	SKIP(31);
	return(4);
    case BINARY_EXT:
	CHKSIZE(4);
	i = (**ext << 24) | ((*ext)[1] << 16) |((*ext)[2] << 8) | (*ext)[3];
	SKIP(4+i);
	return(1);
    default:
	return(0);
    }
}

int decode_size(t, size)
    byte *t;
    int size;
{
    int heap_size;
    int ok = 1;

    if (size > 0 && *t == VERSION_MAGIC)
	t++;
    else
	return -1;
    size--;
    heap_size = decode_size2(&t, t+size, &ok);
    return ok ? heap_size : -1;
}
