/* ``The contents of this file are subject to the Erlang Public License,
 * Version 1.0, (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.erlang.org/EPL1_0.txt
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 * 
 * The Original Code is Erlang-4.7.3, December, 1998.
 * 
 * The Initial Developer of the Original Code is Ericsson Telecom
 * AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
 * Telecom AB. All Rights Reserved.
 * 
 * Contributor(s): ______________________________________.''
 */
/*
 * This file is copyright (c) Ellemtel in January 1991
 *
 * Author Mike Williams
 *
 * Tony:
 * 950227:  Rewrote process table to be a table of pointers to
 *          processes. The Process structures are allocated by
 *          fix_alloc.
 * 960923:  High priority as a new priority level
 */

#include "sys.h"
#include "config.h"
#include "global.h"
#include "erl_process.h"
#include "error.h"
#include "bif.h"
#include "db.h"
#include "dist.h"

#if defined(JAM)
#  include "jam_opcodes.h"
#elif defined(BEAM)
#  include "error.h"
#endif /* BEAM */

#define MAX_BIT          (1 << PRIORITY_MAX)
#define HIGH_BIT         (1 << PRIORITY_HIGH)
#define NORMAL_BIT       (1 << PRIORITY_NORMAL)
#define LOW_BIT          (1 << PRIORITY_LOW)


#if defined(JAM)
extern byte jam_apply[];
extern byte jam_exit[];
#elif defined(BEAM)
extern uint32* beam_exit;
extern uint32* beam_apply_op;
#endif

static int p_next;
static int p_serial;

typedef struct schedule_q {
    Process* first;
    Process* last;
} ScheduleQ;

static ScheduleQ queue[NPRIORITY_LEVELS];
static int processes_busy;
static int bg_count;
static unsigned qmask = 0;

Process**  process_tab;
uint32     context_switches;   /* no of context switches */
uint32     reductions;	       /* total number of reductions */
uint32     last_reds;	       /* used in process info */

/*
** Kill all processes.
*/
static void atexit_scheduler(arg)
void* arg;
{
    int i;
    Process* p;

    /* Perhaps take one round and turn of tracing to minimize
     ** message passing during exit
     */
    for (i = 0; i < max_process; i++) {
	if (((p = process_tab[i]) != NULL) && (p->status != P_EXITING))
	    /* do_exit(p, am_killed); */
	    ;
    }
    DEBUGF(("atexit_scheduler: live processes = %d\n", processes_busy));

    /* XXX one round and delete processes EXITING */
    sys_free(process_tab);
    process_tab = NULL;
}


/* initialize the scheduler */
void init_scheduler()
{
    int i;

    erl_at_exit(atexit_scheduler, NULL);
    process_tab = (Process**) 
	safe_alloc_from(91, max_process * sizeof(Process*));
    sys_memzero(process_tab, max_process * sizeof(Process*));

    p_next = 0;
    p_serial = 0;

    /* mark the schedule queue as empty */
    for(i = 0; i < NPRIORITY_LEVELS; i++)
	queue[i].first = queue[i].last = (Process*) 0;
    qmask = 0;
    processes_busy = 0;
    bg_count = 0;
    context_switches = 0;
    reductions = 0;
    last_reds = 0;
}

int
sched_q_len(void)
{
    int i;
    int len = 0;

    for (i = 0; i < NPRIORITY_LEVELS; i++) {
	Process* p;

	for (p = queue[i].first; p != NULL; p = p->next) {
	    len++;
	}
    }
    return len;
}

/* schedule a process */
void add_to_schedule_q(p)
Process *p;
{
    ScheduleQ* sq = &queue[p->prio];

    /* Never schedule a suspended process */
    ASSERT(p->status != P_SUSPENDED);

    qmask |= (1 << p->prio);
    p->next = NULL;
    if (sq->first == (Process *) 0)
	sq->first = p;
    else
	sq->last->next = p;
    sq->last = p;
    if (p->status != P_EXITING) {
	p->status = P_RUNABLE;
    }
}

/* Possibly remove a scheduled process we need to suspend */

int remove_proc_from_sched_q(p)
Process *p;
{
    Process *tmp, *prev;
    int i;

    for(i = 0; i < NPRIORITY_LEVELS; i++) {
	ScheduleQ *sq = &queue[i];

	if (sq->first == (Process*) NULL)
	    continue;
	if (sq->first == sq->last && sq->first == p) {
	    sq->first = sq->last = NULL;
	    qmask &= ~(1 << p->prio);
	    return 1;
	}
	if (sq->first == p) {
	    sq->first = sq->first->next;
	    return 1;
	}
	tmp = sq->first->next;
	prev = sq->first;
	while (tmp) {
	    if (tmp == p) {
		prev->next = tmp->next;
		if (p == sq->last)
		    sq->last = prev;
		return 1;
	    }
	    prev = tmp;
	    tmp = tmp->next;
	}
    }
    return 0;
}

/* note that P_RUNNING is only set so that we don't try to remove
** running processes from the schedule queue if they exit - a running
** process not being in the schedule queue!! 
** Schedule for up to INPUT_REDUCTIONS context switches,
** return 1 if more to do.
*/

int
schedule(void)
{
    Process *p;
    ScheduleQ *sq;
    int function_calls = 0;
    int calls;

    if (do_time) {
	bump_timer();
    }
    
    do {
	switch (qmask) {
	case MAX_BIT:
	case MAX_BIT|HIGH_BIT:
	case MAX_BIT|NORMAL_BIT:
	case MAX_BIT|LOW_BIT:
	case MAX_BIT|HIGH_BIT|NORMAL_BIT:
	case MAX_BIT|HIGH_BIT|LOW_BIT:
	case MAX_BIT|NORMAL_BIT|LOW_BIT:
	case MAX_BIT|HIGH_BIT|NORMAL_BIT|LOW_BIT:
	    sq = &queue[PRIORITY_MAX];
	    break;
	case HIGH_BIT:
	case HIGH_BIT|NORMAL_BIT:
	case HIGH_BIT|LOW_BIT:
	case HIGH_BIT|NORMAL_BIT|LOW_BIT:
	    sq = &queue[PRIORITY_HIGH];
	    break;
	case NORMAL_BIT:
	    sq = &queue[PRIORITY_NORMAL];
	    break;
	case NORMAL_BIT|LOW_BIT:
	    bg_count++;
	    if ((bg_count % BG_PROPORTION) == 0)
		sq = &queue[PRIORITY_LOW];
	    else
		sq = &queue[PRIORITY_NORMAL];
	    break;
	case LOW_BIT:
	    bg_count++;
	    sq = &queue[PRIORITY_LOW];
	    break;
	case 0:			/* No process at all */
	    return 0;
#ifdef DEBUG
	default:
	    ASSERT(0);
#endif
	}

	/*
	 * Take the chosen process out of the queue.
	 */
	ASSERT(sq->first != NULL); /* Wrong bitmask in qmask? */
	p = sq->first;
	sq->first = p->next;
	if (sq->first == NULL) {
	    sq->last = NULL;
	    qmask &= ~(1 << p->prio);
	}

	ASSERT(p->status != P_SUSPENDED); /* Never run a suspended process */

	context_switches++;
	calls = CONTEXT_REDS;
	if (p->status != P_EXITING) {
	    if (IS_TRACED_FL(p, F_TRACE_SCHED))
		trace_sched(p, am_in);
	    if (p->flags & F_NEED_GC || 
		(((p->mbuf_sz + p->mbuf_struct_sz)*MBUF_GC_FACTOR) >= p->heap_sz)) {
		calls -= do_gc(p, 0);
		if (calls < 0) {
		    calls = 1;
		}
	    }
	    p->status = P_RUNNING;
	}

	calls = process_main(p, calls);
	function_calls += calls;
	reductions += calls;

	p->reds += calls;
	if (p->status == P_FREE) {
	    fix_free(process_desc, (uint32 *) p);
	} else if (IS_TRACED_FL(p, F_TRACE_SCHED)) {
	    trace_sched(p, am_out);
	}

	if (do_time) {
	    bump_timer();
	}
    } while (function_calls <= INPUT_REDUCTIONS);
    
    return qmask;
}

/*
** Fix allocate a process
*/
static Process*
alloc_process(void)
{
    Process* p;
    int p_prev;

    if (p_next == -1)
	return NULL;

    if ((p = (Process*) fix_alloc_from(92,process_desc)) == NULL)
	return NULL;
    p->id = make_pid(p_serial, THIS_NODE, p_next);
    p->rstatus = P_FREE;
    p->rcount = 0;

    /* set p_next to the next available slot */
    p_prev = p_next;
    p_next = (p_next+1) % max_process;

    while(p_prev != p_next) {
	if (p_next == 0)
	    p_serial = (p_serial+1) % MAX_SERIAL;
	if (process_tab[p_next] == NULL) /* found a free slot */
	    return p;
	p_next = (p_next+1) % max_process;
    }
    p_next = -1;
    return p;
}

uint32
erl_create_process(Process* parent, /* Parent of process (default group leader). */
		   uint32 mod,	/* Tagged atom for module. */
		   uint32 func,	/* Tagged atom for function. */
		   uint32 args,	/* Arguments for function (must be well-formed list). */
		   ErlSpawnOpts* so) /* Options for spawn. */
{
    Process *p;
    uint32 arg_size;		/* Size of arguments. */
    uint32 sz;			/* Needed words on heap. */
    uint32 arity;		/* Number of arguments. */
    uint32 heap_need;		/* Size needed on heap. */
    ScheduleQ* sq;
#ifdef JAM
    int stack_need;		/* Words needed on stack. */
#endif

    /*
     * Check for errors.
     */

    if (is_not_atom(mod) || is_not_atom(func) || ((arity = list_length(args)) < 0)) {
	so->error_code = BADARG;
	return 0;
    }
    if ((p = alloc_process()) == NULL) {
	erl_printf(CBUF, "Too many processes\n");
	send_error_to_logger(parent->group_leader);
	so->error_code = SYSTEM_LIMIT;
	return 0;
    }

    processes_busy++;
    arg_size = size_object(args);
    heap_need = arg_size + H_MARGIN;
    p->group_leader = parent->group_leader;

    if (so->flags & SPO_USE_ARGS) {
	p->min_heap_size = so->min_heap_size;
	p->gc_switch = so->gc_switch;
	p->prio = so->priority;
	p->flags = 0;
    } else {
	p->min_heap_size = H_MIN_SIZE;
	p->gc_switch = switch_gc_threshold;
	p->prio = PRIORITY_NORMAL;
	p->flags = 0;
    }
    ASSERT(p->min_heap_size == next_heap_size(p->min_heap_size, 0));
    
    p->initial[0] = 0;
    p->initial[INITIAL_MOD] = mod;
    p->initial[INITIAL_FUN] = func;
    p->initial[INITIAL_ARI] = arity;

    /*
     * Must initialize binary lists here before copying binaries to process.
     */
    p->mso_weight = 0;
    p->mso = p->old_mso = NULL;

#if defined(JAM)
    stack_need = S_MARGIN + 7 + 2; /* 2 is for fence patterns */
    if (stack_need < S_MIN_SIZE) {
	sz = stack_need = S_MIN_SIZE;
    } else {
	sz = next_heap_size(stack_need, 0);
    }

    /* Allocate memory for the stack and heap */
    p->stack = (uint32 *) safe_alloc_from(4,sizeof(uint32)*sz);
    p->stack_sz = sz-2;  /* We must compensate for this when we recalculate */

    /* Add stack patterns at end of stack */
    p->stack[p->stack_sz] = S_PATTERN;
    p->stack[p->stack_sz+1] = S_PATTERN;

    p->stop = p->stack;
    p->stack_margin = p->stack + p->stack_sz - S_MARGIN;
    p->fp = p->ap = p->stack;

    if (heap_need < p->min_heap_size) {
	sz = heap_need = p->min_heap_size;
    } else {
	sz = next_heap_size(heap_need, 0);
    }

    p->heap = (uint32*) safe_alloc_from(8,sizeof(uint32)*sz);
    p->old_hend = p->old_htop = p->old_heap = NULL; 
    p->high_water = p->low_water = p->heap;
    p->hend = p->heap + sz;
    p->heap_sz = sz;
    p->htop = p->heap;
    SET_HEAP_MARGIN(p, p->heap + p->heap_sz - H_MARGIN);

    p->pc = jam_apply;
    p->cc = NULL;
    p->catches = ENULL;

    p->fvalue = NIL;  /* must always be values (survives garb on Jam) */

    INIT_FRAME(p);
    *p->stop++ = mod;
    *p->stop++ = func;
    *p->stop++ = copy_struct(args, arg_size, &p->htop, &p->mso);

#elif defined(BEAM)
    if (heap_need < p->min_heap_size) {
	sz = heap_need = p->min_heap_size;
    } else {
	sz = next_heap_size(heap_need, 0);
    }

    p->heap = (uint32*) safe_alloc_from(8, sizeof(uint32)*sz);
    p->old_hend = p->old_htop = p->old_heap = NULL;
    p->high_water = p->low_water = p->heap;
    p->arith_avail = 0;		/* No arithmetic heap. */
#ifdef DEBUG
    p->arith_check_me = NULL;
#endif

    p->stop = p->stack = p->hend = p->heap + sz;
    p->htop = p->heap;
    p->heap_sz = sz;
    p->catches = 0;
    /* No need to initialize p->fcalls. */

    p->current = p->initial+INITIAL_MOD;
    p->beam_apply[0] = (uint32) beam_apply_op;
    p->beam_apply[1] = mod;
    p->beam_apply[2] = func;
    p->beam_apply[3] = make_rreg();   /* Put argument list in r(0) */
    p->beam_apply[4] = (uint32) ready;

    p->call = beam_apply_op;
    p->cp = p->beam_apply+4;	/* Point to ready instruction */
    p->i = p->beam_apply;

    p->arg_reg = p->def_arg_reg;
    p->max_arg_reg = sizeof(p->def_arg_reg)/sizeof(p->def_arg_reg[0]);
    p->arg_reg[0] = copy_struct(args, arg_size, &p->htop, &p->mso);
    p->arity = 1;
#endif

    p->freason = 0;
    p->reds = 0;
    sys_memset(&p->tm, 0, sizeof(ErlTimer));

    p->reg = NULL;
    p->dslot = -1;
    p->error_handler = am_error_handler;    /* default */
    p->tracer_proc = NIL;
    p->links = NULL;

    p->msg.first = NULL;
    p->msg.last = &p->msg.first;
    p->msg.save = &p->msg.first;
    p->msg.len = 0;
    p->mbuf = NULL;
    p->mbuf_sz = 0;
    p->mbuf_struct_sz = 0;
    p->dictionary = NIL;
#ifdef SEQ_TRACE
    p ->seq_trace_lastcnt = 0;
    p ->seq_trace_clock = 0;
    SEQ_TRACE_TOKEN(p) = NIL;
#endif

    process_tab[get_number(p->id)] = p;

    if (IS_TRACED(parent)) {
	if (parent->flags & F_TRACE_SOS) {
	    p->flags |= (parent->flags & TRACE_FLAGS);
	    p->tracer_proc = parent->tracer_proc;
	}
	if (parent->flags & F_TRACE_PROCS) 
	    trace_proc(parent, am_spawn, p->id);
	if (parent->flags & F_TRACE_SOS1) { /* Overrides TRACE_CHILDREN */
	    p->flags |= (parent->flags & TRACE_FLAGS);
	    p->tracer_proc = parent->tracer_proc;
	    p->flags &= ~(F_TRACE_SOS1 | F_TRACE_SOS);
	    parent->flags &= ~(F_TRACE_SOS1 | F_TRACE_SOS);
	}
    }

    /*
     * Check if this process should be initially linked to its parent.
     */

    if (so->flags & SPO_LINK) {
	if (IS_TRACED(parent) && (parent->flags & F_TRACE_PROCS) != 0) {
	    trace_proc(parent, am_link, p->id);
	}
	parent->links = new_link(parent->links, LNK_LINK, p->id, NIL);
	p->links = new_link(p->links, LNK_LINK, parent->id, NIL);
	if (IS_TRACED(parent)) {
	    if (parent->flags & F_TRACE_SOL)  {
		p->flags |= (parent->flags & TRACE_FLAGS);
		p->tracer_proc = parent->tracer_proc;    /* maybe steal */
	    }
	    if (parent->flags & F_TRACE_SOL1)  { /* maybe override */
		p->flags |= (parent->flags & TRACE_FLAGS);
		p->tracer_proc = parent->tracer_proc;   
		p ->flags &= ~(F_TRACE_SOL1 | F_TRACE_SOL);
		parent->flags &= ~(F_TRACE_SOL1 | F_TRACE_SOL);
	    }
	}
    }

    /*
     * Schedule process for execution.
     */
    sq = &queue[p->prio];
    qmask |= (1 << p->prio);
    p->next = NULL;
    if (sq->first == (Process *) 0)
	sq->first = p;
    else
	sq->last->next = p;
    sq->last = p;
    p->status = P_RUNABLE;
    return p->id;
}


void delete_process(p)
Process* p;
{
    ProcBin *bptr;
    ErlLink* lnk;
    ErlMessage* mp;
    ErlMessageBuffer* bp;
    int i;

    if (p->reg != NULL)
	unregister_process(p->reg->name);

    cancel_timer(p);  /* Always cancel timer just in case */

#if defined(BEAM)
    if (p->arg_reg != p->def_arg_reg)
	sys_free(p->arg_reg);
#endif

#if defined(JAM)
    sys_free((char *) p->stack);
#endif
    sys_free((void*) p->heap);
    if (p->old_heap != NULL)
	sys_free(p->old_heap);

    /* free all chunks of binaries */
    bptr = p->mso;
    while(bptr) {
	ProcBin* next_bptr = bptr->next;
	maybe_delete_contents(bptr);
	fix_free(proc_bin_desc, (uint32*)bptr);
	bptr = next_bptr;
    }
    
    bptr = p->old_mso;
    while(bptr) {
	ProcBin* next_bptr = bptr->next;
	maybe_delete_contents(bptr);
	fix_free(proc_bin_desc, (uint32*)bptr);
	bptr = next_bptr;
    }

    /*
     * These should not be used anymore, but if they are, make sure that
     * we'll notice.
     */
    p->old_mso = p->mso = (void *) 0x8DEFFACD;

    /*
     * Free all pending message buffers.
     */
    bp = p->mbuf;
    while(bp != NULL) {
	ErlMessageBuffer* next_bp = bp->next;
	free_message_buffer(bp);
	bp = next_bp;
    }

    /* free all pending messages */
    mp = p->msg.first;
    while(mp != NULL) {
	ErlMessage* next_mp = mp->next;
	free_message(mp);
	mp = next_mp;
    }

    /* free all links */
    lnk = p->links;
    while(lnk != NULL) {
	ErlLink* next_link = lnk->next;
	fix_free(link_desc, (uint32*)lnk);
	lnk = next_link;
    }

    if (p->flags & F_USING_DB)
	db_proc_dead(p->id);

    i = get_number(p->id);
    process_tab[i] = NULL;
    if (p_next == -1)
	p_next = i;

    /*
     * Don't free it here, just mark it.
     */
    p->status = P_FREE;

    processes_busy--;
}


void schedule_exit(p, reason)
Process *p; uint32 reason;
{
    uint32 copy;
    uint32 status = p->status;
    /*
     * If this is the currently running process, we'll only change its
     * status to P_EXITING, and do nothing more.  It's the responsibility
     * of the caller to make the current process exit.
     */
    p->status = P_EXITING;
    if (status == P_RUNNING)
	return;
    copy_object(reason, p, 0, &copy, (Process*)0);
    p->fvalue = copy;

#if defined(JAM)
    p->pc = jam_exit;
#elif defined(BEAM)
    cancel_timer(p);
    p->freason = USER_EXIT;
    KILL_CATCHES(p);
    p->call = beam_exit;
#endif
    if (status != P_RUNABLE)
	add_to_schedule_q(p);
}

/* this function fishishes a process and propagates exit messages - called
   by process_main when a process dies */
void do_exit(p, reason)
Process* p; uint32 reason;
{
    Process *rp;
    ErlLink* lnk;
    uint32 item;
    int slot;
    int ix;

    p->status = P_EXITING;

    if (IS_TRACED_FL(p,F_TRACE_PROCS))
	trace_proc(p, am_exit, reason);

    lnk = p->links;
    p->links = NULL;

    while(lnk != NULL) {
	item = lnk->item;
	switch(lnk->type) {
	case LNK_LINK:
	    if (is_port(item)) {
		if ((slot = get_node_reference(item)) != THIS_NODE)
		    dist_exit(slot, p->id, item, reason);
		else {
		    ix = get_number_reference(item);
		    if (port[ix].status != FREE) {
			del_link(find_link(&port[ix].links,LNK_LINK,
					   p->id,NIL));
			do_exit_port(item, p->id, reason);
		    }
		}
	    }
	    else if (is_pid(item)) {
		if ((slot = get_node(item)) != THIS_NODE) {
#ifdef SEQ_TRACE
		    if (SEQ_TRACE_TOKEN(p) != NIL) {
			seq_trace_update_send(p);
		    }
		    dist_exit_tt(slot, p->id, item, reason, SEQ_TRACE_TOKEN(p));
#else
		    dist_exit(slot, p->id, item, reason);
#endif
		}
		else {
		    if ((rp = pid2proc(item)) != NULL) {
			del_link(find_link(&rp->links,LNK_LINK,p->id,NIL));
			if (rp->flags & F_TRAPEXIT) {
#ifdef SEQ_TRACE
			    if (SEQ_TRACE_TOKEN(p) != NIL ) {
				seq_trace_update_send(p);
			    }
			    deliver_exit_message_tt(p->id, rp, reason, SEQ_TRACE_TOKEN(p));
#else
			    deliver_exit_message(p->id, rp, reason);
#endif
			}
			else if (reason != am_normal)
			    schedule_exit(rp, reason);
		    }
		}
	    }
	    break;
	case LNK_NODE:
	    del_link(find_link(&dist_addrs[lnk->data].links,LNK_NODE,
			       p->id,NIL));
	    break;

	case LNK_OMON:
	case LNK_TMON:
	default:
	    erl_exit(1, "bad type in link list\n");
	    break;
	}
	del_link(&lnk);  /* will set lnk to next as well !! */
    }

    if ((p->flags & F_DISTRIBUTION) && (p->dslot != -1))
	do_net_exits(p->dslot);
    delete_process(p);
}

/* Callback for process timeout */
static void timeout_proc(p)
Process* p;
{
#if defined(BEAM)
    p->i = p->action_time_out;
    p->call = (uint32 *) *(p->i);
#endif /* BEAM */
    p->flags |= F_TIMO;
    p->flags &= ~F_INSLPQUEUE;

    if (p->status == P_WAITING)
	add_to_schedule_q(p); 
    if (p->status == P_SUSPENDED)
	p->rstatus = P_RUNABLE;   /* MUST set resume status to runnable */
}


void cancel_timer(p)
Process* p;
{
    erl_cancel_timer(&p->tm);
    p->flags &= ~(F_INSLPQUEUE|F_TIMO);
}

/*
** Insert a process into the time queue, with a timeout 'timeout'
*/
void set_timer(p, timeout)
Process* p;
uint32 timeout;   /* time in ms */
{
    /* check for special case timeout=0 DONT ADD TO time queue */
    if (timeout == 0) {
	p->flags |= F_TIMO;
	return;
    }
    erl_set_timer(&p->tm,
		  (ErlTimeoutProc) timeout_proc,
		  NULL,
		  (void*) p,
		  timeout);
    p->flags |= F_INSLPQUEUE;
    p->flags &= ~F_TIMO;
}
