/* ``The contents of this file are subject to the Erlang Public License,
 * Version 1.0, (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.erlang.org/EPL1_0.txt
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 * 
 * The Original Code is Erlang-4.7.3, December, 1998.
 * 
 * The Initial Developer of the Original Code is Ericsson Telecom
 * AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
 * Telecom AB. All Rights Reserved.
 * 
 * Contributor(s): ______________________________________.''
 */
/*
 * This file is copyright (c) Ellemtel in January 1991
 *
 * Author: Tony Rogvall
 */

#include "sys.h"
#include "config.h"
#include "global.h"
#include "erl_process.h"
#include "erl_version.h"
#include "db.h"

uint32     max_process;
byte*      tmp_buf;
int        tot_bin_allocated;
uint32     display_items;	/* no of items to display in traces etc */
uint32     display_loads;	/* print info about loaded modules */
uint32     do_time;		/* set at clock interupt */
uint32     garbage_cols;	/* no of garbage collections */
uint32     reclaimed;		/* no of words reclaimed in GCs */
int        switch_gc_threshold;	/*
				 * Threshold when to switch from
				 * fullsweep to generational GC.
				 */
int	   error_on_unregistered_send = 1;
int	   count_instructions;

int heap_series = HS_FIBONACCI;	/* Series to use for heap size. */

#ifdef INSTRUMENT
uint32 instr_send_sizes[INSTR_SEND_SIZES_MAX];
#endif

#ifdef JAM
int S_MIN_SIZE;			/* The minimum stack grain */
#endif
int H_MIN_SIZE;			/* The minimum heap grain */

static ErlExitStruct erl_exit_stack[ERL_MAX_EXITS];
static int erl_exit_sp;


#ifdef DEBUG
uint32 verbose;		/* noisy mode = 1 */
#endif
static uint32 progress = 0;	/* Print progress reports */

#ifdef SEQ_TRACE
uint32 system_seq_tracer;
#endif

#if defined(BEAM)

/* used to store some error information  while crashing in bifs */
uint32 error_info[7];

/* address.c */
EXTERN_FUNCTION(void, init_emulator, (_VOID_));

#endif

static FUNCTION(void, usage, (_VOID_));


/* static variables that must not change (use same values at restart) */
static char* program;
static char* init = "init";
static char* boot = "boot";
static int    boot_argc;
static char** boot_argv;

/*
** At exit (install exit function to handle clean termination)
*/

void erl_at_exit(func, arg)
FUNCTION(void, (*func),(void*)); void* arg;
{
    ASSERT(erl_exit_sp < ERL_MAX_EXITS);
    erl_exit_stack[erl_exit_sp].exit_proc = func;
    erl_exit_stack[erl_exit_sp].exit_arg = arg;
    erl_exit_sp++;
}


static char* get_arg(rest, next, ip)
char* rest; char* next; int* ip;
{
    if (*rest == '\0') {
	if (next == NULL) {
	    erl_printf(CERR, "too few arguments\n");
	    usage();
	}
	(*ip)++;
	return next;
    }
    return rest;
}


/* Called last in the at exit chain */
static void atexit_bottom(arg)
void* arg;
{
#ifdef DEBUG
    sys_free(tmp_buf);
    DEBUGF(("MEMORY REMAINING = %d\n", tot_allocated));
#endif
}


/* reset the system by calling all the exit handles */
void erl_reset()
{
#ifdef NOT_NOW_ANYWAY
    /*
     * XXX This doesn't work properly yet, and can cause a core
     * dump when exiting, which is confusing and will an incorrect
     * exit code to erlc.
     */

#ifndef PURIFY
    int i;

    for (i = erl_exit_sp-1; i >= 0; i--)
	(*erl_exit_stack[i].exit_proc)(erl_exit_stack[i].exit_arg);
#endif
#endif

    erl_exit_sp =  0;
}


/* initalize the system (may be used to reset the system) */
static void erl_init()
{
    uint32 mod;
    uint32 fun;
    uint32 args;
    int i;
    Process parent;
    Process* p;
    uint32 pid;
    ErlSpawnOpts so;
    ErlMessageBuffer* mbuf;
    uint32* hp;

    erl_exit_sp = 0;
    garbage_cols = 0;
    reclaimed = 0;
    erl_at_exit(atexit_bottom, NULL);

    /* Init of global variables */
    if (TMP_BUF_SIZE < TMP_BUF_MIN)
	TMP_BUF_SIZE = TMP_BUF_MIN;
    tmp_buf = (byte *)safe_alloc_from(130,TMP_BUF_SIZE * sizeof(byte));
    erl_progressf("+ Initializing fixed allocator\r\n");
    init_alloc();
    erl_progressf("+ Initializing garbage collector\r\n");
    init_gc();

    H_MIN_SIZE = next_heap_size(H_MIN_SIZE, 0);
#if defined(JAM)
    S_MIN_SIZE = next_heap_size(S_MIN_SIZE, 0);
#endif

    erl_progressf("+ Initializing atom table\r\n");
    init_atom_table();
    erl_progressf("+ Initializing db\r\n");
    init_db();
    erl_progressf("+ Initializing export table\r\n");
    init_export_table();
    erl_progressf("+ Initializing module table\r\n");
    init_module_table();
    erl_progressf("+ Initializing register table\r\n");
    init_register_table();
    erl_progressf("+ Initializing message buffers\r\n");
    init_message();
    erl_progressf("+ Initializing emulator\r\n");
    init_emulator();
    erl_progressf("+ Initializing scheduler\r\n");
    init_scheduler();
    erl_progressf("+ Initializing time queue\r\n");
    init_time();
    erl_progressf("+ Initializing distribution\r\n");
    init_dist();
    erl_progressf("+ Initializing I/O system\r\n");
    init_io();
    erl_progressf("+ Initializing tables for copy routines\r\n");
    init_copy();

    erl_progressf("+ Initializing loader\r\n");
    init_load();

#ifdef INSTRUMENT
    {int j;
     for (j=0;j<INSTR_SEND_SIZES_MAX;j++) 
	 instr_send_sizes[j]=0;
 }
#endif
    /*
     * Verify that the boot function (normally init:boot/1) exists.
     */
    if ((i = atom_get((byte*)init, sys_strlen(init))) == -1)
	erl_exit(3, "Module %s undefined\n", init);
    mod = make_atom(i);
    if ((i = atom_get((byte*)boot, sys_strlen(boot))) == -1)
	erl_exit(4, "Function %s undefined\n", boot);
    fun = make_atom(i);
    if (find_function(mod, fun, 1) == -1)
	erl_exit(5, "No function %s:%s/1\n", init, boot);

    erl_progressf("+ Creating init process\r\n");

    /*
     * Build arguments for mod:fun([A1,A2,...,An])
     */ 
    mbuf = new_message_buffer(boot_argc * 2 + 2);
    hp = mbuf->mem;
    args = NIL;
    for (i = boot_argc-1; i >= 0; i--) {
	int len = sys_strlen(boot_argv[i]);
	args = CONS(hp, am_atom_put(boot_argv[i], len), args);
	hp += 2;
    }
    args = CONS(hp, args, NIL);

    /*
     * We need a dummy parent process to be able to call erl_create_process().
     */
    parent.group_leader = NIL;
    parent.tracer_proc = NIL;	/* Not traced, of course. */
    parent.flags = 0;

    so.flags = 0;
    pid = erl_create_process(&parent, mod, fun, args, &so);
    p = process_tab[get_number(pid)];
    p->group_leader = pid;
    free_message_buffer(mbuf);

    erl_progressf("+ Entering scheduling loop\r\n");
}
 

/* be helpful (or maybe downright rude:-) */
static void usage()
{
    erl_printf(CERR, "usage: %s [flags] [ -- [init_args] ]\n", program);
    erl_printf(CERR, "The flags are:\n\n");
    erl_printf(CERR, "-v         turn on chatty mode, GCs will be reported etc\n");
    erl_printf(CERR, "-l         turn on auto load tracing\n");
    erl_printf(CERR, "-i module  set the boot module (default init)\n");
    erl_printf(CERR, "-b fun     set the boot function (default boot)\n");
#ifdef JAM
    erl_printf(CERR, "-s number  set minimum stack size in words (default %d)\n",
	       S_DEFAULT_SIZE);
#endif
    erl_printf(CERR, "-h number  set minimum heap size in words (default %d)\n",
	       H_DEFAULT_SIZE);
    erl_printf(CERR, "-# number  set the number of items to be used in traces etc\n");
    erl_printf(CERR, "-B         turn break handler off\n");
    erl_printf(CERR, "-P number  set maximum number of processes on this node\n");
    erl_printf(CERR, "           valid range is [%d-%d]\n",
	       MIN_PROCESS, MAX_PROCESS);
    erl_printf(CERR, "\n\n");
    erl_exit(-1, "");
}


void erl_start(argc,argv)
int argc; char **argv;
{
    int i = 1;
    char* arg=NULL;
    int have_break_handler = 1;
    char* tmpenvbuf;

    program = argv[0];
    display_items = 100;   /* approx chars */
    H_MIN_SIZE = H_DEFAULT_SIZE;
#ifdef JAM
    S_MIN_SIZE = S_DEFAULT_SIZE;
#endif
    max_process = MAX_PROCESS;
    display_loads = 0;
    tot_bin_allocated = 0;
    tmpenvbuf = getenv("ERL_GC_TYPE");
    switch_gc_threshold = 0;
    if (tmpenvbuf != NULL) {
	if (strcmp(tmpenvbuf,"generational") == 0)
	    switch_gc_threshold = 0;
	else if (strcmp(tmpenvbuf,"fullsweep") == 0)
	    switch_gc_threshold = MAX_SMALL; /* Never migrate */
	else if (strncmp(tmpenvbuf,"switch:",7) == 0)
	    switch_gc_threshold = atoi(tmpenvbuf+7);
	else
	    erl_printf(CERR, "warning: ERL_GC_TYPE is set to unknown"
		       " value, assuming generational.\n");
    }
    

#ifdef DEBUG
    verbose = 0;
#endif

#ifdef SEQ_TRACE
    system_seq_tracer = 0;
#endif

    while (i < argc) {
	if (argv[i][0] != '-') {
	    usage();
	}
	if (strcmp(argv[i], "--") == 0) { /* end of emulator options */
	    i++;
	    break;
	}
	switch (argv[i][1]) {
	case '#' :
	    arg = get_arg(argv[i]+2, argv[i+1], &i);
	    if ((display_items = atoi(arg)) == 0) {
		erl_printf(CERR, "bad display items%s\n", arg);
		usage();
	    }
	    VERBOSE(erl_printf(COUT,"using display items %d\n",
			       display_items););
	    break;

	case 'l':
	    display_loads++;
	    break;
	    
	case 'v':
#ifdef DEBUG
	    verbose++;
#else
	    erl_printf(CERR, "warning: -v (only in debug compiled code)\n");
#endif
	    break;
	case 'V' :
	    {
	       char tmp[100];

	       tmp[0] = tmp[1] = '\0';
#ifdef DEBUG
			  strcat(tmp, ",DEBUG");
#endif
#ifdef INSTRUMENT
			  strcat(tmp, ",INSTRUMENT");
#endif
	       erl_printf(CERR, "Erlang (%s) (%s) emulator version "
			  ERLANG_VERSION "\n", 
			  tmp+1,
			  EMULATOR);
	    }
	    break;

	case 'H':
	    if (argv[i][2] == 'f') {
		heap_series = HS_FIBONACCI;
	    } else if (argv[i][2] == 'T') {
		heap_series = HS_POWER_TWO;
	    } else if (argv[i][2] == 't') {
		heap_series = HS_POWER_TWO_MINUS_ONE;
	    } else {
		erl_printf(CERR, "bad heap series %s (use 'f', 'T', or 't')\n", arg);
		usage();
	    }
	    break;

	case 'h':
	    /* set default heap size */
	    arg = get_arg(argv[i]+2, argv[i+1], &i);
	    if ((H_MIN_SIZE = atoi(arg)) <= 0) {
		erl_printf(CERR, "bad heap size %s\n", arg);
		usage();
	    }
	    VERBOSE(erl_printf(COUT, "using minimum heap size %d\n",
			       H_MIN_SIZE););
	    break;

	case 's':
#if defined(JAM)
	    /* set default stack size */
	    arg = get_arg(argv[i]+2, argv[i+1], &i);
	    if ((S_MIN_SIZE = atoi(arg)) <= 0) {
		erl_printf(CERR, "bad stack size %s\n", arg);
		usage();
	    }
	    VERBOSE(erl_printf(COUT, "using minimum stack size %d\n", 
			       S_MIN_SIZE););
#endif
	    break;

	case 'i':
	    /* define name of module for initial function */
	    init = get_arg(argv[i]+2, argv[i+1], &i);
	    break;

	case 'b':
	    /* define name of initial function */
	    boot = get_arg(argv[i]+2, argv[i+1], &i);
	    break;

	case 'B':
	    have_break_handler = 0;
	    break;

	case 'p':
	    progress = 1;
	    break;

	case 'P':
	    /* set maximum number of processes */
	    arg = get_arg(argv[i]+2, argv[i+1], &i);
	    if (((max_process = atoi(arg)) < MIN_PROCESS) ||
		(max_process > MAX_PROCESS)) {
		erl_printf(CERR, "bad number of processes %s\n", arg);
		usage();
	    }
	    break;

	case 'n':   /* XXX obsolete */
	    break;

	case 'W':
	    error_on_unregistered_send = 1;
	    break;
	case 'w':
	    error_on_unregistered_send = 0;
	    break;
	case 'c':
	    if (argv[i][2] == 'i') {
		count_instructions = 1;
	    }
	    break;
	default:
	    erl_printf(CERR, "%s unknown flag %s\n", argv[0], argv[i]);
	    usage();
	}
	i++;
    }

    /* Restart will not reinstall the break handler */
    if (have_break_handler)
	init_break_handler();

    boot_argc = argc - i;  /* Number of arguments to init */
    boot_argv = &argv[i];
    erl_init();
}

#if defined(__STDC__) || defined(_MSC_VER)
void
erl_progressf(char* fmt, ...)
#else
void
erl_progressf(fmt, va_alist)
char* fmt;
va_dcl
#endif /* __STDC__ */
{
    char sbuf[1024];		/* Temporary buffer. */
    va_list va;
    
    if (progress) {
	VA_START(va, fmt);
	vsprintf(sbuf, fmt, va);
	va_end(va);
	erl_printf(COUT, "%s", sbuf);
    }
}
