/* ``The contents of this file are subject to the Erlang Public License,
 * Version 1.0, (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.erlang.org/EPL1_0.txt
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 * 
 * The Original Code is Erlang-4.7.3, December, 1998.
 * 
 * The Initial Developer of the Original Code is Ericsson Telecom
 * AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
 * Telecom AB. All Rights Reserved.
 * 
 * Contributor(s): ______________________________________.''
 */
/* 
 * This file is copyright (c) Ellemtel in October 1994
 *
 * Special code for very large dictionaries 
 * Author Claes Wikstrom, klacke@erix.ericsson.se
 * Rewritten by Tony Rogvall, tony@erix.ericsson.se (1996)
 */

#include "sys.h"
#include "config.h"
#include "global.h"
#include "erl_process.h"
#include "error.h"
#include "db.h"
#include "bif.h"
#include "big.h"

#define MEM_COST_GROUP 3

/*  #define HARDDEBUG 1  */

#define KEY   1
#define VALUE 2
#define NBIND 8  

#define SZEXP   8
#define SEGSZ   (1 << SZEXP)
#define SZMASK  ((1 << SZEXP)-1)


#define SEG_LEN         256   /* When growing init segs */
#define SEG_INCREAMENT  128   /* Number of segments to grow */

#define BUCKET(tb, i) (tb)->seg[(i) >> SZEXP][(i) & SZMASK]

/* ix is a NAME parameter :-) */
#define HASH(tb, hval, ix) \
  do { \
     if ((ix = ((hval) & (tb)->szm)) < (tb)->p) \
        ix = (hval) & (((tb)->szm << 1) | 1); \
  } while(0)

/* optimised version of make_hash (normal case? atomic key) */
#define MAKE_HASH(term) \
    (is_atom(term) ? (atom_tab(unsigned_val(term))->slot.bucket.hvalue) : \
     make_hash(term, 0))



#define GETKEY(tb, tplp)   (*((tplp) +  tb->keypos))
#define ZEROB(bind)        sys_memzero(bind.ptr, sizeof(uint32)*bind.size)

/* optimised version of copy_object (normal case? atomic object) */
#define COPY_OBJECT(obj, p, objp) \
   if (IS_CONST(obj)) { *(objp) = (obj); } \
   else { copy_object(obj, p, 0, objp, (Process*) 0); }

/* optimised version of eq, first check for equality of pointers !! */
#define EQ(a,b) ( ((a) == (b)) ? 1 : eq(a,b) )

#define DB_READ  (DB_PROTECTED|DB_PUBLIC)
#define DB_WRITE DB_PUBLIC
#define DB_INFO  (DB_PROTECTED|DB_PUBLIC|DB_PRIVATE)

#ifdef HARDDEBUG
#define CHECK_TABLES() check_tables()
#else 
#define CHECK_TABLES()
#endif

/* This is a static hashlist of all tables we have */

static struct tab_entry {
    DbTable *t;
    uint32 id;            /* Automatically initialized */
    uint32 name;          /* An atom */
} db_tables[DB_MAX_TABS];  /* Local variable db_tables */

struct bindings {
    int size;
    uint32 *ptr;
};


/* Forward decls */
FUNCTION(static int, has_variable, (uint32));
static FUNCTION(void, free_table, (DbTable*));

static int no_tabs;		/* Number of active tables */
static uint32 am_eot;		/* Atom '$end_of_table' */

/* The id in a tab_entry slot is                                       */
/* 0 if it's never been used                                             */
/* USED if it's been freed                                                  */
/* This is so that we shall be able to terminate a search when we        */
/* reach a point in the table that is impossible to reach if the id    */
/* is there, we have to consider that tables can be removed thogh, so if */
/* we come to a removed slot, we must continue the search                */

#define USED 0xffffffff
#define ISFREE(i) ((db_tables[i].id == USED) || (db_tables[i].id == 0))
#define ISNOTUSED(i) (db_tables[i].id == 0)

/* Cleaup db */

static void atexit_db(arg)
void* arg;
{
    int i;

    DEBUGF(("atexit_db: live tables = %d\n", no_tabs));

    for (i = 0; i < 0; i++) {
	DbTable* t = db_tables[i].t;
	if (t != NULL) {
	    free_table(t);
	    db_tables[i].id = 0;
	    db_tables[i].t = NULL;
	}
    }
}

/* Init the db */

void init_db()
{
    int i;

    no_tabs = 0;
    for (i=0; i<DB_MAX_TABS; i++) {
	db_tables[i].id = 0;
	db_tables[i].t = NULL;
    }
    am_eot = am_magic_word("$end_of_table");
    erl_at_exit(atexit_db, NULL);
}

/* Used by match funcs to extract key from pattern */

static uint32 getkey(tb, obj)
DbTable *tb; uint32 obj;
{
    if (is_tuple(obj)) {
	uint32 *tptr = ptr_val(obj);
	if (arityval(*tptr) >= tb->keypos)
	    return *(tptr + tb->keypos);
    }
    return 0;
}


static DbTable* get_table(p, id, what)
Process* p; uint32 id; int what;
{
    int i, j;

    if (!is_table_id(id)) {
	return NULL;
    }

    i = j = unsigned_val(id) % DB_MAX_TABS;
    while (1) {
	if (db_tables[i].id == id) {
	    DbTable* tb = db_tables[i].t;
	    if ((tb->status & what) != 0 || p->id == tb->owner) {
		return tb;
	    }
	    return NULL;
	}
	if (ISNOTUSED(i++))
	    return NULL;
	if (i == DB_MAX_TABS) 
	    i = 0; 
	if (i == j)
	    return NULL;
    }
    return NULL;
}

static DbTerm** alloc_seg()
{
    DbTerm** bp;
    int sz = sizeof(DbTerm*)*SEGSZ;

    if ((bp = (DbTerm**) sys_alloc_from(51,sz)) == NULL)
	return NULL;
    memset(bp, 0, sz);
    return bp;
}

/*
** Copy term into table (possibly reallocate)
*/
static DbTerm* get_term(tb, old, obj, hval)
DbTable* tb; DbTerm* old; uint32 obj; uint32 hval;
{
    int size = size_object(obj);
    DbTerm* p;
    uint32 copy;
    uint32* top;

    if (old != 0) {
	ProcBin* ptr = old->mso;
	/* release old binary */
	while(ptr) {
	    ProcBin* next = ptr->next;
	    maybe_delete_contents(ptr);
	    fix_free(proc_bin_desc, (uint32*) ptr);
	    ptr = next;
	}
	if (size != old->size)
	    p = (DbTerm*) safe_realloc((char*)old,
				       sizeof(DbTerm)+sizeof(uint32)*(size-1));
	else
	    p = old;
    }
    else
	p = (DbTerm*) safe_alloc(sizeof(DbTerm)+sizeof(uint32)*(size-1));

    p->hvalue = hval;
    p->size = size;
    p->mso = 0;

    top = p->v;
    copy = copy_struct(obj, size, &top, &p->mso);
    p->tpl = ptr_val(copy);
    return p;
}

/*
** Copy terms from ptr1 until ptr2
** works for ptr1 == ptr2 == 0  => []
** or ptr2 == 0
*/
static uint32 put_term_list(p, ptr1, ptr2)
Process* p; DbTerm* ptr1; DbTerm* ptr2;
{
    int sz = 0;
    DbTerm* ptr;
    uint32 list = NIL;
    uint32 copy;
    uint32* hp;

    ptr = ptr1;
    while(ptr != ptr2) {
	sz += ptr->size + 2;
	ptr = ptr->next;
    }

    hp = HAlloc(p, sz);

    ptr = ptr1;
    while(ptr != ptr2) {
	copy = copy_shallow(ptr->v, ptr->size, &hp, &p->mso);
	list = CONS(hp, copy, list);
	hp  += 2;
	ptr = ptr->next;
    }
    return list;
}

static void free_term(p)
DbTerm* p;
{
    ProcBin* ptr = p->mso;

    while(ptr) {
	ProcBin* next = ptr->next;

	maybe_delete_contents(ptr);
	fix_free(proc_bin_desc, (uint32*) ptr);
	ptr = next;
    }
    sys_free(p);
}


static void grow(tb)
DbTable* tb;
{
    DbTerm** bp;
    DbTerm** bps;
    DbTerm* b;
    int ix;
    int nszm = (tb->szm << 1) | 1;

    /* Ensure that that the slot nacive exists */
    if (tb->nactive >= tb->nslots) {
	/* Time to get a new array */    
	if ((tb->nactive & SZMASK) == 0) {
	    int next = tb->nactive >> SZEXP;
	    DbTerm** new_segment = alloc_seg();
	    DbTerm*** new_seg;

	    if (new_segment == NULL)
		return;

	    if (next == tb->nsegs) {
		int i, sz;

		if (tb->nsegs == 1)
		    sz = SEG_LEN;
		else
		    sz = tb->nsegs + SEG_INCREAMENT;
		new_seg = (DbTerm***) sys_realloc(tb->seg,
						  sizeof(DbTerm**)*sz);
		if (new_seg == NULL) {
		    sys_free(new_segment);
		    return;
		}

		tb->seg = new_seg;
		tb->nsegs = sz;
		for (i = next+1; i < sz; i++)
		    tb->seg[i] = 0;
	    }
	    tb->seg[next] = new_segment;
	    tb->nslots += SEGSZ;
	}
    }

    ix = tb->p;
    bp = &BUCKET(tb, ix);
    ix += (tb->szm+1);
    bps = &BUCKET(tb, ix);
    b = *bp;

    while (b != 0) {
	ix = b->hvalue & nszm;

	if (ix == tb->p)
	    bp = &b->next;
	else {
	    *bp = b->next;  	    /* unlink */
	    b->next = *bps;         /* link */
	    *bps = b;
	}
	b = *bp;
    }

    tb->nactive++;
    if (tb->p == tb->szm) {
	tb->p = 0;
	tb->szm = nszm;
    }
    else
	tb->p++;
}


/*
** Shrink the hash table
** Remove segments if they are empty
** but do not reallocate the segment index table !!!
*/
static void shrink(tb)
DbTable* tb;
{
    DbTerm** bp;

    if (tb->nactive == SEGSZ)
	return;

    tb->nactive--;
    if (tb->p == 0) {
	tb->szm >>= 1;
	tb->p = tb->szm;
    }
    else
	tb->p--;

    bp = &BUCKET(tb, tb->p);
    while(*bp != 0) bp = &(*bp)->next;

    *bp = BUCKET(tb, tb->nactive);
    BUCKET(tb, tb->nactive) = 0;

    if ((tb->nactive & SZMASK) == SZMASK) {
	int six = (tb->nactive >> SZEXP)+1;

	sys_free(tb->seg[six]);
	tb->seg[six] = 0;
	tb->nslots -= SEGSZ;
    }
}


/* release all memory occupied by a single table */

static void free_table(tb)
DbTable* tb;
{
    DbTerm*** sp = tb->seg;
    int n = tb->nsegs;

    no_tabs--;

    while(n--) {
	DbTerm** bp = *sp;
	if (bp != 0) {
	    int m = SEGSZ;

	    while(m--) {
		DbTerm* p = *bp++;

		while(p != 0) {
		    DbTerm* next = p->next;
		    free_term(p);
		    p = next;
		}
	    }
	    sys_free(*sp);
	}
	sp++;
    }
    sys_free(tb->seg);
    fix_free(table_desc, (uint32*) tb);
}


/* Search a list of tuples for a matching key */

static DbTerm* search_list(tb, key, hval, list)
DbTable* tb; uint32 key; uint32 hval; DbTerm* list;
{
    while (list != 0) {
	if ((list->hvalue == hval) && EQ(key, GETKEY(tb, list->tpl)))
	    return list;
	list = list->next;
    }
    return 0;
}


/* This function is called by the next AND the match BIF */
/* It return the next object in a table                   */

static DbTerm* next(tb, iptr, list)
DbTable* tb; int *iptr; DbTerm* list;
{
    int i;

    list = list->next;

    if (list != 0)
	return list;
    i = *iptr + 1;
    while (i < tb->nactive) {
	if ((list = BUCKET(tb,i)) != 0) {
	    *iptr = i;
	    return list;
	}
	i++;
    }
    *iptr = i;
    return 0;
}


/*
** Check if object represents a "match" variable 
** i.e and atom $N where N is an integer < NBIND
**
*/

static int variable(obj)
uint32 obj;
{
    byte *b;
    int n;
    int N;

    if (is_not_atom(obj))
        return -1;
    b = atom_tab(unsigned_val(obj))->name;
    if ((n = atom_tab(unsigned_val(obj))->len) < 2)
        return -1;
    if (*b++ != '$')
        return -1;
    n--;
    /* Handle first digit */
    if (*b == '0')
        return (n == 1) ? 0 : -1;
    if (*b >= '1' && *b <= '9')
        N = *b++ - '0';
    else
        return -1;
    n--;
    while(n--) {
        if (*b >= '0' && *b <= '9') {
            N = N*10 + (*b - '0');
            b++;
        }
        else
            return -1;
    }
    return N;
}


/* check if obj is (or contains) a variable */
/* return 1 if obj contains a variable or underscore */
/* return 0 if obj is fully ground                   */

static int has_variable(obj)
uint32 obj;
{
    switch(tag_val_def(obj)) {
    case LIST_DEF: {
	while (is_list(obj)) {
	    if (has_variable(*ptr_val(obj)))
		return 1;
	    obj = *(ptr_val(obj) +1);
	}
	return(has_variable(obj));  /* Non wellformed list or [] */
    }
    case TUPLE_DEF: {
	uint32* tuple = ptr_val(obj);
	int arity = arityval(*tuple++);
	while(arity--) {
	    if (has_variable(*tuple))
		return 1;
	    tuple++;
	}
	return(0);
    }
    case ATOM_DEF:
	if (obj == am_underscore || variable(obj) >= 0)
	    return 1;
    }
    return 0;
}

/* If the index for a variable is > than size of bindings */
/* we allocate more space for the bindings                */

static void grow_bindings(bs, need)
struct bindings *bs;
int need;
{
    int newsz;
    uint32 *uptr;

    if (bs == NULL) return;
    newsz = need * 2;

    if (bs->size != NBIND) { /* Not the first time */
	uptr = (uint32*) sys_realloc(bs->ptr, newsz * sizeof(uint32));
	sys_memzero(uptr + bs->size, (newsz - bs->size) * sizeof(uint32));
	bs->ptr = uptr;
    }
    else {   /* For the first time */
	uptr = (uint32*) sys_alloc_from(52,newsz * sizeof(uint32));
	sys_memzero(uptr + bs->size, (newsz - bs->size) * sizeof(uint32));
	sys_memcpy(uptr, bs->ptr,  NBIND * sizeof(uint32));
	bs->ptr = uptr;
    }
    bs->size = newsz;
}



/* Return 1 if obj match pattern, as well as fill in bindings */
/*        0 if no match                                       */
/* Possibly reallocs the bindings                             */

static int match(obj, pattern, bs, sz_bs)
uint32 obj; uint32 pattern; struct bindings *bs; struct bindings *sz_bs;
{
    uint32 *aa,*bb;
    int i;

    if ((i = variable(pattern)) >= 0) { /* Binding time */
	if (i >= bs->size) { /* We gotta allocate  new space      */
			      /* for the variable bindings         */
	    grow_bindings(bs, i);
	    grow_bindings(sz_bs, i);
	}

	if (bs->ptr[i] == 0) {  /* Var match anything */
	    bs->ptr[i] = obj;
	    return 1;  
	}
	if (EQ(bs->ptr[i], obj))   /* Already bound !! */
	    return 1;
	return 0; /* Allready bound to different value */
    }

    if ((obj == pattern) || (pattern == am_underscore)) 
	return 1;
    if (not_eq_tags(obj,pattern)) 
	return 0; 

    switch(tag_val_def(obj)) {
    case LIST_DEF:
	aa = ptr_val(obj);
	bb = ptr_val(pattern);
	while (1) {
	    if (!match(*aa++, *bb++,  bs, sz_bs)) 
		return 0;
	    if (*aa == *bb)
		return 1;
	    if (is_not_list(*aa) || is_not_list(*bb)) 
		return(match(*aa, *bb,  bs, sz_bs));
	    aa = ptr_val(*aa);
	    bb = ptr_val(*bb);
	}
    case TUPLE_DEF:
	aa = ptr_val(obj);
	bb = ptr_val(pattern);
	if (*aa != *bb) 
	    return 0;         /* different arities */
	i = arityval(*aa);    /* get the arity*/
	while (i--) {
	    if (match(*++aa, *++bb,  bs, sz_bs) == 0)
		return 0;
	}
	return 1;
    default:
	return(EQ(obj, pattern));
    }
}


/* Called when  a process which has created any tables dies */
/* So that we can remove the tables ceated by the process   */

void db_proc_dead(pid)
uint32 pid;
{
    int i;

    for (i=0; i<DB_MAX_TABS; i++) {
	DbTable* tb = db_tables[i].t;
	if ((tb != NULL)  && (tb->owner == pid)) {
	    free_table(tb);
	    db_tables[i].id = USED;
	    db_tables[i].t = NULL;
	}
    }
}

/**********************************************************************/

BIF_RETTYPE db_fixtable_2(BIF_ALIST_2)
BIF_ADECL_2
{
    DbTable* tb;
    uint32 arg;

    if ((tb = get_table(BIF_P, BIF_ARG_1, DB_WRITE)) == NULL) {
	BIF_ERROR2(BADARG, am_db_fixtable, BIF_ARG_1, BIF_ARG_2);
    }
    arg = BIF_ARG_2;
    if (arg == am_true) 
      tb->status |= DB_FIXED;
    else if (arg == am_false)
      tb->status &= ~DB_FIXED;
    else {
	BIF_ERROR2(BADARG, am_db_fixtable, BIF_ARG_1, BIF_ARG_2);
    }
    BIF_RET(am_true);
}

/**********************************************************************/

/* Returns the first Key in a table */
BIF_RETTYPE db_first_1(BIF_ALIST_1)
BIF_ADECL_1
{
    DbTable* tb;
    int i = 0;

    if ((tb = get_table(BIF_P, BIF_ARG_1, DB_READ)) == NULL) {
	BIF_ERROR1(BADARG, am_db_first, BIF_ARG_1);
    }

    while (i < tb->nactive) {
	DbTerm* list = BUCKET(tb, i);
	if (list != 0) {
	    uint32 key = GETKEY(tb, list->tpl);
	    uint32 copy;

	    COPY_OBJECT(key, BIF_P, &copy);
            BIF_RET(copy);
	}
	i++;
    }
    BIF_RET(am_eot);
}

/**********************************************************************/
/* The next BIF, given a key, return the "next" key */

BIF_RETTYPE db_next_key_2(BIF_ALIST_2)
BIF_ADECL_2
{
    DbTable* tb;
    HashValue hval;
    int ix;
    uint32 key;
    DbTerm* b1;

    if ((tb = get_table(BIF_P, BIF_ARG_1, DB_READ)) == NULL) {
	BIF_ERROR2(BADARG, am_db_next_key, BIF_ARG_1, BIF_ARG_2);
    }

    key = BIF_ARG_2;
    hval = MAKE_HASH(key);
    HASH(tb, hval, ix);
    b1 = BUCKET(tb, ix);

    while(b1 != 0) {
	if ((b1->hvalue == hval) && EQ(key, GETKEY(tb, b1->tpl))) {
	    DbTerm* b2 = next(tb, &ix, b1);
	    if ((tb->status & DB_BAG) || (tb->status & DB_DUPLICATE_BAG)) {
		while (b2 != 0) {
		    uint32 key2 = GETKEY(tb, b2->tpl);
		    if (EQ(key, key2)) {
			b2 = next(tb, &ix, b2);
			continue;
		    }
		    break;
		}
	    }
	    if (b2 == 0) {
		BIF_RET(am_eot);
	    }
	    else {
		uint32 copy;

		COPY_OBJECT(GETKEY(tb, b2->tpl), BIF_P, &copy);
		BIF_RET(copy);
	    }
	}
	b1 = b1->next;
    }
    BIF_ERROR2(BADARG, am_db_next_key, BIF_ARG_1, BIF_ARG_2);
}

/**********************************************************************/
/* update_counter(Tab, Key, Increment) 
** Returns new value (integer)
*/

#define FIX_BIG_SIZE 16
#define MAX_NEED(x,y) (((x)>(y)) ? (x) : (y))

static uint32  big_tmp[2];
static uint32  big_buf[FIX_BIG_SIZE];

static uint32 add_counter(counter, incr)
uint32 counter; uint32 incr;
{
    uint32 res;
    sint32 ires;
    uint32 arg1;
    uint32 arg2;
    uint32 sz1;
    uint32 sz2;
    uint32 need;
    uint32* ptr;
    int i;

    if (is_small(counter) && is_small(incr)) {
	ires = signed_val(counter) + signed_val(incr);
	if (IS_SSMALL(ires))
	    return make_small(ires);
	else
	    return small_to_big(ires, big_buf);
    }
    else {
#if defined(BEAM)
        if (is_nil(counter) || is_nil(incr)) {
	    return 0;
        }
#endif /* BEAM */
	switch(i = NUMBER_CODE(counter, incr)) {
	case SMALL_BIG:
	    arg1 = small_to_big(signed_val(counter), big_tmp);
	    arg2 = incr;
	    break;
	case BIG_SMALL:
	    arg1 = counter;
	    arg2 = small_to_big(signed_val(incr), big_tmp);
	    break;
	case BIG_BIG:
	    arg1 = incr;
	    arg2 = counter;
	    break;
	default:
	    return 0;
	}
	sz1 = big_size(arg1);
	sz2 = big_size(arg2);
	sz1 = MAX_NEED(sz1,sz2)+1;
	need = BIG_NEED_SIZE(sz1);
	if (need <= FIX_BIG_SIZE)
	    ptr = big_buf;
	else {
	    if ((ptr = sys_alloc_from(53,need*sizeof(uint32))) == NULL)
		return NIL;  /* system limit */
	}
	res = big_plus(arg1, arg2, ptr);
	if (is_small(res) || is_nil(res)) {
	    if (ptr != big_buf)
		sys_free(ptr);
	}
	return res;
    }
}

/*
** Copy old counter with counter set to ZERO
** but first make the new space to the correct size
** return -1 on memory failure
**         0 otherwise
*/
static int realloc_counter(bp, b, sz, new_counter, keypos)
DbTerm** bp; DbTerm* b; uint32 sz; uint32 new_counter; int keypos;
{
    DbTerm* new;
    uint32  old_counter;
    uint32  old_sz;
    uint32  new_sz;
    uint32  basic_sz;
    uint32  copy;
    uint32* top;
    uint32* ptr;

    old_counter = b->tpl[keypos+1];

    if (is_small(old_counter))
	old_sz = 0;
    else {
	top = ptr_val(old_counter);
	old_sz = BIG_ARITY(top) + 1;
	if (sz == old_sz) {  /* OK we fit in old space */
	    sys_memcpy(top, ptr_val(new_counter), sz*sizeof(uint32));
	    return 0;
	}
    }

    basic_sz = b->size - old_sz;
    new_sz = basic_sz + sz;

    new = (DbTerm*) safe_alloc(sizeof(DbTerm)+sizeof(uint32)*(new_sz-1));
    if (new == NULL)
	return -1;
    new->next = b->next;
    new->hvalue = b->hvalue;
    new->size = new_sz;
    new->mso  = 0;
    top = new->v;

    b->tpl[keypos+1] = SMALL_ZERO;               /* zap, do not copy */

    /* copy term (except old counter) */
    copy = copy_struct(make_tuple(b->tpl), basic_sz, &top, &new->mso);
    new->tpl = ptr_val(copy);

    free_term(b);  /* free old term */
    *bp = new;     /* patch new */

    /* copy new counter */
    if (sz == 0)
	new->tpl[keypos+1] = new_counter;  /* must be small !!! */
    else {
	ptr = ptr_val(new_counter);
	sys_memcpy(top, ptr, sz*sizeof(uint32));
	new->tpl[keypos+1] = make_big(top);
    }
    return 0;
}


BIF_RETTYPE db_update_counter_3(BIF_ALIST_3)
BIF_ADECL_3
{
    DbTable* tb;
    DbTerm* b;
    DbTerm** bp;
    int ix;
    uint32 hval;
    uint32 key;
    uint32 incr;
    uint32 counter;
    uint32* counterp;
    uint32 res;

    if ((tb = get_table(BIF_P, BIF_ARG_1, DB_WRITE)) == NULL) {
	BIF_ERROR3(BADARG, am_db_update_counter,BIF_ARG_1,BIF_ARG_2,BIF_ARG_3);
    }
    if (!(tb->status & DB_SET)) {
	BIF_ERROR3(BADARG, am_db_update_counter,BIF_ARG_1,BIF_ARG_2,BIF_ARG_3);
    }

    key = BIF_ARG_2;
    hval = MAKE_HASH(key);
    HASH(tb, hval, ix);
    bp = &BUCKET(tb, ix);
    b = *bp;

    while(b != 0) {
	if ((b->hvalue == hval) && EQ(key,GETKEY(tb, b->tpl)))
	    break;
	bp = &b->next;
	b = *bp;
    }

    if (b == 0) {
	BIF_ERROR3(BADARG, am_db_update_counter,BIF_ARG_1,BIF_ARG_2,BIF_ARG_3);
    }

    if (make_arityval(tb->keypos+1) != *b->tpl) {
	BIF_ERROR3(BADARG, am_db_update_counter,BIF_ARG_1,BIF_ARG_2,BIF_ARG_3);
    }

    incr = BIF_ARG_3;
    counterp = b->tpl + tb->keypos + 1;
    counter = *counterp;

    if ((res = add_counter(counter, incr)) == NIL) {
	BIF_ERROR3(SYSTEM_LIMIT, am_db_update_counter,
		   BIF_ARG_1,BIF_ARG_2,BIF_ARG_3);
    }
    else if (res == 0) {
	BIF_ERROR3(BADARG, am_db_update_counter,BIF_ARG_1,BIF_ARG_2,BIF_ARG_3);
    }

    if (is_small(res)) {
	if (is_small(counter))
	    *counterp = res;
	else {
	    if (realloc_counter(bp, b, 0, res, tb->keypos) < 0) {
		BIF_ERROR3(SYSTEM_LIMIT, am_db_update_counter,
			   BIF_ARG_1,BIF_ARG_2,BIF_ARG_3);
	    }
	}
	BIF_RET(res);
    }
    else {
	uint32* ptr = ptr_val(res);
	uint32 sz = BIG_ARITY(ptr) + 1;
	uint32* hp;

	if (realloc_counter(bp, b, sz, res, tb->keypos) < 0) {
	    BIF_ERROR3(SYSTEM_LIMIT, am_db_update_counter,
		       BIF_ARG_1,BIF_ARG_2,BIF_ARG_3);
	}
	hp = HAlloc(BIF_P, sz);
	sys_memcpy(hp, ptr, sz*sizeof(uint32));
	res = make_big(hp);
	hp += sz;
	if (ptr != big_buf)
	    sys_free(ptr);
	BIF_RET(res);
    }
}

/**********************************************************************/
/* The put BIF */
    
BIF_RETTYPE db_put_2(BIF_ALIST_2)
BIF_ADECL_2
{
    DbTable* tb;
    uint32 hval;
    int ix;
    uint32 key;
    DbTerm** bp;
    DbTerm* b;
    DbTerm* q;

    CHECK_TABLES();

    if ((tb = get_table(BIF_P, BIF_ARG_1, DB_WRITE)) == NULL) {
	BIF_ERROR2(BADARG, am_db_put, BIF_ARG_1, BIF_ARG_2);
    }
    if (is_not_tuple(BIF_ARG_2) || 
	(arityval(*ptr_val(BIF_ARG_2)) < tb->keypos)) {
	BIF_ERROR2(BADARG, am_db_put, BIF_ARG_1, BIF_ARG_2);
    }

    key = GETKEY(tb, ptr_val(BIF_ARG_2));
    hval = MAKE_HASH(key);
    HASH(tb, hval, ix);
    bp = &BUCKET(tb, ix);
    b = *bp;

    while(b != 0) {
	if ((b->hvalue == hval) && EQ(key, GETKEY(tb, b->tpl))) {
	    if (tb->status & DB_SET) {
		DbTerm* bnext = b->next;
		q = get_term(tb, b, BIF_ARG_2, hval);
		q->next = bnext;
		*bp = q;
		BIF_RET(am_true);
	    }
	    else if (tb->status & DB_BAG) {
		DbTerm** tp = bp;
                DbTerm* p = b;
		
                if (eq(make_tuple(b->tpl), BIF_ARG_2)) {
                    BIF_RET(am_true);
                }
                bp = &b->next;
                b = b->next;
                while ((b != 0) && (b->hvalue == hval) && 
                       EQ(key, GETKEY(tb, b->tpl))) {
                    if (eq(make_tuple(b->tpl), BIF_ARG_2)) {
                        BIF_RET(am_true);
                    }
                    bp = &b->next;
                    b = b->next;
                }

                q = get_term(tb, 0, BIF_ARG_2, hval);
                q->next = p;
                *tp = q;
		goto Lupdate;
	    }
	    else {  /* if (tb->status & DB_DUPLICATE_BAG) */
		q = get_term(tb, 0, BIF_ARG_2, hval);
		q->next = b;
		*bp = q;
		goto Lupdate;
	    }
	}
	bp = &b->next;
	b = b->next;
    }

    q = get_term(tb, 0, BIF_ARG_2, hval);
    q->next = b;
    *bp = q;

 Lupdate:
    tb->nitems++;

    if ( ((tb->nitems / tb->nactive) > CHAIN_LEN) && 
	((tb->status & DB_FIXED) == 0))
	grow(tb);
    CHECK_TABLES();
    BIF_RET(am_true);
}

/**********************************************************************/
/* The create table BIF     */
/* Args: (Name, Properties) */

static int last_slot = 0;

BIF_RETTYPE db_create_2(BIF_ALIST_2)
BIF_ADECL_2
{
    DbTable* tb;
    int slot;
    uint32 list;
    uint32 val;
    uint32 ret;
    uint32 status;
    int keypos;
    int is_named;

    if (is_not_atom(BIF_ARG_1)) {
	BIF_ERROR2(BADARG, am_db_create, BIF_ARG_1, BIF_ARG_2);
    }
    if (is_not_nil(BIF_ARG_2) && is_not_list(BIF_ARG_2)) {
	BIF_ERROR2(BADARG, am_db_create, BIF_ARG_1, BIF_ARG_2);
    }
    if (no_tabs > DB_MAX_TABS * 0.8) {
	erl_printf(CBUF, "** Too many db tables **\n");
	send_error_to_logger(BIF_P->group_leader);
	BIF_ERROR2(SYSTEM_LIMIT, am_db_create, BIF_ARG_1, BIF_ARG_2);
    }

    status = DB_NORMAL | DB_SET | DB_LHASH | DB_PROTECTED;
    keypos = 1;
    is_named = 0;

    list = BIF_ARG_2;
    while(is_list(list)) {
	val = *ptr_val(list);
	if (val == am_bag) {
	    status |= DB_BAG;
	    status &= ~(DB_SET | DB_DUPLICATE_BAG);
	}
	else if (val == am_duplicate_bag) {
	    status |= DB_DUPLICATE_BAG;
	    status &= ~(DB_SET | DB_BAG);
	}
	else if (is_tuple(val)) {
	    uint32* tp = ptr_val(val);

	    if ((arityval(tp[0]) == 2) && (tp[1] == am_keypos) &&
		is_small(tp[2]) && (signed_val(tp[2]) > 0)) {
		keypos = signed_val(tp[2]);
	    }
	    else {
		BIF_ERROR2(BADARG, am_db_create, BIF_ARG_1, BIF_ARG_2);
	    }
	}
	else if (val == am_public) {
	    status |= DB_PUBLIC;
	    status &= ~DB_PROTECTED;
	}
	else if (val == am_private) {
	    status |= DB_PRIVATE;
	    status &= ~DB_PROTECTED;
	}
	else if (val == am_named_table) {
	    is_named = 1;
	}
	else if (val == am_set || val == am_protected)
	    ;
	else {
	    BIF_ERROR2(BADARG, am_db_create, BIF_ARG_1, BIF_ARG_2);
	}
	list = *(ptr_val(list) + 1);
    }
    if (is_not_nil(list)) /* it must be a well formed list */
	    BIF_ERROR2(BADARG, am_db_create, BIF_ARG_1, BIF_ARG_2);

    /* allocate the slot for the table */
    if (++last_slot == DB_MAX_TABS) 
	last_slot = 0;
    if (is_named) {
	slot = unsigned_val(BIF_ARG_1) % DB_MAX_TABS;
	while (1) {
	    if (ISFREE(slot)) {
		ret = BIF_ARG_1;
		break;
	    }
	    if (db_tables[slot].id == BIF_ARG_1) {
		BIF_ERROR2(BADARG, am_db_create, BIF_ARG_1, BIF_ARG_2);
	    }
	    if (++slot == DB_MAX_TABS) {
		slot=0; 
	    }
	}
    }
    else {  /* Allocate number slot */
	slot = last_slot;
	while(1) {
	    if (ISFREE(slot)) {
		ret = make_small(slot);
		break;
	    }
	    if (++slot == DB_MAX_TABS) {
		slot=0; 
	    }
	}
    }

    /* Now slot and ret are properly set */
    /* ret will be the id for the table as well */
    /* Creat a new table and insert in db_tables */

    tb = (DbTable*) fix_alloc_from(55,table_desc);
    tb->id = ret;
    tb->the_name = BIF_ARG_1;
    tb->status = status;
    tb->keypos = keypos;
    tb->owner = BIF_P->id;

    tb->szm = SZMASK;
    tb->nslots = SEGSZ;
    tb->nactive = SEGSZ;
    tb->nitems = 0;
    tb->p = 0;
    tb->nsegs = 1;
    tb->seg = (DbTerm***) sys_alloc_from(54,sizeof(DbTerm**));
    tb->seg[0] = alloc_seg();

    tb->slot = slot;           /* store slot for erase */
    db_tables[slot].id = ret;  /* Insert the table */
    db_tables[slot].t = tb;
    
    BIF_P->flags |= F_USING_DB;        /* So we can remove tb if p dies */

    no_tabs++;

    BIF_RET(ret);
}

/**********************************************************************/
    
/* The lookup BIF */
BIF_RETTYPE db_get_2(BIF_ALIST_2)
BIF_ADECL_2
{
    DbTable* tb;
    HashValue hval;
    int ix;
    uint32 key;
    DbTerm* b1;

    if ((tb = get_table(BIF_P, BIF_ARG_1, DB_READ)) == NULL) {
	BIF_ERROR2(BADARG, am_db_get, BIF_ARG_1, BIF_ARG_2);
    }

    key = BIF_ARG_2;
    hval = MAKE_HASH(key);
    HASH(tb, hval, ix);
    b1 = BUCKET(tb, ix);

    while(b1 != 0) {
	if ((b1->hvalue == hval) && EQ(key, GETKEY(tb, b1->tpl))) {
	    DbTerm* b2 = b1->next;
	    uint32 copy;

	    if ((tb->status & DB_BAG) || (tb->status & DB_DUPLICATE_BAG)) {
		while((b2 != 0) && (b2->hvalue == hval) &&
		      EQ(key, GETKEY(tb, b2->tpl)))
		    b2 = b2->next;
	    }
	    copy = put_term_list(BIF_P, b1, b2);
	    CHECK_TABLES();
	    BIF_RET(copy);
	}
	b1 = b1->next;
    }
    BIF_RET(NIL);
}

/**********************************************************************/
/* Get an element from a term
** get_element_3(Tab, Key, Index)
** return the element or a list of elements if bag
*/
BIF_RETTYPE db_get_element_3(BIF_ALIST_3)
BIF_ADECL_3
{
    DbTable* tb;
    HashValue hval;
    int ix;
    uint32 key;
    DbTerm* b1;
    int index;

    if ((tb = get_table(BIF_P, BIF_ARG_1, DB_READ)) == NULL) {
	BIF_ERROR3(BADARG, am_db_get_element, BIF_ARG_1, BIF_ARG_2, BIF_ARG_3);
    }

    if (is_not_small(BIF_ARG_3) || ((index = signed_val(BIF_ARG_3)) < 1)) {
	BIF_ERROR3(BADARG, am_db_get_element, BIF_ARG_1, BIF_ARG_2, BIF_ARG_3);
    }

    key = BIF_ARG_2;
    hval = MAKE_HASH(key);
    HASH(tb, hval, ix);
    b1 = BUCKET(tb, ix);

    while(b1 != 0) {
	if ((b1->hvalue == hval) && EQ(key, GETKEY(tb, b1->tpl))) {
	    uint32 copy;

	    if (index > arityval(b1->tpl[0]))
		goto badarg;

	    if ((tb->status & DB_BAG) || (tb->status & DB_DUPLICATE_BAG)) {
		DbTerm* b;
		DbTerm* b2 = b1->next;
		uint32 elem_list = NIL;

		while((b2 != 0) && (b2->hvalue == hval) &&
		      EQ(key, GETKEY(tb, b2->tpl))) {
		    if (index > arityval(b2->tpl[0]))
			goto badarg;
		    b2 = b2->next;
		}

		b = b1;
		while(b != b2) {
		    uint32* hp;
		    uint32 sz = size_object(b->tpl[index])+2;
		    
		    hp = HAlloc(BIF_P, sz);
		    copy = copy_struct(b->tpl[index], sz-2,
				       &hp, &BIF_P->mso);
		    elem_list = CONS(hp, copy, elem_list);
		    hp += 2;
		    b = b->next;
		}
		BIF_RET(elem_list);
	    }
	    else {
		COPY_OBJECT(b1->tpl[index], BIF_P, &copy);
		BIF_RET(copy);
	    }
	}
	b1 = b1->next;
    }

 badarg:
    BIF_ERROR3(BADARG, am_db_get_element, BIF_ARG_1, BIF_ARG_2, BIF_ARG_3);
}

/**********************************************************************/

/* BIF to erase a whole table and release all memory it holds */
BIF_RETTYPE db_erase_1(BIF_ALIST_1)
BIF_ADECL_1
{
    DbTable* tb;

    if ((tb = get_table(BIF_P, BIF_ARG_1, DB_WRITE)) == NULL) {
	BIF_ERROR1(BADARG, am_db_erase, BIF_ARG_1);
    }
    db_tables[tb->slot].id = USED;
    db_tables[tb->slot].t = NULL;

    free_table(tb);
    BIF_RET(am_true);
}

/**********************************************************************/

/* Erase an object, or maybe several objects if we have a bag  */
/* Called as db_erase(Tab, Key), where Key is element 1 of the */
/* object(s) we want to erase                                  */

BIF_RETTYPE db_erase_2(BIF_ALIST_2)
BIF_ADECL_2
{
    DbTable* tb;
    HashValue hval;
    int ix;
    uint32 key;
    DbTerm** bp;
    DbTerm* b;
    int found = 0;

    if ((tb = get_table(BIF_P, BIF_ARG_1, DB_WRITE)) == NULL) {
	BIF_ERROR2(BADARG, am_db_erase, BIF_ARG_1, BIF_ARG_2);
    }

    key = BIF_ARG_2;
    hval = MAKE_HASH(key);
    HASH(tb, hval, ix);
    bp = &BUCKET(tb, ix);
    b = *bp;

    while(b != 0) {
	if ((b->hvalue == hval) && EQ(key, GETKEY(tb, b->tpl))) {
	    *bp = b->next;
	    free_term(b);
	    tb->nitems--;
	    b = *bp;
	    found = 1;
	}
	else {
	    if (found)
		break;
	    bp = &b->next;
	    b = b->next;
	}
    }

    if (found && ((tb->nitems / tb->nactive) < CHAIN_LEN) &&
	((tb->status & DB_FIXED) == 0))
	shrink(tb);

    BIF_RET(am_true);
}


/**********************************************************************/
/* Call this BIF as db_erase(Tab,Pattern)                          */ 
/* If The first element in Pattern is bound, we go directly to the */
/* right slot, otherwise, we search the whole table                */

/* Auxilliary MACRO used in the erase BIF's */

BIF_RETTYPE db_match_erase_2(BIF_ALIST_2)
BIF_ADECL_2
{
    DbTable* tb;
    int      j;
    int      ix;
    DbTerm** bp;
    DbTerm*  b;
    int      found;
    uint32   pattern;
    uint32   bnd[NBIND];
    struct bindings bs;

    if ((tb = get_table(BIF_P, BIF_ARG_1, DB_WRITE)) == NULL) {
	BIF_ERROR2(BADARG, am_db_match_erase, BIF_ARG_1, BIF_ARG_2);
    }
    pattern = BIF_ARG_2;
    found = 0;
    bs.ptr = &bnd[0];
    bs.size = NBIND;
    if (is_not_atom(pattern) && pattern != am_underscore && 
	variable(pattern) == -1) {
	uint32   hval;
	uint32   key;

	if ((key = getkey(tb, pattern)) == 0)
	    BIF_RET(am_true);  /* Can't possibly match anything */

	if (!has_variable(key)) {  /* Bound key */
	    hval = MAKE_HASH(key);
	    HASH(tb, hval, ix);
	    bp = &BUCKET(tb, ix); 
	    b = *bp;

	    while (b != 0) {
		ZEROB(bs);

		if (match(make_tuple(b->tpl),pattern, &bs, NULL)) {
		    *bp = b->next;
		    free_term(b);
		    tb->nitems--;
		    b = *bp;
		    found = 1;
		    continue;   /* Might be a bag, we continue until NIL */
		}
		else {
		    if (found)
			break;
		    bp = &b->next;
		    b = b->next;
		}
	    }
	    if (found && ((tb->nitems / tb->nactive) < CHAIN_LEN) &&
		((tb->status & DB_FIXED) == 0))
		shrink(tb);
	    if (bs.size != NBIND) 
		sys_free(bs.ptr);
	    BIF_RET(am_true);
	}
    }

    /* We gotta search the entire table and do the thing */

    for (j= 0; j < tb->nactive; j++) {
	bp = &BUCKET(tb, j);
	if ((b = *bp) == 0)
	    continue;

	while (b != 0) {
	    ZEROB(bs);

	    if (match(make_tuple(b->tpl),pattern, &bs, NULL)) {
		*bp = b->next;
		free_term(b);
		tb->nitems--;
		b = *bp;
		found = 1;
		continue;   /* Might be a bag, we continue until NIL */
	    }
	    else {
		bp = &b->next;
		b = b->next;
	    }
	}
    }
    if (found && ((tb->nitems / tb->nactive) < CHAIN_LEN) &&
	((tb->status & DB_FIXED) == 0))
	shrink(tb);

    if (bs.size != NBIND) 
	sys_free(bs.ptr);
    BIF_RET(am_true);
}

/**********************************************************************/
/* Return a list of tables on this node */

BIF_RETTYPE db_all_tables_0(BIF_ALIST_0)
BIF_ADECL_0
{
    DbTable* tb;
    uint32 previous;
    int i, j;
    uint32* hp = HAlloc(BIF_P, 2*no_tabs);

    previous = NIL;
    j = 0;
    for(i = 0; (i < DB_MAX_TABS && j < no_tabs); i++) {
	if (!ISFREE(i)) {
	    j++;
	    tb = db_tables[i].t;
	    previous = CONS(hp, tb->id, previous);
	    hp += 2;
	}
    }
    ASSERT(j == no_tabs);
    BIF_RET(previous);
}


/**********************************************************************/

BIF_RETTYPE db_slot_2(BIF_ALIST_2) 
BIF_ADECL_2
{
    DbTable* tb;
    uint32 copy;
    int slot;

    if ((tb = get_table(BIF_P, BIF_ARG_1, DB_READ)) == NULL) {
	BIF_ERROR2(BADARG, am_db_slot, BIF_ARG_1, BIF_ARG_2);
    }
    slot = signed_val(BIF_ARG_2);

    if (is_not_small(BIF_ARG_2) || (slot < 0) || (slot > tb->nactive))
	BIF_ERROR(BADARG);
    
    if (slot == tb->nactive)
	BIF_RET(am_eot);

    copy = put_term_list(BIF_P, BUCKET(tb, slot), 0);

    BIF_RET(copy);
}

/**********************************************************************/
/* The match BIF,  called as db_match(Table, Pattern) */

BIF_RETTYPE db_match_2(BIF_ALIST_2)
BIF_ADECL_2
{
    int i;
    DbTable* tb;
    uint32 chain_pos;
    uint32 key_given;
    uint32 key = NIL;		/* suppress use-before-set warning */
    DbTerm* list = NULL;	/* suppress use-before-set warning */
    uint32 hval = NIL;		/* suppress use-before-set warning */
    uint32 bnd[NBIND];
    uint32 size_bnd[NBIND];
    struct bindings bs, sz_bs;
    uint32 match_list;
    uint32 pattern = BIF_ARG_2;

    if ((tb = get_table(BIF_P, BIF_ARG_1, DB_READ)) == NULL) {
	BIF_ERROR2(BADARG, am_db_match, BIF_ARG_1, BIF_ARG_2);
    }
    bs.size = sz_bs.size = NBIND;  /* Initialize bind structure */
    bs.ptr = &bnd[0];
    sz_bs.ptr = &size_bnd[0];

    if (pattern == am_underscore || variable(pattern) != -1)
	key_given = 0;
    else {
	if ((key = getkey(tb, pattern)) == 0)
	    BIF_RET(NIL);  /* can't possibly match anything */
	if (!has_variable(key)) {   /* Bound key */
	    int ix;
	    hval = MAKE_HASH(key);
	    HASH(tb, hval, ix);
	    
	    if ((list = search_list(tb, key, hval, BUCKET(tb, ix))) == 0)
		BIF_RET(NIL);
	    key_given = 1;
	}
	else
	    key_given = 0;
    }

    if (!key_given) {
	/* Run this code if pattern is variable or GETKEY(pattern)  */
	/* is a variable                                            */
	for(chain_pos = 0; chain_pos < tb->nactive; chain_pos++) {
	    if ((list = BUCKET(tb,chain_pos)) != 0)
		break;
	}
	if (list == 0)
	    BIF_RET(NIL);
    }

    match_list = NIL;

    while(1) {
	ZEROB(bs);

	if (match(make_tuple(list->tpl), pattern, &bs, &sz_bs) != 0) {
	    uint32 sz = 2;
	    uint32 binding_list;
	    uint32* hp;

	    for (i = 0; i < bs.size; i++) {
		if (bs.ptr[i] != 0) {
		    sz_bs.ptr[i] = size_object(bs.ptr[i]);
		    sz += sz_bs.ptr[i] + 2;
		}
	    }
	    
	    hp = HAlloc(BIF_P, sz);
	    binding_list = NIL;

            for (i = bs.size - 1; i >= 0; i--) {
                if (bs.ptr[i] != 0) {
                    uint32 bound = copy_struct(bs.ptr[i], 
					       sz_bs.ptr[i],
                                               &hp, &BIF_P->mso);
                    binding_list = CONS(hp, bound, binding_list);
                    hp += 2;
                }
            }
            match_list = CONS(hp, binding_list, match_list);
	    hp += 2;
	}

	/* Update the list variable */
        if (key_given) {  /* Key is bound */
	    list = list->next;
            if (list == 0 || (hval != list->hvalue) || 
		!EQ(key, GETKEY(tb,list->tpl)))
                break;
        }
        else { /* Key is variable */
            if ((list = next(tb, &chain_pos, list)) == 0)
                break;
        }
    }

    /* Possibly free space malloced by match()  */
    if (bs.size != NBIND) {
	sys_free(bs.ptr);
	sys_free(sz_bs.ptr);
    }

    BIF_RET(match_list);
}

/**********************************************************************/
/* db_match_object(Table, Pattern, State) */

BIF_RETTYPE db_match_object_3(BIF_ALIST_3)
BIF_ADECL_3
{
    DbTable* tb;
    uint32 chain_pos;
    uint32 key_given;
    uint32 key = NIL;		/* suppress use-before-set warning */
    DbTerm* list = NULL;	/* suppress use-before-set warning */
    uint32 hval = NIL;		/* suppress use-before-set warning */
    uint32 bnd[NBIND];
    struct bindings bs;
    uint32 match_list = NIL;	/* Resulting match list. */
    uint32 pattern = BIF_ARG_2;
    uint32 records;		/* Records done so far. */
    uint32 max_counter;		/* Maximum number of slots to search. */
    uint32 result;

    /*
     * Make sure that the table exists.
     */

    if ((tb = get_table(BIF_P, BIF_ARG_1, DB_READ)) == NULL) {
    error:
	BIF_ERROR3(BADARG, am_db_match_object, BIF_ARG_1, BIF_ARG_2, BIF_ARG_3);
    }
    
    /*
     * Get the third argument, which must either be the maximum number of
     * records to look at in each invocation of this BIF, or a tuple
     * containing the wrapped up status from a previous invocation.
     * This tuple has three elements: {Matches, ChainNumber, Max},
     * where Matches is a list of Matches found earlier, ChainNumber is
     * the number of the next hash chain to look in, and Max is the
     * the maximum number of records to look at before returning.
     *
     * Note that the Max parameter is not exact; we will finish the
     * current chain before returning.
     */

    if (is_small(BIF_ARG_3)) {
	max_counter = signed_val(BIF_ARG_3);
	chain_pos = 0;
    } else if (is_tuple(BIF_ARG_3)) {
	uint32 *tupleptr = ptr_val(BIF_ARG_3);
	if (arityval(*tupleptr) != 3)
	    goto error;
	match_list = tupleptr[1];
	if (is_not_list(match_list) && is_not_nil(match_list))
	    goto error;
	if (!is_small(tupleptr[2]))
	    goto error;
	chain_pos = signed_val(tupleptr[2]);
	if (chain_pos >= tb->nactive)
	    goto error;
	if (!is_small(tupleptr[3]))
	    goto error;
	max_counter = signed_val(tupleptr[3]);
    } else {
	goto error;
    }
    
    if (max_counter < 1)
	goto error;

    /*
     * Find out whether the pattern has a constant key in
     * the key position of the tuple or not.
     */
    
    if (pattern == am_underscore || variable(pattern) != -1) {
	key_given = 0;
    } else if ((key = getkey(tb, pattern)) == 0) {
	BIF_RET(match_list);
    } else {
	key_given = !has_variable(key);
    }

    /*
     * Locate the first chain to search in.
     */

    if (key_given) {
	int ix;

	hval = MAKE_HASH(key);
	HASH(tb, hval, ix);
	list = search_list(tb, key, hval, BUCKET(tb, ix));
    } else {
	for ( ; chain_pos < tb->nactive; chain_pos++) {
	    if ((list = BUCKET(tb, chain_pos)) != 0)
		break;
	}
    }

    if (list == 0)
	BIF_RET(match_list);

    /*
     * Now start matching with all records in the chain given by list.
     */
    
    bs.size = NBIND;
    bs.ptr = &bnd[0];
    records = 0;
    for (;;) {
	uint32 term = make_tuple(list->tpl);
	ZEROB(bs);

	/*
	 * See if this record matches, and if so, cons it to the
	 * list of matches.
	 */

	if (match(term, pattern, &bs, NULL) != 0) {
	    uint32 copy;
	    uint32* hp;

	    hp = HAlloc(BIF_P, list->size+2);
	    copy = copy_struct(term, list->size, &hp, &BIF_P->mso);
            match_list = CONS(hp, copy, match_list);
            hp += 2;
	}

	/*
	 * Point to the next record in the table.
	 */

	records++;
	list = list->next;
        if (key_given) {
	    /*
	     * When given a key, we will collect all records
	     * before returning (they must be in the same chain).
	     */
            if (list == 0 || (hval != list->hvalue) ||
		!EQ(key, GETKEY(tb, list->tpl))) {
		result = match_list;
                goto return_result;
	    }
        } else if (list == 0) {
	    /*
	     * We have reached the end of this chain.  Find the next
	     * non-empty chain.  We are done if none left.
	     */
	    do {
		chain_pos++;
		if (chain_pos >= tb->nactive) {
		    /*
		     * No chain left. Done.
		     */
		    result = match_list;
		    goto return_result;
		}
	    } while ((list = BUCKET(tb, chain_pos)) == NULL);

	    /*
	     * We have the pointer to the next chain.  But if we have already
	     * looked at enough records, we must save the state and return
	     * from the BIF.
	     */

	    if (records >= max_counter) {
		uint32* hp = HAlloc(BIF_P, 4);
		
		result = TUPLE3(hp, match_list, make_small(chain_pos),
				make_small(max_counter));
		goto return_result;
	    }
	}
    }

 return_result:
    if (bs.size != NBIND) 
	sys_free(bs.ptr);

    BIF_RET(result);
}


/**********************************************************************/
/* BIF to extract information about a particular table */

BIF_RETTYPE db_info_2(BIF_ALIST_2)
BIF_ADECL_2
{
    DbTable* tb;

    if ((tb = get_table(BIF_P, BIF_ARG_1, DB_INFO)) == NULL) {
	BIF_ERROR2(BADARG, am_db_info, BIF_ARG_1, BIF_ARG_2);
    }
    if (BIF_ARG_2 == am_size) 
	BIF_RET(make_small(tb->nitems));
    if (BIF_ARG_2 == am_type) {
	if (tb->status & DB_SET)  {
	    BIF_RET(am_set);
	}
	else if (tb->status & DB_DUPLICATE_BAG) {
	    BIF_RET(am_duplicate_bag);
	}
	else {
	    BIF_RET(am_bag);
	}
    }
    if (BIF_ARG_2 == am_memory) {
	DbTerm* list;
	int tot = 0;
	int i;

	for (i = 0; i < tb->nactive; i++) {
	    list = BUCKET(tb,i);
	    while(list != 0) {
		tot += (sizeof(DbTerm)/sizeof(uint32)) + (list->size-1);
		list = list->next;
	    }
	}
	tot += sizeof (DbTable) / sizeof (uint32);
	tot += tb->nsegs * SEGSZ;
	BIF_RET2(make_small(tot),tb->nitems+1/20);
    }
    if (BIF_ARG_2 == am_owner) 
	BIF_RET(tb->owner);
    if (BIF_ARG_2 == am_protection) {
	if (tb->status & DB_PRIVATE) 
	    BIF_RET(am_private);
	if (tb->status & DB_PROTECTED)
	    BIF_RET(am_protected);
	if (tb->status & DB_PUBLIC)
	    BIF_RET(am_public);
    }
    if (BIF_ARG_2 == am_name)
	BIF_RET(tb->the_name);
    if (BIF_ARG_2 == am_keypos) 
	BIF_RET(make_small(tb->keypos));

    BIF_ERROR2(BADARG, am_db_info, BIF_ARG_1, BIF_ARG_2);
}


static void print_table(show, tb)
int show; DbTable* tb;
{
    int i;
    int sum = 0;

    erl_printf(CERR, "Table "); display(tb->id,CERR);
    erl_printf(CERR, "(with name)" );
    display(tb->the_name, CERR);
    erl_printf(CERR, "\n\r");

    for (i = 0; i < tb->nactive; i++) {
	DbTerm* list = BUCKET(tb,i);
	if (show)
	    erl_printf(CERR,"%d: [", i);
	while(list != 0) {
	    sum += list->size;
	    if (show) {
		display(make_tuple(list->tpl), CERR);
		if (list->next != 0)
		    erl_printf(CERR, ",");
	    }
	    list = list->next;
	}
	if (show) 
	    erl_printf(CERR, "]\n\r");
    }
    erl_printf(CERR,"Tables got %d objects \n\r", tb->nitems);
    erl_printf(CERR,"Table's got %d words of active data \n\r", sum);
}


void db_info()    /* Called by break handler */
{
    int i;
    for (i=0; i < DB_MAX_TABS; i++) 
	if (!ISFREE(i)) {
	    erl_printf(CERR, "In slot %d\n", i);
	    print_table(1, db_tables[i].t);
	}
}


#ifdef DEBUG   /* Here comes some debug functions */

void check_tables()
{
    int i;

    for (i = 0; i < DB_MAX_TABS; i++) {
	if (!ISFREE(i)) {
	    DbTable* tb = db_tables[i].t; 
	    DbTerm* list;
	    int j;

	    for (j = 0; j < tb->nactive; j++) {
		if ((list = BUCKET(tb,j)) == 0)
		    continue;
		while (list != 0) {
		    check_struct(make_tuple(list->tpl));
		    list = list->next;
		}
	    }
	}
    }
}


void db_bin_check()
{
    int i;

    for (i = 0; i < DB_MAX_TABS; i++) {
	if (!ISFREE(i)) {
	    DbTable* tb = db_tables[i].t;
	    DbTerm* list;
	    int printed = 0;
	    int j;

	    for (j = 0; j < tb->nslots; j++) {
		if ((list = BUCKET(tb,j)) == 0)
		    continue;
		while (list != 0) {
		    ProcBin *bp = list->mso;

		    while(bp != 0) {
			if (printed == 0) {
			    erl_printf(COUT,"Table "); 
			    display(tb->id, COUT);
			    erl_printf(COUT, "(with name)" );
			    display(tb->the_name, COUT);
			    erl_printf(COUT," holding binary data \n");
			    printed = 1;
			}
			erl_printf(COUT,"0x%08lx orig_size: %d, norefs = %d\n",
				   (uint32)bp->val, bp->val->orig_size, bp->val->refc);
			bp = bp->next;
		    }
		    list = list->next;
		}
	    }
	    if (printed == 1)
		erl_printf(COUT,"--------------------------------------\n");
	}
    }
}
		
#endif

