/* ``The contents of this file are subject to the Erlang Public License,
 * Version 1.0, (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.erlang.org/EPL1_0.txt
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 * 
 * The Original Code is Erlang-4.7.3, December, 1998.
 * 
 * The Initial Developer of the Original Code is Ericsson Telecom
 * AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
 * Telecom AB. All Rights Reserved.
 * 
 * Contributor(s): ______________________________________.''
 */
/*
** Copy struct & size struct
**
*/

#include "sys.h"
#include "config.h"
#include "global.h"
#include "erl_process.h"
#include "big.h"

FUNCTION(uint32, copy_struct, (uint32, uint32, uint32**, ProcBin**));

#define SMALL_STACK_SIZE 5

typedef struct erl_stack {
    uint32* sp;
    uint32* start;
    uint32* end;
    uint32  default_stack[SMALL_STACK_SIZE];
} ErlStack;

#define EPUSH(s, x) do { \
   if ((s).sp == (s).end) erl_grow_stack(&(s)); \
   *(s).sp++ = (x); \
} while(0)

/* never store a 0 !!! (if you dont know what you are doing) */
#define EPOP(s) (((s).sp == (s).start) ? 0 : *--(s).sp)

#define INIT_ESTACK(s) do { \
   (s).start = (s).default_stack; \
   (s).end = (s).default_stack + SMALL_STACK_SIZE; \
   (s).sp = (s).start; \
} while(0)

#define DESTROY_ESTACK(s) do { \
   if ((s).start != (s).default_stack) sys_free((s).start); \
} while(0)

void
init_copy(void)
{
}

static void erl_grow_stack(s)
ErlStack* s;
{
    int sz;
    if (s->start == s->default_stack) {
	sz = SMALL_STACK_SIZE;
	s->start = (uint32*) sys_alloc_from(4,2*sz*sizeof(uint32));
	sys_memcpy(s->start, s->default_stack, sizeof(uint32)*sz);
    }
    else {
	sz = s->end - s->start;
	s->start = (uint32*) sys_realloc ((char*)s->start,
					  2*sz*sizeof(uint32));
    }
    s->end = s->start + 2*sz;
    s->sp = s->start + sz;
}

/*
**  Copy object "obj" to process p
*/
int copy_object(obj, to, extra, res, from)
uint32 obj; Process* to; uint32 extra;
uint32* res; Process* from;
{
    uint32 size = size_object(obj);
    uint32* hp = HAlloc(to, size+extra);

    *res = copy_struct(obj, size, &hp, &to->mso);
    return 0;
}

/*
**  Copy n objects objs
**
*/
int copy_objects(obj, n, to, extra, res, from)
uint32* obj; int n; Process* to; uint32 extra;
uint32* res; Process* from;
{
    int i;
    int size;
    uint32* hp;

    /* Use "res" to store temporary sizes */
    size = 0;
    for (i = 0; i < n; i++) {
	res[i] = size_object(obj[i]);
	size += res[i];
    }

    hp = HAlloc(to, size+extra);
    for (i = 0; i < n; i++)
	res[i] = copy_struct(obj[i], res[i], &hp, &to->mso);
    return 0;
}

/*
** Return the "flat" size of the object.
*/
uint32 size_object(obj)
uint32 obj;
{
    ErlStack s;
    uint32 sum = 0;
    uint32* ptr;
    int arity;

    INIT_ESTACK(s);
    for (;;) {
	switch (tag_val_def(obj)) {
	case BIG_DEF:
#if defined(BEAM)
	    if (is_not_nil(obj)) {
#endif
		sum += big_arity(obj)+1;
#if defined(BEAM)
	    }
#endif
	    /* Fall through */
	case SMALL_DEF:
	case ATOM_DEF:
	case REFER_DEF:
	case PORT_DEF:
	case PID_DEF:
	case BINARY_DEF:
#if defined(JAM)
	case NIL_DEF:
#endif
	    if ((obj = EPOP(s)) == 0) {
		DESTROY_ESTACK(s);
		return sum;
	    }
	    break;
	case FLOAT_DEF:
	    sum += 3;
	    if ((obj = EPOP(s)) == 0) {
		DESTROY_ESTACK(s);
		return sum;
	    }
	    break;
	case TUPLE_DEF:
	    ptr = ptr_val(obj);
	    arity = arityval(*ptr);
	    sum += arity + 1;
	    if (arity == 0) {
		if ((obj = EPOP(s)) == 0) {
		    DESTROY_ESTACK(s);
		    return sum;
		}
	    } else {
		while (arity-- > 1) {
		    obj = *++ptr;
		    if (!IS_ONE_CELL(obj)) {
			EPUSH(s, obj);
		    }
		}
		obj = *++ptr;
	    }
	    break;
	case LIST_DEF:
	    sum += 2;
	    ptr = ptr_val(obj);
	    obj = *ptr++;
	    if (!IS_ONE_CELL(obj)) {
		EPUSH(s, obj);
	    }
	    obj = *ptr;
	    break;
#if defined(JAM)
	case FRAME_DEF:
#endif
	case ARITYVAL_DEF:
	case THING_DEF:
#if defined(BEAM)
	case CP0:
	case CP4:
	case CP8:
	case CP12:
#endif
	    erl_exit(1, "size_object: bad tag %d for %d\n", tag_val_def(obj), obj);
	    break;
	}
    }
}

/*
**  Copy a structure to the heap of p
*/
uint32 copy_struct(obj, sz, hpp, msoh)
uint32 obj; uint32 sz; uint32** hpp; ProcBin** msoh;
{
#define COPIED(x) (ptr_val(x) >= hstart && ptr_val(x) < hend)
    uint32* hstart;
    uint32* hend;
    uint32* htop;
    uint32* hbot;
    uint32* hp;
    uint32* objp;
    uint32* tp;
    uint32  res;
    uint32  elem;
    uint32* tailp;
    uint32* argp;
    uint32* const_tuple;
    ProcBin* bin;
    int const_flag;
    int i;

    if (IS_CONST(obj))
	return obj;

    hstart = *hpp;
    hend   = hstart + sz;
    htop   = hstart;
    hbot   = hend;
    hp     = hstart;
    const_tuple = 0;

    /* Copy the object onto the heap */
    switch (tag_val_def(obj)) {
    case FLOAT_DEF:  argp = &res; goto L_copy_float;
    case BIG_DEF:    argp = &res; goto L_copy_big;
    case TUPLE_DEF:  argp = &res; goto L_copy_tuple;
    case LIST_DEF:   argp = &res; goto L_copy_list;
    case BINARY_DEF: 
	bin = copy_binary(msoh, (ProcBin*) ptr_val(obj));
	return make_binary(bin);
    default:
	erl_exit(1, "Internal error: in copy_struct (1) 0x%08x\n", obj);
    }

 L_copy:
    while (hp != htop) {
	obj = *hp;
	switch(tag_val_def(obj)) {
	case FLOAT_DEF:  
	    if (COPIED(obj)) {
		hp++;
		break;
	    }
	    argp = hp++; goto L_copy_float;
	case BIG_DEF:
#if defined(BEAM)
	    if (is_nil(obj)) {
		hp++;
		break;
	    }
#endif
	    if (COPIED(obj)) {
		hp++;
		break;
	    }
	    argp = hp++; goto L_copy_big;
	case TUPLE_DEF: 
	    if (COPIED(obj)) {
		hp++;
		break;
	    }
	    argp = hp++; goto L_copy_tuple;
	case LIST_DEF:
	    objp = ptr_val(obj);
	    if (objp >= hstart && objp < hend) {
		hp++;
		break;
	    }
	    argp = hp++; goto L_copy_list;
	case BINARY_DEF:
	    bin = copy_binary(msoh, (ProcBin*) ptr_val(obj));
	    *hp++ = make_binary(bin);
	    break;
	case SMALL_DEF:
#if defined(JAM)
	case NIL_DEF:
#endif
	case ATOM_DEF:
	case PID_DEF:
	case REFER_DEF:
	case PORT_DEF:
	    hp++;
	    break;
	case ARITYVAL_DEF:
	    if (hp == const_tuple)
		hp += (arityval(obj)+1);
	    else
		hp++;
	    break;
	default:
	    erl_exit(1, "Internal error: in copy_struct (2) 0x%08x\n", *hp);
	}
    }

    if (htop > hbot)
	erl_exit(1, "Internal error: in copy_struct (3)\n");
    ASSERT(htop == hbot);
    *hpp = hend;
    return res;


/* -----------------------------------------------
   Copy list:
   Arguments are argp: pointer to store the result
                  obj: the list to copy
   Cells with constant car value is copied at the
   bottom of the heap otherwise cell is copied at
   the top of the stack
   ----------------------------------------------- */

 L_copy_list:
    tailp = argp;
    while(is_list(obj)) {
	objp = ptr_val(obj);
	tp = tailp;
	elem = *objp;
	if (IS_CONST(elem)) {
	    *(hbot-2) = elem;
	    tailp = hbot-1;
	    hbot -= 2;
	}
	else {
	    *htop = elem;
	    tailp = htop+1;
	    htop += 2;
	}
	*tp = make_list(tailp - 1);
	obj = *(objp+1);
    }
    if (IS_CONST(obj)) {
	*tailp = obj;
	goto L_copy;
    }
    else {
	switch(tag_val_def(obj)) {
	case FLOAT_DEF:  argp = tailp; goto L_copy_float;
	case BIG_DEF:    argp = tailp; goto L_copy_big;
	case TUPLE_DEF:  argp = tailp; goto L_copy_tuple;
	case BINARY_DEF:
	    bin = copy_binary(msoh, (ProcBin*) ptr_val(obj));
	    *tailp = make_binary(bin);
	    goto L_copy;
	default:
	    erl_exit(1, "Internal error: copy_struct (cdr)\n");
	}
    }
    
/* -----------------------------------------------
   Copy float:
   Arguments are argp: pointer to store the result
                  obj: the object to copy
   Thing is placed at the bottom of heap (hbot)
   ----------------------------------------------- */

 L_copy_float:
    objp = ptr_val(obj);
    hbot -= 3;
    tp = hbot;
    *argp = make_float(hbot);
    *tp++ = *objp++;
    *tp++ = *objp++;
    *tp   = *objp++;
    goto L_copy;

/* -----------------------------------------------
   Copy big:
   Arguments are argp: pointer to store the result
                  obj: the object to copy
   Thing is placed at the bottom of heap (hbot)
   ----------------------------------------------- */

 L_copy_big:
    objp = ptr_val(obj);
    i = thing_arityval(*objp)+1;
    hbot -= i;
    tp = hbot;
    *argp = make_big(hbot);
    while(i--)
	*tp++ = *objp++;
    goto L_copy;

/* -----------------------------------------------
   Copy tuple:
   Arguments are argp: pointer to store the result
                  obj: the object to copy
   Thing is placed at the top of heap (htop)
   ----------------------------------------------- */

 L_copy_tuple:
    const_flag = 1;             /* assume constant tuple */
    objp = ptr_val(obj);
    i = arityval(*objp);
    *argp = make_tuple(htop);
    tp = htop;               /* tp is pointer to new arity value */
    *htop++ = *objp++;       /* copy arity value */
    while(i--) {
	elem = *objp++;
	if (!IS_CONST(elem))
	    const_flag = 0;
	*htop++ = elem;
    }
    if (const_flag)
	const_tuple = tp;     /* this is the latest const_tuple */
    goto L_copy;
#undef COPIED
}

/*
 * Copy the cells in a tuple, adjusting the pointers to its subterms but
 * not copying them.
 */

uint32 copy_shallow(ptr, sz, hpp, msoh)
uint32* ptr; uint32 sz; uint32** hpp; ProcBin** msoh;
{
    uint32* tp = ptr;
    uint32* hp = *hpp;
    sint32 offs = hp - tp;
    ProcBin* bin;

    while(sz--) {
	uint32 val = *tp++;
	switch(tag_val_def(val)) {
	case BIG_DEF:
#if defined(BEAM)
	    if (is_nil(val)) {
		*hp++ = val;
		break;
	    }
#endif /* BEAM */
	case FLOAT_DEF:
	case LIST_DEF:
	case TUPLE_DEF:
	  *hp++ = offset_ptr(val, offs);
	  break;

	case THING_DEF: {
	    uint32 tari = thing_arityval(val);
	    *hp++ = val;
	    sz -= tari;
	    while(tari--)
		*hp++ = *tp++;
	    break;
	}
	case SMALL_DEF:
	case NIL_DEF:
	case ATOM_DEF:
	case PID_DEF:
	case REFER_DEF:
	case PORT_DEF:
	case ARITYVAL_DEF:
	    *hp++ = val;
	    break;
	case BINARY_DEF:
	    bin = copy_binary(msoh, (ProcBin*) ptr_val(val));
	    *hp++ = make_binary(bin);
	    break;
#if defined(JAM)
	case CATCH_DEF:
	case FRAME_DEF:
#endif /* JAM */
	case MOVED_DEF:
	    erl_exit(1, "Internal error: in copy_linear 0x%08x\n", val);
	}
    }
    *hpp = hp;
    return make_tuple(ptr + offs); 
}
