/* ``The contents of this file are subject to the Erlang Public License,
 * Version 1.0, (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.erlang.org/EPL1_0.txt
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 * 
 * The Original Code is Erlang-4.7.3, December, 1998.
 * 
 * The Initial Developer of the Original Code is Ericsson Telecom
 * AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
 * Telecom AB. All Rights Reserved.
 * 
 * Contributor(s): ______________________________________.''
 */
/*
 * This file is copyright (c) Ellemtel in January 1991
 *
 * Author Mike Williams
 *
 * Co-Authors:
 *  Cleas Wikstrom
 *  Tony Rogvall
 *
 * 1997 Tony Rogvall rehacked distribution
 *      bifs out: disconnect_node, alive 
 *      bifs in:  setnode/2 setnode/3
 *      added BIF_TRAP (trap to erlang functions)
 *      Changed all links into linked lists of ErlLink for
 *      processes and ports.
 *
 *      15/10
 *      added port_command, port_control, port_close, port_connect
 *
 *      14/11
 *      added set_timer, cancel_timer
 *
 * 1998 Arndt replaced set_timer/3 with start_timer/3 and send_after/3.
 */

#include "sys.h"
#include "config.h"
#include "global.h"
#include "erl_process.h"
#include "error.h"
#include "driver.h"
#include "bif.h"
#include "big.h"
#include "dist.h"
#include "erl_version.h"

#if defined(sparc)
#include <sys/ioccom.h>
#define PERFMON_SETPCR			_IOW('P', 1, unsigned long long)
#define PERFMON_GETPCR			_IOR('P', 2, unsigned long long)
#endif

extern DriverEntry fd_driver_entry;
extern DriverEntry vanilla_driver_entry;
extern DriverEntry spawn_driver_entry;

#ifdef SEQ_TRACE
static void new_seq_trace_token(Process* p); /* help func for seq_trace_2*/
#endif
uint32 bif_gc;

/* Wrap info numbers to 27 bit */
#define WRAP27(x)  ((x) & (MAX_SMALL-1))

/*
 * The BIF's now follow, see the Erlang Manual for a description of what
 * each individual BIF does.
 *
 * Beam note: Guards BIFs must not build anything at all on the heap.
 * They must use the ArithAlloc() macro instead of HAlloc() (on Jam,
 * ArithAlloc() is just an alias for HAlloc()).
 */

BIF_RETTYPE spawn_3(BIF_ALIST_3)
BIF_ADECL_3
{
    ErlSpawnOpts so;
    uint32 pid;

    so.flags = 0;
    pid = erl_create_process(BIF_P, BIF_ARG_1, BIF_ARG_2, BIF_ARG_3, &so);
    if (pid == 0) {
	BIF_ERROR3(so.error_code, am_spawn, BIF_ARG_1, BIF_ARG_2, BIF_ARG_3);
    } else {
	BIF_RET(pid);
    }
}

/**********************************************************************/

/* utility to add a new link between processes p and rp */

static void insert_link(p, rp)
Process* p; Process* rp;
{
    p->links = new_link(p->links, LNK_LINK, rp->id, NIL);
    rp->links = new_link(rp->links, LNK_LINK, p->id, NIL);

    if (IS_TRACED(p)) {
	if (p->flags & F_TRACE_SOL)  {
	    rp->flags |= (p->flags & TRACE_FLAGS);
	    rp->tracer_proc = p->tracer_proc;    /* maybe steal */
	}
	if (p->flags & F_TRACE_SOL1 )  { /* maybe override */
	    rp->flags |= (p->flags & TRACE_FLAGS);
	    rp->tracer_proc = p->tracer_proc;   
	    rp ->flags &= ~(F_TRACE_SOL1 | F_TRACE_SOL);
	    p->flags &= ~(F_TRACE_SOL1 | F_TRACE_SOL);
	}
    }
    if (IS_TRACED_FL(rp, F_TRACE_PROCS))
	trace_proc(rp, am_getting_linked, p->id);
}


/* create a link to the process */
BIF_RETTYPE link_1(BIF_ALIST_1)
BIF_ADECL_1
{
    Process *rp;
    int slot;
    int ix;
    int code;

    if (IS_TRACED(BIF_P)) {
	if (BIF_P->flags & F_TRACE_PROCS)
	    trace_proc(BIF_P, am_link, BIF_ARG_1);
    }
    /* check that the pid which is our argument is OK */

    if (is_not_pid(BIF_ARG_1)) {
	if (is_not_port(BIF_ARG_1))
	    BIF_ERROR1(BADARG, am_link, BIF_ARG_1);
    
	/* we are linking to a port */
	if (find_link(&BIF_P->links,LNK_LINK,BIF_ARG_1,NIL) != NULL)
	    BIF_RET(am_true); /* already linked */

	if (get_node_reference(BIF_ARG_1) != THIS_NODE) {
	    /* Don't allow link to a remote port */
	    BIF_ERROR1(BADARG, am_link, BIF_ARG_1);
	} else {
	    ix = get_number_reference(BIF_ARG_1);
	    if (port[ix].status == FREE) {
		if (BIF_P->flags & F_TRAPEXIT) {
		    deliver_exit_message(BIF_ARG_1, BIF_P, am_noproc);
		    BIF_RET(am_true);
		}
		else
		    BIF_ERROR1(NOPROC, am_link, BIF_ARG_1);
	    }
	    port[ix].links = new_link(port[ix].links,LNK_LINK,BIF_P->id,NIL);
	}
	BIF_P->links = new_link(BIF_P->links,LNK_LINK,BIF_ARG_1,NIL);
	BIF_RET(am_true);
    }

    /* we are linking to another process */
    /* check that the PID is OK */
    
    if (BIF_ARG_1 == BIF_P->id)
	BIF_RET(am_true);  /* Some people try the silliest things... */

    if (find_link(&BIF_P->links,LNK_LINK,BIF_ARG_1, NIL) != NULL)
	BIF_RET(am_true);	/*  already linked */	
    
    if ((slot = get_node(BIF_ARG_1)) != THIS_NODE) {  /* link to net */
	if (dist_addrs[slot].cid == NIL)
	    BIF_TRAP1(dlink_trap, BIF_ARG_1);

	if ((code = dist_link(slot, BIF_P->id, BIF_ARG_1)) == 1) {
	    ASSERT(is_port(dist_addrs[slot].cid)); 
	    erl_suspend(BIF_P, dist_addrs[slot].cid);
	    BIF_ERROR(RESCHEDULE);
	}
	else if (code < 0) { /* XXX is this the correct behaviour ??? */
	    BIF_ERROR1(NOTALIVE, am_link, BIF_ARG_1);
	}
	/* insert the link in our own process */
	BIF_P->links = new_link(BIF_P->links,LNK_LINK,BIF_ARG_1,NIL);
	dist_addrs[slot].links = new_link(dist_addrs[slot].links,
					  LNK_LINK, BIF_P->id, BIF_ARG_1);
	BIF_RET(am_true);
    }

    if (get_number(BIF_ARG_1) >= max_process) {
	BIF_ERROR1(BADARG, am_link, BIF_ARG_1);
    }
    
    /* get a pointer to the process struct of the linked process */
    rp = process_tab[get_number(BIF_ARG_1)];
    
    /* is the right process - ie not free and the pid corresponds?? */
    if (INVALID_PID(rp, BIF_ARG_1)) {
	/* Schedule an exit and insert the link in our own queue only */
	if (BIF_P->flags & F_TRAPEXIT) {
	    deliver_exit_message(BIF_ARG_1, BIF_P, am_noproc);
	    BIF_RET(am_true);
	}
	else
	    BIF_ERROR1(NOPROC, am_link, BIF_ARG_1);
    }
    insert_link(BIF_P, rp);
    BIF_RET(am_true);
}


/**********************************************************************/
/* this is a combination of the spawn and link BIFs */

BIF_RETTYPE spawn_link_3(BIF_ALIST_3)
BIF_ADECL_3
{
    ErlSpawnOpts so;
    uint32 pid;

    so.flags = SPO_LINK;
    pid = erl_create_process(BIF_P, BIF_ARG_1, BIF_ARG_2, BIF_ARG_3, &so);
    if (pid == 0) {
	BIF_ERROR3(so.error_code, am_spawn, BIF_ARG_1, BIF_ARG_2, BIF_ARG_3);
    } else {
	BIF_RET(pid);
    }
}

/**********************************************************************/

BIF_RETTYPE spawn_opt_2(BIF_ALIST_2)
BIF_ADECL_2
{
    ErlSpawnOpts so;
    uint32 pid;
    uint32* tp;
    uint32 ap;
    uint32 arg;

    /*
     * Check that the first argument is a tuple of three elements.
     */
    if (is_not_tuple(BIF_ARG_1)) {
    error:
	BIF_ERROR2(BADARG, am_spawn_opt, BIF_ARG_1, BIF_ARG_2);
    }
    tp = ptr_val(BIF_ARG_1);
    if (*tp != make_arityval(3))
	goto error;

    /*
     * Store default values for options.
     */
    so.flags = SPO_USE_ARGS;
    so.min_heap_size = H_MIN_SIZE;
    so.gc_switch = switch_gc_threshold;
    so.priority = PRIORITY_NORMAL;
    so.process_flags = 0;

    /*
     * Walk through the option list.
     */
    ap = BIF_ARG_2;
    while (is_list(ap)) {
	arg = CAR(ptr_val(ap));
	if (arg == am_link) {
	    so.flags |= SPO_LINK;
	} else if (is_tuple(arg)) {
	    uint32* tp2 = ptr_val(arg);
	    uint32 val;
	    if (*tp2 != make_arityval(2))
		goto error;
	    arg = tp2[1];
	    val = tp2[2];
	    if (arg == am_priority) {
		if (val == am_max)
		    so.priority = PRIORITY_MAX;
		else if (val == am_high)
		    so.priority = PRIORITY_HIGH;
		else if (val == am_normal)
		    so.priority = PRIORITY_NORMAL;
		else if (val == am_low)
		    so.priority = PRIORITY_LOW;
		else
		    goto error;
	    } else if (arg == am_gc_switch) {
		if (val == am_infinity) {
		    so.gc_switch = MAX_SMALL;
		} else if (is_small(val) && signed_val(val) >= 0) {
		    so.gc_switch = signed_val(val);
		} else {
		    goto error;
		}
	    } else if (arg == am_min_heap_size && is_small(val)) {
		so.min_heap_size = signed_val(val);
		if (so.min_heap_size < 0) {
		    goto error;
		} else if (so.min_heap_size < H_MIN_SIZE) {
		    so.min_heap_size = H_MIN_SIZE;
		} else {
		    so.min_heap_size = next_heap_size(so.min_heap_size, 0);
		}
	    } else {
		goto error;
	    }
	} else {
	    goto error;
	}
	ap = CDR(ptr_val(ap));
    }
    if (is_not_nil(ap))
	goto error;

    /*
     * Spawn the process.
     */
    pid = erl_create_process(BIF_P, tp[1], tp[2], tp[3], &so);
    if (pid == 0) {
	BIF_ERROR2(so.error_code, am_spawn_opt, BIF_ARG_1, BIF_ARG_2);
    } else {
	BIF_RET(pid);
    }
}

  
/**********************************************************************/
/* remove a link from a process */
BIF_RETTYPE unlink_1(BIF_ALIST_1)
BIF_ADECL_1
{
    Process *rp;
    int slot;
    int ix;

    if (IS_TRACED(BIF_P)) {
	if (BIF_P->flags & F_TRACE_PROCS) 
	    trace_proc(BIF_P, am_unlink, BIF_ARG_1);
    }

    if (is_not_pid(BIF_ARG_1)) {
	if (is_not_port(BIF_ARG_1))
	    BIF_ERROR1(BADARG, am_unlink, BIF_ARG_1);

	if (get_node_reference(BIF_ARG_1) != THIS_NODE)
	    BIF_ERROR1(BADARG, am_unlink, BIF_ARG_1);

	del_link(find_link(&BIF_P->links,LNK_LINK,BIF_ARG_1,NIL));
	ix = get_number_reference(BIF_ARG_1);
	if (port[ix].status != FREE)
	    del_link(find_link(&port[ix].links,LNK_LINK,BIF_P->id,NIL));
	BIF_RET(am_true);
    }

    del_link(find_link(&BIF_P->links,LNK_LINK,BIF_ARG_1,NIL));

    if ((slot = get_node(BIF_ARG_1)) != THIS_NODE) {
	if (dist_addrs[slot].cid == NIL)
	    BIF_TRAP1(dunlink_trap, BIF_ARG_1);

	if (dist_unlink(slot, BIF_P->id, BIF_ARG_1) == 1) {
	    ASSERT(is_port(dist_addrs[slot].cid)); 
	    erl_suspend(BIF_P, dist_addrs[slot].cid);
	    BIF_ERROR(RESCHEDULE);
	}
	del_link(find_link(&dist_addrs[slot].links, LNK_LINK,
			   BIF_P->id, BIF_ARG_1));
	BIF_RET(am_true);
    }

     /* process ok ? */
    if (get_number(BIF_ARG_1) >= max_process)
	BIF_ERROR1(BADARG, am_unlink, BIF_ARG_1);
     
    /* get process struct */
    rp = process_tab[get_number(BIF_ARG_1)];

    /* and we mean this process */
    if (INVALID_PID(rp, BIF_ARG_1))
	BIF_RET(am_true);

    /* unlink and ignore errors */
    del_link(find_link(&rp->links, LNK_LINK, BIF_P->id, NIL));
     
     BIF_RET(am_true);
}

/**********************************************************************/
/* this is the same as throw/1 except that we set freason to USER_EXIT */

BIF_RETTYPE exit_1(BIF_ALIST_1)
BIF_ADECL_1
{
    BIF_P->fvalue = BIF_ARG_1;  /* exit value */
    BIF_ERROR(USER_EXIT);
}

/**********************************************************************/
/* send an exit message to another process (if trapping exits) or
   exit the other process */

BIF_RETTYPE exit_2(BIF_ALIST_2)
BIF_ADECL_2
{
     Process *rp;
     int slot;
     int code;
     uint32 exit_value = (BIF_ARG_2 == am_kill) ? am_killed : BIF_ARG_2;
     /*
      * If the first argument is not a pid, it must a be port
      * or it is an error.
      */

     if (is_not_pid(BIF_ARG_1)) {
	 if (is_not_port(BIF_ARG_1)) 
	     BIF_ERROR2(BADARG, am_exit, BIF_ARG_1, BIF_ARG_2);

	 if (get_node_reference(BIF_ARG_1) != THIS_NODE) {
	     /* remote port */
	     BIF_ERROR2(BADARG, am_exit, BIF_ARG_1, BIF_ARG_2);
	 }
	 do_exit_port(BIF_ARG_1, BIF_P->id, BIF_ARG_2);
	 if (BIF_P->status != P_RUNNING) {
	     BIF_P->fvalue = exit_value;
	     KILL_CATCHES(BIF_P);
	     BIF_ERROR(USER_EXIT);
	 }
	 BIF_RET(am_true);
     }
     
     /*
      * It is a pid.  If it is a remote pid, send a message to the remote
      * node.
      */

     if ((slot = get_node(BIF_ARG_1)) != THIS_NODE) {
	 if (dist_addrs[slot].cid == NIL)
	     BIF_TRAP2(dexit_trap,BIF_ARG_1,BIF_ARG_2);

	 if ((code = dist_exit2(slot, BIF_P->id, BIF_ARG_1, BIF_ARG_2)) == 1) {
	     ASSERT(is_port(dist_addrs[slot].cid)); 
	     erl_suspend(BIF_P, dist_addrs[slot].cid);
	     BIF_ERROR(RESCHEDULE);
	 }
	 else if (code < 0) {
	     BIF_ERROR2(NOTALIVE, am_exit, BIF_ARG_1, BIF_ARG_2);
	 }
	 BIF_RET(am_true);
     }

     /*
      * The pid is local.  Verify that it refers to an existing process.
      */

     if (get_number(BIF_ARG_1) >= max_process)
	 BIF_ERROR2(BADARG, am_exit, BIF_ARG_1, BIF_ARG_2);
     rp = process_tab[get_number(BIF_ARG_1)];
     if (INVALID_PID(rp, BIF_ARG_1))
	 BIF_RET(am_true);

     /*
      * Send an 'EXIT' message or schedule an process exit.
      */

     if (rp->flags & F_TRAPEXIT && BIF_ARG_2 != am_kill)
	 deliver_exit_message(BIF_P->id, rp, BIF_ARG_2);
     else if (BIF_ARG_2 != am_normal || rp->id == BIF_P->id)
	 schedule_exit(rp, exit_value);

     /*
      * If the current process is not running (because schedule_exit()
      * was called for it above), make sure it exits.
      */

     if (BIF_P->status != P_RUNNING) {
	 BIF_P->fvalue = exit_value;
	 KILL_CATCHES(BIF_P);
	 BIF_ERROR(USER_EXIT);
     }

     BIF_RET(am_true);
}

/**********************************************************************/
/* this sets some process info- trapping exits or the error handler */

BIF_RETTYPE process_flag_2(BIF_ALIST_2)
BIF_ADECL_2
{
    uint32 old_value = NIL;	/* shut up warning about use before set */

    if (BIF_ARG_1 == am_pre_empt) {
	old_value = BIF_P->flags & F_DONT_PRE_EMPT ?  am_false : am_true;
	if (BIF_ARG_2 == am_false)
	    BIF_P->flags |= F_DONT_PRE_EMPT;
	else if (BIF_ARG_2 == am_true )
	    BIF_P->flags &= ~F_DONT_PRE_EMPT;
	else {
	    BIF_ERROR2(BADARG, am_process_flag, BIF_ARG_1, BIF_ARG_2); 
	}
	BIF_RET(old_value);
    }
    else if (BIF_ARG_1 == am_error_handler) {
	if (is_not_atom(BIF_ARG_2)) {
	    BIF_ERROR2(BADARG, am_process_flag, BIF_ARG_1, BIF_ARG_2); 
	}
	old_value = BIF_P->error_handler;
	BIF_P->error_handler = BIF_ARG_2;
	BIF_RET(old_value);
    }
    else if (BIF_ARG_1 == am_priority) {
	switch(BIF_P->prio) {
	case PRIORITY_MAX:
	    old_value = am_max; break;
	case PRIORITY_HIGH:
	    old_value = am_high; break;
	case PRIORITY_NORMAL:
	    old_value = am_normal; break;
	case PRIORITY_LOW:
	    old_value = am_low; break;
	}
	if (BIF_ARG_2 == am_max)
	    BIF_P->prio = PRIORITY_MAX;
	else if (BIF_ARG_2 == am_high)
	    BIF_P->prio = PRIORITY_HIGH;
	else if (BIF_ARG_2 == am_normal)
	    BIF_P->prio = PRIORITY_NORMAL;
	else if (BIF_ARG_2 == am_low)
	    BIF_P->prio = PRIORITY_LOW;
	else {
	    BIF_ERROR2(BADARG, am_process_flag, BIF_ARG_1, BIF_ARG_2);
	}
	BIF_RET(old_value);
    }
    else if (BIF_ARG_1 == am_trap_exit) {
	if (BIF_P->flags & F_TRAPEXIT) {
	    old_value = am_true;
	} else {
	    old_value = am_false;
	}
	if (BIF_ARG_2 == am_true) {
	    BIF_P->flags = BIF_P->flags  | F_TRAPEXIT;
	    BIF_RET(old_value);
	}
	if (BIF_ARG_2 == am_false) {
	    BIF_P->flags = BIF_P->flags & ~F_TRAPEXIT;
	    BIF_RET(old_value);
	} 
    }
    BIF_ERROR2(BADARG, am_process_flag, BIF_ARG_1, BIF_ARG_2);
 }

/**********************************************************************/

/* register(atom, Process) registers a global process (for this node) */

BIF_RETTYPE register_2(BIF_ALIST_2)   /* (Atom,Pid)   */
BIF_ADECL_2
{
     Process *rp;
     
     if ((is_not_atom(BIF_ARG_1)) || ((rp = pid2proc(BIF_ARG_2)) == NULL)) {
	 BIF_ERROR2(BADARG, am_register, BIF_ARG_1, BIF_ARG_2);
     }
     /* Check that we don't register undefined */
     if (BIF_ARG_1 == am_undefined)
	 BIF_ERROR2(BADARG, am_register, BIF_ARG_1, BIF_ARG_2);
     if (rp->reg != NULL) {
	 display(rp->id, CBUF);
	 erl_printf(CBUF," already registered as ");
	 print_atom(rp->reg->name, CBUF);
	 erl_printf(CBUF,"\n");
	 send_error_to_logger(BIF_P->group_leader);
	 BIF_ERROR2(BADARG, am_register, BIF_ARG_1, BIF_ARG_2);
     }
     if (register_process(unsigned_val(BIF_ARG_1), rp) != rp) {
	 print_atom(unsigned_val(BIF_ARG_1), CBUF);
	 erl_printf(CBUF," already registered\n");
	 send_error_to_logger(BIF_P->group_leader);
	 BIF_ERROR2(BADARG, am_register, BIF_ARG_1, BIF_ARG_2);
     }
     BIF_RET(am_true);
}


/**********************************************************************/

/* removes the registration of a process */

BIF_RETTYPE unregister_1(BIF_ALIST_1)
BIF_ADECL_1
{
    if (is_not_atom(BIF_ARG_1)) {
	BIF_ERROR1(BADARG, am_unregister, BIF_ARG_1);
    }
    if ((unregister_process(unsigned_val(BIF_ARG_1))) == NULL) {
	BIF_ERROR1(BADARG, am_unregister, BIF_ARG_1);
    }
    BIF_RET(am_true);
}

/**********************************************************************/

/* find out the pid of a regsistered process */
/* this is a rather unsafe BIF as it allows users to do nasty things. */

BIF_RETTYPE whereis_1(BIF_ALIST_1)
BIF_ADECL_1
{
    Process *rp;
 
    if (is_not_atom(BIF_ARG_1)) {
	BIF_ERROR1(BADARG, am_whereis, BIF_ARG_1);
    }
    if ((rp = whereis_process(unsigned_val(BIF_ARG_1))) == NULL) {
	BIF_RET(am_undefined);
    }
    BIF_RET(rp->id);
}

/**********************************************************************/

/* return a list of the registered processes */

BIF_RETTYPE registered_0(BIF_ALIST_0)
BIF_ADECL_0
{
    int i;
    uint32 res;
    uint32 need;
    uint32* hp;
     
    /* work out how much heap we need & maybe garb, by scanning through
       the registered process table */
    need = 0;
    for (i = 0; i < max_process; i++) {
	if ((process_tab[i] != NULL) && (process_tab[i]->reg != NULL))
	    need += 2;
    }
    if (need == 0)
	BIF_RET(NIL);
    hp = HAlloc(BIF_P, need);
     
     /* scan through again and make the list */ 
    res = NIL;
    for (i = 0; i < max_process; i++) {
	if ((process_tab[i] != NULL) && (process_tab[i]->reg != NULL)) {
	    res = CONS(hp, make_atom(process_tab[i]->reg->name), res);
	    hp += 2;
	}
    }
    BIF_RET(res);
}

/**********************************************************************/

/*
 * Send a message to Process, Port or Registered Process
 * returns the message to be sent
 */

Process *the_sender = NULL;

BIF_RETTYPE send_2(BIF_ALIST_2)
BIF_ADECL_2
{
    Process         *rp;
    int             slot;
    int             creation;
    uint32          *tp;

#ifdef SEQ_TRACE
    the_sender = BIF_P;
#endif

    switch(tag_val_def(BIF_ARG_1)) {
    case PID_DEF:   /* Send to pid */
	if ((slot = get_node(BIF_ARG_1)) != THIS_NODE) { /* Remote Pid */
	    if (dist_addrs[slot].cid == NIL)
		BIF_TRAP2(dsend_trap,BIF_ARG_1,BIF_ARG_2);
	    if (dist_send(slot, BIF_ARG_1, BIF_ARG_2) == 1) {
		/* Only ports can be busy */
		ASSERT(is_port(dist_addrs[slot].cid)); 
		erl_suspend(BIF_P, dist_addrs[slot].cid);
		BIF_ERROR(RESCHEDULE);
	    }
	    if (IS_TRACED(BIF_P))
		trace_send(BIF_P, BIF_ARG_1, BIF_ARG_2);  /* (p, to, msg )*/
	    BIF_RET2(BIF_ARG_2, 50);
	}

	if (IS_TRACED(BIF_P))
	    trace_send(BIF_P, BIF_ARG_1, BIF_ARG_2);  /* (p, to, msg )*/
	if (get_number(BIF_ARG_1) >= MAX_PROCESS)
	    BIF_ERROR2(BADARG, am_send, BIF_ARG_1, BIF_ARG_2);
	creation = get_creation(BIF_ARG_1);

	if ((creation != 0)  && (creation != this_creation)) {
	    erl_printf(CBUF, "Discarding message ");
	    display(BIF_ARG_2,CBUF);
	    erl_printf(CBUF,"to a process in an old incarnation of this node\n");
	    send_error_to_logger(BIF_P->group_leader);
	    BIF_RET(BIF_ARG_2);
	}
	rp = process_tab[get_number(BIF_ARG_1)];
	if (INVALID_PID(rp, BIF_ARG_1)) {
	    BIF_RET(BIF_ARG_2);
	}
	break;

    case PORT_DEF:  /* Send to port */
	/* XXX let port_command handle the busy stuf !!! */
	if (port[get_number_reference(BIF_ARG_1)].status & PORT_BUSY) {
	    erl_suspend(BIF_P, BIF_ARG_1);
	    BIF_ERROR(RESCHEDULE);
	}

	if (IS_TRACED(BIF_P)) 	/* trace once only !! */
	    trace_send(BIF_P, BIF_ARG_1, BIF_ARG_2);

#ifdef SEQ_TRACE
	if (SEQ_TRACE_TOKEN(BIF_P) != NIL) {
	    seq_trace_update_send(BIF_P);
	    seq_trace_output(SEQ_TRACE_TOKEN(BIF_P), BIF_ARG_2, SEQ_TRACE_SEND, BIF_ARG_1);
	}	    
#endif

	/* XXX NO GC in port command */
	port_command(BIF_ARG_1, BIF_ARG_2);

	if (BIF_P->status == P_EXITING) {
	    KILL_CATCHES(BIF_P); /* Must exit */
	    BIF_ERROR2(USER_ERROR, am_send, BIF_ARG_1, BIF_ARG_2);
	}
	BIF_RET(BIF_ARG_2);

    case ATOM_DEF:  /* Send to registered process */
	if (IS_TRACED(BIF_P))
	    trace_send(BIF_P, BIF_ARG_1, BIF_ARG_2);  /* (p, to, msg )*/

	if ((rp = whereis_process(unsigned_val(BIF_ARG_1))) == NULL) {
	    if (error_on_unregistered_send) {
		BIF_ERROR2(BADARG, am_send, BIF_ARG_1, BIF_ARG_2);
	    } else {
		BIF_RET(BIF_ARG_2);
	    }
	}
	if (rp->status == P_EXITING) {
	    BIF_RET(BIF_ARG_2);
	}
	break;

    case TUPLE_DEF: /* Send remote */
	tp = ptr_val(BIF_ARG_1);
	if (*tp != make_arityval(2))
	    BIF_ERROR2(BADARG, am_send, BIF_ARG_1, BIF_ARG_2);
	if (is_not_atom(tp[1]) || is_not_atom(tp[2]))
	    BIF_ERROR2(BADARG, am_send, BIF_ARG_1, BIF_ARG_2);

	/* find_or_insert_dist_slot will complain */
	if ((slot = find_or_insert_dist_slot(tp[2])) < 0) {
	    if (IS_TRACED(BIF_P))
		trace_send(BIF_P, BIF_ARG_1, BIF_ARG_2);
	    BIF_RET(BIF_ARG_2);
	}

	if (slot == THIS_NODE) {
	    if (IS_TRACED(BIF_P))
		trace_send(BIF_P, BIF_ARG_1, BIF_ARG_2);
	    if ((rp = whereis_process(unsigned_val(tp[1]))) == NULL) {
		BIF_RET(BIF_ARG_2);
	    }
	    if (rp->status == P_EXITING) {
		BIF_RET(BIF_ARG_2);
	    }
	    break; /* jump to send_message */
	}

	if (dist_addrs[slot].cid == NIL) {
	    BIF_TRAP2(dsend_trap,BIF_ARG_1,BIF_ARG_2);
	}
	if (dist_reg_send(slot, BIF_P->id, tp[1], BIF_ARG_2) == 1) {
	    ASSERT(is_port(dist_addrs[slot].cid));
	    erl_suspend(BIF_P, dist_addrs[slot].cid);
	    BIF_ERROR(RESCHEDULE);
	}
	if (IS_TRACED(BIF_P))
	    trace_send(BIF_P, BIF_ARG_1, BIF_ARG_2);
	BIF_RET(BIF_ARG_2);

    default:
	if (IS_TRACED(BIF_P)) /* XXX Is this really neccessary ??? */
	    trace_send(BIF_P, BIF_ARG_1, BIF_ARG_2);
	BIF_ERROR2(BADARG, am_send, BIF_ARG_1, BIF_ARG_2);
    }
    send_message(BIF_P, rp, BIF_ARG_2);
    BIF_RET(BIF_ARG_2);
}


/**********************************************************************/
/* apply is implemented as an instruction */
/* THIS BIF DOES NOT NEED TO BE IMPLEMENTED */
/* calls to the BIF apply are inserted inb the byte code files
   as enter/call remote erlang:apply/3
   The loader recognises these call and replaces them by
   apply_enter or apply_call - see the emulator code
   */

/* ARGSUSED */

BIF_RETTYPE apply_3(BIF_ALIST_3)
BIF_ADECL_3
{
    BIF_ERROR3(BADARG, am_apply, BIF_ARG_1, BIF_ARG_2, BIF_ARG_3);
}


/* maths abs function */
BIF_RETTYPE abs_1(BIF_ALIST_1)
BIF_ADECL_1
{
    uint32 res;
    sint32 i;
    uint32* hp;

    /* integer arguments */
    if (is_small(BIF_ARG_1)) {
	i = signed_val(BIF_ARG_1);
	i = abs(i);
	BIF_RET(make_small(i));
    }
    else if (is_big(BIF_ARG_1)) {
	if (big_sign(BIF_ARG_1)) {
	    int sz = big_arity(BIF_ARG_1) + 1;
	    uint32* x;

	    hp = ArithAlloc(BIF_P, sz);	/* See note at beginning of file */
	    sz--;
	    res = make_big(hp);
	    x = ptr_val(BIF_ARG_1);
	    *hp++ = make_thing(sz);  /* write thing without signbit */
	    x++;                          /* skip thing */
	    while(sz--)
		*hp++ = *x++;
	    BIF_RET(res);
	}
	else
	    BIF_RET(BIF_ARG_1);
    }
    else if (is_float(BIF_ARG_1)) {
	FloatDef f;

	GET_DOUBLE(BIF_ARG_1, f);
	if (f.fd < 0.0) {
	    hp = ArithAlloc(BIF_P, 3); 	/* See note at beginning of file */
	    f.fd = fabs(f.fd);
	    res = make_float(hp);
	    PUT_DOUBLE(f, hp);
	    BIF_RET(res);
	}
	else
	    BIF_RET(BIF_ARG_1);
    }
    BIF_ERROR1(BADARG, am_abs, BIF_ARG_1);
}

/**********************************************************************/

/* integer to float */
BIF_RETTYPE float_1(BIF_ALIST_1)
BIF_ADECL_1
{
     uint32 res;
     uint32* hp;
     sint32 i;
     FloatDef f;
     
     /* check args */
     if (is_not_integer(BIF_ARG_1)) {
	 if (is_float(BIF_ARG_1)) 
	     BIF_RET(BIF_ARG_1);
	 BIF_ERROR1(BADARG, am_float, BIF_ARG_1);
     }
     if (is_small(BIF_ARG_1)) {
	 i = signed_val(BIF_ARG_1);
	 f.fd = i;		/* use "C"'s auto casting */
     }
     else {
	 if (!FP_PRE_CHECK_OK()) {
	     BIF_ERROR1(BADARG, am_float, BIF_ARG_1);
	 }
	 f.fd = big_to_double(BIF_ARG_1);
	 if (!FP_RESULT_OK(f.fd)) {
	     BIF_ERROR1(BADARG, am_float, BIF_ARG_1);
	 }
     }
     hp = ArithAlloc(BIF_P, 3);	/* See note at beginning of file */
     res = make_float(hp);
     PUT_DOUBLE(f, hp);
     BIF_RET(res);
}

/**********************************************************************/

/* truncate a float returning an integer */
BIF_RETTYPE trunc_1(BIF_ALIST_1)
BIF_ADECL_1
{
    uint32 res;
    FloatDef f;
     
    /* check arg */
    if (is_not_float(BIF_ARG_1)) {
	if (is_integer(BIF_ARG_1)) 
	    BIF_RET(BIF_ARG_1);
	BIF_ERROR1(BADARG, am_trunc, BIF_ARG_1);
    }
    /* get the float */
    GET_DOUBLE(BIF_ARG_1, f);

    /* truncate it and return the resultant integer */
    res = double_to_integer(BIF_P, (f.fd >= 0.0) ? floor(f.fd) : ceil(f.fd));
    BIF_RET(res);
}

/**********************************************************************/

/* round a 'number' to an integer */

BIF_RETTYPE round_1(BIF_ALIST_1)
BIF_ADECL_1
{
    uint32 res;
    FloatDef f;
     
    /* check arg */ 
    if (is_not_float(BIF_ARG_1)) {
	if (is_integer(BIF_ARG_1)) 
	    BIF_RET(BIF_ARG_1);
	BIF_ERROR1(BADARG, am_round, BIF_ARG_1);
    }
     
    /* get the float */
    GET_DOUBLE(BIF_ARG_1, f);

    /* round it and return the resultant integer */
    res = double_to_integer(BIF_P, (f.fd > 0.0) ? f.fd + 0.5 : f.fd - 0.5);
    BIF_RET(res);
}
/**********************************************************************/

/* 
 * Open a port. Most of the work is not done here but rather in
 * the file io.c.
 * Error returns: -1 or -2 returned from open_driver (-2 implies
 * 'errno' contains error code; -1 means we don't really know what happened),
 * -3 if argument parsing failed.
 */
static int open_port(pid, name, settings)
uint32 pid, name, settings;
{
    int i, port_num;
    uint32 option;
    uint32 arity;
    uint32 *tp;
    uint32 *nargs;
    DriverEntry* driver;
    char* name_buf;
    SysDriverOpts opts;
    int binary_io;
    int soft_eof;
    int linebuf;

    /* These are the defaults */
    opts.packet_bytes = 0;
    opts.use_stdio = 1;
    opts.redir_stderr = 0;
    opts.read_write = 0;
    opts.hide_window = 0;
    opts.wd = NULL;
    opts.envir = NULL;
    binary_io = 0;
    soft_eof = 0;
    linebuf = 0;

    if (is_not_list(settings) && is_not_nil(settings))
	return -3;

    /*
     * Parse the settings.
     */

    if (is_not_nil(settings)) {
	nargs = ptr_val(settings);
	while (1) {
	    if (is_tuple(*nargs)) {
		tp = ptr_val(*nargs);
		arity = *tp++;
		if (arity != make_arityval(2))
		    return -3;
		option = *tp++;
		if (option == am_packet) {
		   if (is_not_small(*tp))
		      return -3;
		   opts.packet_bytes = signed_val(*tp);
		   switch (opts.packet_bytes) {
		    case 1:
		    case 2:
		    case 4:
		      break;
		    default:
		      return -3;
		   }
		} else if (option == am_line) {
		    if (is_not_small(*tp))
			return -3;
		    linebuf = signed_val(*tp);
		    if(linebuf <= 0)
			return -3;
		} else if (option == am_env) {
		   if (is_not_binary(*tp))
		      return -3;
		   opts.envir = (char *) ((ProcBin *) ptr_val(*tp))->bytes;
		} else if (option == am_cd) {
		   if (is_not_binary(*tp))
		      return -3;
		   opts.wd = (char *) ((ProcBin *) ptr_val(*tp))->bytes;
		} else
		   return -3;
	    } else if (*nargs == am_stream) {
		opts.packet_bytes = 0;
	    } else if (*nargs == am_use_stdio) {
		opts.use_stdio = 1;
	    } else if (*nargs == am_stderr_to_stdout) {
		opts.redir_stderr = 1;
	    } else if (*nargs == am_line) {
		linebuf = 512;
	    } else if (*nargs == am_nouse_stdio) {
		opts.use_stdio = 0;
	    } else if (*nargs == am_binary) {
		binary_io = 1;
	    } else if (*nargs == am_in) {
		opts.read_write |= DO_READ;
	    } else if (*nargs == am_out) {
		opts.read_write |= DO_WRITE;
	    } else if (*nargs == am_eof) {
		soft_eof = 1;
	    } else if (*nargs == am_hide) {
		opts.hide_window = 1;
	    } else {
		return -3;
	    }
	    if (is_nil(*++nargs)) 
		break;
	    if (is_not_list(*nargs)) 
		return -3;
	    nargs = ptr_val(*nargs);
	}
    }
    if (opts.read_write == 0)	/* implement default */
	opts.read_write = DO_READ|DO_WRITE;

    /* Mutually exclusive arguments. */
    if((linebuf && opts.packet_bytes) || 
       (opts.redir_stderr && !opts.use_stdio))
	return -3; 

    /*
     * Parse the first argument and start the appropriate driver.
     */

    if (is_atom(name) || is_string(name)) {
	/* a vanilla port */
	if (is_atom(name)) {
	    if (atom_tab(unsigned_val(name))->len >= TMP_BUF_SIZE)
		return -3;
	    sys_memcpy(tmp_buf, atom_tab(unsigned_val(name))->name, 
		       atom_tab(unsigned_val(name))->len);
	    tmp_buf[atom_tab(unsigned_val(name))->len] = '\0';
	} else {
	    i = intlist_to_buf(name, tmp_buf, TMP_BUF_SIZE);
	    tmp_buf[i] = '\0';
	}
	name_buf = tmp_buf;
	driver = &vanilla_driver_entry;
    } else {   
	if (is_not_tuple(name))
	    return -3;		/* Not a process or fd port */
	tp = ptr_val(name);
	arity = *tp++;

	if (*tp == am_spawn) {	/* A process port */
	    if (arity != make_arityval(2)) {
		return -3;
	    }
	    name = tp[1];
	    if (is_atom(name)) {
		if (atom_tab(unsigned_val(name))->len >= TMP_BUF_SIZE)
		    return -3;
		sys_memcpy(tmp_buf, atom_tab(unsigned_val(name))->name,
			   atom_tab(unsigned_val(name))->len);
		tmp_buf[atom_tab(unsigned_val(name))->len] = '\0';
	    } else  if (is_string(name)) {
		 i = intlist_to_buf(name,tmp_buf, TMP_BUF_SIZE);
		 tmp_buf[i] = '\0';
	    } else
		return -3;
	    name_buf = tmp_buf;
	    driver = &spawn_driver_entry;
	} else if (*tp == am_fd) { /* An fd port */
	    int n;
	    char sbuf[16];
	    char* p;

	    opts.ifd = unsigned_val(tp[1]);
	    opts.ofd = unsigned_val(tp[2]);
	    if (arity != make_arityval(3)) {
		return -3;
	    }
	    if (is_not_small(tp[1]) || is_not_small(tp[2])) {
		return -3;
	    }

	    /* Syntesize name from input and output descriptor. */
	    name_buf = tmp_buf;
	    p = int_to_buf(opts.ifd, sbuf);
	    n = sys_strlen(p);
	    sys_strncpy(name_buf, p, n);
	    name_buf[n] = '/';
	    p = int_to_buf(opts.ofd, sbuf);
	    sys_strcpy(name_buf+n+1, p);

	    driver = &fd_driver_entry;
	} else
	    return -3;
    }

    if ((port_num = open_driver(driver, pid, name_buf, &opts)) < 0) {
	DEBUGF(("open_driver returned %d\n", port_num));
	return port_num;
    }

    if (binary_io)
	port[port_num].status |= BINARY_IO;
    if (soft_eof)
	port[port_num].status |= SOFT_EOF;
    if (linebuf && port[port_num].linebuf == NULL){
	port[port_num].linebuf = allocate_linebuf(linebuf); 
	port[port_num].status |= LINEBUF_IO;
    }

    return port_num;
}

BIF_RETTYPE open_port_prim_2(BIF_ALIST_2)
BIF_ADECL_2
{
    int port_num;
    uint32 port_val;
    int am;
    char *str;

    if ((port_num = open_port(BIF_P->id, BIF_ARG_1, BIF_ARG_2)) < 0) {
       if (port_num == -3)
       {
	  BIF_ERROR2(BADARG, am_open_port_prim, BIF_ARG_1, BIF_ARG_2);
       }

       if (port_num == -2)
	  str = erl_errno_id(errno);
       else
	  str = "einval";

       am = atom_put(str, strlen(str));

       BIF_P->fvalue = make_atom(am);
       BIF_ERROR2(USER_ERROR, am_open_port_prim, BIF_ARG_1, BIF_ARG_2);
    }

    port_val = make_port2(THIS_NODE, port_num);
    port[port_num].links = new_link(port[port_num].links,LNK_LINK,
				    BIF_P->id, NIL);
    BIF_P->links = new_link(BIF_P->links, LNK_LINK, port_val, NIL);
    BIF_RET(port_val);
}


/**********************************************************************/

/* return the length of a list */

BIF_RETTYPE length_1(BIF_ALIST_1)
BIF_ADECL_1
{
     uint32 list;
     uint32 i;
     
     if (is_nil(BIF_ARG_1)) 
	 BIF_RET(SMALL_ZERO);
     if (is_not_list(BIF_ARG_1)) {
	 BIF_ERROR1(BADARG, am_length, BIF_ARG_1);
     }
     list = BIF_ARG_1;
     i = 0;
     while (is_list(list)) {
	 i++;
	 list = CDR(ptr_val(list));
     }
     if (is_not_nil(list))  {
	 BIF_ERROR1(BADARG, am_length, BIF_ARG_1);
     }
     BIF_RET(make_small(i));
}


/**********************************************************************/

/* append a list to any object */

BIF_RETTYPE append_2(BIF_ALIST_2)
BIF_ADECL_2
{
     uint32 list;
     uint32 copy;
     uint32 last;
     uint32 need;
     uint32* hp;
     int i;

     if ((i = list_length(BIF_ARG_1)) < 0) {
	 BIF_ERROR1(BADARG, am_append, BIF_ARG_1);
     }
     if (i == 0)
	 BIF_RET(BIF_ARG_2);
     if (is_nil(BIF_ARG_2))
	 BIF_RET(BIF_ARG_1);

     /* XXX HAlloc is MACRO and will use second argument multiple times */
     need = 2*i;
     hp = HAlloc(BIF_P, need); 
     list = BIF_ARG_1;
     copy = last = CONS(hp, CAR(ptr_val(list)), make_list(hp+2));
     list = CDR(ptr_val(list));
     hp += 2;
     i--;
     while(i--) {
	 uint32* listp = ptr_val(list);
	 last = CONS(hp, CAR(listp), make_list(hp+2));
	 list = CDR(listp);
	 hp += 2;
     }
     CDR(ptr_val(last)) = BIF_ARG_2;
     BIF_RET(copy);
 }

/**********************************************************************/

/* append a list to any object */

#define SUBTRACT_SZ 10  /* preallocated memory for small lists */

BIF_RETTYPE subtract_2(BIF_ALIST_2)
BIF_ADECL_2
{
     uint32  list;
     uint32* hp;
     uint32  need;
     uint32  res;
     uint32  small_vec[SUBTRACT_SZ];
     uint32* vec_p;
     uint32* vp;
     int     i;
     int     n;
     int     m;

     if ((n = list_length(BIF_ARG_1)) < 0) {
	 BIF_ERROR2(BADARG, am_subtract, BIF_ARG_1, BIF_ARG_2);
     }
     if ((m = list_length(BIF_ARG_2)) < 0) {
	 BIF_ERROR2(BADARG, am_subtract, BIF_ARG_1, BIF_ARG_2);
     }

     if (n == 0)
	 BIF_RET(NIL);
     if (m == 0)
	 BIF_RET(BIF_ARG_1);

     /* allocate element vector */
     if (n <= SUBTRACT_SZ)
	 vec_p = small_vec;
     else
	 vec_p = (uint32*) safe_alloc(n * sizeof(uint32));

     /* PUT ALL ELEMENTS IN VP */
     vp = vec_p;
     list = BIF_ARG_1;
     i = n;
     while(i--) {
	 uint32* listp = ptr_val(list);
	 *vp++ = CAR(listp);
	 list = CDR(listp);
     }

     /* UNMARK ALL DELETED CELLS */
     list = BIF_ARG_2;
     m = 0;  /* number of deleted elements */
     while(is_list(list)) {
	 uint32* listp = ptr_val(list);
	 uint32  elem = CAR(listp);
	 i = n;
	 vp = vec_p;
	 while(i--) {
	     if ((*vp != 0) && eq(*vp, elem)) {
		 *vp = 0;
		 m++;
		 break;
	     }
	     vp++;
	 }
	 list = CDR(listp);
     }

     if (m == n)      /* All deleted ? */
	 res = NIL;
     else if (m == 0)  /* None deleted ? */
	 res = BIF_ARG_1;
     else { 	 /* REBUILD LIST */
	 res = NIL;
	 need = 2*(n - m);
	 hp = HAlloc(BIF_P, need);
	 vp = vec_p + n - 1;
	 while(vp >= vec_p) {
	     if (*vp != 0) {
		 res = CONS(hp, *vp, res);
		 hp += 2;
	     }
	     vp--;
	 }
     }
     if (vec_p != small_vec)
	 sys_free(vec_p);
     BIF_RET(res);
}

/**********************************************************************/

BIF_RETTYPE member_2(BIF_ALIST_2)
BIF_ADECL_2
{
     uint32 term;
     uint32 list;
     int max_iter = 10 * CONTEXT_REDS;

     if (is_nil(BIF_ARG_2)) {
	 BIF_RET(am_false);
     } else if (is_not_list(BIF_ARG_2)) {
	 BIF_ERROR1(BADARG, am_member, BIF_ARG_1);
     }

     term = BIF_ARG_1;
     list = BIF_ARG_2;
     while (is_list(list)) {
	 if (--max_iter < 0) {
	     BIF_RET2(list, CONTEXT_REDS);
	 } else if (eq(CAR(ptr_val(list)), term)) {
	     BIF_RET2(am_true, CONTEXT_REDS - max_iter/10);
	 }
	 list = CDR(ptr_val(list));
     }
     if (is_not_nil(list))  {
	 BIF_ERROR1(BADARG, am_member, BIF_ARG_1);
     }
     BIF_RET2(am_false, CONTEXT_REDS - max_iter/10);
}

/**********************************************************************/

BIF_RETTYPE reverse_2(BIF_ALIST_2)
BIF_ADECL_2
{
     uint32 list;
     uint32 result;
     uint32* hp;
     uint32* hend;

     int max_iter = CONTEXT_REDS * 10;
     
     if (is_nil(BIF_ARG_1)) {
	 BIF_RET(BIF_ARG_2);
     } else if (is_not_list(BIF_ARG_1)) {
	 BIF_ERROR1(BADARG, am_reverse, BIF_ARG_1);
     }

     list = BIF_ARG_1;
     result = BIF_ARG_2;
     hp = hend = NULL;
     while (is_list(list)) {
	 uint32* pair = ptr_val(list);
	 if (--max_iter == 0) {
#ifdef DEBUG
	     while (hp < hend) {
		 *hp++ = NIL;
	     }
#endif
	     hp = HAlloc(BIF_P, 3);
	     BIF_RET2(TUPLE2(hp, list, result), CONTEXT_REDS);
	 }
	 if (hp == hend) {
	     hp = HAlloc(BIF_P, 64);
	     hend = hp + 64;
	 }
	 result = CONS(hp, CAR(pair), result);
	 hp += 2;
	 list = CDR(pair);
     }
#ifdef DEBUG
     while (hp < hend) {
	 *hp++ = NIL;
     }
#endif
     if (is_not_nil(list))  {
	 BIF_ERROR1(BADARG, am_reverse, BIF_ARG_1);
     }
     BIF_RET2(result, CONTEXT_REDS - max_iter / 10);
}

/**********************************************************************/

/* returns the head of a list - this function is unecessary
   and is only here to keep Robert happy (Even more, since it's OP as well) */
BIF_RETTYPE hd_1(BIF_ALIST_1)
BIF_ADECL_1
{
     if (is_not_list(BIF_ARG_1)) {
	 BIF_ERROR1(BADARG, am_hd, BIF_ARG_1);
     }
     BIF_RET(CAR(ptr_val(BIF_ARG_1)));
}

/**********************************************************************/

/* returns the tails of a list - same comment as above */

BIF_RETTYPE tl_1(BIF_ALIST_1)
BIF_ADECL_1
{
    if (is_not_list(BIF_ARG_1)) {
	BIF_ERROR1(BADARG, am_tl, BIF_ARG_1);
    }
    BIF_RET(CDR(ptr_val(BIF_ARG_1)));
}

/**********************************************************************/

/* returns the size of a tuple  or a binary */
/* btw: size used to be called aritu in the old version */

BIF_RETTYPE size_1(BIF_ALIST_1)
BIF_ADECL_1
{
    if (is_tuple(BIF_ARG_1)) {
	uint32 *tupleptr = ptr_val(BIF_ARG_1);

	BIF_RET(make_small(arityval(*tupleptr)));
    }
    else if (is_binary(BIF_ARG_1)) {
	ProcBin *bptr = (ProcBin*) ptr_val(BIF_ARG_1);
	BIF_RET(make_small(bptr->size));
    }
    BIF_ERROR1(BADARG, am_size, BIF_ARG_1);
}

/**********************************************************************/

/* return the N'th element of a tuple */

BIF_RETTYPE element_2(BIF_ALIST_2)
BIF_ADECL_2
{
    if (is_not_small(BIF_ARG_1)) {
	BIF_ERROR2(BADARG, am_element, BIF_ARG_1, BIF_ARG_2);
    }
    if (is_tuple(BIF_ARG_2)) {
	uint32 *tuple_ptr = ptr_val(BIF_ARG_2);
	sint32 ix = signed_val(BIF_ARG_1);

	if ((ix >= 1) && (ix <= arityval(*tuple_ptr)))
	    BIF_RET(tuple_ptr[ix]);
    }
    BIF_ERROR2(BADARG, am_element, BIF_ARG_1, BIF_ARG_2);
}

/**********************************************************************/

/* we have discussed a destructive set elements - maybe someday */
/* set the n'th element in a tuple */

BIF_RETTYPE setelement_3(BIF_ALIST_3)
BIF_ADECL_3
{
    uint32* ptr;
    uint32* hp;
    uint32* resp;
    uint32  ix;
    uint32  size;

    if (is_not_small(BIF_ARG_1) || is_not_tuple(BIF_ARG_2))
	BIF_ERROR3(BADARG, am_setelement, BIF_ARG_1, BIF_ARG_2, BIF_ARG_3);
    ptr = ptr_val(BIF_ARG_2);
    ix = signed_val(BIF_ARG_1);
    size = arityval(*ptr) + 1;   /* include arity */
    if ((ix < 1) || (ix >= size))
	BIF_ERROR3(BADARG, am_setelement, BIF_ARG_1, BIF_ARG_2, BIF_ARG_3);

    hp = HAlloc(BIF_P, size);

    /* copy the tuple */
    resp = hp;
    while (size--)  /* XXX memcpy ? */
	*hp++ = *ptr++;
    resp[ix] = BIF_ARG_3;
    BIF_RET(make_tuple(resp));
}

/**********************************************************************/

BIF_RETTYPE make_tuple_2(BIF_ALIST_2)
BIF_ADECL_2
{
    int n;
    uint32* hp;
    uint32 res;

    if (is_not_small(BIF_ARG_1) || (n = signed_val(BIF_ARG_1)) < 0) {
	BIF_ERROR2(BADARG, am_make_tuple, BIF_ARG_1, BIF_ARG_2);
    }
    hp = HAlloc(BIF_P, n+1);
    res = make_tuple(hp);
    *hp++ = make_arityval(n);
    while (n--) {
	*hp++ = BIF_ARG_2;
    }
    BIF_RET(res);
}


/**********************************************************************/

BIF_RETTYPE append_element_2(BIF_ALIST_2)
BIF_ADECL_2
{
    uint32* ptr;
    uint32* hp;
    uint32 arity;
    uint32 res;

    if (is_not_tuple(BIF_ARG_1)) {
	BIF_ERROR2(BADARG, am_append_element, BIF_ARG_1, BIF_ARG_2);
    }
    ptr = ptr_val(BIF_ARG_1);
    arity = arityval(*ptr);
    hp = HAlloc(BIF_P, arity + 2);
    res = make_tuple(hp);
    *hp = make_arityval(arity+1);
    while (arity--) {
	*++hp = *++ptr;
    }
    *++hp = BIF_ARG_2;
    BIF_RET(res);
}

/**********************************************************************/

/* convert an atom to a list of ascii integer */

BIF_RETTYPE atom_to_list_1(BIF_ALIST_1)
BIF_ADECL_1
{
    uint32 need;
    uint32* hp;
    Atom* ap;

    if (is_not_atom(BIF_ARG_1))
	BIF_ERROR1(BADARG, am_atom_to_list, BIF_ARG_1);
     
    /* read data from atom table */
    ap = atom_tab(unsigned_val(BIF_ARG_1));
    if (ap->len == 0)
	BIF_RET(NIL);	/* the empty atom */
    need = ap->len*2;
    hp = HAlloc(BIF_P, need);
    BIF_RET(buf_to_intlist(&hp,ap->name,ap->len, NIL));
}

/**********************************************************************/

/* convert a list of ascii intgers to an atom */
 
BIF_RETTYPE list_to_atom_1(BIF_ALIST_1)
BIF_ADECL_1
{
    int i;

    if ((i = intlist_to_buf(BIF_ARG_1,tmp_buf,TMP_BUF_SIZE)) < 0) {
	BIF_ERROR1(BADARG, am_list_to_atom, BIF_ARG_1);
    }
    else if (i > MAX_ATOM_LENGTH) {
	BIF_ERROR1(SYSTEM_LIMIT, am_list_to_atom, BIF_ARG_1);
    }
    else {
	BIF_RET(am_atom_put(tmp_buf, i));
    }
}


/**********************************************************************/

/* convert an integer to a list of ascii integers */

BIF_RETTYPE integer_to_list_1(BIF_ALIST_1)
BIF_ADECL_1
{
    uint32* hp;
    uint32* hp_end;
    uint32 need;

    if (is_not_integer(BIF_ARG_1)) {
	BIF_ERROR1(BADARG, am_integer_to_list, BIF_ARG_1);
    }

    if (is_small(BIF_ARG_1)) {
	byte *c;
	int n;
	char ibuf[12];

	c = (byte*) int_to_buf(signed_val(BIF_ARG_1), ibuf);
	n = sys_strlen((char*)c);
	need = 2*n;
	hp = HAlloc(BIF_P, need);
	BIF_RET(buf_to_intlist(&hp, c, n, NIL));
    }
    else {
	int n = big_decimal_estimate(BIF_ARG_1);
	uint32 res;

	need = 2*n;
	hp = HAlloc(BIF_P, need);
	hp_end = hp + need;
	res = big_to_list(BIF_ARG_1, &hp);
#ifdef DEBUG
	while (hp < hp_end) *hp++ = NIL;
#endif
	BIF_RET(res);
    }
}

/**********************************************************************/

/* convert a list of ascii ascii integer value to an integer */

BIF_RETTYPE list_to_integer_1(BIF_ALIST_1)
BIF_ADECL_1
{
     uint32 *pos;
     sint32 i = 0;
     int skip = 0;
     int neg = 0;
     int n = 0;
     int m;
     int lg2;
     uint32 res;
     uint32* hp;
     uint32* hp_end;

     /* must be a list */
     if (is_not_list(BIF_ARG_1)) {
	 BIF_ERROR1(BADARG, am_list_to_integer, BIF_ARG_1);
     }

     /* if first char is a '-' then it is a negative integer */
     pos = ptr_val(BIF_ARG_1);
     if (*pos == make_small('-')) {
	  neg = 1;
	  skip = 1;
	  if (is_not_list(*(pos + 1))) {
	      BIF_ERROR1(BADARG, am_list_to_integer, BIF_ARG_1);
	  }
	  pos = ptr_val(*(pos + 1));
     } else {
	  if (*pos == make_small('+')) {
	       /* ignore plus */
	      skip = 1;
	      if (is_not_list(*(pos + 1))) {
		  BIF_ERROR1(BADARG, am_list_to_integer, BIF_ARG_1);
	      }
	      pos = ptr_val(*(pos + 1));
	  }
      }

     /* Calculate size and do type check */

     while(1) {
	 if (is_not_small(*pos)) {
	     BIF_ERROR1(BADARG, am_list_to_integer, BIF_ARG_1);
	 }
	 if (unsigned_val(*pos) < '0' || unsigned_val(*pos) > '9') {
	     BIF_ERROR1(BADARG, am_list_to_integer, BIF_ARG_1);
	 }
	 i = i * 10;
	 i = i + unsigned_val(*pos) - '0';
	 n++;
	 if (*(pos + 1) == NIL) break;
	 if (is_not_list(*(pos + 1))) {
	     BIF_ERROR1(BADARG, am_list_to_integer, BIF_ARG_1);
	 }
	 pos = ptr_val(*(pos + 1));
     }

      /* If n <= 8 then we know it's a small int 
      ** since 2^27 = 134217728. If n > 8 then we must
      ** construct a bignum and let that routine do the checking
      */

     if (n <= 8)  /* It must be small */
     {
	 if (neg) i = -i;
	 BIF_RET(make_small(i));
     }

     lg2 =  (n+1)*230/69+1;
     m  = (lg2+D_EXP-1)/D_EXP; /* number of digits */
     m  = ((m+1)>>1) + 1;      /* number of words + thing */

     hp = HAlloc(BIF_P, m);
     hp_end = hp + m;

     pos = ptr_val(BIF_ARG_1);
     if (skip)
	 pos = ptr_val(*(pos+1));

     /* load first digits (at least one digit) */
     if ((i = (n % 4)) == 0)
	 i = 4;
     n -= i;
     m = 0;
     while(i--) {
	 m = 10*m + (unsigned_val(*pos) - '0');
	 pos = ptr_val(*(pos + 1));
     }
     res = small_to_big(m, hp);  /* load first digits */

     while(n) {
	 i = 4;
	 n -= 4;
	 m = 0;
	 while(i--) {
	     m = 10*m + (unsigned_val(*pos) - '0');
	     pos = ptr_val(*(pos+1));
	 }
	 if (is_small(res))
	     res = small_to_big(signed_val(res), hp);
	 res = big_times_small(res, D_DECIMAL_BASE, hp);
	 if (is_small(res))
	     res = small_to_big(signed_val(res), hp);
	 res = big_plus_small(res, m, hp);
     }

     if (is_big(res))  /* check if small */
	 res = big_plus_small(res, 0, hp); /* includes convert to small */

     if (neg) {
	 if (is_small(res))
	     res = make_small(-signed_val(res));
	 else {
	     pos = ptr_val(res); /* point to thing */
	     *pos |= BIG_SIGN_BIT;
	 }
     }
     if (is_big(res))
	 hp += (big_arity(res)+1);
#ifdef DEBUG	
	while (hp < hp_end) *hp++ = NIL;
#endif     
     BIF_RET(res);
 }

/**********************************************************************/

/* convert a float to a list of ascii characters */

BIF_RETTYPE float_to_list_1(BIF_ALIST_1)
BIF_ADECL_1
{
     int i;
     uint32 need;
     uint32* hp;
     FloatDef f;
     char fbuf[30];
     
     /* check the arguments */
     if (is_not_float(BIF_ARG_1))
	 BIF_ERROR1(BADARG, am_float_to_list, BIF_ARG_1);
     GET_DOUBLE(BIF_ARG_1, f);
     if ((i = sys_double_to_chars(f.fd, fbuf)) <= 0)
	 BIF_ERROR1(INTERNAL_ERROR, am_float_to_list, BIF_ARG_1);
     need = i*2;
     hp = HAlloc(BIF_P, need);
     BIF_RET(buf_to_intlist(&hp, fbuf, i, NIL));
 }

/**********************************************************************/

/* convert a list of ascii  integer values e's +'s and -'s to a float */

BIF_RETTYPE list_to_float_1(BIF_ALIST_1)
BIF_ADECL_1
{
    int i;
    FloatDef f;
    uint32 res;
    uint32* hp;

    if ((i = intlist_to_buf(BIF_ARG_1,tmp_buf,TMP_BUF_SIZE-1)) < 0) {
	BIF_ERROR1(BADARG, am_list_to_float, BIF_ARG_1);
    }
    tmp_buf[i] = '\0';		/* null terminal */

    if (!FP_PRE_CHECK_OK())
	BIF_ERROR1(BADARG, am_list_to_float, BIF_ARG_1);

    if (sys_chars_to_double((char*)tmp_buf, &f.fd) != 0)
	BIF_ERROR1(BADARG, am_list_to_float, BIF_ARG_1);

    if (FP_RESULT_OK(f.fd)) {
	hp = HAlloc(BIF_P, 3);

	res = make_float(hp);
	PUT_DOUBLE(f, hp);
	BIF_RET(res);
    }
    else
	BIF_ERROR1(BADARG, am_list_to_float, BIF_ARG_1);
}


/**********************************************************************/

/* convert a tuple to a list */

BIF_RETTYPE tuple_to_list_1(BIF_ALIST_1)
BIF_ADECL_1
{
    uint32 n;
    uint32 *tupleptr;
    uint32 list = NIL;
    uint32* hp;

    if (is_not_tuple(BIF_ARG_1))  {
	BIF_ERROR1(BADARG, am_tuple_to_list, BIF_ARG_1);
    }

    tupleptr = ptr_val(BIF_ARG_1);
    n = arityval(*tupleptr);
    hp = HAlloc(BIF_P, 2 * n);
    tupleptr++;

    while(n--) {
	list = CONS(hp, tupleptr[n], list);
	hp += 2;
    }
    BIF_RET(list);
}

/**********************************************************************/

/* convert a list to a tuple */

BIF_RETTYPE list_to_tuple_1(BIF_ALIST_1)
BIF_ADECL_1
{
    uint32  list = BIF_ARG_1;
    uint32* cons;
    uint32 res;
    uint32* hp;
    int len;

    if ((len = list_length(list)) < 0)
	BIF_ERROR1(BADARG, am_list_to_tuple, BIF_ARG_1);

    hp = HAlloc(BIF_P, len+1);
    res = make_tuple(hp);
    *hp++ = make_arityval(len);
    while(is_list(list)) {
	cons = ptr_val(list);
	*hp++ = CAR(cons);
	list = CDR(cons);
    }
    BIF_RET(res);
}

/**********************************************************************/

/* return the pid of our own process, in most cases this has been replaced by
   a machine instruction */

BIF_RETTYPE self_0(BIF_ALIST_0)
BIF_ADECL_0
{
     BIF_RET(BIF_P->id);
     
}

/**********************************************************************/

/* make a unique reference - not so unique eh? */

static uint32 reference = 0;   /* XXX fix proper init */

/* For internal use */
static uint32 make_ref()
{
   reference++;
   if (reference >= MAX_REFERENCE)
      reference = 0;
   return make_refer(THIS_NODE,reference);
}

BIF_RETTYPE make_ref_0(BIF_ALIST_0)
BIF_ADECL_0
{
    reference++;
    if (reference >= MAX_REFERENCE)
	reference = 0;
    BIF_RET(make_refer(THIS_NODE,reference));
}

/**********************************************************************/

/* return the time of day */

BIF_RETTYPE time_0(BIF_ALIST_0)
BIF_ADECL_0
{
     int hour, minute, second;
     uint32* hp;

     get_time(&hour, &minute, &second);
     hp = HAlloc(BIF_P, 4);	/* {hour, minute, second}  + arity */
     BIF_RET(TUPLE3(hp, make_small(hour), make_small(minute),
		    make_small(second)));
}
/**********************************************************************/

/* return the date */

BIF_RETTYPE date_0(BIF_ALIST_0)
BIF_ADECL_0
{
     int year, month, day;
     uint32* hp;
     
     get_date(&year, &month, &day);
     hp = HAlloc(BIF_P, 4);	/* {year, month, day}  + arity */
     BIF_RET(TUPLE3(hp, make_small(year), make_small(month), make_small(day)));
}

/**********************************************************************/

/* return the universal time */

BIF_RETTYPE universaltime_0(BIF_ALIST_0)
BIF_ADECL_0
{
     int year, month, day;
     int hour, minute, second;
     uint32 res1, res2;
     uint32* hp;

     /* read the clock */
     get_universaltime(&year, &month, &day, &hour, &minute, &second);

     hp = HAlloc(BIF_P, 4+4+3);

     /* and return the tuple */
     res1 = TUPLE3(hp,make_small(year),make_small(month),make_small(day));
     hp += 4;
     res2 = TUPLE3(hp,make_small(hour),make_small(minute),make_small(second));
     hp += 4;
     BIF_RET(TUPLE2(hp, res1, res2));
 }

/**********************************************************************/

/* return the universal time */

BIF_RETTYPE localtime_0(BIF_ALIST_0)
BIF_ADECL_0
{
     int year, month, day;
     int hour, minute, second;
     uint32 res1, res2;
     uint32* hp;

     /* read the clock */
     get_localtime(&year, &month, &day, &hour, &minute, &second);

     hp = HAlloc(BIF_P, 4+4+3);

     /* and return the tuple */
     res1 = TUPLE3(hp,make_small(year),make_small(month),make_small(day));
     hp += 4;
     res2 = TUPLE3(hp,make_small(hour),make_small(minute),make_small(second));
     hp += 4;
     BIF_RET(TUPLE2(hp, res1, res2));
}
/**********************************************************************/

/* type check and extract components from a tuple on form: {{Y,M,D},{H,M,S}} */
static int time_to_parts(date, year, month, day, hour, minute, second)
uint32 date;
int* year; int* month; int* day;
int* hour;int* minute; int* second;
{
    uint32 *t1;
    uint32 *t2;

    if (is_not_tuple(date))
	return 0;
    t1 = ptr_val(date);
    if (arityval(t1[0]) !=2 || 
	is_not_tuple(t1[1]) || is_not_tuple(t1[2]))
	return 0;
    t2 = ptr_val(t1[1]);
    t1 = ptr_val(t1[2]);
    if (arityval(t2[0]) != 3 || 
	is_not_small(t2[1]) || is_not_small(t2[2]) || is_not_small(t2[3]))
	return 0;
    *year  = signed_val(t2[1]);
    *month = signed_val(t2[2]);
    *day   = signed_val(t2[3]);
    if (arityval(t1[0]) != 3 || 
	is_not_small(t1[1]) || is_not_small(t1[2]) || is_not_small(t1[3]))
	return 0;
    *hour   = signed_val(t1[1]);
    *minute = signed_val(t1[2]);
    *second = signed_val(t1[3]);
    return 1;
}


/* return the universal time */

BIF_RETTYPE localtime_to_universaltime_1(BIF_ALIST_1)
BIF_ADECL_1
{
    int year, month, day;
    int hour, minute, second;
    uint32 res1, res2;
    uint32* hp;

    if (!time_to_parts(BIF_ARG_1, &year, &month, &day, 
		       &hour, &minute, &second))
	BIF_ERROR1(BADARG, am_localtime_to_universaltime, BIF_ARG_1);
    if (!local_to_univ(&year, &month, &day, 
		       &hour, &minute, &second))
	BIF_ERROR1(BADARG, am_localtime_to_universaltime, BIF_ARG_1);
    
    hp = HAlloc(BIF_P, 4+4+3);
    res1 = TUPLE3(hp,make_small(year),make_small(month),
		  make_small(day));
    hp += 4;
    res2 = TUPLE3(hp,make_small(hour),make_small(minute),
		  make_small(second));
    hp += 4;
    BIF_RET(TUPLE2(hp, res1, res2));
 }
	 

/**********************************************************************/

/* return the universal time */

BIF_RETTYPE universaltime_to_localtime_1(BIF_ALIST_1)
BIF_ADECL_1
{
    int year, month, day;
    int hour, minute, second;
    uint32 res1, res2;
    uint32* hp;

    if (!time_to_parts(BIF_ARG_1, &year, &month, &day, 
		       &hour, &minute, &second))
	BIF_ERROR1(BADARG, am_universaltime_to_localtime, BIF_ARG_1);
    if (!univ_to_local(&year, &month, &day, 
		       &hour, &minute, &second))
	BIF_ERROR1(BADARG, am_universaltime_to_localtime, BIF_ARG_1);
    
    hp = HAlloc(BIF_P, 4+4+3);
    res1 = TUPLE3(hp,make_small(year),make_small(month),
		  make_small(day));
    hp += 4;
    res2 = TUPLE3(hp,make_small(hour),make_small(minute),
		  make_small(second));
    hp += 4;
    BIF_RET(TUPLE2(hp, res1, res2));
}

/**********************************************************************/


 /* return a timestamp */
BIF_RETTYPE now_0(BIF_ALIST_0)
BIF_ADECL_0
{
    uint32 megasec, sec, microsec;
    uint32* hp;

    get_now(&megasec, &sec, &microsec);
    hp = HAlloc(BIF_P, 4);
    BIF_RET(TUPLE3(hp,make_small(megasec),make_small(sec),
		   make_small(microsec)));
}


/* this is a general call which return some possibly useful information */

BIF_RETTYPE statistics_1(BIF_ALIST_1)
BIF_ADECL_1
{
    uint32 res;
    uint32* hp;

    if (is_not_atom(BIF_ARG_1))
	BIF_ERROR1(BADARG, am_statistics, BIF_ARG_1);

    if (BIF_ARG_1 == am_context_switches) {
	hp = HAlloc(BIF_P, 3);
	res = TUPLE2(hp, make_small(WRAP27(context_switches)), SMALL_ZERO);
	BIF_RET(res);
    }
    else if (BIF_ARG_1 == am_garbage_collection) {
	hp = HAlloc(BIF_P, 4);
	res = TUPLE3(hp, make_small(WRAP27(garbage_cols)), 
		     make_small(WRAP27(reclaimed)),
		     SMALL_ZERO);
	BIF_RET(res);
    }
    else if (BIF_ARG_1 == am_reductions) {
	uint32 reds;
	uint32 b1, b2;
#if defined(BEAM)
	reds = reductions + (CONTEXT_REDS - BIF_P->fcalls);
#elif defined(JAM)
	reds = reductions + BIF_P->fcalls;
#endif
	b1 = make_small_or_big(reds,BIF_P);
	b2 = make_small_or_big(reds - last_reds,BIF_P);
	hp = HAlloc(BIF_P,3);
	res = TUPLE2(hp, b1, b2); 
	last_reds  = reds;
	BIF_RET(res);
    }
    else if (BIF_ARG_1 == am_runtime) {
	unsigned long u1, u2, dummy;
	uint32 b1, b2;
	elapsed_time_both(&u1,&dummy,&u2,&dummy);
	b1 = make_small_or_big(u1,BIF_P);
	b2 = make_small_or_big(u2,BIF_P);
	hp = HAlloc(BIF_P,3);
	res = TUPLE2(hp, b1, b2);
	BIF_RET(res);
    }
    else if (BIF_ARG_1 ==  am_run_queue) {
	res = sched_q_len();
	BIF_RET(make_small(res));
    }
    else if (BIF_ARG_1 == am_wall_clock) {
	uint32 w1, w2;
	uint32 b1, b2;
	wall_clock_elapsed_time_both(&w1, &w2);
	b1 = make_small_or_big(w1,BIF_P);
	b2 = make_small_or_big(w2,BIF_P);
	hp = HAlloc(BIF_P,3);
	res = TUPLE2(hp, b1, b2);
	BIF_RET(res);
    }
    else if (BIF_ARG_1 == am_io) {
	uint32 r1, r2;
	uint32 in, out;
	in = make_small_or_big(bytes_in,BIF_P);
	out = make_small_or_big(bytes_out,BIF_P); 
	hp = HAlloc(BIF_P, 9);
	r1 = TUPLE2(hp,  am_input, in);
	hp += 3;
	r2 = TUPLE2(hp, am_output, out);
	hp += 3;
	BIF_RET(TUPLE2(hp, r1, r2));
    }
    BIF_ERROR1(BADARG, am_statistics, BIF_ARG_1);
}

/**********************************************************************/

BIF_RETTYPE garbage_collect_1(BIF_ALIST_1)
BIF_ADECL_1
{
    Process *rp;

    if (is_not_pid(BIF_ARG_1))
	BIF_ERROR1(BADARG, am_garbage_collect, BIF_ARG_1);

    if (BIF_P->id == BIF_ARG_1) {
	BIF_P->flags |= F_NEED_GC;
	BIF_RET2(am_true, CONTEXT_REDS); /* Force GC later */
    }
    if ((rp = pid2proc(BIF_ARG_1)) == NULL)
	BIF_RET(am_false);
    do_gc(rp, 0);
    BIF_RET(am_true);
}

BIF_RETTYPE garbage_collect_0(BIF_ALIST_0)
BIF_ADECL_0
{
    BIF_P->flags |= F_NEED_GC;
    BIF_RET2(am_true, CONTEXT_REDS); /* Force GC later */
}

/**********************************************************************/

/* returns a list of the active processes in the system */
/* scans the whole of the process table */

BIF_RETTYPE processes_0(BIF_ALIST_0)
BIF_ADECL_0
{
    int i;
    int need = 0;
    uint32 res = NIL;
    uint32* hp;
     
    /* first work out how many processes there are */
    for (i = 0; i < max_process; i++)
	if (process_tab[i] != (Process*) 0)
	    need += 2;
     
    hp = HAlloc(BIF_P, need);     /* we need two heap words for each pid */

    /* make the list by scanning again (bakward) */
    for (i = max_process-1; i >= 0; i--) {
	if (process_tab[i] != NULL) {
	    res = CONS(hp, process_tab[i]->id, res);
	    hp += 2;
	}
    }
    BIF_RET(res);
}

/**********************************************************************/

static uint32
make_bin_list(p, pb)
Process* p;
ProcBin* pb;
{
    uint32 res = NIL;
    uint32* hp;
    uint32 tuple;

    while (pb) {
	hp = HAlloc(p, 4+2);
	tuple = TUPLE3(hp, make_small(WRAP27((uint32)pb->val)),
		       make_small(pb->size), make_small(pb->val->refc));
	hp += 4;
	res = CONS(hp, tuple, res);
	pb = pb->next;
    }
    return res;
}

BIF_RETTYPE process_info_2(BIF_ALIST_2) 
BIF_ADECL_2
{
    uint32 item;
    uint32 res;
    Process *rp;
    uint32* hp;
    int i;
    uint32 pid = BIF_ARG_1;

    if (is_not_pid(pid) || (get_node(pid) != THIS_NODE) ||
	(get_number(pid) >= max_process)) {
	BIF_ERROR2(BADARG, am_process_info, BIF_ARG_1, BIF_ARG_2);
    }

    i = get_creation(pid);
    if ((i != this_creation) && (i != 0))
	BIF_RET(am_undefined);

    if (is_not_atom(BIF_ARG_2))
	BIF_ERROR2(BADARG, am_process_info, BIF_ARG_1, BIF_ARG_2);

    item = BIF_ARG_2;
    rp = process_tab[get_number(BIF_ARG_1)];

    /* if the process is not active return undefined */
    if (INVALID_PID(rp, BIF_ARG_1))
	BIF_RET(am_undefined);
    res = NIL;

    if (item == am_registered_name) {
	if (rp->reg != NULL) {
	    hp = HAlloc(BIF_P, 3);
	    res = make_atom(rp->reg->name);
	}
	else {
	    BIF_RET(NIL);
	}
    }
    else if (item == am_current_function) {
#if defined(BEAM)
	if (rp->current == NULL) {
	    rp->current = find_function_from_pc(rp->i);
	}
	ASSERT(rp->current != NULL);
	if (rp->current == NULL) {
	    hp = HAlloc(BIF_P, 3);
	    res = am_undefined;
	}
	else {
	    hp = HAlloc(BIF_P, 3+4);
	    res = TUPLE3(hp, rp->current[0],
			 rp->current[1], make_small(rp->current[2]));
	    hp += 4;
	}
#elif defined(JAM)
	int module;
	int function;
	byte *cc;

	if (rp->cc == NULL) {
	    hp = HAlloc(BIF_P, 3);
	    res = am_undefined;
	}
	else {
	    hp = HAlloc(BIF_P, 3+4);
	    cc = rp->cc + 1;

	    module = make_atom(make_16(cc[1], cc[2]));
	    function = make_atom(make_16(cc[3], cc[4]));
	    res = TUPLE3(hp, module, function,
			 make_small(cc[0]));
	    hp += 4;
	}
#endif
    }
    else if (item == am_initial_call) {
	hp = HAlloc(BIF_P, 3+4);
	res = TUPLE3(hp,
		     rp->initial[INITIAL_MOD],
		     rp->initial[INITIAL_FUN],
		     make_small(rp->initial[INITIAL_ARI]));
	hp += 4;
    }
    else if (item == am_status ) {
	hp = HAlloc(BIF_P, 3);
	switch (rp->status) {
	case P_RUNABLE:
	    res = am_runnable;
	    break;
	case P_WAITING:
	    res = am_waiting;
	    break;
	case P_RUNNING:
	    res = am_running;
	    break;
	case P_SUSPENDED:
	    res = am_suspended;
	    break;
	default:
	    res = am_undefined;
	}
    }
    else if (item == am_messages) {
	ErlMessage* mp;
	uint32* cons;
	int n = rp->msg.len;
	uint32 size;

	if (n == 0) {
	    hp = HAlloc(BIF_P, 3);
	    res = NIL;
	}
	else {
	    size = 0;
	    if (rp != BIF_P) {
		mp = rp->msg.first;
		while(mp != NULL) {
		    size += size_object(mp->mesg);
		    mp = mp->next;
		}
	    }
	    hp = HAlloc(BIF_P, 3 + size + 2*n);
	    hp += 2*n;  /* skip the list !!! */
	    cons = hp - 2;
	    res = make_list(cons);  /* first cons cell */
	    /* Build with back-pointers (as cons whould have done) */
	    mp = rp->msg.first;
	    while(mp != NULL) {
		if (rp == BIF_P)
		    cons[0] = mp->mesg;        /* write head */
		else {
		    size = size_object(mp->mesg); /* XXX I know */
		    cons[0] = copy_struct(mp->mesg, size,
					  &hp, &BIF_P->mso);
		}
		cons -= 2;                 /* next cell */
		cons[3] = make_list(cons); /* write tail */
		mp = mp->next;
	    }
	    cons[3] = NIL; 
	}
    }
    else if (item == am_message_queue_len) {
	hp = HAlloc(BIF_P, 3);
	res = make_small(rp->msg.len);
    }
    else if (item == am_links) {
	int n = 0;
	ErlLink* lnk;

	lnk = rp->links;
	while(lnk != NULL) {
	    if (lnk->type == LNK_LINK)
		n++;
	    lnk = lnk->next;
	}
	hp = HAlloc(BIF_P, 3 + 2*n);
	lnk = rp->links;
	while(lnk != NULL) {
	    if (lnk->type == LNK_LINK) {
		res = CONS(hp, lnk->item, res);
		hp += 2;
	    }
	    lnk = lnk->next;
	}
    }
    else if (item == am_dictionary) {
	uint32 size = size_object(rp->dictionary);
	hp = HAlloc(BIF_P, size+3);
	res = copy_struct(rp->dictionary, size, &hp, &BIF_P->mso);
    }
    else if (item == am_trap_exit) {
	hp = HAlloc(BIF_P, 3);
	if (rp->flags  & F_TRAPEXIT)
	    res = am_true;
	else
	    res = am_false;
    }
    else if (item == am_error_handler ) {
	hp = HAlloc(BIF_P, 3);
	res = rp->error_handler;
    }
    else if (item == am_heap_size) {
	hp = HAlloc(BIF_P, 3);
	res = make_small(rp->heap_sz);
    }
    else if (item == am_stack_size) {
	hp = HAlloc(BIF_P, 3);
#if defined(BEAM)
	res = make_small(0);
#elif defined(JAM)
	res = make_small(rp->stack_sz);
#endif
    }
    else if (item == am_memory) {
	uint32 size = 0;
	ErlLink* lnk;

	lnk = rp->links;
	while(lnk != NULL) {
	    size += sizeof(ErlLink);
	    lnk = lnk->next;
	}
	size += (rp->heap_sz + rp->mbuf_sz) * sizeof(uint32) + sizeof(Process);
#if defined(JAM)
	size += rp->stack_sz*sizeof(uint32);
#endif
	hp = HAlloc(BIF_P, 3);
	res = make_small(WRAP27(size));
    }
    else if(item == am_garbage_collection){
	uint32 gc;
	if (!IS_GEN_GC(rp) && rp->gc_switch > 0)
	    gc = am_fullsweep;
	else
	    gc = am_generational;
	hp = HAlloc(BIF_P, 6);
	if (rp->gc_switch == 0)
	    res = TUPLE2(hp, gc, make_small(0));
	else if(rp->gc_switch >= MAX_SMALL)
	    res = TUPLE2(hp, gc, am_infinity);
	else
	    res = TUPLE2(hp, gc, make_small(rp->gc_switch));
	hp += 3;
    }
    else if (item == am_group_leader) {
	hp = HAlloc(BIF_P, 3);
	res = rp->group_leader;
    }
    else if (item == am_reductions) {
	uint32 reds;

	hp = HAlloc(BIF_P, 3);
	if (BIF_P != rp)
	    reds = rp->reds;
	else {
#if defined(BEAM)
	    reds = rp->reds + (CONTEXT_REDS - rp->fcalls);
#elif defined(JAM)
	    reds = rp->reds + rp->fcalls;
#endif
	}
	res = make_small(WRAP27(reds));
    }
    else if (item == am_priority) {
	hp = HAlloc(BIF_P, 3);
	switch(rp->prio) {
	case PRIORITY_MAX:
	    res = am_max; break;
	case PRIORITY_HIGH:
	    res = am_high; break;
	case PRIORITY_NORMAL:
	    res = am_normal; break;
	case PRIORITY_LOW:
	    res = am_low; break;
	}
    }
    else if (item == am_trace) {
	hp = HAlloc(BIF_P, 3);
	res = make_small(rp->flags & TRACE_FLAGS);
    }
    else if (item == am_binary) {
	uint32 curr = make_bin_list(BIF_P, rp->mso);
	uint32 old = make_bin_list(BIF_P, rp->old_mso);
	hp = HAlloc(BIF_P, 7);
	res = TUPLE3(hp, make_small(WRAP27(rp->mso_weight)), curr, old);
	hp += 4;
    }
#ifdef SEQ_TRACE
    else if (item == am_sequential_trace_token) {
	uint32 size = size_object(rp->seq_trace_token);
	hp = HAlloc(BIF_P, size+3);
	res = copy_struct(rp->seq_trace_token, size, &hp, &BIF_P->mso);
    }
#endif
    else
	BIF_ERROR2(BADARG, am_process_info, BIF_ARG_1, BIF_ARG_2);

    BIF_RET(TUPLE2(hp, item, res));
}


/**********************************************************************/
/* Return a list of active ports */

BIF_RETTYPE ports_0(BIF_ALIST_0)
BIF_ADECL_0
{
    int i;
    int need = 0;
    uint32 res = NIL;
    uint32* hp;
     
    /* first work out how many processes there are */
    for (i = 0; i < MAX_PORTS; i++) {
	if (port[i].status != FREE)
	    need += 2;
    }
    hp = HAlloc(BIF_P, need);

    for (i = MAX_PORTS-1; i >= 0; i--) {
	if (port[i].status != FREE) {
	    res = CONS(hp, make_port2(THIS_NODE,i), res);
	    hp += 2;
	}
    }
    BIF_RET(res);
}

/**********************************************************************/ 
/* Return information on ports */
/* Info:
**    id          Port index
**    connected   (Pid)
**    links       List of pids
**    name        String
**    input       Number of bytes input from port program
**    output      Number of bytes output to the port program
*/

BIF_RETTYPE port_info_2(BIF_ALIST_2)
BIF_ADECL_2
{
    uint32 portid = BIF_ARG_1;
    uint32 item = BIF_ARG_2;
    uint32 res;
    uint32* hp;
    int i;
    int count;
    int portix;

    if (is_not_port(portid) || (get_node_reference(portid) != THIS_NODE) ||
	((portix = get_number_reference(portid)) >= MAX_PORTS)) {
	BIF_ERROR2(BADARG, am_port_info, BIF_ARG_1, BIF_ARG_2);
    }
    i = get_creation(portid);
    if (((i != this_creation) && (i != 0)) || (port[portix].status == FREE)) {
	BIF_RET(am_undefined);
    }

    if (item == am_id) {
	hp = HAlloc(BIF_P, 3);
	res = make_small(portix);
    }
    else if (item == am_links) {
	int n = 0;
	ErlLink* lnk;

	lnk = port[portix].links;
	while(lnk != NULL) {
	    if (lnk->type == LNK_LINK)
		n++;
	    lnk = lnk->next;
	}
	hp = HAlloc(BIF_P, 3 + 2*n);

	lnk = port[portix].links;
	res = NIL;
	while(lnk != NULL) {
	    if (lnk->type == LNK_LINK) {
		res = CONS(hp, lnk->item, res);
		hp += 2;
	    }
	    lnk = lnk->next;
	}
    }
    else if (item == am_name) {
	count = sys_strlen(port[portix].name);

	hp = HAlloc(BIF_P, 3 + 2*count);
	res = buf_to_intlist(&hp,(byte*)port[portix].name,count,NIL);
    }
    else if (item == am_connected) {
	hp = HAlloc(BIF_P, 3);
	res = port[portix].connected;
    }
    else if (item == am_input) {
	hp = HAlloc(BIF_P, 3);
	res = make_small(WRAP27(port[portix].bytes_in));
    }
    else if (item == am_output) {
	hp = HAlloc(BIF_P, 3);
	res = make_small(WRAP27(port[portix].bytes_out));
    }
    else
	BIF_ERROR2(BADARG, am_port_info, BIF_ARG_1, BIF_ARG_2);
    BIF_RET(TUPLE2(hp, item, res));
}


/**********************************************************************/

/* this may be hard to understand. Don't worry your'e not stupid. */

BIF_RETTYPE throw_1(BIF_ALIST_1)
BIF_ADECL_1
{
    BIF_P->fvalue = BIF_ARG_1;
    BIF_ERROR(THROWN);
}

/**********************************************************************/


/* 
  this is a non standard undoccumented BIF which is used for debuuging
  "proper" programs whould use the io primitives provided by the io server
  */
BIF_RETTYPE display_1(BIF_ALIST_1)
BIF_ADECL_1
{
     display(BIF_ARG_1, COUT);
     erl_putc('\r', COUT);
     erl_putc('\n', COUT);
     BIF_RET(am_true);
}
/**********************************************************************/

/* load a module which is a binary (files now removed /Mike) */

BIF_RETTYPE load_module_2(BIF_ALIST_2)
BIF_ADECL_2
{
    uint32   reason;
    uint32*  hp;
    int      i;
    int      sz;
    ProcBin* bptr;
    byte*    code;
    
    if (is_not_atom(BIF_ARG_1) || is_not_binary(BIF_ARG_2)) {
	BIF_ERROR2(BADARG, am_load_module, BIF_ARG_1, BIF_ARG_2);
    }
    
    hp = HAlloc(BIF_P, 3);
    bptr = (ProcBin*) ptr_val(BIF_ARG_2);
    sz = bptr->size;
    code = bptr->bytes;
    if ((i = do_load(BIF_P->group_leader, BIF_ARG_1, code, sz)) < 0) { 
	switch (i) {
	case -1: reason = am_badfile; break; 
	case -2: reason = am_nofile; break;
	case -3: reason = am_not_purged; break;
	default: reason = am_badfile; break;
	}
	BIF_RET(TUPLE2(hp, am_error, reason));
    }
    BIF_RET(TUPLE2(hp, am_module, BIF_ARG_1));
}


/**********************************************************************/

/* get/0 gets the whole of a process dictionary */
/* we need to copy since put/2 is destructive as is erase/1 */

BIF_RETTYPE get_0(BIF_ALIST_0)
BIF_ADECL_0
{
    uint32  copy;
    uint32* hp;
    uint32 size;

    size = size_object(BIF_P->dictionary);
    hp = HAlloc(BIF_P, size);
    copy = copy_struct(BIF_P->dictionary, size, &hp, &BIF_P->mso);
    BIF_RET(copy);
}

/**********************************************************************/

/* erase the whole of the dictionary returning the old one */

BIF_RETTYPE erase_0(BIF_ALIST_0)
BIF_ADECL_0
{
     uint32 res;
     res = BIF_P->dictionary;
     BIF_P->dictionary = NIL;
     BIF_RET(res);
}

/**********************************************************************/

/* sort of reverse get/1, but some values may have several keys so this
   returns a list
   */

BIF_RETTYPE get_keys_1(BIF_ALIST_1)
BIF_ADECL_1
{
    uint32 *tp;
    uint32 res;
    uint32 pos = BIF_P->dictionary;
    uint32* hp;
    int n = 0;

    /* run through the list to find the amount of heap needed */
    while (pos != NIL) {
	uint32* cons = ptr_val(pos);
	tp = ptr_val(CAR(cons));
	if (eq(BIF_ARG_1, tp[2])) 
	    n++;
	pos = CDR(cons);
    }
     
    if (n == 0)  /* no matching key found */
	BIF_RET(NIL);
    /* run through again and build the return list */

    hp = HAlloc(BIF_P, 2*n);

     /* run through the list again to find the matching keys */
    pos = BIF_P->dictionary;
    res = make_list(hp);
    while (pos != NIL) { 
	uint32* cons = ptr_val(pos);
	tp = ptr_val(CAR(cons));
	if (eq(BIF_ARG_1, tp[2])) {
	    *hp++ = tp[1];
	    *hp = make_list(hp + 1);
	    hp++;
	}
	pos = CDR(cons);
    }
    *(hp - 1) = NIL;
    BIF_RET(res);
}

/**********************************************************************/

/* process dictionary get/1 
   The dictionary is of the form 
   [{Key1,Value1}, {Key2,Value2}, {Key3,Value3} | The_Rest]
   */

BIF_RETTYPE get_1(BIF_ALIST_1)
BIF_ADECL_1
{
    uint32 pos = BIF_P->dictionary;
    uint32 key = BIF_ARG_1;

    while (pos != NIL) {
	uint32* cons = ptr_val(pos);
	uint32* tp = ptr_val(CAR(cons));

	if (eq(key, tp[1]))
	    BIF_RET(tp[2]);
	pos = CDR(cons);
    }
    BIF_RET(am_undefined); 
}

/**********************************************************************/



static uint32 do_erase_1(p, key)
Process *p;
uint32 key;
{
    uint32 pos = p->dictionary;
    uint32* last = &p->dictionary;

    while (pos != NIL) {
	uint32* cons = ptr_val(pos);
	uint32 *tp = ptr_val(CAR(cons));

	if (eq(key, tp[1])) {
	    *last = CDR(cons);
	    return tp[2];
	}
	last = cons + 1;
	pos = CDR(cons);
    }
    return am_undefined;
}


/* process dictionary put/2
   The dictionary is of the form 
   [{Key1,Value1}, {Key2,Value2}, {Key3,Value3} | The_Rest]
   */


BIF_RETTYPE put_2(BIF_ALIST_2)
BIF_ADECL_2
{
    uint32 new;
    uint32 old;
    uint32 tup;
    uint32* hp;

    hp = HAlloc(BIF_P, 5);
    old = do_erase_1(BIF_P, BIF_ARG_1);
    tup = TUPLE2(hp, BIF_ARG_1, BIF_ARG_2);
    hp += 3;
    new = CONS(hp, tup, BIF_P->dictionary);
    BIF_P->dictionary = new;
    BIF_RET(old);
}

/**********************************************************************/

/* process dictionary erase/1
   The dictionary is of the form 
   [{Key1,Value1}, {Key2,Value2}, {Key3,Value3} | The_Rest]
   
   Return the old value  */


BIF_RETTYPE erase_1(BIF_ALIST_1)
BIF_ADECL_1
{
    BIF_RET(do_erase_1(BIF_P, BIF_ARG_1));
}

/**********************************************************************/


/**********************************************************************/

/* stop the system */
/* ARGSUSED */
BIF_RETTYPE halt_0(BIF_ALIST_0)
BIF_ADECL_0
{
    VERBOSE(erl_printf(COUT, "System halted by BIF halt/0\n"););
    erl_exit(0, "");
    return 0;  /* Pedantic (lint does not know about erl_exit) */
}

/**********************************************************************/

#define MSG_SIZE	200

/* stop the system with exit code */
/* ARGSUSED */
BIF_RETTYPE halt_1(BIF_ALIST_1)
BIF_ADECL_1
{
    int code;
    static char msg[MSG_SIZE];
    int i;

    if (is_small(BIF_ARG_1) && (code = signed_val(BIF_ARG_1)) >= 0) {
       VERBOSE(erl_printf(COUT, "System halted by BIF halt(%d)\n", code););
       erl_exit(-code, "");
    }
    else if (is_string(BIF_ARG_1)) {
       if ((i = intlist_to_buf(BIF_ARG_1, msg, MSG_SIZE-1)) < 0)
	  BIF_ERROR1(BADARG, am_halt, BIF_ARG_1);
       msg[i] = '\0';
       VERBOSE(erl_printf(COUT, "System halted by BIF halt(%s)\n", msg););
       erl_exit(127, "%s\n", msg);
    }
    else
       BIF_ERROR1(BADARG, am_halt, BIF_ARG_1);

    return 0;  /* Pedantic (lint does not know about erl_exit) */
}

/**********************************************************************/

/* remove any old versions of code for a module */

BIF_RETTYPE purge_module_1(BIF_ALIST_1)
BIF_ADECL_1
{
    if (is_not_atom(BIF_ARG_1)) {
	BIF_ERROR1(BADARG, am_purge_module, BIF_ARG_1);
    }
    if (purge_module(unsigned_val(BIF_ARG_1)) < 0) {
	BIF_ERROR1(BADARG, am_purge_module, BIF_ARG_1);
    }
    BIF_RET(am_true);
}


/**********************************************************************/

/* Check the stack of a process (arg1) to see if it is executing any old code
   in module (arg2). All the work is done in load.c
   */ 

BIF_RETTYPE check_process_code_2(BIF_ALIST_2)
BIF_ADECL_2
{
    int i;

    if (is_not_atom(BIF_ARG_2))
	BIF_ERROR2(BADARG, am_check_process_code, BIF_ARG_1, BIF_ARG_2);    

    if ((i = check_process_code(BIF_P,BIF_ARG_1,unsigned_val(BIF_ARG_2)))<0) {
	BIF_ERROR2(BADARG, am_check_process_code, BIF_ARG_1, BIF_ARG_2);
    }
    if (i != 0) 
	BIF_RET(am_true);
    BIF_RET(am_false);
}


/**********************************************************************/

BIF_RETTYPE delete_module_1(BIF_ALIST_1)
BIF_ADECL_1
{
     int i;

     if (is_not_atom(BIF_ARG_1)) {
	 BIF_ERROR1(BADARG, am_delete_module, BIF_ARG_1);
     }
     if ((i = delete_module(BIF_P->group_leader, 
			    unsigned_val(BIF_ARG_1))) == 0)
	 BIF_RET(am_true);
     if (i == 1) 
	 BIF_RET(am_undefined);
     BIF_ERROR1(BADARG, am_delete_module, BIF_ARG_1);
}

/**********************************************************************/    

BIF_RETTYPE module_loaded_1(BIF_ALIST_1)
BIF_ADECL_1
{
    int i;
    if (is_not_atom(BIF_ARG_1)) {
	BIF_ERROR1(BADARG, am_module_loaded, BIF_ARG_1);
    }
    if ((i = module_get(unsigned_val(BIF_ARG_1))) == -1)
	BIF_RET(am_false);
    if (module_code(i)->code == NULL) {
	BIF_RET(am_false);
    }
    BIF_RET(am_true);
}

/**********************************************************************/    

BIF_RETTYPE function_exported_3(BIF_ALIST_3)
BIF_ADECL_3
{
    if (is_not_atom(BIF_ARG_1) ||
	is_not_atom(BIF_ARG_2) || 
	is_not_small(BIF_ARG_3)) {
	BIF_ERROR3(BADARG,am_function_exported,BIF_ARG_1,BIF_ARG_2,BIF_ARG_3);
    }
    if (find_function(BIF_ARG_1,
		      BIF_ARG_2, signed_val(BIF_ARG_3)) < 0) {
	BIF_RET(am_false);
    }
    BIF_RET(am_true);
}

/**********************************************************************/    

/* convert a pid to an erlang list (for the linked cons cells) of the form
   %node.serial.number% to a PID
 */

BIF_RETTYPE pid_to_list_1(BIF_ALIST_1)
BIF_ADECL_1
{
    int n;
    char* cp;
    char* tmpp;
    char ibuf[12];
    uint32 pid = BIF_ARG_1;
    uint32* hp;

    /* check arg */
    if (is_not_pid(pid)) {
	BIF_ERROR1(BADARG, am_pid_to_list, BIF_ARG_1);
    }

    tmpp = (char*) tmp_buf;
    *tmpp++ = '<';
    
    /* node part */
    cp = int_to_buf(get_node(pid), ibuf);
    n = sys_strlen(cp);
    sys_strncpy(tmpp, cp, n);
    tmpp += n;
    *tmpp++ = '.';

    /* number part */
    cp = int_to_buf(get_number(pid), ibuf);
    n = sys_strlen(cp);
    sys_strncpy(tmpp, cp, n);
    tmpp += n;
    *tmpp++ = '.';

    /* serial part */
    cp = int_to_buf(get_serial(pid), ibuf);
    n = sys_strlen(cp);
    sys_strncpy(tmpp, cp, n);
    tmpp += n;
    *tmpp++ = '>';
    *tmpp = '\0';

    n = tmpp - (char*)tmp_buf;  /* total length */
    hp = HAlloc(BIF_P, n*2);    /* we need length * 2 heap words */
    BIF_RET(buf_to_intlist(&hp, tmp_buf, n, NIL));
}

/**********************************************************************/

/* convert a list of ascii characeters of the form
   <node.serial.number> to a PID
*/

BIF_RETTYPE list_to_pid_1(BIF_ALIST_1)
BIF_ADECL_1
{
    uint32 a = 0, b = 0, c = 0;
    char* cp;
    int i;

    /* walk down the list and create a C string */
    if ((i = intlist_to_buf(BIF_ARG_1,tmp_buf,TMP_BUF_SIZE-1)) < 0) {
	BIF_ERROR1(BADARG, am_list_to_pid, BIF_ARG_1);
    }
    tmp_buf[i] = '\0';		/* null terminal */

    cp = (char*) tmp_buf;
    if (*cp++ != '<') goto bad;
    
    if (*cp < '0' || *cp > '9') goto bad;
    while(*cp >= '0' && *cp <= '9') { a = 10*a + (*cp - '0'); cp++; }

    if (*cp++ != '.') goto bad;

    if (*cp < '0' || *cp > '9') goto bad;
    while(*cp >= '0' && *cp <= '9') { b = 10*b + (*cp - '0'); cp++; }

    if (*cp++ != '.') goto bad;

    if (*cp < '0' || *cp > '9') goto bad;
    while(*cp >= '0' && *cp <= '9') { c = 10*c + (*cp - '0'); cp++; }

    if (*cp++ != '>') goto bad;
    if (*cp != '\0') goto bad;

    /* <a.b.c> a = node, b = process number, c = serial */

    /* bounds check */
    /* NOT max_process here !!! */

    if ((b >= MAX_PROCESS) || (c >= MAX_SERIAL) || (a >= (1 << P_NODE)) ||
	( (a != THIS_NODE) && (dist_addrs[a].cid == NIL))) {
	BIF_ERROR1(BADARG, am_list_to_pid, BIF_ARG_1);
    }
    BIF_RET(make_pid(c, a, b));

 bad:
    BIF_ERROR1(BADARG, am_list_to_pid, BIF_ARG_1);
}

/**********************************************************************/

/* convert an object to an array of bytes */

BIF_RETTYPE term_to_binary_1(BIF_ALIST_1)
BIF_ADECL_1
{
    int size;
    uint32 bin;
    ProcBin* pb;
    byte *c;

    size = encode_size_struct(BIF_ARG_1);
    bin = new_binary(BIF_P, (byte *)NULL, size);
    maybe_gc_binary(BIF_P, &bin); /* No GC Just MARK !! */
    pb = (ProcBin *) ptr_val(bin);
    c = pb->bytes;
    to_external(-1, BIF_ARG_1, &c);
    if (c > (pb->bytes + size))
	erl_exit(1, "Internal error in term_to_binary %d\n",(c-pb->bytes));
    /* adjust since encode_size_struct will return more than needed */
    pb->size = c - pb->bytes;
    BIF_RET(bin);
}

/**********************************************************************/

/* convert an array of bytes to an object */

BIF_RETTYPE binary_to_term_1(BIF_ALIST_1)
BIF_ADECL_1
{
    ProcBin* bp;
    byte* c;
    int i;
    uint32 res;
    uint32* hp;
#ifdef DEBUG
    uint32* end;
#endif

    if (is_not_binary(BIF_ARG_1))
	BIF_ERROR1(BADARG, am_binary_to_term, BIF_ARG_1);

    bp = (ProcBin*) ptr_val(BIF_ARG_1);

    if ((i = dec_size(bp->bytes, bp->size)) < 0)
	BIF_ERROR1(BADARG, am_binary_to_term, BIF_ARG_1);

    /* XXX i may be 0 ??? */
    hp = HAlloc(BIF_P, i);
#ifdef DEBUG
    end = hp+i;
#endif
    c = bp->bytes;

    if ((res = from_external(-1, &hp, &c, &BIF_P->mso)) == 0)
	BIF_ERROR1(BADARG, am_binary_to_term, BIF_ARG_1);
    ASSERT(hp <= end);
#ifdef DEBUG
    while (hp < end) {
	*hp++ = NIL;
    }
#endif
    maybe_gc_binary(BIF_P, &res);
    BIF_RET(res);
}

/**********************************************************************/

/* convert an array of bytes to an object */

BIF_RETTYPE old_binary_to_term_1(BIF_ALIST_1)
BIF_ADECL_1
{
    ProcBin* bp;
    byte* c;
    int i;
    uint32 res;
    uint32* hp;
#ifdef DEBUG
    uint32* end;
#endif

    if (is_not_binary(BIF_ARG_1))
	BIF_ERROR1(BADARG, am_old_binary_to_term, BIF_ARG_1);

    bp = (ProcBin*) ptr_val(BIF_ARG_1);

    if ((i = decode_size(bp->bytes, bp->size)) < 0)
	BIF_ERROR1(BADARG, am_old_binary_to_term, BIF_ARG_1);

    /* XXX i may be 0 ??? */
    hp = HAlloc(BIF_P, i);
#ifdef DEBUG
    end = hp+i;
#endif

    c = bp->bytes;
    if ((res = from_external(-1, &hp, &c, &BIF_P->mso)) == 0)
	BIF_ERROR1(BADARG, am_old_binary_to_term, BIF_ARG_1);
    ASSERT(hp <= end);
#ifdef DEBUG
    while (hp < end) {
	*hp++ = NIL;
    }
#endif
    maybe_gc_binary(BIF_P, &res);
    BIF_RET(res);
}

/**********************************************************************/

BIF_RETTYPE group_leader_0(BIF_ALIST_0)
BIF_ADECL_0
{
    BIF_RET(BIF_P->group_leader);
}

/**********************************************************************/
/* arg1 == leader , arg2 == new member */

BIF_RETTYPE group_leader_2(BIF_ALIST_2)
BIF_ADECL_2
{
    Process* new_member;
    int code;
    int slot;

    if (is_not_pid(BIF_ARG_1) || is_not_pid(BIF_ARG_2)) {
	BIF_ERROR2(BADARG, am_group_leader, BIF_ARG_1, BIF_ARG_2);
    }

    if ((slot = get_node(BIF_ARG_2)) != THIS_NODE) {
	if (dist_addrs[slot].cid == NIL)
	    BIF_TRAP2(dgroup_leader_trap,BIF_ARG_1,BIF_ARG_2);
	if ((code = dist_group_leader(slot, BIF_ARG_1, BIF_ARG_2)) == 1) {
	    ASSERT(is_port(dist_addrs[slot].cid)); 
	    erl_suspend(BIF_P, dist_addrs[slot].cid);
	    BIF_ERROR(RESCHEDULE);
	}
	else if (code < 0) {  /* XXX is this the correct behaviour ??? */
	    BIF_ERROR2(NOTALIVE, am_group_leader, BIF_ARG_1, BIF_ARG_2);
	}
	BIF_RET(am_true);
    }
    if ((new_member = pid2proc(BIF_ARG_2)) == NULL) {
	BIF_ERROR2(BADARG, am_group_leader, BIF_ARG_1, BIF_ARG_2);
    }
    new_member->group_leader = BIF_ARG_1;
    BIF_RET(am_true);
}
    

/*****************************************
   This is the tracing BIF,
    erlang:trace(Pid, true|false, Flaglist) turns on or off
    trace messages from Pid, All the messages are sent to the 
    process executing the BIF.
    FLAGlist can contain any number of the following atoms.
    'send' traces messages the process Pid sends.
    'receive' traces messages the process Pid receives
    'set_on_spawn' makes any process created by Pid inherit the flags of Pid.
    'set_on_first_spawn' as above but only for the first process created by Proc
    'procs' to trace process releted events such as spawn, link, death etc
    'bifs' to trace calls to bifs
    'call' to trace function calls

   The tracing process receives messages on the form
   {trace, 'receive', Proc, Message} when the traced Proc receivs anything
   {trace, send, Proc, To, Msg} when Proc send a massage
   {trace, call, {M,F,A}} when Proc makes a function/bif call.
   {trace, Op, Data} when a process related event occurs in Proc.
                     Op can be link/spawn/exit

   If the tracing process dies, the flags will be silently removed.

   Threre can be only 1 process tracing a particular process. So an 
   attempt to trace an allready traced process will fail.

***********************************************/


BIF_RETTYPE trace_3(BIF_ALIST_3)    
BIF_ADECL_3
{
    Process *tracee;
    uint32 mask;
    uint32 *cons;
    uint32 list;
    uint32 item;

    if (is_not_pid(BIF_ARG_1))
	BIF_ERROR3(BADARG,am_trace,BIF_ARG_1,BIF_ARG_2,BIF_ARG_3);

    if (is_nil(BIF_ARG_3))
	BIF_RET(am_true);

    if (is_not_list(BIF_ARG_3))
	BIF_ERROR3(BADARG,am_trace,BIF_ARG_1,BIF_ARG_2,BIF_ARG_3);

    if (IS_TRACED(BIF_P)) {
	erl_printf(CBUF,
		   "** Can't call trace BIF from a process beeing traced\n");
	send_error_to_logger(BIF_P->group_leader);
	BIF_ERROR3(BADARG,am_trace,BIF_ARG_1,BIF_ARG_2,BIF_ARG_3);
    }

    /* check the tracee PID's and get its process struct */

    if (get_node(BIF_ARG_1) != THIS_NODE)
	BIF_ERROR3(BADARG,am_trace,BIF_ARG_1,BIF_ARG_2,BIF_ARG_3);

    if (get_number(BIF_ARG_1) >= max_process)
	BIF_ERROR3(BADARG,am_trace,BIF_ARG_1,BIF_ARG_2,BIF_ARG_3);

    tracee = process_tab[get_number(BIF_ARG_1)];
    if (INVALID_PID(tracee, BIF_ARG_1) ||
	tracee->id == BIF_P->id ||
	tracee->flags & F_IS_TRACING)
	BIF_RET(am_false);

    /* There can be only 1 trace proc  */
    if ((tracee->flags & (TRACE_FLAGS)) &&
	(tracee->tracer_proc != BIF_P->id)) {
	Process *tracer = process_tab[get_number(tracee->tracer_proc)];

	if (INVALID_PID(tracer, tracee->tracer_proc)) {
	    tracee->flags &= ~TRACE_FLAGS;
	    tracee->tracer_proc = NIL;  /* reset it */
	}
	else {
	    erl_printf(CBUF,
		       "** can only have one tracer process per process\n");
	    send_error_to_logger(BIF_P->group_leader);
	    BIF_ERROR3(BADARG,am_trace,BIF_ARG_1,BIF_ARG_2,BIF_ARG_3);
	}
    }

    list = BIF_ARG_3;
    mask = 0;
    while (is_list(list)) {
	cons = ptr_val(list);
	item = CAR(cons);

	if (item == am_suspend) {
	    if (BIF_ARG_2 == am_true) {
		erl_suspend(tracee, NIL);
		BIF_RET(am_true);
	    }
	    else if (BIF_ARG_2 == am_false) {
		erl_resume(tracee);
		BIF_RET(am_true);
	    }
	    else
		BIF_ERROR3(BADARG,am_trace,BIF_ARG_1,BIF_ARG_2,BIF_ARG_3);
	}
	else if (item == am_send)
	    mask |= F_TRACE_SEND;
	else if (item == am_receive)
	    mask |= F_TRACE_RECEIVE;
	else if (item == am_set_on_spawn)
	    mask |= F_TRACE_SOS;
	else if (item == am_call)
	    mask |= F_TRACE_CALLS;
	else if (item == am_bifs)
	    mask |= F_TRACE_BIFS;
	else if (item == am_procs)
	    mask |= F_TRACE_PROCS;
	else if (item == am_set_on_first_spawn)
	    mask |= F_TRACE_SOS1;
	else if (item == am_set_on_link)
	    mask |= F_TRACE_SOL;
	else if (item == am_set_on_first_link)
	    mask |= F_TRACE_SOL1;
	else if (item == am_timestamp)
	    mask |= F_TIMESTAMP;
	else if (item == am_running)
	    mask |= F_TRACE_SCHED;
	else if (item == am_garbage_collection)
	    mask |= F_TRACE_GC;
	else
	    BIF_ERROR3(BADARG,am_trace,BIF_ARG_1,BIF_ARG_2,BIF_ARG_3);
	list = CDR(cons);
    }
    if (is_not_nil(list))
	BIF_ERROR3(BADARG,am_trace,BIF_ARG_1,BIF_ARG_2,BIF_ARG_3);

    tracee->tracer_proc = BIF_P->id;  /* Allways set it */

    /* This means that a process that has once called the trace/3 BIF */
    /* can itself never be traced :-(                                 */

    BIF_P->flags |= F_IS_TRACING;

    if (BIF_ARG_2 == am_false) 
	tracee->flags &= ~mask;
    else if (BIF_ARG_2 == am_true)
	tracee->flags |= mask;
    else {
	BIF_ERROR3(BADARG,am_trace,BIF_ARG_1,BIF_ARG_2,BIF_ARG_3);
    }
    if (!(tracee->flags & TRACE_FLAGS)) {
	BIF_P->flags &= ~F_IS_TRACING;  
	tracee->tracer_proc = NIL;  /* All flags cleared */
    }
    BIF_RET(am_true);
}



/*
 * Sequential tracing
 *
 * The sequential trace token is internally implemented as
 * a tuple
 *         {Flags, Label, Serial, Sender, LastSerial}
 * 
 * where 
 *       - Flags is an integer (using masks 1, 2, and 4, for send,
 *         receive and print, respectively), 
 *       - Label is any term, Serial (for now XXX) is an integer (it should
 *         be a list reflecting split traces), and 
 *       - Sender is the Pid of the sender (i.e. the current process, 
 *         except immediately after a message reception, in case it is
 *         the pid of the process that sent the message).
 *
 */

BIF_RETTYPE seq_trace_2(BIF_ALIST_2)    
BIF_ADECL_2
{
#ifndef SEQ_TRACE
    BIF_ERROR2(BADARG, am_seq_trace, BIF_ARG_1, BIF_ARG_2);
#else
    uint32 flags, old_value;
    uint32* hp;
    int current_flag;

    if (!is_atom(BIF_ARG_1)) {
	BIF_ERROR2(BADARG, am_seq_trace, BIF_ARG_1, BIF_ARG_2);
    }


    if (BIF_ARG_1 == am_send) {
	current_flag = SEQ_TRACE_SEND;
    } else if (BIF_ARG_1 == am_receive) {
	current_flag = SEQ_TRACE_RECEIVE; 
    } else if (BIF_ARG_1 == am_print) {
	current_flag = SEQ_TRACE_PRINT; 
    } else if (BIF_ARG_1 == am_timestamp) {
	current_flag = SEQ_TRACE_TIMESTAMP; 
    }
    else
	current_flag = 0;

    if (current_flag && ( (BIF_ARG_2 == am_true) || (BIF_ARG_2 == am_false)) ) {
	/* Flags */
        new_seq_trace_token(BIF_P);
        flags = unsigned_val(SEQ_TRACE_TOKEN_FLAGS(BIF_P));
	old_value = flags & current_flag ? am_true : am_false;
	if (BIF_ARG_2 == am_true)
	    SEQ_TRACE_TOKEN_FLAGS(BIF_P) = make_small(flags|current_flag);
	else if (BIF_ARG_2 == am_false)
	    SEQ_TRACE_TOKEN_FLAGS(BIF_P) = make_small(flags&~current_flag);
	else { 
	    BIF_ERROR2(BADARG, am_seq_trace, BIF_ARG_1, BIF_ARG_2); 
	}
	BIF_RET(old_value);
    }
    else if (BIF_ARG_1 == am_label) {
	if (!(is_atom(BIF_ARG_2) || is_small(BIF_ARG_2))) {
	    BIF_ERROR2(BADARG, am_seq_trace, BIF_ARG_1, BIF_ARG_2);
	}
        new_seq_trace_token(BIF_P);
	old_value = SEQ_TRACE_TOKEN_LABEL(BIF_P);
	SEQ_TRACE_TOKEN_LABEL(BIF_P) = BIF_ARG_2;
    	BIF_RET(old_value);
    }
    else if (BIF_ARG_1 == am_serial) {
	uint32* tp;
	if (is_not_tuple(BIF_ARG_2)) {
	    BIF_ERROR2(BADARG, am_seq_trace, BIF_ARG_1, BIF_ARG_2);
	}
	tp = ptr_val(BIF_ARG_2);
	if ((*tp != make_arityval(2)) || is_not_small(*(tp+1)) || is_not_small(*(tp+2))) {
	    BIF_ERROR2(BADARG, am_seq_trace, BIF_ARG_1, BIF_ARG_2);
        }
        new_seq_trace_token(BIF_P);
	hp = HAlloc(BIF_P,3);
 	old_value = TUPLE2(hp, SEQ_TRACE_TOKEN_LASTCNT(BIF_P),
			   SEQ_TRACE_TOKEN_SERIAL(BIF_P));
	SEQ_TRACE_TOKEN_LASTCNT(BIF_P) = *(tp+1);
 	SEQ_TRACE_TOKEN_SERIAL(BIF_P) = *(tp+2);
	BIF_P->seq_trace_clock = unsigned_val(*(tp+2));
	BIF_P->seq_trace_lastcnt = unsigned_val(*(tp+1));
    	BIF_RET(old_value);
    }
    else if (BIF_ARG_1 == am_sequential_trace_token) {
	if (is_not_nil(BIF_ARG_2)) {
	    BIF_ERROR2(BADARG, am_seq_trace, BIF_ARG_1, BIF_ARG_2);
        }
        old_value = SEQ_TRACE_TOKEN(BIF_P);
        SEQ_TRACE_TOKEN(BIF_P) = NIL;
        BIF_RET(old_value);
    }
    else {
	BIF_ERROR2(BADARG, am_seq_trace, BIF_ARG_1, BIF_ARG_2);
    }
#endif
}

#ifdef SEQ_TRACE
void new_seq_trace_token(Process* p) {
    uint32* hp;
    if (SEQ_TRACE_TOKEN(p) == NIL) {
      hp = HAlloc(p, 6);
      SEQ_TRACE_TOKEN(p) = TUPLE5(hp, make_small(0), /*Flags*/ 
				      make_small(0), /*Label*/
                                      make_small(0), /*Serial*/
				      p->id,         /*From*/
	                              make_small(p->seq_trace_lastcnt));
    }
}
#endif

BIF_RETTYPE seq_trace_info_1(BIF_ALIST_1)
BIF_ADECL_1
{
#ifndef SEQ_TRACE
    BIF_ERROR1(BADARG, am_seq_trace_info, BIF_ARG_1);
#else
    uint32 item, res;
    uint32* hp;
    uint32 current_flag;

    if (is_not_atom(BIF_ARG_1)) {
	BIF_ERROR1(BADARG, am_seq_trace_info, BIF_ARG_1);
    }

    item = BIF_ARG_1;

    if (SEQ_TRACE_TOKEN(BIF_P) == NIL) {
	if ((item == am_send) || (item == am_receive) || 
	    (item == am_print) || (item == am_timestamp)) {
	    hp = HAlloc(BIF_P,3);
	    res = TUPLE2(hp, item, am_false);
	    BIF_RET(res);
	} 
	else if ((item == am_label) || (item == am_serial)) {
	    BIF_RET(NIL);
	}
	else {
	    BIF_ERROR1(BADARG, am_seq_trace_info, BIF_ARG_1);
	}
    }

    if (BIF_ARG_1 == am_send) {
	current_flag = SEQ_TRACE_SEND;
    } else if (BIF_ARG_1 == am_receive) {
	current_flag = SEQ_TRACE_RECEIVE; 
    } else if (BIF_ARG_1 == am_print) {
	current_flag = SEQ_TRACE_PRINT; 
    } else if (BIF_ARG_1 == am_timestamp) {
	current_flag = SEQ_TRACE_TIMESTAMP; 
    }
    else
	current_flag = 0;


    if (current_flag) {
      res = unsigned_val(SEQ_TRACE_TOKEN_FLAGS(BIF_P)) & current_flag ? 
	  am_true : am_false;
    }
    else if (item == am_label) {
      res = SEQ_TRACE_TOKEN_LABEL(BIF_P);
    }
    else if (item  == am_serial) {
	hp = HAlloc(BIF_P, 3);
	res = TUPLE2(hp, SEQ_TRACE_TOKEN_LASTCNT(BIF_P), SEQ_TRACE_TOKEN_SERIAL(BIF_P));
    }
    else {
	BIF_ERROR1(BADARG, am_seq_trace_info, BIF_ARG_1);
    }
    hp = HAlloc(BIF_P, 3);
    res = TUPLE2(hp, item, res);
    BIF_RET(res);
#endif
}

/*
   seq_trace_print(Label,Message) -> true | false
   This function passes Message to the system_tracer
   if the trace_token is not NIL and the trace_token label is equal to
   Label. Returns true if Message is passed else false
   Note! That true is returned if the conditions to pass Message is
   fulfilled, but nothing is passed if system_seq_tracer is not set.
 */
BIF_RETTYPE seq_trace_print_2(BIF_ALIST_2)    
BIF_ADECL_2
{
#ifndef SEQ_TRACE
    BIF_ERROR2(BADARG, am_seq_trace_print, BIF_ARG_1, BIF_ARG_2);
#else
    if (SEQ_TRACE_TOKEN(BIF_P) == NIL) 
	BIF_RET(am_false);
    if (!(is_atom(BIF_ARG_1) || is_small(BIF_ARG_1))) {
	BIF_ERROR2(BADARG, am_seq_trace_print, BIF_ARG_1, BIF_ARG_2);
    }
    if (SEQ_TRACE_TOKEN_LABEL(BIF_P) != BIF_ARG_1)
	BIF_RET(am_false);
    seq_trace_update_send(BIF_P);
    seq_trace_output(SEQ_TRACE_TOKEN(BIF_P), BIF_ARG_2, SEQ_TRACE_PRINT, NIL);
    BIF_RET(am_true);
#endif
}

BIF_RETTYPE system_flag_2(BIF_ALIST_2)    
BIF_ADECL_2
{
#ifdef SEQ_TRACE
    uint32 old_value;
    Process* tracer;

    if (BIF_ARG_1 == am_sequential_tracer) {

	old_value = is_pid(system_seq_tracer) ? system_seq_tracer : am_false;

	if(BIF_ARG_2 == am_false) {
	    system_seq_tracer = NIL;
	    BIF_RET(old_value);
	} 
	else if (is_pid(BIF_ARG_2) && (get_node(BIF_ARG_2) == THIS_NODE)) {
	    /* must be a local PID */
	    tracer = process_tab[get_number(BIF_ARG_2)];
	    if (!INVALID_PID(tracer, BIF_ARG_2)) {
		system_seq_tracer = BIF_ARG_2;
		BIF_RET(old_value);
	    }
	}
    }
    else if (BIF_ARG_1 == make_small(1)) {
	uint32 i;
	ErlMessage* mp;
	for (i = 0; i < max_process; i++) {
	    if (process_tab[i] != (Process*) 0) {
		Process* p = process_tab[i];
		p->seq_trace_token = NIL;
		p->seq_trace_clock = 0;
		p->seq_trace_lastcnt = 0;
		mp = p->msg.first;
		while(mp != NULL) {
		    mp->seq_trace_token = NIL;
		    mp = mp->next;
		}
	    }
	}
	BIF_RET(am_true);    
    }
#endif
    BIF_ERROR2(BADARG, am_system_flag, BIF_ARG_1, BIF_ARG_2);
}

BIF_RETTYPE system_info_1(BIF_ALIST_1)    
BIF_ADECL_1
{
#ifdef SEQ_TRACE
    uint32 res;
    uint32* hp;
    uint32 val;

    if (BIF_ARG_1 == am_sequential_tracer) {
	val = is_pid(system_seq_tracer) ? system_seq_tracer : am_false;
	hp = HAlloc(BIF_P, 3);
	res = TUPLE2(hp, am_sequential_tracer, 
		     val);
	BIF_RET(res);
    }
#endif
    BIF_ERROR1(BADARG, am_system_info, BIF_ARG_1);
}

/**********************************************************************/

BIF_RETTYPE hash_2(BIF_ALIST_2)
BIF_ADECL_2
{
    uint32 hash;
    sint32 range;

    if (is_not_small(BIF_ARG_2)) {
	BIF_ERROR2(BADARG, am_hash, BIF_ARG_1, BIF_ARG_2);
    }
    if ((range = signed_val(BIF_ARG_2)) <= 0) {  /* [1..MAX_SMALL] */
	BIF_ERROR2(BADARG, am_hash, BIF_ARG_1, BIF_ARG_2);
    }
    hash = make_hash(BIF_ARG_1, 0);
    BIF_RET(make_small(1 + (hash % range)));   /* [1..range] */
}

/**********************************************************************/

BIF_RETTYPE type_2(BIF_ALIST_2)
BIF_ADECL_2
{
#if defined(BEAM)  
    /* not supported in beam */
    BIF_ERROR2(BADARG, am_type, BIF_ARG_1, BIF_ARG_2);    
#else
    uint32 tmp;
    int k;
    int i;

    if (is_not_small(BIF_ARG_2)) {
	BIF_ERROR2(BADARG, am_type, BIF_ARG_1, BIF_ARG_2);
    }
    tmp = signed_val(BIF_ARG_2);
    k = typeval[tag_val_def(BIF_ARG_1)]; /* map to type value bit */
    /* Check type mask */
    if ((k < 1) || (k > 10) || (((1 << k) & tmp) == 0))
	BIF_RET(SMALL_ZERO);
    /* The type number is depending on the mask, we is only
       intressted in the ordinal from the types we are 
       interested in (i.e a mask with only two types 
       always return 1 or 2) */
    i = k-1;
    while (i > 0) {
	if (((1 << i) & tmp) == 0)
	    k--;
	i--;
    }
    BIF_RET(make_small(k));
#endif
}

/**********************************************************************/

BIF_RETTYPE pre_loaded_0(BIF_ALIST_0)
BIF_ADECL_0
{
    uint32 previous;
    int j;
    uint32 need;
    uint32 mod;
    uint32* hp;
    char* name;
    const Preload *preload = sys_preloaded();

    j = 0;
    while (preload[j].name != NULL)
	j++;
    previous = NIL;
    need = 2*j;
    hp = HAlloc(BIF_P, need);
    j = 0;
    while((name = preload[j].name) != NULL)  {
	mod = am_atom_put(name, sys_strlen(name));
	previous = CONS(hp, mod, previous);
	hp += 2;
	j++;
    }
    BIF_RET(previous);
}

/**********************************************************************/

BIF_RETTYPE loaded_0(BIF_ALIST_0)
BIF_ADECL_0
{
    uint32 previous = NIL;
    uint32* hp;
    int i;
    int j = 0;
    
    for (i = 0; i < module_code_size; i++) {
	if (module_code(i) != NULL &&
	    ((module_code(i)->code_length != 0) ||
	     (module_code(i)->old_code_length != 0))) {
	    j++;
	}
    }
    if (j > 0) {
	hp = HAlloc(BIF_P, j*2);

	for (i = 0; i < module_code_size; i++) {
	    if (module_code(i) != NULL &&
		((module_code(i)->code_length != 0) ||
		 (module_code(i)->old_code_length != 0))) {
		previous = CONS(hp, make_atom(module_code(i)->module), 
				previous);
		hp += 2;
	    }
	}
    }
    BIF_RET(previous);
}


/**********************************************************************/

/*
 * This function takes care of calls to erlang:info/1 when the argument
 * is a tuple.
 */
static BIF_RETTYPE
info_1_tuple(Process* BIF_P,	/* Pointer to current process. */
	     uint32* tp,	/* Pointer to first element in tuple */
	     int arity)		/* Arity of tuple (untagged). */
{
    uint32 sel;
    uint32 str;
    int len;
    char *val;
    uint32* hp;

    sel = *tp++;
    if (sel == am_getenv) {
	if (arity != 2)
	    return 0;
	if (!is_string(*tp))
	    return 0;
       
	if ((len = intlist_to_buf(*tp, tmp_buf, TMP_BUF_SIZE-1)) < 0)
	    return 0;
	tmp_buf[len] = '\0';
	val = getenv(tmp_buf);
	if (val == NULL)
	    str = am_false;
	else {
	    len = strlen(val);
	    hp = HAlloc(BIF_P, len*2);
	    str = buf_to_intlist(&hp, val, len, NIL);
	}
	BIF_RET(str);
#ifdef INSTRUMENT
    } else if (sel == am_allocated) {
	if (!is_string(*tp))
	    return 0;
       
	if ((len = intlist_to_buf(*tp, tmp_buf, TMP_BUF_SIZE-1)) < 0)
	    return 0;
	tmp_buf[len] = '\0';

	if (dump_memory_data(tmp_buf))
	    BIF_RET(am_true);
	else
	    return 0;		/* Return {error, Errno} instead? */
#endif
#ifdef PURIFY
    } else if (sel == am_purify) {
	if (*tp == am_memory) {
	    BIF_RET(make_small_or_big(purify_new_leaks(), BIF_P));
	} else if (*tp == am_fd) {
	    BIF_RET(make_small_or_big(purify_new_fds_inuse(), BIF_P));
	} else if (*tp == am_running) {
	    BIF_RET(purify_is_running() ? am_true : am_false);
	} else if (is_list(*tp)) {
	    int iopos = 0;
	    int r;

	    r = io_list_to_buf(*tp, (char*) tmp_buf, &iopos, TMP_BUF_SIZE - 5);
	    if (r == 0) {
		tmp_buf[iopos] = 0;
		purify_printf("%s\n", tmp_buf);
	    } else {
		return 0;
	    }
	    BIF_RET(am_true);
	}
#endif
#ifdef QUANTIFY
    } else if (sel == am_quantify) {
	if (*tp == am_clear) {
	    quantify_clear_data();
	    BIF_RET(am_true);
	} else if (*tp == am_start) {
	    quantify_start_recording_data();
	    BIF_RET(am_true);
	} else if (*tp == am_stop) {
	    quantify_stop_recording_data();
	    BIF_RET(am_true);
	} else if (*tp == am_running) {
	    BIF_RET(quantify_is_running() ? am_true : am_false);
	}
#endif
#ifdef BEAM
    } else if (sel == am_exports) {
	if (arity != 2)
	    return 0;
	return exported_from_module(BIF_P, *tp);
    } else if (sel == am_attributes) {
	if (arity != 3)
	    return 0;
	return attributes_for_module(BIF_P, tp[0], tp[1]);
#endif
#if defined(__GNUC__) && defined(sparc)
    } else if (sel == am_ultrasparc_set_pcr) {
	unsigned long long tmp;
	int fd;
	int rc;

	if (arity != 2 || !is_small(*tp)) {
	    return 0;
	}
	tmp = signed_val(*tp);
	if ((fd = open("/dev/perfmon", O_RDONLY)) == -1) {
	    BIF_RET(am_false);
	}
	rc = ioctl(fd, PERFMON_SETPCR, &tmp);
	close(fd);
	if (rc < 0) {
	    BIF_RET(am_false);
	}
	BIF_RET(am_true);
#endif
    } else if (sel == am_reductions) {
	int reds;
	
	if (arity != 2)
	    return 0;
	if (is_not_integer(*tp) || ((reds = signed_val(*tp)) < 0))
	    return 0;
	if (reds > CONTEXT_REDS)
	    reds = CONTEXT_REDS;
	BIF_RET2(am_true, reds);
    }
    return 0;
}

BIF_RETTYPE info_1(BIF_ALIST_1)
BIF_ADECL_1
{
    uint32 res;
    uint32* hp;
    cerr_pos = 0;

    if (is_tuple(BIF_ARG_1)) {
	uint32* tp = ptr_val(BIF_ARG_1);
	uint32 arity = *tp++;
	res = info_1_tuple(BIF_P, tp, unsigned_val(arity));
	if (res == 0)
	    goto error;
	return res;
    } else if (BIF_ARG_1 == am_info)
	info(CBUF);
    else if (BIF_ARG_1 == am_procs)
	process_info(CBUF);
    else if (BIF_ARG_1 == am_loaded)
	loaded(CBUF);
    else if (BIF_ARG_1 == am_dist)
	distribution_info(CBUF);
    else if (BIF_ARG_1 == am_system_version) {
	int n;

	sys_strcpy((char*)tmp_buf, "Erlang (");
	n = sys_strlen((char*)tmp_buf);
	sys_strcpy((char*)(tmp_buf+n), EMULATOR);
	n += sys_strlen(EMULATOR);
	sys_strcpy((char*)(tmp_buf+n), ") emulator version ");
	n += 19;
	sys_strcpy((char*)(tmp_buf+n), ERLANG_VERSION);
	n += sizeof(ERLANG_VERSION)-1;
#ifdef DEBUG
	sys_strcpy((char *)(tmp_buf+n), " [debug-compiled]");
	n = strlen(tmp_buf);
#endif	
#ifdef INSTRUMENT
	sys_strcpy((char *)(tmp_buf+n), " [instrumented]");
	n = strlen(tmp_buf);
#endif	
	tmp_buf[n] = '\n';
	tmp_buf[n+1] = '\0';
	n++;
	hp = HAlloc(BIF_P, n*2);
	BIF_RET(buf_to_intlist(&hp, tmp_buf, n, NIL));
    }
    else if (BIF_ARG_1 == am_getenv) {
       GETENV_STATE state;
       char *cp;
       uint32 ret, str;
       int len;

       init_getenv_state(&state);

       ret = NIL;
       while ((cp = getenv_string(&state)) != NULL)
       {
	  len = strlen(cp);
	  hp = HAlloc(BIF_P, len*2);
	  str = buf_to_intlist(&hp, cp, len, NIL);
	  hp = HAlloc(BIF_P, 2);
	  ret = CONS(hp, str, ret);
       }

       BIF_RET(ret);
    }
#ifdef INSTRUMENT
    else if (BIF_ARG_1 == am_allocated) {
       uint32 val;

       val = collect_memory(BIF_P);
       BIF_RET(val);
    }
#endif
    else if (BIF_ARG_1 == am_os_type) {
       uint32 type = am_atom_put(os_type, strlen(os_type));
       uint32 flav, tup;

       os_flavor(tmp_buf, TMP_BUF_SIZE);
       flav = am_atom_put(tmp_buf, strlen(tmp_buf));
       hp = HAlloc(BIF_P, 3);
       tup = TUPLE2(hp, type, flav);
       BIF_RET(tup);
    }
    else if (BIF_ARG_1 == am_os_version) {
       int major, minor, build;
       uint32 tup;

       os_version(&major, &minor, &build);
       hp = HAlloc(BIF_P, 4);
       tup = TUPLE3(hp,
		    make_small(major),
		    make_small(minor),
		    make_small(build));
       BIF_RET(tup);
    }
    else if (BIF_ARG_1 == am_version) {
	int n = strlen(ERLANG_VERSION);
	hp = HAlloc(BIF_P, ((sizeof ERLANG_VERSION)-1) * 2);
	BIF_RET(buf_to_intlist(&hp, (byte*)ERLANG_VERSION, n, NIL));
    }
    else if (BIF_ARG_1 == am_machine) {
	int n = strlen(EMULATOR);
	hp = HAlloc(BIF_P, n*2);
	BIF_RET(buf_to_intlist(&hp, (byte*)EMULATOR, n, NIL));
    }
    else if (BIF_ARG_1 == am_garbage_collection) {
	if (switch_gc_threshold == 0)
	    BIF_RET(am_generational);
	else if(switch_gc_threshold >= MAX_SMALL)
	    BIF_RET(am_fullsweep);
	else {
	    uint32 tup;
	    hp = HAlloc(BIF_P, 3);
	    tup = TUPLE2(hp, am_switch, make_small(switch_gc_threshold));
	    BIF_RET(tup);
	}
#if defined(BEAM)
    } else if (BIF_ARG_1 == am_instruction_counts) {
	int i;
	hp = HAlloc(BIF_P, num_instructions*5);
	res = NIL;
	for (i = num_instructions-1; i >= 0; i--) {
	    uint32 tuple;
	    uint32 atom = am_atom_put(opc[i].name, strlen(opc[i].name));
	    uint32 count;

	    if (IS_USMALL(0, opc[i].count)) {
		count = make_small(opc[i].count);
	    } else {
		uint32* num_hp = ArithAlloc(BIF_P, 2);
		count = uint32_to_big(opc[i].count, num_hp);
	    }
	    tuple = TUPLE2(hp, atom, count);
	    hp += 3;
	    res = CONS(hp, tuple, res);
	    hp += 2;
	}
	BIF_RET(res);
#endif
#if defined(__GNUC__) && defined(sparc)
    } else if (BIF_ARG_1 == am_ultrasparc_read_tick1) {
	register unsigned high asm("%l0");
	register unsigned low asm("%l1");

	hp = HAlloc(BIF_P, 5);
	asm volatile (".word 0xa3410000;" /* rd %tick, %l1 */
		      ".word 0xa1347020" /* srlx  %l1, 0x20, %l0 */
		      : "=r" (high), "=r" (low));
	res = TUPLE4(hp, make_small(high >> 16),
		     make_small(high & 0xFFFF),
		     make_small(low >> 16),
		     make_small(low & 0xFFFF));
	BIF_RET(res);
    } else if (BIF_ARG_1 == am_ultrasparc_read_tick2) {
	register unsigned high asm("%l0");
	register unsigned low asm("%l1");

	asm volatile (".word 0xa3410000;" /* rd %tick, %l1 */
		      ".word 0xa1347020" /* srlx  %l1, 0x20, %l0 */
		      : "=r" (high), "=r" (low));
	hp = HAlloc(BIF_P, 5);
	res = TUPLE4(hp, make_small(high >> 16),
		     make_small(high & 0xFFFF),
		     make_small(low >> 16),
		     make_small(low & 0xFFFF));
	BIF_RET(res);
    } else if (BIF_ARG_1 == am_ultrasparc_read_pic1) {
	register unsigned high asm("%l0");
	register unsigned low asm("%l1");

	hp = HAlloc(BIF_P, 5);
	asm volatile (".word 0xa3444000;" /* rd %asr17, %l1 */
		      ".word 0xa1347020" /* srlx  %l1, 0x20, %l0 */
		      : "=r" (high), "=r" (low));
	res = TUPLE4(hp, make_small(high >> 16),
		     make_small(high & 0xFFFF),
		     make_small(low >> 16),
		     make_small(low & 0xFFFF));
	BIF_RET(res);
    } else if (BIF_ARG_1 == am_ultrasparc_read_pic2) {
	register unsigned high asm("%l0");
	register unsigned low asm("%l1");

	asm volatile (".word 0xa3444000;" /* rd %asr17, %l1 */
		      ".word 0xa1347020" /* srlx  %l1, 0x20, %l0 */
		      : "=r" (high), "=r" (low));
	hp = HAlloc(BIF_P, 5);
	res = TUPLE4(hp, make_small(high >> 16),
		     make_small(high & 0xFFFF),
		     make_small(low >> 16),
		     make_small(low & 0xFFFF));
	BIF_RET(res);
#endif
    } else {
    error:
	cerr_pos = 0;
	BIF_ERROR1(BADARG, am_info, BIF_ARG_1);
    }
    res = new_binary(BIF_P, tmp_buf, cerr_pos);
    cerr_pos = 0;
    BIF_RET(res);
}

/* XXX   Rewrite this !!!! */
BIF_RETTYPE float_to_words_1(BIF_ALIST_1)
BIF_ADECL_1
{
    uint32 e1,e2;
    uint32* hp;
    FloatDef f;

    if (is_not_float(BIF_ARG_1)) 
	BIF_ERROR1(BADARG, am_float_to_words, BIF_ARG_1);
    
    GET_DOUBLE(BIF_ARG_1, f);
    
    hp = HAlloc(BIF_P, 3+5+5);
    
    if (abs(f.fw[0]) > MAX_SMALL) {
	e1 = small_to_big(f.fw[0], hp);
	hp += big_arity(e1)+1;
    }
    else
	e1 = make_small(f.fw[0]);

    if (abs(f.fw[1]) > MAX_SMALL) {
	e2 = small_to_big(f.fw[1], hp);
	hp += big_arity(e2)+1;
    }
    else
	e2 = make_small(f.fw[1]);

    BIF_RET(TUPLE2(hp, e1, e2));
}

/****************************************************************************

  PORT BIFS:

           port_command/2   -- replace Port ! {..., {command, Data}}
               port_command(Port, Data) -> true
               when port(Port), io-list(Data)

           port_control/3   -- new port_control(Port, Ctl, Data) -> Reply
	      port_control(Port, Ctl, Data) -> Reply
              where integer(Ctl), io-list(Data), io-list(Reply)

           port_close/1     -- replace Port ! {..., close}
             port_close(Port) -> true
             when port(Port)

           port_connect/2   -- replace Port ! {..., {connect, Pid}}
              port_connect(Port, Pid) 
              when port(Port), pid(Pid)

 ***************************************************************************/

static Port* id2port(id)
uint32 id;
{
    int ix;

    if (is_not_port(id) || (get_node_reference(id) != THIS_NODE))
	return NULL;
    ix = get_number_reference(id);
    if ((port[ix].status == FREE) || (port[ix].status & CLOSING))
	return NULL;
    return &port[ix];
}


BIF_RETTYPE port_command_2(BIF_ALIST_2)
BIF_ADECL_2
{
    Port* p;

    if ((p = id2port(BIF_ARG_1)) == NULL)
	BIF_ERROR2(BADARG, am_port_command, BIF_ARG_1, BIF_ARG_2);

    if (p->status & PORT_BUSY) {
	erl_suspend(BIF_P, BIF_ARG_1);
	BIF_ERROR(RESCHEDULE);
    }

    if (write_port(get_number_reference(BIF_ARG_1), BIF_ARG_2) != 0)
    {
       BIF_ERROR2(BADARG, am_port_command, BIF_ARG_1, BIF_ARG_2);
    }

    if (BIF_P->status == P_EXITING) {
       KILL_CATCHES(BIF_P);	/* Must exit */
       BIF_ERROR2(USER_ERROR, am_port_command, BIF_ARG_1, BIF_ARG_2);
    }
    BIF_RET(am_true);
}


BIF_RETTYPE port_control_3(BIF_ALIST_3)
BIF_ADECL_3
{
    Port* p;
    int op;
    uint32 res;

    if ((p = id2port(BIF_ARG_1)) == NULL) {
    error:
	BIF_ERROR3(BADARG, am_port_control, BIF_ARG_1, BIF_ARG_2, BIF_ARG_3);
    }
    if (is_not_small(BIF_ARG_2))
	goto error;
    op = signed_val(BIF_ARG_2);
    
    if (port_control(BIF_P, p, op, BIF_ARG_3, &res) == 0)
	goto error;
    BIF_RET(res);
}

BIF_RETTYPE port_close_1(BIF_ALIST_1)
BIF_ADECL_1
{
    Port* p;
    if ((p = id2port(BIF_ARG_1)) == NULL)
	BIF_ERROR1(BADARG, am_port_close, BIF_ARG_1);
    do_exit_port(BIF_ARG_1, p->connected, am_normal);
    /* since we terminate port with reason normal 
       we SHOULD never get an exit signal out selfs !!!
       */
    BIF_RET(am_true);
}


BIF_RETTYPE port_connect_2(BIF_ALIST_2)
BIF_ADECL_2
{
    Port* p;
    if (is_not_pid(BIF_ARG_2) || (p = id2port(BIF_ARG_1)) == NULL)
	BIF_ERROR2(BADARG, am_port_connect, BIF_ARG_1, BIF_ARG_2);
    p->connected = BIF_ARG_2;
    BIF_RET(am_true);
}

/****************************************************************************
** BIF Timer support
****************************************************************************/

typedef struct bif_timer_rec {
    struct bif_timer_rec* next;
    ErlTimer tm;
    ErlMessageBuffer* bp;
    uint32 message;
    uint32 pid;
    uint32 ref;
} BifTimerRec;

#define TIMER_HASH_VEC 251

static BifTimerRec* bif_tm_vec[TIMER_HASH_VEC];  /* XXX fix proper init */

static BifTimerRec** find_timer(ref)
uint32 ref;
{
    int ix = get_number_reference(ref) % TIMER_HASH_VEC;
    BifTimerRec** tp = &bif_tm_vec[ix];

    while(*tp != NULL) {
	if ((*tp)->ref == ref)
	    return tp;
	tp = &(*tp)->next;
    }
    return NULL;
}

static void bif_timeout_proc(btm)
BifTimerRec* btm;
{
    Process* rp;
    BifTimerRec** tp = find_timer(btm->ref);
    int invalid_pid;

    ASSERT((tp != NULL) && (*tp == btm));
    *tp = btm->next;

    if (is_atom(btm->pid))
    {
       rp = whereis_process(unsigned_val(btm->pid));
       invalid_pid = (rp == NULL);
    }
    else
    {
       rp = process_tab[get_number(btm->pid)];
       invalid_pid = (INVALID_PID(rp, btm->pid));
    }

    if (invalid_pid)
       free_message_buffer(btm->bp);
    else
       queue_message(rp, btm->bp, btm->message);

    sys_free(btm);
}

/* tm arg contains the BifTimerRec */
static void bif_cancel_proc(btm)
BifTimerRec* btm;
{
    free_message_buffer(btm->bp);
    sys_free(btm);
}

static BifTimerRec *do_timer(pack, process, arg1, arg2, arg3)
int pack;
Process *process;
uint32 arg1, arg2, arg3;
{
    BifTimerRec* btm;
    ErlMessageBuffer* bp;
    sint32 timeout;
    uint32 size;
    uint32 term, msg;
    uint32 ref;
    uint32* hp;
    int ix;

    if (is_not_small(arg1)
	|| (is_not_pid(arg2) && is_not_atom(arg2)))
       return NULL;
    timeout = signed_val(arg1);
    if ((timeout < 0) || (is_pid(arg2) && get_node(arg2) != THIS_NODE))
	return NULL;

    /* find an acceptable reference */
    do {
	ref = make_ref();
    } while(find_timer(ref) != NULL);

    msg = arg3;

    if (pack)
    {
       hp = HAlloc(process, 4);
       term = TUPLE3(hp, am_timeout, ref, msg);
    }
    else
       term = msg;

    size = size_object(term);
    btm = (BifTimerRec*) safe_alloc(sizeof(BifTimerRec));
    btm->bp = bp = new_message_buffer(size);
    btm->ref = ref;
    btm->pid = arg2;
    ix = get_number_reference(ref) % TIMER_HASH_VEC;
    btm->next = bif_tm_vec[ix];
    bif_tm_vec[ix] = btm;
    hp = bp->mem;
    btm->message = copy_struct(term, size, &hp, &bp->mso);
    btm->tm.active = 0; /* MUST be initalized */
    erl_set_timer(&btm->tm,
		  bif_timeout_proc,
		  bif_cancel_proc,
		  (void*)btm,
		  timeout);

    return btm;
}

/* send_after(Pid, Time, Message) -> Ref */
BIF_RETTYPE send_after_3(BIF_ALIST_3)
BIF_ADECL_3
{
    BifTimerRec* btm;

    btm = do_timer(0, BIF_P,
		   BIF_ARG_1, BIF_ARG_2, BIF_ARG_3);

    if (btm != NULL)
       BIF_RET(btm->ref);
    else
       BIF_ERROR3(BADARG, am_send_after, BIF_ARG_1, BIF_ARG_2, BIF_ARG_3);
}

/* start_timer(Pid, Time, Message) -> Ref */
BIF_RETTYPE start_timer_3(BIF_ALIST_3)
BIF_ADECL_3
{
    BifTimerRec* btm;

    btm = do_timer(1, BIF_P,
		   BIF_ARG_1, BIF_ARG_2, BIF_ARG_3);

    if (btm != NULL)
       BIF_RET(btm->ref);
    else
       BIF_ERROR3(BADARG, am_start_timer, BIF_ARG_1, BIF_ARG_2, BIF_ARG_3);
}

/* cancel_timer(Ref)) -> Bool */
BIF_RETTYPE cancel_timer_1(BIF_ALIST_1)
BIF_ADECL_1
{
    BifTimerRec* btm;
    BifTimerRec** tp;
    uint32 left;

    if (is_not_refer(BIF_ARG_1))
	BIF_ERROR1(BADARG, am_cancel_timer, BIF_ARG_1);
    if ((tp = find_timer(BIF_ARG_1)) != NULL) {
	btm = *tp;
	*tp = btm->next;
	left = time_left(&btm->tm);
	erl_cancel_timer(&btm->tm);
	BIF_RET(make_small(left));
    }
    else 
	BIF_RET(am_false);
}

/* return the pid of the erlang process. For internal use*/

BIF_RETTYPE get_os_pid_0(BIF_ALIST_0)
BIF_ADECL_0
{
     char pid_string[21]; /* enough for a 64 bit number */
     int n;
     uint32* hp;
     sys_get_pid(pid_string); /* In sys.c */
     n = sys_strlen(pid_string);
     hp = HAlloc(BIF_P, n*2);
     BIF_RET(buf_to_intlist(&hp, pid_string, n, NIL));
}

BIF_RETTYPE os_put_env_1(BIF_ALIST_1)
BIF_ADECL_1
{
    int i;
    if(!is_string(BIF_ARG_1))
	BIF_ERROR1(BADARG, am_os_put_env, BIF_ARG_1);
    i = intlist_to_buf(BIF_ARG_1, tmp_buf, TMP_BUF_SIZE);
    tmp_buf[i] = '\0';
    if(sys_putenv(tmp_buf))
	BIF_ERROR1(BADARG, am_os_put_env, BIF_ARG_1);
    BIF_RET(am_true);
}

    
