/* ``The contents of this file are subject to the Erlang Public License,
 * Version 1.0, (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.erlang.org/EPL1_0.txt
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 * 
 * The Original Code is Erlang-4.7.3, December, 1998.
 * 
 * The Initial Developer of the Original Code is Ericsson Telecom
 * AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
 * Telecom AB. All Rights Reserved.
 * 
 * Contributor(s): ______________________________________.''
 */
/*
 * Defines the interfaces between the generic efile driver and its
 * operating-system dependent helpers.
 */

#include "sys.h"

/*
 * Open modes for efile_openfile().
 */
#define EFILE_MODE_READ 1
#define EFILE_MODE_WRITE 2	/* Implies truncating file when used alone. */
#define EFILE_MODE_READ_WRITE 3
#define EFILE_COMPRESSED 4

/*
 * Seek modes for efile_seek().
 */
#define	EFILE_SEEK_SET	0
#define	EFILE_SEEK_CUR	1
#define	EFILE_SEEK_END	2

/*
 * File types returned by efile_fileinfo().
 */
#define FT_DEVICE 	1
#define FT_DIRECTORY 	2
#define FT_REGULAR	3
#define FT_SYMLINK	4
#define FT_OTHER	5

/*
 * Access attributes returned by efile_fileinfo() (the bits can be ORed
 * together).
 */
#define FA_NONE    	0
#define FA_WRITE 	1
#define FA_READ		2

/*
 * An handle to an open directory.  To be cast to the correct type
 * in the system-dependent directory functions.
 */

typedef struct _Efile_Dir_Handle* EFILE_DIR_HANDLE;

/*
 * Error information from the last call.
 */  
typedef struct _Efile_error {
    int posix_errno;		/* Posix error number, as in <errno.h>. */
    int os_errno;		/* Os-dependent error number. */
} Efile_error;

/*
 * This structure contains date and time.
 */
typedef struct _Efile_time {
    unsigned year;		/* (4 digits). */
    unsigned month;		/* (1..12). */
    unsigned day;		/* (1..31). */
    unsigned hour;		/* (0..23). */
    unsigned minute;		/* (0..59). */
    unsigned second;		/* (0..59). */
} Efile_time;


/*
 * Describes what is returned by file:file_info/1.
 */

typedef struct _Efile_info {
    uint32 size_low;		/* Size of file, lower 32 bits.. */
    uint32 size_high;		/* Size of file, higher 32 bits. */
    uint32 type;		/* Type of file -- one of FT_*. */
    uint32 access;		/* Access to file -- one of FA_*. */
    uint32 mode;		/* Access permissions -- bit field. */
    uint32 links;		/* Number of links to file. */
    uint32 major_device;	/* Major device or file system. */
    uint32 minor_device;	/* Minor device (for devices). */
    uint32 inode;		/* Inode number. */
    uint32 uid;			/* User id of owner. */
    uint32 gid;			/* Group id of owner. */
    Efile_time accessTime;	/* Last time the file was accessed. */
    Efile_time modifyTime;	/* Last time the file was modified. */
    Efile_time cTime;		/* Creation time (Windows) or last
				 * inode change (Unix).
				 */
} Efile_info;

/*
 * Functions.
 */

FUNCTION(int, efile_mkdir, (Efile_error* errInfo, char* name));
FUNCTION(int, efile_rmdir, (Efile_error* errInfo, char* name));
FUNCTION(int, efile_delete_file, (Efile_error* errInfo, char* name));
FUNCTION(int, efile_rename, (Efile_error* errInfo, char* src, char* dst));
FUNCTION(int, efile_chdir, (Efile_error* errInfo, char* name));
FUNCTION(int, efile_getdcwd, (Efile_error* errInfo, int drive,
			      char* buffer, unsigned size));
FUNCTION(int, efile_readdir, (Efile_error* errInfo, char* name, 
			      EFILE_DIR_HANDLE* dir_handle,
			      char* buffer, unsigned size));
FUNCTION(int, efile_openfile, (Efile_error* errInfo, char* name, int flags,
			       int* pfd, unsigned* pSize));
FUNCTION(void, efile_closefile, (int fd));
FUNCTION(int, efile_fsync, (Efile_error* errInfo, int fd));
FUNCTION(int, efile_fileinfo, (Efile_error* errInfo, Efile_info* pInfo,
			       char *name));
FUNCTION(int, efile_write_info, (Efile_error* errInfo, Efile_info* pInfo,
			       char *name));
FUNCTION(int, efile_write, (Efile_error* errInfo, int fd, char* buf,
			    unsigned count));
FUNCTION(int, efile_read, (Efile_error* errInfo, int fd, char* buf,
			   unsigned count, unsigned* pBytesRead));
FUNCTION(int, efile_seek, (Efile_error* errInfo, int fd, int offset, int origin,
			   unsigned* new_location));
FUNCTION(int, efile_truncate_file, (Efile_error* errInfo, int fd));

FUNCTION(int, efile_pwrite, (Efile_error* errInfo, int fd, char* buf,
			     unsigned count, int offset));
FUNCTION(int, efile_pread, (Efile_error* errInfo, int fd, int offset, char* buf,
			    unsigned count, unsigned* pBytesRead));
