/* ``The contents of this file are subject to the Erlang Public License,
 * Version 1.0, (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.erlang.org/EPL1_0.txt
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 * 
 * The Original Code is Erlang-4.7.3, December, 1998.
 * 
 * The Initial Developer of the Original Code is Ericsson Telecom
 * AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
 * Telecom AB. All Rights Reserved.
 * 
 * Contributor(s): ______________________________________.''
 */
/* Copyright (C) 1997, Ericsson Telecom */

/* 
 * Dynamic driver loader and linker
 */
#include "sys.h"
#include "config.h"
#include "driver.h"
#include "global.h"
#include "erl_ddll.h"

#define LOAD_DRIVER   'l'
#define UNLOAD_DRIVER 'u'
#define GET_DRIVERS   'g'

/***********************************************************************
 * R e p l i e s
 *
 * The following replies are sent from this driver.
 *
 * Reply		Translated to on Erlang side
 * -----		----------------------------
 * [$o]			ok
 * [$o|List]		{ok, List}
 * [$e|Atom]		{error, Atom}
 * [$E|[Atom, 0, Message]]	{error, {Atom, Message}}
 * [$i|String]		"..."    (This is an item in a list.)
 *
 * List of strings are sent as multiple messages:
 *
 *	[$i|String]
 *	.
 *	.
 *	.
 *	[$i|String]
 *	[$o]
 *
 * This will be represented as {ok, [String1, ..., StringN]} in Erlang.
 ************************************************************************/
#define LOAD_FAILED   2

typedef FUNCTION(DriverEntry*, (*DRV_INITFN), (void*));

static long dyn_start();
static int dyn_stop();
static int handle_command();

struct driver_entry ddll_driver_entry = {
    null_func,
    dyn_start,
    dyn_stop,
    handle_command,
    null_func,
    null_func,
    "ddll",
    NULL,
    NULL
};

static long erlang_port = -1;

static long reply(long port, int success, char *str)
{
    char tmp[200];
    *tmp = success;
    strncpy(tmp + 1, str, 198);
    tmp[199] = 0; /* in case str was more than 198 bytes */
    driver_output(port, tmp, strlen(tmp));
    return 0;
}

static long
error_reply(char* atom, char* string)
{
    char* reply;
    int alen;			/* Length of atom. */
    int slen;			/* Length of string. */
    int rlen;			/* Length of reply. */

    alen = strlen(atom);
    slen = strlen(string);
    rlen = 1+alen+1+slen;
    reply = sys_alloc(rlen+1);
    reply[0] = 'E';
    strcpy(reply+1, atom);
    strcpy(reply+alen+2, string);
    driver_output(erlang_port, reply, rlen);
    sys_free(reply);
    return 0;
}

static long dyn_start(long port, char *buf)
{
    if (erlang_port != -1) {
	return -1;			/* Already started! */
    }
    return erlang_port = port;
}

static int dyn_stop()
{
    erlang_port = -1;
    return 0;
}

static int handle_command(long inport, char *buf, int count)
{
    int j;
    uint32 *initfn;
    DriverEntry *dp;
    void *lib;
    char *driver_name;
    
    /*
     * Do some sanity checking on the commands passed in buf. Make sure the
     * last string is null terminated, and that buf isn't empty.
     */
    
    if (count == 0 || (count > 1 && buf[count-1] != 0)) {
	return driver_failure(erlang_port, 100);
    }
  
    switch (*buf++) {
    case LOAD_DRIVER:
	{
	    char* full_name;

	    DE_List *de;

	    if (count < 5) {
		return driver_failure(erlang_port, 110);
	    }

	    driver_name = buf;
	    j = strlen(driver_name);
	    if (j == 0 || j+3 >= count) {
		return driver_failure(erlang_port, 111);
	    }

	    full_name = buf+j+1;
	    
	    for (de = driver_list; de != NULL; de = de->next) {
		if (strcmp(de->drv->driver_name, driver_name) == 0) {
		    return reply(erlang_port, 'e', "already_loaded");
		}
	    }
	    
	    if ((lib = ddll_open(full_name)) == NULL) {
		return error_reply("open_error", ddll_error());
	    }
	    
	    /* Some systems still require an underscore at the beginning of the
	     * symbol name. If we can't find the name without the underscore, try
	     * again with it.
	     */
	
	    if ((initfn = ddll_sym(lib, "driver_init")) == NULL)
		if ((initfn = ddll_sym(lib, "_driver_init")) == NULL) {
		    ddll_close(lib);
		    return reply(erlang_port, 'e', "no_driver_init");
		}

	    /* 
	     * Here we go, lets hope the driver write knew what he was doing...
	     */
	    
	    if ((dp = (((DriverEntry *(*)())initfn)(lib))) == NULL) {
		ddll_close(lib);
		return reply(erlang_port, 'e', "driver_init_failed");
	    }
    
	    if (strcmp(driver_name, dp->driver_name) != 0) {
		ddll_close(lib);
		return reply(erlang_port, 'e', "bad_driver_name");
	    }
    
	    add_driver_entry(dp);
	    return reply(erlang_port, 'o', "");
	}

    case UNLOAD_DRIVER:
	{
	    DE_List *de;

	    if (count < 3) {
		return driver_failure(erlang_port, 200);
	    }

	    driver_name = buf;
	    for (de = driver_list; de != NULL; de = de->next) {
		if (strcmp(de->drv->driver_name, driver_name) == 0) {
		    if (de->drv->finish == NULL || de->drv->handle == NULL) {
			return reply(erlang_port, 'e', "linked_in_driver");
		    }

		    /*
		     * Kill all ports that depend on this driver.
		     */
		    for (j = 0; j < MAX_PORTS; j++) {
			if (port[j].status != FREE 
			    && port[j].drv_ptr == de->drv) {
			    driver_failure(j, -1);
			}
		    }

		    /* 
		     * Let the driver clean up its mess before unloading it.
		     * We ignore errors from the finish funtion and
		     * ddll_close().
		     */

		    (*(de->drv->finish))();
		    ddll_close(de->drv->handle);
		    remove_driver_entry(de->drv);
		    return reply(erlang_port, 'o', "");
		}
	    }
	    return reply(erlang_port, 'e', "not_loaded");
	}
    
    case GET_DRIVERS:
	{
	    DE_List *pos;

	    for (pos = driver_list; pos != NULL; pos = pos->next) {
		reply(erlang_port, 'i', pos->drv->driver_name);
	    }
	    return reply(erlang_port, 'o', "");
	}
	
    default:
	driver_failure(erlang_port, 999);
    }
    return 0;
}
