%% ``The contents of this file are subject to the Erlang Public License,
%% Version 1.0, (the "License"); you may not use this file except in
%% compliance with the License. You may obtain a copy of the License at
%% http://www.erlang.org/EPL1_0.txt
%% 
%% Software distributed under the License is distributed on an "AS IS"
%% basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
%% the License for the specific language governing rights and limitations
%% under the License.
%% 
%% The Original Code is Erlang-4.7.3, December, 1998.
%% 
%% The Initial Developer of the Original Code is Ericsson Telecom
%% AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
%% Telecom AB. All Rights Reserved.
%% 
%% Contributor(s): ______________________________________.''
%%
%% Copyright (C) 1990, Ellemtel Telecommunications Systems Laboratories
%% File    : sys_pre_pj.erl
%% Author  : Robert Virding
%% Purpose : The pattern matching compiler from Simon PJ's book.

%% This module implements the algorithm for an optimizing compiler for
%% pattern matching given "The Implementation of Functional Programming
%% Languages" by Simon Peyton Jones. The code is much longer, mainly due
%% to the lack of the list comprehension mechanism present in Miranda.
%%
%% As we are using the standard Erlang "case", patterns consisting of
%% constants have not been modified into guard tests, but are matched
%% against explicitly in the "case". An explicit catch all error clause
%% is added to every "case" as we have no type checking to tell us which
%% types can be had. Repeated variables in heads are NOT handled properly,
%% and no use of any guard information is made. In each guard expression
%% a clause that explicitly generates the error case is added.
%%
%% The module contains code to do both depth first and breadth first
%% matching of variables, the difference is just 3 lines. Comment out
%% that which is not wanted.
%%
%% It is strictly Erlang source-to-source so the optimization of the
%% constructor rule has not been done; FATBAR and FAIL have no Erlang
%% counterparts.
%%
%% A function has the form:
%%
%%	{function,Line,Name,Arity,ClauseList}
%%
%% A clause has the form:
%%
%%	{clause,Line,HeadList,GuardList,BodyList}

-module(sys_pre_pj).
-copyright('Copyright (c) 1991-97 Ericsson Telecom AB').
-vsn('$Revision: /main/release/free/1').

-export([module/2,file/2]).
-import(ordsets, [new_set/0,list_to_set/1,
		  is_element/2,add_element/2,del_element/2,
		  union/1,union/2,intersection/1,intersection/2,subtract/2]).

%% file(InFile, OutFile)
%%  Test function. Do the standard compiler preprocessing first.

file(Ifile, Ofile) ->
    case epp:parse_file(Ifile, ["."], []) of
	{ok,Forms0} ->
	    case file:open(Ofile, write) of
		{ok,O} ->
		    {Mod,Exp,Forms1,Opts} = sys_pre_expand:module(Forms0, []),
		    pp_forms(O, module([{attribute,0,module,Mod},
					{attribute,0,export,Exp}|Forms1],
				       Opts)),
		    file:close(O),
		    ok;
		Error ->
		    error
	    end;
	Error ->
	    error
    end.

pp_forms(O, [F|Fs]) ->
    io:put_chars(O, [erl_pp:form(F),$\n]),
    pp_forms(O, Fs);
pp_forms(O, []) -> ok.

%% module(Forms, Options) ->
%%	MatchCompiledForms

module([{attribute,L,Attr,Val}|Fs], Opts) ->
    [{attribute,L,Attr,Val}|module(Fs, Opts)];
module([Func|Fs], Opts) ->
    [match(Func)|module(Fs, Opts)];
module([], Opts) -> [].

%% match(Function)

match({function,L,Name,Arity,Cs0}) ->
    Us = make_vars(Arity, 0, L),
    {Cs,C} = clauses(Cs0, Arity),
    Oc = match(C,Us,Cs,[{call,L,{remote,L,{atom,L,erlang},{atom,L,exit}},
			 [{atom,L,function_clause}]}]),
    {function,L,Name,Arity,[{clause,L,Us,[],Oc}]}.

%% CaseExpression = match(VarNum, Vars, Clauses, Default)
%%  Return a call list (body) containing a case call to handle matching.

match(K,[],Cs,Def) ->
	match_if(Cs,Def);
match(K,[U|Us],Cs,Def) ->
	Pcs = partition(Cs),
	foldr_match_varcon(K,[U|Us],Def,Pcs).

%% IfCall = match_if(Clauses, Default)
%%  Build a call to "if" from the remaining clauses (if necessary).

match_if([{clause,L,_,[],B}|_], Def) ->		%No guard in first clause
	B;
match_if(Cs, Def) ->
	Opts = match_if1(Cs, Def),
	[{'if',0,Opts}].

match_if1([{clause,L,_,[],B}|_], Def) ->	%Empty guard in clause
	[{clause,L,[],[],B}];
match_if1([{clause,L,_,G,B}|Cs], Def) ->
	O = {clause,L,[],G,B},
	Os = match_if1(Cs, Def),
	[O|Os];
match_if1([], no_default) -> [];
match_if1([], Def) ->
	[{clause,0,[],[],Def}].

%% PartitionList = partition(Clauses)
%%  Partition a list of clauses into groups, either all with a variable as
%%  first argument, or all with a constructor (non-variable).

partition([]) -> [];
partition([X]) -> [[X]];
partition([C1,C2|Cs]) ->
	V1 = is_var(C1),
	V2 = is_var(C2),
	if
	    V1 == V2 ->
		Pcs = partition([C2|Cs]),
		tack(C1, Pcs);
	    true ->
		Pcs = partition([C2|Cs]),
		[[C1]|Pcs]
	end.

tack(C, [Cs|Css]) ->
	[[C|Cs]|Css].

is_var({clause,L,[{var,_,_}|_],_,_}) ->
	true;
is_var(_) ->
	false.

foldr_match_varcon(A1, A2, Last, []) ->
	Last;
foldr_match_varcon(A1, A2, Last, [Cs|Css]) ->
	Rs = foldr_match_varcon(A1, A2, Last, Css),
	match_varcon(A1, A2, Cs, Rs).

%% match_varcon(VarNum, Vars, Clause, Def)

match_varcon(K, Us, [C|Cs], Def) ->
	case is_var(C) of
	    true ->
		match_var(K, Us, [C|Cs], Def);
	    false ->
		match_con(K, Us, [C|Cs], Def)
	end.

match_var(K, [U|Us], Cs, Def) ->
	Scs = subst_var(U, Cs),
	match(K, Us, Scs, Def).

%% subst_var(NewVar, Clauses)
%%  Substitute NewVar for the variable that is first in the head list of
%%  each clause. DON'T do it if variable an anonymous, "_".
%%  This is smart coded. We KNOW the basic formats of things.

subst_var(U, [{clause,L,[{var,_,'_'}|As],G,B}|Cs]) ->
    [{clause,L,As,G,B}|subst_var(U, Cs)];
subst_var(U, [{clause,L,[V|As],G0,B0}|Cs]) ->
    G = subst_var(G0, V, U),
    B = subst_var(B0, V, U),
    [{clause,L,As,G,B}|subst_var(U, Cs)];
subst_var(_, []) ->
    [].

subst_var({var,_,V},{var,_,V},New) ->
    New;
subst_var({Tag,L}, Old, New) ->
    {Tag,L};
subst_var({Tag,L,B0}, Old, New) ->
    B = subst_var(B0, Old, New),
    {Tag,L,B};
subst_var({Tag,L,B0,C0}, Old, New) ->
    B = subst_var(B0, Old, New),
    C = subst_var(C0, Old, New),
    {Tag,L,B,C};
subst_var({Tag,L,B0,C0,D0}, Old, New) ->
    B = subst_var(B0, Old, New),
    C = subst_var(C0, Old, New),
    D = subst_var(D0, Old, New),
    {Tag,L,B,C,D};
subst_var({Tag,L,B0,C0,D0,E0}, Old, New) ->
    B = subst_var(B0, Old, New),
    C = subst_var(C0, Old, New),
    D = subst_var(D0, Old, New),
    E = subst_var(E0, Old, New),
    {Tag,L,B,C,D,E};
subst_var([H|T], Old, New) ->
    [subst_var(H, Old, New)|subst_var(T, Old, New)];
subst_var([], Old, New) -> [];
subst_var(C, _, _) when constant(C) ->		%Catch all constants
    C.

%% CaseCall = match_con(VarNum, Variables, Clauses, Default)
%%  Build a call to "case" from a list of clauses all containing a
%%  constructor/constant as first argument.

match_con(K, [U|Us], Cs0, Def) ->
    Css = group_con(Cs0),
    Os = match_con1(K, [U|Us], Css, Def),
    [{'case',0,U,Os}].

match_con1(K, Us, [Cs|Css], Def) ->
	O = match_clause(K, Us, Cs, Def),
	Os = match_con1(K, Us, Css, Def),
	[O|Os];
match_con1(K, Us, [], no_default) -> [];
match_con1(K, Us, [], Def) ->
	[{clause,0,[{var,0,'_'}], [], Def}].

%% ClauseGroupList = group_con(Clauses)
%%  Group clauses after constructor in first variable.

group_con([C|Cs0]) ->
	Type = select_type(C),
	{More,Cs1} = select(Type, Cs0),		%Select (remove) same clauses
	Cs = group_con(Cs1),
	[[C|More]|Cs];
group_con([]) ->
	[].

select_type({clause,_,[{tuple,_,Args}|_],_,_}) ->
    Arity = length(Args),
    {tuple,Arity};
select_type({clause,_,[{cons,_,_,_}|_],_,_}) ->
    cons;
select_type({clause,_,[{nil,_}|_],_,_}) ->
    nil;
select_type({clause,_,[{atom,_,A}|_],_,_}) ->
    {atom,A};
select_type({clause,_,[{float,_,F}|_],_,_}) ->
    {float,F};
select_type({clause,_,[{integer,_,I}|_],_,_}) ->
    {integer,I}.

select(Type, [C|Cs0]) ->
    {More,Cs} = select(Type, Cs0),
    case select_type(C) of
	Type -> {[C|More],Cs};
	Other -> {More,[C|Cs]}
    end;
select(_, []) ->
    {[],[]}.

%% CaseClause = match_clause(VarNum, Variables, Clauses, Default)
%%  Build a "case" clause from a list of clauses all starting with the
%%  same constructor/constant.

match_clause(K, Us, Cs, Def) ->
    match_clause(get_con(Cs), K, Us, Cs, Def).

match_clause(Con, K, [_|Us], Cs, Def) ->
    {Match,Vs,Arity,Guards} = get_match(Con, K),
    K1 = K + Arity,
    Us1 = combine_args(Vs, Us),
    Cs1 = new_clauses(Cs),
    Oc = match(K1, Us1, Cs1, Def),
    {clause,0,[Match],Guards,Oc}.

%% NewArgs = combine_args(ConstructorArgs, RestArgs)
%%  Combine the variables/arguments from the constructor with the remaining
%%  clause arguments. How the arguments are combined gives the match ordering.

combine_args(ConArgs, RestArgs) ->
    append(ConArgs, RestArgs).		%Depth first matching
%%append(RestArgs, ConArgs).		%Breadth first matching

get_match({cons,L,_,_}, K) ->
    Mas = [H,T] = make_vars(2, K, L),
    {{cons,L,H,T},Mas,2,[]};
get_match({tuple,L,Args}, K) ->
    Arity = length(Args),
    Mas = make_vars(Arity, K, L),
    {{tuple,L,Mas},Mas,Arity,[]};
get_match(M, _) ->
    {M,[],0,[]}.

get_con([{clause,L,[Con|_],_,_}|_]) ->
    Con.

new_clauses([C0|Cs]) ->
    C = new_clause(C0),
    [C|new_clauses(Cs)];
new_clauses([]) ->
    [].

new_clause({clause,L,[{cons,_,H,T}|Rest],G,B}) ->
    New = combine_args([H,T], Rest),
    {clause,L,New,G,B};
new_clause({clause,L,[{tuple,_,Args}|Rest],G,B}) ->
    New = combine_args(Args, Rest),
    {clause,L,New,G,B};
new_clause({clause,L,[_|Rest],G,B}) ->
    {clause,L,Rest,G,B}.

%% make_vars(Count, FirstVar, Line)
%%  Make a list of Count variables starting from FirstVar.

make_vars(C, Next, L) when C > 0 ->
    V = list_to_atom("__" ++ integer_to_list(Next) ++ "__"),
    [{var,L,V}|make_vars(C-1, Next+1, L)];
make_vars(_, _, L) ->
    [].

%%
%% Miscellaneous utilities.
%%

append([H|T], X) ->
	[H|append(T, X)];
append([], X) -> X.

append([L]) -> L;
append([L|Ls]) ->
    append(L, append(Ls));
append([]) -> [].

%% clauses(Clauses)

clauses([{clause,L,H0,G0,B0}|Cs0], C0) ->
    {H,G1,Vs,C1} = head(H0, [], [], C0),
%    B = body(B0),
    {Cs,C} = clauses(Cs0, C1),
    {[{clause,L,H,append(G1, G0),B0}|Cs],C};
clauses([], C) ->
    {[],C}.

%% head(Head, Guard, Variables, Counter)
%%  Remove all repeated occurences of variables and replace with explicit
%%  equality test in guard.

head([A0|As0], G0, Vs0, C0) ->
    {A,G1,Vs1,C1} = arg(A0, G0, Vs0, C0),
    {As,G,Vs,C} = head(As0, G1, Vs1, C1),
    {[A|As],G,Vs,C};
head([], G, Vs,C) ->
    {[],G,Vs,C}.

arg({var,L,'_'}, G, Vs,C) ->
    {{var,L,C},G,Vs,C+1};
arg({var,L,V}, G, Vs, C) ->
    case is_element(V, Vs) of
	true ->
	    %% Don't bother to add new temporary.
	    {{var,L,C},[{op,L,'=:=',{var,L,C},{var,L,V}}|G],Vs,C+1};
	false ->
	    {{var,L,V},G,add_element(V, Vs),C}
    end;
arg({cons,L,H0,T0}, G0, Vs0, C0) ->
    {H,G1,Vs1,C1} = arg(H0, G0, Vs0, C0),
    {T,G,Vs,C} = arg(T0, G1, Vs1, C1),
    {{cons,L,H,T},G,Vs,C};
arg({tuple,L,Es0}, G0, Vs0, C0) ->
    {Es,G,Vs,C} = arg_list(Es0, G0, Vs0, C0),
    {{tuple,L,Es},G,Vs,C};
arg({string,L,S}, G, Vs, C) ->
    {string_to_cons(S,L),G,Vs,C};
arg(A, G, Vs, C) ->
    {A,G,Vs,C}.

arg_list([A0|As0], G0, Vs0, C0) ->
    {A,G1,Vs1,C1} = arg(A0, G0, Vs0, C0),
    {As,G,Vs,C} = arg_list(As0, G1, Vs1, C1),
    {[A|As],G,Vs,C};
arg_list([], G, Vs, C) ->
    {[],G,Vs,C}.

%% string_to_cons(StringChars)

string_to_cons([C|Cs],L) ->
    {cons,L,{number,L,C},string_to_cons(Cs,L)};
string_to_cons([],L) ->
    {nil,L}.
