%% ``The contents of this file are subject to the Erlang Public License,
%% Version 1.0, (the "License"); you may not use this file except in
%% compliance with the License. You may obtain a copy of the License at
%% http://www.erlang.org/EPL1_0.txt
%% 
%% Software distributed under the License is distributed on an "AS IS"
%% basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
%% the License for the specific language governing rights and limitations
%% under the License.
%% 
%% The Original Code is Erlang-4.7.3, December, 1998.
%% 
%% The Initial Developer of the Original Code is Ericsson Telecom
%% AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
%% Telecom AB. All Rights Reserved.
%% 
%% Contributor(s): ______________________________________.''
%%
-module(erl_mkbif).
-copyright('Copyright (c) 1991-97 Ericsson Telecom AB').
-vsn('$Revision: /main/release/free/1').
-author('bjorn@erix.ericsson.se').

%% Purpose: Creates the sys_bifs.erl module given the bif.tab file.

-export([file/2, file/3]).

-include("erl_compile.hrl").

file(Input, Output) ->
    file(Input, Output, #options{}).

file(Input, Output, Options) ->
    {ok, Fd} = file:open(Input++".tab", read),
    List = process(Fd, []),
    file:close(Fd),
    {ok, Out} = file:open(Output++".erl", write),
    make(Out, List, filename:basename(Output)),
    ok.

%% Processes input file.

process(Fd, Result) ->
    case scan(io:get_line(Fd, none), [], []) of
	eof ->
	    case Result of
		error ->
		    error;
		List ->
		    lists:reverse(Result)
	    end;
	[] ->
	    process(Fd, Result);
	List ->
	    case parse(List) of
		{ok, Tuple} ->
		    process(Fd, [Tuple|Result]);
		error ->
		    process(Fd, error)
	    end
    end.

%% Splits input line into fields.

scan(eof, Token, Result) ->
    eof;
scan([$#|_], Token, Result) ->
    [];
scan([C|Rest], [], Result) when C >= 0, C =< $  ->
    scan(Rest, [], Result);
scan([C|Rest], Token, Result) when C >= 0, C =< $  ->
    scan(Rest, [], [lists:reverse(Token)|Result]);
scan([C|Rest], Token, Result) ->
    scan(Rest, [C|Token], Result);
scan([], [C|Rest], Result) ->
    scan([], [], [[C|Rest]|Result]);
scan([], [], Result) ->
    lists:reverse(Result).


%% Parses field list.

parse([Func, Arity, AutoImport]) ->
    parse(list_to_atom(Func), catch list_to_integer(Arity),
	  list_to_atom(AutoImport));
parse(_) ->
    io:format("Illegal number of fields.~n").

parse(Func, {'EXIT', _}, AutoImport) ->
    io:format("Bad arity.~n"),
    error;
parse(Func, Arity, true) ->
    {ok, {Func, Arity, true}};
parse(Func, Arity, false) ->
    {ok, {Func, Arity, false}};
parse(Func, Arity, Other) ->
    io:format("Bad auto-import field: ~p.~n", [Other]),
    error.

%% Makes output file.

make(Fd, List, Name) ->
    io:format(Fd, "% Automatically generated from bif.tab -- don't edit.~n", []),
    io:format(Fd, "-module(~s).~n", [Name]),
    io:format(Fd, "-export([bif/2]).~n~n", []),
    make1(Fd, List, 0),
    io:format(Fd, "bif(_, _) -> false.~n", []).

make1(Fd, [{Func, Arity, AutoImport}|Rest], Number) ->
    io:format(Fd, "bif(~p, ~p) -> {~p, ~p};~n",
	      [Func, Arity, Number, AutoImport]),
    make1(Fd, Rest, Number+1);
make1(_, [], _) ->
    ok.
