%% ``The contents of this file are subject to the Erlang Public License,
%% Version 1.0, (the "License"); you may not use this file except in
%% compliance with the License. You may obtain a copy of the License at
%% http://www.erlang.org/EPL1_0.txt
%% 
%% Software distributed under the License is distributed on an "AS IS"
%% basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
%% the License for the specific language governing rights and limitations
%% under the License.
%% 
%% The Original Code is Erlang-4.7.3, December, 1998.
%% 
%% The Initial Developer of the Original Code is Ericsson Telecom
%% AB. Portions created by Ericsson are Copyright (C), 1998, Ericsson
%% Telecom AB. All Rights Reserved.
%% 
%% Contributor(s): ______________________________________.''
%%
%%% File    : beam_cleanup.erl
%%% Author  : Bjorn Gustavsson <bjorn@strider>
%%% Purpose : Cleanups output from compiler before presenting it for the assembler.
%%% Created : 18 Mar 1998 by Bjorn Gustavsson <bjorn@strider>

-module(beam_cleanup).
-author('bjorn@strider').

-export([module/3]).

-record(state,
	{module,				% Module name (atom).
	 label,					% Next free label.
	 func_info}).				% Label for func_info instruction.

module(Mod, List, Exports) ->
    St = #state{module = Mod, label = get(c_next_label)},
    module(List, [], St, Exports).

module([H|T], Acc, St0, Exports) ->
    {Code, St} = function(H, St0),
    module(T, [Code|Acc], St, Exports);
module([], Code, #state{module=Mod, label=Lcount}, Exports) ->
    {Mod, Exports, Code, Lcount}.

function({code, Mod, Name, Arity, Type, [Label| Code]}, St0) ->
    {Entry, Prefix, St1} = cleanup_label(Label, St0),
    {Code1, St2} = cleanup(Code, Prefix, St1),
    {{function, Name, Arity, Entry, Code1}, St2}.

cleanup_label({label, Name, Arity, Entry, E}, St0) ->
    Mod = St0#state.module,
    {Label, St1} = next_label(St0),
    Prefix = [{label, Entry}, {func_info, {atom, Mod}, {atom, Name}, Arity},
	      {label, Label}],
    {Entry, Prefix, St1#state{func_info = Label}}.

%%
%% Cleanup code to make it acceptable to assembler.
%%

%% Add labels instead of various end constructs.

cleanup([{label, Lbl}| T], Acc, St) ->		% Added by build_receive.
    cleanup(T, add_label(Lbl, Acc), St);

cleanup([{clause_end, L}|T], Acc, St) ->
    cleanup(T, add_label(L, Acc), St);

cleanup([{commonCl_end, L}|T], Acc, St) ->
    cleanup(T, add_label(L, Acc), St);

cleanup([{commonTestPattern_end, L}|T], Acc, St) ->
    cleanup(T, add_label(L, Acc), St);

cleanup([{testPattern_end, Lb1, 0}|T], Acc, St) ->
    cleanup(T, add_if_not_dead({jump, {f, Lb1}}, Acc), St);

cleanup([{testPattern_end, Lb1, Lb2}|T], Acc, St) ->
    cleanup(T, add_label(Lb2, add_if_not_dead({jump, {f, Lb1}}, Acc)), St);

cleanup([{receiveEnd, Lb}|T], Acc, St) ->
    cleanup(T, add_label(Lb, Acc), St);

cleanup([{fail, Const, W}|T], Acc, St) ->
    cleanup(T, [cleanup_fail(Const, W, St)|Acc], St);

cleanup([{'catch', {catchEnd, Y, Lb}}|T], Acc, St) ->
    cleanup(T, [{catch_end, Y}| add_label(Lb, Acc)], St);

cleanup([{caseEnd, Lb, _, R}|T], Acc, St) ->
    Ce = {case_end, {p, func_info(St)}, R},
    cleanup(T, add_label(Lb, add_if_not_dead(Ce, Acc)), St);

cleanup([{ifEnd, Lb, _}|T], Acc, St) ->
    IfEnd = {if_end, {p, func_info(St)}},
    cleanup(T, add_label(Lb, add_if_not_dead(IfEnd, Acc)), St);

cleanup([{switchlabel, N}|T], Acc, St) ->
    cleanup(T, add_label(N, Acc), St);

cleanup([{'receive', Lb, Y}|T], Acc, St0) ->
    {Code, St} = build_receive(T, Lb, Y, [], St0),
    cleanup(Code, ['receive'| add_label(Lb, Acc)], St);

%% Remove dead code (everything after, for instance, 'jump' up to the next label).

cleanup([H|T], [DeadAfter| Acc], St) when element(1, DeadAfter) == jump ->
    cleanup(T, [DeadAfter| Acc], St);

cleanup([H|T], [DeadAfter| Acc], St) when element(1, DeadAfter) == badmatch ->
    cleanup(T, [DeadAfter| Acc], St);

cleanup([H|T], [DeadAfter| Acc], St) when element(1, DeadAfter) == call_last ->
    cleanup(T, [DeadAfter| Acc], St);

cleanup([H|T], [DeadAfter| Acc], St) when element(1, DeadAfter) == call_only ->
    cleanup(T, [DeadAfter| Acc], St);

cleanup([H|T], [DeadAfter| Acc], St) when element(1, DeadAfter) == callExLast ->
    cleanup(T, [DeadAfter| Acc], St);

cleanup([H|T], [DeadAfter| Acc], St) when element(1, DeadAfter) == return ->
    cleanup(T, [DeadAfter| Acc], St);

%% Cleanup other stuff.

cleanup([{allocate, Need, Regs}| T], Acc, St) ->
    cleanup_allocation(T, {allocate, Need, Regs}, Need, [], Acc, St);

cleanup([{allocateH, Need, Regs, Heap}|T], Acc, St) ->
    cleanup_allocation(T, {allocate_heap, Need, Regs, Heap}, Need, [], Acc, St);

cleanup([{init, Y}|T], Acc, St) ->
    cleanup(T, [{init, {y, Y}}|Acc], St);

cleanup([{testHeap, Need, Alive}| T], Acc, St) ->
    cleanup(T, [{test_heap, Need, Alive}| Acc], St);

cleanup([{call, Label, _, _, _}|T], Acc, St) ->
    cleanup(T, [{call, {p, Label}}|Acc], St);

cleanup([{callOnly, Label, _, _, _}|T], Acc, St) ->
    cleanup(T, [{call_only, {p, Label}}|Acc], St);

cleanup([{callLast, Label, Arity, Deallocate, _, _}|T], Acc, St) ->
    cleanup(T, [{call_last, {p, Label}, Deallocate}|Acc], St);

cleanup([{return}|T], Acc, St) ->
    cleanup(T, [return|Acc], St);

cleanup([removeMessage|T], Acc, St) ->
    cleanup(T, [remove_message|Acc], St);

cleanup([{test, Test, W, _}|T], Acc, St) ->
    cleanup(T, [cleanup_test(Test, W, St)|Acc], St);

cleanup([{bif, {Bif, [], Dest}, _, _}|T], Acc, St) ->
    Args = [Dest],
    cleanup(T, [{bif, Bif, Args, 0}|Acc], St);

cleanup([{bif, {Bif, Args0, Dest}, W, _}|T], Acc, St) ->
    Args = [tag_label(W, St)|Args0++[Dest]],
    cleanup(T, [{bif, Bif, Args, length(Args0)}|Acc], St);

cleanup([{bif_gc, {send, Args0}, W, _}|T], Acc, St) ->
    Args = [tag_label(W, St)|Args0],
    cleanup(T, [{bif, send, Args, length(Args0)}|Acc], St);

cleanup([{bif_gc, {Bif, [], {x, 0}}, W, _}|T], Acc, St) ->
    Args = [],
    cleanup(T, [{bif, Bif, [], 0}|Acc], St);

cleanup([{bif_gc, {Bif, Args0, {x, 0}}, W, _}|T], Acc, St) ->
    Args = [tag_label(W, St)|Args0],
    cleanup(T, [{bif, Bif, Args, length(Args0)}|Acc], St);

cleanup([{get, {getList2, R1, R2, R3}}| T], Acc, St) ->
    cleanup(T, [{get_list, R1, R2, R3}| Acc], St);

cleanup([{get, {getTupleElement, Src, Dest, Element}}| T], Acc, St) ->
    cleanup(T, [{get_tuple_element, Src, Element, Dest}| Acc], St);

cleanup([{put, {putTuple, Dest, Arity, Value}}| T], Acc, St) ->
    cleanup(T, [{put, Value}, {put_tuple, {arity, Arity}, Dest}| Acc], St);

cleanup([{put, Put}|T], Acc, St) ->
    cleanup(T, [cleanup_put(Put)| Acc], St);

cleanup([{'catch', {'catch', Y, Lb}}|T], Acc, St) ->
    cleanup(T, [{'catch', Y, {p, Lb}}|Acc], St);

cleanup([{error_func_clause, _}|T], Acc, St) ->
    cleanup(T, [{function_clause_error, {p, func_info(St)}}|Acc], St);

cleanup([{arith, Arith, W, _}|T], Acc, St) ->
    cleanup(T, [{arith, Arith, tag_label(W, St)}|Acc], St);

cleanup(['if'|T], Acc, St) ->
    cleanup(T, Acc, St);

cleanup([H|T], Acc, St) ->
    case garbage(H) of
	true -> cleanup(T, Acc, St);
	false -> cleanup(T, [H|Acc], St)
    end;

cleanup([], Acc, St) ->
    {lists:reverse(Acc), St}.

cleanup_put({putTupleElement, Src}) ->
    {put, Src};
cleanup_put({putList2, Dest, H, T}) ->
    {put_list, H, T, Dest};
cleanup_put({putString, Dest, _, Str}) ->
    {put_string, length(Str), {string, Str}, Dest};
cleanup_put({putFloat, Dest, Float}) ->
    {B1, B2} = float_to_ieee(Float),
    {put_float, B1, B2, Dest};
cleanup_put({putInt, Dest, Arity, Sign}) ->
    {put_bignum, {bignum, Arity, Sign}, Dest};
cleanup_put({putIntVal, Val}) ->
    {put_bigval, {bignum_part, Val}}.

float_to_ieee(F) when float(F) ->
    case erlang:float_to_words(1.0) of
	{0, _} ->				% Little-endian.
	    {B1, B2} = erlang:float_to_words(F),
	    {B2, B1};
	_ ->					% Big-endian.
	    erlang:float_to_words(F)
    end.

%% Cleanup test instructions.

cleanup_test({test, tuple, R}, W, St) ->
    {is_tuple, tag_label(W, St), R};
cleanup_test({test, binary, R}, W, St) ->
    {is_binary, tag_label(W, St), R};
cleanup_test({test, nil, R}, W, St) ->
    {is_nil, tag_label(W, St), R};
cleanup_test({test, constant, R}, W, St) ->
    {is_constant, tag_label(W, St), R};
cleanup_test({test, atom, R}, W, St) ->
    {is_atom, tag_label(W, St), R};
cleanup_test({test, number, R}, W, St) ->
    {is_number, tag_label(W, St), R};
cleanup_test({test, integer, R}, W, St) ->
    {is_integer, tag_label(W, St), R};
cleanup_test({test, float, R}, W, St) ->
    {is_float, tag_label(W, St), R};
cleanup_test({test, pid, R}, W, St) ->
    {is_pid, tag_label(W, St), R};
cleanup_test({test, reference, R}, W, St) ->
    {is_ref, tag_label(W, St), R};
cleanup_test({test, port, R}, W, St) ->
    {is_port, tag_label(W, St), R};
cleanup_test({test, list, R}, W, St) ->
    {is_list, tag_label(W, St), R};
cleanup_test({test, nonEmptyList, R}, W, St) ->
    {is_nonempty_list, tag_label(W, St), R};
cleanup_test({equal_atom, R1, Atom}, W, St) ->
    {is_eq_exact, tag_label(W, St), R1, Atom};
cleanup_test({equal_int, R1, Int}, W, St) ->
    {is_eq_exact, tag_label(W, St), R1, {i, Int}};
cleanup_test({equal, _, R1, R2}, W, St) ->
    {is_eq_exact, tag_label(W, St), R1, R2};
cleanup_test({nEqual, _, R1, R2}, W, St) ->
    {is_ne_exact, tag_label(W, St), R1, R2};
cleanup_test({testTuple, R1, R2}, W, St) ->
    {is_tuple_of_arity, tag_label(W, St), R1, R2};
cleanup_test({testTupleArity, R1, R2}, W, St) ->
    {test_arity, tag_label(W, St), R1, R2};
cleanup_test({Test, R1, R2}, W, St) ->
    {Test, tag_label(W, St), R1, R2};
cleanup_test({intComp, Op, R1, R2}, W, St) ->
    {cleanup_comp(Op), tag_label(W, St), R1, R2};
cleanup_test({comp, Op, R1, R2}, W, St) ->
    {cleanup_comp(Op), tag_label(W, St), R1, R2}.

cleanup_comp('>')   -> is_gt;
cleanup_comp('<')   -> is_lt;
cleanup_comp('>=')  -> is_ge;
cleanup_comp('=<')  -> is_le;
cleanup_comp('==')  -> is_eq;
cleanup_comp('/=')  -> is_ne;
cleanup_comp('=:=') -> is_eq_exact;
cleanup_comp('=/=') -> is_ne_exact.
    

cleanup_fail(Const, {head, L}, _) -> {jump, {f, L}};
cleanup_fail(Const, {head_case, L}, _) -> {jump, {f, L}};
cleanup_fail(Const, body, St) -> {badmatch, Const, {p, func_info(St)}};
cleanup_fail(Const, body_case, St) -> {badmatch, Const, {p, func_info(St)}}.

tag_label({head, L}, _) -> {f, L};
tag_label({head_case, L}, _) -> {f, L};
tag_label(body, St) -> {p, func_info(St)};
tag_label(body_case, St) -> {p, func_info(St)}.

add_label(0, Acc) ->
    Acc;
add_label(Lbl, [{jump, {f, Lbl}}| Acc]) ->
    [{label, Lbl}| Acc];
add_label(Lbl, Acc) ->
    [{label, Lbl}| Acc].

%% Add an instruction only if it can be reached.

add_if_not_dead(_, [return|Acc]) ->
    [return| Acc];
add_if_not_dead(_, [CallExLast|Acc]) when element(1, CallExLast) == callExLast ->
    [CallExLast| Acc];
add_if_not_dead(_, [CallLast|Acc]) when element(1, CallLast) == call_last ->
    [CallLast| Acc];
add_if_not_dead(_, [CallOnly|Acc]) when element(1, CallOnly) == call_only ->
    [CallOnly| Acc];
add_if_not_dead(_, [Jump|Acc]) when element(1, Jump) == jump ->
    [Jump| Acc];
add_if_not_dead(Instr, Acc) ->
    [Instr| Acc].


cleanup_allocation([{init, Y}| T], Alloc, Need, Inits, Acc, St) ->
    cleanup_allocation(T, Alloc, Need, [{init, {y, Y}}| Inits], Acc, St);
cleanup_allocation(T, Alloc, Need, [], Acc, St) ->
    cleanup(T, [Alloc| Acc], St);
cleanup_allocation(T, Alloc, Need, Inits, Acc, St) when Need > length(Inits)*2 ->
    cleanup(T, Inits ++ [Alloc| Acc], St);
cleanup_allocation(T, Alloc, Need, Inits, Acc, St) ->
    cleanup(T, [zero_alloc(Alloc)| Acc], St).

zero_alloc({allocate, Need, Regs}) ->    
    {allocate_zero, Need, Regs};
zero_alloc({allocate_heap, Need, Regs, Heap}) ->
    {allocate_heap_zero, Need, Regs, Heap}.

%% Returns true if a {Garbage, ...} tuple is not useful for the assembler.

garbage(Tuple) when tuple(Tuple) ->
    garbage1(element(1, Tuple));
garbage(_) ->
    false.

garbage1(clause_header) -> true;
garbage1(commonCl) -> true;
garbage1(testPattern) -> true;
garbage1(commonTestPattern) -> true;
garbage1(waitTimeOutEnd) -> true;
garbage1(receiveEnd) -> true;
garbage1('case') -> true;
garbage1(_) -> false.

%% Generates a receive statement.  The skeleton looks like this:
%%
%%      L0:           <-------------------+
%%     	       	receive
%%      L1:           <-----------+   	  |
%%              loop_rec L2 ------+---+   |
%%              ...               |   |   |
%%              ...	          |   |   |
%%		...	          |   |   |
%%		Loop_rec_end L1 --+   |   |
%%      L2:           <---------------+   |
%%	   	wait L0  -----------------+	%% or wait_timeout
%%		timeout

build_receive([{wait, Lb, W}|T], Lb, Y, Acc, St) ->
    {Lb2, St1} = next_label(St),
    {Lb3, St2} = next_label(St1),
    LoopRec = {loop_rec, {p, Lb3}, {y, Y}},
    LoopRecEnd = {loop_rec_end, {p, Lb2}},
    Wait = {wait, {p, Lb}, {p, func_info(St2)}},
    End = lists:reverse(Acc, [LoopRecEnd, {label, Lb3}, Wait, timeout|T]),
    Code = [{label, Lb2}, LoopRec | End],
    {Code, St2};
build_receive([{waitTimeOut, Lb, Time, W}|T], Lb, Y, Acc, St) ->
    {Lb2, St1} = next_label(St),
    {Lb3, St2} = next_label(St1),
    LoopRec = {loop_rec, {p, Lb3}, {y, Y}},
    LoopRecEnd = {loop_rec_end, {p, Lb2}},
    Wait = {wait_timeout, {p, Lb}, {p, func_info(St2)}, Time},
    End = lists:reverse(Acc, [LoopRecEnd, {label, Lb3}, Wait, timeout|T]),
    Code = [{label, Lb2}, LoopRec | End],
    {Code, St2};
build_receive([H|T], Lb, Y, Acc, St) ->
    build_receive(T, Lb, Y, [H|Acc], St).

next_label(St0) ->
    Label = St0#state.label,
    {Label, St0#state{label = Label+1}}.

func_info(St) ->
    St#state.func_info.
