/* 
   filedialog.c [2000-10-31]
   (c) 2000 by Dieter Mittelmaier <dieter.mittelmaier@freenet.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <dirent.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <SDL.h>

#include "button.h"
#include "drawutil.h"
#include "listbox.h"


#define OK				1
#define CANCEL			2

#define ARRAY_SIZE		100

#define REGULAR_FILE	0
#define DIRECTORY		1

#define FILENAME_SIZE	256

typedef struct _FileItem {
	int isDir;
	char name[FILENAME_SIZE];
}FILEITEM;

typedef struct _Label {
	SDL_Surface *area;
	SDL_Rect rect;
	char *text;
	Uint32 backcolor;
}LABEL;


SDL_Rect dialog;
static SDL_Surface *backingstore = NULL;

int file_count;
int array_size;
char separator[] = {"/"};
char fullpath[1024];

FILEITEM **filelist;

LISTBOX *lb;
LABEL pathLabel;
BUTTON *okButton, *cancelButton;




void freeFilelist()
{
	int i;

	if (filelist == NULL)
		return;
	
	for ( i = 0; i < file_count; i++)
		free(filelist[i]);
	
	free(filelist);
	filelist = NULL;
}

int dirisReadable(char *path)
{
	if(access(path,R_OK) == -1)
		return 0;

	return 1;
}

int isrootDir(char *path)
{
	if (strcmp(path, "/"))
		return 0;

	return 1;
}

int fileInteresting(char *file)
{
	struct stat st;
	
	if (lstat(file,&st) < 0) return -1;
	
	if (S_ISDIR(st.st_mode)) {
		return DIRECTORY;
	} else if (S_ISLNK(st.st_mode)) {
		struct stat st1;
		if (stat(file,&st1) < 0) {
			//stalledLink
			return -1;
		} else {
			if (S_ISDIR(st1.st_mode)) {
				return DIRECTORY;
			} else {
				return REGULAR_FILE;
			}
		}
	} else if (S_ISREG(st.st_mode)) {
		return REGULAR_FILE;
	}

	return -1;
}

void inSort(FILEITEM *fi)
{
	int i;
	int ready = 0;
	for ( i = 0; i < file_count; i++) {
		if (fi->isDir) {
			if (filelist[i]->isDir) {
				if (strcmp(fi->name, filelist[i]->name) < 0) {
					ready = i;
					break;
				}
			} else {
				ready = i;
				break;
			}
		} else {
			if (!filelist[i]->isDir) {
				if (strcmp(fi->name, filelist[i]->name) < 0) {
					ready = i;
					break;
				}
			}
		}
	}
		
	if(!ready) {
		filelist[file_count] = fi;
		return;
	}
	
	for ( i = file_count; i > ready; i--)
		filelist[i] = filelist[i - 1];

	filelist[ready] = fi;
}

int readDirectory(char *path)
{
	DIR *dir;
	FILEITEM *fi;
	struct dirent *entry;
	char tmp[1024];
	int filetype;

	file_count = 0;
	array_size = ARRAY_SIZE;
	
	filelist = (FILEITEM **)malloc(array_size * sizeof(FILEITEM *));
	if (filelist == NULL) {
		printf("Malloc FileArray failed\n");
		return 0;
	}
	
	if (!dirisReadable(path)) {
		filelist[file_count] = (FILEITEM *)malloc(sizeof(FILEITEM));
		if (filelist[file_count] == NULL) {
			freeFilelist();
			printf("Malloc FileItem failed\n");
			return 0;
		}
		filelist[file_count]->isDir = 1;
		memset(filelist[file_count]->name, 0, FILENAME_SIZE);
		strcpy(filelist[file_count]->name, "..");
		file_count = 1;
		filelist[file_count] = NULL;
		return 1;
	}

	dir = opendir(path);
	if (!dir) {
		freeFilelist();
		return 0;
	}
	
	while ((entry = readdir(dir))) {
		if (strcmp(entry->d_name, ".") == 0)
			continue;
		if ((strcmp(entry->d_name, "..") != 0) && (entry->d_name[0] == '.'))
			continue;

		strcpy(tmp,path);
		if (!isrootDir(path))
			strcat(tmp, "/");
		strcat(tmp, entry->d_name);
		
		filetype = fileInteresting(tmp);
		if (filetype < 0)
			continue;
			
		if (file_count == array_size) {
			FILEITEM **newfiles;
			array_size += ARRAY_SIZE;
			newfiles = (FILEITEM **)realloc(filelist, array_size * sizeof(FILEITEM *));
			if (newfiles == NULL) {
				printf("Realloc FileArray failed\n");
				freeFilelist();
				closedir(dir);
				return 0;
			}
			filelist = newfiles;
		}

		filelist[file_count] = NULL;
		fi = (FILEITEM *)malloc(sizeof(FILEITEM));
		if (fi == NULL) {
			printf("Malloc FileItem failed\n");
			freeFilelist();
			closedir(dir);
			return 0;
		}

		strcpy(fi->name, entry->d_name);
		fi->isDir = filetype;
		inSort(fi);

		file_count++;
	}

	closedir(dir);
	return 1;
}

int changeDir(char *path)
{
	char *tmp = NULL;
	char last[256];
	int i;
	int focusitem = ListBoxFocusItem(lb);
	
	memset(last, 0, 256);
	if ((strcmp(filelist[focusitem]->name, "..")) == 0) {
		if (!isrootDir(path)) {
			if ((tmp = strrchr(path, '/')) != NULL) {
				*tmp = '\0';
				++tmp;
				strcpy(last, tmp);
			}
			if (strlen(path) == 0)
				strcpy(path, "/");
		} else {
			return 1;
		}
	} else {
		if (!isrootDir(path))
			strcat(path, "/");
		strcat(path,filelist[focusitem]->name);
	}

	cleanupListBox(lb);
	freeFilelist();

	if (!readDirectory(path)) {
		printf("Read Directory failed\n");
		return 0;
	}

	for (i = 0; i < file_count; i++) {
		if (filelist[i]->isDir)
			addListBoxItem(lb, separator, filelist[i]->name);
		else
			addListBoxItem(lb, filelist[i]->name, NULL);
	}

	if (strlen(last) > 0) {
		int i;
		for (i = 0; i < file_count; i++) {
			if (strcmp(filelist[i]->name, last) == 0) {
				setListBoxFocusItem(lb, i);
				break;
			}
		}
	}
	
	return 1;
}

void updateLabel(SDL_Surface *surface, LABEL *l)
{
	SDL_Rect src;
	SDL_Rect dst;
	int maxlen;
	int pathlen;
	char tmp[1024];

	src.x = 0;
	src.y = 0;
	src.w = l->area->w;
	src.h = l->area->h;

	pathlen = strlen(l->text);
	maxlen = src.w / charWidth();
	if (pathlen > maxlen) {
		strcpy(tmp, "...");
		strcat(tmp, &l->text[pathlen - maxlen + 3]);
	} else {
		strncpy(tmp, l->text, 1024);
	}

	SDL_FillRect(l->area, &src, l->backcolor);

	drawText(l->area, src.x, src.y, tmp);

	dst.x = l->rect.x;
	dst.y = l->rect.y;
	dst.w = l->rect.w;
	dst.h = l->rect.h;

	SDL_BlitSurface(l->area, &src, surface, &dst);
}

void createFiledialog(SDL_Surface *surface, char *path)
{
	SDL_Rect label;
	SDL_Rect buttonarea;
	SDL_Rect button;

	int w = 300;
	int h = 300;
	int margin = 4;
	int i;
	
	if (surface->w < w)
		w = surface->w;
	if (surface->h < h)
		h = surface->h;

	dialog.x = (surface->w - w) / 2;
	dialog.y = (surface->h - h) / 2;
	dialog.w = w;
	dialog.h = h;

	backingstore = SDL_CreateRGBSurface(SDL_SWSURFACE, dialog.w, dialog.h,
								surface->format->BitsPerPixel,
								surface->format->Rmask, surface->format->Gmask,
								surface->format->Bmask, surface->format->Amask);

	if (backingstore)
		SDL_BlitSurface(surface, &dialog, backingstore, NULL);

	label.x = dialog.x + 10;
	label.y = dialog.y + 10;
	label.w = dialog.w - 20;
	label.h = charHeight() + 4 + margin * 2;

	pathLabel.rect.x = label.x + 2 + margin;
	pathLabel.rect.y = label.y + 2 + margin;
	pathLabel.rect.w = label.w - 4 - margin * 2;
	pathLabel.rect.h = label.h - 4 - margin * 2;
	pathLabel.text = path;

	buttonarea.x = dialog.x + 10;
	buttonarea.y = dialog.y + dialog.h - 1 - 20 - label.h;
	buttonarea.w = dialog.w - 20;
	buttonarea.h = dialog.y + dialog.h - buttonarea.y;

	button.x = buttonarea.x;
	button.y = buttonarea.y + 10;
	button.w = (strlen("Cancel") + 2) * charWidth();
	button.h = label.h;
	okButton = Button(&button, OK);
	setButtonText(okButton, "OK");

	button.x += button.w + 10;
	cancelButton = Button(&button, CANCEL);
	setButtonText(cancelButton, "Cancel");

	lb = ListBox(label.x, label.y + label.h + 10,
					label.w, buttonarea.y - (label.y + label.h + 10), margin, 1);

	for (i = 0; i < file_count; i++) {
		if (filelist[i]->isDir)
			addListBoxItem(lb, separator, filelist[i]->name);
		else
			addListBoxItem(lb, filelist[i]->name, NULL);
	}

	initListBox(lb, surface);

	pathLabel.area = SDL_CreateRGBSurface(SDL_SWSURFACE, pathLabel.rect.w, pathLabel.rect.h,
									surface->format->BitsPerPixel,
									surface->format->Rmask, surface->format->Gmask,
									surface->format->Bmask, surface->format->Amask);

	pathLabel.backcolor = SDL_MapRGB(pathLabel.area->format, 204, 204, 204);

	drawRaisedWin(surface, &dialog, light_color, shadow_color, back_color);
	drawSunkenWin(surface, &label, light_color, shadow_color, pathLabel.backcolor);
	updateLabel(surface, &pathLabel);
	drawListBox(lb, surface);
	drawButton(surface, okButton);
	drawButton(surface, cancelButton);

    SDL_UpdateRect(surface,dialog.x,dialog.y,dialog.w,dialog.h);	
}

char* openFile(SDL_Surface *surface, char *path)
{
	int done = 0;
	int cancel = 0;
	okButton = NULL;
	cancelButton = NULL;
	
	if(strlen(path) == 0)
		strcpy(path, getenv("HOME"));

	if (!readDirectory(path)) {
		printf("Read Directory failed\n");
		return NULL;
	}

	setTransparent(1);
	createFiledialog(surface, path);

	SDL_EnableKeyRepeat(SDL_DEFAULT_REPEAT_DELAY, SDL_DEFAULT_REPEAT_INTERVAL);
	while ( ! done ) {
		SDL_Event event;
		SDL_WaitEvent(&event);
		if (!handleListBoxEvent(&event, lb, surface)) {
			switch (event.type) {
				case SDL_KEYDOWN:
					if ( (event.key.keysym.sym == SDLK_ESCAPE) || (event.key.keysym.sym == SDLK_q) ) {
						cancel = 1;
						done = 1;
					} else if (event.key.keysym.sym == SDLK_RETURN) {
						int focusitem = ListBoxFocusItem(lb);
						if (filelist[focusitem]->isDir) {
							if (!changeDir(path)) {
								done = 1;
								cancel = 1;
							} else {
								updateLabel(surface, &pathLabel);
								setListBoxSizes(lb);
								drawListBox(lb, surface);
								SDL_UpdateRect(surface,dialog.x,dialog.y,dialog.w,dialog.h);
							}
						} else {
							memset(fullpath, 0, 1024);
							strncpy(fullpath, path, 1024);
							if (!isrootDir(path))
								strcat(fullpath, "/");
							strcat(fullpath,filelist[focusitem]->name);
							done = 1;
						}
					}
					break;
				case SDL_MOUSEBUTTONUP:
				case SDL_MOUSEBUTTONDOWN:
					if ((event.button.state == SDL_PRESSED) && (event.button.button == SDL_BUTTON_LEFT)) {
						if (isButtonPressed(okButton, event.button.x, event.button.y)) {
							drawButton(surface, okButton);
							SDL_UpdateRect(surface,dialog.x,dialog.y,dialog.w,dialog.h);
						} else if (isButtonPressed(cancelButton, event.button.x, event.button.y)) {
							drawButton(surface, cancelButton);
							SDL_UpdateRect(surface,dialog.x,dialog.y,dialog.w,dialog.h);
						}
					} else if ((event.button.state == SDL_RELEASED) && (event.button.button == SDL_BUTTON_LEFT)) {
						if (isButtonReleased(okButton, event.button.x, event.button.y)) {
							SDL_Event e;
							e.type = SDL_KEYDOWN;
							e.key.type = SDL_KEYDOWN;
							e.key.state = SDL_PRESSED;
							e.key.keysym.sym = SDLK_RETURN;
							SDL_PeepEvents(&e, 1, SDL_ADDEVENT, SDL_KEYDOWNMASK);
						} else if (isButtonReleased(cancelButton, event.button.x, event.button.y)) {
							cancel = 1;
							done = 1;
						} 
						drawButton(surface, okButton);
						drawButton(surface, cancelButton);
						SDL_UpdateRect(surface,dialog.x,dialog.y,dialog.w,dialog.h);
					}
					break;
				default:
					break;
			}
		}
	}

	SDL_EnableKeyRepeat(0, SDL_DEFAULT_REPEAT_INTERVAL);
	if (backingstore) {
		SDL_BlitSurface(backingstore, NULL, surface, &dialog);
		SDL_UpdateRect(surface,dialog.x,dialog.y,dialog.w,dialog.h);
		SDL_FreeSurface(backingstore);
		backingstore = NULL;
	}

	freeFilelist();
	if (okButton) free(okButton);
	if (cancelButton) free(cancelButton);
	if (pathLabel.area) SDL_FreeSurface(pathLabel.area);
	pathLabel.area = NULL;
	deleteListBox(lb);
	
	if (cancel)
		return NULL;

	return fullpath;

}		
