/* 
   enjoympeg.c [2000-10-8]
   (c) 2000 by Dieter Mittelmaier <dieter.mittelmaier@freenet.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   This file is a modified version of plaympeg.c, (C) 1999 Loki Entertainment Software
*/

#include <stdlib.h>
#include <string.h>
#include <smpeg.h>
#include <SDL.h>
#include <SDL_thread.h>
#include "controlbar.h"
#include "msgbox.h"
#include "preferences.h"

#define BUFFER_SIZE 1024

typedef struct _vidprops {
	int width, height;
	int pause, loop; 
	int use_audio, use_video, use_timer;
	int fullscreen_toggled, filter;
	int keep_aspect;
	char path[1024];
	char *filename;
}VIDEO_PROPS;

extern char* openFile(SDL_Surface *surface, char *path);

SMPEG *mpeg;

int seeking = 0;
int update_done = 0;
int controlbar = 1;
int controlbar_height;

void update(SDL_Surface *screen, Sint32 x, Sint32 y, Uint32 w, Uint32 h)
{
    //if ( screen->flags & SDL_DOUBLEBUF ) {
    //    SDL_Flip(screen);
    //}
	//printf("update\n");
	if (update_done) {
		update_done = 0;
	} else if (!seeking && controlbar) {
		float slidepos;
		SMPEG_Info info;

		SMPEG_getinfo(mpeg, &info);
		slidepos = calcSlideOffset(info.total_size, info.current_offset);
		setSlideButton(slidepos);
	}
	
}

void clearScreen(SDL_Surface *screen)
{
	SDL_Rect r;
	r.x = 0;
	r.y = 0;
	r.w = screen->w;
	r.h = screen->h - controlbar_height;
	SDL_FillRect(screen, &r, SDL_MapRGB(screen->format, 0, 0, 0));
	SDL_UpdateRects(screen, 1, &r);
}

void playVideo()
{
	SMPEG_play(mpeg);
}

void pauseVideo()
{
	SMPEG_stop(mpeg);
}

void stopVideo(SDL_Surface *surface)
{
	SMPEG_stop(mpeg);
	SMPEG_rewind(mpeg);
	resetButtons();
	clearScreen(surface);
}

void resizeVideo(SDL_Surface *surface, SDL_mutex *surflock, VIDEO_PROPS *vp)
{
	int w, h;
	if (!mpeg)
		return;

	w = surface->w;
	h = surface->h - controlbar_height * controlbar;
	if (vp->keep_aspect) {
		SMPEG_Info info;
		SDL_Rect r;
		float vid_aspect;
		float surf_aspect;
		SMPEG_getinfo(mpeg, &info);
		vid_aspect = info.height / (info.width * 1.0);
		surf_aspect = h / (w * 1.0);
		//printf("aspect %f %f\n", vid_aspect, surf_aspect);
		if (vid_aspect > surf_aspect) {
			w = (int)(w * surf_aspect / vid_aspect);
			//printf("win %d %d\n", w, h);

			r.x = 0;
			r.y = 0;
			r.w = (surface->w - w) / 2;
			r.h = h;
			SMPEG_scaleXY(mpeg, w, h);
			SMPEG_move(mpeg, r.w, 0);

			SDL_mutexP(surflock);
			SDL_FillRect(surface, &r, SDL_MapRGB(surface->format, 0, 0, 0));
			SDL_UpdateRects(surface, 1, &r);
			r.x += w + r.w;
			r.w = surface->w - r.x;
			SDL_FillRect(surface, &r, SDL_MapRGB(surface->format, 0, 0, 0));
			SDL_UpdateRects(surface, 1, &r);
			SDL_mutexV(surflock);

		} else if (vid_aspect < surf_aspect) {
			//printf("win %d %d\n", w, (int)(h * vid_aspect / surf_aspect));
			h = (int)(h * vid_aspect / surf_aspect);
			
			r.x = 0;
			r.y = 0;
			r.w = w;
			r.h = (surface->h - h) / 2;
			SMPEG_scaleXY(mpeg, w, h);
			SMPEG_move(mpeg, 0, r.h);

			SDL_mutexP(surflock);
			SDL_FillRect(surface, &r, SDL_MapRGB(surface->format, 0, 0, 0));
			SDL_UpdateRects(surface, 1, &r);
			r.y += h + r.h;
			r.h = surface->h - r.y;
			SDL_FillRect(surface, &r, SDL_MapRGB(surface->format, 0, 0, 0));
			SDL_UpdateRects(surface, 1, &r);
			SDL_mutexV(surflock);
		} else {
			SMPEG_move(mpeg, 0, 0);
			SMPEG_scaleXY(mpeg, w, h);
		}
	} else {
		SMPEG_move(mpeg, 0, 0);
		SMPEG_scaleXY(mpeg, w, h);
	}
}

void toggleControlBar(SDL_Surface *surface, SDL_mutex *surflock, VIDEO_PROPS *vp)
{
	if (!mpeg)
		return;

	if (SMPEG_status(mpeg) == SMPEG_PLAYING) {
		SDL_mutexP(surflock);
		controlbar = !controlbar;
		SDL_mutexV(surflock);
		resizeVideo(surface, surflock, vp);
		if (controlbar)
			showControlbar();
		if (vp->fullscreen_toggled) {
			if ( surface->flags & SDL_FULLSCREEN ) {
					SDL_ShowCursor(controlbar);
			}
		}
	}
}

Uint32 checkVideoEnd(Uint32 ms, void *param)
{
	int status;
	static int sendEvent = 0;
	if (!mpeg)
		return ms;
		
	status = SMPEG_status(mpeg);
	if (!sendEvent) {
		if (status == SMPEG_STOPPED) {
			SDL_Event event;

			event.type = SDL_USEREVENT;
			event.user.code = 1;
			event.user.data1 = 0;
			event.user.data2 = 0;
			SDL_PushEvent(&event);
			//printf("Video stopped %d\n", ms);
			sendEvent = 1;
		}
	} else {
		if (status == SMPEG_PLAYING) {
			sendEvent = 0;
			//printf("Video playing %d\n", ms);
		}
	}
	
	return ms;
}

int doSeek(SDL_mutex *surflock, int pause)
{
	SMPEG_Info info;
	SDL_Event event;
	float slidepos;
	int frameondisplay;
	int seek = SEEK;
	int seekfailure = 1;
	int seektime = 0;
	int was_playing = 0;
	SDL_mutexP(surflock);
	seeking = 1;
	SDL_mutexV(surflock);

	SMPEG_getinfo(mpeg, &info);
	if (!pause) {
		pauseVideo();
		was_playing = 1;
	}

	while (seek == SEEK) {
		SDL_WaitEvent(&event);
		switch (event.type) {
			case SDL_MOUSEMOTION:
				seek = updateSlideButton(event.motion.x);
				break;
			case SDL_MOUSEBUTTONUP:
			case SDL_MOUSEBUTTONDOWN:
				seek = 0;
				break;
			default:
				break;
		}
	}

	slidepos = currentSlideOffset();
	seek = (int)(info.total_size * (slidepos / 100.5));
	//printf("%d %d %f\n",info.total_size,seek,slidepos);

		//if (use_audio) SMPEG_enableaudio(mpeg, 0);
		SMPEG_seek(mpeg, seek);
		//printf("Finish seek\n");
		frameondisplay = 1;
		SDL_mutexP(surflock);
		update_done = 1;
		SDL_mutexV(surflock);
		while (frameondisplay) {
			SMPEG_getinfo(mpeg, &info);
			//printf("%d %f\n",info.current_frame,info.current_time);
			SMPEG_renderFrame(mpeg, info.current_frame + 1);
			SDL_Delay(50);
			seektime += 50;

			SDL_mutexP(surflock);
			frameondisplay = update_done;
			SDL_mutexV(surflock);

			if (seektime > 2000) {
				seekfailure = 0;
				break;
			}
		}
		
		//printf("seektime %d\n",seektime);
		//if (use_audio) SMPEG_enableaudio(mpeg, 1);
	//SMPEG_getinfo(mpeg, &info);
	//printf("%d\n",info.current_frame);

	SDL_mutexP(surflock);
	seeking = 0;
	SDL_mutexV(surflock);

	if (was_playing) {
		playVideo();
		SDL_Delay(500);
	}

	return seekfailure;
}

/*
int doSeek(SDL_mutex *surflock, int pause)
{
	SMPEG_Info info;
	SDL_Event event;
	float slidepos;
	int frameondisplay;
	int seek = SEEK;
	int seekfailure = 1;
	int seektime = 0;

	SDL_mutexP(surflock);
	seeking = 1;
	SDL_mutexV(surflock);

	SMPEG_getinfo(mpeg, &info);

	while (seek == SEEK) {
		SDL_WaitEvent(&event);
		switch (event.type) {
			case SDL_MOUSEMOTION:
				seek = updateSlideButton(event.motion.x);
				break;
			case SDL_MOUSEBUTTONUP:
			case SDL_MOUSEBUTTONDOWN:
				seek = 0;
				break;
			default:
				break;
		}
	}

	slidepos = currentSlideOffset();
	seek = (int)(info.total_size * (slidepos / 100.5));
	//printf("%d %d %f\n",info.total_size,seek,slidepos);
	if (pause) {
		//if (use_audio) SMPEG_enableaudio(mpeg, 0);
		SMPEG_seek(mpeg, seek);
		//printf("Finish seek\n");
		frameondisplay = 1;
		SDL_mutexP(surflock);
		update_done = 1;
		SDL_mutexV(surflock);
		while (frameondisplay) {
			SMPEG_getinfo(mpeg, &info);
			//printf("%d %f\n",info.current_frame,info.current_time);
			SMPEG_renderFrame(mpeg, info.current_frame + 1);
			SDL_Delay(50);
			seektime += 50;

			SDL_mutexP(surflock);
			frameondisplay = update_done;
			SDL_mutexV(surflock);

			if (seektime > 2000) {
				seekfailure = 0;
				break;
			}
		}
		
		//printf("seektime %d\n",seektime);
		//if (use_audio) SMPEG_enableaudio(mpeg, 1);
	} else {
		SMPEG_seek(mpeg, seek);
		SDL_Delay(300);
		
	}
	//SMPEG_getinfo(mpeg, &info);
	//printf("%d\n",info.current_frame);

	SDL_mutexP(surflock);
	seeking = 0;
	SDL_mutexV(surflock);

	return seekfailure;
}
*/

void changePrefs(SDL_Surface *surface, SDL_mutex *surflock, PREFS *p, VIDEO_PROPS *vp)
{
	int was_playing = 0;
	int resize_done = 0;
	int keep_aspect = vp->keep_aspect;

	if (SMPEG_status(mpeg) == SMPEG_PLAYING) {
		pauseVideo();
		was_playing = 1;
	}

	prefDialog(surface, p);

	vp->keep_aspect = p->keep_aspect_ratio;
	vp->loop = p->loop;


	if ((p->win_width != vp->width) && !vp->fullscreen_toggled) {
		surface = SDL_SetVideoMode(p->win_width, p->win_height,
									surface->format->BitsPerPixel,
									surface->flags);

		if (!surface) {
			printf("Resize window failed\n");
			SDL_Quit();
			exit(1);
		}

		SMPEG_setdisplay(mpeg, surface, surflock, update);
		vp->width = p->win_width;
		vp->height = p->win_height;
		resizeVideo(surface, surflock, vp);
		resize_done = 1;
		resizeControlbar(surface, controlbar);
	}

	if ((vp->keep_aspect != keep_aspect) && !resize_done)
		resizeVideo(surface, surflock, vp);

	if (was_playing)
		playVideo();
//printf("#This is the enjoympeg preferences file\n\n");
//printf("WINDOW_SIZE %dx%d\n", p->win_width, p->win_height);
//printf("KEEP_ASPECT_RATIO %d\n", p->keep_aspect_ratio);
//printf("ALWAYS_LOOP %d\n", p->loop);
//printf("PLAY_ON_START %d\n", p->playonstart);
//printf("SAVE_PREFS %d\n", p->save_prefs);
//printf("FILTER %d\n", p->filter);

}

void toggleFullScreen(SDL_Surface *surface, SDL_mutex *surflock, PREFS *p, VIDEO_PROPS *vp)
{
	SDL_Rect **modes;
	int resize_w = 0;
	int resize_h = 0;
	int was_playing = 0;

	if (!vp->fullscreen_toggled) {
		modes = SDL_ListModes(NULL, surface->flags | SDL_FULLSCREEN);
		if (modes == (SDL_Rect **)0){
			printf("No fullscreen modes available!\n");
			return;
		} else if (modes == (SDL_Rect **)-1) {
			printf("All resolutions available.\n");
			resize_w = vp->width;
			resize_h = vp->height;
		} else {
			int m;
			for (m = 0; modes[m]; ++m) {
				if ((modes[m]->w >= vp->width) && (modes[m]->h >= vp->height)) {
					//printf("%d x %d\n", modes[m]->w, modes[m]->h);
					resize_w = modes[m]->w;
					resize_h = modes[m]->h;
				}
			}
			
			if ((resize_w > 0) && (resize_h > 0)) {
				printf("Switch to %d x %d fullscreen\n", resize_w, resize_h);
			} else {
				printf("Cannot switch to %d x %d fullscreen\n", resize_w, resize_h);
				return;
			}
		}

		if (SMPEG_status(mpeg) == SMPEG_PLAYING) {
			pauseVideo();
			was_playing = 1;
		}

		if ((resize_w != vp->width) || (resize_h != vp-> height)) {
			surface = SDL_SetVideoMode(resize_w, resize_h,
									surface->format->BitsPerPixel,
									surface->flags);

			if (!surface) {
				printf("Resize window failed\n");
				SDL_Quit();
				exit(1);
			}

			SMPEG_setdisplay(mpeg, surface, surflock, update);
			vp->width = resize_w;
			vp->height = resize_h;
			resizeVideo(surface, surflock, vp);
			resizeControlbar(surface, controlbar);
		}
		if (SDL_WM_ToggleFullScreen(surface)){
			vp->fullscreen_toggled = 1;
		} else {
			printf("Switching to fullscreen failed\n");
		}
		if (was_playing)
			playVideo();
	} else {
		if (SDL_WM_ToggleFullScreen(surface)){
			vp->fullscreen_toggled = 0;
		} else {
			printf("Switching to window failed\n");
			return;
		}

		if (SMPEG_status(mpeg) == SMPEG_PLAYING) {
			pauseVideo();
			was_playing = 1;
		}
		resize_w = p->win_width;
		resize_h = p->win_height;

		if ((resize_w != vp->width) || (resize_h != vp-> height)) {
			surface = SDL_SetVideoMode(resize_w, resize_h,
									surface->format->BitsPerPixel,
									surface->flags);

			if (!surface) {
				printf("Resize window failed\n");
				SDL_Quit();
				exit(1);
			}

			SMPEG_setdisplay(mpeg, surface, surflock, update);
			vp->width = resize_w;
			vp->height = resize_h;
			resizeVideo(surface, surflock, vp);
			resizeControlbar(surface, controlbar);
		}

		if (was_playing)
			playVideo();

	}		
}

void mpegInfo(SDL_Surface *surface, char *filename)
{
    SMPEG_Info info;
	char buffer[1024];
	char *basefile = NULL;
	int was_playing = 0;
	
	SMPEG_getinfo(mpeg, &info);

	if ( SMPEG_status(mpeg) == SMPEG_PLAYING ) {
		pauseVideo();
		was_playing = 1;
	}

    basefile = strrchr(filename, '/');
    if ( basefile ) {
    	++basefile;
    } else {
    	basefile = filename;
    }

    if ( info.has_audio && info.has_video ) {
		if (was_playing) {
			sprintf(buffer, "File: %s\nMPEG system stream (audio/video)\n"
							"Video %dx%d resolution\n"
							"Audio %s\n"
							"Size: %d\n"
							"\nTotal time: %f\n"
							"Current time: %f\n"
							"\nFrame: %d\n"
							"Frames per sec.: %2.2f\n"
							"Audio Frame: %d\n"
							"File Offset: %d\n",
							basefile, info.width, info.height,
							info.audio_string, info.total_size,
							info.total_time, info.current_time,
							info.current_frame, info.current_fps,
							info.audio_current_frame, info.current_offset);
		} else {
			sprintf(buffer, "File: %s\nMPEG system stream (audio/video)\n"
							"Video %dx%d resolution\n"
							"Audio %s\n"
							"Size: %d\n"
							"\nTotal time: %f\n"
							"Current time: %f\n",
							basefile, info.width, info.height,
							info.audio_string, info.total_size,
							info.total_time, info.current_time);
		}
	} else if ( info.has_video ) {
		if (was_playing) {
			sprintf(buffer, "File: %s\nMPEG video stream\n"
							"Video %dx%d resolution\n"
							"Size: %d\n"
							"\nTotal time: %f\n"
							"Current time: %f\n"
							"\nFrame: %d\n"
							"Frames per sec.: %2.2f\n"
							"File Offset: %d\n",
							basefile, info.width, info.height, info.total_size,
							info.total_time, info.current_time,
							info.current_frame, info.current_fps,
							info.current_offset);
		} else {
			sprintf(buffer, "File: %s\nMPEG video stream\n"
							"Video %dx%d resolution\n"
							"Size: %d\n"
							"\nTotal time: %f\n"
							"Current time: %f\n",
							basefile, info.width, info.height, info.total_size,
							info.total_time, info.current_time);
		}
	}

	messageBox(surface, buffer);
	if (was_playing) {
		playVideo();
		SDL_Delay(500);
	}
}

void setCaption(char *filename)
{
	char caption[256];
	char *basefile;
	
    basefile = strrchr(filename, '/');
    if ( basefile ) {
    	++basefile;
    } else {
    	basefile = filename;
    }

	strcpy(caption, "Enjoy ");
	strcat(caption, basefile);
	
	SDL_WM_SetCaption(caption, "enjoympeg");	
}

void initMPEG(SDL_Surface *surface, SDL_mutex *surflock, char *file, VIDEO_PROPS *vp)
{
    SMPEG_Info info;
    SMPEG *newmpeg = NULL;
	
	/* Create the MPEG stream */
	newmpeg = SMPEG_new(file, &info, 1);
	if ( SMPEG_error(newmpeg) ) {
    	fprintf(stderr, "%s: %s\n", file, SMPEG_error(newmpeg));
   	    SMPEG_delete(newmpeg);
		return;
	}

	if (!info.has_video) {
    	fprintf(stderr, "No MPEG video stream found in: %s\n", file);
   	    SMPEG_delete(newmpeg);
		return;
	}
/*
	if (!info.has_audio) {
    	fprintf(stderr, "No MPEG audio stream found in: %s\n", file);
   	    SMPEG_delete(newmpeg);
		return;
	}
*/
	vp->use_audio = info.has_audio;
	vp->use_video = info.has_video;

	mpeg = newmpeg;
    SMPEG_enableaudio(mpeg, info.has_audio);
    SMPEG_enablevideo(mpeg, info.has_video);
    if (info.has_audio) SMPEG_setvolume(mpeg, 100);
	
	SMPEG_setdisplay(mpeg, surface, surflock, update);
	resizeVideo(surface, surflock,  vp);
	
	setCaption(file);

	//return newmpeg;
}

int openVideo(SDL_Surface *surface, SDL_mutex *surflock, VIDEO_PROPS *vp)
{
	char *newfile;
	int was_playing = 0;

	if (mpeg) {
		if (SMPEG_status(mpeg) == SMPEG_PLAYING) {
			pauseVideo();
			vp->pause = 1;
			was_playing = 1;
		}
	}

	newfile = openFile(surface, vp->path);

	if (newfile) {
		vp->use_audio = 1;
		vp->use_video = 1;
		if (mpeg) {
			stopVideo(surface);
			SMPEG_delete(mpeg);
			mpeg = NULL;
		}
		initMPEG(surface, surflock, newfile, vp);
		if (mpeg) {
			vp->filename = newfile;
			//SMPEG_getinfo(mpeg, &info);
		} else {
			return 0;
		}
	} else {
		if (mpeg) {
			if (was_playing) {
				playVideo();
				vp->pause = 0;
				SDL_Delay(500);
			}
		} else {
			return 0;
		}
	}

	return 1;
}	

int main(int argc, char *argv[])
{
	const SDL_VideoInfo *video_info;
	Uint32 video_flags;
	int video_bpp;
    int done; 
	int buttonstate;
    SDL_Surface *screen;
	SDL_mutex *screenlock;
	SDL_TimerID timerID;
    //SMPEG_Info info;
	VIDEO_PROPS vid;
	PREFS pref;
	
	loadPrefs(&pref);

	vid.width = pref.win_width;
	vid.height = pref.win_height;
    vid.loop = pref.loop;
	vid.filter = pref.filter;
	vid.pause = 1; //!pref.playonstart;
	vid.keep_aspect = pref.keep_aspect_ratio;
	vid.use_audio = 1;
    vid.use_video = 1;
	vid.use_timer = 1;
	vid.fullscreen_toggled = 0;
	vid.filename = NULL;
	memset(vid.path ,0, 1024);

	mpeg = NULL;
	done = 0;
	buttonstate = 0;

	
	/* Initialize SDL */
	if ( SDL_Init(SDL_INIT_VIDEO) < 0 ) {
		fprintf(stderr, "Couldn't init SDL video: %s\n", SDL_GetError());
		exit(1);
	} else {
		if ( SDL_Init(SDL_INIT_AUDIO) < 0 ) {
			fprintf(stderr, "Couldn't init SDL audio: %s\n", SDL_GetError());
			fprintf(stderr, "Will ignore audio stream\n");
			vid.use_audio = 0;
		}
		if ( SDL_Init(SDL_INIT_TIMER) < 0 ) {
			fprintf(stderr, "Couldn't init SDL timer: %s\n", SDL_GetError());
			fprintf(stderr, "Will ignore timer callback\n");
			vid.use_timer = 0;
		}
	}

	/* Get the "native" video mode */
	video_info = SDL_GetVideoInfo();
	switch (video_info->vfmt->BitsPerPixel) {
		case 16:
		case 32:
			video_bpp = video_info->vfmt->BitsPerPixel;
			break;
		default:
			video_bpp = 16;
			break;
	}

	video_flags = SDL_SWSURFACE|SDL_ASYNCBLIT;

	screen = SDL_SetVideoMode(vid.width, vid.height, video_bpp, video_flags);
	if ( screen == NULL ) {
    	fprintf(stderr, "Unable to set %dx%d video mode: %s\n",
                               	vid.width, vid.height, SDL_GetError());
		SDL_Quit();
		exit(1);
	}

	SDL_WM_SetCaption("enjoympeg", "enjoympeg");

	screenlock = SDL_CreateMutex();
	controlbar_height = setupControlbar(screen, screenlock, vid.pause);
	controlbar = vid.pause;

	if(argc < 2) {
		if (!openVideo(screen, screenlock, &vid)) {
			printf("No file selected\nExit\n");
			SDL_Quit();
			exit(1);
		}
	} else {
		vid.filename = argv[1];
		initMPEG(screen, screenlock, vid.filename, &vid);
		if (!mpeg) {
			SDL_Quit();
			exit(1);
		}
	}

	if (!mpeg) {
		printf("No mpeg\n");
		SDL_Quit();
		exit(1);
	}

	/* Play it, and wait for playback to complete */
	if (vid.pause) {
		showControlbar();
	} else {
		playVideo();
	}
	
	timerID = SDL_AddTimer(1000, checkVideoEnd, (void *)1);
	if (!timerID) {
		printf("timer error %s\n",SDL_GetError());
		vid.use_timer = 0;
	}

	while (!done) {
		SDL_Event event;
		if (vid.use_timer) {
			SDL_WaitEvent(&event);
		} else {
			if (!SDL_PollEvent(&event))
				SDL_Delay(500);
		}
		switch (event.type) {
			case SDL_MOUSEBUTTONUP:
			case SDL_MOUSEBUTTONDOWN:
				if (event.button.state == SDL_PRESSED) {
					if ((event.button.button == SDL_BUTTON_LEFT) && controlbar) {
						if (slideButtonPressed(event.button.x, event.button.y - (screen->h - controlbar_height))) {
							if (!doSeek(screenlock, vid.pause)) {
								stopVideo(screen);
								vid.pause = 1;
							}
						} else {
							buttonstate = controlButtonPressed(event.button.x, event.button.y - (screen->h - controlbar_height));
							SDL_Delay(200);
						}
					}
				} else if (event.button.state == SDL_RELEASED) {
					if (event.button.button == SDL_BUTTON_RIGHT) {
						toggleControlBar(screen, screenlock, &vid);
					} else if ((event.button.button == SDL_BUTTON_LEFT) && controlbar) {
						buttonstate = controlButtonReleased(event.button.x, event.button.y - (screen->h - controlbar_height));
						if (buttonstate == PLAY) {
							if ( SMPEG_status(mpeg) != SMPEG_PLAYING ) {
								playVideo();
								vid.pause = 0;
							} else {
								pauseVideo();
								vid.pause = 1;
							}
						} else if (buttonstate == STOP) {
							stopVideo(screen);
							vid.pause = 1;
						} else if (buttonstate == FULL) {
							toggleFullScreen(screen, screenlock, &pref, &vid);
						} else if (buttonstate == QUIT) {
							pauseVideo();
							vid.pause = 1;
							done = 1;
						} else if (buttonstate == FOLDER) {
							if (!openVideo(screen, screenlock, &vid))
								done = 1;
						} else if (buttonstate == INFO) {
							mpegInfo(screen, vid.filename);
						} else if (buttonstate == PREF) {
							changePrefs(screen, screenlock, &pref, &vid);
						}
					}
				}
				break;
			case SDL_KEYDOWN:
				if (event.key.keysym.sym == SDLK_ESCAPE) {
					pauseVideo();
					done = 1;
				} else if (event.key.keysym.sym == SDLK_s) {
					SMPEG_skip(mpeg, 10);
				}
				break;
			case SDL_QUIT:
				done = 1;
				break;
			case SDL_USEREVENT:
				//printf("Video stop event\n");
				if (!vid.pause) {
					printf("Video end\n");
					//SDL_Delay(200);
				}
				break;
			default:
				break;
		}

		if (!done) {
			if (!vid.pause && (SMPEG_status(mpeg) == SMPEG_STOPPED)) {
				if (vid.loop) {
					pauseVideo();
					SMPEG_rewind(mpeg);
					playVideo();
				} else {
					vid.pause = 1;
					if (!controlbar) {
						controlbar = 1;
						resizeVideo(screen, screenlock, &vid);
					}
					stopVideo(screen);
					if (screen->flags & SDL_FULLSCREEN) SDL_ShowCursor(1);
				}
			}
		}
	}

	if (pref.save_prefs) savePrefs(&pref);
	if (timerID) SDL_RemoveTimer(timerID);
	if (mpeg) SMPEG_delete(mpeg);
	SDL_DestroyMutex(screenlock);
    SDL_Quit();
    exit(0);
}

