/* 
   controlbar.c [2000-10-8]
   (c) 2000 by Dieter Mittelmaier <dieter.mittelmaier@freenet.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <stdlib.h>
#include <string.h>
#include "controlbar.h"
#include "button.h"
#include "drawutil.h"

#define LAST_PRESSED	0
#define CONTROL_HEIGHT	28

typedef struct _SlideButton {
	SDL_Rect area;
	SDL_Rect button;
	Sint16 min_x;
	Sint16 max_x;
}SLIDEBUTTON;

BUTTON *buttons[MAX_BUTTONS];
SLIDEBUTTON slide;

SDL_Surface *display = NULL;
SDL_Surface *control = NULL;
SDL_mutex *displaylock;

static float slidepos;

int drawSlideButton(SDL_Surface *surface, Sint16 x, Sint16 y, Uint16 w, Uint16 h)
{
	int error = 0;
	SDL_Rect r;

	r.x = x;
	r.y = y;
	r.w = w;
	r.h = h;

	error = drawRaisedWin(surface, &r, light_color, shadow_color, back_color);
	
	return error;
}

int controlButtonPressed(Uint16 x, Uint16 y)
{
	int i;
	
	for(i = 1; i < MAX_BUTTONS; i++) {
		if (isButtonPressed(buttons[i], x, y)) {
			drawButton(control, buttons[i]);
			buttons[LAST_PRESSED] = buttons[i];
			showControlbar();
			return ButtonType(buttons[i]);
		}
	}

	buttons[LAST_PRESSED] = NULL;
	return 0;
}

int controlButtonReleased(Uint16 x, Uint16 y)
{
	int type = 0;

	if(!buttons[LAST_PRESSED])
		return 0;
		
	if(isButtonReleased(buttons[LAST_PRESSED], x, y))
		type = ButtonType(buttons[LAST_PRESSED]);

	drawButton(control, buttons[LAST_PRESSED]);
	buttons[LAST_PRESSED] = NULL;
	showControlbar();

	return type;
}

int slideButtonPressed(Uint16 x, Uint16 y)
{
	if ((x > slide.button.x) && (x < slide.button.x + slide.button.w)) {
		if ((y > slide.button.y) && (y < slide.button.y + slide.button.h)) {
			return 1;
		}
	}

	return 0;
}

int updateSlideButton(Uint16 x)
{
	int sx = x;

	if ((sx > slide.min_x) && (sx < (slide.max_x + slide.button.w / 2))) {
		SDL_FillRect(control,&slide.button,back_color);
		sx -= slide.button.w / 2;
		if ((sx) < slide.min_x)
			sx = slide.min_x;
		if (sx > slide.max_x)
			sx = slide.max_x;
		slide.button.x = sx;
		slidepos = (sx - slide.min_x) * 100.0 / (slide.area.w - 1);
		drawSlideButton(control, slide.button.x, slide.button.y, slide.button.w, slide.button.h);
		showControlbar();
		return SEEK;
	}
	
	return -1;
}

int setSlideButton(float percent)
{
	Uint16 pos;

	if(percent > 100.0)
		percent = 100.0;

	slidepos = percent;
	pos = slide.min_x + ((slide.area.w - 1) / 100.0) * percent;
	if (slide.button.x == pos)
		return 0;

	SDL_FillRect(control,&slide.button,back_color);
	slide.button.x = pos;
	drawSlideButton(control, slide.button.x, slide.button.y, slide.button.w, slide.button.h);
	showControlbar();

	return 1;
}	

float calcSlideOffset(Uint32 total, Uint32 current)
{
	float percent;
	percent = 100.0 * current / total;

	return percent;
}

float currentSlideOffset()
{
	return slidepos;
}

void showControlbar()
{
	SDL_Rect s;
	SDL_Rect d;
	
	s.x = 0;
	s.y = 0;
	s.w = control->w;
	s.h = control->h;
	
	d.x = s.x;
	d.y = display->h - control->h;
	d.w = control->w;
	d.h = control->h;

	SDL_mutexP(displaylock);
	
	SDL_BlitSurface(control, &s, display, &d);
    SDL_UpdateRect(display,d.x,d.y,d.w,d.h);

	SDL_mutexV(displaylock);
}

void resetButtons()
{
	setToggleButton(buttons[PLAY], 1, 0);
	drawButton(control, buttons[PLAY]);
	setSlideButton(0.0);
	slidepos = 0.0;
	showControlbar();
}

int resizeControlbar(SDL_Surface *newSurface, int isvisible)
{
	int i;
	int s;
    SDL_Rect c;
	Uint32 slider_scale_color;
		
	display = newSurface;
	if (control) {
		if (control->w == newSurface->w) {
			if (isvisible)
				showControlbar();
			return 1;
		} else {
			SDL_FreeSurface(control);
			control = NULL;
		}
	}
	
	control = SDL_CreateRGBSurface(SDL_SWSURFACE, newSurface->w, CONTROL_HEIGHT,
						newSurface->format->BitsPerPixel,
						newSurface->format->Rmask, newSurface->format->Gmask,
						newSurface->format->Bmask, newSurface->format->Amask);

	if (control == NULL) {
    	fprintf(stderr, "Unable to set %dx%d controlbar_surface: %s\n",
                               	newSurface->w, CONTROL_HEIGHT, SDL_GetError());
		return 0;
	}

	slider_scale_color = SDL_MapRGB(control->format, 0, 0, 0);

    c.x = 0;
    c.y = 0;
    c.w = control->w;
    c.h = control->h;

	drawRaisedWin(control, &c, light_color, shadow_color, back_color);
	for(i = PLAY; i < MAX_BUTTONS; i++) {
		drawButton(control, buttons[i]);
	}
	
	slide.area.x = slide.min_x + slide.button.w / 2;
	slide.area.y = slide.button.y + 18;
	slide.area.w = control->w - 3 - slide.area.x - slide.button.w / 2;
	slide.area.h = 22;
	slide.max_x = slide.area.x + slide.area.w  - slide.button.w / 2;
	slide.button.x = slide.min_x + ((slide.area.w - 1) / 100.0) * slidepos;
	
	drawLine(control, slide.area.x, slide.area.y, slide.area.w, 2, slider_scale_color);
	s = slide.area.w / 10;
	for (i = 0; i < 10; i++) {
		drawLine(control, slide.area.x + i*s, slide.area.y - 3, 1, 3, slider_scale_color);
	}
	drawLine(control, slide.area.x + slide.area.w - 1, slide.area.y - 3, 1, 3, slider_scale_color);
	drawSlideButton(control, slide.button.x, slide.button.y, slide.button.w, slide.button.h);
	
	if (isvisible)
		showControlbar();

	return 1;
}

int setupControlbar(SDL_Surface *surface, SDL_mutex *surflock, int pause)
{
	int i;
    SDL_Rect c;
	char bmpfile1[256];
	char bmpfile2[256];

	displaylock = surflock;

	slidepos = 0.0;
    back_color = SDL_MapRGB(surface->format, 179, 179, 179);
	light_color = SDL_MapRGB(surface->format, 225, 225, 225);
	shadow_color = SDL_MapRGB(surface->format, 104, 104, 104);
	

	loadFont();
	
	for(i = 0; i < MAX_BUTTONS; i++) {
		buttons[i] = NULL;
	}

	c.x = 3;
	c.y = 3;
	c.w = 22;
	c.h = 22;

	strcpy(bmpfile1, BMPDIR);
	strcat(bmpfile1, "/quit.bmp");
	buttons[QUIT] = Button(&c, QUIT);
	setButtonImages(buttons[QUIT], bmpfile1, NULL);

	c.x +=24;
	strcpy(bmpfile1, BMPDIR);
	strcat(bmpfile1, "/folder.bmp");
	buttons[FOLDER] = Button(&c, FOLDER);
	setButtonImages(buttons[FOLDER], bmpfile1, NULL);

	c.x +=24;
	strcpy(bmpfile1, BMPDIR);
	strcat(bmpfile1, "/info.bmp");
	buttons[INFO] = Button(&c, INFO);
	setButtonImages(buttons[INFO], bmpfile1, NULL);
	
	c.x +=24;
	strcpy(bmpfile1, BMPDIR);
	strcat(bmpfile1, "/pref.bmp");
	buttons[PREF] = Button(&c, PREF);
	setButtonImages(buttons[PREF], bmpfile1, NULL);

	c.x +=30;
	strcpy(bmpfile1, BMPDIR);
	strcat(bmpfile1, "/play.bmp");
	strcpy(bmpfile2, BMPDIR);
	strcat(bmpfile2, "/pause.bmp");
	buttons[PLAY] = Button(&c, PLAY);
	setToggleButton(buttons[PLAY], 1, !pause);
	setButtonImages(buttons[PLAY], bmpfile1, bmpfile2);

	c.x +=24;
	strcpy(bmpfile1, BMPDIR);
	strcat(bmpfile1, "/stop.bmp");
	buttons[STOP] = Button(&c, STOP);
	setButtonImages(buttons[STOP], bmpfile1, NULL);

	c.x +=30;
	strcpy(bmpfile1, BMPDIR);
	strcat(bmpfile1, "/full.bmp");
	strcpy(bmpfile2, BMPDIR);
	strcat(bmpfile2, "/win.bmp");
	buttons[FULL] = Button(&c, FULL);
	setToggleButton(buttons[FULL], 1, 0);
	setButtonImages(buttons[FULL], bmpfile1, bmpfile2);

	c.x += 36;
	slide.button.x = c.x;
	slide.button.y = c.y;
	slide.button.w = 13;
	slide.button.h = 15;
	slide.min_x = slide.button.x;

	if (resizeControlbar(surface, 0))
		return CONTROL_HEIGHT;

	return 0;
}

