/* 
   button.c [2000-10-29]
   (c) 2000 by Dieter Mittelmaier <dieter.mittelmaier@freenet.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <stdlib.h>
#include <string.h>
#include "button.h"
#include "drawutil.h"

struct _Button {
	int type;
	int state;
	int isToggleButton;
	int toggled;
	int	isCheckButton;
	SDL_Rect rect;
	char *text;
	SDL_Surface *image[2];
};

BUTTON *Button(SDL_Rect *r, int type)
{
	BUTTON *b;
	
	b = (BUTTON *)malloc(sizeof(BUTTON));
	if (b) {
		b->type = type;
		b->state = BUTTON_UP;
		b->isToggleButton = 0;
		b->toggled = 0;
		b->isCheckButton = 0;
		b->rect.x = r->x;
		b->rect.y = r->y;
		b->rect.w = r->w;
		b->rect.h = r->h;
		b->text = NULL;
		b->image[0] = NULL;
		b->image[1] = NULL;
		return b;
	} else {
		printf("Button %d not allocated\n", type);
	}
	
	return NULL;
}

void setButtonText(BUTTON * b, char *text)
{
	b->text = text;
}

void setButtonImages(BUTTON * b, char *image1, char *image2)
{
	b->image[0] = SDL_LoadBMP(image1);
	if (b->image[0] == NULL) {
		printf("Loading Button-Image %s failed\n", image1);
	}

	if (!image2)
		return;

	b->image[1] = SDL_LoadBMP(image2);
	if (b->image[1] == NULL) {
		printf("Loading Button-Image %s failed\n", image2);
	}
}

void setToggleButton(BUTTON *b, int on, int toggled)
{
	b->isToggleButton = on;
	b->toggled = toggled;
}

void setCheckButton(BUTTON *b, int checked)
{
	b->isCheckButton = 1;
	b->state = checked;
}

int isButtonChecked(BUTTON *b)
{
	return b->state;
}

int drawButton(SDL_Surface *surface, BUTTON * b)
{
	int error = 0;
	SDL_Rect r;

	r.x = b->rect.x;
	r.y = b->rect.y;
	r.w = b->rect.w;
	r.h = b->rect.h;

	if (b->state == BUTTON_UP) {
		if (b->isCheckButton) {
			r.h = 10;
			r.w = r.h;
			r.x += (b->rect.h - r.h) / 2;
			r.y += (b->rect.h - r.h) / 2;
			error = drawRaisedWin(surface, &r, light_color, shadow_color, back_color);
		} else {
			error = drawRaisedWin(surface, &r, light_color, shadow_color, back_color);
		}
	} else {
		if (b->isCheckButton) {
			r.h = 10;
			r.w = r.h;
			r.x += (b->rect.h - r.h) / 2;
			r.y += (b->rect.h - r.h) / 2;
			error = drawSunkenWin(surface, &r, light_color, shadow_color, back_color);
		} else {
			error = drawSunkenWin(surface, &r, light_color, shadow_color, back_color);
		}
	}


	if (b->image[b->toggled]) {
		r.x += 3;
		r.y += 3;
		r.w = b->image[b->toggled]->w;
		r.h = b->image[b->toggled]->h;
		SDL_BlitSurface(b->image[b->toggled], NULL, surface, &r);
	} else if (b->text) {
		int textwidth = strlen(b->text) * charWidth();
		if (b->isCheckButton) {
			Sint16 x = b->rect.x + b->rect.h;
			Sint16 y = (b->rect.h - charHeight()) / 2 + b->rect.y;
			drawText(surface, x, y, b->text);
		} else {
			Sint16 x = (b->rect.w - textwidth) / 2 + b->rect.x;
			Sint16 y = (b->rect.h - charHeight()) / 2 + b->rect.y;
			drawText(surface, x, y, b->text);
		}
	}

	return error;
}

int insideButton(BUTTON *b, Sint16 x, Sint16 y)
{
	if ((x >= b->rect.x) && (x < b->rect.x + b->rect.w)) {
		if ((y >= b->rect.y) && (y < b->rect.y + b->rect.h)) {
			return 1;
		}
	}

	return 0;
}

int isButtonPressed(BUTTON *b, Sint16 x, Sint16 y)
{
	if (insideButton(b, x, y)) {
		if (b->isCheckButton) {
			if (b->state == BUTTON_DOWN)
				b->state = BUTTON_UP;
			else
				b->state = BUTTON_DOWN;
		} else {
			b->state = BUTTON_DOWN;
		}
		return 1;
	}

	return 0;
}

int isButtonReleased(BUTTON *b, Sint16 x, Sint16 y)
{
	if (b->state == BUTTON_UP)
		return 0;
	if (insideButton(b, x, y)) {
		if (!b->isCheckButton)
			b->state = BUTTON_UP;
		if (b->isToggleButton) {
			if (b->toggled)
				b->toggled = 0;
			else
				b->toggled = 1;
		}
		return 1;
	} else {
		if (!b->isCheckButton)
			b->state = BUTTON_UP;
	}

	return 0;
}

int ButtonType(BUTTON *b)
{
	return b->type;
}
