//======================================================================
// Copyright (C) 2002 Daniel Heck
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//  
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//  
// You should have received a copy of the GNU General Public License along
// with this program; if not, write to the Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//======================================================================
#include "world.hh"
#include "objects.hh"
#include "display.hh"
#include "player.hh"
#include "sound.hh"
#include "px/math.hh"
#include "px/tools.hh"
#include "px/array2.hh"
#include "lua.hh"

#include <iostream>
#include <algorithm>
#include <functional>
#include <map>

using namespace std;
using namespace world;
using namespace px;


tools::Timer world::g_timer;

 // ActorInfo
ActorInfo::ActorInfo(Actor *a, px::V3 p, px::V3 v) 
    : actor(a), pos(p), vel(v), oldpos(p), oldvel(v), grabbed(false)
{}
void ActorInfo::dispose() { delete actor; }

//----------------------------------------
 // Field
//----------------------------------------
namespace
{
    class Field {
    public:
	Field();
	~Field();

        Floor *floor;
        Item *item;
        Stone *stone;
    };
    typedef px::Array2<Field> FieldArray;
    typedef std::vector<ActorInfo> ActorList;
}

Field::Field() 
{
    floor=0;
    item=0;
    stone=0;
}

Field::~Field() 
{
    DisposeObject(floor);
    DisposeObject(item);
    DisposeObject(stone);
}

 // Forces
namespace
{
    class ForceField {
    public:
	virtual ~ForceField() {}
	virtual V3 get_force(ActorInfo &a,const V3 & x, 
                             const V3 & v, double time)=0;
	virtual void tick(double dtime) {}
    };

    class ConstantForce : public ForceField {
    public:
	ConstantForce(const V3 &f) : force(f)
	{}
	V3 get_force(ActorInfo &a,const V3 & x, const V3 & v, 
                     double time) 
	{
	    return force;
	}
    private:
	V3 force;
    };

    class MouseForce : public ForceField {
    public:
	MouseForce() 
	{
	    duration = 30./1000;
	    time_since_reset = 0;
	    maxforce = 50;
	}

	void set_force(const V3 &f) { force=f; limit_force(); }
	void add_force(const V3 &f) { force+=f; limit_force(); }
	
	void limit_force() {
	    if (length(force) > maxforce)
		force *= maxforce/length(force);
	}
	
	V3 get_force(ActorInfo &a, const V3 & x, const V3 & v,
                     double time) 
	{
	    Actor *ac = a.actor;
	    if (time < duration && ac->get_attrib("mouseforce"))
		return force;
	    else
		return V3();
	}

	void tick(double dtime)
        {
	    time_since_reset += dtime;
	    if (time_since_reset >= duration) 
	    {
		time_since_reset -= duration;
		force = V3();
	    }
	}
    private:
	double duration;        // how long does the force have effect?
	double time_since_reset;
	double maxforce;	// maximum force possible
	V3 force;
    };

    typedef vector<ForceField*> ForceList;
}

 // Local variables

namespace
{
    map<string, Object *> named_objects;
    ForceList forces;
    MouseForce *mouseforce;

    FieldArray *fields;         // the grid of static objects
    ActorList actors;           // list of movable, dynamic objects
}

 // Level access functions

static bool
level_contains(GridPos p)
{
    return (p.x >= 0 && p.y >= 0 &&
            p.x < int(fields->width()) && p.y < int(fields->height()));
}

 // Physics & Collisions
namespace
{
    typedef vector<StoneContact> StoneContactList;
}

//----------------------------------------
// Sphere
//----------------------------------------
namespace
{
    class Sphere {
        V3 center;
        double radius;
    public:
        Sphere(const V3 &c, double r)
            : center(c), radius(r)
        {}
        bool contains(const V3 &p) const {
            return length(center-p) < radius;
        }
    };
}

// Calculate the total acceleration on an actor A at a time TIME.  The
// actor's current position and velocity are also passed.  Note that
// the position and velocity entries in ActorInfo may be updated only
// _after_ a successful time step, so we cannot use them here.
static V3
get_accel (ActorInfo &a, const V3 & x, const V3 & v, double time)
{
    V3 f;
    GridPos p(x[0], x[1]);

    // mouse force
    Floor *floor = GetFloor(p);
    if (floor)
        f += mouseforce->get_force(a, x, v, time) * floor->mousefactor();

    // all other force fields
    for (ForceList::iterator i=forces.begin(); i != forces.end(); ++i)
        f += (*i)->get_force(a, x, v, time);

    // friction
    if (floor && length(v) > 0.00001)
        f -= floor->friction() * v/length(v);

    return f / a.actor->get_mass();
}

static void 
move_actor(ActorInfo &a, double time, double h)
{
    const double MAXVEL = 15;
    
    V3 accel = get_accel(a, a.pos, a.vel, time);
    
    a.pos += h*a.vel;
    a.vel += h*accel;

    // limit maximum velocity
    double q = length(a.vel) / MAXVEL;
    if (q > 1)
        a.vel /= q;
}

static V3 
reflect(V3 v, V3 normal)
{
    double n = length(normal);
    normal /= n;
    return V3(v-2*(v*normal)*normal);
}

class actor_equal : public binary_function<StoneContact,StoneContact,bool> {
  public:
    bool operator()(const StoneContact &a, const StoneContact &b) const 
    { return a.actorinfo == b.actorinfo; }
};

static void
add_contact(StoneContactList &cl, const StoneContact &c)
{
    if (c.stresponse == STONE_PASS)
        return;
    StoneContactList::iterator i;
    i=find_if(cl.begin(), cl.end(), bind2nd(actor_equal(), c));
    if (i!=cl.end()) {
    	i->normal += c.normal;
        i->normal /= length(i->normal);
    }
    else
    	cl.push_back(c);
}

//----------------------------------------
// find_collision_with_stone
//----------------------------------------
static void
find_collision_with_stone(ActorInfo &ad, StoneContactList &cl, int x, int y)
{
    double r = ad.actor->get_radius();
    double ax = ad.pos[0];
    double ay = ad.pos[1];
    const double contact_e = 0.01;
    const double erad = 2.0/32; // edge radius

    Stone* st = GetStone(GridPos(x, y));

    if (!st || !Sphere(make_v3(x+0.5,y+0.5,0), r+0.5*sqrt(2)).contains(ad.pos))
         return;

    V3 cp; // contact point on box
    V3 normal; // face normal


    // contact with north or south face of the stone?
    if (ax>x+erad && ax<x+1-erad) { 
        double dist = r+5;

        // south
        if (ay>y+1 && ad.vel[1]<0) { 
            dist = ay-(y+1);
            cp = make_v3(ax, y+1,0);
            normal = make_v3(0,+1,0);
        }
        // north
        else if (ay<y && ad.vel[1]>0) {
            dist = y-ay;
            cp = make_v3(ax, y,0);
            normal = make_v3(0,-1,0);
        }
        
	if (dist-r < contact_e) {
            StoneContact contact(&ad, st, GridPos(x,y), cp, normal);
            contact.stresponse = st->actor_hit(contact);
            add_contact(cl, contact);
        }
    }
    // contact with west or east face of the stone?
    else if (ay>y+erad && ay<y+1-erad) { 
        double dist=r+5;
        if (ax>x+1 && ad.vel[0]<0) { // east
            dist = ax-(x+1);
            cp=make_v3(x+1,ay,0);
            normal=make_v3(+1,0,0);
        }
        else if (ax<x && ad.vel[0]>0) { // west
            dist = x-ax;
            cp=make_v3(x,ay,0);
            normal=make_v3(-1,0,0);
        }
        
	if (dist-r < contact_e) 
	{
            StoneContact contact(&ad, st, GridPos(x,y), cp, normal);
            contact.stresponse = st->actor_hit(contact);
            add_contact(cl, contact);
        }
    } 
    // contact with any of the four corners?
    else {                    
        V3 corner;
        double cx[2] = {erad, -erad};
        for (int xcorner=0; xcorner<=1; xcorner++) 
        {
            for (int ycorner=0; ycorner<=1; ycorner++) 
            {
                corner[0] = x+xcorner+cx[xcorner];
                corner[1] = y+ycorner+cx[ycorner];

                V3 a=corner-make_v3(ax,ay,0);

                // If the actor is close enough to the corner and doesn't
                // move away from it, we have a collision.
                bool close_enough = (length(a)-r-erad < contact_e);
                bool moves_closer = (a*ad.vel > 0);
                if (close_enough && moves_closer)
                {
                    V3 normal = -a/length(a);
                    V3 cp = corner + normal*erad;

                    StoneContact contact(&ad, st, GridPos(x,y), cp, normal);
                    contact.stresponse = st->actor_hit(contact);
                    add_contact(cl, contact);
                    return;
                }
            }
        }
    }    
}

//----------------------------------------
// find_stone_contacts
//----------------------------------------
static void 
find_stone_contacts(ActorInfo &ad, StoneContactList &cl)
{
    int fieldx = int(ad.pos[0]);
    int fieldy = int(ad.pos[1]);

    // test for collisions with the stones nearby...
    for (int y=fieldy-1; y<=fieldy+1; y++) 
        for (int x=fieldx-1; x<=fieldx+1; x++)
            find_collision_with_stone(ad,cl,x,y);
}

static void
handle_stone_contact(StoneContact &sc)
{
    // was the actor already in contact with this particular stone? ###

    ActorInfo *ai = const_cast<ActorInfo *>(sc.actorinfo);

    if (sc.stresponse == STONE_REBOUND) {
        ai->vel = reflect(ai->vel, sc.normal);
        AddSprite(display::SPRITE_EFFECT, sc.cpoint, "ring-anim");
    }
}

static void
handle_collisions(ActorInfo &a)
{
    GridPos  field(int(a.pos[0]),int(a.pos[1]));
    GridPos ofield(int(a.oldpos[0]),int(a.oldpos[1]));

    if (field != ofield) {
        if (Floor *fl = GetFloor(field)) 
            fl->actor_enter(a);
        if (Floor *ofl = GetFloor(ofield)) 
            ofl->actor_leave(a);

        if (Item *it = GetItem(field)) 
            it->actor_enter(a);
        if (Item *oit = GetItem(ofield)) 
            oit->actor_leave(a);
    }
    if (Floor *fl = GetFloor(field)) {
        fl->actor_hit(a);
    }
    if (Item *it = GetItem(field)) {
        if (it->actor_hit(a)) {
            player::PickupItem(a.actor, field);
        }
    }
    
    StoneContactList cl;
    find_stone_contacts(a, cl);
    for_each(cl.begin(), cl.end(), handle_stone_contact);
}

//----------------------------------------
// tick_actor
//----------------------------------------
static void 
tick_actor(ActorInfo& a, double dtime)
{
    a.oldpos = a.pos;
    a.oldvel = a.vel;

    double rest_time = dtime;
    double dt = dtime;

    while (rest_time > 0) 
    {
        move_actor(a, dtime-rest_time, dt);
                
        double l = length(a.pos-a.oldpos);

        // time step too large? try again
        if (l > 0.02 && dt > 1/1000.) {
            dt /= 2;
            a.pos = a.oldpos;
            a.vel = a.oldvel;
            continue;
        }

        // ok, accept this step

        handle_collisions(a);

        rest_time -= dt;
        a.oldpos = a.pos;
        a.oldvel = a.vel;

        // time step too small?
        if (l < 0.005) {
            dt *= 2;
            dt = dt > rest_time? rest_time : dt;
        }
    }
}

 // world:: global functions

void 
world::Tick(double dtime) 
{
    for (ActorList::iterator i=actors.begin(); i!=actors.end(); ++i) {
        if (!i->grabbed)
            tick_actor(*i, dtime);
    }
    // Tell actors about their new position
    for (ActorList::iterator i=actors.begin(); i!=actors.end(); ++i) {
        Actor *a = i->actor;
        a->move(i->pos);
    }
    mouseforce->tick(dtime);
    for_each(forces.begin(), forces.end(), 
             bind2nd(mem_fun(&ForceField::tick), dtime));

    g_timer.tick(dtime);
    recalc_light_now();         // recalculate laser beams if necessary
}

static void
clear_world()
{
    delete fields;
    delete mouseforce;
    fields=0;
    mouseforce=0;
    delete_sequence(forces.begin(), forces.end());
    forces.clear();

    for_each(actors.begin(), actors.end(), mem_fun_ref(&ActorInfo::dispose));
    actors.clear();
}

void 
world::Create(int w, int h) 
{
    clear_world();
    fields = new FieldArray(w,h);
    mouseforce = new MouseForce;
//    forces.push_back(new ConstantForce(make_v3(-20,0.2,0)));
    display::NewWorld(w, h);
    player::NewWorld(1);        // ## allow more than one player
}

void world::Shutdown()
{
    clear_world();
}

void world::Load(const string &name)
{
    string filename = enigma::FindDataFile(string("levels/") + name + ".lua");
    try {
        lua::Dofile(filename);
    } 
    catch (lua::Error) {
        cerr << "Error while loading landscape " << name << endl;
        clear_world();
        throw;
    }
}

void world::SetMouseForce(const V3 &f) 
{
    mouseforce->add_force(f);
}

void 
world::NameObject(Object *obj, const std::string &name)
{
    named_objects[name] = obj;
}

Object *
world::GetNamedObject(const std::string &name)
{
    return named_objects[name];
}


namespace
{
    Field &get_field(GridPos p) {
        return fields->get(p.x, p.y);
    }
}


//----------------------------------------
// Layer
//
// Changes to a layer are cached and only applied at the end of a
// tick. [### not implemented]
//----------------------------------------
template <class T>
class Layer {
    T *defaultval;
public:
    Layer(T* deflt = 0) : defaultval(deflt) {}
    T *get(GridPos p) {
        return (level_contains(p)) ? raw_get(p) : defaultval;
    }

    T *yield(GridPos p) {
        if (level_contains(p)) {
            T *x = raw_get(p);
            if (x) {
                raw_set(p, 0);
                x->removal();
            }
            return x;
        } else
            return defaultval;
    }

    void set(GridPos p, T *x) {
        if (x && level_contains(p)) {
            kill(p);
            raw_set(p, x);
            x->creation(p);
        }
        else
            dispose(x);
    }
    void kill(GridPos p) { dispose(yield(p)); }

    virtual T* raw_get(GridPos) = 0;
    virtual void raw_set(GridPos, T *) = 0;

private:
    void dispose(T *x) { if(x) x->dispose(); }
};

//----------------------------------------
// Floor manipulation.
//----------------------------------------

namespace
{
    class FloorLayer : public Layer<Floor> {
    private:
        Floor *raw_get(GridPos p) { return get_field(p).floor; }
        void raw_set(GridPos p, Floor *x) {get_field(p).floor = x;}
    };
    FloorLayer fl_layer;
}

void world::KillFloor(GridPos p) {fl_layer.kill(p);}    
Floor *world::GetFloor(GridPos p) {return fl_layer.get(p);}
void world::SetFloor(GridPos p, Floor* st) {fl_layer.set(p,st);}

//----------------------------------------
// Stone manipulation.
//----------------------------------------

namespace 
{
    class BorderStone : public Stone {
    public:
        BorderStone() : Stone("borderstone") {}
        Object *clone() { return this; }
        void dispose() {}
    };

    BorderStone borderstone;


    class StoneLayer : public Layer<Stone> {
    public:
        StoneLayer() : Layer<Stone>(&borderstone) {}
    private:
        Stone *raw_get(GridPos p) { return get_field(p).stone; }
        void raw_set(GridPos p, Stone *st) {get_field(p).stone = st;}
    };

    StoneLayer st_layer;

    void stone_change(GridPos p) {
        Stone *st = GetStone(p);
        if (Item *it = GetItem(p))
            it->stone_change(st);
        if (Floor *fl = GetFloor(p))
            fl->stone_change(st);
    }

}


void world::KillStone(GridPos p) {
    st_layer.kill(p);
    stone_change(p);
}
Stone *world::GetStone(GridPos p) {return st_layer.get(p);}

Stone *
world::YieldStone(GridPos p) 
{
    Stone *st = st_layer.yield(p);
    stone_change(p);
    return st;
}

void 
world::SetStone(GridPos p, Stone* st) 
{
    st_layer.set(p,st);
    stone_change(p);
}

void 
world::MaybeMoveStone(GridPos p, Direction dir)
{
    GridPos newp = move(p,dir);
    if (GetStone(newp) == 0) {
        sound::PlaySound("st-move");
        SetStone(newp, YieldStone(p));
    }
}

void world::SwapStones(GridPos p, GridPos newp)
{
    Stone *tmp = st_layer.yield(newp);
    if (tmp->get_kind() != "borderstone") {
        sound::PlaySound("st-move");
        SetStone(newp, st_layer.yield(p));
        SetStone(p, tmp);
    }
}

//----------------------------------------
// Item manipulation.
//----------------------------------------

namespace
{
    class ItemLayer : public Layer<Item> {
    private:
        Item *raw_get(GridPos p) { return get_field(p).item; }
        void raw_set(GridPos p, Item *x) {get_field(p).item = x;}
    };
    ItemLayer it_layer;
}

void world::KillItem(GridPos p) {it_layer.kill(p);}    
Item *world::GetItem(GridPos p) {return it_layer.get(p);}
Item *world::YieldItem(GridPos p) {return it_layer.yield(p);}
void world::SetItem(GridPos p, Item* st) {it_layer.set(p,st);}

//----------------------------------------
// Actor manipulation.
//----------------------------------------

namespace
{
    ActorInfo &
    get_actor_info(Actor *a) 
    {
        for (ActorList::iterator i=actors.begin(); i!=actors.end(); ++i)
        {
            if (i->actor == a)
                return *i;
        }
        assert(0);
    }
}

void 
world::SetActor(double x, double y, Actor* a)
{
    V3 pos=make_v3(x,y,0);
    actors.push_back(ActorInfo(a,pos, V3()));
    a->on_creation(pos);
}

void
world::GrabActor(Actor *a)
{
    ActorInfo &ai = get_actor_info(a);
    ai.grabbed = true;
}

void
world::ReleaseActor(Actor *a)
{
    ActorInfo &ai = get_actor_info(a);
    ai.grabbed = false;
}

const ActorInfo &
world::GetActorInfo(Actor *a)
{
    return get_actor_info(a);
}
